/**
 * @fileOverview DataGraph - render 'spreadsheet like' data into a graph
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.Layer');
OU.require('OU.util.Instruction');
OU.require('OU.util.HTMLEditor');
OU.require('OU.util.FunctionEditor');
OU.require('OU.util.PopUpForm');
/**
 * @class DataGraph - renders points on a 2D graph based on a data set that the user can enter in a spreadsheet like format.
 * @extends OU.util.Activity
 * @param {object} data - data.js content
 * @param {string} instance - unique instance name
 * @param {OU.util.Controller} controller - (optional) reference of parent controller
 */
OU.activity.DataGraph = function(data, instance, controller) {
    /**
     * Starting point when running with HTML5 Canvas Support
     */
    OU.activity.DataGraph.prototype.canvasView = function() {
        var bH = OU.controlHeight, self = this;
        this.config = {
            depth: 5,
            fps: 40 // 40ms = 25 fps
        };
        this.numTicks = 20;
        this.numDecimals = 2;
        this.lineWidth = 1;
        // create Canvas Layers & Contexts
        this.PAD = this.h * .045;
        this.xColumn = 0;
        this.titlesDiv = new OU.util.Div({
            container: this,
            style: "background:#fff",
            'id': 'titles',
            x: this.x,
            y: this.y,
            w: this.w,
            'h': this.h - bH
        });
        this.modelLayer = new OU.util.Layer({
            container: this,
            'id': 'model',
            'h': this.h - bH,
            hasEvents: true
        });
        this.controlsLayer = new OU.util.Layer({
            container: this,
            'id': 'controls',
            hasEvents: true
        });

        // Set dataset to 1st and render
        for (var i = this.data.datasets.length; i--; ) {
            this.data.datasets[i].container = this;
            this.dataset = new this.Dataset(this.data.datasets[i]);
            this.dataset.save(true);
        }
        this.initControls();

        this.resize();

        OU.__loadData = function() {
            self.loadData();
            return false;
        };
        OU.__importData = function() {
            self.importData();
            return false;
        };
        OU.__saveData = function() {
            self.saveData();
            return false;
        };
        OU.__clearData = function() {
            self.clearData();
            return false;
        };
        OU.__showGraph = function() {
            self.closeData();
            return false;
        };
        OU.__addRows = function() {
            self.dataEntry();
            return false;
        };
        OU.__addCol = function() {
            self.addColumn();
            return false;
        };
        OU.__addFnCol = function() {
            self.addFnColumn();
            return false;
        };
        OU.__editCol = function(n) {
            self.editColumn(n);
            return false;
        };
        OU.__deleteRow = function(n) {
            self.deleteRow(n);
            return false;
        };
        OU.__cellUpdate = function(that) {
            self.cellUpdate({
                id: that.getAttribute('id'),
                value: that.value,
                cell: that
            });
            return false;
        };
        OU.__togglePlot = function(that) {
            self.togglePlot({
                id: that.getAttribute('id'),
                value: that.value,
                cell: that
            });
            return false;
        };
        OU.__updateTitles = function() {
            self.updateTitles();
            return false;
        };

        // open data view
        this.dataEntry();
    };
    /**
     * Resize the activity
     */
    OU.activity.DataGraph.prototype.resize = function() {
        OU.activity.DataGraph.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight, bX = bH * 5;
        this.PAD = this.h * .045;
        this.PAD = this.PAD < 100 ? 100 : this.PAD;
        this.titlesDiv.resize({
            x: this.x,
            y: this.y,
            w: this.w,
            'h': this.h - bH
        });
        this.modelLayer.resize({
            'h': this.h - bH
        });
        this.controlsLayer.resize({
            y: this.y + this.h - bH,
            h: bH
        });
        this.xScale = (this.w - this.PAD * 2) / this.dataset.dx;
        this.yScale = ((this.h * .9) - this.PAD * 2) / this.dataset.dy;

        if (this.dataDiv) {
            this.dataDiv.resize({
                x: this.x,
                y: this.y,
                w: this.w,
                h: this.h
            });
        }
        this.dataButton.resize({
            x: 0,
            y: 0,
            w: bH * 3,
            h: bH
        });
        this.printButton.resize({
            x: bH * 3,
            y: 0,
            w: bH * 3,
            h: bH
        });
        this.settingsButton.resize({
            x: bH * 6,
            y: 0,
            w: bH * 3,
            h: bH
        });

        this.straightLineButton.resize({
            x: this.w - bX,
            y: 0,
            w: bH * 5,
            h: bH
        });
        bX = bX + bH * 4;
        this.curveLineButton.resize({
            x: this.w - bX,
            y: 0,
            w: bH * 4,
            h: bH
        });
        bX = bX + bH * 4;
        this.noLineButton.resize({
            x: this.w - bX,
            y: 0,
            w: bH * 4,
            h: bH
        });
        bX = bX - bH * 3;
        this.render();
        if (OU._popupHighlander)
            OU._popupHighlander.resize();
    };

    /**
     * Render the current view
     */
    OU.activity.DataGraph.prototype.render = function() {
        var ctx = this.modelLayer.context, x, y, i, j, set, xset,
                bH = OU.controlHeight, circ = Math.PI * 2,
                n, sumXY = 0, sumX = 0, sumX2 = 0, sumY = 0, a, b, theta,
                ds = this.dataset,
                xS, yS, x1, x2, r2,
                r = this.w / 200,
                PAD = this.PAD,
                yOff = this.h - bH - PAD,
                intersectX, intersectY;

        ctx.clearRect(0, 0, this.w, this.h - bH);

        if (!ds.isValid()) // duck out now if not enough data to render graph
            return;
        if (this.gradientDiv) {
            this.gradientDiv.remove();
            this.gradientDiv = null;
        }

        this.renderFrame();
        xS = this.xScale;
        yS = this.yScale;
        ctx.save();
        ctx.lineWidth = this.lineWidth;
        xset = ds.sets[this.xColumn];
        if (this.curveFit && ds.numKnots > 1) {
            ds.bezierCurve(ctx);
        }
        this.intersectingLine = null;
        for (i = 0; i < ds.sets.length; i++) {
            if (i !== this.xColumn) {
                set = ds.sets[i];
                if (set.plot !== false) {
                    n = set.points.length;
                    sumXY = 0;
                    sumX = 0;
                    sumX2 = 0;
                    sumY = 0;
                    for (j = n; j--; ) {
                        x = xset.points[j];
                        y = set.points[j];
                        if (this.isNum(x) && this.isNum(y)) {

                            sumXY = sumXY + (x * y);
                            sumX = sumX + x;
                            sumX2 = sumX2 + (x * x);
                            sumY = sumY + y;

                            ctx.fillStyle = set.fillStyle || "#f44";
                            ctx.strokeStyle = set.strokeStyle || "#444";
                            if (x >= ds.x1 && x <= ds.x2 && y >= ds.y1 && y <= ds.y2) {
                                x = PAD + (x - ds.x1) * xS;
                                y = yOff - (y - ds.y1) * yS;
                                if (set.style.toLowerCase() === 'circle') {
                                    ctx.beginPath();
                                    ctx.arc(x, y, r, 0, circ, false);
                                    ctx.fill();
                                    ctx.stroke();
                                }
                                else if (set.style.toLowerCase() === 'rectangle') {
                                    ctx.beginPath();
                                    ctx.fillRect(x - r, y - r, r * 2, r * 2);
                                    ctx.stroke();
                                }
                                else { // triangle
                                    ctx.beginPath();
                                    ctx.moveTo(x, y - r);
                                    ctx.lineTo(x - r, y + r);
                                    ctx.lineTo(x + r, y + r);
                                    ctx.closePath();
                                    ctx.fill();
                                    ctx.stroke();
                                }
                            }
                        }
                    }
                    if (this.lineFit || (this.curveFit && ds.numKnots < 2)) { // best line fit
                        a = (n * sumXY - sumX * sumY) / (n * sumX2 - sumX * sumX);
                        theta = Math.atan(a);
                        b = (sumY - a * sumX) / n;
                        x = ds.x1;
                        y = x * a + b;

                        // uncertainty values
                        var xx, yy, sigmaSum = 0, sigmaY2, sigmaA, sigmaB, xdiffSum = 0;
                        for (j = n; j--; ) {
                            xx = xset.points[j];
                            yy = set.points[j];
                            if (this.isNum(xx) && this.isNum(yy)) {
                                sigmaSum += Math.pow(yy - (a * xx + b), 2);
                                xdiffSum += Math.pow(xx - (sumX / n), 2);
                            }
                        }
                        sigmaY2 = sigmaSum / (n - 2); // Variance of Y
                        sigmaB = Math.sqrt((sumX2 * sigmaY2) / (n * xdiffSum)); // standard deviation of intercept =  uncertainty of intercept
                        sigmaA = Math.sqrt(sigmaY2 / xdiffSum); // standard deviation of slope = gradient uncertainty


                        if (y < ds.y1) { // Y value at x1 is below graph
                            y = ds.y1;
                            x = (y - b) / a;
                        }
                        else if (y > ds.y2) { // Y value at x1 ia above graph
                            y = ds.y2;
                            x = (y - b) / a;
                        }
                        x1 = x;
                        x = PAD + (x - ds.x1) * xS; // x at start of line - left hand side
                        y = yOff - (y - ds.y1) * yS; // y at start of line
                        ctx.beginPath();
                        ctx.moveTo(x, y);
                        x = ds.x2;
                        y = x * a + b;
                        if (y < ds.y1) {
                            y = ds.y1;
                            x = (y - b) / a;
                        }
                        else if (y > ds.y2) {
                            y = ds.y2;
                            x = (y - b) / a;
                        }
                        x2 = x;
                        x = PAD + (x - ds.x1) * xS; // x at end of line - right hand side
                        y = yOff - (y - ds.y1) * yS; // y at end of line
                        ctx.lineTo(x, y);
                        ctx.strokeStyle = set.fillStyle || '#44f';
                        ctx.stroke();

                        if (ds.gradX === undefined)
                            ds.gradX = x1 + (x2 - x1) / 2;
                        ds.gradY = ds.gradX * a + b;
                        x = PAD + (ds.gradX - ds.x1) * xS; // x at gradient point
                        y = yOff - (ds.gradY - ds.y1) * yS; // y at gradient point

                        if (ds.numPlotSets === 1) {
                            if (ds.gradX < x2 && ds.gradX > x1) {
                                ctx.strokeStyle = '#999';
                                ctx.moveTo(PAD, y);
                                ctx.lineTo(x, y);
                                ctx.lineTo(x, yOff);
                                ctx.stroke();
                            }
                            r2 = ds.calcR2({
                                a: a,
                                b: b,
                                ysetIdx: i
                            });
                            this.gradientInfo('<strong><u>Straight line details</u></strong><br/>'
                                    + 'x: ' + this.niceNumberFormatx10(ds.gradX) + '<br/>'
                                    + 'y: ' + this.niceNumberFormatx10(ds.gradY) + '<br/>'
                                    + 'gradient: ' + this.niceNumberFormatx10(a) + ' (± ' + this.niceNumberFormatx10(sigmaA) + ')<br/>'
                                    + 'intercept: ' + this.niceNumberFormatx10(b) + ' (± ' + this.niceNumberFormatx10(sigmaB) + ')<br/>'
//                                + 'gradient angle: ' + this.niceNumberFormatx10(theta * 180 / Math.PI) + '&deg;<br/>'
                                    + 'R<sup>2</sup>: ' + this.niceNumberFormatx10(r2) + '<br/>');
                        }
                        else if (ds.numPlotSets === 2) {
                            if (this.intersectingLine) {
                                intersectX = (this.intersectingLine.b - b) / (a - this.intersectingLine.a);
                                intersectY = intersectX * a + b;
                                this.gradientInfo('<strong><u>Lines intersect at</u></strong><br/>'
                                        + 'x: ' + this.niceNumberFormatx10(intersectX) + '<br/>'
                                        + 'y: ' + this.niceNumberFormatx10(intersectY) + '<br/>');
                            }
                            else {
                                this.intersectingLine = {
                                    a: a,
                                    b: b
                                };
                            }
                        }
                    }
                }
            }
        }
        ctx.restore();
        this.renderControls();
    };
    OU.activity.DataGraph.prototype.gradientInfo = function(html) {
        if (!this.gradientDiv) {
            this.gradientDiv = new OU.util.Div({
                container: this,
                x: this.x + this.w - 380,
                y: this.y,
                w: 320,
                h: 'auto',
                htmlClass: 'infoBox'
            });
        }
        this.gradientDiv.div.innerHTML = html;
    };
    /**
     * Determine if given object is a number
     * @param {object} n - possibly a number
     * @returns {boolean} true if input is a number
     */
    OU.activity.DataGraph.prototype.isNum = function(n) {
        n = parseFloat(n);
        return !isNaN(n) && isFinite(n);
    };
    /*
     * Find a nice tick size for the axis
     */
    OU.activity.DataGraph.prototype.roundTicks = function(delta) {
        var unroundedTickSize = delta / (this.numTicks - 1),
                x = Math.ceil((Math.log(unroundedTickSize) / Math.LN10) - 1),
                pow10x = Math.pow(10, x);
        return Math.ceil(unroundedTickSize / pow10x) * pow10x;
    };

    /**
     * draw the axises
     */
    OU.activity.DataGraph.prototype.renderFrame = function() {
        var ctx = this.modelLayer.context,
                bH = OU.controlHeight,
                ds = this.dataset,
                xAxis, yAxis, i, step, val,
                xS = this.xScale,
                yS = this.yScale,
                PAD = this.PAD;

        ds.calcRanges();
        xS = this.xScale;
        yS = this.yScale;

//        ctx.gradRect();
        ctx.save();
        ctx.font = new OU.font({size: 16}).str();
        ctx.fillStyle = '#fff';
        ctx.fillRect(PAD, PAD, this.w - 2 * PAD, this.h - bH - (2 * PAD));
        ctx.strokeStyle = "#000";
        ctx.lineWidth = 1;
        ctx.fillStyle = "#444";
        yAxis = PAD; // -ds.x1 * xS + PAD; - alt version to place y-axis at X=0
        yAxis = yAxis < PAD ? PAD : (yAxis > this.w - PAD ? this.w - PAD : yAxis); // ensure YAxis is always visible
        xAxis = this.h - PAD - bH; // ds.y2 * yS + PAD; - alt version to place the x-axis at Y=0
        xAxis = xAxis < PAD ? PAD : (xAxis > this.h - PAD - bH ? this.h - PAD - bH : xAxis);
        ctx.beginPath();
        ctx.moveTo(yAxis, PAD);
        ctx.lineTo(yAxis, this.h - bH - PAD);
        ctx.closePath();
        ctx.stroke();
        step = ds.yticks;
        ctx.textAlign = 'right';
        //      ctx.fillText("x:"+ds.x1+" to "+ds.x2+"    y:"+ds.y1+" to "+ds.y2+"    dX:"+ds.dx+"    dY:"+ds.dy,this.w-10,10);
        ctx.save();
        ctx.lineWidth = 0.25;
        ctx.strokeStyle = '#aaa';

        // Y axis
        // determine if we can use ints & make sure labels don't overlap
        var needDecimals = false;
        var origNumDecimals = this.numDecimals;
        var maxHeight = 16, stepJumps = 1, ticks = 0;
        for (i = parseFloat(ds.y2); i >= parseFloat(ds.y1); i = i - step) {
            if (i !== (i | 0)) {
                needDecimals = true;
            }
        }
        var origNumDecimals = this.numDecimals;
        if (!needDecimals) {
            this.numDecimals = 0;
        }
        if (maxHeight > step * yS - 5) { // If tick value are going to overlap, then drop some out
            stepJumps = ((maxHeight / (step * yS - 5)) | 0) + 1;
        }
        for (i = parseFloat(ds.y2); i >= parseFloat(ds.y1); i = i - step) {
            ctx.beginPath();
            ctx.moveTo(this.x + this.w - PAD, PAD + (ds.y2 - i) * yS);
            ctx.lineTo(PAD, PAD + (ds.y2 - i) * yS);
            ctx.stroke();
            ctx.save();
            ctx.beginPath();
            ctx.lineWidth = 2;
            ctx.strokeStyle = '#666';
            ctx.moveTo(PAD, PAD + (ds.y2 - i) * yS);
            ctx.lineTo(PAD - 8, PAD + (ds.y2 - i) * yS);
            ctx.stroke();
            ctx.restore();
            if (ticks / stepJumps === ((ticks / stepJumps) | 0)) {
                val = this.niceNumberFormat(i);
                ctx.fillText(val, yAxis - 15, PAD + (ds.y2 - i) * yS);
            }
            ticks++;
        }

        this.numDecimals = origNumDecimals;

        ctx.restore();
        ctx.beginPath();
        ctx.moveTo(PAD, xAxis);
        ctx.lineTo(this.w - PAD, xAxis);
        ctx.closePath();
        ctx.stroke();
        step = ds.xticks;
        ctx.textAlign = 'center';
        ctx.save();
        ctx.strokeStyle = '#aaa';

        // X axis
        // determine if we can drop back to ints
        // and maximum tick mark width
        needDecimals = false;
        var maxWidth = 0;
        for (i = parseFloat(ds.x1) + step; i <= parseFloat(ds.x2); i = i + step) {
            if (i !== (i | 0)) {
                needDecimals = true;
            }
            val = this.niceNumberFormat(i);
            var m = ctx.measureText(val, PAD + (i - ds.x1) * xS, xAxis + 30);
            if (m.width > maxWidth) {
                maxWidth = m.width;
            }
        }
        if (!needDecimals) {
            this.numDecimals = 0;
        }
        stepJumps = 1;
        ticks = 0;

        if (maxWidth > step * xS - 5) { // If tick value are going to overlap, then drop some out
            stepJumps = ((maxWidth / (step * xS - 5)) | 0) + 1;
        }
        for (i = parseFloat(ds.x1) + step; i <= parseFloat(ds.x2); i = i + step) {
            ctx.beginPath();
            ctx.moveTo(PAD + (i - ds.x1) * xS, xAxis);
            ctx.lineTo(PAD + (i - ds.x1) * xS, PAD);
            ctx.stroke();
            ctx.save();
            ctx.beginPath();
            ctx.lineWidth = 2;
            ctx.strokeStyle = '#666';
            ctx.moveTo(PAD + (i - ds.x1) * xS, xAxis);
            ctx.lineTo(PAD + (i - ds.x1) * xS, xAxis + 8);
            ctx.stroke();
            ctx.restore();
            if (ticks / stepJumps === ((ticks / stepJumps) | 0)) {
                val = this.niceNumberFormat(i);
                ctx.fillText(val, PAD + (i - ds.x1) * xS, xAxis + 30);
            }
            ticks++;
        }
        this.numDecimals = origNumDecimals;

        ctx.restore();
        if (this.titleDiv) {
            this.titleDiv.resize({
                x: this.x + PAD,
                y: PAD / 2,
                w: this.w - PAD,
                h: 'auto'
            });
            this.titleDiv.div.innerHTML = ds.graphTitle;
        }
        else {
            this.titleDiv = new OU.util.Div({
                container: this,
                parentDiv: this.titlesDiv.div,
                style: 'text-align:left;',
                innerHTML: ds.graphTitle,
                x: this.x + PAD,
                y: PAD / 2,
                w: this.w - PAD,
                h: 'auto'
            });
        }
        if (this.xAxisDiv) {
            this.xAxisDiv.resize({
                x: this.x,
                y: this.h - bH - PAD / 2,
                w: this.w,
                h: 'auto'
            });
            this.xAxisDiv.div.innerHTML = ds.xAxisTitle;
        }
        else {
            this.xAxisDiv = new OU.util.Div({
                container: this,
                parentDiv: this.titlesDiv.div,
                style: 'text-align:center;',
                innerHTML: ds.xAxisTitle,
                x: this.x,
                y: this.h - bH - PAD / 2,
                w: this.w - PAD * 2,
                h: 'auto'
            });
        }
        if (this.yAxisDiv) {
            this.yAxisDiv.resize({
                x: this.x + 20 - (this.h - bH - PAD * 2) / 2,
                y: this.h / 2 - bH,
                w: this.h - bH - PAD * 2,
                h: 'auto'
            });
            this.yAxisDiv.div.innerHTML = ds.yAxisTitle;
        }
        else {
            this.yAxisDiv = new OU.util.Div({
                container: this,
                parentDiv: this.titlesDiv.div,
                style: 'text-align:center;',
                htmlClass: 'rotateAC90',
                innerHTML: ds.yAxisTitle,
                x: this.x + 20 - (this.h - bH - PAD * 2) / 2,
                y: this.h / 2 - bH,
                w: this.h - bH - PAD * 2,
                h: 'auto'
            });
        }
    };
    /**
     * Initialise the controls
     */
    OU.activity.DataGraph.prototype.initControls = function() {
        var bH = OU.controlHeight,
                clickable = this.controlsLayer.events.clickable,
                self = this, bX = bH * 5;
        clickable.length = 0;
        this.dataButton = new OU.util.ControlButton({
            txt: 'Edit Data',
            x: 0,
            y: 0,
            w: bH * 3,
            h: bH,
            layer: this.controlsLayer,
            onClick: function() {
                self.dataEntry();
            }
        });
        this.printButton = new OU.util.ControlButton({
            txt: 'Print',
            x: bH * 3,
            y: 0,
            w: bH * 3,
            h: bH,
            layer: this.controlsLayer,
            onClick: function() {
                window.print();
            }
        });
        this.settingsButton = new OU.util.ControlButton({
            txt: 'Settings',
            x: bH * 6,
            y: 0,
            w: bH * 3,
            h: bH,
            layer: this.controlsLayer,
            onClick: function() {
                self.settings();
            }
        });
        this.straightLineButton = new OU.util.RadioButton({
            group: 'fit',
            txt: 'Straight Line',
            x: this.w - bX,
            y: 0,
            w: bH * 5,
            h: bH,
            layer: this.controlsLayer,
            onOffIndicator: true,
            onOff: false,
            background: {col: '#fff'},
            onClick: function() {
                self.straightLine();
            }
        });
        bX = bX + bH * 5;
        this.curveLineButton = new OU.util.RadioButton({
            group: 'fit',
            txt: 'Curve Line',
            x: this.w - bX,
            y: 0,
            w: bH * 5,
            h: bH,
            layer: this.controlsLayer,
            onOffIndicator: true,
            onOff: false,
            background: {col: '#fff'},
            onClick: function() {
                self.curveLine();
            }
        });
        bX = bX + bH * 5;
        this.noLineButton = new OU.util.RadioButton({
            group: 'fit',
            txt: 'No Line',
            x: this.w - bX,
            y: 0,
            w: bH * 5,
            h: bH,
            layer: this.controlsLayer,
            onOffIndicator: true,
            onOff: true,
            background: {col: '#fff'},
            onClick: function() {
                self.noLine();
            }
        });
        this.noLineButton.state(true);

    };
    /**
     * Switch the straightline render on or off
     */
    OU.activity.DataGraph.prototype.curveLine = function() {
        this.curveFit = true;
        this.lineFit = false;
        this.render();
    };
    /**
     * Switch the straightline render on or off
     */
    OU.activity.DataGraph.prototype.straightLine = function() {
        if (this.gradientDiv)
            this.gradientDiv.remove();
        this.gradientDiv = null;
        this.dataset.gradX = undefined;

        this.curveFit = false;
        this.lineFit = true;
        this.render();
    };
    /**
     * Switch the straightline render on or off
     */
    OU.activity.DataGraph.prototype.noLine = function() {
        if (this.gradientDiv)
            this.gradientDiv.remove();
        this.gradientDiv = null;

        this.curveFit = false;
        this.lineFit = false;
        this.render();
    };
    /**
     * Render the control elements
     */
    OU.activity.DataGraph.prototype.renderControls = function() {
        this.controlsLayer.context.gradRect();
        this.dataButton.render();
        this.printButton.render();
        this.settingsButton.render();
        this.straightLineButton.render();
        this.curveLineButton.render();
        this.noLineButton.render();
    };
    OU.activity.DataGraph.prototype.countRows = function() {
        var ds = this.dataset, numCols = ds.sets.length, i, rows, numRows = 0;
        numCols = numCols < 2 ? 2 : numCols;
        for (i = numCols; i--; ) {
            if (ds.sets[i] && ds.sets[i].points) {
                rows = ds.sets[i].points.length;
                if (rows > numRows)
                    numRows = rows;
            }
        }
        return numRows;
    };
    /**
     * Display the data in "spreadsheet" format
     */
    OU.activity.DataGraph.prototype.dataEntry = function() {
        var div, h, titlesHtml, i, j, col, ds = this.dataset,
                numCols = ds.sets.length, name, checked,
                val, numRows = 0, self = this;
        if (this.gradientDiv)
            this.gradientDiv.remove();
        this.gradientDiv = null;

        if (!this.dataDiv) {
            this.dataDiv = new OU.util.Div({
                x: this.x,
                y: this.y,
                w: this.w,
                h: this.h,
                container: this,
                zIndex: OU.POP_UP_LEVEL,
                htmlClass: "hidden3d",
                overflow: "auto",
                style: "background:#fff"
            });
        }
        div = this.dataDiv.div;
        numCols = numCols < 2 ? 2 : numCols;
        numRows = this.countRows() + 3;
        h = "<div class='dgColNums'><span>&nbsp;</span><span title='Select columns to include on the graph by ticking the boxes in this row.'>Plot?</span>";
        for (j = 0; j < numRows; j++) {
            h = h + "<span><strong title='Delete this row' onclick='OU.__deleteRow(" + j + ");'>\u2612</strong> " + (j + 1) + "</span>";
        }
        h = h + "<span title='Add more rows' onclick='return OU.__addRows();'> + </span>";
        h = h + "</div>";
        for (i = 0; i < numCols; i++) {
            col = ds.sets[i];
            if (col) {
                name = col.name;
            }
            else {
                name = null;
                col = {fillStyle: '#444'};
            }
            if (name) {
                if (i < 1)
                    name = "X";
                else
                    name = String.fromCharCode(64 + i) + ' [' + name + ']';
            }
            else {
                if (i < 1)
                    name = "X";
                else
                    name = String.fromCharCode(64 + i);
            }

            if (i > 0) {
                name = name + '<span style="font-size:1.4em;line-height:0.7em;color:' + col.fillStyle + '">';
                if (col.style.toLowerCase() === 'circle')
                    name = name + '  ●</span>';
                else if (col.style.toLowerCase() === 'rectangle')
                    name = name + '  ■</span>';
                else
                    name = name + '  ▲</span>';

                h = h + "<div class='dgColumn cog' >";
            }
            else {
                h = h + "<div class='dgColumn'>";
            }
            if (col.plot !== false)
                checked = 'checked';
            else
                checked = '';
            h = h + "<div style='margin-right:30px;' onclick='OU.__editCol(" + i + ");' title='Click to edit column settings'>" + name + "</div>";
            if (i > 0)
                h = h + "<span class='plotTick' title='If ticked this column will be included on the graph'><input type='checkbox' id='plottick-" + i + "_" + j + "' onchange='return OU.__togglePlot(this);' " + checked + "/></span>";
            else
                h = h + "<span class='plotTick'></span>";
            for (j = 0; j < numRows; j++) {
                val = this.niceNumberFormat(col.points[j]);
                if (col.function)
                    h = h + "<input title='This column is calculated using a function' disabled='true' id='dgcell-" + i + "_" + j + "' value='" + val + "'/>";
                else
                    h = h + "<input onchange='return OU.__cellUpdate(this);' id='dgcell-" + i + "_" + j + "' value='" + val + "'/>";
            }
            h = h + "</div>";
        }
        if (ds.sets.length < 6) {
            h = h + "<div  style='width:20px;'class='dgColumn'><div title='Add an empty column' onclick='return OU.__addCol();'> + </div></div>"
                    + "<div  style='width:30px;'class='dgColumn'><div title='Add a column using a function' onclick='return OU.__addFnCol();'> +Fn </div></div>";
        }

        titlesHtml = "<table id='graphTitles'><tr><th>Graph title</th><th>X-axis title</th><th>Y-axis title</th></tr>"
                + "<tr><td><div id='_graphTitle' tabIndex=1>" + ds.graphTitle + "</div></td>"
                + "<td><div id='_xAxisTitle' tabIndex=2>" + ds.xAxisTitle + "</div></td>"
                + "<td><div id='_yAxisTitle' tabIndex=3>" + ds.yAxisTitle + "</div></td></tr></table>";

        div.innerHTML = "<p style='text-align:right; float:right;width:auto;'>"
                + "<input type='button' class='_abuttonNOFLOAT' onclick='return OU.__showGraph();' tabIndex=4 value='View Graph'/>&nbsp;&nbsp;"
                + "<input type='button' class='_abuttonNOFLOAT' onclick='return OU.__loadData();'tabIndex=5 value='Load data'/>&nbsp;&nbsp;"
                + "<input type='button' class='_abuttonNOFLOAT' onclick='return OU.__saveData();'tabIndex=6 value='Save data'/>&nbsp;&nbsp;"
                + "<input type='button' class='_abuttonNOFLOAT' onclick='return OU.__clearData();'tabIndex=7 value='Clear data'/>&nbsp;&nbsp;"
                + "<input type='button' class='_abuttonNOFLOAT' onclick='return OU.__importData();'tabIndex=8 value='Import data'/>&nbsp;&nbsp;</p>"
                + titlesHtml
                + '<div id="tableWrapper" style="overflow:auto; width:600%;height:80%"><div id="tableFixed" style="display:inline-block">' + h + '</div></div>';

        var fTable = document.getElementById("tableFixed"),
                wTable = document.getElementById("tableWrapper");
        fTable.style.width = (fTable.clientWidth + 60) + 'px';
        wTable.style.width = '100%';

        OU.addClass(div, "show3d"); // bring the div into view
        this.modelLayer.canvas.className = 'hidden3d';
        this.controlsLayer.canvas.className = 'noPrint hidden3d';
        if (this.titlesDiv)
            this.titlesDiv.div.className = '_overlaydiv hidden3d';

        this.titleEditor = new OU.util.HTMLEditor({
            container: this,
            targetDiv: document.getElementById('_graphTitle'),
            onClose: function() {
                self.updateTitles();
            }
        });
        this.xAxisEditor = new OU.util.HTMLEditor({
            container: this,
            targetDiv: document.getElementById('_xAxisTitle'),
            onClose: function() {
                self.updateTitles();
            }
        });
        this.yAxisEditor = new OU.util.HTMLEditor({
            container: this,
            targetDiv: document.getElementById('_yAxisTitle'),
            onClose: function() {
                self.updateTitles();
            }
        });
    };

    OU.activity.DataGraph.prototype.recalcFns = function() {
        var c, r, col, j, vars, val,
                ds = this.dataset,
                numCols = ds.sets.length,
                numRows = this.countRows();
        //work through columns, if functions then re-calc
        for (c = 0; c < numCols; c++) {
            col = ds.sets[c];
            if (col.function) {

                // populate the column using the given function
                for (r = numRows; r--; ) {
                    try {
                        vars = [];
                        for (j = ds.sets.length; j--; )
                            vars[j] = ds.sets[j].points[r];
                        val = this.evalFn(col.function, vars);
                        if (this.isNum(val)) {
                            col.points[r] = val;
                            document.getElementById('dgcell-' + c + "_" + r).value = this.niceNumberFormat(val);
                        }
                    }
                    catch (e) {
                        col.points[r] = '';
                        document.getElementById('dgcell-' + c + "_" + r).value = '';
                        console.log("Error populating column: " + e.message);
                    }
                }
            }
        }
    };

    // Extend Math object to include log10 function
    if (!Math.log10) {
        Math.log10 = function(val) {
            return Math.log(val) / Math.LN10;
        };
    }
    /**
     * Evaluate a function defined in a string using the variables passed in an array
     * @param {String} fnStr - the function to evaluate
     * @param {Object[]} vars - the variables in an array in order [X,A,B,C,...]
     * @throws {Error} Error if eval fails
     */
    OU.activity.DataGraph.prototype.evalFn = function(fnStr, vars) {
        var i, c, chunk1, chunk2, reg,
                replaceStrs = ['sin', 'cos', 'tan', 'asin', 'acos', 'atan', 'abs', 'ln', 'log10', 'exp', 'sqrt', 'e', 'E'],
                withStrs = ['Math.sin', 'Math.cos', 'Math.tan', 'Math.asin', 'Math.acos', 'Math.atan', 'Math.abs', 'Math.log', 'Math.log10', 'Math.exp', 'Math.sqrt', 'Math.E', '*10^'];

        // replace the variables with their values
        for (i = fnStr.length; i--; ) {
            c = fnStr.charCodeAt(i);
            if (c >= 65 && c < 91) {
                if (c === 88)
                    c = 0;
                else
                    c = c - 64;
                if (c < 5) { // restrict to variables X,A,B,C,D
                    chunk1 = fnStr.substr(0, i);
                    chunk2 = fnStr.substr(i + 1);
                    if (vars[c] === undefined)
                        throw new Error("Unknown variable: " + fnStr.charAt(i));
                    fnStr = chunk1 + vars[c] + chunk2;
                }
            }
        }
        // convert function names to javascript functions
        for (i = replaceStrs.length; i--; ) { // replace strs
            reg = new RegExp("\\b" + replaceStrs[i] + "\\b", "g");
            fnStr = fnStr.replace(reg, withStrs[i]);

        }
        // replace single chars
        fnStr = fnStr.replace('÷', '/');
        fnStr = fnStr.replace('×', '*');
        fnStr = fnStr.replace('Π', 'Math.PI');
//        console.log('evalling: ' + fnStr);
        return eval(fnStr); // Assumes calling function is going to catch and deal with any errors
    };
    OU.activity.DataGraph.prototype.niceNumberFormat = function(num) {
        if (num === undefined || num === '' || isNaN(num))
            return '';
        var nice = parseFloat(num);
        if ((Math.abs(nice) < Math.pow(10, -this.numDecimals) && nice !== 0) || nice > 1000000) {
            nice = nice.toExponential(this.numDecimals);
        }
        else {
            nice = nice.toFixed(this.numDecimals);
        }
        if (isNaN(nice)) {
            return '';
        }
        return nice;
    };
    OU.activity.DataGraph.prototype.niceNumberFormatx10 = function(num) {
        if (num === undefined || num === '')
            return '';
        if (isNaN(num)) {
            return 'NaN';
        }
        var nice = parseFloat(num);
        if (isNaN(nice)) {
            return 'NaN';
        }
        if ((Math.abs(nice) < Math.pow(10, -this.numDecimals) && nice !== 0) || nice > 1000000) {
            nice = nice.toExponential(this.numDecimals);
            if (nice.indexOf('e') > 0) {
                nice = nice.replace('e', 'x10<sup>') + '</sup>';
            }
        }
        else {
            nice = nice.toFixed(this.numDecimals);
        }
        return nice;
    };
    OU.activity.DataGraph.prototype.cellUpdate = function(params) {
        var id = params.id.split('-')[1],
                ij = id.split('_'),
                i = ij[0], j = ij[1];
        if (params.value === '')
            this.dataset.sets[i].points[j] = null;
        else
            this.dataset.sets[i].points[j] = parseFloat(params.value);
        params.cell.value = this.niceNumberFormat(params.value);
        this.recalcFns();
    };
    OU.activity.DataGraph.prototype.togglePlot = function(params) {
        var id = params.id.split('-')[1],
                ij = id.split('_'),
                i = ij[0], j = ij[1];
        this.dataset.sets[i].plot = !this.dataset.sets[i].plot;
    };
    OU.activity.DataGraph.prototype.updateTitles = function() {
        var ds = this.dataset, val = document.getElementById('_graphTitle').innerHTML;
        if (val)
            ds.graphTitle = val;
        val = document.getElementById('_xAxisTitle').innerHTML;
        if (val)
            ds.xAxisTitle = val;
        val = document.getElementById('_yAxisTitle').innerHTML;
        if (val)
            ds.yAxisTitle = val;
    };
    OU.activity.DataGraph.prototype.settings = function() {
        var self = this, i, form = '<p><span style="width: 100px"/>Line width: </span><input type="text" id="_lineWidth" value="' + this.lineWidth + '"/></p>'
                + '<p><span style="width: 100px"/>Significant figures: </span><select id="_sigfigs">';
        for (i = 0; i < 7; i++) {
            if (i === this.numDecimals)
                form = form + '<option selected>' + i + '</option>';
            else
                form = form + '<option>' + i + '</option>';
        }
        form = form + '</select></p><p style="text-align:right"><input type="submit" value="Done"/>';
        new OU.util.PopUpForm({
            container: this,
            formHTML: form,
            onSubmit: function() {
                self._updateSettings();
            }
        });
    };
    OU.activity.DataGraph.prototype._updateSettings = function() {
        this.numDecimals = document.getElementById('_sigfigs').value;
        this.lineWidth = document.getElementById('_lineWidth').value;
        this.lineWidth = this.lineWidth > 10 ? 10 : this.lineWidth;
        this.lineWidth = this.lineWidth < 0.5 ? 0.5 : this.lineWidth;
        this.render();
    };
    OU.activity.DataGraph.prototype.loadData = function() {
        var self = this, form = "<p>Select the data set to load</p><p><select size='6' id='_dataChoices'>",
                setsStr = OU.LocalStorage.load('OU.datagraph_.datasets'), sets, i;
        if (setsStr)
            sets = JSON.parse(setsStr);
        if (setsStr && sets.length > 0) {
            sets.sort();
            for (i = 0; i < sets.length; i++)
                form += "<option>" + sets[i] + "</option>";
            form += "</select></p><p><input type='button' value='Cancel' onclick='OU._popupHighlander.remove(); return false;'/>&nbsp;<input type='submit' value='Load'/></p>";

            new OU.util.PopUpForm({
                container: this,
                formHTML: form,
                onSubmit: function() {
                    self._loadData(document.getElementById('_dataChoices').value);
                }
            });
        }
        else {
            new OU.util.PopUpForm({
                container: this,
                formHTML: "<p>There are no saved data files</p><p><input type='submit' value='OK'/></p>"
            });
        }
    };
    OU.activity.DataGraph.prototype._loadData = function(dataName) {
        this.dataset = new this.Dataset({
            container: this,
            name: dataName
        });
        this.dataset.load();
        this.dataEntry();
        this.recalcFns();
    };
    OU.activity.DataGraph.prototype.importData = function() {
        var self = this,
                form = "<p>Paste new CSV formatted data below</p>"
                + "<textarea id='_newcsv' style='width: 90%; height: 160px;margin-left: 5%;'></textarea>"
                + "<p>Data set name: <input type='text' id='_newName'/>&nbsp;&nbsp;&nbsp;&nbsp;<input type='checkbox' id='_incTitles'/> Use 1st row for titles</p>"
                + "<p><input type='button' value='Cancel' onclick='OU._popupHighlander.remove(); return false;'/>&nbsp;<input type='submit' value='Import'/></p>";

        new OU.util.PopUpForm({
            container: this,
            formHTML: form,
            onSubmit: function() {
                self._importData();
            }
        });
    };
    OU.activity.DataGraph.prototype._importData = function() {
        var dataName = document.getElementById('_newName').value,
                incTitles = document.getElementById('_incTitles').checked;
        dataName = dataName || 'importedData';
        dataName = this.createNewName(dataName);

        var csv = document.getElementById('_newcsv').value;
        if (!csv || csv === '') {
            console.log('nothing to see here!');
            return;
        }
        this.dataset = new this.Dataset({
            container: this,
            name: dataName
        });
        var newDataArr = this.CSVToArray(csv);
        this.dataset.import(newDataArr, incTitles);
        this.dataEntry();
        this.recalcFns();
    };
    OU.activity.DataGraph.prototype.createNewName = function(newName) {
        var i, setsStr = OU.LocalStorage.load('OU.datagraph_.datasets'), sets, validName = newName, failed = true, attempt = 1;
        if (setsStr)
            sets = JSON.parse(setsStr);
        if (setsStr && sets.length > 0) {
            while (failed === true) {
                failed = false;
                for (i = sets.length; i--; ) {
                    if (sets[i] === validName)
                        failed = true;
                }
                if (failed) {
                    validName = newName + attempt;
                    attempt++;
                }
            }
        }
        return validName;
    };
    OU.activity.DataGraph.prototype.CSVToArray = function(strData) {
        var strDelimiter = ",",
                objPattern = new RegExp(// Regular expression to parse the CSV values.
                (// Delimiters.
                        "(\\" + strDelimiter + "|\\r?\\n|\\r|^)" +
                        // Quoted fields.
                        "(?:\"([^\"]*(?:\"\"[^\"]*)*)\"|" +
                        // Standard fields.
                        "([^\"\\" + strDelimiter + "\\r\\n]*))"
                        ),
                "gi"
                ),
                arrData = [[]], arrMatches = null;

        // Keep looping over the regular expression matches
        // until we can no longer find a match.
        while (arrMatches = objPattern.exec(strData)) {

            // Get the delimiter that was found.
            var strMatchedDelimiter = arrMatches[ 1 ];

            // Check to see if the given delimiter has a length
            // (is not the start of string) and if it matches
            // field delimiter. If id does not, then we know
            // that this delimiter is a row delimiter.
            if (strMatchedDelimiter.length &&
                    (strMatchedDelimiter !== strDelimiter)) {
                // Since we have reached a new row of data,
                // add an empty row to our data array.
                arrData.push([]);
            }

            // Now that we have our delimiter out of the way,
            // let's check to see which kind of value we
            // captured (quoted or unquoted).
            if (arrMatches[ 2 ]) {
                // We found a quoted value. When we capture
                // this value, unescape any double quotes.
                var strMatchedValue = arrMatches[ 2 ].replace(
                        new RegExp("\"\"", "g"),
                        "\""
                        );
            } else { // We found a non-quoted value.
                var strMatchedValue = arrMatches[ 3 ];
            }
            // Now that we have our value string, let's add
            // it to the data array.
            arrData[ arrData.length - 1 ].push(strMatchedValue);
        }

        return this.flipArray(arrData);
    };
    OU.activity.DataGraph.prototype.flipArray = function(ijArray) {
        var jiArray = [], i, j;
        for (i = 0; i < ijArray.length; i++) {
            for (j = 0; j < ijArray[i].length; j++) {
                if (jiArray[j] === undefined)
                    jiArray[j] = [];
                jiArray[j][i] = ijArray[i][j];
            }
        }
        return jiArray;
    };
    OU.activity.DataGraph.prototype.saveData = function() {
        var self = this, form = "<p>Enter a name for this dataset</p>"
                + "<p><input type='text' id='_dataName' value='" + this.dataset.name + "'/></p>"
                + "<p><input type='button' value='Cancel' onclick='OU._popupHighlander.remove(); return false;'/>&nbsp;<input type='submit' value='Save'/></p>";
        new OU.util.PopUpForm({
            container: this,
            formHTML: form,
            onSubmit: function() {
                self._saveData(document.getElementById('_dataName').value);
            }
        });
    };
    OU.activity.DataGraph.prototype._saveData = function(name) {
        this.dataset.name = name;
        this.dataset.save(true);
    };
    OU.activity.DataGraph.prototype.clearData = function() {
        var ds = this.dataset;

        ds.sets[0].points = [];
        ds.sets[1].points = [];
        ds.sets.splice(2, ds.sets.length - 2);
        this.dataEntry();
    };
    OU.activity.DataGraph.prototype.closeData = function() {
        var div = this.dataDiv.div, ds = this.dataset;
        if (!ds.isValid()) { // not enough data, so warn user
            if (this._minDataInfo) {
                this._minDataInfo.grow();
            }
            else {
                this._minDataInfo = new OU.util.Instruction({
                    container: this,
                    message: "To view a graph, please ensure you have at least 1 column containing valid data selected for plot."
                });
            }
            return false;
        }
        if (this._minDataInfo) { // if min data message has previsouly been displayed remove it
            this._minDataInfo.remove();
            this._minDataInfo = null;
        }
        this.titleEditor.remove();
        this.xAxisEditor.remove();
        this.yAxisEditor.remove();
        div.className = "_overlaydiv hidden3d"; // hide the div
        this.modelLayer.canvas.className = 'hidden3d show3d';
        this.controlsLayer.canvas.className = 'noPrint hidden3d show3d';
        if (this.titlesDiv)
            this.titlesDiv.div.className = '_overlaydiv hidden3d show3d';
        this.dataset.calc();
        this.render();
        return false;
    };
    OU.activity.DataGraph.prototype.editColumn = function(colNum) {
        var self = this, i,
                shapes = ["Circle", "Triangle", "Rectangle"],
                colours = ["Aqua", "Black", "Blue", "Fuchsia", "Gray", "Green", "Lime", "Maroon", "Navy", "Olive", "Purple", "Red", "Silver", "Teal", "White", "Yellow"],
                set = this.dataset.sets[colNum],
                form = "<table style='width:auto; float:none;'><tr><td>Label name:</td><td><input type='text' id='_labelName' value='" + set.name + "'/></td></tr>"
                + "<tr><td>Marker shape:</td><td><select size='" + shapes.length + "' id='_marker'>";
        for (i = shapes.length; i--; ) {
            if (set.style === shapes[i])
                form += "<option selected='true'>" + shapes[i] + "</option>";
            else
                form += "<option>" + shapes[i] + "</option>";
        }
        form += "</select></td></tr><tr><td>Colour:</td><td><select size='6' id='_colour'>";
        for (i = colours.length; i--; ) {
            if (set.fillStyle === colours[i])
                form += "<option selected='true'>" + colours[i] + "</option>";
            else
                form += "<option>" + colours[i] + "</option>";
        }
        OU.___dataGraph_deleteColumn = false;
        form += "</select></td></tr></table>"
                + "<p><input type='submit' onclick='OU.___dataGraph_deleteColumn=true;' value='Delete Column'/></p>"
                + "<p style='text-align:right'><input type='submit' value='Done'/></p>";
        new OU.util.PopUpForm({
            container: this,
            formHTML: form,
            onSubmit: function() {
                self._editColumn(colNum);
            }
        });
    };
    OU.activity.DataGraph.prototype._editColumn = function(colNum) {
        var name = document.getElementById('_labelName').value,
                shape = document.getElementById('_marker').value,
                colour = document.getElementById('_colour').value,
                set = this.dataset.sets[colNum];
        if (OU.___dataGraph_deleteColumn) {
            if (window.confirm('Are you sure you want to delete this column?')) {
                this.dataset.sets.splice(colNum, 1);
            }
        }
        else {
            set.name = name;
            set.style = shape;
            set.fillStyle = colour;
        }
        this.dataEntry();
    };
    OU.activity.DataGraph.prototype.deleteRow = function(rowIndex) {
        var i, set;
        if (window.confirm('Are you sure you want to delete this row?')) {
            for (i = this.dataset.sets.length; i--; ) {
                set = this.dataset.sets[i];
                set.points.splice(rowIndex, 1);
            }
        }
        this.dataEntry();
    };
    OU.activity.DataGraph.prototype.addFnColumn = function() {
        var self = this;
        new OU.util.FunctionEditor({
            container: this,
            onSubmit: function(s) {
                self._addFnColumn(s);
            }
        });
    };
    OU.activity.DataGraph.prototype._addFnColumn = function(fnStr) {
        if (fnStr && fnStr !== '') {
            var i, j, val, vars, newColIdx, ds = this.dataset, numRows = this.countRows();
            // Add an empty column
            ds.sets.push({
                name: fnStr,
                function: fnStr,
                fillStyle: '#666',
                strokeStyle: '#444',
                style: 'rect',
                points: []
            });
            newColIdx = ds.sets.length - 1;
            // populate the column using the given function
            for (i = numRows; i--; ) {
                try {
                    vars = [];
                    for (j = ds.sets.length; j--; )
                        vars[j] = ds.sets[j].points[i];
                    val = this.evalFn(fnStr, vars);
                    if (this.isNum(val))
                        ds.sets[newColIdx].points[i] = val;
                }
                catch (e) {
                    console.log("Error populating column: " + e.message);
                }
            }
        }
        this.dataEntry();
    };

    OU.activity.DataGraph.prototype.addColumn = function() {
        this.dataset.sets.push({
            name: 'Y' + this.dataset.sets.length,
            fillStyle: '#666',
            strokeStyle: '#444',
            style: 'rect',
            points: []
        });
        this.dataEntry();
    };
    /**
     * @class Dataset contains information for each dataset as an object
     * @param {object} params
     * @private
     */
    OU.activity.DataGraph.prototype.Dataset = function(params) {
        this.name = params.name || 'y=?';
        this.firstControlPoints = [];
        this.secondControlPoints = [];
        this.container = params.container;
        this.graphTitle = params.graphTitle || "Graph title";
        this.xAxisTitle = params.xAxisTitle || "X axis";
        this.yAxisTitle = params.yAxisTitle || "Y axis";
        this.sets = params.sets || [[]];
        this.gradX = 6.5;
        this.numPlotSets = 0;

        /**
         * determine if dataset is valid  -  ie can it be graphed?
         * @returns boolean - true if at least 2 valid points, else false
         */
        OU.activity.DataGraph.prototype.Dataset.prototype.isValid = function() {
            var x, xval, set, yval, validPoints = 0;
            if (!this.sets[this.container.xColumn] || !this.sets[this.container.xColumn].points)
                return false;
            for (x = this.sets[this.container.xColumn].points.length; x--; ) {
                xval = this.sets[this.container.xColumn].points[x];
                if (this.container.isNum(xval)) {
                    for (set = 0; set < this.sets.length; set++) {
                        if (set !== this.container.xColumn && this.sets[set].plot) {
                            yval = this.sets[set].points[x];
                            if (this.container.isNum(yval)) {
                                validPoints++;
                                if (validPoints > 1)
                                    return true;
                            }
                        }
                    }
                }
            }
            return false;
        };
        /**
         * Saves the dataset to localStorage
         * @param {boolean} overwrite - overwrite any existing data if true
         */
        OU.activity.DataGraph.prototype.Dataset.prototype.save = function(overwrite) {
            var setsStr = OU.LocalStorage.load('OU.datagraph_.datasets'), sets, i, found = false;
            if (setsStr) {
                sets = JSON.parse(setsStr);
                for (i = sets.length; i--; ) {
                    if (sets[i] === this.name)
                        found = true;
                }
                if (!found)
                    sets.push(this.name);
            }
            else {
                sets = [this.name];
            }
            OU.LocalStorage.save('OU.datagraph_.datasets', JSON.stringify(sets));
            if (overwrite || !OU.LocalStorage.load('OU.datagraph.' + this.name)) {
                OU.LocalStorage.save('OU.datagraph.' + this.name, JSON.stringify({
                    sets: this.sets,
                    graphTitle: this.graphTitle,
                    xAxisTitle: this.xAxisTitle,
                    yAxisTitle: this.yAxisTitle
                }));
            }
        };
        OU.activity.DataGraph.prototype.Dataset.prototype.import = function(csvArray, incTitles) {
            var i;
            for (i = 0; i < csvArray.length; i++) {
                this.sets[i] = {
                    name: 'Y' + this.sets.length,
                    fillStyle: '#666',
                    strokeStyle: '#444',
                    style: 'rect'
                };
                this.sets[i].points = csvArray[i];
                if (incTitles) {
                    this.sets[i].name = csvArray[i][0];
                    this.sets[i].points.splice(0, 1);
                }
            }
            this.calc();
            this.save();
        };
        /**
         * Loads the dataset from localStorage
         */
        OU.activity.DataGraph.prototype.Dataset.prototype.load = function() {
            var savedData, savedStr = OU.LocalStorage.load('OU.datagraph.' + this.name);
            if (savedStr && savedStr !== '') {
                savedData = JSON.parse(savedStr);
                this.sets = savedData.sets;
                this.graphTitle = savedData.graphTitle;
                this.xAxisTitle = savedData.xAxisTitle;
                this.yAxisTitle = savedData.yAxisTitle;
                this.calc();
            }
        };
        OU.activity.DataGraph.prototype.Dataset.prototype.calc = function() {
            var i;
            this.numPlotSets = -1;
            for (i = 0; i < this.sets.length; i++) {
                if (!this.sets[i].style)
                    this.sets[i].style = 'rect';
                if (this.sets[i].plot === undefined)
                    this.sets[i].plot = true;
                if (this.sets[i].plot)
                    this.numPlotSets++;
            }
            if (this.isValid()) {

                this.reorder();
                this.calcRanges();
            }
            this.gradX = this.x1 + (this.x2 - this.x1) / 2;
        };
        /**
         * calculate the axis ranges
         */
        OU.activity.DataGraph.prototype.Dataset.prototype.calcRanges = function() {
            var i, v, min = Number.MAX_VALUE, max = -Number.MAX_VALUE, sumv,
                    sets, set = this.sets[this.container.xColumn];

            // calc X axis
            for (i = set.points.length; i--; ) {
                v = set.points[i];
                if (v > max)
                    max = v;
                if (v < min)
                    min = v;
            }
            this.xticks = this.container.roundTicks(max - min);
            this.x1 = this.xticks * (((min / this.xticks) | 0));
            this.x2 = this.xticks * ((1 + (max / this.xticks)) | 0);

            if (this.x1 === min)
                this.x1 = this.x1 - this.xticks;
            if (this.x2 === max)
                this.x2 = this.x2 + this.xticks;

            // calc Y axis
            min = Number.MAX_VALUE;
            max = -Number.MAX_VALUE;
            for (sets = 0; sets < this.sets.length; sets++) {
                if (sets !== this.container.xColumn) {
                    set = this.sets[sets];
                    if (set.plot !== false) {
                        sumv = 0;
                        for (i = set.points.length; i--; ) {
                            v = set.points[i];
                            if (v > max)
                                max = v;
                            if (v < min)
                                min = v;
                            sumv = sumv + v;
                        }
                        set.meanY = sumv / set.points.length;
                    }
                }
            }
            this.yticks = this.container.roundTicks(max - min);

            this.y1 = this.yticks * (((min / this.yticks) | 0));
            this.y2 = this.yticks * (((max / this.yticks)) | 0);
            if (this.y1 > min)
                this.y1 = min;
            if (this.y2 < max)
                this.y2 = max;

            this.deltas();
            // calc visible scale
            if (this.dx !== 0) {
                this.container.xScale = (this.container.w - this.container.PAD * 2) / this.dx;
            }
            else {
                this.container.xScale = 10;
                console.log('dx is zero!!!')
            }
            if (this.dy !== 0) {
                this.container.yScale = ((this.container.h - OU.controlHeight) - this.container.PAD * 2) / this.dy;
            }
            else {
                this.container.yScale = 10;
                console.log('dy is zero!!!')
            }

        };
        /**
         * Calculates the R Squared value for the straight line fit.
         */
        OU.activity.DataGraph.prototype.Dataset.prototype.calcR2 = function(params) {
            var i, a = params.a, b = params.b,
                    yset = this.sets[params.ysetIdx],
                    xset = this.sets[this.container.xColumn],
                    x, y, fny, sstot = 0, sserr = 0;

            for (i = yset.points.length; i--; ) {
                x = xset.points[i];
                y = yset.points[i];
                fny = a * x + b; // straight line Y value at x
                sstot = sstot + Math.pow(y - yset.meanY, 2); // sum of squares to Mean Y
                sserr = sserr + Math.pow(y - fny, 2); // sum of squares to line
            }
            if (sstot === 0)
                return 1;
            return 1 - (sserr / sstot);
        };
        /**
         * calculate deltas
         */
        OU.activity.DataGraph.prototype.Dataset.prototype.deltas = function() {
            this.dx = Math.abs(this.x2 - this.x1);
            this.dy = Math.abs(this.y2 - this.y1);
        };
        /**
         * sorts the data according to X values & recalcs control points for bezier curve
         */
        OU.activity.DataGraph.prototype.Dataset.prototype.reorder = function() {
            var i, j, pointSets = [], pset,
                    numX = this.sets[this.container.xColumn].points.length,
                    numY = this.sets.length,
                    clickable = this.container.modelLayer.events.clickable;
            clickable.length = 0;
            clickable.push(this); // push this dataset into the clickable array of the model layer

            for (i = numX; i--; ) {
                pset = {
                    x: this.sets[this.container.xColumn].points[i],
                    yvals: []
                };
                for (j = 1; j < numY; j++) {
                    pset.yvals[j] = this.sets[j].points[i];
                }
                pointSets.push(pset);
            }
            pointSets.sort(function(a, b) {
                return a.x - b.x;
            });
            this.sets[this.container.xColumn].points = [];
            for (i = 0; i < numX; i++) {
                this.sets[this.container.xColumn].points[i] = pointSets[i].x;
            }

            for (j = 1; j < numY; j++) {
                this.sets[j].points = [];
                for (i = 0; i < numX; i++) {
                    this.sets[j].points[i] = pointSets[i].yvals[j];
                }
            }

            this.getCurveControlPoints();

        };
        /*
         * Handles touch/mouse events. Moves the gradient X point.
         */
        OU.activity.DataGraph.prototype.Dataset.prototype.isHit = function(x, y, state) {
            var gp = this.container, dx, xS = gp.xScale;
            if (state) {
                if (gp.lineFit || (gp.curveFit && this.numKnots < 2)) {
                    this.gradX = (x - gp.PAD) / xS + this.x1;
                    gp.render();
                }
                else {
                    if (this.lastHitX) {
                        dx = this.lastHitX - x;
                        this.gradX -= dx / xS;
                        this.lastHitX = x;

                        //this.gradX = ((x - gp.PAD) / xS) + this.x1; // moves gradX to the point touched
                        if (this.gradX < this.x1)
                            this.gradX = this.x1;
                        if (this.gradX > this.x2)
                            this.gradX = this.x2;
                        gp.render();
                    }
                    else {
                        this.lastHitX = x;
                    }
                }
            }
            else {
                this.lastHitX = null;
            }
        };

        /**
         * Calculates the point on a bezier curve at a time interval (t).
         * Also calculates the gradient of the line at that point using differentiate of the bezier curve equation.
         * @param defines the parameters and points to use in calculation:
         * <ul>
         * <li> {int} yset - the index of the Y values to calculate against</li>
         * <li> {int} a - the index of the left hand point</li>
         * <li> {float} t - time between point a & b (0 to 1)
         * </ul>
         * @returns {object} co-ordinate and gradient characteristics ie. {x,y,theta,a,b}
         */
        OU.activity.DataGraph.prototype.Dataset.prototype.pointAtTbezierCurve = function(params) {
            var xpoints = this.sets[this.container.xColumn].points,
                    ypoints = this.sets[params.yset].points,
                    a = {x: xpoints[params.a], y: ypoints[params.a]}, // starting point
            b = {x: xpoints[params.a + 1], y: ypoints[params.a + 1]}, // ending point
            c = this.firstControlPoints[params.yset][params.a], // first control point
                    d = this.secondControlPoints[params.yset][params.a], // second control point
                    t = params.t,
                    x, y, dx, dy, a, b, theta;

            x = a.x * Math.pow((1 - t), 3)
                    + 3 * Math.pow((1 - t), 2) * t * c.x
                    + 3 * (1 - t) * Math.pow(t, 2) * d.x
                    + Math.pow(t, 3) * b.x;
            y = a.y * Math.pow((1 - t), 3)
                    + 3 * Math.pow((1 - t), 2) * t * c.y
                    + 3 * (1 - t) * Math.pow(t, 2) * d.y
                    + Math.pow(t, 3) * b.y;

            // Calculate dx and dy using differentiation.
            dx = -3 * a.x * Math.pow((1 - t), 2)
                    + 3 * Math.pow((1 - t), 2) * c.x
                    - 6 * t * (1 - t) * c.x
                    - 3 * Math.pow(t, 2) * d.x
                    + 6 * t * (1 - t) * d.x
                    + 3 * b.x * Math.pow(t, 2);
            dy = -3 * a.y * Math.pow((1 - t), 2)
                    + 3 * Math.pow((1 - t), 2) * c.y
                    - 6 * t * (1 - t) * c.y
                    - 3 * Math.pow(t, 2) * d.y
                    + 6 * t * (1 - t) * d.y
                    + 3 * b.y * Math.pow(t, 2);


            // calculate A and B in the equation 'y=ax+b' that defines the gradient line
            if (dx === 0) { // this is incredibly unlikely, but let's catch divide-by-zero error
                a = Number.MAX_VALUE;
                b = 0;
                theta = 0;
            }
            else {
                a = dy / dx;
                b = y - (a * x);
                // calculate the gradient angle
                theta = Math.atan(a);
            }

            // return the point and gradient characteristics of the curve at that point
            return {x: x, y: y, a: a, b: b, theta: theta};
        };
        /**
         * Renders bezier curves for each Y value set
         * @param ctx - context to render to
         */
        OU.activity.DataGraph.prototype.Dataset.prototype.bezierCurve = function(ctx) {
            var i, j, n, set, bH = OU.controlHeight, t, pointAtT, lastx, x, y, c1x, c1y, c2x, c2y, ex, ey,
                    xset = this.sets[this.container.xColumn],
                    xS = this.container.xScale,
                    yS = this.container.yScale,
                    PAD = this.container.PAD,
                    yOff = this.container.h - bH - PAD;
            for (i = 0; i < this.sets.length; i++) {
                if (i !== this.container.xColumn) {
                    set = this.sets[i];
                    if (set.plot !== false) {
                        n = set.points.length;
                        lastx = x = xset.points[0];
                        y = set.points[0];
                        x = PAD + (x - this.x1) * xS;
                        y = yOff - (y - this.y1) * yS;
                        ctx.beginPath();
                        ctx.moveTo(x, y);
                        ctx.strokeStyle = set.fillStyle || "#444";

                        for (j = 0; j < n - 1; j++) {
                            x = parseFloat(xset.points[j + 1]);
                            y = parseFloat(set.points[j + 1]);
                            if (this.container.isNum(x) && this.container.isNum(y)) {
                                c1x = this.firstControlPoints[i][j].x;
                                c1y = this.firstControlPoints[i][j].y;
                                c2x = this.secondControlPoints[i][j].x;
                                c2y = this.secondControlPoints[i][j].y;

                                c1x = PAD + (c1x - this.x1) * xS;
                                c2x = PAD + (c2x - this.x1) * xS;
                                c1y = yOff - (c1y - this.y1) * yS;
                                c2y = yOff - (c2y - this.y1) * yS;

                                ctx.bezierCurveTo(c1x, c1y, c2x, c2y, PAD + (x - this.x1) * xS, yOff - (y - this.y1) * yS);
                            }
                            else {
                                console.log('not number: j=' + j + ' x=' + x + ' y=' + y);
                            }
                            if (this.gradX !== undefined && 1 === this.numPlotSets && lastx <= this.gradX && x >= this.gradX) {
                                t = (this.gradX - lastx) / (x - lastx);
                                pointAtT = this.pointAtTbezierCurve({
                                    a: j,
                                    t: t,
                                    yset: i
                                });
                            }
                            lastx = x;
                        }
                        ctx.stroke();
                        // if we have a gradient, then display it.
                        if (pointAtT) {
                            ctx.beginPath();
                            ctx.save();
                            ctx.rect(PAD / 2, yOff - ((this.y2 - this.y1) * yS) - PAD / 2, (this.x2 - this.x1) * xS + PAD, ((this.y2 - this.y1) * yS) + PAD);
                            ctx.clip();
                            ctx.beginPath();
                            ctx.setLineDash && ctx.setLineDash([5, 5]);
                            ctx.strokeStyle = '#444';
                            // draw the rectangle to point from th axis
                            ctx.moveTo(PAD, yOff - (pointAtT.y - this.y1) * yS);
                            ctx.lineTo(PAD + (pointAtT.x - this.x1) * xS, yOff - (pointAtT.y - this.y1) * yS);
                            ctx.lineTo(PAD + (pointAtT.x - this.x1) * xS, yOff);
                            ctx.lineWidth = 0.5;
                            ctx.stroke();
                            ctx.beginPath();
                            // draw the gradient line by calculating the points at the start and end of the range.
                            ex = this.x1;
                            ey = (ex * pointAtT.a + pointAtT.b)
                            ctx.moveTo(PAD + (ex - this.x1) * xS, yOff - (ey - this.y1) * yS);
                            ex = this.x2;
                            ey = (ex * pointAtT.a + pointAtT.b)
                            ctx.lineTo(PAD + (ex - this.x1) * xS, yOff - (ey - this.y1) * yS);
                            ctx.lineWidth = 2;
                            ctx.stroke();
                            ctx.restore();

                            this.container.gradientInfo('<strong><u>Gradient details</u></strong><br/>'
                                    + 'x: ' + this.container.niceNumberFormatx10(pointAtT.x) + '<br/>'
                                    + 'y: ' + this.container.niceNumberFormatx10(pointAtT.y) + '<br/>'
                                    + 'gradient: ' + this.container.niceNumberFormatx10(pointAtT.a) + '<br/>'
                                    + 'intercept: ' + this.container.niceNumberFormatx10(pointAtT.b) + '<br/>');
                            // + 'gradient angle: ' + this.container.niceNumberFormatx10(pointAtT.theta * 180 / Math.PI) + '&deg;<br/>');
                        }
                    }
                }
            }
        };

        /**
         * Generate control points for bezier curve between data points.
         */
        OU.activity.DataGraph.prototype.Dataset.prototype.getCurveControlPoints = function() {
            var knots = [], i, x, y, n, rhs, setIdx;

            for (setIdx = 0; setIdx < this.sets.length; setIdx++) {
                if (setIdx !== this.container.xColumn) {
                    knots = [];
                    for (i = 0; i < this.sets[0].points.length; i++) {
                        x = this.sets[this.container.xColumn].points[i];
                        y = this.sets[setIdx].points[i];
                        if (x !== undefined && y !== undefined) {
                            knots.push({
                                X: x,
                                Y: y
                            });
                        }
                    }
                    this.numKnots = n = knots.length - 1;
                    if (n < 2) {
                        // not enough points to create curved line fit
                        return;
                    }

                    // Calculate first Bezier control points
                    // Right hand side vector
                    rhs = [];

                    // Set right hand side X values
                    for (i = 1; i < n - 1; ++i)
                        rhs[i] = 4 * knots[i].X + 2 * knots[i + 1].X;
                    rhs[0] = knots[0].X + 2 * knots[1].X;
                    rhs[n - 1] = (8 * knots[n - 1].X + knots[n].X) / 2.0;

                    // Get first control points X-values
                    x = this.getFirstControlPoints(rhs);

                    // Set right hand side Y values
                    for (i = 1; i < n - 1; ++i)
                        rhs[i] = 4 * knots[i].Y + 2 * knots[i + 1].Y;
                    rhs[0] = knots[0].Y + 2 * knots[1].Y;
                    rhs[n - 1] = (8 * knots[n - 1].Y + knots[n].Y) / 2.0;
                    // Get first control points Y-values
                    y = this.getFirstControlPoints(rhs);

                    // Fill output arrays.
                    this.firstControlPoints[setIdx] = [];
                    this.secondControlPoints[setIdx] = [];
                    for (i = 0; i < n; ++i) {
                        // First control point
                        this.firstControlPoints[setIdx][i] = {x: x[i], y: y[i]};
                        // Second control point
                        if (i < n - 1) {
                            this.secondControlPoints[setIdx][i] = {
                                x: 2 * knots[i + 1].X - x[i + 1],
                                y: 2 * knots[i + 1].Y - y[i + 1]
                            };
                        }
                        else {
                            this.secondControlPoints[setIdx][i] = {
                                x: (knots[n].X + x[n - 1]) / 2,
                                y: (knots[n].Y + y[n - 1]) / 2
                            };
                        }
                    }
                }
            }
        };

        /**
         *  Solves a tridiagonal system for one of coordinates (x or y)
         *  of first Bezier control points.
         */
        OU.activity.DataGraph.prototype.Dataset.prototype.getFirstControlPoints = function(rhs) {
            var i, n = rhs.length,
                    x = [], // Solution vector.
                    tmp = [], // Temp workspace.
                    b = 2.0;
            x[0] = rhs[0] / b;
            for (i = 1; i < n; i++) { // Decomposition and forward substitution.
                tmp[i] = 1 / b;
                b = (i < n - 1 ? 4.0 : 3.5) - tmp[i];
                x[i] = (rhs[i] - x[i - 1]) / b;
            }
            for (i = 1; i < n; i++)
                x[n - i - 1] -= tmp[n - i] * x[n - i]; // Backsubstitution.

            return x;
        };

        this.calc();
    };


    /**
     * Overides base method for accessible view to display a text version of the activity when canvas is not supported
     */
    OU.activity.DataGraph.prototype.accessibleView = function() {
        var i, h = '<div id="equation3DAccessible">';
        h += '<h1>' + this.data.title + '</h1>';
        h += '<p>' + this.data.description + '</p>';
        for (i = 0; i < this.data.dataset.length; i++) {
            var equation = this.data.dataset[i];
            h += '<h2>' + equation.name + '</h2><p>' + equation.description + '</p>';
        }
        h += '</div>';
        document.body.innerHTML = '';
        var accessible = document.createElement('div');
        accessible.innerHTML = h;
        document.body.appendChild(accessible);
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.DataGraph, OU.util.Activity);
