/**
 * @fileOverview  RelativeScales Class
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.HtmlBox');
OU.require('OU.util.Layer');
OU.require('OU.util.Slider');
/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.RelativeScales = function ( data, instance, controller ) {
    OU.activity.RelativeScales.prototype.canvasView = function () {
        this.bgLayer = new OU.util.Layer({
            container:this
        });
        this.sliderLayer = new OU.util.Layer({
            container:this,
            hasEvents:true
        });
        this.min = this.data.min;
        this.max = this.data.max;
        // Ensure the controller maintains layout for popup activities
        if (!this.controller.section.options)
            this.controller.section.options = {};
        this.controller.section.options.maintainLayout = true;
        this.initControls();
        this.resize();
    };
    OU.activity.RelativeScales.prototype.resize = function () {
        OU.activity.RelativeScales.superClass_.resize.call(this); // call the parent class resize
        this.bgLayer.resize();
        this.sliderLayer.resize();
        if (this.infoBox) {
            this.infoBox.resize({
                x:this.x + 10,
                y:this.y + this.h * .05,
                w:this.w / 2 - 20,
                h:'auto'
            });
        }
        var i, item,
            bH = this.h * .1 - 10,
            iH = (this.h * .85) / this.numItems;
        iH = iH > 80 ? 80 : iH;
        bH = bH < 20 ? 20 : bH > OU.controlHeight ? OU.controlHeight : bH;
        for (i = this.numItems; i--;) {
            item = this.items[i];
            item.resize({
                x:this.w * .6,
                y:this.h * .05 + iH * i,
                w:this.w * .4 - 10,
                h:iH
            });
        }
        this.checkButton.resize({
            x:this.w * .65,
            y:this.h - bH - 10,
            w:this.w * .3 - 10,
            h:bH
        });
        this.render();
    };
    OU.activity.RelativeScales.prototype.render = function () {
        var i, ctx = this.bgLayer.context, iH = this.items[0].h,
            x = this.w * .6,
            y = this.h * .05,
            w = this.w * .4 - 10,
            h = iH * this.numItems + iH / 2,
            f = new OU.font({
                size:16*OU.dpr,
                weight:'bold'
            });
        this.bgLayer.clear();
        this.sliderLayer.clear();
        ctx.save();
        ctx.fillStyle = '#eef';
        ctx.strokeStyle = '#444';
        ctx.lineWidth = 2;
        ctx.roundRect(x, y, w, h, 15);
        ctx.fill();
        ctx.beginPath();
        ctx.moveTo(x + w * .1, y + 15);
        ctx.lineTo(x + w * .1, y + h - 15);
        ctx.moveTo(x + w * .9, y + 15);
        ctx.lineTo(x + w * .9, y + h - 15);
        ctx.stroke();
        ctx.fillStyle = '#444';
        ctx.font = f.str();
        ctx.textAlign = 'left';
        ctx.fillText(this.data.minLabel, x + 10, y - 14*OU.dpr);
        ctx.textAlign = 'right';
        ctx.fillText(this.data.maxLabel, x + w - 10, y - 14*OU.dpr);
        ctx.restore();
        for (i = this.numItems; i--;) {
            this.items[i].render(this.allCorrect);
        }
        if (!this.indicateCorrect)
            this.checkButton.render();
        if (this.nextButton)
            this.nextButton.render();
    };
    OU.activity.RelativeScales.prototype.initControls = function () {
        var i, self = this;
        this.items = [];
        this.numItems = this.data.items.length;
        for (i = 0; i < this.numItems; i++) {
            this.items.push(new this.Item({
                id:i,
                data:this.data.items[i],
                parent:this
            }));
        }
        this.checkButton = new OU.util.Button({
            layer:this.sliderLayer,
            txt:"Check your answer",
            x:this.x + this.w * .6,
            y:this.y + this.h * .9 + 10,
            w:this.w * .4 - 10,
            h:this.h * .1 - 20,
            onClick:function () {
                self.checkAnswer();
            }
        });
        this.infoBox = new OU.util.HtmlBox({
            container:this,
            html:this.data.instructions,
            x:this.x + 10,
            y:this.y + this.h * .05,
            w:this.w / 2 - 20,
            h:'auto'
        });
    };
    OU.activity.RelativeScales.prototype.checkAnswer = function ( p ) {
        var i, passed = true, item;
        for (i = 0; i < this.numItems; i++) {
            item = this.items[i];
            if (Math.abs(item.targetScale - item.currentScale) > item.tolerance) {
                passed = false;
            }
        }
        this.indicateCorrect = true;
        if (this.displayState===undefined || this.displayState!=passed) {
            this.displayState = passed;
            if (this.openActivity)
                this.controller.removeActivity(this.openActivity);
            if (passed) {
                this.allCorrect = true;
                this.correctStageStep = 0;
                this.correctStage();
            }
            else {
                this.infoBox.div.innerHTML = this.data.incorrectFeedback;
                this.allCorrect = false;
            }
        }
        this.render();
    };
    OU.activity.RelativeScales.prototype.correctStage = function () {
        var hHeight, w = this.w / 6, self = this;
        this.infoBox.div.innerHTML = this.data.correctFeedback[this.correctStageStep];
        hHeight = this.infoBox.y + this.infoBox.div.clientHeight + 10;
        if (this.correctStageStep==this.data.correctFeedback.length - 1) { // Add onComplete Button
            if (this.nextButton)
                this.nextButton.remove();
            this.nextButton = new OU.util.Button({
                layer:this.sliderLayer,
                txt:'Next',
                x:this.w / 2 - w - 10,
                y:hHeight,
                w:w,
                h:40,
                onClick:function () {
                    self.controller.overrideSection({
                        activities:[self.data.onCompleteActivity]
                    });
                }

            });
        }
        else { // Add next step button
            this.nextButton = new OU.util.Button({
                layer:this.sliderLayer,
                txt:'Next',
                x:this.w / 2 - w - 10,
                y:hHeight,
                w:w,
                h:40,
                onClick:function () {
                    self.correctStageStep++;
                    self.correctStage();
                    self.render();
                }
            });
        }
    };
    /**
     * @class
     */
    OU.activity.RelativeScales.prototype.Item = function ( p ) {
        var self = this, savedPos;
        this.parent = p.parent;
        this.id = p.id;
        this.activity = p.data.activity;
        this.activity.dims = {
            w:0.5
        };
        this.targetScale = p.data.scale;
        this.currentScale = p.parent.min;
        this.tolerance = p.data.tolerance;
        this.name = p.data.name;
        this.x = this.y = 0;
        this.w = 100;
        this.h = 60;
        OU.activity.RelativeScales.prototype.Item.prototype.resize = function ( p ) {
            this.x = p.x || this.x;
            this.y = p.y || this.y;
            this.w = p.w || this.w;
            this.h = p.h || this.h;
            this.tH = this.h * .5 > 80 ? 80 : this.h * .5;
            this.sH = this.h * .5 > 80 ? 80 : this.h * .5;
            this.slider.resize({
                x:this.x,
                y:this.y + this.h-this.sH,
                w:this.w,
                h:this.sH
            });
            this.loadButton.resize({
                x:this.x - this.parent.w * .1,
                y:this.y + this.tH,
                w:this.parent.w * .1,
                h:this.sH
            });
        };
        OU.activity.RelativeScales.prototype.Item.prototype.render = function ( disableSlider ) {
            var ctx = this.parent.sliderLayer.context, glyph;
            if (this.parent.indicateCorrect) {
                glyph = {
                    type:'tickBox',
                    align:'left',
                    x:this.x + this.w - this.tH / 2 - 10,
                    y:this.y + this.h - this.tH / 2 - 10,
                    w:this.tH / 2,
                    h:this.tH / 2
                };
                if (Math.abs(this.targetScale - this.currentScale) > this.tolerance) {
                    glyph.type = 'crossBox';
                }
            }

            if (disableSlider)
                this.slider.disabled = true;
            else
                this.slider.disabled = false;
            this.slider.render((this.currentScale - this.parent.min) / (this.parent.max - this.parent.min));
            new OU.util.DynText({
                context:ctx,
                txt:this.name,
                colour:'#444',
                fontWeight:'bold',
                x:this.x,
                y:this.y + this.tH / 2,
                w:this.w,
                h:this.tH *.65,
                align:'center',
                background:{
                    clear:true
                }
            });
            if (!this.parent.allCorrect)
                this.loadButton.render();
            if (glyph)
                ctx.glyph(glyph);
        };
        OU.activity.RelativeScales.prototype.Item.prototype.setScale = function ( v, me ) {
            var p = me.parent;
            me.currentScale = p.min + v * (p.max - p.min);
            //            me.slider.render(v);
            me.render();
            OU.LocalStorage.save("OU.relativeScales." + p.instance + "." + me.id, v);
            if (p.indicateCorrect)
                p.checkAnswer();
        };
        OU.activity.RelativeScales.prototype.Item.prototype.loadActivity = function () {
            var p = this.parent;
            if (p.allCorrect)
                return;
            p.infoBox.div.innerHTML = '';
            if (p.openActivity)
                p.controller.removeActivity(p.openActivity);
            p.openActivity = p.controller.addActivity(this.activity, true);
        };
        savedPos = OU.LocalStorage.load("OU.relativeScales." + this.parent.instance + "." + this.id) || 0.5;
        this.currentScale = this.parent.min + savedPos * (this.parent.max - this.parent.min);
        this.slider = new OU.util.Slider({
            sliderPos:savedPos,
            w:100,
            h:40,
            container:this.parent,
            drawContainer:false,
            callback:this.setScale,
            callbackParam:this,
            context:this.parent.sliderLayer.context
        });
        this.parent.sliderLayer.events.clickable.push(this.slider);
        this.loadButton = new OU.util.PrevButton({
            layer:this.parent.sliderLayer,
            txt:'',
            x:this.x - this.parent.w * .1,
            y:this.y,
            w:this.parent.w * .1,
            h:this.h*OU.dpr,
            onClick:function () {
                self.loadActivity();
            }
        });
        this.loadButton.glyph.align = 'right';
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.RelativeScales, OU.util.Activity);
