/**
 * @fileOverview Events provides Event handling for Mouse & Touch events
 *
 * Provides a uniform interface to the events generated by various platforms, ie. iPad, Desktop browsers, mobiles, etc.
 *
 *
 * @auther Nigel Clarke <nigel.clarke@pentahedra.com>
 */
/**
 * @class Events handler - typically attached to a canvas (OU.util.Layer) object.
 * The event handler sets up an array ("clickable") which contains instances of "object" classes which react to events.
 * Each clickable object is required to have a isHit() function, which will be called each time an event happens.
 * It is up to each clickable object to determine if it should react to the event. ie. if it has been "hit" if it needs to act.
 *
 * @param {object} settings contains:
 * <ul>
 *  <li><strong>{dom object} target: </strong> (required) DOM target, eg. canvas</li>
 *  <li><strong>{function} moveLeft: </strong> callback on swipe left event</li>
 *  <li><strong>{function} moveRight: </strong> callback on swipe right event</li>
 *  <li><strong>{function} clickTouch: </strong> callback on mouseDown/touchDown</li>
 *  <li><strong>{function} keyPress: </strong> callback on key event</li>
 *  <li><strong>{function} pinch: </strong> callback on pinch action</li>
 *  <li><strong>{object} pinchMe: </strong> object to pass back the pinch callback function</li>
 * </ul>
 */
OU.util.Events = function ( settings ) {
    var self = this;
    this.x = 0;
    this.y = 0;
    this.pressed = false;
    this.touched = false;
    this.dragStart = 0;
    this.lastDrag = new Date().getTime();
    this.clickable = new OU.util.TypedArray();
    this.dragId = 0;
    this.dragTaken = false;
    this.swipeEnabled = true;
    this.doubleTap = false;
    this.lastTouch = {
        when:new Date().getTime(),
        x:0,
        y:0
    };
    if (settings.target===undefined) {
        alert('Events Object requires a target');
        return;
    }
    this.target = settings.target;
    this.target.events = this; // self reference held in the target object, so it is accessible from the event functions
    // Store action callback functions or set to do nothing if not set
    this.keyPressFn = settings.keyPress || function () {
    };
    this.moveRight = settings.moveRight || function () {
    };
    this.moveLeft = settings.moveLeft || function () {
    };
    this.pinch = settings.pinch;
    this.pinchMe = settings.pinchMe;

    if (settings.clickTouch===undefined)
        this.clickTouch = function () {};
    else
        this.clickTouch = settings.clickTouch;
    // init the event listeners
    settings.target.addEventListener("mousedown", this.mouseDown, false, 0, true);
    settings.target.addEventListener("dblclick", this.mouseDouble, false, 0, true);
    settings.target.addEventListener("mouseup", this.mouseUp, false, 0, true);
    settings.target.addEventListener("mouseout", this.mouseOut, false, 0, true);
    settings.target.addEventListener("mousemove", this.mouseMove, false, 0, true);
    settings.target.addEventListener("mousewheel", this.mouseWheel, false, 0, true);
    settings.target.addEventListener("touchstart", this.touchStart, false, 0, true);
    settings.target.addEventListener("touchend", this.touchUp, false, 0, true);
    settings.target.addEventListener("touchmove", this.touchMove, false, 0, true);
    settings.target.addEventListener("gestureend", this.touchGesture, false, 0, true);
    document.addEventListener("keydown", this.listener = function ( e ) {
        self.keyDown(e);
    }, false, 0, true);
};
/**
 * Removes the event handlers and empties the clickable array.
 * This would typically be ccalled by the Layer's remove function
 */
OU.util.Events.prototype.remove = function () {
    document.removeEventListener("keydown", this.listener);
    this.target.removeEventListener("gestureend", this.touchGesture, false);
    this.target.removeEventListener("touchmove", this.touchMove, false);
    this.target.removeEventListener("touchend", this.touchUp, false);
    this.target.removeEventListener("touchstart", this.touchStart, false);
    this.target.removeEventListener("mousewheel", this.mouseWheel, false);
    this.target.removeEventListener("mousemove", this.mouseMove, false);
    this.target.removeEventListener("mouseout", this.mouseOut, false);
    this.target.removeEventListener("mouseup", this.mouseUp, false);
    this.target.removeEventListener("dblclick", this.mouseDouble, false);
    this.target.removeEventListener("mousedown", this.mouseDown, false);
    for (var i = this.clickable.length; i--;)
        delete this.clickable[i];
    this.clickable.length = 0;
    //delete this.targets.events;
    this.target.events = null;
};
/**
 * Switches off the swipe action, ie. disables the left and right callbacks
 */
OU.util.Events.prototype.disableSwipe = function () {
    this.swipeEnabled = false;
    this.dragStart = 0;
};
/**
 * Switches on the swipe action, ie. enables the left and right callbacks
 */
OU.util.Events.prototype.enableSwipe = function () {
    this.swipeEnabled = true;
};
/**
 * Key down event handler, combines all platforms' variables into a single value.
 * Calls left or right callbacks if arrow keys pressed.
 * Sends keypress details to keypress callback if present.
 * @private
 */
OU.util.Events.prototype.keyDown = function ( e ) {
    var k = null;
    if (e._dealtWith)
        return;
    if (window.event)
        k = window.event.keyCode;
    else if (e)
        k = e.which || e.keyCode;
    if (k==39) {
        if (this.moveRight)
            this.moveRight();
    }
    if (k==37) {
        if (this.moveLeft)
            this.moveLeft();
    }
    if (this.keyPressFn)
        this.keyPressFn(k);
};
/**
 * Mouse Double Click event handler.
 * Also calls the pinch callback function.
 * @private
 */
OU.util.Events.prototype.mouseDouble = function ( e ) {
    this.events.doubleTap = true;
    this.events.trigger();

    if(this.events.pinch)
        this.events.pinch(this.width/10, 0, this.events.x, this.events.y, 0, 0, this.events.pinchMe);

};
/**
 * MouseDown event handler. Combines all platforms' variables for event co-ordinates
 * @private
 */
OU.util.Events.prototype.mouseDown = function ( e ) {
    e.preventDefault();
    e = OU.combineMouseTouch(e);
    this.events.x = e.pageX - this.offsetLeft;
    this.events.y = e.pageY - this.offsetTop;
    this.events.dragStart = this.events.x; // set Drag start
    this.events.dragId++;
    this.events.dragTaken = false;
    this.events.pressed = true;
    this.events.clickTouch();
    this.events.trigger("mouseDown");
};
/**
 * MouseUp event handler. Combines all platforms' variables for event co-ordinates
 * @private
 */
OU.util.Events.prototype.mouseOut = OU.util.Events.prototype.mouseUp = function ( e ) {
    e.preventDefault();
    e = OU.combineMouseTouch(e);
    this.events.x = e.pageX - this.offsetLeft;
    this.events.y = e.pageY - this.offsetTop;
    this.events.pressed = false;
    if (this.events.dragStart > 0) {
        this.events.swipeX(this.events.dragStart, this.events.x);
    }
    this.events.trigger("mouseUp");
};
/**
 * Checks to see if a "swipe" action has happened, and calls the relevant callback function.
 * @private
 */
OU.util.Events.prototype.swipeX = function ( start, end ) {
    if (!this.swipeEnabled)
        return;
    if (end > start + 10) {
        this.moveLeft();
        this.lastDrag = new Date().getTime();
    }
    if (end < start - 10) {
        this.moveRight();
        this.lastDrag = new Date().getTime();
    }
};
/**
 * MouseMove event handler - combine platforms' variables and pass event on to trigger()
 * @private
 */
OU.util.Events.prototype.mouseMove = function ( e ) {
    this.events.x = e.pageX - this.offsetLeft;
    this.events.y = e.pageY - this.offsetTop;
    this.events.trigger("mouseMove");
};
/**
 * MouseWheel event handler - call left/right callback functions if mouse wheel up or down
 * @private
 */
OU.util.Events.prototype.mouseWheel = function ( e ) { // limited support cross browser/device
    e.preventDefault();
    if (e.wheelDelta > 0)
        this.events.moveLeft();
    if (e.wheelDelta < 0)
        this.events.moveRight();
};
/**
 * TouchStart event handler - Combines all platforms' variable and handles the event
 * @private
 */
OU.util.Events.prototype.touchStart = function ( e ) {
    var x, y, now = new Date().getTime(), i, lT = this.events.lastTouch;
    e.preventDefault();
    e = OU.combineMouseTouch(e);
    if (e.touches && e.touches.length > 1) {
        this.events.startTouches = [];
        for (i = 0; i < e.touches.length; i++)
            this.events.startTouches[i] = {
                pageX:e.touches[i].pageX,
                pageY:e.touches[i].pageY
            };
        return;
    }
    x = this.events.x = e.pageX - this.offsetLeft;
    y = this.events.y = e.pageY - this.offsetTop;
    this.events.dragStart = this.events.x; // set Drag start
    this.events.dragId++;
    this.events.dragTaken = false;
    this.events.touched = true;
    this.events.clickTouch();
    if (now - lT.when < 500 && lT.x - x > -20 && lT.x - x < 20 && lT.y - y > -20 && lT.y - y < 20)
        this.events.doubleTap = true;
    this.events.lastTouch = {
        when:now,
        x:x,
        y:y
    };
    this.events.trigger();
};
/**
 * TouchUp event handler - Combines all platforms' variable and handles the event
 * @private
 */
OU.util.Events.prototype.touchUp = function ( e ) {
    e.preventDefault();
    e = OU.combineMouseTouch(e);
    this.events.touched = false;
    if (e.touches && e.touches.length > 1) {
        this.events.processTouches(e.touches, this);
        return;
    }
    this.events.x = e.pageX - this.offsetLeft;
    this.events.y = e.pageY - this.offsetTop;
    this.events.trigger();
};
/**
 * TouchMove event handler - Combines all platforms' variable and handles the event
 * @private
 */
OU.util.Events.prototype.touchMove = function ( e ) {
    var now = new Date().getTime();
    e.preventDefault();
    e = OU.combineMouseTouch(e);
    if (e.touches && e.touches.length > 1) {
        this.events.processTouches(e.touches, this);
        return;
    }
    this.events.x = e.pageX - this.offsetLeft;
    this.events.y = e.pageY - this.offsetTop;
    if (now - this.events.lastDrag > 1000) {
        this.events.swipeX(this.events.dragStart, this.events.x);
        this.events.touched = true;
    }
    this.events.trigger();
};
/**
 * ProcessTouches event handler - Processes multi-touch events. Combines all platforms' variable and handles the event
 * @private
 */
OU.util.Events.prototype.processTouches = function ( latest, c ) {
    if (!this.startTouches)
        return;
    var i, start = this.startTouches,
    sdX = start[1].pageX - start[0].pageX,
    sdY = start[1].pageY - start[0].pageY,
    startDelta = Math.sqrt(sdX * sdX + sdY * sdY),
    edX = latest[1].pageX - latest[0].pageX,
    edY = latest[1].pageY - latest[0].pageY,
    endDelta = Math.sqrt(edX * edX + edY * edY),
    cX = latest[1].pageX - edX / 2 - c.offsetLeft,
    cY = latest[1].pageY - edY / 2 - c.offsetTop,
    dX = sdX==0 ? 0 : edX / sdX,
    dY = sdY==0 ? 0 : edY / sdY;
    if (startDelta!=endDelta) { //default to pinch //TODO determine multi touch type (ie. 2 finger swipe, etc.)
        this.startTouches.length = 0;
        for (i = 0; i < latest.length; i++) {
            this.startTouches[i] = {
                pageX:latest[i].pageX,
                pageY:latest[i].pageY
            };
        }
        if(this.pinch)
            this.pinch(endDelta, startDelta, cX, cY, dX, dY, this.pinchMe);
    }
};
/**
 * TouchGesture event handler - handle gesture events.
 * //TODO re-visit this, as it didn't used to work in ePub - may now be working
 * @private
 */
OU.util.Events.prototype.touchGesture = function ( e ) {
    //    alert('GESTURE');
    };
/**
 * Handler all events - Cycle through the clickable array and call each one's isHit() function
 * @private
 */
OU.util.Events.prototype.trigger = function () {
    var i;
    if (this.clickable.length > 0) {
        for (i = this.clickable.length; i--;) { // process in reverse, so object render last (ie. on top) are checked first
            if (this.clickable[ i ]) {// check this, as array could have been emptied mid loop
                if (this.clickable[ i ].isHit(this.x, this.y, this.touched || this.pressed, this)) {
                    this.touched = this.pressed = false;
                }
            }
        }
    }
};
/**
 * Return the Touched/Pressed state
 * @returns {boolean} state
 */
OU.util.Events.prototype.active = function () {
    return this.touched || this.pressed;
};
/**
 * Switch off the Touched/Pressed state
 */
OU.util.Events.prototype.flush = function () {
    this.touched = this.pressed = false;
};