/**
 * @fileOverview Instruction - pop up message window showing user instructions
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.DynText');
OU.require('OU.util.Layer');

/**
  * @class Instruction - pop up message window showing user instructions.
  * Message times out or goes away when hit - after which can be re-opened using an "info icon" button
  * @param {object} params - Options:
  * <ul>
  * <li><string>{object} container:</strong> A reference to the calling object, typically an OU.util.Activity</li>
  * <li><string>{string} message:</strong> The instructions to display</li>
  * <li><string>{int} duration:</strong> (optional) length of timeout in milliseconds, defaults to 6000 (6 seconds)</li>
  * <li><string>{int} x:</strong> X co-ordinate of text box</li>
  * <li><string>{int} y:</strong> Y co-ordinate of text box</li>
  * <li><string>{int} w:</strong> Width of text box</li>
  * <li><string>{int} h:</strong> Height of text box</li>
  * <li><string>{object} shrinkArea:</strong> optional position of the info icon, defaults to topLeft of the calling activity</li>
  * <li><string>{string} bgCol:</strong> optional background colour of the message box, defaults to OU.theme.instructionsBg</li>
  * <li><string>{string} colour:</strong> optional text colour of the message, defaults to OU.theme.instructionsCol</li>
  * </ul>
 */
OU.util.Instruction = function(params) {
    var container=params.container;
    this.container = container;
    this.message = params.message || 'Oops! Undefined instruction';
    this.duration = params.duration || 6000; // default 6 seconds
    this.kill=false;
    this.params = params;
    this.onClose = params.onClose || function() {};
    this.dims = {
        x: params.x || container.w*.3,
        y: params.y || container.h*.4,
        w: params.w || container.w*.4,
        h: params.h || container.h*.2
    };
    if(OU.baseController.headerDiv) {
        this.shrinkArea = {x: OU.baseController.w-OU.controlHeight*1.5,y:OU.baseController.headerHeight-OU.controlHeight,w: OU.controlHeight,h:OU.controlHeight};
        this.dims = {
          x: OU.controlHeight,
          y: OU.baseController.headerHeight+10,
          w: OU.baseController.w-2*OU.controlHeight,
          h: OU.baseController.h*.25
        };
    }
    else {
        this.shrinkArea = params.shrinkArea || {x: OU.controlHeight,y:OU.controlHeight,w: OU.controlHeight,h:OU.controlHeight};
    }

    this.bgCol = params.bgCol || OU.theme.instructionsBg;
    this.colour = params.colour || OU.theme.instructionsCol;
    /**
     * @private
     */
    OU.util.Instruction.prototype.init=function() {
        var self=this,ctx,pt,sc=this.container,bH=OU.controlHeight;

        this.started = new Date().getTime();
        this.layer = new OU.util.Layer({
            container: sc,
            hasEvents:true,
            zIndex: OU.POP_UP_LEVEL
        });
        this.layer.events.clickable.push(this);
        ctx=this.layer.context;
        this.layer.opacity(1);

        if(this.params.point) {
            pt=this.params.point;

            ctx.beginPath();
            ctx.moveTo(pt.x,pt.y);
            ctx.fillStyle=this.bgCol;
            if(pt.x<=sc.w/2)  // top left
                this.dims.x=pt.x-this.dims.w/2<0?0:pt.x-this.dims.w/2;
            else  //top right
                this.dims.x=pt.x+this.dims.w/2>sc.w?sc.w-this.dims.w:pt.x-this.dims.w/2;
            if(pt.y<=sc.h/2) {
                this.dims.y=pt.y+sc.h*.2;
                /*
                ctx.bezierCurveTo(pt.x,this.dims.y+this.dims.h,
                    this.dims.x,pt.y,
                    this.dims.x,this.dims.y+this.dims.h/5);
                ctx.lineTo(this.dims.x+this.dims.w,this.dims.y+this.dims.h/5);
                ctx.bezierCurveTo(this.dims.x+this.dims.w,pt.y,
                    pt.x,this.dims.y+this.dims.h,
                    pt.x,pt.y);
                //*/
                ctx.lineTo(this.dims.x+this.dims.w/2-15,this.dims.y);
                ctx.lineTo(this.dims.x+this.dims.w/2+15,this.dims.y);
                ctx.fill();
                /*
                ctx.beginPath();
                ctx.moveTo(pt.x,pt.y);
                ctx.lineTo(pt.x+20,pt.y+20);
                ctx.lineTo(pt.x-20,pt.y+20);
                ctx.closePath();
                ctx.fill();
                //*/
            }
            else {
                this.dims.y=pt.y-sc.h*.4;
                /*
                ctx.bezierCurveTo(pt.x,this.dims.y,
                    this.dims.x,pt.y,
                    this.dims.x,this.dims.y+this.dims.h*.8);
                ctx.lineTo(this.dims.x+this.dims.w,this.dims.y+this.dims.h*.8);
                ctx.bezierCurveTo(this.dims.x+this.dims.w,pt.y,
                    pt.x,this.dims.y,
                    pt.x,pt.y);
                    //*/
                ctx.lineTo(this.dims.x+this.dims.w/2-15,this.dims.y+this.dims.h);
                ctx.lineTo(this.dims.x+this.dims.w/2+15,this.dims.y+this.dims.h);
                ctx.fill();
                /*
                ctx.beginPath();
                ctx.moveTo(pt.x,pt.y);
                ctx.lineTo(pt.x+20,pt.y-20);
                ctx.lineTo(pt.x-20,pt.y-20);
                ctx.closePath();
                ctx.fill();
                //*/
            }
        }
        new OU.util.DynText({
            txt: this.message,
            context: ctx,
            x: this.dims.x,
            y: this.dims.y,
            w: this.dims.w,
            h: this.dims.h,
            align:'left',
            padding:bH*2,
            background: {
                borderCol: '#296E8F',
                col: '#ccc',
                radius: 5
            },
            colour: '#000'
        });

        this.infoButton = new OU.util.InfoButton({
                txt:"",
                x: this.dims.x+10,
                y: this.dims.y+10,
                w:this.shrinkArea.w,
                h:this.shrinkArea.h,
                layer:this.layer,
                background:{
                    clear:false
                },
                onClick:function () {
                    self.shrink();
                }
            });
        this.layer.context.closeIcon({
            x: this.dims.x+this.dims.w-45,
            y: this.dims.y+10,
            r: 30
        });

    };
    /**
     * @private
     */
    OU.util.Instruction.prototype.isHit=function(x,y,evState) {
        if(evState)
            this.shrink();
    };
    /**
     * @private
     */
    OU.util.Instruction.prototype.grow=function() {
        if(this.infoButton) {
            this.infoButton.remove();
            this.infoButton=undefined;
        }
        this.layer.events.flush();
        this.layer.remove();
        this.kill=false;
        this.init();
    };
    /**
     * @private
     */
    OU.util.Instruction.prototype.shrink=function() {
        var self=this;
        this.layer.events.flush();
        if(this.infoButton)
            this.infoButton.remove();
        this.layer.clear();
        this.layer.resize({
            x:this.dims.x+10,
            y:this.dims.y+10,
            w:this.shrinkArea.w,
            h:this.shrinkArea.h
        });
        this.infoButton = new OU.util.InfoButton({
                txt:"",
                x:0,
                y:0,
                w:this.shrinkArea.w,
                h:this.shrinkArea.h,
                layer:this.layer,
                onClick:function () {
                    self.grow();
                }
            });
        this.layer.slideTo(this.shrinkArea);
        this.onClose();
    };
    OU.util.Instruction.prototype.remove=function() {
        if(this.infoButton)
            this.infoButton.remove();
        this.layer.remove();
    };
    this.init();
};