/**
 * @fileOverview Navigation buttons objects - places a left and right button within the defined button area (the bar)
 * 
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */
/**
 * @class NavButtons - Navigation bar, with left & right buttons
 * 
 * @param {object} params - options:
 * <ul>
 * <li><strong>{canvas.context} context:</strong> (required) context of the layer to put the buttons on</li>
 * <li><strong>{OU.util.Layer} layer:</strong> (required) reference to the Layer the buttons are going on</li>
 * <li><strong>{int} x:</strong> X coordinate of the bar</li>
 * <li><strong>{int} y:</strong> Y coordinate of the bar</li>
 * <li><strong>{int} w:</strong> Width of the bar</li>
 * <li><strong>{int} h:</strong> Height of the bar</li>
 * <li><strong>{double} bw:</strong> button Width in pixels, defaults to match Height if undefined as bar is expected to have letterbox style dims</li>
 * <li><strong>{double} pad:</strong> Button padding in pixels - gap around each button</li>
 * <li><strong>{function} leftFunc:</strong> callback function for left button</li>
 * <li><strong>{function} rightFunc:</strong> callback function for right button</li>
 * <li><strong>{boolean} leftOn:</strong> Makes left Button visible or not, default to true</li>
 * <li><strong>{boolean} rightOn:</strong> Makes right Button visible or not, default to true</li>
 * </ul>
 */
OU.util.NavButtons = function(params) {
    this.x = params.x || 0;
    this.y = params.y || 0;
    this.w = params.w || 400;
    this.h = params.h || 60;
    this.bw = params.bw || this.h;
    this.pad = params.pad || 0;
    this.context = params.context;
    this.layer = params.layer;
    this.leftFunc = params.leftFunc || function() {
    };
    this.rightFunc = params.rightFunc || function() {
    };
    this.leftOn = params.leftOn === undefined ? true : params.leftOn;
    this.rightOn = params.rightOn === undefined ? true : params.rightOn;

    /**
     * Render the buttons
     */
    OU.util.NavButtons.prototype.render = function() {
        this.context.clearRect(this.x, this.y, this.w, this.h);

        if (this.leftOn) {
            if (!this.leftButton) {
                this.leftButton = new OU.util.PrevButton({
                    x: this.x + this.pad,
                    y: this.y + this.pad,
                    w: this.bw - (2 * this.pad),
                    h: this.h - (2 * this.pad),
                    tabIndex: 10,
                    layer: this.layer,
                    onClick: this.leftFunc
                });
            }
            else {
                this.leftButton.render();
            }
        }
        else {
            if (this.leftButton) {
                this.leftButton.remove();
                this.leftButton = null;
            }
        }
        if (this.rightOn) {
            if (!this.rightButton) {
                this.rightButton = new OU.util.NextButton({
                    x: this.x + this.w - this.bw + this.pad,
                    y: this.y + this.pad,
                    w: this.bw - (2 * this.pad),
                    h: this.h - (2 * this.pad),
                    tabIndex: 20,
                    layer: this.layer,
                    onClick: this.rightFunc
                });
            }
            else {
                this.rightButton.render();
            }
        }
        else {
            if (this.rightButton) {
                this.rightButton.remove();
                this.rightButton = null;
            }
        }
    };
    /**
     * Move (resize) the entire "bar", and buttons accordingly
     * 
     * @param {object} params - new dims:
     * <ul>
     * <li><strong>{int} x:</strong> X coordinate of the bar</li>
     * <li><strong>{int} y:</strong> Y coordinate of the bar</li>
     * <li><strong>{int} w:</strong> Width of the bar</li>
     * <li><strong>{int} h:</strong> Height of the bar</li>
     * </ul>
     */
    OU.util.NavButtons.prototype.move = function(params) {
        this.x = params.x || this.x;
        this.y = params.y || this.y;
        this.w = params.w || this.w;
        this.h = params.h || this.h;

        if (this.leftButton) {
            this.leftButton.resize({
                x: this.x + this.pad,
                y: this.y + this.pad,
                w: this.bw - (2 * this.pad),
                h: this.h - (2 * this.pad)
            });
        }
        if (this.rightButton) {
            this.rightButton.resize({
                x: this.x + this.w - this.bw + this.pad,
                y: this.y + this.pad,
                w: this.bw - (2 * this.pad),
                h: this.h - (2 * this.pad)
            });
        }
    };
};