/**
 * @fileOverview PopUpInfo utility
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Div');

/**
 *  @class PopUpInfo - A very simple pop up box that displays a title and contents, with a close icon.
 *  Calls a callback function when it is closed.
 *  Takes ownership of the entire screen and fills with an invisible backdrop layer, so that clicking outside the pop-up
 *  acts like hitting the close button.
 *  
 *  @param {object} params - options:
 *  <ul>
 *  <li><strong>{object} container:</strong> (required) The calling object, typically an OU.util.Acitivity </li>
 *  <li><strong>{function} onClose:</strong> Callback function to run when the pop-up is closed</li>
 *  <li><strong>{string} txt:</strong> Contents to put in the pop-up, can be HTML content</li>
 *  <li><strong>{int} zIndex:</strong> Optional zIndex for the Pop-up</li>
 *  <li><strong>{int} x:</strong> X co-ordinate of the pop-up</li>
 *  <li><strong>{int} y:</strong> Y co-ordinate of the pop-up</li>
 *  <li><strong>{int} w:</strong> Width of the pop-up</li>
 *  <li><strong>{int} h:</strong> Height of the pop-up</li>
 *  </ul>
 * 
 */
OU.util.PopUpInfo = function ( params ) {
    var self=this;
    this.container = params.container || this;
    this.zIndex=(params.zIndex || OU.POP_UP_LEVEL) - 1;
    this.layer = new OU.util.Layer({
        container:this.container,
        hasEvents:true,
        id:'popup',
        zIndex: this.zIndex
    });
    this.context = this.layer.context;
    this.events = this.layer.events;
    this.events.clickable.push(this);
    this.onClose = params.onClose;
    this.txt = params.txt || '';
    this.w = params.w || this.context.canvas.width * .8;
    this.h = params.h || this.context.canvas.height * .8;
    this.x = params.x || this.context.canvas.width * .1;
    this.y = params.y || this.context.canvas.height * .1;
    this.render();
    window.addEventListener("keydown", this.listener=function(e) { self.keyPress(e); }, false,0,true);
};
/**
 * Function to handle key presses, specifically to close the pop-up if ESCAPE is hit
 * @private
 */
OU.util.PopUpInfo.prototype.keyPress = function (e) {
    var k=null;
    if (window.event)
        k= window.event.keyCode;
    else if (e)
        k= e.which || e.keyCode;
    if(k==27)
        this.remove();
};
/**
 * Renders the frame and contents
 * @private
 */
OU.util.PopUpInfo.prototype.render = function () {
    var g, ctx = this.context,cW = this.w*.025;
    cW = cW>15?15:cW;
    ctx.save();
    ctx.fillStyle = 'rgba(0,0,0,0.3)';
    ctx.fillRect(this.x + 10, this.y + 10, this.w, this.h);
    ctx.restore();
    g = ctx.createLinearGradient(this.x, this.y, this.x, this.y + this.h);
    //g = ctx.createRadialGradient(this.x + (this.w / 2), this.y + this.h, 0, this.x + (this.w / 2), this.y + this.h, this.h);
    g.addColorStop(0, '#afdae7');
    g.addColorStop(0.41, '#bedfe9');
    g.addColorStop(1, '#98ccdb');
    //color-stop(0%,#93cede), color-stop(41%,#75bdd1), color-stop(100%,#49a5bf)
    ctx.background({
        RGB:'235,244,251',
        borderCol:'#666',
        gradient:g
    }, {
        x:this.x,
        y:this.y,
        w:this.w,
        h:this.h
    });
    ctx.save();
    ctx.strokeStyle = "rgba(255,255,255,0.3)";
    ctx.beginPath();
    ctx.moveTo(this.x + this.w, this.y + 1);
    ctx.lineTo(this.x + 1, this.y);
    ctx.lineTo(this.x + 1, this.y + this.h - 1);
    ctx.stroke();
    ctx.strokeStyle = "rgba(0,0,0,0.3)";
    ctx.beginPath();
    ctx.moveTo(this.x + 1, this.y + this.h - 1);
    ctx.lineTo(this.x + this.w - 1, this.y + this.h - 1);
    ctx.lineTo(this.x + this.w - 1, this.y + 1);
    ctx.stroke();
    
    ctx.closeIcon({
        x:this.x + this.w - (cW * 2),
        y:this.y+cW/2,
        r:cW
    });
    this.htmlDiv = new OU.util.Div({
        container:this.container,
        innerHTML:this.txt,
        zIndex: this.zIndex+1,
        style: 'padding: 0 20px;',
        overflow:'auto',
        x: (this.container.x+this.x) | 0,
        y: (this.container.y+this.y + cW*2) | 0,
        w: (this.w -20) | 0,
        h: (this.h - cW*2-10) |0
    });   
};
/**
 * Closes the pop-up, calling the callback function if defined
 */
OU.util.PopUpInfo.prototype.remove=OU.util.PopUpInfo.prototype.close = function () {
    if (this.events.clickable.length > 0) {
        this.events.clickable.length = 0;
        this.htmlDiv.remove();
        this.layer.remove();
        this.htmlDiv=this.layer=null;
        window.removeEventListener("keydown", this.listener);
        
        if (this.onClose)
            this.onClose();
    }
};
/**
 * Resizes the pop-up
 * @param {object} params - new dims:
 * <ul>
 *  <li><strong>{int} x:</strong> X co-ordinate of the pop-up</li>
 *  <li><strong>{int} y:</strong> Y co-ordinate of the pop-up</li>
 *  <li><strong>{int} w:</strong> Width of the pop-up</li>
 *  <li><strong>{int} h:</strong> Height of the pop-up</li>
 *  </ul>
 */
OU.util.PopUpInfo.prototype.resize = function ( params ) {
    var cW = this.w*.025;
    cW = cW>15?15:cW;
    this.w = params.w || this.context.canvas.width * .8;
    this.h = params.h || this.context.canvas.height * .8;
    this.x = params.x || this.context.canvas.width * .1;
    this.y = params.y || this.context.canvas.height * .1;
    this.htmlDiv.resize({
        x:this.container.x + this.x,
        y:this.container.y + this.y + cW*2,
        w:this.w -20,
        h:this.h - cW*2-10
    });
};
/**
 * isHit function, which is applied to the backdrop layer, to catch clicks/taps outside the pop-up.
 * Clicks outside the pop-up, trigger the close action
 * @private
 */
OU.util.PopUpInfo.prototype.isHit = function ( x, y ) {
    if (this.events.pressed || this.events.touched) {
        if (x < this.x + this.w * 0.9 || x > this.x + this.w)
            return;
        if (y < this.y || y > this.y + this.h)
            return;
        this.remove();
    }
};