/**
 * @fileOverview ScrubBar - provides a scrub bar at the specified location & size
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */
/**
 * @class - ScrubBar - it's like a slider, but with defined position points.
 * 
 *  Note, render does not handle animation internally, so relies on the calling activity to render durung each render cycle.
 *  We should probably improve this, to make it more self contained.
 * 
 * @param {object} params - options:
 * <ul>
 * <li><strong>{canvas.context} context:</strong> (required) Context of the canvas/layer to render to</li>
 * <li><strong>{int} x:</strong> X co-ordinate of the baar</li>
 * <li><strong>{int} y:</strong> Y co-ordinate of the bar</li>
 * <li><strong>{int} w:</strong> Width of the bar</li>
 * <li><strong>{int} h:</strong> Height of the bar</li>
 * <li><strong>{int} points:</strong> Number of points on the bar</li>
 * <li><strong>{function} newPointFunc:</strong> callback function to call when the scrubbar moves to a new point</li>
 * <li><strong>{object} callbackParam:</strong> (optional) data to pass back to the newPointFunc</li>
 * <li><strong>{int} startPos:</strong> Point to start on, defaults to 0</li>
 * <li><strong>{canvas.context.background} background:</strong> (optional) background object to render</li>
 * <li><strong>{string[]} labels:</strong> (optional) labels for points</li>
 * <li><strong>{double} labelHeight:</strong> (optional) height of Labels above scrub bar</li>
 * </ul>
 */
OU.util.ScrubBar = function(params) {
    this.params = params;
    this.x = params.x || 0;
    this.y = params.y || 0;
    this.w = params.w || 0;
    this.h = params.h || 0;
    this.context = params.context;
    this.sliderX = 0;
    this.sliderWidth = 20;
    this.sliderSpeed = 4;
    this.numPoints = params.points;
    this.currentPoint = 0;
    this.highlightColour = params.highlightColour || 'rgba(41,171,225,1)';
    if (params.startPos !== undefined)
        this.currentPoint = params.startPos;
    this.labels = params.labels;
    this.labelHeight = params.labelHeight || 0;

    this.newPointFunc = params.newPointFunc || function() {
    };
    this.callbackParam = params.callbackParam;
    this.resize(params);
};
/**
 * Moves the slider to a new point programmatically
 * @param {int} position - the new position to move to
 */
OU.util.ScrubBar.prototype.newTarget = function(position) {
    this.currentPoint = position;
    this.targetX = position * (this.pointWidth);
};
/**
 * Resize the scrubbar
 * @param {object} dims - Dimensions:
 * <ul>
 * <li><strong>{int} x:</strong> X co-ordinate of the baar</li>
 * <li><strong>{int} y:</strong> Y co-ordinate of the bar</li>
 * <li><strong>{int} w:</strong> Width of the bar</li>
 * <li><strong>{int} h:</strong> Height of the bar</li>
 * </ul>
 */
OU.util.ScrubBar.prototype.resize = function(dims) {
    this.context.clearRect(this.x, this.y, this.w, this.h);

    this.x = dims.x || this.x || 0;
    this.y = dims.y || this.y || 0;
    this.w = dims.w || this.w || 0;
    this.h = dims.h || this.h || 0;

    this.lineStart = this.x + (this.sliderWidth / 2);
    this.lineEnd = this.x + this.w - (this.sliderWidth / 2);
    this.pointWidth = (this.lineEnd - this.lineStart) / (this.numPoints - 1);
    this.targetX = this.currentPoint * (this.pointWidth);

    this.render();
};
/**
 * Renders the slider.
 */
OU.util.ScrubBar.prototype.render = function() {
    var i, scrubY = this.y + this.labelHeight, scrubH = this.h - this.labelHeight, bg, labelFontSize = 100, button;

    if (this.params.background)
        this.context.background(this.params.background, this);

    // move the slider towards the target X
    if (this.sliderX !== this.targetX)
        this.sliderX += (this.targetX - this.sliderX) / this.sliderSpeed;

    this.context.strokeStyle = "#666";
    // render backdrop slider points
    this.context.beginPath();
    this.context.moveTo(this.lineStart, scrubY + (scrubH / 2));
    this.context.lineTo(this.lineEnd, scrubY + (scrubH / 2));
    this.context.stroke();
    if (this.labels) {
        for (i = this.numPoints; i--; ) {
            if (this.labels[i]) {
                button = new OU.util.DynText({
                    context: this.context,
                    txt: this.labels[i],
                    x: this.lineStart + (this.pointWidth * i) - this.pointWidth / 2 + 5,
                    y: this.y + this.labelHeight * .1,
                    w: this.pointWidth - 10,
                    h: this.labelHeight * .8,
                    fontSize: labelFontSize,
                    measureOnly: true,
                    padding: this.pointWidth * .1, // (function(v,a,b) {return v<a?a:(v>b?b:v);})(this.pointWidth*.1,0,10),
                    autoWidth: true,
                    autoCentre: true
                });
                if (button.font.size < labelFontSize)
                    labelFontSize = button.font.size;
            }
        }
    }
    for (i = this.numPoints; i--; ) {
        this.context.beginPath();
        this.context.moveTo(this.lineStart + (this.pointWidth * i), scrubY + (3 * scrubH / 8));
        this.context.lineTo(this.lineStart + (this.pointWidth * i), scrubY + (5 * scrubH / 8));
        this.context.stroke();
        if (this.labels && this.labels[i]) {
            if (this.currentPoint === i) {
                bg = {
                    radius: this.labelHeight / 4,
                    col: this.highlightColour
                };
            }
            else {
                bg = null;
            }
            new OU.util.DynText({
                context: this.context,
                txt: this.labels[i],
                x: this.lineStart + (this.pointWidth * i) - this.pointWidth / 2 + 5,
                y: this.y + this.labelHeight * .1,
                w: this.pointWidth - 10,
                h: this.labelHeight * .8,
                fontSize: labelFontSize,
                background: bg,
                padding: this.pointWidth * .1, // (function(v,a,b) {return v<a?a:(v>b?b:v);})(this.pointWidth*.1,0,10),
                autoWidth: true,
                autoCentre: true
            });
        }
    }

    // render slider
    this.context.roundRect(this.x + this.sliderX, scrubY + (scrubH / 4), this.sliderWidth, scrubH / 2, this.sliderWidth / 2);
    this.context.fillStyle = "rgba( 144, 144, 144, 0.5 )";
    this.context.strokeStyle = "#000";
    this.context.fill();
    this.context.stroke();
};
/**
 * isHit function called by the Layer's event handler
 * @param {int} x - X co-ordinate of event
 * @param {int} y - Y co-ordinate of event
 * @param {boolean} pressed - state of the event, ie. true if mouse down or touch down, etc.
 * @private
 */
OU.util.ScrubBar.prototype.isHit = function(x, y, pressed) {
    if (pressed && x > this.x && x < this.x + this.w && y > this.y && y < this.y + this.h) {
        this.sliderX = x - this.x - (this.sliderWidth / 2);
        var newPoint = (((x - this.lineStart) / this.pointWidth) + 0.5) | 0;
        if (newPoint !== this.currentPoint) {
            this.currentPoint = newPoint;
            this.newTarget(newPoint);
            this.newPointFunc(newPoint, this.callbackParam); // call the new point handler to let the container know we've moved'
        }
    }

};


