/**
 * @fileOverview Tile Viewer - provides a view of an image that has been tiled into smaller images
 *               with the option of zooming in
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require("OU.util.Vector2D");
OU.require("OU.util.Blur");
OU.require('OU.util.PopUpForm');

OU.VM_RULER_WIDTH = 30;

/**
 * @class TileViewer, renders a "zoomified" image using small tiles (typically 256x256 pixels/tile)
 * Used in activities: Microscope, Panorama, TileZoom, etc.
 *
 * @param {object} params - options:
 * <ul>
 * <li><strong>{object} container:</strong> (required) The calling object, typically an OU.util.Acitivity </li>
 * <li><strong>{canvas.context} context:</strong> (required) Context of the canvas/layer to render to</li>
 * <li><strong>{object} window:</strong> defines the dimensions with elements</li>
 * <ul>
 * <li><strong>{int} x:</strong> X co-ordinate</li>
 * <li><strong>{int} y:</strong> Y co-ordinate</li>
 * <li><strong>{int} w:</strong> Width</li>
 * <li><strong>{int} h:</strong> Height</li>
 * </ul>
 * <li><strong>{object} image:</strong> dimensions of the original image before being cut into tiles, with:</li>
 * <ul>
 * <li><strong>{int} w:</strong> Width</li>
 * <li><strong>{int} h:</strong> Height</li>
 * </ul>
 * <li><strong>{object} imageBase:</strong> a json structure that defines the images (see data.js format)</li>
 * <li><strong>{int} zoomLevels:</strong> (required) number of zoom levels in the zoomify files</li>
 * <li><strong>{boolean} useTileGroups:</strong> true if the Zoomify files are in TileGroup folders, defaults to true</li>
 * <li><strong>{int} tileSize:</strong> size of the tiles in pixels, defaults to 256</li>
 * <li><strong>{function} zoomCallback:</strong> Callback function, called with new zoom value, ie to set a slider in the calling activity</li>
 * <li><strong>{boolean} renderAngle:</strong> set true to show angle (to horizontal) on measuring tool, defaults to false</li>
 * <li><strong>{boolean} vectorAngles:</strong> set true to show the angle between 2 lines under measurements</li>
 * <li><strong>{string} rotationImage:</strong> file name/path of the graphic to indicate a rotation point for the VM. relative to the data directory </li>
 * <li><strong>{boolean} showHotspots:</strong> Set true to highlight the hotspot areas (default:false) </li>
 * <li><strong>{string} hotspotOutlineColour:</strong>  colour of the highlight if showing hotspots</li>
 * <li><strong>{boolean} requireDouble:</strong> if true, then a double tap/click is required to open a hotspot (default:false)</li>
 * <li><strong>{boolean} horizontalStitch:</strong> if true, then image is wrapped horizontally - ie for panoramas (default:false)</li>
 * <li><strong>{boolean} useGoogleMapsFileConvention:</strong> if true, uses Google tile format for files (deprecated - use Zoomify format instead)</li>
 * <li><strong>{function} markerCallback:</strong> callback function when marker/hotspot is hit</li>
 * <li><strong>{boolean} fitScreen:</strong> if true, then the initial view leaves no gaps, therefore some of the image may be offscreen (default:false)</li>
 * <li><strong>{object} crop:</strong> for use with google file format to crop unwanted parts of image (deprecated) </li>
 * <li><strong>{boolean} showRulers:</strong> set true to show rulers on top and left of window (default:false)</li>
 * <li><strong>{boolean} extendDrag:</strong> set true to allow the user to drag the view further off the content,default false</li>
 * </ul>
 *
 */
OU.util.TileViewer = function(params) {
    var NOTLOADED = 0, LOADING = 1, LOADED = 2;
    this.context = params.context;
    this.container = params.container;
    this.imageBase = params.imageBase || [];
    this.renderAngle = params.renderAngle || false;
    this.rotationImageFile = params.rotationImage;
    this.numberMarkers = params.numberMarkers;
    this.showHotspots = params.showHotspots || false;
    this.hotspotOutlineColour = params.hotspotOutlineColour;
    this.numImages = this.imageBase.length;
    this.requireDouble = params.requireDouble || false;
    this.vectorAngles = params.vectorAngles || true;
    this.micronThreshold = params.micronThreshold === undefined ? 1 : params.micronThreshold;
    this.numDecimals = params.numDecimals === undefined ? 2 : params.numDecimals;
    this.horizontalStitch = params.horizontalStitch || false;
    this.useGoogleMapsFileConvention = params.useGoogleMapsFileConvention || false;
    this.markerCallback = params.markerCallback || function() {
    };
    this.imageIdx = 0;
    this.window = params.window || {// viewing window on the canvas
        w: 1024,
        h: 768
    };
    this.fitScreen = params.fitScreen || false;
    this.v = {}; // viewport of the image (section of the image being displayed)
    this.crop = params.crop || {
        x: 0,
        y: 0
    }; // crop offsets
    this.dims = params.image || {}; // image dims
    this.useTileGroups = params.useTileGroups;
    this.tileSize = params.tileSize || 256;
    this.nZ = params.zoomLevels || 0; // total zoom levels
    this.z = 0; // current zoom level
    this.s = 0; // scale (0 <= s <=1) - % of total zoom range
    this.x = this.y = 0; // offsets
    this.measureMM = 0;
    this.gridAngle = 0;
    this.doRender = false;
    this.zoomCallback = params.zoomCallback || function() {
    };
    this.extendDrag = params.extendDrag;
    this.edgeSnap = !params.noEdgeSnap;
    this.hasDragged = false;
    this.inDrag = false;
    this.mouseDown = false;
    this.history = [];
    this.inertia = {
        x: 0,
        y: 0
    };
    this.lastTouch = {
        x: 0,
        y: 0
    };
    this.blurRadius = 0;
    this.brightenFactor = 1;
    this.showRulers = params.showRulers || false;
    this.showCrosshairs = params.showCrosshairs || false;
    this.centreReadout = params.centreReadout || false;
    this.gridColour = '#c00';
    this.rulerGuide = {
        out: false,
        x: 0,
        y: 0
    };
    this.dragRuler = false;
    this.dataDir = this.container.dataDir;

    this.uncroppedW = (this.dims.w + this.crop.x * 2);
    this.uncroppedH = (this.dims.h + this.crop.y * 2);

    if (params.thumbnail) {
        params.thumbnail.tileViewer = this;
        this.thumbnail = new this.Thumbnail(params.thumbnail);
    }

    /**
     * @private
     */
    OU.util.TileViewer.prototype.init = function() {
        var URLVars = OU.getUrlVars(), self = this,
                uncroppedW = this.uncroppedW,
                uncroppedH = this.uncroppedH;
        this.imageIdx = URLVars['s'] || 0;

        this.initImageBase();

        this.context.font = '18px ' + OU.theme.font;
        this.context.lineWidth = 2;
        this.context.strokeStyle = this.gridColour;
        this.context.fillStyle = this.gridColour;
        this.context.textAlign = 'center';
        this.renderCycle();

        // load rotation marker image
        this.rotationImageLoaded = false;
        if (this.rotationImageFile) {
            this.rotationImage = new Image();
            this.rotationImage.src = this.dataDir + this.rotationImageFile;
            this.rotationImage.onload = function() {
                self.rotationImageLoaded = true;
                self.doRender = true;
            };
        }
    };
    OU.util.TileViewer.prototype.remove = function() {
        this.removeMarkers();
    };
    OU.util.TileViewer.prototype.removeMarkers = function() {
        if (this.markers) { // remove the tababble markers
            for (var i = this.markers.length; i--; ) {
                this.markers[i].remove();
            }
        }
    };
    OU.util.TileViewer.prototype.newDims = function(newSize) {
        this.dims = newSize;
    };
    OU.util.TileViewer.prototype.initImageBase = function() {
        var i, z, zoomLevels = this.nZ,
                squareTileSize = this.tileSize * Math.pow(2, zoomLevels - 1),
                pX, pY, gmfn, fn, nX, nY, tileGroup, tGCount, fileType,
                URLVars = OU.getUrlVars(), x, y, fitWidth, fitHeight,
                p2m = this.imageBase[this.imageIdx].pixelsPerMM,
                uncroppedW, uncroppedH,
                rulerHeight = this.showRulers ? 30 : 0;

        this.measureStartX = this.measureStartY = this.measureEndX = this.measureEndY = null;

        if (!this.dims.w || this.dims.w < 1 || !this.dims.h || this.dims.h < 1) {// if image size not specified, calculate from the max zoom & tile size
            this.dims.w = this.dims.h = squareTileSize;
        }
        pX = this.propX = this.dims.w / squareTileSize;
        pY = this.propY = this.dims.h / squareTileSize;

        uncroppedW = this.uncroppedW = (this.dims.w + this.crop.x * 2);
        uncroppedH = this.uncroppedH = (this.dims.h + this.crop.y * 2);

        if (this.horizontalStitch) {
            this.viewAllScale = (this.window.h - rulerHeight) / uncroppedH;
        }
        else {
            fitWidth = (this.window.w - rulerHeight) / (uncroppedW);
            fitHeight = (this.window.h - rulerHeight) / (uncroppedH);
            if (this.fitScreen)
                this.viewAllScale = fitWidth > fitHeight ? fitWidth : fitHeight;
            else
                this.viewAllScale = fitWidth < fitHeight ? fitWidth : fitHeight;
        }
        this.scaleFactor = 1 - this.viewAllScale; // factor to normalise zoom scale to a range of 0 to 1
        if (URLVars['zoom']) {
            this.s = parseFloat(URLVars['zoom']);
            this.s = this.s < 0 ? 0 : (this.s > 1 ? 1 : this.s);
        }
        if (URLVars['x']) {
            this.x = this.window.w / 2 - parseFloat(URLVars['x']) * (p2m * (this.s * this.scaleFactor + this.viewAllScale));
        }
        else {
            if (this.showRulers) {
                this.x = this.horizontalStitch ? 0 : (OU.VM_RULER_WIDTH + this.window.w - uncroppedW * this.viewAllScale) / 2;
            }
            else {
                this.x = this.horizontalStitch ? 0 : (this.window.w - uncroppedW * this.viewAllScale) / 2;
            }
        }
        if (URLVars['y']) {
            this.y = this.window.h / 2 - parseFloat(URLVars['y']) * (p2m * (this.s * this.scaleFactor + this.viewAllScale));
        }
        else {
            if (this.showRulers) {
                this.y = (OU.VM_RULER_WIDTH + this.window.h - uncroppedH * this.viewAllScale) / 2;
            }
            else {
                this.y = (this.window.h - uncroppedH * this.viewAllScale) / 2;
            }
        }
        //set up array of image place holders
        for (i = this.numImages; i--; ) {
            nX = nY = 1;
            tileGroup = tGCount = 0;
            this.imageBase[i]._tiles = [];
            fileType = this.imageBase[i].fileType || '.jpg';
            for (z = 0; z < zoomLevels; z++) {
                this.imageBase[i]._tiles[z] = [];
                for (x = 0; x < (nX * pX) + 1; x++) {
                    this.imageBase[i]._tiles[z][x] = [];
                }
                nX = nX * 2;
            }
            nX = nY = 1;

            for (z = 0; z < zoomLevels; z++) {
                for (y = 0; y < (nY * pY); y++) {
                    for (x = 0; x < (nX * pX); x++) {
                        if (this.useGoogleMapsFileConvention) {
                            gmfn = this.getTileURL({
                                x: x,
                                y: y
                            }, z);
                            fn = this.dataDir + this.imageBase[i].fileBase + gmfn + fileType;
                        }
                        else if (this.useTileGroups) {
                            fn = this.dataDir + this.imageBase[i].fileBase + '/TileGroup' + tileGroup + '/' + z + '-' + x + '-' + y + fileType;
                            tGCount++;
                            if (tGCount > 255) {
                                tGCount = 0;
                                tileGroup++;
                            }
                        }
                        else {
                            fn = this.dataDir + this.imageBase[i].fileBase + z + '_' + x + '_' + y + fileType;
                        }
                        this.imageBase[i]._tiles[z][x][y] = {
                            file: fn,
                            loadState: NOTLOADED
                        };
                    }
                }
                nX = nX * 2;
                nY = nY * 2;
            }
            this.loadTile(this.imageBase[i]._tiles[0][0][0]);
        }

        //load initial image
        this._tiles = this.imageBase[this.imageIdx]._tiles;
        this.rough = this._tiles[0][0][0].image;

        if (this.dims.w > this.dims.h)
            this.roughSF = this.rough.width / (this.tileSize);
        else
            this.roughSF = this.rough.height / (this.tileSize);
        this.hotspots = this.imageBase[this.imageIdx].rotations || this.imageBase[this.imageIdx].layers || this.imageBase[this.imageIdx].hotspots || undefined;

        this.loadChondrules();

        this.removeMarkers();
        // Init the tababble markers
        if (this.hotspots && this.hotspots.length > 0) {
            this.markers = [];
            function createMarker(index, tileViewer) {
                return new tileViewer.Marker({
                    tileViewer: tileViewer,
                    getHotspot: function() {
                        return tileViewer.hotspots[index];
                    }
                });
            }
            ;
            for (var i = 0; i < this.hotspots.length; i++) {
                this.markers.push(createMarker(i, this));
            }
        }
    };
    /**
     * Tababble hotspot objects
     */
    OU.util.TileViewer.prototype.Marker = function(params) {
        var tileViewer = params.tileViewer;
        this.getHotspot = params.getHotspot;
        OU.util.TileViewer.prototype.Marker.prototype.focus = function() {
            this.getHotspot().hasFocus = true;
            tileViewer.doRender = true;
        };
        OU.util.TileViewer.prototype.Marker.prototype.blur = function() {
            this.getHotspot().hasFocus = false;
            tileViewer.doRender = true;
        };
        OU.util.TileViewer.prototype.Marker.prototype.hit = function() {
            tileViewer.markerCallback(this.getHotspot(), tileViewer.container);
        };
        OU.util.TileViewer.prototype.Marker.prototype.getX = function() {
            return this.getHotspot().x;
        };
        OU.base(this, params);
    };
    OU.inherits(OU.util.TileViewer.prototype.Marker, OU.util.Tabbable);

    /**
     * Change specimen image to a new one.
     * Requires the data to be set up to use multiple specimens
     */
    OU.util.TileViewer.prototype.changeDataset = function(imageBase) {
        this._tiles = null;
        this.imageBase = imageBase;
        this.numImages = this.imageBase.length;
        this.scale(0);
        this.initImageBase();
    };
    /**
     * Resizes the tileviewer and thumbnail if thumbnail used.
     * @param {object} params - new dimensions, all are optional:
     * <ul>
     * <li><strong>{double} w:</strong> new width</li>
     * <li><strong>{double} h:</strong> new height</li>
     * <li><strong>{object} thumbnail:</strong> new thumbnail dimensions, relative to the canvas it sits on, ie { x:1,y:1,w:256,h:256 }</li>
     * </ul>
     */
    OU.util.TileViewer.prototype.resize = function(params) {
        this.window.w = params.w || this.window.w;
        this.window.h = params.h || this.window.h;
        if (this.thumbnail && params.thumbnail)
            this.thumbnail.resize(params.thumbnail);
    };

    /**
     * Used if Google file format is used - which should be deprecated
     * @deprecated
     * @private
     */
    OU.util.TileViewer.prototype.getTileURL = function(a, b) { //converts tile x,y into keyhole string
        var c = Math.pow(2, b), d = a.x, e = a.y, f = "t", g;
        for (g = 0; g < b; g++) {
            c = c / 2;
            if (e < c) {
                if (d < c) {
                    f = f + "q";
                }
                else {
                    f = f + "r";
                    d -= c;
                }
            }
            else {
                if (d < c) {
                    f = f + "t";
                    e = e - c;
                }
                else {
                    f = f + "s";
                    d = d - c;
                    e = e - c;
                }
            }
        }
        return f;
    };
    /**
     * Change image set
     * @param {int} i - index of new image
     */
    OU.util.TileViewer.prototype.changeImage = function(i) {
        this.imageIdx = i < 0 ? 0 : i > this.numImages ? this.numImages : i;
        this._tiles = this.imageBase[this.imageIdx]._tiles;
        this.rough = this._tiles[0][0][0].image;
        this.doRender = true;
    };
    /**
     * @private
     * @param {object} t - the tile
     */
    OU.util.TileViewer.prototype.loadTile = function(t) {
        if (t.loadState === NOTLOADED) {
            var self = this;
            t.loadState = LOADING;
            t.image = new Image();
            t.image.src = t.file;
            t.image.onload = function() {
                t.loadState = LOADED;
                t.width = t.image.width;
                t.height = t.image.height;
                self.doRender = true;
            };
            t.image.onerror = function(e) {
            };
        }
    };
    /**
     * @private
     */
    OU.util.TileViewer.prototype.isHit = function(x, y, evState, ev) {
        var dX, dY, hitMarker = false,
                uncroppedW = this.uncroppedW,
                uncroppedH = this.uncroppedH,
                s = this.s * this.scaleFactor + this.viewAllScale,
                bH = 30,
                xPx, xO = this.x + this.crop.x * s,
                yPx, yO = this.y + this.crop.y * s,
                p2m = this.imageBase[this.imageIdx].pixelsPerMM;
        if (evState !== undefined)
            this.mouseDown = evState;
        if (ev.doubleTap) {
            ev.doubleTap = false;
            if (this.requireDouble) {
                this.markerHit(x, y, ev);
                return;
            }
        }
        if (this.mouseDown) {
            this.lastTouch.x = x;
            this.lastTouch.y = y;
            if (this.showRulers && !this.dragRuler) {
                if (this.rulerGuide.out) {
                    xPx = this.x + (this.rulerGuide.x * p2m) * s;
                    yPx = this.y + (this.rulerGuide.y * p2m) * s;
                    if (Math.abs(x - xPx) < 25 && Math.abs(y - yPx) < 25)
                        this.dragRuler = true;
                }
                else {
                    if (x < bH && y < bH) {
                        this.rulerGuide.out = true;
                        this.dragRuler = true;
                    }
                }
            }
            else {
                this.rulerGuide.x = ((x - this.x) / s) / p2m;
                this.rulerGuide.y = ((y - this.y) / s) / p2m;
                this.doRender = true;
            }
            if (!this.dragRuler) {
                if (this.inDrag) {
                    if (this.measureOn) {
                        this.renderMeasure = true;
                        this.measureEndX = (x - xO) / s;
                        this.measureEndY = (y - yO) / s;
                    }
                    else {
                        dX = x - this.dragStartX;
                        dY = y - this.dragStartY;
                        if (this.edgeSnap) {
                            this.x = (this.x + dX) % (uncroppedW * s);
                            this.y = (this.y + dY) % (uncroppedH * s);
                        }
                        else {
                            this.x = (this.x + dX);
                            this.y = (this.y + dY);
                        }
                        this.dragStartX = x;
                        this.dragStartY = y;
                    }
                    this.doRender = true;
                }
                else {
                    if (!this.requireDouble)
                        hitMarker = this.markerHit(x, y, ev);
                    if (!hitMarker) {
                        this.inertia = {
                            x: 0,
                            y: 0
                        };
                        this.dragStartX = x;
                        this.dragStartY = y;
                        this.inDrag = true;
                        if (this.measureOn) {
                            if (this.measurePrevStartX) { // Start new measuring pair
                                this.measurePrevStartX = null;
                            }
                            else { // half way through measuring pair, so record 1st measurement
                                this.measurePrevStartX = this.measureStartX;
                                this.measurePrevStartY = this.measureStartY;
                                this.measurePrevEndX = this.measureEndX;
                                this.measurePrevEndY = this.measureEndY;
                            }
                            this.measureStartX = (x - xO) / s;
                            this.measureStartY = (y - yO) / s;
                        }
                    }
                }
            }
        }
        else {
            this.dragRuler = false;
            if (x == 0 && y == 0) { // Touch UP events don't include the XY data, so use the last touched point
                x = this.lastTouch.x;
                y = this.lastTouch.y;
            }
            if (x < bH && y < bH && (x > 0 || y > 0)) { // if drag ends in the top-left ruler corner, then put ruler guide away
                if (this.rulerGuide.out)
                    this.doRender = true;
                this.rulerGuide.out = false;
            }
            if (this.inDrag && this.history.length > 0) {
                this.inertia = {// if mouse/touch just ended, then apply inertia
                    x: (this.x - this.history[0].x) / 4,
                    y: (this.y - this.history[0].y) / 4
                };
                if (this.measureChondrules) {
                    this.addChondruleMeasurement();
                }
            }
            this.inDrag = false;
            if (this.circleOn)
                this.doRender = true;
        }
    };

    /**
     * @private
     */
    OU.util.TileViewer.prototype.markerHit = function(x, y, event) {
        var rX, rY, s = this.s * this.scaleFactor + this.viewAllScale,
                xO = this.x + this.crop.x * s,
                yO = this.y + this.crop.y * s,
                i, hs, isHit, hitMarker = false,
                hotspots = this.hotspots;

        if (hotspots && hotspots.length > 0) {
            for (i = hotspots.length; i--; ) {
                hs = hotspots[i];
                isHit = false;
                if (hs.hotspot !== undefined) {
                    if (hs.hotspot.radius !== undefined) { // circle hotspot
                        rX = xO + hs.hotspot.x * s;
                        rY = yO + hs.hotspot.y * s;
                        if (Math.sqrt((rX - x) * (rX - x) + (rY - y) * (rY - y)) <= hs.hotspot.radius * s) {
                            isHit = true;
                        }
                    }
                    else if (hs.hotspot.polygon) {
                        isHit = this.polygonHit({
                            x: x,
                            y: y,
                            xO: xO,
                            yO: yO,
                            points: hs.hotspot.polygon,
                            scale: s
                        });
                    }
                    else { // rectangular hotspot
                        rX = (xO + hs.hotspot.x * s);
                        rY = (yO + hs.hotspot.y * s);
                        if (x > rX && x < rX + hs.hotspot.w * s && y > rY && y < rY + hs.hotspot.h * s) {
                            isHit = true;
                        }
                    }
                }
                else {
                    rX = (xO + hs.x * s) - this.rotationImage.width / 2;
                    rY = (yO + hs.y * s) - this.rotationImage.height / 2;
                    if (x > rX && x < rX + this.rotationImage.width && y > rY && y < rY + this.rotationImage.height) {
                        isHit = true;
                    }
                }
                if (isHit) {
                    hitMarker = true;
                    this.mouseDown = false;
                    hs.arrayIndex = i;
                    this.markerCallback(hs, this.container, event);
                }
            }
        }
        return hitMarker;
    };
    /**
     * Determines if a polygon shaped hotspot has been hit.
     * Don't worry about how this works, but it does!
     * @returns {int} oddNodes - False if hit outside the polygon. True if hit inside
     * @private
     */
    OU.util.TileViewer.prototype.polygonHit = function(params) {

        var points = params.points, i, j = points.length - 1,
                x = params.x,
                y = params.y,
                xO = params.xO,
                yO = params.yO,
                sf = params.scale,
                xI, xJ, yI, yJ, oddNodes = false;

        for (i = 0; i < points.length; i++) {
            xI = xO + points[i].x * sf;
            xJ = xO + points[j].x * sf;
            yI = yO + points[i].y * sf;
            yJ = yO + points[j].y * sf;
            if ((yI < y && yJ >= y || yJ < y && yI >= y) && (xI <= x || xJ <= x)) {
                if (xI + (y - yI) / (yJ - yI) * (xJ - xI) < x) {
                    oddNodes = !oddNodes;
                }
            }
            j = i;
        }
        console.log('returning: ' + oddNodes);
        return oddNodes;
    };
    /**
     * @private
     */
    OU.util.TileViewer.prototype.stop = function() {
        this.inertia = {
            x: 0,
            y: 0
        };
        this.doRender = false;
    };
    /**
     * Switches features on/off
     * @param {object} params - options:
     * <ul>
     * <li><strong>{boolean} showRulers: </strong> True to show rulers, False to hide them, undefined to leave the state as is</li>
     * <li><strong>{boolean} showGrid: </strong> True to show grid, False to hide them, undefined to leave the state as is</li>
     * <li><strong>{boolean} measure: </strong> True to switch measuring on, False to switch measuring off, undefined to leave the state as is</li>
     * </ul>
     */
    OU.util.TileViewer.prototype.setFeatures = function(params) {
        if (params.showRulers !== undefined)
            this.showRulers = params.showRulers;
        if (params.showGrid !== undefined)
            this.showGrid = params.showGrid;
        if (params.showGraticule !== undefined)
            this.showGraticule = params.showGraticule;
        if (params.measure !== 'off')
            this.measureOn = params.measure;
        if (params.measureChondrules !== undefined) {
            if (params.measureChondrules) {
                this.chondrules = this.chondrules || [];
                if (this.chondrules.length === 0) {
                    this.loadChondrules();
                }
                this.measureChondrules = true;
                this.measureOn = true;
                this.renderMeasure = true;
                this.renderAngle = false;
                this.vectorAngles = false;
                this.updateChondrulesCount();
            }
            else {
                this.measureChondrules = false;
                this.measureOn = false;
            }
        }
        this.doRender = true;
    };
    /**
     * setPosition - changes the view to a specific X,Y position with a set Zoom level
     *
     * @param {object} newPos - optional parameters (x/y pairs should be specified in either pixels or MM):
     * <ul>
     * <li><strong>xPixels</strong>: new X coordinate in pixels</li>
     * <li><strong>yPixels</strong>: new Y coordinate in pixels</li>
     * <li><strong>xMM</strong>: new X coordinate in mm</li>
     * <li><strong>yMM</strong>: new Y coordinate in mm</li>
     * <li><strong>zoom</strong>: new Zoom (0 to 1)</li>
     * </ul>
     */
    OU.util.TileViewer.prototype.setPosition = function(newPos) {
        var p2m = this.imageBase[this.imageIdx].pixelsPerMM;
        if (newPos.zoom) {
            this.s = newPos.zoom;
            this.s = this.s < 0 ? 0 : (this.s > 1 ? 1 : this.s);
        }
        if (newPos.xMM && newPos.yMM) {
            this.x = this.window.w / 2 - newPos.xMM * (p2m * (this.s * this.scaleFactor + this.viewAllScale));
            this.y = this.window.h / 2 - newPos.yMM * (p2m * (this.s * this.scaleFactor + this.viewAllScale));
        }
        if (newPos.xPixels && newPos.yPixels) {
            this.x = this.window.w / 2 - newPos.xPixels * ((this.s * this.scaleFactor + this.viewAllScale));
            this.y = this.window.h / 2 - newPos.yPixels * ((this.s * this.scaleFactor + this.viewAllScale));
        }
        this.doRender = true;
    };
    OU.util.TileViewer.prototype.nudge = function(direction) {
        var horizontalHalf = this.window.w / 2,
                verticalHalf = this.window.h / 2;
        switch (direction) {
            case 'left':
                this.x = this.x + horizontalHalf;
                break;
            case 'right':
                this.x = this.x - horizontalHalf;
                break;
            case 'up':
                this.y = this.y + verticalHalf;
                break;
            case 'down':
                this.y = this.y - verticalHalf;
                break;
        }
        this.doRender = true;
    };
    /**
     * Returns the current positon in pixels of the centre point, and the zoom level
     */
    OU.util.TileViewer.prototype.getPositionPixels = function() {
        var s = this.s * this.scaleFactor + this.viewAllScale,
                x = ((this.window.w / 2) - this.x) / s,
                y = ((this.window.h / 2) - this.y) / s,
                zoom = (s - this.viewAllScale) / (1 - this.viewAllScale);
        return {
            x: x | 0,
            y: y | 0,
            zoom: zoom.nDecimals(6)
        };
    };
    /**
     * Returns the current positon in Millimetres of the centre point, and the zoom level
     */
    OU.util.TileViewer.prototype.getPositionMM = function() {
        var s = this.s * this.scaleFactor + this.viewAllScale,
                p2m = this.imageBase[this.imageIdx].pixelsPerMM,
                xMM = ((this.window.w / 2) - this.x) / (p2m * s),
                yMM = ((this.window.h / 2) - this.y) / (p2m * s),
                zoom = (s - this.viewAllScale) / (1 - this.viewAllScale);
        return {
            x: xMM.nDecimals(6),
            y: yMM.nDecimals(6),
            zoom: zoom.nDecimals(6)
        };
    };
    /**
     * Returns the current Measurement (distance) in Millimetres
     */
    OU.util.TileViewer.prototype.getMeasureMM = function() {
        var p2m = this.imageBase[this.imageIdx].pixelsPerMM;
        var response = {
            distance: this.measureMM,
            x1pixels: this.measureStartX,
            y1pixels: this.measureStartY,
            x2pixels: this.measureEndX,
            y2pixels: this.measureEndY,
            x1: this.measureStartX / p2m,
            y1: this.measureStartY / p2m,
            x2: this.measureEndX / p2m,
            y2: this.measureEndY / p2m,
            pixelsPerMM: p2m,
            state: this.container._measureState
        };
        if (this.vectorAngles) {
            response.altx1pixels = this.measurePrevStartX;
            response.alty1pixels = this.measurePrevStartY;
            response.altx2pixels = this.measurePrevEndX;
            response.alty2pixels = this.measurePrevEndY;
            response.altx1 = this.measurePrevStartX / p2m;
            response.alty1 = this.measurePrevStartY / p2m;
            response.altx2 = this.measurePrevEndX / p2m;
            response.alty2 = this.measurePrevEndY / p2m;
        }
        return response;
    };
    /**
     * Returns the current Measurement (distance) in Millimetres
     */
    OU.util.TileViewer.prototype.setMeasureMM = function(newPos) {
        var p2m = this.imageBase[this.imageIdx].pixelsPerMM;
        if (newPos.setInPixels) {
            this.measureStartX = newPos.x1;
            this.measureStartY = newPos.y1;
            this.measureEndX = newPos.x2;
            this.measureEndY = newPos.y2;
            this.measurePrevStartX = newPos.altx1;
            this.measurePrevStartY = newPos.alty1;
            this.measurePrevEndX = newPos.altx2;
            this.measurePrevEndY = newPos.alty2;
        }
        else {
            this.measureStartX = newPos.x1 * p2m;
            this.measureStartY = newPos.y1 * p2m;
            this.measureEndX = newPos.x2 * p2m;
            this.measureEndY = newPos.y2 * p2m;
            this.measurePrevStartX = newPos.altx1 * p2m;
            this.measurePrevStartY = newPos.alty1 * p2m;
            this.measurePrevEndX = newPos.altx2 * p2m;
            this.measurePrevEndY = newPos.alty2 * p2m;
        }
        if (newPos.state === 'angle')
            this.vectorAngles = true;
        else
            this.vectorAngles = false;
        this.measureOn = true; //params.measure;
        this.renderMeasure = true;
        this.doRender = true;
    };
    /**
     * Returns the current view's parameters in a GET parameter format
     * @returns {string} params
     */
    OU.util.TileViewer.prototype.getDirectParams = function() {
        var s = this.s * this.scaleFactor + this.viewAllScale,
                p2m = this.imageBase[this.imageIdx].pixelsPerMM,
                xMM = ((this.window.w / 2) - this.x) / (p2m * s),
                yMM = ((this.window.h / 2) - this.y) / (p2m * s),
                zoom = (s - this.viewAllScale) / (1 - this.viewAllScale);
        return "x=" + xMM.nDecimals(6) + "&y=" + yMM.nDecimals(6) + "&zoom=" + zoom.nDecimals(6) + "&s=" + this.imageIdx;
    };
    /**
     * Sets new scale
     * @param {double} scale - new scale value between 0 and 1
     * @param {object} centre - point to keep as centre of the zoom change in form eg. {x:100,y:200}
     */
    OU.util.TileViewer.prototype.scale = function(scale, centre) {
        var ns = scale > 1 ? 1 : (scale < 0 ? 0 : scale), // ensure new scale is within limits
                nnS = ns * this.scaleFactor + this.viewAllScale,
                noS = this.s * this.scaleFactor + this.viewAllScale,
                dS = nnS / noS,
                cX = this.window.w / 2, // center zoom on middle of window
                cY = this.window.h / 2, offX, offY;
        if (centre) { // if zooming via pinch, then center on touch point
            cX = centre.x;
            cY = centre.y;
            this.history.length = 0;// zero history so that pinch release doesn't trigger inertia movement
        }
        offX = cX - this.x,
                offY = cY - this.y;
        this.x += offX - (offX * dS); // set new position to maintain zoom center
        this.y += offY - (offY * dS);
        this.s = ns; // set new Scale
        this.doRender = true;
    };
    /**
     * @private
     */
    OU.util.TileViewer.prototype.renderCycle = function() {
        var self = this, s,
                uncroppedW = this.uncroppedW,
                uncroppedH = this.uncroppedH;
        if (this.history.length > 4)
            this.history.shift();
        this.history.push({
            x: this.x,
            y: this.y
        });
        if (!this.mouseDown && (this.inertia.x || this.inertia.y)) {
            s = this.s * this.scaleFactor + this.viewAllScale;
            if (this.edgeSnap) {
                this.x = (this.x + this.inertia.x) % (uncroppedW * s);
                this.y = (this.y + this.inertia.y) % (uncroppedH * s);
            }
            else {
                this.x = (this.x + this.inertia.x);
                this.y = (this.y + this.inertia.y);
            }
            this.inertia.x = this.inertia.x * .9;
            this.inertia.y = this.inertia.y * .9;
            this.inertia.x = this.inertia.x < 2 && this.inertia.x > -2 ? 0 : this.inertia.x;
            this.inertia.y = this.inertia.y < 2 && this.inertia.y > -2 ? 0 : this.inertia.y;
            this.doRender = true;
        }
        if (this.doRender)
            this.render();
        setTimeout(function() {
            self.renderCycle();
        }, 40);
    };
    /**
     * Render the current view
     * @private
     */
    OU.util.TileViewer.prototype.render = function() {
        var i, hs, nX, nY, x1, x2, y1, y2, x, y, ax, tile, rts, hX, hY, tX, tY, tW, tH, zoomLevels = this.nZ,
                ctx = this.context, gap,
                rulerHeight = this.showRulers ? 30 : 0,
                uncroppedW = this.uncroppedW,
                uncroppedH = this.uncroppedH,
                hotspots = this.hotspots,
                s = this.s * this.scaleFactor + this.viewAllScale, z; // factored scale between 100% and max zoom

        if (!this.rough)
            return;

        // Determine the best zoom level to use
        z = (zoomLevels * ((1 - this.viewAllScale) * s + this.viewAllScale)) | 0; // calcs relevant zoom level for best quality images
        z = z >= zoomLevels - 1 ? zoomLevels - 1 : z;
        if (this.dims.w < this.dims.h) {
            do {
                z++;
                nX = Math.pow(2, z); // number of tiles across or down @ zoom level
                rts = (uncroppedW / nX) / this.propX; // relative tile size @ zoom level
            }
            while (rts * s > 1.5 * this.tileSize && z < zoomLevels - 1);
            if (z === zoomLevels)
                nX = nX / 2;
            //            nX = ((nX*this.roughSF)|0)+1;
            nY = ((nX * uncroppedH / uncroppedW) | 0); // +1;
            rts = (uncroppedW / nX) / this.propX; // relative tile size @ zoom level
        }
        else {
            do {
                z++;
                nY = Math.pow(2, z); // number of tiles across or down @ zoom level
                rts = (uncroppedH / nY) / this.propY; // relative tile size @ zoom level
            }
            while (rts * s > 1.5 * this.tileSize && z < zoomLevels - 1);
            if (z === zoomLevels)
                nY = nY / 2;
            //            nY = ((nY*this.roughSF)|0)+1;
            nX = ((nY * uncroppedW / uncroppedH) | 0); // +1;
            rts = (uncroppedH / nY) / this.propY; // relative tile size @ zoom level
        }
        this.z = z = z >= zoomLevels - 1 ? zoomLevels - 1 : z;

        while ((this._tiles[z][nX] === undefined || this._tiles[z][nX][0] === undefined) && nX > 0)
            nX--;
        nX++;
        while ((this._tiles[z][0][nY] === undefined || this._tiles[z][0][nY] === undefined) && nY > 0)
            nY--;
        nY++;

        // Ensure that content stays in view vertically
        if (this.edgeSnap) {
            if (this.extendDrag) {
                if (this.y > this.window.h / 2 + rulerHeight - this.crop.y * s) // Don't allow content to move down too far
                    this.y = this.window.h / 2 + rulerHeight - this.crop.y * s;
                if (this.y < (this.window.h - (this.dims.h + this.crop.y) * s) - this.window.h / 2) { // Don't allow content to move up too far
                    this.y = (this.window.h - (this.dims.h + this.crop.y) * s) - this.window.h / 2;
                }
            }
            else { //
                if (this.y > rulerHeight - this.crop.y * s) // Don't allow content to move down too far
                    this.y = rulerHeight - this.crop.y * s;
                if (this.y < (this.window.h - (this.dims.h + this.crop.y) * s)) { // Don't allow content to move up too far
                    if ((this.window.h - (this.dims.h + this.crop.y) * s) < 0)
                        this.y = (this.window.h - (this.dims.h + this.crop.y) * s); // if off screen, just make sure it doesn't go too far
                    else
                        this.y = (this.window.h - (this.dims.h + this.crop.y) * s) / 2; // if all visible vertically, then centre vertically
                }
            }
        }
        // Ensure content stays in view horizontally , unless we are using horizontal stitch, ie 360 degree wrap around
        if (!this.horizontalStitch) {
            if (this.extendDrag) {
                if (this.x > this.window.w / 2) {
                    this.x = this.window.w / 2;
                }
            }
            else {
                if (this.x > 0) {
                    this.x = 0;
                }
                if (this.x < (this.window.w - (this.dims.w + this.crop.x) * s)) {
                    if ((this.window.w - (this.dims.w + this.crop.x) * s) < 0)
                        this.x = (this.window.w - (this.dims.w + this.crop.x) * s); // if off screen, just make sure it doesn't go too far
                    else
                        this.x = (this.window.w - (this.dims.w + this.crop.x) * s) / 2; // if all visible vertically, then centre vertically
                }
            }
        }

        // calc viewport dims & location
        this.v.x = -this.x / s;
        this.v.y = -this.y / s;
        this.v.w = this.window.w / s;
        this.v.h = this.window.h / s;

        x1 = ((this.v.x / rts) | 0) - 1;
        x2 = (((this.v.x + this.v.w) / rts) | 0) + 1;
        y1 = (this.v.y / rts) | 0;
        y2 = (((this.v.y + this.v.h) / rts) | 0) + 1;

        if (!this.horizontalStitch) {
            x1 = x1 < 0 ? 0 : x1 > nX - 1 ? nX - 1 : x1;
            x2 = x2 < 0 ? 0 : x2 > nX - 1 ? nX - 1 : x2;
        }
        else {
            x1 = x1 <= 0 ? x1 - 1 : x1;
            x2 = x2 <= 0 ? x2 - 1 : x2;
        }
        y1 = y1 < 0 ? 0 : y1 > nY - 1 ? nY - 1 : y1;
        y2 = y2 < 0 ? 0 : y2 > nY ? nY : y2;
        ctx.clearRect(0, 0, this.window.w, this.window.h + 60);

        ctx.drawImage(this.rough, this.x, this.y, uncroppedW * s, uncroppedH * s);

        if (this.horizontalStitch) {
            if (this.x > 0)
                ctx.drawImage(this.rough, this.x - uncroppedW * s, this.y, uncroppedW * s, uncroppedH * s);
            if (this.x < 0 - (this.dims.w * s - this.window.w))
                ctx.drawImage(this.rough, this.x + uncroppedW * s, this.y, uncroppedW * s, uncroppedH * s);
        }
        gap = ((this.tileSize - this._tiles[z][nX - 1][0].width) / this.tileSize) * rts * s;
        if (isNaN(gap) && this._tiles[z][nX - 2]) {
            gap = ((this.tileSize - this._tiles[z][nX - 2][0].width) / this.tileSize) * rts * s;
        }
        if (isNaN(gap))
            gap = 0;
        for (x = x1; x <= x2; x++) { // render images if loaded, otherwise load relevant tiles
            for (y = y1; y <= y2; y++) {
                ax = this.horizontalStitch ? x % nX : x;
                ax = ax < 0 ? ax + nX : ax;
                //                console.log("x:"+x+" ax:"+ax+" nX:"+nX);
                if (this._tiles[z][ax]) {
                    tile = this._tiles[z][ax][y];
                    if (tile) {
                        if (tile.loadState === LOADED) {
                            tX = this.x + x * rts * s;
                            tY = this.y + y * rts * s;
                            if (x < 0) {
                                tX = tX + gap; // close the gap left by the last column of tiles
                                //                                console.log('gap+:'+gap+" ax:"+ax+" x:"+x+" z:"+z+" nX:"+nX);
                            }
                            if (x > ax) {
                                tX = tX - gap; // close the gap left by the last column of tiles
                                //                                console.log('gap-:'+gap+" ax:"+ax+" x:"+x+" z:"+z+" nX:"+nX);
                            }


                            tW = (tile.width / this.tileSize) * rts * s;
                            tH = (tile.height / this.tileSize) * rts * s;
                            ctx.drawImage(tile.image, tX, tY, tW, tH);
                            /* debug
                             ctx.strokeRect(tX, tY, tW, tH);
                             /*
                             console.clear();
                             console.log("rts: "+rts+" w: "+tW+"   s: "+s);
                             console.log("tile.width: "+tile.width);
                             console.log("this.tileSize: "+this.tileSize);
                             console.log("z: "+z);
                             console.log("wH/dimsW: "+this.window.w/this.dims.w);
                             console.log("roughSF: "+this.roughSF);

                             //*/
                        }
                        else if (tile.loadState === NOTLOADED) {
                            this.loadTile(tile);
                        }
                    }
                }
            }
        }
        this.filters();
        if (hotspots && hotspots.length > 0) {
            if (this.showHotspots) {
                this.renderHotspotOutlines();
            }
            if (this.rotationImageLoaded) {
                for (i = hotspots.length; i--; ) {
                    if (hotspots[i].hotspot)
                        hs = hotspots[i].hotspot;
                    else
                        hs = hotspots[i];
                    hX = this.x + hs.x * s;
                    hY = this.y + hs.y * s;
                    if (hX) {
                        ctx.drawImage(this.rotationImage, hX - this.rotationImage.width / 2, hY - this.rotationImage.height / 2, this.rotationImage.width, this.rotationImage.height);
                        if (this.numberMarkers) {
                            ctx.fillText(i + 1, hX, hY);
                        }
                        if (hotspots[i].hasFocus) {
                            this.renderHotspotOutline(hotspots[i]);
                        }
                    }
                }
            }
        }
        if (this.showRulers)
            this.renderRulers();

        if (this.renderMeasure && this.measureOn)
            this.renderMeasurements();

        if (this.thumbnail) {
            this.thumbnail.render();
        }
        if (this.monitorPositionPixels)
            this.monitorPositionPixels(this.getPositionPixels());

        if (this.showGrid)
            this.renderGrid();
        else if (this.showGraticule)
            this.renderGraticule();
        else if (this.showCrosshairs)
            this.renderCrosshairs();

        this.doRender = false;
    };
    OU.util.TileViewer.prototype.renderHotspotOutline = function(hotspot) {
        var hs, ctx = this.context, s = this.s * this.scaleFactor + this.viewAllScale;
        ctx.save();
        ctx.strokeStyle = this.hotspotOutlineColour || 'rgba(0,0,200,0.5)';
        if (hotspot.hasFocus) {
            ctx.strokeStyle = 'rgba(200,0,0,0.8)';
        }

        ctx.lineWidth = 4;

        if (hotspot.hotspot)
            hs = hotspot.hotspot;
        else
            hs = hotspot;
        ctx.beginPath();
        if (hs.radius) {
            ctx.arc(this.x + (this.crop.x + hs.x) * s, this.y + (this.crop.y + hs.y) * s, hs.radius * s, Math.PI * 2, 0, false);
        }
        if (this.rotationImage) {
            ctx.arc(this.x + (this.crop.x + hs.x) * s, this.y + (this.crop.y + hs.y) * s, this.rotationImage.width / 2 + 10 * s, Math.PI * 2, 0, false);
        }
        else if (hs.polygon) {
            var p, px, py, points = hs.polygon;
            ctx.beginPath();
            for (p = points.length; p--; ) {
                px = this.x + (this.crop.x + points[p].x) * s;
                py = this.y + (this.crop.y + points[p].y) * s;
                if (p === points.length - 1) {
                    ctx.moveTo(px, py);
                }
                else {
                    ctx.lineTo(px, py);
                }
            }
            ctx.closePath();
            ctx.stroke();
        }
        else {
            ctx.rect(this.x + (this.crop.x + hs.x) * s, this.y + (this.crop.y + hs.y) * s, hs.w * s, hs.h * s);
        }
        ctx.stroke();
        ctx.restore();
    };
    /**
     * Displays the hotspot outlines
     * @private
     */
    OU.util.TileViewer.prototype.renderHotspotOutlines = function() {
        var i, hotspots = this.hotspots;
        for (i = hotspots.length; i--; ) {
            this.renderHotspotOutline(hotspots[i]);
        }
    };
    /**
     * Applies filters to the image
     */
    OU.util.TileViewer.prototype.filters = function() {
        OU.util.Blur({
            context: this.context,
            radius: this.blurRadius
        });
        OU.util.brighten({
            context: this.context,
            factor: this.brightenFactor
        });
    };
    OU.util.TileViewer.prototype.addChondruleMeasurement = function() {
        var line1, p2m = this.imageBase[this.imageIdx].pixelsPerMM,
                thetaAngle, hypotenuseText;
        if (this.measureStartX > 0 && this.measureStartY > 0 && this.measureEndX > 0 && this.measureEndY) {

            line1 = new OU.util.Vector2D({
                x1: this.measureStartX,
                y1: this.measureStartY,
                x2: this.measureEndX,
                y2: this.measureEndY
            });
            this.measureMM = (line1.hypotenuse / p2m);
            this.measureMicrons = this.measureMM * 1000;
            this.measureMicrons = (this.measureMicrons + 0.5) | 0;
            hypotenuseText = this.measureMicrons + ' \u03BCm';

            if (this.measureStartY > this.measureEndY) {
                if (this.measureStartX < this.measureEndX) {
                    thetaAngle = -line1.trigRadians;
                }
                else {
                    thetaAngle = line1.trigRadians;
                }
            }
            else {
                if (this.measureStartX > this.measureEndX) {
                    thetaAngle = -line1.trigRadians;
                }
                else {
                    thetaAngle = line1.trigRadians;
                }
            }
            this.chondrules.push({
                x1: this.measureStartX,
                y1: this.measureStartY,
                x2: this.measureEndX,
                y2: this.measureEndY,
                thetaAngle: thetaAngle,
                hypotenuseText: hypotenuseText,
                lengthInMicrons: this.measureMicrons
            });
            this.saveChondrules();
        }
    };
    OU.util.TileViewer.prototype.updateChondrulesCount = function() {
        var counterDiv = document.getElementById('chondruleCount');
        if (counterDiv) {
            counterDiv.innerHTML = this.chondrules.length;
        }
    };
    OU.util.TileViewer.prototype.clearChondrules = function() {
        this.chondrules = [];
        this.measureStartX = this.measureStartY = this.measureEndX = this.measureEndY = null;
        this.saveChondrules();
    };
    OU.util.TileViewer.prototype.saveChondrules = function() {
        var datasetName = '';
        if (this.container && this.container.imageBase && this.container.imageBase.name) {
            datasetName = this.container.imageBase.name;
        }
        OU.LocalStorage.save('VM_chondrules_' + datasetName, JSON.stringify(this.chondrules));
        this.updateChondrulesCount();
    };
    OU.util.TileViewer.prototype.loadChondrules = function() {
        var arrayString, datasetName = '';
        this.chondrules = [];
        if (this.container && this.container.imageBase && this.container.imageBase.name) {
            datasetName = this.container.imageBase.name;
        }
        arrayString = OU.LocalStorage.load('VM_chondrules_' + datasetName);
        if (arrayString) {
            this.chondrules = JSON.parse(arrayString);
        }
        if (!this.chondrules || this.chondrules.length < 1) {
            this.chondrules = [];
        }
        this.updateChondrulesCount();
    };
    OU.util.TileViewer.prototype.undoChondrules = function() {
        this.chondrules.splice(this.chondrules.length - 1);
        this.measureStartX = this.measureStartY = this.measureEndX = this.measureEndY = null;
        this.saveChondrules();
    };
    OU.util.TileViewer.prototype.exportChondrules = function() {
        var i, form = "<p>Copy and paste the data below:</p><textarea style='margin:20px;width: 300px;height: 200px;'>Measurements (microns)\n";
        for (i = this.chondrules.length; i--; ) {
            form = form + this.chondrules[i].lengthInMicrons + "\n";
        }
        form = form + "</textarea><p align='right'><input type='submit' value='Done'/>";
        new OU.util.PopUpForm({
            container: this.container,
            formHTML: form
        });
    };
    OU.util.TileViewer.prototype.renderChondruleMeasurements = function() {
        var i;
        if (this.measureChondrules) {
            for (i = this.chondrules.length; i--; ) {
                this.renderLine(this.chondrules[i]);
            }
        }
    };
    OU.util.TileViewer.prototype.renderLine = function(params) {
        var ctx = this.context,
                s = this.s * this.scaleFactor + this.viewAllScale,
                x1 = params.x1,
                y1 = params.y1,
                x2 = params.x2,
                y2 = params.y2,
                thetaAngle = params.thetaAngle,
                hypotenuseText = params.hypotenuseText,
                dX = x2 - x1,
                dY = y2 - y1;

        ctx.beginPath();
        ctx.moveTo(this.x + x1 * s, this.y + y1 * s);
        ctx.lineTo(this.x + x2 * s, this.y + y2 * s);
        ctx.stroke();
        ctx.beginPath();

        ctx.save();
        ctx.translate(this.x + (x1 + dX / 2) * s, this.y + (y1 + dY / 2) * s);
        ctx.rotate(thetaAngle);
        ctx.fillStyle = '#fff';
        ctx.fillRect(-50, -27, 100, 20);
        ctx.fillStyle = this.gridColour;
        ctx.fillText(hypotenuseText, 0, -16);
        ctx.restore();
    };
    /**
     * Renders the measuring tool, either as a single line or as two lines with an angle between them
     * @private
     */
    OU.util.TileViewer.prototype.renderMeasurements = function() {
        var hypotenuseText, sa, se,
                dX, dY, thetaAngle, s = this.s * this.scaleFactor + this.viewAllScale,
                p2m = this.imageBase[this.imageIdx].pixelsPerMM, ctx = this.context, intersect,
                line2, line1 = new OU.util.Vector2D({
                    x1: this.measureStartX,
                    y1: this.measureStartY,
                    x2: this.measureEndX,
                    y2: this.measureEndY
                });

        dX = this.measureEndX - this.measureStartX;
        dY = this.measureEndY - this.measureStartY;

        this.renderChondruleMeasurements();

        if (line1.hypotenuse > 0) {
            this.measureMM = (line1.hypotenuse / p2m);
            this.measureMicrons = this.measureMM * 1000;
            this.measureMM = this.measureMM.toFixed(this.numDecimals);
            this.measureMicrons = this.measureMicrons.toFixed(0);
            if (this.measureMM < this.micronThreshold)
                hypotenuseText = this.measureMicrons + ' \u03BCm';
            else
                hypotenuseText = this.measureMM + ' mm';

            if (this.monitorMeasureMM)
                this.monitorMeasureMM(this.getMeasureMM());
            if (this.vectorAngles && this.measurePrevStartX) {
                // we have a pair of measurements, so show angle between them
                //
                line2 = new OU.util.Vector2D({
                    x1: this.measurePrevStartX,
                    y1: this.measurePrevStartY,
                    x2: this.measurePrevEndX,
                    y2: this.measurePrevEndY
                });
                intersect = line1.intersectPoint(line2);
                ctx.beginPath();
                ctx.moveTo(this.x + line2.x1 * s, this.y + line2.y1 * s);
                ctx.lineTo(this.x + line2.x2 * s, this.y + line2.y2 * s);
                ctx.moveTo(this.x + line1.x1 * s, this.y + line1.y1 * s);
                ctx.lineTo(this.x + line1.x2 * s, this.y + line1.y2 * s);
                ctx.stroke();
                if (intersect) {
                    ctx.beginPath();
                    ctx.moveTo(this.x + line1.x1 * s, this.y + line1.y1 * s);
                    ctx.lineTo(this.x + intersect.x * s, this.y + intersect.y * s);
                    ctx.lineTo(this.x + line2.x1 * s, this.y + line2.y1 * s);
                    ctx.stroke();
                    ctx.beginPath();
                    if (this.measureStartX < this.measureEndX) {
                        ctx.arc(this.x + intersect.x * s, this.y + intersect.y * s, line1.radius * s, line1.trigRadians, line2.trigRadians, false);
                    }
                    else {
                        ctx.arc(this.x + intersect.x * s, this.y + intersect.y * s, line1.radius * s, line1.trigRadians, line2.trigRadians, false);
                    }
                    if (!isNaN(intersect.x)) {
                        var x = this.x + intersect.x * s - 20;
                        var y = this.y + intersect.y * s + 20;
                        x = x < 60 ? 60 : x;
                        x = x > this.window.w - 60 ? this.window.w - 60 : x;
                        y = y < 30 ? 30 : y;
                        y = y > this.window.h - 30 ? this.window.h - 30 : y;
                        ctx.fillStyle = '#fff';
                        ctx.fillRect(x - 30, y - 10, 60, 20);
                        ctx.fillStyle = this.gridColour;
                        ctx.fillText(intersect.angle.degrees.nDecimals(2) + '\u00B0', x, y);
                        /* sort this out if ever needed, should draw the angle arc
                         if(line1.fullRadians<line2.fullRadians) {
                         sa = line1.fullRadians;
                         se = line2.fullRadians;
                         }
                         else {
                         sa = line2.fullRadians;
                         se = line1.fullRadians;
                         }
                         ctx.arc(x+20,y-20,120,sa,se,false);
                         //*/
                    }
                }
                ctx.stroke();
            }
            else {
                if (this.renderAngle) {
                    // we have a single measurement, so show angle relative to horizontal
                    //
                    ctx.beginPath();
                    ctx.moveTo(this.x + this.measureStartX * s, this.y + this.measureStartY * s);
                    ctx.lineTo(this.x + this.measureEndX * s, this.y + this.measureEndY * s);
                    if (this.measureStartY > this.measureEndY) {
                        ctx.lineTo(this.x + this.measureEndX * s, this.y + this.measureStartY * s);
                    }
                    else {
                        ctx.lineTo(this.x + this.measureStartX * s, this.y + this.measureEndY * s);
                    }
                    ctx.closePath();
                    ctx.stroke();
                    ctx.beginPath();
                    if (this.measureStartY > this.measureEndY) {
                        if (this.measureStartX < this.measureEndX) {
                            ctx.arc(this.x + this.measureStartX * s, this.y + this.measureStartY * s, line1.radius * s, -line1.trigRadians, 0, false);
                            thetaAngle = -line1.trigRadians;
                        }
                        else {
                            ctx.arc(this.x + this.measureStartX * s, this.y + this.measureStartY * s, line1.radius * s, Math.PI, Math.PI + line1.trigRadians, false);
                            thetaAngle = line1.trigRadians;
                        }
                        ctx.fillStyle = '#fff';
                        ctx.fillRect(this.x + this.measureStartX * s - 50, this.y + this.measureStartY * s + 10, 60, 20);
                        ctx.fillStyle = this.gridColour;
                        ctx.fillText(line1.trigDegrees.nDecimals(2) + '\u00B0', this.x + this.measureStartX * s - 20, this.y + this.measureStartY * s + 20);
                        ctx.save();
                        ctx.translate(this.x + (this.measureStartX + dX / 2) * s, this.y + (this.measureStartY + dY / 2) * s);
                        ctx.rotate(thetaAngle);
                        ctx.fillStyle = '#fff';
                        ctx.fillRect(-50, -27, 100, 20);
                        ctx.fillStyle = this.gridColour;
                        ctx.fillText(hypotenuseText, 0, -16);
                        ctx.restore();
                    }
                    else {
                        if (this.measureStartX > this.measureEndX) {
                            ctx.arc(this.x + this.measureEndX * s, this.y + this.measureEndY * s, line1.radius * s, -line1.trigRadians, 0, false);
                            thetaAngle = -line1.trigRadians;
                        }
                        else {
                            ctx.arc(this.x + this.measureEndX * s, this.y + this.measureEndY * s, line1.radius * s, Math.PI, Math.PI + line1.trigRadians, false);
                            thetaAngle = line1.trigRadians;
                        }
                        ctx.fillStyle = '#fff';
                        ctx.fillRect(this.x + this.measureEndX * s - 50, this.y + this.measureEndY * s + 10, 60, 20);
                        ctx.fillStyle = this.gridColour;
                        ctx.fillText(line1.trigDegrees.nDecimals(2) + '\u00B0', this.x + this.measureEndX * s - 20, this.y + this.measureEndY * s + 20);
                        ctx.save();
                        ctx.translate(this.x + (this.measureStartX + dX / 2) * s, this.y + (this.measureStartY + dY / 2) * s);
                        ctx.rotate(thetaAngle);
                        ctx.fillStyle = '#fff';
                        ctx.fillRect(-50, -27, 100, 20);
                        ctx.fillStyle = this.gridColour;
                        ctx.fillText(hypotenuseText, 0, -16);
                        ctx.restore();
                    }
                    ctx.stroke();
                }
                else {
                    // Single measurement with no angle displayed
                    //
                    if (this.measureStartY > this.measureEndY) {
                        if (this.measureStartX < this.measureEndX) {
                            thetaAngle = -line1.trigRadians;
                        }
                        else {
                            thetaAngle = line1.trigRadians;
                        }
                    }
                    else {
                        if (this.measureStartX > this.measureEndX) {
                            thetaAngle = -line1.trigRadians;
                        }
                        else {
                            thetaAngle = line1.trigRadians;
                        }
                    }
                    this.renderLine({
                        x1: this.measureStartX,
                        y1: this.measureStartY,
                        x2: this.measureEndX,
                        y2: this.measureEndY,
                        thetaAngle: thetaAngle,
                        hypotenuseText: hypotenuseText
                    });
                }
            }
        }
        if (this.mouseDown && this.measureOn) { // In drag, so draw a touch circle
            ctx.save();
            ctx.globalAlpha = .5;
            ctx.beginPath();
            ctx.arc(this.x + s * this.measureEndX, this.y + s * this.measureEndY, 50, 0, Math.PI * 2, false);
            ctx.moveTo(0, this.y + s * this.measureEndY); // (this.x + s * this.measureEndX - 50, this.y + s * this.measureEndY);
            ctx.lineTo(this.window.w, this.y + s * this.measureEndY); // (this.x + s * this.measureEndX + 50, this.y + s * this.measureEndY);
            ctx.moveTo(this.x + s * this.measureEndX, 0); // (this.x + s * this.measureEndX, this.y + s * this.measureEndY - 50);
            ctx.lineTo(this.x + s * this.measureEndX, this.window.h); // (this.x + s * this.measureEndX, this.y + s * this.measureEndY + 50);
            ctx.stroke();
            ctx.restore();
            this.circleOn = true;
        }
    };
    /**
     * @private
     */
    OU.util.TileViewer.prototype.renderCrosshairs = function() {
        var ctx = this.context,
                w = this.container.w,
                h = this.container.h - OU.controlHeight,
                p2m = this.imageBase[this.imageIdx].pixelsPerMM * (this.s * this.scaleFactor + this.viewAllScale),
                xMM = (w / 2 - this.x) / p2m,
                yMM = (h / 2 - this.y) / p2m,
                xPixels = xMM * this.imageBase[this.imageIdx].pixelsPerMM | 0,
                yPixels = yMM * this.imageBase[this.imageIdx].pixelsPerMM | 0,
                measure, txt;
        ctx.save();
        ctx.translate(w / 2, h / 2);

        ctx.beginPath();

        ctx.strokeStyle = this.gridColour;
        ctx.moveTo(-10, 0);
        ctx.lineTo(10, 0);
        ctx.moveTo(0, -10);
        ctx.lineTo(0, 10);
        ctx.stroke();
        ctx.restore();

        if (this.centreReadout) {
            if (this.centreReadout === 'mms') {
                txt = 'Centre: ' + xMM.toFixed(2) + ',' + yMM.toFixed(2) + ' mm';
            }
            else if (this.centreReadout === 'pixels') {
                txt = 'Centre: ' + xPixels + ',' + yPixels + ' pixels';
            }
            else { // both
                txt = 'Centre: ' + xMM.toFixed(2) + ',' + yMM.toFixed(2) + ' mm : ' + xPixels + ',' + yPixels + ' pixels';
            }
            ctx.save();
            ctx.font = '14px sans-serif';
            measure = ctx.measureText(txt, OU.VM_RULER_WIDTH, this.container.h);
            ctx.fillStyle = '#777';
            ctx.roundRect(OU.VM_RULER_WIDTH + 5, this.container.h - OU.controlHeight - 30, measure.width + 20, 28, 10);
            ctx.fill();
            ctx.fillStyle = '#eee';
            ctx.textAlign = 'left';
            ctx.fillText(txt, OU.VM_RULER_WIDTH + 15, this.container.h - OU.controlHeight - 15);
            ctx.restore();
        }
    };
    /**
     * @private
     */
    OU.util.TileViewer.prototype.renderGrid = function() {
        var i, ctx = this.context,
                w = this.container.w,
                h = this.container.h - OU.controlHeight,
                gap = h < w ? h / 12 : w / 12;

        ctx.save();
        ctx.translate(w / 2, h / 2);
        ctx.rotate(this.gridAngle);

        ctx.beginPath();

        ctx.strokeStyle = this.gridColour;
        for (i = -5; i <= 5; i++) {
            ctx.moveTo(-5 * gap, i * gap);
            ctx.lineTo(5 * gap, i * gap);
            ctx.moveTo(i * gap, -5 * gap);
            ctx.lineTo(i * gap, 5 * gap);
        }
        ctx.stroke();
        ctx.restore();
    };
    /**
     * @private
     */
    OU.util.TileViewer.prototype.renderGraticule = function() {
        var i, ctx = this.context,
                w = this.container.w,
                h = this.container.h - OU.controlHeight,
                p2m = this.imageBase[this.imageIdx].pixelsPerMM * (this.s * this.scaleFactor + this.viewAllScale);

        ctx.save();
        ctx.translate(w / 2, h / 2);
        ctx.rotate(this.gridAngle);

        ctx.beginPath();

        ctx.strokeStyle = this.gridColour;
//        ctx.lineWidth = 0.5;
        ctx.moveTo(-5 * p2m, 0);
        ctx.lineTo(5 * p2m, 0);
        for (i = 100; i >= 0; i--) {
            if (parseInt(i / 10) === i / 10) { // if int
                h = p2m;
            }
            else if (parseInt(i / 5) === i / 5) { // if int
                h = p2m / 2;
            }
            else {
                h = p2m / 4;
            }
            ctx.moveTo(-(i - 50) * p2m / 10, h / 2);
            ctx.lineTo(-(i - 50) * p2m / 10, -h / 2);
        }
        ctx.stroke();
        ctx.restore();
    };
    /**
     * @private
     */
    OU.util.TileViewer.prototype.renderRulers = function() {
        var ctx = this.context, s = this.s * this.scaleFactor + this.viewAllScale,
                bH = OU.VM_RULER_WIDTH, unitModulus,
                p2m = this.imageBase[this.imageIdx].pixelsPerMM,
                xMM = (0 - this.x) / (p2m * s),
                yMM = (0 - this.y) / (p2m * s),
                xInt, yInt, coords, coordW,
                xPx, xPos = xMM.nDecimals(1) - 0.1,
                yPx, yPos = yMM.nDecimals(1) - 0.1,
                x, y;
        ctx.save();
        ctx.font = '12px ' + OU.theme.font;
        ctx.lineWidth = 0.5;
        ctx.fillStyle = 'rgba(255,255,255,0.7)';
        ctx.strokeStyle = '#00c';
        ctx.fillRect(0, 0, this.window.w, bH);
        ctx.fillRect(0, bH, bH, this.window.h);
        ctx.fillStyle = '#00c';
        ctx.beginPath();
        unitModulus = (50 / (p2m * s)) | 0;
        unitModulus = unitModulus < 1 ? 1 : unitModulus;
        do {
            xPx = this.x + (xPos * p2m) * s;
            if (xPx > bH) {
                ctx.moveTo(xPx, bH);
                xInt = xPos | 0;
                if (xPos === xInt) {
                    ctx.lineTo(xPx, bH * .4);
                    if (xInt % unitModulus == 0)
                        ctx.fillText(xPos, xPx, bH * .25);
                }
                else if (xPos * 2 - 1 == xInt * 2 || xPos * 2 + 1 == xInt * 2) {
                    ctx.lineTo(xPx, bH * .6);
                }
                else {
                    ctx.lineTo(xPx, bH * .75);
                }
            }
            xPos = (xPos + 0.1).nDecimals(1);
        } while (xPx < this.window.w);
        ctx.stroke();
        ctx.fillStyle = '#fff';
        ctx.beginPath();
        ctx.roundRect(this.window.w - 60, bH * .5, 40, bH * .5, 5);
        ctx.fill();
        ctx.fillStyle = '#00c';
        ctx.fillText("mm", this.window.w - 40, bH * .75);
        do {
            yPx = this.y + (yPos * p2m) * s;
            if (yPx > bH) {
                ctx.moveTo(bH, yPx);
                yInt = yPos | 0;
                if (yPos === yInt) {
                    ctx.lineTo(bH * .4, yPx);
                    if (yInt % unitModulus == 0)
                        ctx.fillText(yPos, bH * .25, yPx);
                }
                else if (yPos * 2 - 1 == yInt * 2 || yPos * 2 + 1 == yInt * 2) {
                    ctx.lineTo(bH * .6, yPx);
                }
                else {
                    ctx.lineTo(bH * .75, yPx);
                }
            }
            yPos = (yPos + 0.1).nDecimals(1);
        } while (yPx < this.window.h);
        ctx.stroke();
        if (this.rulerGuide.out) {
            xPx = this.x + (this.rulerGuide.x * p2m) * s;
            yPx = this.y + (this.rulerGuide.y * p2m) * s;
            ctx.lineWidth = 1;
            ctx.beginPath();
            ctx.moveTo(0, yPx);
            ctx.lineTo(this.window.w, yPx);
            ctx.moveTo(xPx, 0);
            ctx.lineTo(xPx, this.window.h);
            ctx.stroke();
            ctx.beginPath();
            ctx.arc(xPx, yPx, 25, 0, Math.PI * 2, false);
            ctx.stroke();
            ctx.globalAlpha = 0.4;
            ctx.fillStyle = "#fff";
            ctx.fill();
            ctx.globalAlpha = 1;
            coords = (this.rulerGuide.x).nDecimals(2) + ", " + (this.rulerGuide.y).nDecimals(2) + " mm";
            coordW = ctx.measureText(coords, 0, 0);
            if (xPx < this.window.w / 2) { // left Half of screen
                if (yPx < this.window.h / 2) { // top Half of screen
                    x = xPx + coordW.width / 2 + 35;
                    y = yPx + 35;
                }
                else { // bottom
                    x = xPx + coordW.width / 2 + 35;
                    y = yPx - 35;
                }
            }
            else { // right half of screen
                if (yPx < this.window.h / 2) { // top Half of screen
                    x = xPx - coordW.width / 2 - 35;
                    y = yPx + 35;
                }
                else { // bottom
                    x = xPx - coordW.width / 2 - 35;
                    y = yPx - 35;
                }
            }
            ctx.roundRect(x - coordW.width / 2 - 10, y - 10, coordW.width + 20, 20, 5);
            ctx.fill();
            ctx.fillStyle = '#00c';
            ctx.fillText(coords, x, y);
        }
        else {
            ctx.fillStyle = '#fff';
            ctx.fillRect(0, 0, bH, bH);
            ctx.beginPath();
            ctx.arc(bH / 2, bH / 2, bH / 4, 0, Math.PI * 2, false);
            ctx.moveTo(0, bH / 2);
            ctx.lineTo(bH, bH / 2);
            ctx.moveTo(bH / 2, 0);
            ctx.lineTo(bH / 2, bH);
            ctx.stroke();
        }
        ctx.restore();
    };
    this.init();
};
/**
 * @class Thumbnail view of the tileviewer - Renders a thumbnail of the full image and current view.
 * The thumbnail should be on a separate canvas to the main view
 * @param {object} params - options:
 * <ul>
 * <li><strong>{OU.util.TileViewer} tileViewer:</strong> The tileviewer that is using this thumbnail</li>
 * <li><strong>{OU.util.Layer} layer:</strong> The layer that the thumbnail will be rendered to (must have events turned on)</li>
 * <li><strong>{int} x:</strong> X co-ordinate</li>
 * <li><strong>{int} y:</strong> Y co-ordinate</li>
 * <li><strong>{int} w:</strong> Width</li>
 * <li><strong>{int} h:</strong> Height</li>
 * </ul>
 * @private
 */
OU.util.TileViewer.prototype.Thumbnail = function(params) {

    this.view = params.view || 'rectangle';
    this.tileViewer = params.tileViewer;
    this.context = params.layer.context;
    params.layer.events.clickable.push(this); // push the thumbnail into the clickable array, so it can handle events to move the thumbnail around

    /**
     * Renders the current thumbnail view
     */
    OU.util.TileViewer.prototype.Thumbnail.prototype.render = function() {
        var ctx = this.context, tileViewer = this.tileViewer,
                uncroppedW = tileViewer.uncroppedW,
                uncroppedH = tileViewer.uncroppedH,
                windowScale = (tileViewer.s * tileViewer.scaleFactor + tileViewer.viewAllScale),
                thumbScale = (this.w / uncroppedW),
                tX = this.x + (-tileViewer.x / windowScale) * thumbScale,
                tY = this.y + (-tileViewer.y / windowScale) * thumbScale,
                wPerc = (tileViewer.window.w / (uncroppedW * windowScale)),
                hPerc = (tileViewer.window.h / (uncroppedH * windowScale)),
                tW = this.w * wPerc,
                tH = this.h * hPerc;
        this.thumbScale = thumbScale;
        this.windowScale = windowScale;
        if (ctx === undefined)
            return;

        // These are the thumbnail dimensions
        this.tX = tX;
        this.tY = tY;
        this.tW = tW;
        this.tH = tH;
        this.wPerc = wPerc;
        this.hPerc = hPerc;
//        console.log('field of view: '+wPerc.toFixed(2)+'% width, '+hPerc.toFixed(2)+'% height, ');

        if (this.view === 'rectangle') {
            // Adjust thumb outline, so it stays within the image
            if (tX < this.x) {
                tW = tW - (this.x - tX);
                tX = this.x;
            }
            else if (tX > this.x + this.w) {
                tW = 0;
                tX = this.x + this.w;
            }
            if (tY < this.y) {
                tH = tH - (this.y - tY);
                tY = this.y;
            }
            else if (tY > this.y + this.h) {
                tH = 0;
                tY = this.y + this.h;
            }
            if (tX + tW > this.x + this.w) {
                tW = this.x + this.w - tX;
            }
            if (tY + tH > this.y + this.h) {
                tH = this.y + this.h - tY;
            }
        }
        ctx.save();
        ctx.beginPath();
        ctx.strokeStyle = "#000";
        ctx.drawImage(tileViewer.rough, this.x, this.y, this.w, this.h);
        ctx.rect(this.x, this.y, this.w, this.h);
        ctx.stroke();

        ctx.beginPath();
        ctx.lineWidth = 2;
        ctx.strokeStyle = "#f22";
        if (this.view === 'rectangle')
            ctx.rect(tX, tY, tW, tH);
        else // circle
            ctx.arc(tX + tW / 2, tY + tH / 2, tW / 3, 0, Math.PI * 2, false);
        ctx.stroke();
        ctx.restore();
    };
    OU.util.TileViewer.prototype.Thumbnail.prototype.move = function(x, y) {
        var tileViewer = this.tileViewer;

        this.tX = x;
        this.tY = y;
        tileViewer.x = -((x - this.x) / this.thumbScale) * this.windowScale;
        tileViewer.y = -((y - this.y) / this.thumbScale) * this.windowScale;
        tileViewer.doRender = true;
    };
    OU.util.TileViewer.prototype.Thumbnail.prototype.isHit = function(x, y, state) {
        //TODO handle movement of the thumbnail
        var dX, dY;

        if (state) {
            if (this.drag) {
                dX = x - this.startDragX;
                dY = y - this.startDragY;
                if ((dX !== 0 || dY !== 0) && !isNaN(dX) && !isNaN(dY)) {
                    this.move(this.tX + dX, this.tY + dY);
                    this.startDragX = x;
                    this.startDragY = y;
                }
            }
            else {
                this.drag = true;
                this.startDragX = x;
                this.startDragY = y;
            }
        }
        else {
            if (this.drag) {
                this.drag = false;
                if (x > 0 || y > 0) { // don't trigger move on 'touch up' with crazy values
                    dX = x - this.startDragX;
                    dY = y - this.startDragY;
                    if ((dX !== 0 || dY !== 0) && !isNaN(dX) && !isNaN(dY)) {
                        this.move(this.tX + dX, this.tY + dY);
                    }
                }
            }
        }
    };
    OU.util.TileViewer.prototype.Thumbnail.prototype.resize = function(params) {
        params.x = params.x === undefined ? this.x : params.x;
        params.y = params.y === undefined ? this.y : params.y;
        params.w = params.w === undefined ? this.w : params.w;
        params.h = params.h === undefined ? this.h : params.h;
        var aspectRatio = this.tileViewer.uncroppedW / this.tileViewer.uncroppedH,
                fitWidthScale = params.w / this.tileViewer.uncroppedW,
                fitHeightScale = params.h / this.tileViewer.uncroppedH;
        if (fitWidthScale < fitHeightScale) { // fit width
            this.w = params.w;
            this.h = this.w / aspectRatio;
            this.x = params.x;
            this.y = params.y + (params.h - this.h) / 2;
        }
        else { // fit height
            this.h = params.h;
            this.w = this.h * aspectRatio;
            this.x = params.x + (params.w - this.w) / 2;
            this.y = params.y;
        }
    };
    this.resize(params);
};