/**
 * @fileOverview Gallery - New slideshow activity
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

// Load in any util elements that are required
OU.require('OU.util.Div');

/**
 * @class EmptyActivity - A template for activities that extend the OU.util.Activity class
 * @extends OU.util.Activity
 * @param {Object} data - Holds the data content for a specific instance of the activity
 * @param {String} instance - (Optional) A unique identifier name for this instance, defaults to 'a1'
 * @param {OU.util.Controller} controller - (Optional) A reference to the controller that initialised this instance, if undefined, the superclass will generate a new controller and create the reference to it as 'this.controller'
 */
OU.activity.Gallery = function(data, instance, controller) {
    var currentSlide = 0, slides, transition, hiddenCaptions, captionsStartOpen, swipeListenerIdx, keyListener, gallery = this, progressBar;
    var SLIDE_Z = 100, PROGRESS_Z = 115, ARROW_Z = 200, CAPTION_Z = 210;
    /**
     * canvasView - start of activity
     */
    OU.activity.Gallery.prototype.canvasView = function() {
        var self = this;
        // insert a full window div to catch events in Chrome, otherwise "off-image" events don't trigger
        this.bgDiv = new OU.util.Div({
        });
        progressBar = new this.ProgressBar({
            container: this,
            position: 0,
            positions: this.data.activityContent.length,
            progressBackground: this.data.activityOptions.progressBackground,
            progressHighlight: this.data.activityOptions.progressHighlight,
            progressBorder: this.data.activityOptions.progressBorder
        });

        if (this.data.activityOptions.nextArrow !== undefined && this.data.activityOptions.previousArrow !== undefined) {
            this.showArrows = true;
            this.renderArrows();
        }
        this.initSlides();
        this.animate();
        swipeListenerIdx = OU.events.addSwipeListener(function(events) {
            if (events.dx > 0) {
                self.movePrevious();
            }
            else {
                self.moveNext();
            }
        });
        keyListener = OU.events.addKeyListener(function(key) {
            switch (key) {
                case 37: // LEFT
                    self.movePrevious();
                    break;
                case 39: // RIGHt
                    self.moveNext();
                    break;
                case 38: // UP
                case 40: // down
                    toggleCaption(slides[currentSlide]);
                    break;
            }
        });
        OU.events.addListener(document.body, function(e, events) {
            var pos = events.startPosition;
            if (pos.pageX > self.w / 2) {
                //tap right side of gallery
                self.moveNext();
            }
            else {
                // tap left side
                self.movePrevious();
            }
        });
    };
    OU.activity.Gallery.prototype.renderArrows = function() {
        if (this.showArrows) {
            if (this.nextArrowDiv) {
                this.nextArrowDiv.resize({
                    x: this.w * .75,
                    y: 0,
                    w: this.w * .25,
                    h: this.h
                });
            }
            else {
                this.nextArrowDiv = new OU.util.Div({
                    x: this.w * .75,
                    y: 0,
                    w: this.w * .25,
                    h: this.h,
                    zIndex: ARROW_Z,
                    htmlClass: 'mousePoint'
                });
            }
            this.nextArrowDiv.div.style.backgroundImage = 'url("' + this.dataDir + this.data.activityOptions.nextArrow.image + '")';
            this.nextArrowDiv.div.style.backgroundPosition = 'center right';
            this.nextArrowDiv.div.style.backgroundRepeat = 'no-repeat';

            if (this.previousArrowDiv) {
                this.previousArrowDiv.resize({
                    x: 0,
                    y: 0,
                    w: this.w / 4,
                    h: this.h
                });
            }
            else {
                this.previousArrowDiv = new OU.util.Div({
                    x: 0,
                    y: 0,
                    w: this.w / 4,
                    h: this.h,
                    zIndex: ARROW_Z,
                    htmlClass: 'mousePoint'
                });
            }
            this.previousArrowDiv.div.style.backgroundImage = 'url("' + this.dataDir + this.data.activityOptions.previousArrow.image + '")';
            this.previousArrowDiv.div.style.backgroundPosition = 'center left';
            this.previousArrowDiv.div.style.backgroundRepeat = 'no-repeat';
        }
    };
    OU.activity.Gallery.prototype.initSlides = function() {
        var slide, i,
                offscreen = this.h + 10;
        hiddenCaptions = (this.data.activityOptions.hiddenCaptions === true);
        captionsStartOpen = this.data.activityOptions.captionsStartOpen;
        transition = this.data.activityOptions.transition;
        slides = this.data.activityContent;
        for (i = slides.length; i--; ) {
            slide = slides[i];
            this.scaleSlide(slide);
            slide.div = new OU.util.Div({
                x: this.w,
                y: 0,
                w: slide.w,
                h: this.h,
                htmlClass: 'fastease',
                zIndex: SLIDE_Z,
                style: "background: url('" + this.dataDir + slide.image + "') center center no-repeat;background-size:contain;"
            });
            slide.div.id = 'captiondiv';
            if (slide.caption) {
                slide.captionDiv = document.createElement('div');
                slide.captionDiv._slide = slide;
                slide.captionHidden = hiddenCaptions && !captionsStartOpen;
                slide.captionDiv.setAttribute('class', 'galleryCaption mousePoint');
                slide.captionDiv.style.opacity = 0;
                slide.captionDiv.style.margin = offscreen + 'px 0 0 0';
                slide.captionDiv.style.zIndex = CAPTION_Z;
                slide.captionDiv.innerHTML = "<div></div>";
                slide.div.appendChild(slide.captionDiv);
                setCaption(slide);
                if (hiddenCaptions) {
                    var captionContents = slide.captionDiv.firstElementChild || slide.captionDiv;
                    OU.events.addListener(captionContents, function(event, events, obj) {
                        toggleCaption(obj.parentNode._slide);
                    });
                    OU.events.addOption(captionContents, 'stopPropagation', true);
                }
            }
        }
    };
    var resizeCaption = function(slide) {
        if (slide.captionDiv) {
            setTimeout(function() {
                slide.captionDiv.setAttribute('style', 'height:auto;'); // reset to auto so we can calculate the actual content height
                var captionH = slide.captionDiv.firstChild.clientHeight, margin, maxHeight = (slide.h / 4) | 0;
                captionH = (captionH > maxHeight ? maxHeight : captionH) - 0;
                margin = gallery.h - captionH - 15;
                slide.captionDiv.setAttribute('style', 'height:' + captionH + 'px;');
                slide.captionDiv.style.margin = margin + 'px 0 0 0';
                slide.captionDiv.style.opacity = 1;
                slide.captionDiv.style.zIndex = CAPTION_Z;
                if (slide.captionHidden) {
                    slide.captionDiv.style.overflow = 'hidden';
                }
                else {
                    slide.captionDiv.style.overflow = 'auto';
                }

            }, 100);
        }
    };
    var setCaption = function(slide) {
        var divObj = slide.captionDiv.firstElementChild;
        if (slide.captionHidden) {
            divObj.innerHTML = '<span style="background: rgba(255,255,255,0.7);"> ≡ </span>';
        }
        else {
            divObj.innerHTML = '<div style="background: rgba(255,255,255,0.7)">' + slide.caption + '</div>';
        }
        resizeCaption(slide);
    };
    var toggleCaption = function(slide) {
        if (hiddenCaptions && slide) { // only toggle if captions started off hidden
            slide.captionHidden = !slide.captionHidden;
            setCaption(slide);
        }
    };
    OU.activity.Gallery.prototype.ProgressBar = function(params) {
        var position = params.position || 0,
                positions = params.positions || 1,
                height = 8,
                progressBg = params.progressBackground || 'rgba(255,255,255,0.4)',
                progressHighlight = params.progressHighlight || 'rgba(50,50,50,0.8)',
                progressBorder = params.progressBorder || '#666',
                div = new OU.util.Div({
                    x: params.container.x + params.container.w * .1,
                    y: params.container.y + params.container.h - height,
                    w: params.container.w * .8,
                    h: height,
                    zIndex: PROGRESS_Z
                });
        var render = function() {
            var i, h = '', spanWidth = (100 / positions) | 0;
            for (i = 0; i < positions; i++) {
                if (i !== position) {
                    h = h + '<div style="background: ' + progressBg + '; width:' + spanWidth + '%;height:10px; float:left;"></div>';
                }
                else {
                    h = h + '<div style="background: ' + progressHighlight + '; width:' + spanWidth + '%;height:10px; float:left;"></div>';
                }
            }
            div.html('<div style="height:' + height + 'px;border: 1px solid ' + progressBorder + '">' + h + '</div>');
        };

        OU.activity.Gallery.prototype.ProgressBar.prototype.setPosition = function(newPos) {
            if (newPos >= 0 && newPos < positions) {
                position = newPos;
                render();
            }
        };
        OU.activity.Gallery.prototype.ProgressBar.prototype.resize = function(container) {
            div.resize({
                x: container.x + container.w * .1,
                y: container.y + container.h - height,
                w: container.w * .8,
                h: height
            });
            render();
        };
        render();
    };
    OU.activity.Gallery.prototype.movePrevious = function() {
        if (currentSlide > 0) {
            currentSlide--;
            this.animate();
        }
    };
    OU.activity.Gallery.prototype.moveNext = function() {
        if (currentSlide < slides.length - 1) {
            currentSlide++;
            this.animate();
        }
    };
    OU.activity.Gallery.prototype.animate = function() {
        var slide, i;

        for (i = slides.length; i--; ) {
            slide = slides[i];
            switch (transition) {
                case 'slide':
                    if (i === currentSlide) {
                        slide.div.resize({
                            x: ((this.w - slide.w) / 2),
                            w: slide.w
                        });
                    }
                    else if (i < currentSlide) {
                        slide.div.resize({
                            x: -this.w
                        });
                    }
                    else {
                        slide.div.resize({
                            x: this.w
                        });
                    }
                    break;
                case 'fade':
                    slide.div.transform({
                        translate: {
                            x: (this.w - slide.w) / 2
                        }
                    });
                    if (i === currentSlide) {
                        slide.div.opacity(1.0);
                    }
                    else {
                        slide.div.opacity(0.0);
                    }
                    break;
            }
        }
        progressBar.setPosition(currentSlide);
        this.showHideArrows();
    };
    OU.activity.Gallery.prototype.showHideArrows = function() {
        if (this.showArrows) {
            if (currentSlide > 0) {
                this.previousArrowDiv.div.style.display = 'block';
            }
            else {
                this.previousArrowDiv.div.style.display = 'none';
            }
            if (currentSlide < slides.length - 1) {
                this.nextArrowDiv.div.style.display = 'block';
            }
            else {
                this.nextArrowDiv.div.style.display = 'none';
            }
        }
    };
    OU.activity.Gallery.prototype.scale = function() {
        var slide, i;
        for (i = slides.length; i--; ) {
            slide = slides[i];
            this.scaleSlide(slide);
            slide.div.resize({
                x: 0,
                y: 0,
                w: slide.w,
                h: this.h //slide.h
            });
            resizeCaption(slide);
        }
    };
    OU.activity.Gallery.prototype.scaleSlide = function(slide) {
        var fitWidthScale = this.w / slide.width, fitHeightScale = this.h / slide.height;
        if (fitWidthScale > fitHeightScale) {
            slide.scale = fitHeightScale;
        }
        else {
            slide.scale = fitWidthScale;
        }
        slide.w = slide.width * slide.scale;
        slide.h = slide.height * slide.scale;
    };
    /**
     * resize - called when activity size is changed
     */
    OU.activity.Gallery.prototype.resize = function() {
        OU.activity.Gallery.superClass_.resize.call(this); // call the superclass resize

        this.bgDiv.resize({
            w: this.w,
            h: this.h
        });
        if (slides !== undefined) {
            this.scale();
            this.animate();
            this.renderArrows();
            progressBar.resize(this);
        }
        window.scrollTo(0, 0);
        this.showHideArrows();
        document.body.style.width = window.innerWidth;
    };
    OU.activity.Gallery.prototype.remove = function() {
        for (var i = slides.length; i--; ) {
            slides[i].div.remove();
        }
        OU.events.removeSwipeListener(swipeListenerIdx);
        OU.events.removeKeyListener(keyListener);

    };
    // call the superclass's constructor
    OU.base(this, data, instance, controller);
};
// Call our inherits function to implement the class inheritance
OU.inherits(OU.activity.Gallery, OU.util.Activity);
