/**
 * @fileOverview Panorama - display a panorama view
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.DynText');
OU.require('OU.util.Layer');
OU.require('OU.util.Slider');
OU.require('OU.util.TileViewer');
OU.require('OU.util.ModalInfo');
/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.Panorama = function(data, instance, controller) {
    OU.activity.Panorama.prototype.canvasView = function() {
        var bH = OU.controlHeight;
        this.bgLayer = new OU.util.Layer({
            container: this,
            id: 'bg'
        });
        if (this.data.activityContent.backgroundColour !== undefined)
            this.bgLayer.context.gradRect({
                col1: this.data.activityContent.backgroundColour,
                col2: this.data.activityContent.backgroundColour
            }); // use specified background colour
        else
            this.bgLayer.context.gradRect(); // use default background
        this.scopeLayer = new OU.util.Layer({
            container: this,
            id: 'canvas',
            x: this.x,
            y: this.y,
            h: this.h - bH,
            hasEvents: true,
            pinch: this.pinch,
            pinchMe: this
        });
        this.controlsLayer = new OU.util.Layer({
            container: this,
            id: 'oucontrols',
            x: this.x,
            y: this.y + this.h - bH,
            h: bH,
            hasEvents: true
        });
        this.showHotspots = this.data.activityContent.showHotspots;
        this.tileView = new OU.util.TileViewer({
            imageBase: this.data.activityContent.images,
            tileSize: this.data.activityContent.tileSize || 256,
            useTileGroups: this.data.activityContent.useTileGroups,
            rotationImage: this.data.activityContent.rotationImage,
            markerCallback: this.loadHotspot,
            hotspotOutlineColour: this.data.activityContent.hotspotOutlineColour,
            zoomLevels: this.data.activityContent.zoomLevels,
            renderAngle: this.data.activityContent.renderAngle,
            horizontalStitch: this.data.activityContent.horizontalLoop === false ? false : true,
            useGoogleMapsFileConvention: this.data.activityContent.useGoogleFileConvention,
            window: {
                w: this.w,
                h: this.h - bH
            },
            image: this.data.activityContent.imageDims,
            crop: this.data.activityContent.crop,
            container: this,
            context: this.scopeLayer.context,
            zoomCallback: this.setSlider
        });
        this.scopeLayer.events.clickable.push(this.tileView);
        this.initControls();
    };
    OU.activity.Panorama.prototype.resize = function() {
        OU.activity.Panorama.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight, bW = 0;
        this.bgLayer.resize();
        if (this.data.activityContent.backgroundColour !== undefined) {
            this.bgLayer.context.gradRect({
                col1: this.data.activityContent.backgroundColour,
                col2: this.data.activityContent.backgroundColour
            }); // use specified background colour
        }
        else {
            this.bgLayer.context.gradRect(); // use default background
        }
        this.scopeLayer.resize({
            h: this.h - bH
        });
        this.controlsLayer.resize({
            x: this.x,
            y: this.y + this.h - bH,
            h: bH
        });
        if (this.showHotspots && this.hotspotsButton) {
            this.hotspotsButton.render({
                x: this.w - bH * 5,
                y: 0,
                w: bH * 5,
                h: bH
            });
            bW = bH * 5;
        }
        this.zoomSlider.resize({
            x: 0,
            y: 0,
            w: this.w - bW,
            h: bH
        });
        this.tileView.resize({
            w: this.w,
            h: this.h - bH
        });
        this.tileView.render();
        if (OU.modalInfo) {
            OU.modalInfo.remove();
        }
    };
    OU.activity.Panorama.prototype.remove = function() {
        OU.activity.Panorama.superClass_.remove.call(this); // call the superclass method
        if (OU.modalInfo) {
            OU.modalInfo.remove();
        }
    };
    OU.activity.Panorama.prototype.initControls = function() {
        var ctx = this.controlsLayer.context, bH = OU.controlHeight,
                clickable = this.controlsLayer.events.clickable, bW = 0, self = this;

        if (this.showHotspots) {
            if (this.hotspotsButton) {
                this.hotspotsButton.render();
            }
            else {
                this.hotspotsButton = new OU.util.CheckBoxButton({
                    txt: 'Show hotspots',
                    x: this.w - bH * 5,
                    y: 0,
                    w: bH * 5,
                    h: bH,
                    layer: this.controlsLayer,
                    state: false,
                    tabIndex: 10,
                    onClick: function() {
                        var newState = !self.tileView.showHotspots;
                        self.tileView.showHotspots = newState;
                        self.tileView.render();
                        self.hotspotsButton.state(newState);
                        self.hotspotsButton.render(newState);
                    }
                });
            }
            bW = bH * 5;
        }
        if (this.zoomSlider) {
            this.zoomSlider.resize({
                x: 0,
                y: 0,
                w: this.w - bW,
                h: bH
            });
        }
        else {
            this.zoomSlider = new OU.util.Slider({
                container: this,
                sliderPos: 0,
                instance: 'zoom' + this.instance,
                x: 0,
                y: 0,
                w: this.w - bW,
                h: bH,
                sliderHeight: bH / 2,
                drawContainer: false,
                callback: this.setZoom,
                callbackParam: this,
                background: {
                    clear: true
                },
                context: ctx
            });
            clickable.push(this.zoomSlider);
        }
        this.renderSlider();
    };
    OU.activity.Panorama.prototype.renderControls = function() {
        this.controlsLayer.clear();
        this.accessibleButton.render();
    };
    OU.activity.Panorama.prototype.renderSlider = function() {
        var self = this, s = this.zoomSlider;
        if (this.controlsLayer.context) {
            this.controlsLayer.context.clearRect(0, 0, s.w, s.h);
            s.render();
            setTimeout(function() {
                self.renderSlider();
            }, 40);
        }
    };
    OU.activity.Panorama.prototype.setZoom = function(val, me) { // Called when scrubBar is moved
        me.zoomSlider.render();
        me.tileView.scale(val);
    };
    OU.activity.Panorama.prototype.loadHotspot = function(hotspot, me, event) { // called when hotspot hit
        me.scopeLayer.events.touched = false;
        me.scopeLayer.events.pressed = false;
        if (hotspot.popupActivity !== undefined) {
            me.controller.addActivity(hotspot.popupActivity);
        }
        if (hotspot.modalInfo) {
            if (!OU.modalInfo) {
                OU.modalInfo = new OU.util.ModalInfo();
            }
            console.log(event);
            OU.modalInfo.view(hotspot.modalInfo, event);
        }
    };
    OU.activity.Panorama.prototype.pinch = function(e, s, x, y, dx, dy, me) { // called when pinch event detected
        var ns = ((e - s) / me.h) + me.tileView.s;
        ns = ns > 1 ? 1 : (ns < 0 ? 0 : ns);
        me.tileView.scale(ns, {
            x: x,
            y: y
        });
        me.zoomSlider.sliderPos = ns;
        me.zoomSlider.render();
    };
    OU.activity.Panorama.prototype.setSlider = function(val, me) { // Called when TileViewer changed via mousewheel, etc.
        me.zoomSlider.setTarget(val);
    };

    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.Panorama, OU.util.Activity);
