/**
 * @fileOverview TableGraph - renders a table and or graph for a specified dataset(s)
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.PopUpInfo');
OU.require('OU.util.Layer');
OU.require('OU.util.Div');
OU.require('OU.util.Instruction');
/** 
 * @class
 * @extends OU.util.Activity
 */
OU.activity.TableGraph = function(data,instance,controller) {

    OU.activity.TableGraph.prototype.canvasView = function() {
        OU.obj[instance] = this; // push this object into the OU object array using it's instance name
        this.config = {
            depth: 5,
            fps:120 // 40ms = 25 fps
        };

        // create Canvas Layers & Contexts
        this.bgLayer = new OU.util.Layer({
            container:this
        });
        this.bgLayer.context.gradRect(); // draw background on backdrop layer

        this.PAD = this.h*.045;

        this.yTics={};
        this.xTics={};

        this.bestFitOn=this.data.settings.straightLineFit || false;

        // Set dataset to 1st and render
        this.initDatasets();
        this.changeDataset(OU.LocalStorage.load("OU.TableGraph.dataset") || 0);
    };
    OU.activity.TableGraph.prototype.resize = function() {
        OU.activity.TableGraph.superClass_.resize.call(this); // call the parent class resize 
        var bH = OU.controlHeight,bX=bH*3;

        this.bgLayer.resize();
        this.bgLayer.context.gradRect();
        this.PAD = this.h*.045;

        this.graphArea = {
            x: this.PAD*3,
            y: this.PAD*2,
            w: this.w - this.PAD*4,
            h: this.h - this.PAD*5
        };
        this.tableArea = {
            x: bH,
            y: bH,
            w: this.w-2*bH,
            h: this.h-2*bH
        };

        if(this.view=='graph') {
            this.modelLayer.resize({});
            this.graphControlLayer.resize({});

            this.initControls();

            this.closeButton.resize({
                x:this.w-bX-bH*2,
                y:this.h-bH,
                w:bH*4,
                h:bH
            });
            bX=bX+bH*4;
            if(this.data.settings.buttons.straightLine) {
                this.straightLineButton.resize({
                    x:this.w-bX-bH*2,
                    y:this.h-bH,
                    w:bH*4,
                    h:bH
                });
                bX=bX+bH*4;
            }
        }
        if(this.tableDiv!==undefined) {
            this.tableDiv.resize({
                x: this.tableArea.x,
                y: this.tableArea.y,
                w: this.tableArea.w,
                h: this.tableArea.h
            });
        }

        this.render();
        this.renderControls();
    };
    OU.activity.TableGraph.prototype.initDatasets = function() {
        var ds,i;
        this.datasets = [];
        for(i=0; i< this.data.datasets.length; i++) {
            ds=this.data.datasets[i];
            ds.container=this;
            this.datasets.push(new this.Dataset(ds));
        }
    };
    OU.activity.TableGraph.prototype.chooseDataset = function() {
        var i,self=this,infoText,
        currentReaction=1*(OU.LocalStorage.load('ou.tableGraph.dataset') || 0),
        cellWidthTotal=this.w*.6*.8;

        infoText='<form name="pieform">';
        infoText+='<table width="'+cellWidthTotal+'" border="0"><tr><td width="'+cellWidthTotal*.8+'" valign="top">';

        for (i=0; i<this.datasets.length;i++){
            if (i==currentReaction){
                infoText+='<input type="radio" name="group1" id="radiofield'+i+'" value="'+i+'" checked="checked" onclick="OU.LocalStorage.save(\'ou.tableGraph.dataset\','+i+')" /><label for="radiofield'+i+'"> '+this.datasets[i].name+'</label>';
            }else{
                infoText+='<input type="radio" name="group1" id="radiofield'+i+'" value="'+i+'" onclick="OU.LocalStorage.save(\'ou.tableGraph.dataset\','+i+')" /><label for="radiofield'+i+'"> '+this.datasets[i].name+'</label>';
            }
            if(i!=this.datasets[i].length-1){
                infoText+='<br/>';
            }
        }
        infoText+='</td></tr>';
        infoText+='</table>';
        infoText+='</form>';

        new OU.util.PopUpInfo({
            container: this,
            txt:infoText,
            x:this.w*.20,
            y:this.h*.15,
            w:this.w*.6,
            h:this.h*.7,
            onClose: function() {
                self.changeDataset();
            }
        });
    };
    OU.activity.TableGraph.prototype.changeDataset = function() {
        this.dataset = this.datasets[OU.LocalStorage.load('ou.tableGraph.dataset') || 0];
        this.resize();
    };
    OU.activity.TableGraph.prototype.resetView = function() {
        var ds = this.dataset,bH=OU.controlHeight;
        this.xScale = (this.w-this.PAD*2)/ds.dx;
        this.yScale = ((this.h-bH)-this.PAD*2)/ds.dy;
        this.resize();
    };
    OU.activity.TableGraph.prototype.xPt = function(v) {
        return this.xOff+(v-this.xMin)*this.xS;
    };
    OU.activity.TableGraph.prototype.yPt = function(v) {
        return this.yOff-(v-this.yMin)*this.yS;
    };
    OU.activity.TableGraph.prototype.render = function() {
        this.renderTable();
        if(this.view=='graph')
            this.renderGraph();
    };
    OU.activity.TableGraph.prototype.renderTable = function() {
        var i,j,row,cell,table = this.dataset.table,h;

        if(table.style!==undefined)
            h='<table style="'+table.style+'">';
        else
            h='<table>';

        if(table.graphColumns!==undefined) {
            h=h+'<tr><th colspan="'+table.headings[0].cols.length+'"><h2><input style="float:right; margin: 0 10px;" type="button" value="Display Graph" onClick="OU.obj[\''+this.instance+'\'].showGraph()"/>'+this.dataset.name;
            if(this.data.datasets.length>1)
                h=h+'<input style="margin: 0 10px;" type="button" value="Change" onClick="OU.obj[\''+this.instance+'\'].chooseDataset()"/>';
            h=h+'</h2></th></tr>';
        }
        if(this.tableDiv===undefined) {
            this.tableDiv = new OU.util.Div({
                x: this.tableArea.x,
                y: this.tableArea.y,
                w: this.tableArea.w,
                h: this.tableArea.h,
                container: this
            });
        }
        for(i=0; i<table.headings.length; i++) {
            row=table.headings[i];
            h=h+'<tr>';
            for(j=0; j<row.cols.length; j++) {
                cell=row.cols[j];
                h = h+'<th>'+cell+'</th>';
            }
            h=h+'</tr>';
        }
        if(table.allowReorder!==undefined || table.calcs!==undefined) {
            row=table.rows[0];
            h=h+'<tr>';
            for(j=0; j<row.cols.length; j++) {
                h = h+'<th>';
                if(table.allowReorder !== undefined && table.allowReorder[j])
                    h = h+'<input type="button" value="\u21d3 Re-order" onClick="OU.obj[\''+this.instance+'\'].reOrder('+j+');"><br/>';
                if(table.calcs!==undefined && table.calcs[j]!==undefined)
                    h = h+'<input type="button" value="Calculate" onClick="OU.obj[\''+this.instance+'\'].calcColumn('+j+');">';
                h = h+'</th>';
            }
            h=h+'</tr>';
        }

        for(i=0; i<table.rows.length; i++) {
            row=table.rows[i];
            h=h+'<tr>';
            for(j=0; j<row.cols.length; j++) {
                cell=row.cols[j];
                h = h+'<td>'+cell+'</td>';
            }
            h=h+'</tr>';
        }
        h=h+'</table>';
        this.tableDiv.div.innerHTML=h;
    };
    OU.activity.TableGraph.prototype.reOrder = function(col) {
        var table = this.dataset.table;
        table.rows.sort(function(a,b){
            return a.cols[col]-b.cols[col];
        });
        this.renderTable();
    };
    OU.activity.TableGraph.prototype.calcColumn = function(col) {
        var ds = this.dataset,i,row,table = this.dataset.table;
        for(i=0; i<table.rows.length; i++) {
            row=table.rows[i];
            table.calcs[col].calc(row);
        }
        this.renderTable();
        if(ds.onOpenGraph!==undefined && ds.onOpenGraph.controllerAction!==undefined) {
            OU.callControl(ds.onOpenGraph.controllerAction);
        }
        if(table.calcs[col].controllerAction!==undefined)
            OU.callControl(table.calcs[col].controllerAction);
    };
    OU.activity.TableGraph.prototype.showGraph = function() {
        var ds = this.dataset, table = ds.table,i,row,incomplete=false;

        row=table.rows[0];
        for(i=table.graphColumns.length; i--;) {
            if(row.cols[table.graphColumns[i]]===undefined || row.cols[table.graphColumns[i]]=='')
                incomplete=true;
        }
        if(incomplete) {
            new OU.util.Instruction({
                message: "Some graph data is missing, make sure you have followed the instructions fully before displaying the graph.",
                container: this
            });
            return;
        }

        this.modelLayer = new OU.util.Layer({
            container: this
        });
        this.modelLayer.context.gradRect();
        this.graphControlLayer = new OU.util.Layer({
            container: this,
            hasEvents: true
        });
        this.view='graph';
        this.renderGraph();
        this.initControls();
        if(ds.onOpenGraph!==undefined && ds.onOpenGraph.controllerAction!==undefined)
            OU.callControl(ds.onOpenGraph.controllerAction);

    };
    OU.activity.TableGraph.prototype.closeGraph = function() {
        var ds = this.dataset;
        this.graphControlLayer.remove();
        this.modelLayer.remove();
        if(this.closeButton)
            this.closeButton.remove();
        if(this.straightLineButton)
            this.straightLineButton.remove();
        this.closeButton=null;
        this.straightLineButton=null;
        this.view='table';
        if(ds.onCloseGraph!==undefined && ds.onCloseGraph.controllerAction!==undefined)
            OU.callControl(ds.onCloseGraph.controllerAction);
    };
    OU.activity.TableGraph.prototype.renderGraph = function() {
        var ctx = this.modelLayer.context,x,y,j,
        circ=Math.PI*2,failed=false,
        n,sumXY=0,sumX=0,sumX2=0,sumY=0,a,b,
        ds = this.dataset,
        r = this.w/200;
        this.xOff = this.graphArea.x;
        this.yOff = this.graphArea.y+this.graphArea.h;

        ds.calcPoints();

        this.xS = ds.xScale;
        this.yS = ds.yScale;
        this.xMin = ds.x1;
        this.yMin = ds.y1;
        this.gxMin = this.xPt(ds.x1);
        this.gxMax = this.xPt(ds.x2);
        this.gyMin = this.yPt(ds.y1);
        this.gyMax = this.yPt(ds.y2);

        this.renderFrame();

        ctx.save();
        ctx.lineWidth=1;
        n=ds.points.length;

        sumXY=0;
        sumX=0;
        sumX2=0;
        sumY=0;

        for(j=n; j--;) {
            x=ds.points[j].x;
            y=ds.points[j].y;
            if(x===undefined || y===undefined || isNaN(x) || isNaN(y)) {
                failed=true;
            }
            else {
                if(this.bestFitOn) {
                    sumXY = sumXY+(x*y);
                    sumX = sumX+x;
                    sumX2 = sumX2+x*x;
                    sumY = sumY+y;
                }
                ctx.fillStyle= ds.fillStyle || "#f44";
                ctx.strokeStyle= "#100";
                ctx.lineWidth=1;
                if(x>=ds.x1 && x<=ds.x2 && y>=ds.y1 && y<=ds.y2) {

                    x = this.xPt(x);
                    y = this.yPt(y);

                    if(ds.style=='rect') {
                        ctx.beginPath();
                        ctx.fillRect(x-r,y-r,r*2,r*2);
                        ctx.stroke();
                    }
                    else {
                        ctx.beginPath();
                        ctx.arc(x,y,r,0,circ,false);
                        ctx.fill();
                        ctx.stroke();
                    }//*/
                }
            }
        }
        if(!failed && this.bestFitOn) {
            a = (n*sumXY - sumX*sumY)/(n*sumX2 - sumX*sumX);
            b = (sumY-a*sumX)/n;
            x=ds.x1;
            y=x*a+b;
            if(y<ds.y1 ) {
                y=ds.y1;
                x= (y-b)/a;
            }
            else if(y>ds.y2) {
                y = ds.y2;
                x= (y-b)/a;
            }
            x = this.xPt(x);
            y = this.yPt(y);
            ctx.beginPath();
            ctx.moveTo(x,y);
            x=ds.x2;
            y=x*a+b;
            if(y<ds.y1 ) {
                y=ds.y1;
                x= (y-b)/a;
            }
            else if(y>ds.y2) {
                y = ds.y2;
                x= (y-b)/a;
            }
            x = this.xPt(x);
            y = this.yPt(y);
            ctx.lineTo(x,y);
            ctx.strokeStyle=ds.fillStyle || '#44f';
            ctx.stroke();
        }
        ctx.restore();
        if(failed) {
            new OU.util.Instruction({
                container: this,
                message: this.data.incompleteMessage || 'Some of the values are missing, please ensure you have completed the relevant table.'
            });
        }
    };
    OU.activity.TableGraph.prototype.isNum = function(n) {
        return !isNaN(parseFloat(n)) && isFinite(n);
    };
    OU.activity.TableGraph.prototype.renderFrame = function() {
        var ctx = this.modelLayer.context,
        ds = this.dataset,
        table = ds.table,xHeading,yHeading,
        xAxis,yAxis,i,j,val,
        xS=ds.xScale,
        yS=ds.yScale,
        PAD = this.PAD,
        xOff = this.graphArea.x;

        ctx.gradRect();
        ctx.save();
        ctx.fillStyle='#fff';
        ctx.fillRect(this.xPt(this.xMin)/*this.graphArea.x*/,this.graphArea.y,this.graphArea.w,this.graphArea.h);
        ctx.textAlign='right';
        ctx.lineWidth=0.7;
        yAxis = -ds.x1*xS + this.graphArea.x;
        yAxis = yAxis<xOff?xOff:(yAxis>this.graphArea.x+this.graphArea.w?this.graphArea.x+this.graphArea.w:yAxis);

        for(i=this.yTics.min;i<=this.yTics.max;i+=this.yTics.interval) {
            var yy=PAD+(ds.y2-i)*yS;
            ctx.beginPath();
            ctx.strokeStyle="#000";
            ctx.moveTo(this.gxMin-5,yy);
            ctx.lineTo(this.gxMin,yy);
            ctx.closePath();
            ctx.stroke();
            ctx.lineWidth=0.7;
            ctx.strokeStyle="#777";
            ctx.beginPath();
            ctx.moveTo(this.gxMin,yy);
            ctx.lineTo(this.gxMax,yy);
            ctx.closePath();
            ctx.stroke();
            val=i.nDecimals(ds.table.nDecimals || 3);
            ctx.fillStyle="#000";
            ctx.fillText(val,yAxis-10,yy);
            for(j=i+this.yTics.subInterval;j<i+this.yTics.interval;j+=this.yTics.subInterval){
                yy=this.yPt(j);
                ctx.strokeStyle="#aaa";
                ctx.lineWidth=0.3;
                ctx.beginPath();
                ctx.moveTo(this.gxMin,yy);
                ctx.lineTo(this.gxMax,yy);
                ctx.closePath();
                ctx.stroke();
            }
        }
        xAxis = ds.y2*yS +this.graphArea.x;
        xAxis = xAxis<this.graphArea.y?this.graphArea.y:(xAxis>this.graphArea.y+this.graphArea.h?this.graphArea.y+this.graphArea.h:xAxis);
        ctx.textAlign='center';
        for(i=this.xTics.min;i<=this.xTics.max;i+=this.xTics.interval) {
            var xx=this.xPt(i);
            ctx.beginPath();
            ctx.strokeStyle="#000";
            ctx.moveTo(xx,this.gyMin);
            ctx.lineTo(xx,this.gyMin+5);
            ctx.closePath();
            ctx.stroke();
            ctx.lineWidth=0.7;
            ctx.strokeStyle="#777";
            ctx.beginPath();
            ctx.moveTo(xx,this.gyMin);
            ctx.lineTo(xx,this.gyMax);
            ctx.closePath();
            ctx.stroke();
            val=i.nDecimals(ds.table.nDecimals || 3);
            ctx.fillStyle="#000";
            ctx.fillText(val,xx,this.gyMin+10);
            for(j=i+this.xTics.subInterval;j<i+this.xTics.interval;j+=this.xTics.subInterval){
                xx=this.xPt(j);
                ctx.lineWidth=0.3;
                ctx.strokeStyle="#aaa";
                ctx.beginPath();
                ctx.moveTo(xx,this.gyMin);
                ctx.lineTo(xx,this.gyMax);
                ctx.closePath();
                ctx.stroke();
            }
        }
        ctx.strokeStyle="#000";
        ctx.lineWidth=0.7;
        ctx.fillStyle="#444";
        ctx.beginPath();
        ctx.moveTo(yAxis,this.graphArea.y);
        ctx.lineTo(yAxis,this.graphArea.y+this.graphArea.h);
        ctx.lineTo(yAxis+this.graphArea.w,this.graphArea.y+this.graphArea.h);
        ctx.lineTo(yAxis+this.graphArea.w,this.graphArea.y);
        ctx.lineTo(yAxis,this.graphArea.y);
        ctx.closePath();
        ctx.stroke();
        ctx.restore();

        xHeading = table.headings[0].cols[table.graphColumns[0]];
        yHeading = table.headings[0].cols[table.graphColumns[1]];
        new OU.util.DynText({
            txt: ds.name,
            container: this,
            context: ctx,
            background: {
                col: '#EBF4FB',
                borderCol: '#444'
            },
            x: this.graphArea.x,
            y: this.graphArea.y-this.PAD,
            w: this.graphArea.w,
            h: this.PAD
        });
        new OU.util.DynText({
            txt: xHeading,
            container: this,
            context: ctx,
            align:'center',
            x: this.graphArea.x,
            y: this.graphArea.y+this.graphArea.h+this.PAD,
            w: this.graphArea.w,
            h: this.PAD
        });
        ctx.save();
        ctx.translate(this.graphArea.x,this.graphArea.y+this.graphArea.h);
        ctx.rotate(-Math.PI/2);
        new OU.util.DynText({
            txt: yHeading,
            container: this,
            context: ctx,
            align:'center',
            x: 0,
            y: -this.graphArea.x,
            w: this.graphArea.h,
            h: this.graphArea.x-this.PAD
        });
        ctx.restore();

        if(this.bestFitOn && ds.bestFitText!==undefined) {
            new OU.util.DynText({
                txt: ds.bestFitText,
                container: this,
                context: ctx,
                background: {
                    col: 'rgba(255,255,255,0.5'
                },
                x: this.graphArea.x+this.graphArea.w/4,
                y: this.graphArea.y+this.PAD/2,
                w: this.graphArea.w/2,
                h: this.PAD
            });
        }
    };
    OU.activity.TableGraph.prototype.calcTics = function(mn,mx){
        ///  ticInterval calculation thanks to M.A.Brown
        var dif=mx-mn,n = Math.floor(Math.log(dif)/Math.log(10.0)),
        a = dif*Math.pow(10.0, -n),ticInterval, dt=5;
        if (a<2.0){
            ticInterval = 0.2;
            dt=2;
        }else if(a<5.0) ticInterval = 0.5;
        else ticInterval = 1.0;
        ticInterval=ticInterval*Math.pow(10.0, n);
        return {
            interval: ticInterval,
            min:Math.floor(mn/ticInterval)*ticInterval,
            max: Math.ceil(mx/ticInterval)*ticInterval,
            subInterval: ticInterval/dt
        }
    };
    OU.activity.TableGraph.prototype.initControls = function() {
        var bH=OU.controlHeight,
        clickable,
        self=this,bX=bH*3;

        if(this.view=='graph') {
            clickable = this.graphControlLayer.events.clickable;
            clickable.length=0;
            this.closeButton = new OU.util.ControlButton({
                txt:'Close Graph',
                x:this.w-bX-bH*2,
                y:this.h-bH,
                w:bH*4,
                h:bH,
                tabIndex: 2,
                layer:this.graphControlLayer,
                onClick: function() {
                    self.closeGraph();
                }
            });
            bX=bX+bH*6;
            if(this.data.settings.buttons.straightLine) {
                this.straightLineButton = new OU.util.CheckBoxButton({
                    txt:'Straight Line Fit',
                    x:this.w-bX-bH*2,
                    y:this.h-bH,
                    w:bH*6,
                    h:bH,
                    tabIndex: 1,
                    layer:this.graphControlLayer,
                    onOffIndicator:true,
                    state: this.bestFitOn,
                    onClick: function() {
                        self.toggleStraightLine();
                    }
                });
                bX=bX+bH*4;
            }
        }
    };
    OU.activity.TableGraph.prototype.toggleStraightLine = function() {
        var ds=this.dataset;
        this.bestFitOn = !this.bestFitOn;
        this.straightLineButton.state(this.bestFitOn);
        if(this.bestFitOn && this.doneBestFitAudio===undefined) {
            if(ds.onBestFit!==undefined && ds.onBestFit.controllerAction!==undefined)
                OU.callControl(ds.onBestFit.controllerAction);
            this.doneBestFitAudio=true;
        }
        this.renderGraph();
        this.renderControls();
    };
    OU.activity.TableGraph.prototype.renderControls = function() {
        if(this.view=='graph') {
            this.graphControlLayer.clear();
            this.closeButton.render();
            if(this.data.settings.buttons.straightLine) {
                this.straightLineButton.render();
            }
        }
    };
    OU.activity.TableGraph.prototype.Dataset = function(params) {
        var t;
        this.name = params.name || 'Dataset';
        this.container = params.container;
        this.table = params.table || {};
        this.bestFitText=params.bestFitText || undefined;
        this.x1 = params.x1 || 0;
        this.x2 = params.x2 || 10;
        this.startY = params.startY || 10;
        this.x2 = params.x2 || 10;
        this.variables = params.variables || {};
        this.onOpenGraph = params.onOpenGraph;
        this.onCloseGraph = params.onCloseGraph;
        this.onBestFit = params.onBestFit;
        this.instructions = params.instructions;
        this.xInterval = params.xInterval || 1;
        if(this.x1>this.x2) {
            t = this.x2;
            this.x2=this.x1;
            this.x1=t;
        }
        this.reset_x1 = this.x1;
        this.reset_x2 = this.x2;
        this.reset_y1 = this.y1;
        this.reset_y2 = this.y2;
        if(params.colour!==undefined)
            this.colour = params.colour;
        if(params.rgb!==undefined)
            this.rgb = params.rgb;
        OU.activity.TableGraph.prototype.Dataset.prototype.tableToPoints = function() {
            var i,table=this.table,row,x,y,dx,dy,points=[];
            this.x1=10000;
            this.x2=-10000;
            this.y1=10000;
            this.y2=-10000;
            for(i=table.rows.length; i--;) {
                row=table.rows[i];
                x = row.cols[table.graphColumns[0]];
                y = row.cols[table.graphColumns[1]];
                if(this.container.isNum(x) && this.container.isNum(y))
                    points.push({
                        x:x,
                        y:y
                    });
                if(x<this.x1)
                    this.x1=x;
                if(x>this.x2)
                    this.x2=x;
                if(y<this.y1)
                    this.y1=y;
                if(y>this.y2)
                    this.y2=y;
            }
            dx=this.x2-this.x1;
            dy=this.y2-this.y1;
            this.x1 = this.x1 - dx*.1;
            this.x2 = this.x2 + dx*.1;
            this.y1 = this.y1 - dy*.1;
            this.y2 = this.y2 + dy*.1;
            //*/
            return points;
        };
        OU.activity.TableGraph.prototype.Dataset.prototype.calcPoints = function() {
            var i,params={},v,y,c=this.container;

            for(i=this.variables.length; i--;) {
                v = this.variables[i];
                params[v.varName]=v.value;
            }
            params.x1=this.x1;
            params.x2=this.x2;
            params.xInterval=this.xInterval;
            params.startY=this.startY;

            this.points = this.tableToPoints();

            this.y1=1.e12;
            this.y2=-this.y1;
            for(i=this.points.length; i--;) {
                if(c.isNum(c.yZero))this.points[i].y-=c.yZero;
                y=this.points[i].y;
                if(y>this.y2)
                    this.y2=y;
                if(y<this.y1)
                    this.y1=y;
            }
            if(c.isNum(c.setYMin)){
                this.y1=c.setYMin;
                this.y2=c.setYMax;
            }
            c.yTics=c.calcTics(this.y1,this.y2);
            this.y1=c.yTics.min;
            this.y2=c.yTics.max;
            c.xTics=c.calcTics(this.x1,this.x2);
            this.x1=c.xTics.min;
            this.x2=c.xTics.max;

            this.dx = Math.abs(this.x2-this.x1);
            this.dy = Math.abs(this.y2-this.y1);

            this.xScale = (c.graphArea.w)/this.dx;
            this.yScale = (c.graphArea.h)/this.dy;
        };
    };
    OU.base(this,data,instance,controller);
};
OU.inherits(OU.activity.TableGraph,OU.util.Activity);

