/**
 * @fileOverview Slider - Creates a slider util
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */
/**
 * @class Slider utility, provides vertical or horizontal slide bars, that allow users to
 * manipulate a value on a sliding scale. All values within the slider are a percentage, represented between 0 and 1.
 * As the slider is moved, it feeds back new value information to the Activity via a callback function.
 *
 * @extends OU.util.Tabbable
 *
 * @param {object} params - options:
 * <ul>
 * <li><strong>{canvas.context} context:</strong> (required) Context of the canvas/layer to render to</li>
 * <li><strong>{object} container:</strong> (required) Calling object, typically an OU.util.Activity object</li>
 * <li><strong>{function} callback:</strong> function to call when the slider moves, updates with new value</li>
 * <li><strong>{object} callbackParam:</strong> (optional) data to pass with the new value </li>
 * <li><strong>{string} orientation:</strong> horizontal (default) or vertical</li>
 * <li><strong>{string} title:</strong> (optional) title/label for the slider (Use not recommended - render the label outside the slider)</li>
 * <li><strong>{int} x:</strong> X co-ordinate of the slider</li>
 * <li><strong>{int} y:</strong> Y co-ordinate of the slider</li>
 * <li><strong>{int} w:</strong> Width of the slider</li>
 * <li><strong>{int} h:</strong> Height of the slider</li>
 * <li><strong>{string} titlePosition:</strong> Where to position the label ('above' = above the slider, aligned left, defaults to Left of the slider)</li>
 * <li><strong>{double} scrubWidth:</strong> (optional) width of the visible slider bar (defaults to .8 of the overall width)</li>
 * <li><strong>{string} sliderStyle:</strong> (optional) style of the slider knob, defaults to circle, any other value gives a rectangle</li>
 * <li><strong>{double} sliderHeight:</strong> (optional) height of the slider knob, defaults to .95 of the full height</li>
 * <li><strong>{double} sliderPos:</strong> Starting point for the slider</li>
 * <li><strong>{boolean} drawContainer:</strong> if true, render the slider background (default: true)</li>
 * <li><strong>{canvas.context.background} background:</strong> (optional) background object to render</li>
 * <li><strong>{boolean} showValue:</strong> If true, then the value of the slider is show, defaults to false</li>
 * <li><strong>{string} valueUnits:</strong> Units string to put after the value if displayed</li>
 * <li><strong>{object} range:</strong> Effect value range, used in conjunction with showValue, has following elements:</li>
 * <ul>
 * <li><strong>{double} min:</strong> Starting value of range at 0</li>
 * <li><strong>{double} max:</strong> Ending value  of range at 1</li>
 * <li><strong>{int} nDecimals:</strong> number of decimals to round the value to (default: 3)</li>
 * </ul>
 * </ul>
 */
OU.util.Slider = function(params) {
    this.params = params;
    this.context = params.context;
    this.frames = OU.IS_IPAD ? 2 : 5;
    this.x = params.x || 0;
    this.y = params.y || 0;
    this.w = params.w || 100;
    this.h = params.h || 20;
    this.titlePosition = params.titlePosition;
    this.valueUnits = params.valueUnits;
    this.scrubX = this.x + (this.w / 10);
    this.scrubY = this.y + (this.h / 2);
    this.scrubWidth = params.scrubWidth || this.w - (2 * (this.w / 10));
    this.callback = params.callback || function() {
    };
    this.callbackParam = params.callbackParam || undefined;
    this.sliderStyle = params.sliderStyle || 'circle';
    this.sliderHeight = params.sliderHeight || this.h * .95;
    this.sliderWidth = this.sliderHeight / 2;
    this.orient = params.orientation || 'horizontal';
    this.sliderPos = params.sliderPos || 0;
    this.target = params.sliderPos || 0;
    this.title = params.title;
    this.halted = false;
    this.disabled = false;
    this.showValue = params.showValue || false;
    this.range = params.range || {};
    if (this.range) {
        this.range.dx = this.range.max - this.range.min;
    }
    this.container = params.container || {};
    OU.addRemovable(this, this.container.zOffset || 0);
    if (params.background === undefined) {
        if (this.orient === 'horizontal') {
            this.params.background = '#CCCCCC';
        }
        else {
            this.params.background = 'rgba(26,26,26,0.9)';
        }
    }
    /**
     * Hit function triggers the call back function. Used by the Tabbable class
     * @private
     */
    OU.util.Slider.prototype.hit = function() {
        this.onClick(this.onClickParam);
        return false;
    };
    /**
     * Called by tabbable, to set focus and render with highlight
     * @private
     */
    OU.util.Slider.prototype.focus = function() {
        this.hasFocus = true;
        this.render();
        return false;
    };
    /**
     * Called by tabbable, to unset focus and remove highlight
     * @private
     */
    OU.util.Slider.prototype.blur = function() {
        this.hasFocus = undefined;
        this.render();
        return false;
    };
    /**
     * Removes the slider
     */
    OU.util.Slider.prototype.remove = function() {
        OU.util.Slider.superClass_.remove.call(this); // call the parent class method
        OU.removables.removeType(this.instanceId);
        this.halted = true;
    };
    /**
     * Resizes the sider
     * @param {object} p - new dims:
     * <ul>
     * <li><strong>{int} x:</strong> X co-ordinate of the slider</li>
     * <li><strong>{int} y:</strong> Y co-ordinate of the slider</li>
     * <li><strong>{int} w:</strong> Width of the slider</li>
     * <li><strong>{int} h:</strong> Height of the slider</li>
     * </ul>
     */
    OU.util.Slider.prototype.resize = function(p) {
        this.x = p.x || this.x;
        this.y = p.y || this.y;
        this.w = p.w || this.w;
        this.h = p.h || this.h;
        this.scrubX = this.x + (this.w / 10);
        this.scrubY = this.y + (this.h / 2);
        this.scrubWidth = this.w - (2 * (this.w / 10));
    };
    /**
     * Renders the slider
     * @param {double} sliderPos - (optional) new position to render to
     */
    OU.util.Slider.prototype.render = function(sliderPos) {
        this.sliderPos = sliderPos === undefined ? this.sliderPos : sliderPos;
        this.sliderPos = this.sliderPos > 1 ? 1 : this.sliderPos;
        this.sliderPos = this.sliderPos < 0 ? 0 : this.sliderPos;
        if (this.params.background !== undefined)
            this.context.background(this.params.background, this);
        if (this.orient === 'vertical')
            this.renderVertical();
        else
            this.renderHorizontal();
    };
    /**
     * Render vertical version
     * @private
     */
    OU.util.Slider.prototype.renderVertical = function() {
        var sX = this.x + this.w / 2,
                sYt = this.y + this.h * .1,
                sYb = this.y + this.h * .9,
                sYm = sYt + (sYb - sYt) / 2,
                sH = this.w / 4,
                sW = this.w / 2,
                ctx = this.context,
                c = Math.PI * 2,
                mw = sW / 10;
        // draw scrub
        ctx.save();
        if (this.disabled)
            ctx.globalAlpha = 1.0;
        if (this.hasFocus) {
            ctx.fillStyle = '#c00';
            ctx.roundRect(this.x + 2, this.y + 2, this.w - 4, this.h - 4, 5);
        }
        else {
            ctx.fillStyle = '#ccc';
            ctx.roundRect(this.x, this.y, this.w, this.h, 5);
        }
        ctx.fill();
        ctx.lineWidth = 1;
        ctx.strokeStyle = "#666";
        ctx.beginPath();
        ctx.moveTo(sX - (8 * mw), sYb + 0.5);
        ctx.lineTo(sX - (4 * mw), sYb + 0.5);
        ctx.moveTo(sX + (4 * mw), sYb + 0.5);
        ctx.lineTo(sX + (8 * mw), sYb + 0.5);
        ctx.moveTo(sX - (8 * mw), sYm + 0.5);
        ctx.lineTo(sX - (6 * mw), sYm + 0.5);
        ctx.moveTo(sX + (6 * mw), sYm + 0.5);
        ctx.lineTo(sX + (8 * mw), sYm + 0.5);
        ctx.moveTo(sX - (8 * mw), sYt + 0.5);
        ctx.lineTo(sX - (4 * mw), sYt + 0.5);
        ctx.moveTo(sX + (4 * mw), sYt + 0.5);
        ctx.lineTo(sX + (8 * mw), sYt + 0.5);
        ctx.closePath();
        ctx.stroke();
        ctx.strokeStyle = "#fff";
        ctx.beginPath();
        ctx.moveTo(sX - 0.5, sYt);
        ctx.lineTo(sX - 0.5, sYb);
        ctx.closePath();
        ctx.stroke();
        ctx.restore();
        ctx.save();
        if (this.disabled)
            ctx.globalAlpha = 0.25;
        // draw slider
        if (this.sliderStyle === 'circle') {
            ctx.save();
            ctx.fillStyle = '#ccc';
            ctx.strokeStyle = "#000";
            ctx.beginPath();
            ctx.arc(sX, sYb - (this.sliderPos * (sYb - sYt)), this.w / 6, 0, c, false);
            ctx.fill();
            ctx.closePath();
            ctx.restore();
            ctx.fillStyle = '#556688';
            ctx.beginPath();
            ctx.arc(sX, sYb - (this.sliderPos * (sYb - sYt)), this.w / 10, 0, c, false);
            ctx.fill();
            ctx.fillStyle = '#ffffff';
            ctx.beginPath();
            ctx.arc(sX, sYb - (this.sliderPos * (sYb - sYt)), this.w / 24, 0, c, false);
            ctx.fill();
        }
        else {
            ctx.fillStyle = "rgba( 144, 144, 144, 0.8 )";
            ctx.strokeStyle = "#000";
            ctx.roundRect(sX - sW / 2, sYb - (sH / 2) - (this.sliderPos * (this.h * .8)), sW, sH, sH / 2);
            ctx.stroke();
            ctx.fill();
        }
        ctx.restore();
    };
    /**
     * render horizontal version
     * @private
     */
    OU.util.Slider.prototype.renderHorizontal = function() {
        var sXl = this.x + this.w * .1,
                sXr = this.x + this.w * .9,
                sY = this.y + this.h / 2,
                sH = this.h / 2 - 2,
                sW = this.sliderWidth,
                ctx = this.context, val,
                c = Math.PI * 2,
                mh = sH / 10;
        // draw scrub
        ctx.save();
        ctx.globalAlpha = 1;
        ctx.fillStyle = '#ccc';
        if (this.params.background.color !== undefined)
            ctx.fillStyle = this.params.background.color;
        if (this.params.background.RGB) {
            if (this.params.background.alpha)
                ctx.fillStyle = 'rgba(' + this.params.background.RGB + ',' + this.params.background.alpha + ')';
            else
                ctx.fillStyle = 'rgba(' + this.params.background.RGB + ',1)';
        }
        if (this.hasFocus)
            ctx.fillStyle = '#c00';
        if (this.params.drawContainer !== false || this.hasFocus) {
            ctx.roundRect(sXl - sW + 1, sY - sH + 1, sXr - sXl + (sW * 2) - 2, sH * 2 - 2, 3);
            ctx.fill();
        }
        else {
            ctx.clearRect(sXl - sW, sY - sH, sXr - sXl + (sW * 2), sH * 2);
        }
        if (this.disabled)
            ctx.globalAlpha = 0.5;
        // draw slider line
        ctx.save();
        ctx.fillStyle = '#fff';
        ctx.strokeStyle = '#666';
        ctx.roundRect(sXl - 5, this.y + this.h / 2 - 5, sXr - sXl + 10, 10, 5);
        ctx.fill();
        ctx.stroke();
        ctx.beginPath();
        ctx.lineWidth = 0.5;
        ctx.strokeStyle = "#000";
        ctx.beginPath();
        ctx.moveTo(sXl, sY);
        ctx.lineTo(sXr, sY);
        ctx.stroke();
        ctx.closePath();
        ctx.restore();
        if (this.disabled)
            ctx.globalAlpha = 0.25;
        // draw slider
        if (this.sliderStyle === 'circle') {
            ctx.save();
            ctx.fillStyle = '#333';
            ctx.strokeStyle = "#eee";
            ctx.lineWidth = 2;
            ctx.beginPath();
            ctx.arc(sXl + (this.sliderPos * (sXr - sXl)), sY, this.h / 3, 0, c, false);
            ctx.fill();
            ctx.stroke();
            ctx.closePath();
            ctx.restore();
            ctx.fillStyle = '#fff';
            ctx.beginPath();
            ctx.arc(sXl + (this.sliderPos * (sXr - sXl)), sY, this.h / 10, 0, c, false);
            ctx.fill();
        }
        else {
            ctx.roundRect(sXl - (sW / 2) + (this.sliderPos * (sXr - sXl)), sY - sH / 2, sW, sH, sW / 2);
            ctx.fillStyle = "rgba( 144, 144, 144, 0.8 )";
            ctx.strokeStyle = "#000";
            ctx.stroke();
            ctx.fill();
        }
        ctx.restore();
        if (this.title || this.showValue) {
            ctx.save();
            ctx.font = (this.h / 2) + 'px ' + OU.theme.font;
            if (this.title) {
                if (this.titlePosition === 'above') {
                    new OU.util.DynText({
                        context: ctx,
                        txt: this.title,
                        notDynamic: true,
                        fontSize: this.h * .4,
                        x: this.x + this.w * .05,
                        y: this.y,
                        w: this.scrubWidth,
                        h: this.h / 2,
                        align: 'left',
                        padding: 0
                    });
                }
                else {
                    new OU.util.DynText({
                        context: ctx,
                        txt: this.title,
                        notDynamic: true,
                        fontSize: this.h * .4,
                        x: this.x,
                        y: this.y,
                        w: this.w * .1,
                        h: this.h
                    });
                }
            }
            if (this.showValue) {
                val = this.range.min + this.sliderPos * this.range.dx;
                val = val.nDecimals(this.range.nDecimals === undefined ? 3 : this.range.nDecimals);
                if (this.valueUnits !== undefined)
                    val = val + this.valueUnits;
                new OU.util.DynText({
                    context: ctx,
                    notDynamic: true,
                    fontSize: this.h * .4,
                    align: 'left',
                    txt: val,
                    x: this.scrubX + this.scrubWidth * 1.05,
                    y: this.y,
                    w: this.w * .3,
                    h: this.h
                });
            }
            ctx.restore();
        }
    };
    /**
     * Provides the mechanism for animation of the slider
     * @private
     */
    OU.util.Slider.prototype.animate = function() {
        var self = this;
        if ((!this.halted) && (Math.abs(this.sliderPos - this.target) > 0.0001)) {
            this.sliderPos += (this.target - this.sliderPos) / this.frames;
            this.callback(this.sliderPos, this.callbackParam);
            setTimeout(function() {
                self.animate();
            }, 40);
        }
        else {
            if (this.sliderPos !== this.target) {
                this.sliderPos = this.target;
                this.callback(this.sliderPos, this.callbackParam);
            }
            this.halted = false;
        }
    };
    /**
     * isHit, called by the Event handler for the layer
     *
     * @private
     */
    OU.util.Slider.prototype.isHit = function(x, y, pressed) {
        if (this.disabled === false && pressed && y > this.y && y < this.y + this.h && x > this.x && x < this.x + this.w) {
            if (this.orient === 'vertical')
                this.target = ((this.y + this.h * .9) - y) / (this.h * .8);
            else
                this.target = (x - this.scrubX) / this.scrubWidth;
            if (this.target < 0)
                this.target = 0;
            if (this.target > 1)
                this.target = 1;
            this.halted = false;
            this.animate();
        }
    };
    /**
     * arrowKey, called by the event handler, when the slider has Tabbable focus
     * @private
     */
    OU.util.Slider.prototype.arrowKey = function(k) {
        if (this.disabled)
            return false;
        if (k === 38 || k === 39)
            this.target = this.sliderPos + 0.1;
        if (k === 37 || k === 40)
            this.target = this.sliderPos - 0.1;

        if (this.target < 0)
            this.target = 0;
        if (this.target > 1)
            this.target = 1;
        this.halted = false;
        this.animate();
        return true;
    };
    /**
     * Animate to a new position
     * @param {double} newPerc - the new position between 0 and 1
     */
    OU.util.Slider.prototype.setTarget = function(newPerc) {
        this.target = newPerc;
        this.halted = false;
        this.animate();
    };
    /**
     * Jump to a new position
     * @param {double} newPerc - the new position between 0 and 1
     */
    OU.util.Slider.prototype.setTarget = function(newPerc) {
        this.sliderPos = newPerc;
        this.halted = false;
        this.render();
    };
    OU.base(this, params);
};
OU.inherits(OU.util.Slider, OU.util.Tabbable);
