/**
 * @fileOverview AlgebraicEquations - Rearranging equations activity
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Button');
OU.require('OU.util.Layer');
OU.require('OU.util.Div');
OU.require('OU.util.ImageLoader');
OU.require('OU.util.Instruction');
/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.AlgebraicEquations = function ( data, instance, controller ) {
    OU.activity.AlgebraicEquations.prototype.canvasView = function () {
        var ctx, self = this;
        this.bgLayer = new OU.util.Layer({
            container:this
        });
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        this.msgLayer = new OU.util.Layer({
            container:this
        });
        ctx = this.msgLayer.context;
        ctx.lineWidth = 2;
        ctx.lineCap = "round";
        ctx.fillStyle = '#999';
        ctx.strokeStyle = '#999';
        this.eqLayer = new OU.util.Layer({
            container:this,
            h:this.h - 80,
            hasEvents:true
        });
        this.actionsLayer = new OU.util.Layer({
            container:this,
            x:this.x,
            y:this.y + this.h - 80,
            h:80,
            hasEvents:true
        });
        this.imageLoader = new OU.util.ImageLoader({
            container:this,
            data:this.data.equations,
            onLoad:function () {
                self.start();
            }
        });
    };
    OU.activity.AlgebraicEquations.prototype.resize = function () {
        OU.activity.AlgebraicEquations.superClass_.resize.call(this); // call the parent class resize
        if(this.imageLoader && this.imageLoader.complete) {
            this.bgLayer.resize();
            this.msgLayer.resize();
            this.bgLayer.context.gradRect();
            this.eqLayer.resize({
                h:this.h - 80
            });
            this.actionsLayer.resize({
                y:this.y + this.h - 80,
                h:80
            });
            this.render();
        }
    };
    OU.activity.AlgebraicEquations.prototype.start = function () {
        this.equation = 0;
        this.step = 0;
        this.selectedAction = '';
        this.showHint = false;
        this.wrongArea = 0;
        this.initActions();
        this.resize();
        this.render();
        this.eqLayer.events.clickable.push(this);
    };
    OU.activity.AlgebraicEquations.prototype.clearActions = function () {
        this.button1.params.onOff = false;
        this.button2.params.onOff = false;
        this.button3.params.onOff = false;
        this.button4.params.onOff = false;
        this.button5.params.onOff = false;
        this.button6.params.onOff = false;
        this.selectedAction = '';
    };
    OU.activity.AlgebraicEquations.prototype.actionHit = function ( p ) {
        this.clearActions();
        this.msgLayer.clear();
        p.button.params.onOff = true;
        this.selectedAction = p.act;
        this.renderActions();
        if (this._help2===undefined) {
            this._help2 = true;
            this.message({
                msg:"Now select an element of the equation to apply it to",
                pt:{
                    x:this.w * .5,
                    y:200 + (this.h - 200) * .5
                },
                mpt:{
                    x:this.w * .4,
                    y:200 + (this.h - 200) * .7
                }
            });
        }
    };
    OU.activity.AlgebraicEquations.prototype.isHit = function ( x, y, pressed ) {
        var ev = this.eqLayer.events,
        eq = this.data.equations[this.equation],
        step = eq.steps[this.step],
        a = step.actionArea,
        w = step.image.width,
        h = step.image.height;
        x = x - (this.w - w) / 2;
        y = y - (this.h - 80 - h) / 2;
        if (pressed) {
            if (this.step==eq.steps.length - 1) {
                return; // we are done!
            }
            ev.touched = ev.pressed = false;
            if (this.selectedAction=='') {
                this.message({
                    msg:"Select an action from the list below first",
                    pt:{
                        x:200,
                        y:this.h - 90
                    },
                    mpt:{
                        x:100,
                        y:this.h - 190
                    }
                });
                return;
            }
            if (this.selectedAction!=step.correctAction) {
                this.fade(204, 0, 0);
                this.message({
                    msg:"You have chosen the wrong action, please try again",
                    pt:{
                        x:200,
                        y:this.h - 90
                    },
                    mpt:{
                        x:100,
                        y:this.h - 190
                    }
                });
                this.showHint = true;
                this.clearActions();
                this.renderActions();
                return;
            }
            if (x >= a.x && x <= a.x + a.w && y >= a.y && y <= a.y + a.h) {
                this.msgLayer.clear();
                this.fade(0, 204, 0);
                this.clearActions();
                this.step++;
                this.showHint = false;
                this.wrongArea = 0;
                this.animateEquation(true);
            }
            else {
                this.fade(204, 0, 0);
                if (++this.wrongArea > 2) {
                    this.showHint = true;
                    this.renderActions();
                }
            }
        }
    };
    OU.activity.AlgebraicEquations.prototype.message = function ( p ) {
        var ctx = this.msgLayer.context, dx, dy;
        this.msgLayer.clear();
        if (p.pt!==undefined) {
            ctx.beginPath();
            ctx.moveTo(p.pt.x, p.pt.y);
            dx = (p.pt.x - p.mpt.x) / 4;
            dy = (p.pt.y - p.mpt.y) / 4;
            ctx.bezierCurveTo(p.pt.x, p.mpt.y - dy, p.mpt.x, p.pt.y - dy, p.mpt.x, p.mpt.y);
            ctx.stroke();
            if (p.pt.y < p.mpt.y) { // point upwards
                ctx.beginPath();
                ctx.moveTo(p.pt.x, p.pt.y - 2);
                ctx.lineTo(p.pt.x + 10, p.pt.y + 15);
                ctx.lineTo(p.pt.x - 10, p.pt.y + 15);
                ctx.fill();
                new OU.util.DynText({
                    txt:p.msg,
                    container:this,
                    context:ctx,
                    x:(p.mpt.x < p.pt.x ? p.mpt.x : p.pt.x) - dx * 4,
                    y:p.mpt.y,
                    w:dx * 8,
                    h:60
                });
            }
            else { // point down
                ctx.beginPath();
                ctx.moveTo(p.pt.x, p.pt.y + 2);
                ctx.lineTo(p.pt.x + 10, p.pt.y - 15);
                ctx.lineTo(p.pt.x - 10, p.pt.y - 15);
                ctx.fill();
                new OU.util.DynText({
                    txt:p.msg,
                    container:this,
                    context:ctx,
                    x:(p.mpt.x < p.pt.x ? p.mpt.x : p.pt.x) - dx * 4,
                    y:p.mpt.y - 60,
                    w:dx * 8,
                    h:60
                });
            }
        }
        else {
            new OU.util.DynText({
                txt:p.msg,
                container:this,
                context:this.msgLayer.context,
                x:this.w * .1,
                y:this.h / 2,
                w:this.w * .8,
                h:this.h / 2 - 80
            });
        }
    };
    OU.activity.AlgebraicEquations.prototype.fade = function ( r, g, b ) {
        var ctx = this.bgLayer.context, self = this;
        if (r===undefined) {
            this._fadeR = this._fadeR + (256 - this._fadeR) / 16;
            this._fadeG = this._fadeG + (256 - this._fadeG) / 16;
            this._fadeB = this._fadeB + (256 - this._fadeB) / 16;
        }
        else {
            this._fadeR = r;
            this._fadeG = g;
            this._fadeB = b;
            if (this._fading)
                return;
        }
        ctx.beginPath();
        ctx.fillStyle = '#fff';
        ctx.rect(0, 0, this.w, this.h);
        ctx.fill();
        if (this._fadeR > 230 && this._fadeG > 230 && this._fadeB > 230) {
            this._fading = false;
        }
        else {
            ctx.beginPath();
            ctx.fillStyle = 'rgba(' + (this._fadeR | 0) + ',' + (this._fadeG | 0) + ',' + (this._fadeB | 0) + ',1)';
            ctx.rect(0, 0, this.w, this.h);
            ctx.fill();
            ctx.beginPath();
            ctx.fillStyle = '#fff';
            ctx.roundRect(40, 40, this.w - 80, this.h - 80 - 80, 20);
            ctx.fill();
            this._fading = true;
            setTimeout(function () {
                self.fade();
            }, 40);
        }
    };
    OU.activity.AlgebraicEquations.prototype.render = function () {
        this.renderActions();
        this.renderEquation();
    };
    OU.activity.AlgebraicEquations.prototype.renderHint = function () {
        var eq = this.data.equations[this.equation],
        step = eq.steps[this.step];
        if (!this.showHint)
            return; // don't show it if the hint button is not visible
        this.msgLayer.clear();
        this.message({
            msg:step.hint
        });
    };
    OU.activity.AlgebraicEquations.prototype.animateEquation = function ( start ) {
        var self = this,
        step = this.data.equations[this.equation].steps[this.step];
        if (start!==undefined) {
            this.inTransition = true;
            this.animStage = 1;
            this.lastStepOpacity = 0;
            this.cancelOpacity = 0;
            this.animOffset = 0;
            this.animTarget = step.image.height + 10;
        }
        if (this.animStage==1) {
            this.animOffset = this.animOffset + (this.animTarget - this.animOffset) / 8;
            if (Math.abs(this.animOffset - this.animTarget) < 0.1) {
                this.animOffset = this.animTarget;
                this.animStage = 2;
            }
        }
        if (this.animStage==2) {
            this.lastStepOpacity = this.lastStepOpacity + (1 - this.lastStepOpacity) / 8;
            if (this.lastStepOpacity > 0.95) {
                this.lastStepOpacity = 1;
                this.animStage = 3;
            }
        }
        if (this.animStage==3) {
            if (step.stepType=='cancellation') {
                this.cancelOpacity = this.cancelOpacity + (1 - this.cancelOpacity) / 8;
                if (this.cancelOpacity > 0.95) {
                    this.cancelOpacity = 1;
                    this.step++;
                    this.inTransition = false;
                    setTimeout(function () {
                        self.animateEquation(true);
                    }, 40);
                    return;
                }
            }
            else {
                this.inTransition = false;
                this.showHint = false;
                this.clearActions();
                this.renderActions();
            }
        }
        this.renderEquation();
        if (this.inTransition) {
            setTimeout(function () {
                self.animateEquation();
            }, 40);
        }
    };
    OU.activity.AlgebraicEquations.prototype.renderEquation = function () {
        var i, j, pt, ctx = this.eqLayer.context, eq = this.data.equations[this.equation], step, w, h, y;
        ctx.globalAlpha = 1;
        ctx.fillStyle = '#ccd';
        this.eqLayer.clear();
        for (i = this.step + 1; i--;) {
            step = eq.steps[i];
            w = step.image.width;
            h = step.image.height;
            if (i==this.step) { // rendering latest step
                y = (this.h - 80 - h) / 2;
                ctx.globalAlpha = this.lastStepOpacity;
            }
            else {
                ctx.globalAlpha = .5;
            }
            if (y + h > 0) {
                ctx.roundRect((this.w - w) / 2, y, w, h, 10);
                ctx.fill();
                ctx.drawImage(step.image, (this.w - w) / 2, y, w, h);
                if (step.stepType=='cancellation') {
                    ctx.globalAlpha = this.step==i ? this.cancelOpacity : .75;
                    for (j = step.cancelPoints.length; j--;) {
                        pt = step.cancelPoints[j];
                        new OU.util.DynText({
                            context:ctx,
                            txt:"X",
                            x:(this.w - w) / 2 + pt.x - 30,
                            y:y + pt.y - 30,
                            w:60,
                            h:60,
                            colour:'#c00',
                            fontWeight:'bold',
                            fontSize:40
                        });
                    }
                }
                if (this.inTransition==false || i!=this.step) {
                    y = y - h - 10;
                }
                else {
                    y = y - this.animOffset;
                }
            }
            else {
                i = 0;
            }
        }
        if (this.step==0) {
            this.message({
                msg:eq.instructions
            });
        }
        if (this.step==eq.steps.length - 1 && !this.inTransition) {
            this.message({
                msg:eq.feedbackOnCompletion
            });
        }
    };
    OU.activity.AlgebraicEquations.prototype.renderActions = function () {
        var l = this.actionsLayer,
        ctx = l.context;
        ctx.gradRect({
            col1:'#999',
            col2:'#444'
        });
        ctx.beginPath();
        ctx.strokeStyle = '#000';
        ctx.moveTo(0, 0);
        ctx.lineTo(this.w, 0);
        ctx.stroke();
        this.button1.render();
        this.button2.render();
        this.button3.render();
        this.button4.render();
        this.button5.render();
        this.button6.render();
        if (this.showHint)
            this.hintButton.render();
    };
    OU.activity.AlgebraicEquations.prototype.initActions = function () {
        var l = this.actionsLayer,
        ctx = l.context,
        self = this;
        l.events.clickable.length = 0;
        this.button1 = new OU.util.Button({
            layer:l,
            x:40,
            y:5,
            w:70,
            h:70,
            container:l,
            align:"center",
            txt:"+",
            padding:1,
            verticalPadding:1,
            onClick:function () {
                self.actionHit({
                    act:"add",
                    button:self.button1
                });
            },
            onOffIndicator:true,
            onOff:false
        });
        this.button2 = new OU.util.Button({
            layer:l,
            x:120,
            y:5,
            w:70,
            h:70,
            container:l,
            align:"center",
            txt:"-",
            padding:1,
            verticalPadding:1,
            onClick:function () {
                self.actionHit({
                    act:"minus",
                    button:self.button2
                });
            },
            onOffIndicator:true,
            onOff:false
        });
        this.button3 = new OU.util.Button({
            layer:l,
            x:200,
            y:5,
            w:70,
            h:70,
            container:l,
            align:"center",
            txt:"\u00d7",
            padding:1,
            verticalPadding:1,
            onClick:function () {
                self.actionHit({
                    act:"multiply",
                    button:self.button3
                });
            },
            onOffIndicator:true,
            onOff:false
        });
        this.button4 = new OU.util.Button({
            layer:l,
            x:280,
            y:5,
            w:70,
            h:70,
            container:l,
            align:"center",
            txt:"\u00f7",
            padding:1,
            verticalPadding:1,
            onClick:function () {
                self.actionHit({
                    act:"divide",
                    button:self.button4
                });
            },
            onOffIndicator:true,
            onOff:false
        });
        this.button5 = new OU.util.Button({
            layer:l,
            x:360,
            y:5,
            w:70,
            h:70,
            container:l,
            align:"center",
            txt:"\u221a",
            padding:1,
            verticalPadding:1,
            onClick:function () {
                self.actionHit({
                    act:"squareRoot",
                    button:self.button5
                });
            },
            onOffIndicator:true,
            onOff:false
        });
        this.button6 = new OU.util.Button({
            layer:l,
            x:440,
            y:5,
            w:70,
            h:70,
            align:"center",
            txt:"x<sup>2</sup>",
            padding:1,
            verticalPadding:1,
            context:ctx,
            onClick:function () {
                self.actionHit({
                    act:"square",
                    button:self.button6
                });
            },
            onOffIndicator:true,
            onOff:false
        });
        this.hintButton = new OU.util.Button({
            layer:l,
            x:520,
            y:20,
            w:70,
            h:40,
            container:l,
            align:"center",
            txt:"Hint",
            padding:1,
            verticalPadding:1,
            onClick:function () {
                self.renderHint();
            }
        });
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.AlgebraicEquations, OU.util.Activity);
