/**
 * @fileOverview Spatial frequencies activity
 *
 * @author Fiona Williams
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.Layer');
OU.require('OU.util.PopUpInfo');
OU.require('OU.util.Slider');
/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.SpatialFreqGraph = function ( data, instance, controller ) { //data and instance are passed in from index.html
    OU.activity.SpatialFreqGraph.prototype.canvasView = function () {
        //var self = this
        var bH = OU.controlHeight;
        // create Canvas Layers & Contexts
        this.initialSliderPos = 0;
	this.isSquare = true;
	this.showCycles = this.data.showCycles;
	
	this.lineCol = "#000";
	 if (this.data.graphLineColour !== undefined)
	 {
	     this.lineCol = this.data.graphLineColour; 
	 }
	 
	 this.backCol = "#FFF";
	 if (this.data.backgroundColour !== undefined)
	 {
	     this.backCol = this.data.backgroundColour;
	 }
	
        this.bgLayer = new OU.util.Layer({
            container:this, //container is the whole area and can contain other stuff
            id:'bg'
        });
	
        this.graphLayer = new OU.util.Layer({
            container:this,
            id:'graphLayer',
            x:this.w*0.05,
            y:2*bH + 0.02*this.h,
            w:this.w*0.35,
            h:this.h*0.58,
            hasEvents:true,
            pinch:this.pinch,
            pinchMe:this
        });

        this.controlsLayer = new OU.util.Layer({
            container:this,
            id:'slidersLayer',
            x:this.graphLayer.x,
            y:this.graphLayer.y+this.graphLayer.h,
            w:this.graphLayer.w,
            h:this.h*0.4,
            hasEvents:true,
            pinch:this.pinch,
            pinchMe:this
        });
	
        this.controlsLayer.context.gradRect(); // use default background

        this.textDiv = new OU.util.Div({
            x:this.graphLayer.x+this.graphLayer.w+this.w*0.05,
            y:this.graphLayer.y,
            w:this.w*0.4,
            h:this.h*0.5,
            container:this
        });
        this.textDiv.div.innerHTML = this.data.instructions;
		
        this.initControls();
        this.resize();    // sets all sizes and calls render()
    };


    OU.activity.SpatialFreqGraph.prototype.initControls = function () {
        var ctx = this.controlsLayer.context, self = this,
        clickable = this.controlsLayer.events.clickable, cH = this.controlsLayer.h, cW = this.controlsLayer.w;
			
        this.slider = new OU.util.Slider({
            container:this.controlsLayer,
            instance:'',
	    x:0,
            y:0.1*cH,
            w:cW,
            h:0.18*cH,
            sliderPos:this.initialSliderPos,
	    titlePosition:'above',
            drawContainer:false,
            title:'spatial frequency',
            showValue:false,
            callback:self.setSlider,
            callbackParam:{i:1,self:this}, //identifies which slider was clicked
            background:{
                clear:true
            },
            context:ctx
        });
        clickable.push(this.slider);

	this.ncyclesDiv = new OU.util.Div({
	    container:this,
            x:this.controlsLayer.x + 0.1*cW,
            y:this.controlsLayer.y + 0.3*cH,
            w:cW,
            h:0.18*cH
        });
	if (this.showCycles)
	{
	    this.ncyclesDiv.visible = true;
	    this.ncyclesDiv.div.innerHTML = "Number of cycles shown = ";
	}
	else
	{
	    this.ncyclesDiv.visible = false;
	}
	
	this.squareButton = new OU.util.CheckBoxButton({
	    txt:"square",
	    layer:this.controlsLayer,
	    y:0.5*cH,
            h:cH*0.18,
            w:cW*0.4,
            x:0.1*cW,
	    state:true,
	    onClick:function () {
		self.isSquare = true;
		self.squareButton.state(true);
		self.sineButton.state(false);
		self.render();
            }	 
	})
	
	this.sineButton = new OU.util.CheckBoxButton({
	    txt:"sine",
	    layer:this.controlsLayer,
	    y:0.7*cH,
            h:cH*0.18,
            w:cW*0.4,
            x:0.1*cW,
	    state:false,
	    onClick:function () {
		self.isSquare = false;
		self.squareButton.state(false);
		self.sineButton.state(true);
		self.render();
            }
	})
	

        this.helpBtn = new OU.util.Button({
            txt:"Help",
            padding:0,
            verticalPadding:0,
            layer:this.controlsLayer,
             y:cH*0.7,
            h:cH*0.2,
            w:cW*0.2,
            x:cW*0.75,
            onClick:function () {
                if (!self.helpDisp) {
                    self.helpDisp = true;
                    self.showHelp();
                }
            }
        });
 
    };

    OU.activity.SpatialFreqGraph.prototype.showHelp = function () {
        var self = this;
        new OU.util.PopUpInfo({
            container:this,
            txt:this.data.help,
            x:this.w*0.17,
            y:this.h*0.17,
            w:this.w*0.66,
            h:this.h*0.66,
            onClose:function () {
                self.helpDisp = false;
            }
        });
    };

    /**
     * The main renderer.
     */
    OU.activity.SpatialFreqGraph.prototype.render = function () {
	this.helpBtn.render();
	this.squareButton.render();
	this.sineButton.render();
	this.slider.render();
	this.drawAxes();
        this.drawGraphs();
    };

    OU.activity.SpatialFreqGraph.prototype.resize = function () {
        OU.activity.SpatialFreqGraph.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight, ctxS = this.controlsLayer.context, cH = this.controlsLayer.h,
        cW = this.controlsLayer.w;
	
        this.bgLayer.resize();
	
	this.graphLayer.resize({
            x:this.w*0.05,
            y:2*bH + 0.02*this.h,
            w:this.w*0.35,
            h:this.h*0.58
        });
	
        this.controlsLayer.resize({
            x:this.graphLayer.x,
            y:this.graphLayer.y+this.graphLayer.h,
            w:this.graphLayer.w,
            h:this.h*0.4
        });
	
        if (this.data.backgroundColour!==undefined)
	// use specified background colour
	{
            this.bgLayer.context.gradRect({
                col1:this.data.backgroundColour,
                col2:this.data.backgroundColour
            });
	    
	    ctxS.gradRect({
                col1:this.data.backgroundColour,
                col2:this.data.backgroundColour
            });	    	    
	}
	else
        {
	    this.bgLayer.context.gradRect(); // use default background
	    ctxS.gradRect();
	}

        this.slider.resize({
	    x:0,
            y:0.1*cH,
            w:cW,
            h:0.18*cH
        });
	
	 this.ncyclesDiv.resize({
	    x:this.controlsLayer.x + 0.1*cW,
            y:this.controlsLayer.y + 0.3*cH,
            w:cW,
            h:0.18*cH
	 });

	this.squareButton.resize({
	    y:0.5*cH,
            h:cH*0.18,
            w:cW*0.4,
            x:0.1*cW   
	})
	
	this.sineButton.resize({
	    y:0.7*cH,
            h:cH*0.18,
            w:cW*0.4,
            x:0.1*cW   
	})
	
        this.helpBtn.resize({
            y:cH*0.7,
            h:cH*0.2,
            w:cW*0.2,
            x:cW*0.75
        });
	
        this.textDiv.resize({
            x:this.graphLayer.x+this.graphLayer.w+this.w*0.05,
            y:this.graphLayer.y,
            w:this.w*0.4,
            h:this.h*0.5
        });
	
		
        this.render();
        
    };
    
    OU.activity.SpatialFreqGraph.prototype.drawAxes = function () {
	var ctx = this.graphLayer.context;
        var w = this.graphLayer.w;
        var h = this.graphLayer.h;
	var origx = w*0.08;    
	var origy = 0.75*h;
	ctx.fillStyle = this.backCol;
        ctx.fillRect(0, 0, w, h);
		
	//draw the axes
	ctx.strokeStyle = this.lineCol;
	ctx.fillStyle = this.lineCol;
	
	var Aht = 0.18*h;
	ctx.beginPath();
	ctx.moveTo(origx, origy - Aht);
	ctx.lineTo(origx, origy + Aht);
	ctx.lineTo(w, origy + Aht);
	ctx.stroke();
	ctx.font = "12px verdana,sans-serif";
	ctx.textBaseline = "top";
	var size = ctx.measureText("distance").width;
	ctx.fillText("distance", w - size - 2, origy + Aht + 2);
	size = ctx.measureText("intensity").width;
	ctx.rotate(-Math.PI/2);
	ctx.fillText("intensity", -origy + Aht - size, origx - 14);
	ctx.rotate(Math.PI/2);	
    }
	
	
    OU.activity.SpatialFreqGraph.prototype.drawGraphs = function () {
        var ctx = this.graphLayer.context;
        var w = this.graphLayer.w;
        var h = this.graphLayer.h;
	var h2 = Math.floor(0.5*h);
	var origx = Math.floor(w*0.1);     
	var origy = Math.floor(0.75*h);   // origy refers to the lower graph
	var A = Math.floor(0.15*h);
	var graphWidth = w - origx;
	
	var minPeriod = 3;
	var maxPeriod = 100;
	var xscale = graphWidth/200;   // rescale so that width of graph area represents 200 
	
	// following original version, slider changes period rather than frequency and is inverted so low slider position
	// corresponds to higher period (lower freq)
	var period = maxPeriod - this.slider.sliderPos*(maxPeriod - minPeriod);
	var freq = 1/period;
	var fscale = 2*freq/xscale;
	if (this.showCycles) this.ncyclesDiv.div.innerHTML = "Number of cycles shown = " + (200/period).toFixed(2);
	
	var intensity;
		
	// clear the graph areas
	var Aht = Math.floor(1.2*A);
	ctx.fillStyle = this.backCol;
        ctx.fillRect(origx-1, origy - Aht, graphWidth + 2, 2*Aht - 1);	
	ctx.fillRect(origx-1, 0, graphWidth + 2, h2);
	
	ctx.strokeStyle = this.lineCol;
	ctx.beginPath();
	ctx.moveTo(origx,origy);
		
	// if square wave
	 if (this.isSquare)
	 {
	    // draw square wave pattern as series of rectangles of fixed width - this removes problem
	    // when frequency is high of uneven widths due to rounding 
	    var stripWidth = 0.5*period*xscale;
	    ctx.fillStyle = "rgb(0,0,0)";
	    var stripCount = 0;
	    for (var i = 0; i < graphWidth; i += stripWidth)
	    {
		if ( stripCount % 2 < 1e-6)
		{
		    intensity = 1;
		    ctx.fillStyle = "#FFF";
		}
		else
		{
		    intensity = -1;
		    ctx.fillStyle = "#000";
		}				
		ctx.fillRect(origx + i, 2, stripWidth, h2-2);
		ctx.lineTo(origx + i, origy - A*intensity);
		ctx.lineTo(origx + i + stripWidth, origy - A*intensity);
		stripCount++;
	    }

	 }
	 else  // if sine wave
	 {		    
	    for (var i = 0; i < graphWidth; i=i+1)
	    {	
		intensity = Math.sin(Math.PI*fscale*i);     // a value between -1 and +1
		var colourVal = 127 + Math.floor(127*intensity);       
		ctx.fillStyle = 'rgb(' + colourVal + ',' + colourVal + ',' + colourVal + ')';
		ctx.fillRect(origx + i,2,1,h2-2);
		ctx.lineTo(origx + i, origy - A*intensity);
	    }
	 }
	
	 ctx.stroke();
	 
	// draw a grey border around the display
	ctx.strokeStyle = "rgb(162,162,162)";
	ctx.strokeRect(origx, 1, graphWidth-1, h2);
		
	/*// clear the graph area
	var Aht = Math.floor(1.2*A);
	ctx.fillStyle = "rgb(255,255,255)";
        ctx.fillRect(origx-1, origy - Aht, graphWidth + 1, 2*Aht - 1);
	
	// draw the graph
	ctx.strokeStyle = "#000000";
	ctx.beginPath();
	ctx.moveTo(origx,origy)
	if (this.isSquare)
	{
	    for (var i=0; i<graphWidth; i=i+1)
	    {	    
		var stripNo = Math.floor(i*fscale);  
		if ( stripNo % 2 < 1e-6)
		    intensity = A;		
		else
		    intensity = -A;    
		ctx.lineTo(origx + i, origy - intensity);
	    }	    
	}
	else
	{
	    for (var i=0; i<graphWidth; i=i+1)
	    {	    
		intensity = Math.floor(A*Math.sin(Math.PI*fscale*i)); 
		ctx.lineTo(origx + i, origy - intensity);
	    }
	}
	ctx.stroke();
*/	
    };

    OU.activity.SpatialFreqGraph.prototype.setSlider = function ( p, v ) { // Called when slider is moved
	v.self.slider.render();
        v.self.drawGraphs();       
    };
   

    /*OU.activity.Cafewall.prototype.pinch = function ( e, s, x, y, dx, dy, me ) { // called when pinch event detected
        var ns = ((e - s) / me.h) + me.tileView.s;
        ns = ns > 1 ? 1 : (ns < 0 ? 0 : ns);
        me.tileView.scale(ns, {
            x:x,
            y:y
        });
        me.zoomSlider.sliderPos = ns;
        me.zoomSlider.render();
    };//*/
    /**
     * @class
     */
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.SpatialFreqGraph, OU.util.Activity);
