/**
 * @fileOverview Slideshow Activity
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.DynText');
OU.require('OU.util.NavButtons');
OU.require('OU.util.ScrubBar');
OU.require('OU.util.Layer');
OU.require('OU.util.ImageLoader');
/**
 * @class Simple slideshow activity.
 * 
 * @extends OU.util.Activity
 * @param {object} data - data.js content
 * @param {string} instance - unique instance name
 * @param {OU.util.Controller} controller - (optional) reference of parent controller
 */
OU.activity.Slideshow = function(data, instance, controller) {

    /**
     * Starting point when running with HTML5 Canvas Support
     */
    OU.activity.Slideshow.prototype.canvasView = function() {

        var bH = OU.controlHeight, self = this, labelHeight, controlCtx;

        this.config = {
            imageLoadTimeout: 10000, // allow max 10 seconds to load images

            // Settings for ScrubBar & Nav Buttons
            overlayControls: false, // set to false to seperate controls from images

            // Settings for Caption object
            useCaptions: true,
            captionHeight: 30,
            // animation settings
            sliderSpeed: 2,
            inertiaFactor: 1000, // set to minimum 2 for no inertia effect

            // internal values below - will be overwritten anyway
            maxWidth: 1024, // overwritten by canvas width
            maxHeight: 768, // overwritten by canvas height (minus controlHeight if not overlaid)
            slideHeight: 768 // Height of slides -recalced, based on control placement
        };

        this.noSliding = this.data.noSliding || false;
        this.viewOption = data.view;
        this.autoInterval = data.autoInterval || 6000;
        this.slides = this.data.slides;
        this.loadedImagesCount = 0;
        this.focus = 0; // image number that is currently in focus
        this.currentFocus = 0; // actual focus
        this.focusOffset = this.w; // image line X offset - starts off right of canvas
        this.autoDirectionNext = true;
        this.inertia = 1;
        this.noRender = false;

        this.useNavButtons = true;
        this.useScrubBar = true;
        this.autoScroll = false;
        if (this.viewOption < 1)
            this.viewOption = 3; // default to Scrub bar and Next/Prev
        if (this.viewOption === 1)
            this.useScrubBar = false;
        if (this.viewOption === 2)
            this.useNavButtons = false;
        if (this.viewOption === 4) {
            this.useNavButtons = false;
            this.useScrubBar = false;
            this.autoScroll = true;
        }
        if (this.viewOption === 5) {
            this.useNavButtons = false;
            this.useScrubBar = false;
            this.autoScroll = false;
            this.config.overlayControls = true;
        }

        this.bgLayer = new OU.util.Layer({
            container: this
        });
        this.bgLayer.context.gradRect(); // draw background on backdrop layer

        this.imageLayer = new OU.util.Layer({
            container: this,
            hasEvents: true
        });
        labelHeight = this.data.scrubLabels ? bH : 0;
        this.controlLayer = new OU.util.Layer({
            container: this,
            y: this.y + this.h - (bH + labelHeight),
            h: bH + labelHeight,
            hasEvents: true
        });
        controlCtx = this.controlLayer.context;
        // setup an interaction handlers
        this.imageLayer.events.moveRight = function() {
            self.nextSlide();
        };
        this.imageLayer.events.moveLeft = function() {
            self.prevSlide();
        };

        // set Proportions of slideshow elements
        this.config.captionHeight = this.h * .1;
        if (this.config.overlayControls)
            this.config.slideHeight = this.h;
        else
            this.config.slideHeight = this.h - bH;
        this.config.maxHeight = this.config.slideHeight;

        // Add nav buttons
        if (this.useNavButtons) {
            this.navButtons = new OU.util.NavButtons({
                x: 0,
                y: labelHeight,
                w: this.w,
                h: bH,
                pad: 0, //bH*.1,
                context: controlCtx,
                layer: this.controlLayer,
                leftFunc: function() {
                    self.prevSlide();
                },
                rightFunc: function() {
                    self.nextSlide();
                }
            });
        }

        if (this.useScrubBar) { //TODO possibly expand to full canvas width if no Nav buttons
            this.scrubBar = new OU.util.ScrubBar({
                points: this.slides.length,
                labelHeight: labelHeight,
                labels: this.data.scrubLabels,
                x: bH,
                y: 0,
                w: this.w - 2 * bH,
                h: bH + labelHeight,
                context: controlCtx,
                newPointFunc: this.gotoSlide,
                callbackParam: this,
                startPos: this.focus
            });
            this.controlLayer.events.clickable.push(this.scrubBar);
        }

        if (this.autoScroll) {
            setInterval(function() {
                self.autoNext();
            }, this.autoInterval);
        }

        if (this.data.waitForCue) {
            this.state = OU.PAUSED;
        }
        this.load();

    };
    /**
     * Removes the activity, halts animation and clears some variables
     */
    OU.activity.Slideshow.prototype.remove = function() {
        OU.activity.Slideshow.superClass_.remove.call(this); // call the superclass method
        this.noRender = true;
        this.data = null;
        this.imageLoader = null;
    };
    /**
     * Loads the images ready for the slideshow, then starts the render
     */
    OU.activity.Slideshow.prototype.load = function() {
        var self = this;
        this.imageLoader = new OU.util.ImageLoader({
            container: this,
            data: this.slides,
            onLoad: function() {
                self.scaleImages();
                if (self.data.waitForCue) {
                    self.state = OU.PAUSED;
                }
                else {
                    self.state = OU.RUNNING;
                    self.render();
                }
            }
        });
    };
    /**
     * Moves the slideshow to a specific slide number 
     * @param {int} n - new slide number (1 to N)
     */
    OU.activity.Slideshow.prototype.step = function(n) {
        if (n > 0 && n <= this.slides.length) {
            if (this.focus !== n - 1) {
                this.focus = n - 1;
                this.inertia = this.config.inertiaFactor;
                if (this.state === OU.PAUSED) {
                    this.state = OU.RUNNING;
                    this.render();
                }
                if (this.useScrubBar)
                    this.scrubBar.newTarget(this.focus);
            }
        }
    };
    /**
     * Resizes the activity
     */
    OU.activity.Slideshow.prototype.resize = function() {
        OU.activity.Slideshow.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight, labelHeight = this.data.scrubLabels?bH * 1.5:0;

        this.bgLayer.resize();
        this.bgLayer.context.gradRect(); // draw background on backdrop layer

        this.imageLayer.resize();
        this.controlLayer.resize({
            y: this.y + this.h - (bH + labelHeight),
            h: bH + labelHeight
        });

        // set Proportions of slideshow elements
        this.config.captionHeight = this.h * .1;
        if (this.config.overlayControls)
            this.config.slideHeight = this.h;
        else
            this.config.slideHeight = this.h - bH;
        this.config.maxHeight = this.config.slideHeight;

        // Add nav buttons
        if (this.navButtons) {
            this.navButtons.move({
                x: 0,
                y: labelHeight,
                w: this.w,
                h: bH
            });
        }
        if (this.scrubBar) {
            this.scrubBar.resize({
                x: bH,
                y: 0,
                w: this.w - 2 * bH,
                h: bH + labelHeight
            });
        }
        this.scaleImages();
    };
    /**
     * Renders the current view
     * @private
     */
    OU.activity.Slideshow.prototype.render = function() {
        var self = this, bH = OU.controlHeight, xOffset = 0, speed = 0, 
                targetFocusOffset = 0 - (this.w * this.focus), // X val of the centre of the focus image
                events = this.imageLayer.events,
                ctx = this.imageLayer.context,
                j, x, y, w, h, closestSlide = 0, captionAlpha = 1, slide, closestCaption;

        if (this.noRender || !ctx) // totally stopped render, activity has been removed
            return;
        // Stop rendering when dormant (no animation needed) to save on processor usage
        if (this.doRender || this.focusOffset !== targetFocusOffset || events.pressed || events.touched) {

            ctx.clearRect(0, 0, this.w, this.h); // clear layer

            // cycle through the slide array and render the slide if it is visible
            for (j = this.slides.length; j--; ) {
                slide = this.slides[j];
                w = slide.w;
                h = slide.h;
                y = slide.y;
                x = this.focusOffset + slide.x;

                if (x < this.w && (x + w) > 0) {
                    ctx.drawImage(slide.image, x, y, w, h);
                    if (this.focusOffset < targetFocusOffset) { // moving right
                        if (x < this.w / 2) {
                            closestSlide = j;
                            captionAlpha = 1 - (targetFocusOffset - this.focusOffset) / this.w;
                        }
                    }
                    else { // moving left
                        if (x > -(this.w / 2)) {
                            closestSlide = j;
                            captionAlpha = 1 - (this.focusOffset - targetFocusOffset) / this.w;
                        }
                    }
                }
                xOffset += this.w;
            }
            closestCaption = this.slides[closestSlide].caption;

            if (this.noSliding) { // just jump through slides
                if (this.currentFocus < this.focus)
                    this.currentFocus++;
                if (this.currentFocus > this.focus)
                    this.currentFocus--;
                this.focusOffset = - (this.w * this.currentFocus);
            }
            else { // perform animated slide movement
                if (this.focusOffset > targetFocusOffset) {
                    speed = -(this.focusOffset - targetFocusOffset) / this.config.sliderSpeed;
                }

                if (this.focusOffset < targetFocusOffset) {
                    speed = (targetFocusOffset - this.focusOffset) / this.config.sliderSpeed;
                }

                if (this.inertia > 1) {
                    this.inertia = this.inertia / 2; // reduce inertia on each render cycle
                }
                else {
                    this.inertia = 1;
                }
                speed = speed / this.inertia; // apply inertia to slow initial movement of animation

                if (Math.abs(speed) < 1 && speed > 0)
                    speed = 1;
                this.focusOffset += speed;

                if (Math.abs(this.focusOffset - targetFocusOffset) < 1)
                    this.focusOffset = targetFocusOffset;
            }
            this.controlLayer.clear();
            this.imageLayer.events.clickable.length = 0;
            // render Nav Buttons
            if (this.navButtons) {
                if (this.focus === 0) {
                    this.navButtons.leftOn = false;
                    this.navButtons.rightOn = true;
                }
                else if (this.focus >= this.slides.length - 1) {
                    this.navButtons.leftOn = true;
                    this.navButtons.rightOn = false;
                }
                else {
                    this.navButtons.leftOn = true;
                    this.navButtons.rightOn = true;
                }
                this.navButtons.render();
            }

            // Render Scrub Bar
            if (this.useScrubBar)
                this.scrubBar.render();

            // Render Caption
            if (this.config.useCaptions && closestCaption && closestCaption !== '') {
                new OU.util.DynText({
                    txt: closestCaption,
                    x: 0,
                    y: this.h - bH - this.config.captionHeight,
                    w: this.w,
                    h: this.config.captionHeight,
                    context: ctx,
                    background: {
                        RGB: '255,255,255',
                        alpha: captionAlpha / 2
                    },
                    alpha: captionAlpha,
                    fontWeight: 'bold'
                });
            }
        }
        // cycle render
        setTimeout(function() {
            self.render();
        }, 40);
    };
    /**
     * Moves to the next slide
     */
    OU.activity.Slideshow.prototype.nextSlide = function() {
        if (this.focus < this.slides.length - 1) {
            this.focus++;
            this.inertia = this.config.inertiaFactor;
            if (this.useScrubBar)
                this.scrubBar.newTarget(this.focus);
        }
        else {
            this.autoDirectionNext = false;
        }
    };
    /**
     * Moves to the previous slide
     */
    OU.activity.Slideshow.prototype.prevSlide = function() {
        if (this.focus > 0) {
            this.focus--;
            this.inertia = this.config.inertiaFactor;
            if (this.useScrubBar)
                this.scrubBar.newTarget(this.focus);
        }
        else {
            this.autoDirectionNext = true;
        }
    };
    /**
     * Moves to the next slide automatically (next slide can be previous if running back through the slides)
     */
    OU.activity.Slideshow.prototype.autoNext = function() {
        if (this.autoDirectionNext)
            this.nextSlide();
        else
            this.prevSlide();
    };
    /**
     * Jumps to a specific slide
     * @param {int} newSlide - the slidenumber (0 to n-1)
     * @param {OU.activity.Slideshow} me - reference to this
     */
    OU.activity.Slideshow.prototype.gotoSlide = function(newSlide, me) {
        me.focus = newSlide;
        me.inertia = me.config.inertiaFactor;
    };
    /**
     * Resize the images according to the dimensions of the activity.
     */
    OU.activity.Slideshow.prototype.scaleImages = function() {
        var j, w, h, img;
        for (j = this.slides.length; j--; ) {
            img = this.slides[j].image;
            w = img.width;
            h = img.height;

            if (w > this.w) {
                h = img.height * (this.w / w); // maintain aspect ratio
                w = this.w;
            }
            if (h > this.config.maxHeight) {
                w = w * (this.config.maxHeight / h); // maintain aspect ratio
                h = this.config.maxHeight;
            }
            this.slides[j].w = w;
            this.slides[j].h = h;
            this.slides[j].y = (this.config.slideHeight - h) / 2;
            this.slides[j].x = (j * this.w) + ((this.w - w) / 2);
        }
        this.doRender = true;
    };
    /**
     * Renders an alternative view if HTML5 canvas is not available
     */
    OU.activity.Slideshow.prototype.accessibleView = function() {
        this.noRender = true;
        var i, slide, h = '<div id="slideshowFallback"><h1>Slideshow</h1>';
        for (i = 0; i < this.slides; i++) {
            slide = this.slides[i];
            h += '<img src="data/' + slide.fileName + '" alt="' + slide.alt + '"/>';
            if (slide.caption !== undefined && slide.caption !== '')
                h += '<p>' + slide.caption + '</p>';
        }
        h += '</div>';
        document.body.innerHTML = '';
        var accessible = document.createElement('div');
        accessible.innerHTML = h;
        document.body.appendChild(accessible);
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.Slideshow, OU.util.Activity);
