/**
 * @fileOverview Spatial frequencies activity
 *
 * @author Fiona Williams
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.Layer');
OU.require('OU.util.PopUpInfo');
OU.require('OU.util.Slider');
/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.SpatialFreqMTF = function ( data, instance, controller ) { //data and instance are passed in from index.html
    OU.activity.SpatialFreqMTF.prototype.canvasView = function () {
        //var self = this
        var bH = OU.controlHeight;
        // create Canvas Layers & Contexts
        this.initialSliderPos = 0.038;
	this.isEye = false;
	this.isCSF = this.data.isCSF;     // set in data file if contrast sensitivity function is to be shown
	
	this.lineCol = "#000";
	 if (this.data.graphLineColour !== undefined)
	 {
	     this.lineCol = this.data.graphLineColour; 
	 }
	 
	 this.backCol = "#FFF";
	 if (this.data.backgroundColour !== undefined)
	 {
	     this.backCol = this.data.backgroundColour;
	 }
	
        this.bgLayer = new OU.util.Layer({
            container:this, //container is the whole area and can contain other stuff
            id:'bg'
        });
	
        this.objectLayer = new OU.util.Layer({
            container:this,
            id:'objectLayer',
            x:this.w*0.05,
            y:2*bH + 0.02*this.h,
            w:this.w*0.25,
            h:this.h*0.5,
            hasEvents:true,
            pinch:this.pinch,
            pinchMe:this
        });
	
	 this.graphLayer = new OU.util.Layer({
            container:this,
            id:'graphLayer',
            x:this.objectLayer.x + this.objectLayer.w + 0.05*this.w,
            y:2*bH + 0.02*this.h,
            w:this.w*0.25,
            h:this.h*0.25,
            hasEvents:true,
            pinch:this.pinch,
            pinchMe:this
        });
	
	
	this.imageLayer = new OU.util.Layer({
            container:this,
            id:'imageLayer',
            x:this.graphLayer.x + this.graphLayer.w + 0.05*this.w,
            y:2*bH + 0.02*this.h,
            w:this.w*0.25,
            h:this.h*0.5,
            hasEvents:true,
            pinch:this.pinch,
            pinchMe:this
        });

        this.controlsLayer = new OU.util.Layer({
            container:this,
            id:'controlsLayer',
            x:this.graphLayer.x,
            y:this.graphLayer.y+this.graphLayer.h,
            w:this.graphLayer.w,
            h:this.h*0.25,
            hasEvents:true,
            pinch:this.pinch,
            pinchMe:this
        });
	
        this.controlsLayer.context.gradRect(); // use default background

        this.textDiv = new OU.util.Div({
            x:this.objectLayer.x,
            y:this.objectLayer.y + this.objectLayer.h + this.h*0.05,
            w:this.w*0.9,
            h:this.h*0.3,
            container:this
        });
        this.textDiv.div.innerHTML = this.data.instructions;
	
	this.debugDiv = new OU.util.Div({
            x:this.objectLayer.x,
            y:this.textDiv.y + this.textDiv.h + this.h*0.05,
            w:this.w*0.3,
            h:this.h*0.3,
            container:this
        });
	//this.debugDiv.div.innerHTML = "colour = " + this.baseColour;
		
        this.initControls();
        this.resize();    // sets all sizes and calls render()
    };


    OU.activity.SpatialFreqMTF.prototype.initControls = function () {
        var ctx = this.controlsLayer.context, self = this,
        clickable = this.controlsLayer.events.clickable, cH = this.controlsLayer.h, cW = this.controlsLayer.w;
			
        this.slider = new OU.util.Slider({
            container:this.controlsLayer,
            instance:'',
	    x:0,
            y:0.1*cH,
            w:cW,
            h:0.2*cH,
            sliderPos:this.initialSliderPos,
	    titlePosition:'above',
            drawContainer:false,
            title:'spatial frequency',
            showValue:false,
            callback:self.setSlider,
            callbackParam:{i:1,self:this}, //identifies which slider was clicked
            background:{
                clear:true
            },
            context:ctx
        });
        clickable.push(this.slider);

	this.ncyclesDiv = new OU.util.Div({
	    container:this,
            x:this.controlsLayer.x + 0.1*cW,
            y:this.controlsLayer.y + 0.3*cH,
            w:cW,
            h:0.18*cH
        });

	this.ncyclesDiv.div.innerHTML = "Spatial frequency = ";

	if (!this.isCSF)
	{
	    this.eyeButton = new OU.util.CheckBoxButton({
		txt:"lens/cornea system MTF",
		layer:this.controlsLayer,
		y:0.5*cH,
		h:cH*0.18,
		w:cW*0.9,
		x:0,
		state:false,
		onClick:function () {
		    self.isEye = true;
		    self.eyeButton.state(true);
		    self.lensButton.state(false);
		    self.render();
		}	 
	    });
	
	    this.lensButton = new OU.util.CheckBoxButton({
		txt:"perfect lens MTF",
		layer:this.controlsLayer,
		y:0.8*cH,
		h:cH*0.18,
		w:cW*0.9,
		x:0,
		state:true,
		onClick:function () {
		    self.isEye = false;
		    self.eyeButton.state(false);
		    self.lensButton.state(true);
		    self.render();
		}
	    });
	}
	    
    };

/*    OU.activity.SpatialFreqMTF.prototype.showHelp = function () {
        var self = this;
        new OU.util.PopUpInfo({
            container:this,
            txt:this.data.help,
            x:this.w*0.17,
            y:this.h*0.17,
            w:this.w*0.66,
            h:this.h*0.66,
            onClose:function () {
                self.helpDisp = false;
            }
        });
    };*/

    /**
     * The main renderer.
     */
    OU.activity.SpatialFreqMTF.prototype.render = function () {
	if (!this.isCSF)
	{
	    this.lensButton.render();
	    this.eyeButton.render();
	}
	this.slider.render();
        this.drawGraphs();
    };

    OU.activity.SpatialFreqMTF.prototype.resize = function () {
        OU.activity.SpatialFreqMTF.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight, ctxS = this.controlsLayer.context, cH = this.controlsLayer.h,
        cW = this.controlsLayer.w;
	
        this.bgLayer.resize();
	
	this.objectLayer.resize({
            x:this.w*0.05,
            y:2*bH + 0.02*this.h,
            w:this.w*0.25,
            h:this.h*0.5
        });
	
	this.graphLayer.resize({
            x:this.objectLayer.x + this.objectLayer.w + 0.05*this.w,
            y:2*bH + 0.02*this.h,
            w:this.w*0.25,
            h:this.h*0.25
        });
	
	this.imageLayer.resize({
            x:this.graphLayer.x + this.graphLayer.w + 0.05*this.w,
            y:2*bH + 0.02*this.h,
            w:this.w*0.25,
            h:this.h*0.5
        });
	
        this.controlsLayer.resize({
            x:this.graphLayer.x,
	    y:this.graphLayer.y+this.graphLayer.h,
            w:this.graphLayer.w,
            h:this.h*0.25
        });
	
        if (this.data.backgroundColour!==undefined)
	// use specified background colour
	{
            this.bgLayer.context.gradRect({
                col1:this.data.backgroundColour,
                col2:this.data.backgroundColour
            });
	    
	    ctxS.gradRect({
                col1:this.data.backgroundColour,
                col2:this.data.backgroundColour
            });	    	    
	}
	else
        {
	    this.bgLayer.context.gradRect(); // use default background
	    ctxS.gradRect();
	}

        this.slider.resize({
	    x:0,
            y:0.1*cH,
            w:cW,
            h:0.2*cH
        });
	
	 this.ncyclesDiv.resize({
	    x:this.controlsLayer.x + 0.1*cW,
            y:this.controlsLayer.y + 0.3*cH,
            w:cW,
            h:0.18*cH
	 });

	if (!this.isCSF)
	{
	    this.eyeButton.resize({
		y:0.5*cH,
		h:cH*0.18,
		w:cW*0.9,
		x:0   
	    });
	
	    this.lensButton.resize({
		y:0.8*cH,
		h:cH*0.18,
		w:cW*0.9,
	     x:0   
	    });
	}
	
        this.textDiv.resize({
            x:this.objectLayer.x,
            y:this.objectLayer.y + this.objectLayer.h + this.h*0.05,
            w:this.w*0.9,
            h:this.h*0.3
        });
			
	this.debugDiv.resize({
            x:this.objectLayer.x,
            y:this.textDiv.y + this.textDiv.h + this.h*0.05,
            w:this.w*0.3,
            h:this.h*0.3
        });
		
        this.render();
        
    };
   
    OU.activity.SpatialFreqMTF.prototype.drawGraphs = function () {	
	this.drawFunc(this.objectLayer, false);
	this.drawFunc(this.imageLayer, true);
	this.drawTransferFunc();
    }
    
     OU.activity.SpatialFreqMTF.prototype.drawFunc = function (layer, isModulated) {	
        var ctx = layer.context;
        var w = layer.w;
        var h = layer.h;
		
	ctx.fillStyle = this.backCol;
        ctx.fillRect(0, 0, w, h);
	
	var displayHt = Math.floor(0.4*h);   // the height of the upper display
	var origy = Math.floor(0.7*h);   // origy refers to the lower graph
	var A = Math.floor(0.2*h);
	
	var minPeriod = 3;
	var maxPeriod = 100;
	var xscale = w/1;   // rescale so that width of graph area represents 1 degree of visual angle
	
	var minFreq = 0.1;
	var maxFreq = 50;
	
	// following original version, slider changes period rather than frequency and is inverted so low slider position
	// corresponds to higher period (lower freq)
	//var period = maxPeriod - this.slider.sliderPos*(maxPeriod - minPeriod);
	var freq = minFreq + this.slider.sliderPos*(maxFreq - minFreq);
	
	var period = 1/freq;
	//var freq = 1/period;
	var modulation = 1;
	if (isModulated) modulation = this.getMod(freq);
	var fscale = 2*freq/xscale;
	this.ncyclesDiv.div.innerHTML = "Spatial frequency = " + freq.toFixed(1);
	
	
	//this.debugDiv.div.innerHTML = "period = " + period + ", freq = " + freq + ", mod = " + modulation;
	var intensity;
		
	ctx.strokeStyle = this.lineCol;
	ctx.beginPath();
	ctx.moveTo(0,origy);
				    
	for (var i = 0; i < w; i=i+1)
	{	
		intensity = modulation*Math.sin(Math.PI*fscale*i);    
		var colourVal = 127 + Math.floor(127*intensity);       
		ctx.fillStyle = 'rgb(' + colourVal + ',' + colourVal + ',' + colourVal + ')';
		ctx.fillRect(i,0,1, displayHt);
		ctx.lineTo(i, origy - A*intensity);
	}
	
	 ctx.stroke();
	 
	// draw grey borders around the graph
	ctx.lineWidth = "1";
	ctx.strokeStyle = "rgb(162,162,162)";
	ctx.strokeRect(0, origy - A*1.2, w-1, A*2.4);
	
    };

    OU.activity.SpatialFreqMTF.prototype.drawTransferFunc = function () {
	var ctx = this.graphLayer.context;
	var w = this.graphLayer.w;
        var h = this.graphLayer.h;
	
	// clear the graph area and draw border rectangle
	ctx.fillStyle = this.backCol;
	ctx.fillRect(0, 0, w, h);
	ctx.strokeStyle = this.lineCol;
        ctx.strokeRect(0, 0, w-1, h-1);
	
	var xscale = w/65;
	var yscale = h/1.7;
	
	var origx = 10*xscale;
	var origy = 1.3*yscale;
	var tickx = 0.02*w;
	var ticky = 0.03*h;
	
	//draw the axes and labels
	ctx.strokeStyle = this.lineCol;
	ctx.fillStyle = this.lineCol;
	
	ctx.beginPath();	
	ctx.moveTo(origx, origy - 1.1*yscale);
	ctx.lineTo(origx, origy);
	ctx.lineTo(origx + 52*xscale, origy);
	ctx.moveTo(origx, origy);
	ctx.lineTo(origx, origy + ticky);
	ctx.moveTo(origx + 25*xscale, origy);
	ctx.lineTo(origx + 25*xscale, origy + ticky);
	ctx.moveTo(origx + 50*xscale, origy);
	ctx.lineTo(origx + 50*xscale, origy + ticky);
	ctx.moveTo(origx, origy);
	ctx.lineTo(origx - tickx, origy);
	ctx.moveTo(origx, origy - 0.5*yscale);
	ctx.lineTo(origx - tickx, origy - 0.5*yscale);
	ctx.moveTo(origx, origy - yscale);
	ctx.lineTo(origx - tickx, origy - yscale);
	ctx.stroke();
	ctx.font= (h/10) + 'px' + ' verdana, sans-serif';	
	ctx.textBaseline = "top";
	var size = ctx.measureText("spatial frequency").width;
	ctx.fillText("spatial frequency", w/2 - 0.5*size, origy + h/10);
	size = ctx.measureText("0").width;
	ctx.fillText("0", origx - 0.5*size, origy + ticky + 2);
	ctx.fillText("0", origx - tickx - size - 2, origy - h/20);
	size = ctx.measureText("50").width;
	ctx.fillText("50", origx + 50*xscale - 0.5*size, origy + ticky + 2);
	var label = "MTF";
	if (this.isCSF) label = "CSF";
	size = ctx.measureText(label).width;
	ctx.rotate(-Math.PI/2);
	ctx.fillText(label, -origy + 0.5*yscale - 0.5*size, 2); 
	ctx.rotate(Math.PI/2);
	if (!this.isCSF)
	{
	    size = ctx.measureText("1.0").width;
	    ctx.fillText("1.0", origx - tickx - size - 2, origy - yscale - h/20);
	}

	// draw the function
	var maxx = 50*xscale;
	ctx.beginPath();
	ctx.moveTo(origx, origy - yscale*this.getMod(0));
	
	for (var i = 0; i < maxx; i++)
	{
	    ctx.lineTo(origx + i, origy - yscale*this.getMod(i/xscale));
	}
	ctx.stroke();
	
	
	// draw a red line at current frequency
	var minFreq = 0.1;
	var maxFreq = 50;	
	var freq = minFreq + this.slider.sliderPos*(maxFreq - minFreq);
	ctx.strokeStyle = "#F00";
	ctx.beginPath();
	ctx.moveTo(origx + freq*xscale, origy + ticky);
	ctx.lineTo(origx + freq*xscale, origy - 1.1*yscale);
	ctx.stroke();
    }
    
     OU.activity.SpatialFreqMTF.prototype.getMod = function(xvalue) {
	var yval;
	if (this.isCSF)
	{
	    yval = 1.6*Math.exp(-0.08*xvalue);
	    if (xvalue <= 7) yval = yval - 0.045*(xvalue - 7)*(xvalue - 7)/2;
	}
	else
	{
	    if (this.isEye)
	    {
		yval = Math.exp(-0.08*xvalue);
	    }
	    else
	    {
		yval = 1 - xvalue/50;
	    }
	}
	
	return yval;
     }
    
    OU.activity.SpatialFreqMTF.prototype.setSlider = function ( p, v ) { // Called when slider is moved
	v.self.slider.render();
        v.self.drawGraphs();       
    };
   

    /*OU.activity.Cafewall.prototype.pinch = function ( e, s, x, y, dx, dy, me ) { // called when pinch event detected
        var ns = ((e - s) / me.h) + me.tileView.s;
        ns = ns > 1 ? 1 : (ns < 0 ? 0 : ns);
        me.tileView.scale(ns, {
            x:x,
            y:y
        });
        me.zoomSlider.sliderPos = ns;
        me.zoomSlider.render();
    };//*/
    /**
     * @class
     */
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.SpatialFreqMTF, OU.util.Activity);
