/**
 *
 * @fileOverview StickyNote - Post It Note Style Application
 *
 * TODO -
 *      <ul>
 *          <li>change the resize method, so that it does a scale rather than
 *        change the size of the pinboard.</li>
 *          <li>Tidy up the drawing code</li>
 *          <li>remove magic numbers</li>
 *          <li>extend the data file to make the activity more flexible</li>
 *      </ul>
 *
 * @author Tegai Lansdell
 */

OU.require('OU.util.Button');
OU.require('OU.util.Draggable');
OU.require('OU.util.Layer');
OU.require('OU.util.ImageLoader');

/**
 * @class StickyNote
 * @extends OU.util.Activity
 * @param {Object} data - Holds the data content for a specific instance of the activity
 * @param {String|undefined} instance - A unique identifier name for this instance, defaults to 'a1'
 * @param {Controller Object|undefined} controller - A reference to the controller that initialised this instance, if undefined, the superclass will generate a new controller and create the reference to it as 'this.controller'
 */

OU.activity.StickyNote = function ( data, instance, controller ) {
    OU.activity.StickyNote.prototype.canvasView = function () {
        var self = this;

        this.bgLayer = new OU.util.Layer({
            container:this
        });
        this.bgLayer.context.gradRect(); // draw background on backdrop layer

        this.noteBgDimsX  = this.data.controlPannelWidth;
        this.noteBgDimsY  = this.data.noteBackgroundMargin;
        this.noteBgDimsW  = this.w - this.noteBgDimsX  - this.data.noteBackgroundMargin;
        this.noteBgDimsH  = this.h-(this.noteBgDimsY*2);

        this.bgLayer.context.fillStyle = this.data.noteBackgroundColour;
        this.bgLayer.context.roundRect( this.noteBgDimsX,  this.noteBgDimsY, this.noteBgDimsW,  this.noteBgDimsH, 0);
        this.bgLayer.context.fill();

        this.imageLayer = new OU.util.Layer({
            container:this,
            hasEvents:true
        });
        this.boundary = {
            x:5,
            y:5,
            w:this.w - 10,
            h:this.h - OU.controlHeight * 2 - 5
        };
        this.fbWidth = this.w;
        this.controlLayer = new OU.util.Layer({
            y:0,
            h:this.h,
            w:this.data.controlPannelWidth,
            container:this,
            hasEvents:true
        });
        this.imageLoader = new OU.util.ImageLoader({
            container:this,
            data:this.data,
            onLoad:function () {
                self.start();
            }
        });

    };


     /**
    *
    * resize - Main resize function
    * resiz ethe sticky note so it fits to the screen
    */
    OU.activity.StickyNote.prototype.resize = function () {
        var count=0, numItems = this.items.length, item;
        OU.activity.StickyNote.superClass_.resize.call(this); // call the parent class resize
        this.bgLayer.resize();
        this.bgLayer.context.gradRect(); // draw background on backdrop layer


        this.imageLayer.resize();
        this.boundary = {
            x:5,
            y:5,
            w:this.w - 10,
            h:this.h - OU.controlHeight * 2 - 5
        };
        this.fbWidth = this.w;
        this.controlLayer.resize({
            y:0,
            h:this.h,
            w:this.data.controlPannelWidth
        });

        this.noteBgDimsX  = this.data.controlPannelWidth;
        this.noteBgDimsY  = this.data.noteBackgroundMargin;
        this.noteBgDimsW  = this.w - this.noteBgDimsX  - this.data.noteBackgroundMargin;
        this.noteBgDimsH  = this.h-(this.noteBgDimsY*2);

        this.bgLayer.context.fillStyle = this.data.noteBackgroundColour;
        this.bgLayer.context.roundRect(this.noteBgDimsX , this.noteBgDimsY , this.noteBgDimsW, this.noteBgDimsH , 0);
        this.bgLayer.context.fill();

        for (count = numItems; count--;) {

            item = this.items[count];

            item.rightBorder = this.w - item.w - this.data.noteBackgroundMargin;
            item.bottomBorder = this.h - item.h - this.data.noteBackgroundMargin;

            if (item.x>item.rightBorder){item.draggable.x = item.x = item.rightBorder; }
            if (item.y>item.bottomBorder){item.draggable.y = item.y = item.bottomBorder-this.data.noteBackgroundMargin;}

        }


        this.copyButton.render();
        this.deleteButton.render();
        this.deleteAllButton.render();
        this.addButton.render();
        this.colourSelector.render();
        this.saveButton.render();

        this.doRender = true;
    };


    /**
     * start - run the activity and start the render loop
     *         also loads any previous notes that have been saved
     */
    OU.activity.StickyNote.prototype.start = function () {
        var i, item, ev = this.imageLayer.events, arrayOfItems = [],
          cX = this.data.controlPannelWidth, cY = this.data.noteBackgroundMargin;

        this.initItems();

        item = OU.LocalStorage.load(this.data.storageUId) || "";
        if (item!=="")
        {
          //console.log('retrievedObject: ', JSON.parse(item));
          arrayOfItems = JSON.parse(item);

          this.stickyNoteCountId=0;

          for (i = arrayOfItems.length; i--;)
          {
            this.stickyNoteCountId++;

            this.items.push(new this.Item({
                StickyNote:this,
                id:arrayOfItems[i].id,
                "notetext":arrayOfItems[i].notetext,
                "bgcolor":arrayOfItems[i].bgcolor,
                "events":ev,
                "x":arrayOfItems[i].x,
                "y":arrayOfItems[i].y,
                "w":this.data.noteWidth,
                "h":this.data.noteHeight,
                "leftBorder":cX,
                "rightBorder":this.w - this.data.noteWidth - this.data.noteBackgroundMargin,
                "topBorder":cY,
                "bottomBorder":this.h - this.data.noteHeight - this.data.noteBackgroundMargin,
                "cbfReOrder":this.reOrderItems
            }));
          }
          this.stickyNoteCountId++;
        }


        // start render loop
        this.doRender = true;
        this.renderLoop();
    };

   /**
    *   setStickyNoteColour - set the bg colour of the text entry textarea
    */
    OU.activity.StickyNote.prototype.setStickyNoteColour = function (colourValue) {
        document.getElementById("snNoteTextAreaId").style.backgroundColor=colourValue;
    };

   /**
    *   setStickyNoteColour - gets the bg colour of the text entry textarea
    */
    OU.activity.StickyNote.prototype.getStickyNoteColour = function () {
        return document.getElementById("snNoteTextAreaId").style.backgroundColor;
    };

   /**
    *   reOrderItems - reorders the zordering of the sticky notes
    */
    OU.activity.StickyNote.prototype.reOrderItems = function (id,parent) {

        var count=0, obj=[], len = parent.items.length;

        if (parent.currentStickyNoteId !== id && len>0)
        {
          for (count=len;count--;)
          {
            if(parent.items[count].id===id)
            {
              obj = parent.items.splice(count,1);
              parent.items.unshift(obj[0]);
              break;
            }
          }
        }
        parent.currentStickyNoteId = id;

    };

   /**
    *   deleteNote - deletes the currently selected sticky note
    */
    OU.activity.StickyNote.prototype.deleteNote = function () {
        var count=0, len = this.items.length, id=this.currentStickyNoteId;
        if (len>0)
        {
          for (count=len;count--;)
          {
            if(this.items[count].id===id)
            {
              this.items.splice(count,1);
              break;
            }
          }

        }
    };

   /**
    *   saveNotes - saves all the sticky notes
    */
    OU.activity.StickyNote.prototype.saveNotes = function () {

      var itemObject, arrayOfDivs = [], pbObj = this.items, i = 0;

      for( i = pbObj.length; i--;) {
        if(pbObj[i]) {
          if(pbObj[i].id !== undefined) {
            itemObject = new Object();
            itemObject.id       = pbObj[i].id;
            itemObject.notetext = pbObj[i].notetext;
            itemObject.bgcolor  = pbObj[i].bgcolor;
            itemObject.x        = pbObj[i].x;
            itemObject.y        = pbObj[i].y;

            arrayOfDivs.push(itemObject);
          }
        }
      }

      OU.LocalStorage.save(this.data.storageUId,JSON.stringify(arrayOfDivs));

    };

   /**
    *   deleteAllNotes - delete all the sticky notes
    */
    OU.activity.StickyNote.prototype.deleteAllNotes = function () {

        var itemArrayLength = this.items.length;
        if (itemArrayLength>0)
        {
          this.items.splice(0,itemArrayLength);
        }
    };

   /**
    *   copyNote - copies the currently selected notes text to
    *   the text entry textarea
    */
    OU.activity.StickyNote.prototype.copyNote = function () {
        var count=0, len = this.items.length, id=this.currentStickyNoteId;
        if (len>0)
        {
          for (count=len;count--;)
          {
            if(this.items[count].id===id)
            {
              document.getElementById("snNoteTextAreaId").value = this.items[count].notetext;
              this.setStickyNoteColour(this.items[count].bgcolor);
              break;
            }
          }
        }
    };

   /**
    *   initItems - initalise al lthe elemnts of the activity, buttons, notes, etc
    */
    OU.activity.StickyNote.prototype.initItems = function () {

        var yPos=0, self = this;

        this.items = [ ];

        this.stickyNoteCountId = 0;

        yPos=5;

       this.colourSelector = new this.colourBox({
            StickyNote:this,
            id:"colourSelector",
            "events":this.controlLayer.events,
            "image":this.data.noteDataList,
            "x":30,
            "y":yPos,
            "w":20,
            "gap":5,
            "ctx":this.controlLayer.context,
            "cbf":this.setStickyNoteColour
        });



        yPos=60;

        this.htmlDiv = new OU.util.Div({
            layer:this.controlLayer,
            x:25,
            y:yPos,
            w:this.data.controlPannelWidth-50,
            h:this.data.controlPannelWidth-50,
            container: this
        });

        yPos+=this.data.controlPannelWidth-50;
        this.htmlDiv.div.innerHTML = "<textarea id='snNoteTextAreaId' style='resize:none; width:"+(this.data.controlPannelWidth-80)+"px; height:"+(this.data.controlPannelWidth-80)+"px; background-color:#FFF7C6;'>"+this.data.defaultNoteText+"</textarea>";

        this.addButton = new OU.util.Button({
            layer:this.controlLayer,
            txt:this.data.addButtonText,
            x:0,
            y:yPos,
            w:this.data.controlPannelWidth-25,
            h:30,
            onClick:function () {
                self.createNote();
            }
         });

        yPos+= 40;

        this.copyButton = new OU.util.Button({
            layer:this.controlLayer,
            txt:this.data.copyButtonText,
            x:0,
            y:yPos,
            w:this.data.controlPannelWidth-25,
            h:30,
            onClick:function () {
                self.copyNote();
            }
         });

        yPos+= 40;

        this.deleteButton = new OU.util.Button({
            layer:this.controlLayer,
            txt:this.data.deleteButtonText,
            x:0,
            y:yPos,
            w:this.data.controlPannelWidth-25,
            h:30,
            onClick:function () {
                self.deleteNote();
            }
         });

         yPos+= 40;

         this.deleteAllButton = new OU.util.Button({
            layer:this.controlLayer,
            txt:this.data.deleteAllButtonText,
            x:0,
            y:yPos,
            w:this.data.controlPannelWidth-25,
            h:30,
            onClick:function () {
                self.deleteAllNotes();
            }
         });

         yPos+= 40;

         this.saveButton = new OU.util.Button({
            layer:this.controlLayer,
            txt:this.data.saveButtonText,
            x:0,
            y:yPos,
            w:this.data.controlPannelWidth-25,
            h:30,
            onClick:function () {
                self.saveNotes();
            }
         });

        this.copyButton.render();
        this.addButton.render();
        this.deleteButton.render();
        this.deleteAllButton.render();
        this.colourSelector.render();
        this.saveButton.render();


    };

   /**
    *   createNote - create new sticky note and put it on screen
    */
    OU.activity.StickyNote.prototype.createNote = function () {
       var  cX = this.data.controlPannelWidth, cY = this.data.noteBackgroundMargin,
            ev = this.imageLayer.events,
            noteTextString="";


        noteTextString = document.getElementById("snNoteTextAreaId").value;

        this.stickyNoteCountId++;

        this.items.unshift(new this.Item({
            StickyNote:this,
            id:this.stickyNoteCountId,
            "notetext":noteTextString,
            "bgcolor":this.getStickyNoteColour(),
            "events":ev,
            "x":cX,
            "y":cY,
            "w":this.data.noteWidth,
            "h":this.data.noteHeight,
            "leftBorder":cX,
            "rightBorder":this.w - this.data.noteWidth - this.data.noteBackgroundMargin,
            "topBorder":cY,
            "bottomBorder":this.h - this.data.noteHeight - this.data.noteBackgroundMargin,
            "cbfReOrder":this.reOrderItems
        }));

        this.currentStickyNoteId = this.stickyNoteCountId;

        this.render();

    };

   /**
    *   renderLoop - render loop
    */
    OU.activity.StickyNote.prototype.renderLoop = function () {
        var self = this;
        if (this.doRender)
            this.render();
        setTimeout(function () {
            self.renderLoop();
        }, 20);
    };

   /**
    *   render - draws all the notes to the image layer
    */
    OU.activity.StickyNote.prototype.render = function () {
        var i, numItems = this.items.length;
        this.imageLayer.clear();

        for (i = numItems; i--;) {
            this.items[i].render();
        }

    };



   /**
    * @class Item
    * @extends OU.util.StickyNote
    * @param {Object} params - Holds the data content for a specific instance of the item
    *
    * the item class is the sticky note that is displayed on the screen
    */
    OU.activity.StickyNote.prototype.Item = function ( params ) {
        this.StickyNote = params.StickyNote;
        this.events = params.events;
        this.id = params.id;
        this.notetext = params.notetext;
        this.bgcolor = params.bgcolor || '#fff',
        this.x = params.x;
        this.y = params.y;
        this.h = params.h;
        this.w = params.w;
        this.startX = this.x;
        this.startY = this.y;
        this.leftBorder = params.leftBorder;
        this.cbfReOrderItems = params.cbfReOrder;
        this.col = "#000";
        this.rightBorder = params.rightBorder;
        this.topBorder = params.topBorder;
        this.bottomBorder = params.bottomBorder;

       /**
        *   render - draw the sticky note
        */
        OU.activity.StickyNote.prototype.Item.prototype.render = function () {

            var ctx = this.StickyNote.imageLayer.context, lineWidth=0, newString="" , splitString =  [], textOffsetY=0, displayX=0;

            ctx.roundRect(this.x , this.y , this.w , this.h , 0);

            if (this.StickyNote.currentStickyNoteId === this.id )
            {
              ctx.lineWidth = 4;
              ctx.strokeStyle = '#C00';

            }
            else
            {
              ctx.lineWidth = 1;
              ctx.strokeStyle = '#000';
            }

            ctx.fillStyle = this.bgcolor;
            ctx.stroke();

            ctx.shadowColor="black";
            ctx.shadowBlur=10;
            ctx.shadowOffsetY=2;
            ctx.shadowOffsetX=2;

            ctx.fill();

            ctx.shadowBlur=0;
            ctx.shadowOffsetY=0;
            ctx.shadowOffsetX=0;

            //Text colour
            ctx.fillStyle = this.col;
            ctx.font = '14px ' + OU.theme.font;
            ctx.textAlign = 'center';


            // now to do the splitting of the String

            lineWidth   = ctx.measureText(this.notetext, 0, 0).width;
            splitString = this.notetext.split(" ");
            textOffsetY = 0;

            while (splitString.length>0)
            {

              newString ="";
              while (splitString.length>0)
              {

                lineWidth = ctx.measureText((newString +" "+ splitString[0]), 0, 0).width;

                if (lineWidth<(this.w-10) )
                {
                  newString += " " + splitString[0];
                  displayX = this.x + (ctx.measureText(newString, 0, 0).width)/2;
                  splitString.shift();
                }
                else
                {
                  if (newString==="")
                  {
                    //string needs spliting as it is too long
                    newString=splitString[0];
                    displayX = this.x + (this.w/2);
                    splitString.shift();
                  }
                  break;
                }

              }

              ctx.fillText(newString, displayX , (this.y + 10 + textOffsetY), this.w);

              textOffsetY+=20;
            }

            return;

        };

       /**
        *   startDrag - call the reorder function when the sticky note is initally dragged
        */
        OU.activity.StickyNote.prototype.Item.prototype.startDrag = function ( p ) {
            //sort the items so that the latest dragged is at the end of the queue, therefore render last next time (on top)
            p.me.cbfReOrderItems(p.me.id,p.me.StickyNote);

        };

       /**
        *   endDrag - action to be taken when the note has finished being dragged
        */
        OU.activity.StickyNote.prototype.Item.prototype.endDrag = function ( p ) {
            //Nothing to do
        };

       /**
        *   newPos - check the note doesn't go out of bounds when being dragged
        */
        OU.activity.StickyNote.prototype.Item.prototype.newPos = function ( p ) {
            var newX = p.x, newY = p.y;

            if (p.x<p.me.leftBorder){
              newX=p.me.leftBorder;
            }else if (p.x>p.me.rightBorder){
              newX=p.me.rightBorder;
            }

            p.me.x = newX;


            if (p.y<p.me.topBorder){
              newY=p.me.topBorder;
            }
            else if (p.y>p.me.bottomBorder){
              newY=p.me.bottomBorder;
            }

            p.me.y = newY;

            p.me.StickyNote.doRender = true;

        };

       /**
        *   dims - return teh notes dimensions
        */
        OU.activity.StickyNote.prototype.Item.prototype.dims = function () {

          return {
              w:this.w,
              h:this.h
          };

        };
      /**
        *   move - updated the note's x and y values
        */
        OU.activity.StickyNote.prototype.Item.prototype.move = function ( x, y ) {
            this.x = x;
            this.y = y;
            this.draggable.x = x;
            this.draggable.y = y;
        };
        this.draggable = new OU.util.Draggable({
            "me":this,
            "events":this.StickyNote.imageLayer.events,
            "x":this.x,
            "y":this.y,
            "h":this.h,
            "w":this.w,
            "onStart":this.startDrag,
            "onEnd":this.endDrag,
            "onMove":this.newPos
        });
        this.events.clickable.push(this.draggable);
    };

    /**
    * @class colourBox
    * @extends OU.util.StickyNote
    * @param {Object} params - Holds the data content for a specific instance of the item
    *
    * the colourBox class handles the colour choice boxes
    */
    OU.activity.StickyNote.prototype.colourBox = function ( params ) {
        this.StickyNote = params.StickyNote;
        this.events = params.events;
        this.id = params.id;
        this.data = params.image;
        this.x = params.x;
        this.y = params.y;
        this.gap = params.gap;
        this.w = params.w;
        this.ctx = params.ctx;
        this.cbf = params.cbf;

       /**
        *   render - draw the sticky note
        */
        OU.activity.StickyNote.prototype.colourBox.prototype.render = function () {

            var count,index=0;

            this.ctx.textAlign = 'left';
            this.ctx.fillStyle = "#c0c0c0";
            this.ctx.font = '10px ' + OU.theme.font;
            this.ctx.fillText("Choose colour:", this.x, this.y+5, (this.w*this.data.length));

            for (count=this.data.length-1; count>=0; count--,index++)
            {
              this.ctx.drawImage(this.data[index].image, this.x+((this.gap+this.w)*index), this.y+20);
            }

            return;

        };

      /**
        *   EventClick - event handler fo rthe colour boxes
        */
        OU.activity.StickyNote.prototype.colourBox.EventClick = function ( evt, colourBox ) {
          this.event = evt;
          this.colourBox = colourBox;
          this.isHit = function ( x, y, pressed ) {
            var ev = this.event, itemSelected=Math.floor((x-this.colourBox.x)/(this.colourBox.gap+this.colourBox.w));
            if (pressed) {
              if (itemSelected>=0 && itemSelected<this.colourBox.data.length){
                this.colourBox.cbf(this.colourBox.data[itemSelected].colour);
              }else{
                this.colourBox.cbf(this.colourBox.data[0].colour);
              }
            }
          }
        };

        this.events.clickable.push(new OU.activity.StickyNote.prototype.colourBox.EventClick(this.events, this)); // push event click area
      };

    /**
     * accessibleView - this function is called instead of canvasView if the browser does not support HTML5 canvas
     */
    OU.activity.StickyNote.prototype.renderAccessible = function () {
        var h;
        clearInterval(this.renderCycle);
        h = '<div id="accessibleView">';
        h += '<h1>' + this.data.title + '</h1>';
        h += '<p>' + this.data.description + '</p>';
        h += '<p></p>';
        h += '</div>';
        document.body.innerHTML = '';
        var accessible = document.createElement('div');
        accessible.innerHTML = h;
        document.body.appendChild(accessible);
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.StickyNote, OU.util.Activity);

