/**
 * @fileOverview Ball and shadow illusion
 *
 * @author Greg Black
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.Layer');
OU.require('OU.util.PopUpInfo');
/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.BallAndShadow = function ( data, instance, controller ) { //data and instance are passed in from index.html
    OU.activity.BallAndShadow.prototype.canvasView = function () {
        //var self = this
        var bH =  OU.controlHeight;
        // create Canvas Layers & Contexts
        this.radiansPerDegree = Math.atan(1.0)/45.0;

        this.bgLayer = new OU.util.Layer({
            container:this, //container is the whole area and can contain other stuff
            id:'bg'
        });
        //set colour of container layer (if specified in data.js) or use a default
        if (this.data.backgroundColour !== undefined)
            this.bgLayer.context.gradRect({
                col1:this.data.backgroundColour,
                col2:this.data.backgroundColour
            }); // use specified background colour
        else
            this.bgLayer.context.gradRect(); // use default background

        this.illusionLayer = new OU.util.Layer({
            container:this,
            id:'illusionLayer',
            x:this.w*.1,
            y:this.h*.1+bH,
            w:this.h*.7,
            h:this.h*.7,
            hasEvents:true,
            pinch:this.pinch,
            pinchMe:this
        });

        this.optionsLayer = new OU.util.Layer({
            container:this,
            id:'optionsLayer',
            x:this.w*.1,
            y:this.illusionLayer.y+this.illusionLayer.h,
            w:this.illusionLayer.w,
            h:this.h*.2,
            hasEvents:true,
            pinch:this.pinch,
            pinchMe:this
        });
        this.optionsLayer.context.gradRect(); // use default background

        var ctx = this.illusionLayer.context;
        if (this.data.illusionLayer!==undefined){
            ctx.stroke();
            ctx.fillStyle = this.baseColour;
            ctx.fillRect(0, 0, this.illusionLayer.w, this.illusionLayer.h);
        }
        else{
            ctx.stroke();
            ctx.fillStyle = this.data.baseColour;
            ctx.fillRect(0, 0, this.illusionLayer.w, this.illusionLayer.h);
        }
        this.textDiv = new OU.util.Div({
            x:this.illusionLayer.x+this.illusionLayer.w+this.w*.05,
            y:this.h*0.2,
            w:this.w*0.3,
            h:this.h/2,
            container:this
        });
        this.textDiv.div.innerHTML = this.data.instructions;

        this.ballRadius = this.illusionLayer.w/20;
        this.ballX = this.ballRadius;
        this.ballY = 0;
        this.shadowX = 0;
        this.shadowY = 0;
        this.noShadow = false;
        this.horizShadow = true;
        this.diagShadow = false;
        this.bothShadow = false;
        this.dX = 1;
	this.goingup = false;
        this.goingdown = false;
        this.tick = 15;
        this.periodx = 3;
        this.periody = this.periodx/3.0;
        this.yOffset = this.illusionLayer.h*.1;
  
        this.initControls();
        this.resize();
        this.render();
        this.renderLoop();
    };

    OU.activity.BallAndShadow.prototype.step = function () {
        if(this.ballX <= 2*this.ballRadius){
            this.goingup = true;
            this.goingdown = false;
        }
        else if(this.ballX >= this.illusionLayer.w*0.75){
            this.goingup = false;
            this.goingdown = true;
        }

        if(this.goingup){
            this.ballX=this.tick*this.illusionLayer.h*0.006;
            this.ballY=this.illusionLayer.h-this.tick*this.illusionLayer.h*0.004-this.yOffset;
            this.shadowX=this.tick*this.illusionLayer.h*0.006;
            this.shadowY=this.illusionLayer.h-this.tick*this.illusionLayer.h*0.004-this.yOffset;
            this.tick++;
        }
        else if(this.goingdown){
            this.ballX=this.tick*this.illusionLayer.h*0.006;
            this.ballY=this.illusionLayer.h-this.tick*this.illusionLayer.h*0.004-this.yOffset;
            this.shadowX=this.tick*this.illusionLayer.h*0.006;
            this.shadowY=this.illusionLayer.h-this.tick*this.illusionLayer.h*0.004-this.yOffset;
            this.tick--;
        }

	this.drawIllusion();
    };

    OU.activity.BallAndShadow.prototype.initControls = function () {
        var i, bH = OU.controlHeight, self = this, slH = this.optionsLayer.h, 
        slW = this.optionsLayer.w;

        this.shadowButtons = [];
        this.shadowButtonLabels = ["No shadow","Diagonal shadow","Horizontal shadow","Combined shadow"];
        for (i = 0; i < 4; i++) {
            this.shadowButtons[i] = new OU.util.CheckBoxButton({
                txt:this.shadowButtonLabels[i],
                x:i<2?i*slW/5:(i-2)*slW/5,
                y:i<2?0:bH,
                w:slW*.25,
                h:slH/3,
                layer:this.optionsLayer,
                onClick:function ( p ) {
                    self.setShadow(p);
                },
                onClickParam:i,
                state:false
            });
        }
        this.shadowButtons[0].state(true);

        this.helpBtn = new OU.util.Button({
            txt:"Help",
            padding:0,
            verticalPadding:0,
            layer:this.optionsLayer,
            onClick:function () {
                if (!self.helpDisp) {
                    self.helpDisp = true;
                    self.showHelp();
                }
            }
        });
    };

    OU.activity.BallAndShadow.prototype.showHelp = function () {
        var self = this;
        new OU.util.PopUpInfo({
            container:this,
            txt:this.data.help,
            x:this.w*.17,
            y:this.h*.17,
            w:this.w*.66,
            h:this.h*.66,
            onClose:function () {
                self.helpDisp = false;
            }
        });
    };

    /**
     * The main renderer.
     */
    OU.activity.BallAndShadow.prototype.render = function () {
        this.drawIllusion();
    };

    OU.activity.BallAndShadow.prototype.resize = function () {
        OU.activity.BallAndShadow.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight, i, ctxS = this.optionsLayer.context, ctxW = this.illusionLayer.context,
        slW = this.optionsLayer.w, slH = this.optionsLayer.h;
        this.bgLayer.resize();
        if (this.data.backgroundColour!==undefined)
            this.bgLayer.context.gradRect({
                col1:this.data.backgroundColour,
                col2:this.data.backgroundColour
            }); // use specified background colour
        else
            this.bgLayer.context.gradRect(); // use default background

        this.illusionLayer.resize({
            x:this.w*.1,
            y:this.h*.1+bH,
            w:this.h*.7,
            h:this.h*.7
        });
        
        this.optionsLayer.resize({
            x:this.w*.1,
            y:this.illusionLayer.y+this.illusionLayer.h,
            w:this.illusionLayer.w,
            h:this.h*.2
        });

        if (this.data.panelColour!==undefined)
            ctxS.gradRect({
                col1:this.data.panelColour,
                col2:this.data.panelColour
            }); // use specified background colour
        else
            ctxS.gradRect(); // use default background

        ctxW.stroke();
        ctxW.fillStyle = this.baseColour;
        ctxW.fillRect(0, 0, this.illusionLayer.w, this.illusionLayer.h);

        ctxS.font = '18px ' + OU.theme.font;
        ctxS.lineWidth = 2;
        ctxS.strokeStyle = '#c00';
        ctxS.fillStyle = '#c00';
        ctxS.textAlign = 'center';

        for (i = this.shadowButtons.length; i--;) {
            this.shadowButtons[i].resize({
                txt:this.shadowButtonLabels[i],
                x:i<2?i*slW/3:(i-2)*slW/3,
                y:i<2?0:bH,
                w:slW*.35,
                h:slH/3
            });
        }
        this.helpBtn.resize({
            x:this.optionsLayer.w*.8,
            y:this.optionsLayer.h*.2,
            w:this.optionsLayer.w*.2,
            h:this.optionsLayer.h*.25
        });
        this.textDiv.resize({
            x:this.illusionLayer.x+this.illusionLayer.w+this.w*.05,
            y:this.h*0.2,
            w:this.w*0.3,
            h:this.h/2
        });
        this.render();
        this.helpBtn.render();
        for(i = this.shadowButtons.length; i--;) {
            this.shadowButtons[i].render();
        }
    };

    OU.activity.BallAndShadow.prototype.renderLoop = function () {
        var self = this;
        this.step();
        this.drawIllusion();
        setTimeout(function() {
            self.renderLoop();
        },40);
    };

    OU.activity.BallAndShadow.prototype.drawIllusion = function () {
        var ctx = this.illusionLayer.context;
        var w = this.illusionLayer.h;
        var h = this.illusionLayer.h;
        ctx.clearRect(0,0,w,h);
        // Draw background in light colour
        ctx.fillStyle = "#FFFFFF";
        ctx.fillRect(0, 0, w-1, h-1);

        ctx.strokeStyle = "#f00";
        ctx.beginPath();
        var spacing = h/10;
        var y = h*0.9;
        var x = w/20;
        ctx.moveTo(x, y);
        ctx.lineTo(w-w/20, y);
        for(var i=0; i<12; i++){
            if(i>0 && i<10){
                var j = (i-5)*2;
                ctx.moveTo(w/2-j*w/2/9, h*.9);
                ctx.lineTo(w/2-j*w/2/14, h*.9-w/2);
            }
            ctx.moveTo(x, y);
            ctx.lineTo(w-x, y);
            spacing *= 0.86;
            y-=spacing;
            x+=spacing/3;
        }
        ctx.stroke();
        ctx.closePath();

        this.ballRadius = w/20;
        ctx.fillStyle = "rgba(127,127,110,0.8)";
        ctx.strokeStyle = "rgba(127,127,110,0.8)";
        ctx.beginPath();
        
        switch(this.shadowType){
            case 0: //no shadow
                break;
            case 1: //diagonal shadow
                ctx.arc(this.shadowX+this.ballRadius/2, this.ballY, this.ballRadius, 0, 360, false);
                break;
            case 2: //horizontal shadow
                ctx.arc(this.shadowX+this.ballRadius, h*.86, this.ballRadius, 0, 360, false);
                break;
            case 3: //combined shadow
                this.shadowX = this.ballX+this.ballRadius*0.8;
                this.shadowY = this.shadowY+0.5*this.ballRadius-Math.abs(10.0*Math.sin(this.tick*0.1));
                ctx.arc(this.shadowX, this.shadowY, this.ballRadius, 0, 360, false);
                break;
            default:
                break;
        }
        ctx.fill();

        ctx.beginPath();
        ctx.arc(this.ballX, this.ballY, this.ballRadius, 0, 360, false);
        var gradient = ctx.createRadialGradient(
        this.ballX-this.ballRadius/4, this.ballY-this.ballRadius/5, this.ballRadius/4,
        this.ballX, this.ballY, this.ballRadius);
        gradient.addColorStop(0, '#08BDFF');
        gradient.addColorStop(1, '#0039C6');
        ctx.fillStyle = gradient;
        ctx.fill();

        // Draw a black border
        ctx.strokeStyle = "#000000";
        ctx.strokeRect(0, 0, w - 1, h - 1);
    };

    OU.activity.BallAndShadow.prototype.setShadow = function ( idx ) {
        this.shadowType = idx;
        var i;
        for (i = this.shadowButtons.length; i--;){ //i = 0; i < this.shadowButtons.length; i++) {
            this.shadowButtons[i].state(i==idx ? true : false);
        }
        for(i = this.shadowButtons.length; i--;) {
            this.shadowButtons[i].render();
        }
    };

    /*OU.activity.BallAndShadow.prototype.pinch = function ( e, s, x, y, dx, dy, me ) { // called when pinch event detected
        var ns = ((e - s) / me.h) + me.tileView.s;
        ns = ns > 1 ? 1 : (ns < 0 ? 0 : ns);
        me.tileView.scale(ns, {
            x:x,
            y:y
        });
        me.zoomSlider.sliderPos = ns;
        me.zoomSlider.render();
    };//*/

    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.BallAndShadow, OU.util.Activity);
