/**
 * @fileoverview  OU.activity.Contrationchart - S104 Chemical Equilibrium Activity
 *
 * @param {JSON array} data Data set
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.DynText');
OU.require('OU.util.PopUpInfo');
OU.require('OU.util.Layer');
OU.require('OU.util.ImageLoader');
/**
 * @class
 */
OU.activity.ConcentrationChart = function ( data, instance, controller ) {
    OU.activity.ConcentrationChart.prototype.canvasView = function () {
        var self = this;
        OU.LocalStorage.save("ou.concentrationchart.reaction", 0);
        this.bgLayer = new OU.util.Layer({
            container:this
        });
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        this.imageLayer = new OU.util.Layer({
            container:this
        });
        this.controlLayer = new OU.util.Layer({
            container:this,
            hasEvents:true
        });
        this.reactions = this.data.reactions;
        this.currentReaction = this.reactions[0];
        this.imageLoader = new OU.util.ImageLoader({
            container:this,
            data:this.data,
            onLoad:function () {
                self.imagesLoaded = true;
                self.start();
            }
        });
    };
    OU.activity.ConcentrationChart.prototype.resize = function () {
        OU.activity.ConcentrationChart.superClass_.resize.call(this); // call the parent class resize 
        var b, barHeight, i, r = this.currentReaction,
            n = r.buttonToggles.length, bH;
        if (this.popUp)
            this.popUp.resize({
                x:this.w * .20,
                y:this.h * .15,
                w:this.w * .6,
                h:this.h * .7
            });
        if (this.imagesLoaded) {
            this.bgLayer.resize();
            this.bgLayer.context.gradRect(); // draw background on backdrop layer
            this.imageLayer.resize();
            if (this.w > this.h) { // landscape
                this.barsBoundary = {
                    x:this.w * .05,
                    y:this.h * .05,
                    w:this.w * .6,
                    h:this.h * .9
                };
                this.buttonsBoundary = {
                    x:this.w * .65 + 10,
                    y:this.h * .05,
                    w:this.w * .3 - 10,
                    h:this.h * .9
                };
            }
            else { // portrait
                this.barsBoundary = {
                    x:this.w * .05,
                    y:this.h * .05,
                    w:this.w * .9,
                    h:this.h * .6
                };
                this.buttonsBoundary = {
                    x:this.w * .05,
                    y:this.h * .65 + 10,
                    w:this.w * .9,
                    h:this.h * .3 - 10
                };
            }
            b = this.barsBoundary;
            barHeight = b.h * .08;
            if (this.data.changeButton==1) {
                this.changeButton.resize({
                    x:b.x + b.w / 3,
                    y:b.y + b.h - barHeight * 1.5,
                    w:b.w / 3,
                    h:barHeight
                });
            }
            b = this.buttonsBoundary;
            bH = b.h / ((n + 1) * 1.5);
            bH = bH > 50 ? 50 : bH;
            if (this.data.resetButton==1) {
                this.resetButton.resize({
                    x:b.x + b.w / 4 + (bH + 10) / 2,
                    y:b.y + bH * 8.5,
                    w:b.w / 2,
                    h:bH
                });
            }
            for (i = n; i--;) {
                if (this.buttonToggles[i] >= 0) {
                    this.buttons[i].resize({
                        x:b.x + b.w / 4 + (bH + 10) / 2,
                        y:b.y + bH * (i + 0.5) * 1.5,
                        w:b.w / 2,
                        h:bH
                    });
                }
            }
            this.controlLayer.resize();
            this.drawControl();
            this.redrawBars();
        }
    };
    OU.activity.ConcentrationChart.prototype.start = function () {
        this.setupReaction(0);
        this.initControls();
        this.resize(); // call resize to perform initial render
    };
    OU.activity.ConcentrationChart.prototype.Button = function ( params ) {
        params.fontWeight = 'normal';
        if (params.background===undefined)
            params.background = {};
        params.background.borderCol = '#444';
        // params.background.shadow=true;
        params.background.radius = 10;
        return new OU.util.Button(params);
    };
    OU.activity.ConcentrationChart.prototype.initControls = function () {
        var r = this.currentReaction, i, clickable = this.controlLayer.events.clickable;
        // create button - note, no need to pass dims here as will be resized before rendering
        clickable.length = 0;
        if (this.data.changeButton==1) {
            this.changeButton = new this.Button({
                context:this.controlLayer.context,
                events:this.controlLayer.events,
                txt:"Change",
                onClick:this.changeReaction,
                onClickParam:{
                    me:this
                },
                colour:"#" + this.data.enabledColour,
                background:{
                    RGB:'255,255,255'
                }
            });
            clickable.push(this.changeButton);
        }
        if (this.data.resetButton==1) {
            if (this.data.resetDisabled) {
                this.resetButton = new this.Button({
                    context:this.controlLayer.context,
                    events:this.controlLayer.events,
                    txt:"Reset",
                    colour:"#" + this.data.disabledColour,
                    background:{
                        RGB:'255,255,255'
                    }
                });
            }
            else {
                this.resetButton = new this.Button({
                    context:this.controlLayer.context,
                    events:this.controlLayer.events,
                    txt:"Reset",
                    onClick:this.resetReaction,
                    onClickParam:{
                        me:this
                    },
                    colour:"#" + this.data.enabledColour,
                    background:{
                        RGB:'255,255,255'
                    }
                });
                clickable.push(this.resetButton);
            }
        }
        this.buttons = [];
        for (i = this.buttonToggles.length; i--;) {
            if (this.buttonToggles[i]==1) {
                clickable.push(this.buttons[i] = new this.Button({
                    context:this.controlLayer.context,
                    events:this.controlLayer.events,
                    txt:r.menuChars[i],
                    onClick:this.buttonClicked,
                    onClickParam:{
                        choice:i,
                        me:this
                    },
                    colour:"#" + this.data.enabledColour,
                    background:{
                        RGB:'255,255,255'
                    }
                }));
            }
            else if (this.buttonToggles[i]==0) {
                this.buttons[i] = new this.Button({
                    context:this.controlLayer.context,
                    events:this.controlLayer.events,
                    txt:r.menuChars[i],
                    colour:"#" + this.data.disabledColour,
                    background:{
                        RGB:'255,255,255'
                    }
                });
            }
        }
    };
    OU.activity.ConcentrationChart.prototype.drawControl = function () {
        var grd, b = this.barsBoundary, barHeight = b.h * .08, i, m = this.buttonsBoundary, bH,
            conCtx = this.controlLayer.context,
            r = this.currentReaction,
            bgCtx = this.bgLayer.context;
        this.controlLayer.clear();
        this.bgLayer.clear();
        // draw plinth
        bgCtx.strokeStyle = "#666666";
        grd = bgCtx.createLinearGradient(0, 0, 0, 175);
        grd.addColorStop(0, "#EEEEEE");
        grd.addColorStop(1, "#FFFFFF");
        bgCtx.fillStyle = grd;
        bgCtx.lineWidth = 1;
        bgCtx.beginPath();
        bgCtx.fillRect(this.barsBoundary.x, this.barsBoundary.y, this.barsBoundary.w, this.barsBoundary.h);
        bgCtx.strokeRect(this.barsBoundary.x, this.barsBoundary.y, this.barsBoundary.w, this.barsBoundary.h);
        bgCtx.fillRect(this.buttonsBoundary.x, this.buttonsBoundary.y, this.buttonsBoundary.w, this.buttonsBoundary.h);
        bgCtx.strokeRect(this.buttonsBoundary.x, this.buttonsBoundary.y, this.buttonsBoundary.w, this.buttonsBoundary.h);
        bgCtx.closePath();
        bgCtx.stroke();
        conCtx.strokeStyle = "#000";
        if (this.data.changeButton==1) {
            this.changeButton.render();
        }
        if (this.data.resetButton==1) {
            this.resetButton.render();
        }
        bH = m.h / ((this.buttonToggles.length + 1) * 1.5);
        bH = bH > 50 ? 50 : bH;
        for (i = this.buttonToggles.length; i--;) {
            if (this.buttonToggles[i] >= 0) {
                this.buttons[i].render();
                if (this.data.menu[i].image!==undefined)
                    bgCtx.drawImage(this.data.menu[i].image, m.x + m.w / 4 - (bH + 10) / 2, m.y + bH * (i + 0.5) * 1.5, bH, bH);
            }
        }
        new OU.util.DynText({
            txt:this.currentReaction.formula,
            x:b.x,
            y:b.y + b.h - barHeight * 2.5,
            w:b.w,
            h:barHeight,
            context:conCtx,
            align:'center'
        });
    };
    OU.activity.ConcentrationChart.prototype.redrawBars = function () {
        var ctx = this.imageLayer.context, bA = this.barsBoundary,
            r = this.currentReaction, levels = [],
            b = {
                x:bA.x + bA.w * .2,
                y:bA.y + bA.h * .2,
                w:bA.w * .6,
                h:bA.h * .6
            },
            barHeight = b.h / 10,
            WidthAt0Point1M = b.w / 2;
        levels[0] = Math.floor((r.NewConcA / 0.1) * WidthAt0Point1M);
        levels[1] = Math.floor((r.NewConcB / 0.1) * WidthAt0Point1M);
        levels[2] = Math.floor((r.NewConcC / 0.1) * WidthAt0Point1M);
        levels[3] = Math.floor((r.NewConcD / 0.1) * WidthAt0Point1M);
        ctx.strokeStyle = '#999';
        ctx.fillStyle = '#fff';
        ctx.beginPath();
        ctx.moveTo(b.x, b.y);
        ctx.lineTo(b.x, b.y + barHeight * 6.5);
        ctx.stroke();
        ctx.beginPath();
        ctx.moveTo(b.x + WidthAt0Point1M, b.y);
        ctx.lineTo(b.x + WidthAt0Point1M, b.y + barHeight * 6.5);
        ctx.stroke();
        ctx.beginPath();
        ctx.moveTo(b.x + 2 * WidthAt0Point1M, b.y);
        ctx.lineTo(b.x + 2 * WidthAt0Point1M, b.y + barHeight * 6.5);
        ctx.stroke();
        new OU.util.DynText({
            context:ctx,
            txt:"0M",
            x:b.x - barHeight,
            y:b.y - barHeight,
            w:barHeight * 2,
            h:barHeight,
            align:'center'
        });
        new OU.util.DynText({
            context:ctx,
            txt:"0.1M",
            x:WidthAt0Point1M + b.x - barHeight,
            y:b.y - barHeight,
            w:barHeight * 2,
            h:barHeight,
            align:'center'
        });
        new OU.util.DynText({
            context:ctx,
            txt:"0.2M",
            x:2 * WidthAt0Point1M + b.x - barHeight,
            y:b.y - barHeight,
            w:barHeight * 2,
            h:barHeight,
            align:'center'
        });
        new OU.util.DynText({
            context:ctx,
            txt:r.barChars[0],
            x:bA.x,
            y:b.y + barHeight * .5,
            w:b.x - bA.x,
            h:barHeight,
            padding:10
        });
        new OU.util.DynText({
            context:ctx,
            txt:r.barChars[1],
            x:bA.x,
            y:b.y + barHeight * 2,
            w:b.x - bA.x,
            h:barHeight,
            padding:10
        });
        new OU.util.DynText({
            context:ctx,
            txt:r.barChars[2],
            x:bA.x,
            y:b.y + barHeight * 3.5,
            w:b.x - bA.x,
            h:barHeight,
            padding:10
        });
        new OU.util.DynText({
            context:ctx,
            txt:r.barChars[3],
            x:bA.x,
            y:b.y + barHeight * 5,
            w:b.x - bA.x,
            h:barHeight,
            padding:10
        });
        ctx.fillRect(b.x, b.y + barHeight * 0.5, WidthAt0Point1M * 2.2, barHeight);
        ctx.fillRect(b.x, b.y + barHeight * 2, WidthAt0Point1M * 2.2, barHeight);
        ctx.fillRect(b.x, b.y + barHeight * 3.5, WidthAt0Point1M * 2.2, barHeight);
        ctx.fillRect(b.x, b.y + barHeight * 5, WidthAt0Point1M * 2.2, barHeight);
        ctx.strokeRect(b.x, b.y + barHeight * 0.5, WidthAt0Point1M * 2.2, barHeight);
        ctx.strokeRect(b.x, b.y + barHeight * 2, WidthAt0Point1M * 2.2, barHeight);
        ctx.strokeRect(b.x, b.y + barHeight * 3.5, WidthAt0Point1M * 2.2, barHeight);
        ctx.strokeRect(b.x, b.y + barHeight * 5, WidthAt0Point1M * 2.2, barHeight);
        ctx.fillStyle = '#0000EF';
        ctx.fillRect(b.x, b.y + barHeight * 0.5, levels[0], barHeight);
        ctx.fillStyle = '#FFFF00';
        ctx.fillRect(b.x, b.y + barHeight * 2, levels[1], barHeight);
        ctx.fillStyle = '#EF0000';
        ctx.fillRect(b.x, b.y + barHeight * 3.5, levels[2], barHeight);
        ctx.fillStyle = '#00EF00';
        ctx.fillRect(b.x, b.y + barHeight * 5, levels[3], barHeight);
    };
    OU.activity.ConcentrationChart.prototype.changeReaction = function ( p ) {
        var self = p.me, me = self, infoText = '<h1>Select a Chemical Equation</h1>',
            currentReaction = self.currentReaction.reactionNumber, fm;
        me.controlLayer.events.flush();
        infoText += '<form name="pieform">';
        infoText += '<table><tr><td>';
        for (fm = 0; fm < me.data.reactions.length; fm++) {
            if (fm==currentReaction) {
                infoText += '<input type="radio" name="group1" id="radiofield' + fm + '" value="' + fm + '" checked="checked" onclick="OU.LocalStorage.save(\'ou.concentrationchart.reaction\',' + fm + ')"/><label for="radiofield' + fm + '"> ' + me.data.reactions[fm].formula + '</label>';
            }
            else {
                infoText += '<input type="radio" name="group1" id="radiofield' + fm + '" value="' + fm + '"  onclick="OU.LocalStorage.save(\'ou.concentrationchart.reaction\',' + fm + ')"/><label for="radiofield' + fm + '"> ' + me.data.reactions[fm].formula + '</label>';
            }
            if (fm!=me.data.reactions.length - 1) {
                infoText += '<br/>';
            }
        }
        infoText += '</td></tr></table></form>';
        this.popUp = new OU.util.PopUpInfo({
            container:self,
            txt:infoText,
            x:me.w * .20,
            y:me.h * .15,
            w:me.w * .6,
            h:me.h * .7,
            onClose:function () {
                self.setupReaction(OU.LocalStorage.load('ou.concentrationchart.reaction'));
                self.initControls();
                self.resize();
            }
        });
        if (self.data.onChange!==undefined)
            self.data.onChange();
    };
    OU.activity.ConcentrationChart.prototype.step = function ( p ) {
        var i, j, action;
        if (p.step==this._currentStep)
            return;
        this._currentStep = p.step;
        for (i = p.actions.length; i--;) {
            action = p.actions[i];
            switch(action.type) {
                case "turnOnReset":
                    this.data.resetDisabled = false;
                    this.initControls();
                    this.resize();
                    break;
                case "turnOffReset":
                    this.data.resetDisabled = true;
                    this.initControls();
                    this.resize();
                    break;
                case "setResetFunction":
                    this.onReset = action.value;
                    break;
                case "setChangeFunction":
                    this.data.onChange = action.value;
                    break;
                case "updateButtonToggles":
                    this.currentReaction.buttonToggles = action.value;
                    for (j = this.currentReaction.buttonToggles.length; j--;)
                        this.buttonToggles[j] = this.currentReaction.buttonToggles[j];
                    this.initControls();
                    this.resize();
                    break;
            }
        }
    };
    OU.activity.ConcentrationChart.prototype.resetReaction = function ( p ) {
        var self = p.me;
        self.setupReaction(self.currentReaction.reactionNumber);
        self.initControls();
        self.resize();
        if (self.onReset!==undefined) {
            self.onReset();
            self.onReset = undefined;
        }
    };
    OU.activity.ConcentrationChart.prototype.setupReaction = function ( n ) {
        var Ca, Cb, Cc, Cd, i,
            r = this.reactions[n];
        r.ConcA = r.initLevels.r1;
        r.ConcB = r.initLevels.r2;
        r.ConcC = r.initLevels.p1;
        r.ConcD = r.initLevels.p2;
        r.NewConcA = r.ConcA;
        r.NewConcB = r.ConcB;
        r.NewConcC = r.ConcC;
        r.NewConcD = r.ConcD;
        r.K = r.initLevels.k;
        Cb = r.ConcB==0 ? 1 : r.ConcB;
        Cd = r.ConcD==0 ? 1 : r.ConcD;
        Ca = r.ConcA;
        Cc = r.ConcC;
        r.K = (Cc * Cd) / (Ca * Cb);
        this.increasingChemical = undefined;
        r.reactionNumber = n;
        this.buttonToggles = [];
        for (i = r.buttonToggles.length; i--;)
            this.buttonToggles[i] = r.buttonToggles[i];
        this.currentReaction = r;
    };
    OU.activity.ConcentrationChart.prototype.buttonClicked = function ( p ) {
        var i = p.choice, self = p.me;
        self.AddTo(i + 1);
    };
    OU.activity.ConcentrationChart.prototype.AddTo = function ( Chemical ) {
        var Increment = 0.02, a, b, c, y2, i,
            cR = this.currentReaction;
        if (this.increasingChemical===undefined) {
            this.increasingChemical = Chemical;
            this.increase = Increment;
            for (i = this.buttonToggles.length; i--;) {
                if (i!=Chemical - 1 && this.buttonToggles[i]==1)
                    this.buttonToggles[i] = 0;
            }
            this.initControls();
        } else if (Chemical!=this.increasingChemical) {
            return;
        }
        else {
            this.increase = this.increase + Increment;
        }
        if (this.increase > 0.2)
            return;
        if ((Chemical==1) || (Chemical==2)) { // If Reactants
            //  {1,1,1,1}
            if ((cR.ConcA!=0) && (cR.ConcB!=0) && (cR.ConcC!=0) && (cR.ConcD!=0)) {
                a = cR.K - 1;
                b = -2 * cR.K * cR.ConcB - cR.K * this.increase - 2 * cR.ConcC;
                c = cR.K * (cR.ConcB * cR.ConcB) + cR.K * cR.ConcB * this.increase - (cR.ConcC * cR.ConcC);
            }
            //   {1,0,1,1}
            if ((cR.ConcA!=0) && (cR.ConcB==0) && (cR.ConcC!=0) && (cR.ConcD!=0)) {
                a = -1;
                b = -cR.K - 2 * cR.ConcC;
                c = cR.K * cR.ConcA + cR.K * this.increase - (cR.ConcC * cR.ConcC);
            }
            //   {1,1,1,0}
            if ((cR.ConcA!=0) && (cR.ConcB!=0) && (cR.ConcC!=0) && (cR.ConcD==0)) {
                a = cR.K;
                b = -2 * cR.K * cR.ConcB - cR.K * this.increase - 1;
                c = cR.K * (cR.ConcB * cR.ConcB) + cR.K * cR.ConcB * this.increase - cR.ConcC;
            }
        }
        if ((Chemical==3) || (Chemical==4)) { // If Products
            if ((cR.ConcA!=0) && (cR.ConcB!=0) && (cR.ConcC!=0) && (cR.ConcD!=0)) {
                //  {1,1,1,1}
                a = (1 / cR.K) - 1;
                b = -2 * (1 / cR.K) * cR.ConcD - (1 / cR.K) * this.increase - 2 * cR.ConcA;
                c = (1 / cR.K) * (cR.ConcD * cR.ConcD) + (1 / cR.K) * cR.ConcD * this.increase - (cR.ConcA * cR.ConcA);
            }
            if ((cR.ConcA!=0) && (cR.ConcB==0) && (cR.ConcC!=0) && (cR.ConcD!=0)) {
                //   {1,0,1,1}
                a = (1 / cR.K);
                b = -2 * (1 / cR.K) * cR.ConcD - (1 / cR.K) * this.increase - 1;
                c = (1 / cR.K) * (cR.ConcD * cR.ConcD) + (1 / cR.K) * cR.ConcD * this.increase - cR.ConcA;
            }
            if ((cR.ConcA!=0) && (cR.ConcB!=0) && (cR.ConcC!=0) && (cR.ConcD==0)) {
                //   {1,1,1,0}
                a = -1;
                b = -(1 / cR.K) - 2 * cR.ConcA;
                c = (1 / cR.K) * cR.ConcC + (1 / cR.K) * this.increase - (cR.ConcA * cR.ConcA);
            }
        }
        y2 = (-b - Math.sqrt((b * b) - 4 * a * c)) / (2 * a);
        switch(Chemical) {
            case 1 :
                cR.NewConcA = cR.ConcA + this.increase - y2;
                if (cR.ConcB!=0) {
                    cR.NewConcB = cR.ConcB - y2;
                }
                cR.NewConcC = cR.ConcC + y2;
                if (cR.ConcD!=0) {
                    cR.NewConcD = cR.ConcD + y2;
                }
                break;
            case 2 :
                cR.NewConcA = cR.ConcA - y2;
                cR.NewConcB = cR.ConcB + this.increase - y2;
                cR.NewConcC = cR.ConcC + y2;
                if (cR.ConcD!=0) {
                    cR.NewConcD = cR.ConcD + y2;
                }
                break;
            case 3 :
                cR.NewConcA = cR.ConcA + y2;
                if (cR.ConcB!=0) {
                    cR.NewConcB = cR.ConcB + y2;
                }
                cR.NewConcC = cR.ConcC + this.increase - y2;
                if (cR.ConcD!=0) {
                    cR.NewConcD = cR.ConcD - y2;
                }
                break;
            case 4 :
                cR.NewConcA = cR.ConcA + y2;
                if (cR.ConcB!=0) {
                    cR.NewConcB = cR.ConcB + y2;
                }
                cR.NewConcC = cR.ConcC - y2;
                cR.NewConcD = cR.ConcD + this.increase - y2;
                break;
        }
        this.resize();
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.ConcentrationChart, OU.util.Activity);
