/**
 * OU.activity.Crystals - renders a 3D representation of a molecule - ball and stick stylie!
 *
 * draw polyhedra from extra data from mol file
 *
 * @author Will Rawes
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.HtmlBox');
OU.require('OU.util.Slider');
OU.require('OU.util.varControlBank');
OU.require('OU.util.Layer');
// Atom Types
OU.OTHER = 0;
OU.HYDROGEN = 1;
OU.CARBON = 2;
OU.NITROGEN = 3;
OU.OXYGEN = 4;
OU.FLUORINE = 5;
OU.CHLORINE = 6;
OU.BROMINE = 7;
OU.IODINE = 8;
OU.NOBLEGAS = 9;
OU.PHOSPHORUS = 10;
OU.SULFUR = 11;
OU.BORON = 12;
OU.ALKALIMETALS = 13;
OU.ALKALINEEARTHMETALS = 14;
OU.TITANIUM = 15;
OU.IRON = 16;
OU.SILICON = 17;
OU.ALUMINIUM = 18;
OU.SODIUM = 19;
OU.CALCIUM = 20;
OU.POTASSIUM = 21;
OU.YELLOW = 22;
OU.BROWN = 23;
OU.PURPLE = 24;
OU.LIGHTBLUE = 25;
OU.ORANGE = 26;
OU.atoms = [];
OU.atoms[OU.OTHER] = {
    name:'?',
    col:'#DD77FF',
    text:'#444',
    radius:0.05
};
OU.atoms[OU.HYDROGEN] = {
    name:'H',
    col:'#dddddd',
    text:'#444',
    radius:0.037
};
OU.atoms[OU.CARBON] = {
    name:'C',
    col:'#663300',
    text:'#eee',
    radius:0.17077
};
OU.atoms[OU.NITROGEN] = {
    name:'N',
    col:'#2233FF',
    text:'#fff',
    radius:0.070
};
OU.atoms[OU.OXYGEN] = {
    name:'O',
    col:'#FF2222',
    text:'#fff',
    radius:0.23073
};
OU.atoms[OU.FLUORINE] = {
    name:'F',
    col:'#55BB00',
    text:'#fff',
    radius:0.072
};
OU.atoms[OU.CHLORINE] = {
    name:'Cl',
    col:'#55BB00',
    text:'#fff',
    radius:0.120
};
OU.atoms[OU.BROMINE] = {
    name:'Br',
    col:'#992200',
    text:'#fff',
    radius:0.114
};
OU.atoms[OU.IODINE] = {
    name:'I',
    col:'#6600BB',
    text:'#fff',
    radius:0.133
};
OU.atoms[OU.NOBLEGAS] = {
    name:'He/Ne/Ar/Xe/Kr',
    col:'#00FFFF',
    text:'#444',
    radius:0.05
};
OU.atoms[OU.PHOSPHORUS] = {
    name:'P',
    col:'#FF9900',
    text:'#444',
    radius:0.11
};
OU.atoms[OU.SULFUR] = {
    name:'S',
    col:'#DDDD00',
    text:'#444',
    radius:0.3103
};
OU.atoms[OU.BORON] = {
    name:'B',
    col:'#FFAA77',
    text:'#444',
    radius:0.09
};
OU.atoms[OU.ALKALIMETALS] = {
    name:'Li/Na/K/Rb/Cs',
    col:'#7700FF',
    text:'#fff',
    radius:0.152
};
OU.atoms[OU.ALKALINEEARTHMETALS] = {
    name:'Be/Mg/Ca/Sr/Ba/Ra',
    col:'#007700',
    text:'#fff',
    radius:0.222
};
OU.atoms[OU.TITANIUM] = {
    name:'Ti',
    col:'#999999',
    text:'#000',
    radius:0.14
};
OU.atoms[OU.IRON] = {
    name:'Fe',
    col:'#FF5500',
    text:'#fff',
    radius:0.2076
};
OU.atoms[OU.SILICON] = {
    name:'Si',
    col:'#8888ff',
    text:'#fff',
    radius:0.1505
};
OU.atoms[OU.ALUMINIUM] = {
    name:'Al',
    col:'#8888ff',
    text:'#fff',
    radius:0.05
};
OU.atoms[OU.SODIUM] = {
    name:'Na',
    col:'#BBBB00',
    text:'#fff',
    radius:0.08
};
OU.atoms[OU.CALCIUM] = {
    name:'Ca',
    col:'#00BB00',
    text:'#fff',
    radius:0.03
};
OU.atoms[OU.POTASSIUM] = {
    name:'K',
    col:'#00AAAA',
    text:'#fff',
    radius:0.04
};
OU.atoms[OU.YELLOW] = {
    name:'Y',
    col:'#BBBB00',
    text:'#fff',
    radius:0.02
};
OU.atoms[OU.BROWN] = {
    name:'Bwn',
    col:'#886600',
    text:'#fff',
    radius:0.13076
};
OU.atoms[OU.PURPLE] = {
    name:'Prp',
    col:'#AA33AA',
    text:'#fff',
    radius:0.04
};
OU.atoms[OU.LIGHTBLUE] = {
    name:'LBl',
    col:'#00EEFF',
    text:'#fff',
    radius:0.03
};
OU.atoms[OU.ORANGE] = {
    name:'Org',
    col:'#f0a010',
    text:'#fff',
    radius:0.13076
};
OU.activity.Crystals = function ( data, instance, controller ) {
    OU.activity.Crystals.prototype.canvasView = function () {
        var bH = OU.controlHeight, ctx;
        this.config = {
            depth:5,
            fps:40, // 40ms = 25 fps
            spaceFillFactor:3.5
        };
        // create Canvas Layers & Contexts
        this.bgLayer = new OU.util.Layer({
            container:this
        });
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        this.PAD = this.h * .045;
        this.inRender = false;
        this.crystal = function () {
        };
        this.controls = [];
        this.modelLayer = new OU.util.Layer({
            container:this,
            'id':'model',
            y:this.y,
            'h':this.h - bH,
            x:this.x + 10,
            w:this.w - 10,
            'hasEvents':true,
            pinch:this.pinch,
            pinchMe:this
        });
        this.modelLayer.events.clickable.push(this);
        ctx = this.modelLayer.context;
        ctx.textAlign = 'center';
        ctx.font = '16px ' + OU.theme.font;
        this.controlsLayer = new OU.util.Layer({
            container:this,
            'y':this.y + this.h - bH,
            w:this.w,
            'h':bH,
            'id':'controls',
            hasEvents:true
        });
        this.rotatable = this.data.settings.rotatable;
        this.spinOn = this.data.settings.spin;
        this.atomicAnimation = this.data.settings.animate;
        this.namesOn = this.data.settings.symbols;
        this.showPitchYaw = this.data.settings.showPitchYaw || false;
        this.yaw0 = this.data.settings.yaw0 || 0;
        this.pitch0 = this.data.settings.pitch0 || 0;
        this.zOrder = [];
        this.window = {
            x:0,
            y:0,
            w:this.w,
            h:this.h - bH
        };
        ctx.translate(this.window.x, this.window.y + this.window.h);
        this.initControls();
        this.midDrag = false;
        this.renderRange = {
            minX:0,
            maxX:0,
            minY:0,
            maxY:0,
            radius:0
        };
        this.gridSpaces = 40;
        this.hexToRGB(OU.atoms);
        this.initCols();
        this.requireRender = false;
        this.renderLoop();
        // Set equation to 1st and render
        this.setCrystals();
        this.changeCrystal(OU.LocalStorage.load("OU.crystal3d.eq") || 0);
        this.inertia = {
            x:0,
            y:0
        };
        this.countToSpin = 0;
        this.spinning = true;
        this.spin();
    };
    OU.activity.Crystals.prototype.initCols = function () {
        var k, j, dk, r, g, b, scl;
        this.cols = [];
        for (k = 0; k < 7; k++) {
            this.cols[k] = [];
            for (j = 200; j--;) {
                dk = 250;
                switch(k) {
                    case 0:
                        r = 30 + parseInt(dk / 4);
                        g = 30 + parseInt(dk / 3);
                        b = parseInt(dk);
                        break;
                    case 1:
                        r = parseInt(dk);
                        g = parseInt(dk);
                        b = 50;
                        break;
                    case 2:
                        r = 80;
                        g = 80 + parseInt(dk / 2);
                        b = 50 + parseInt(dk);
                        break;
                    case 3:
                        r = parseInt(dk);
                        g = parseInt(dk);
                        b = 50;
                        break;
                    case 4:
                        r = parseInt(dk);
                        g = parseInt(dk);
                        b = parseInt(dk);
                        break;
                    case 5:
                        r = 150;
                        g = 80;
                        b = 20;
                        break;
                    case 6:
                        r = 250;
                        g = 200;
                        b = 20;
                        break;
                }
                scl = j / 200;
                r = parseInt(scl * r);
                g = parseInt(scl * g);
                b = parseInt(scl * b);
                if (r > 255)r = 255;
                if (g > 255)g = 255;
                if (b > 255)b = 255;
                this.cols[k][j] = 'rgb(' + r + ',' + g + ',' + b + ')';
            }
        }
    };
    OU.activity.Crystals.prototype.hexToRGB = function ( a ) { // Converts an array of elements with a hex colour in .col into RGB in (.r .g .b)
        var i, cl = function ( s, p ) {
            var j, ii, s1, s2, a = 0;
            s1 = s.substring(p, p + 1).toUpperCase();
            s2 = s.substring(p + 1, p + 2).toUpperCase();
            for (ii = 65; ii < 65 + 7; ii++) {
                s1 = s1.replace("" + String.fromCharCode(ii), "" + ((ii - 65) + 10));
                s2 = s2.replace("" + String.fromCharCode(ii), "" + ((ii - 65) + 10));
            }
            return parseInt(s1) * 16 + parseInt(s2);
        };
        for (i = 0; i < a.length; i++) {
            a[i].r = cl(a[i].col, 1);
            a[i].g = cl(a[i].col, 3);
            a[i].b = cl(a[i].col, 5);
        }
    };
    OU.activity.Crystals.prototype.resize = function () {
        OU.activity.Crystals.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight * 0.7,
            ctx,
            buttons = this.data.settings.showButtons,
            bX = this.w - bH * 3;
        // create Canvas Layers & Contexts
        this.bgLayer.resize();
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        this.window = {
            x:0,
            y:0,
            w:this.w,
            h:this.h - bH
        };
        if (this.selectionLayer!==undefined) {
            this.selectionLayer.resize({
                'x':this.window.x,
                'y':this.window.y + this.window.h - bH * 2,
                'w':this.window.w,
                'h':bH * 2
            });
            if (this.selectionText) {
                this.selectionText = new OU.util.DynText({
                    context:this.selectionLayer.context,
                    x:this.window.x,
                    y:0,
                    w:this.window.w - bH * 3,
                    h:bH * 2,
                    txt:this.crystal.selection.instruction
                });
            }
            if (this.selectionDone) {
                this.selectionDone.resize({
                    x:this.window.x + this.window.w - bH * 2.5,
                    y:bH * .5,
                    w:bH * 2,
                    h:bH
                });
                this.selectionDone.render();
            }
        }
        this.bgLabels();
        this.modelLayer.resize({
            x:this.x,
            y:this.y,
            h:this.h - bH,
            w:this.w
        });
        ctx = this.modelLayer.context;
        ctx.translate(this.window.x, this.window.y + this.window.h);
        ctx.textAlign = 'center';
        ctx.font = '16px ' + OU.theme.font;
        this.controlsLayer.resize({
            x:this.x,
            y:this.y + this.h - bH,
            w:this.w,
            h:bH
        });
        if (buttons.symbols) {
            this.namesButton.resize({
                x:bX,
                y:0,
                w:bH * 2,
                h:bH
            });
            bX -= bH * 2;
        }
        if (buttons.spin) {
            this.spinButton.resize({
                x:bX,
                y:0,
                w:bH * 2,
                h:bH
            });
            bX -= bH * 2;
        }
        this.zoomSlider.resize({
            x:bH,
            y:0,
            w:bX + bH,
            h:bH
        });
        this.resetView();
        this.render();
    };
    OU.activity.Crystals.prototype.setCrystals = function () {
        this.crystals = [];
        for (var i = 0; i < this.data.crystals.length; i++) {
            this.crystals.push(new this.Crystal(this.data.crystals[i], this));
        }
    };
    OU.activity.Crystals.prototype.crystalSelector = function () {
        if (this.crystals.length < 2 || this.data.settings.showMenu===false)
            return; // don't include if only 1 equation
        var h = '<form>Crystal: <select onchange="OU.obj.' + instance + '.changeCrystal();" id="eqList' + this.instance + '">';
        for (var eq in this.crystals) {
            h += '<option value="' + eq + '">' + this.crystals[eq].name + '</option>';
        }
        h += '</select></form>';
        this.eqSelector = new OU.util.HtmlBox({
            html:h,
            x:this.x + 0,
            y:this.y + this.h * .125,
            w:5,
            h:5,
            unclosable:true,
            handleScrolling:false,
            style:'overflow:visible'
        });
    };
    OU.activity.Crystals.prototype.step = function ( n ) {
        if (n - 1!=this.crystalIndex)
            this.changeCrystal(n - 1);
    };
    OU.activity.Crystals.prototype.changeCrystal = function ( eqNum ) {
        var bH = OU.controlHeight * 0.7,
            ctx, self = this;
        if (eqNum===undefined) {
            var eqList = document.getElementById('eqList' + this.instance);
            eqNum = eqList.value;
        }
        this.crystal = this.crystals[eqNum || 0];
        if (this.crystal===undefined)
            this.crystal = this.crystals[0];
        OU.LocalStorage.save("OU.crystal3d.eq", eqNum);
        this.crystalIndex = eqNum;
        this.zoom = 0.25;
        if (this.controls.length > 0)
            this.controls[0].closeControls();
        if (this.crystal.selection!==undefined) {
            if (this.selectionLayer===undefined) {
                this.selectionLayer = new OU.util.Layer({
                    container:this,
                    'x':this.window.x,
                    'y':this.window.y + this.window.h - bH * 2,
                    'w':this.window.w,
                    'h':bH * 2,
                    'id':'selLayer',
                    hasEvents:true
                });
            }
            ctx = this.selectionLayer.context;
            this.selectionText = new OU.util.DynText({
                context:ctx,
                x:this.window.x,
                y:0,
                w:this.window.w - bH * 3,
                h:bH * 2,
                txt:this.crystal.selection.instruction
            });
            this.selectionDone = new OU.util.ControlButton({
                txt:"I'm done",
                x:this.window.x + this.window.w - bH * 2.5,
                y:bH * .5,
                w:bH * 2,
                h:bH,
                layer:this.selectionLayer,
                onClick:function () {
                    self.validateSelection();
                }
            });
        }
        else {
            if (this.selectionLayer!==undefined) {
                this.selectionLayer.remove();
                this.selectionLayer = undefined;
            }
        }
        this.controlsWidth = 0;
        this.renderRange = {
            minX:0,
            maxX:0,
            minY:0,
            maxY:0,
            radius:0
        };
        this.resetView();
        this.bgLabels();
    };
    OU.activity.Crystals.prototype.isHit = function ( x, y, pressed ) {
        var damper = 0.25;
        if (pressed) {
            y = -(this.h - OU.controlHeight) + y;
            this.zOrder.sort(function ( a, b ) {
                return b.z - a.z;
            });
            if (this.rotatable) {
                // touch is in white space so rotate
                if (this.midDrag && this.movingAtom===undefined) {
                    this.spinning = false;
                    this.countToSpin = 0;
                    this.rotation.y = this.rotation.y - (x - this.dragStart.x) * damper;
                    this.rotation.x = this.rotation.x - (y - this.dragStart.y) * damper;
                    var r2d = Math.PI / 180;
                    this.rotCX = Math.cos(this.rotation.x * r2d + this.pitch0);
                    this.rotSX = Math.sin(this.rotation.x * r2d + this.pitch0);
                    this.rotCY = Math.cos(this.rotation.y * r2d + this.yaw0);
                    this.rotSY = Math.sin(this.rotation.y * r2d + this.yaw0);
                    this.render();
                }
            }
            this.dragStart = {
                'x':x,
                'y':y
            };
            this.midDrag = true;
        }
        else {
            this.midDrag = false;
            this.movingAtom = undefined;
        }
    };
    OU.activity.Crystals.prototype.spin = function () {
        var self = this, r2d = Math.PI / 180, i = this.inertia, maxVelocity = 3;
        if (this.spinOn && this.spinning) {
            i.x += Math.random() - 0.5;
            i.y += Math.random() - 0.5;
            i.x = i.x > maxVelocity ? maxVelocity : (i.x < -maxVelocity ? -maxVelocity : i.x);
            i.y = i.y > maxVelocity ? maxVelocity : (i.y < -maxVelocity ? -maxVelocity : i.y);
            this.rotation.y += i.y;
            this.rotation.x += i.x;
            this.rotCX = Math.cos(this.rotation.x * r2d + this.pitch0);
            this.rotSX = Math.sin(this.rotation.x * r2d + this.pitch0);
            this.rotCY = Math.cos(this.rotation.y * r2d + this.yaw0);
            this.rotSY = Math.sin(this.rotation.y * r2d + this.yaw0);
            this.render();
        }
        else if (this.countToSpin++ > 100) {
            this.spinning = true;
        }
        else {
            this.inertia = {
                x:0,
                y:0
            };
        }
        setTimeout(function () {
            self.spin();
        }, 40);
    };
    OU.activity.Crystals.prototype.resetView = function ( a, b ) {
        var r2d = Math.PI / 180, keepZm = false,
            self = this, rr, newA = 0.5, newB = 0.5;
        if (a) {
            newA = a;
            keepZm = true;
        }
        if (b) {
            newB = b;
            keepZm = true;
        }
        this.rotation = {
            'x':newA,
            'y':newB
        };
        this.rotCX = Math.cos(this.rotation.x * r2d + this.pitch0);
        this.rotSX = Math.sin(this.rotation.x * r2d + this.pitch0);
        this.rotCY = Math.cos(this.rotation.y * r2d + this.yaw0);
        this.rotSY = Math.sin(this.rotation.y * r2d + this.yaw0);
        rr = this.renderRange;
        this.render();
        if ((Math.abs(rr.maxX - rr.minX) < this.window.w * .5) || (Math.abs(rr.maxY - rr.minY) < this.window.h * .5)) {
            if (!keepZm) {
                if (this.zoom < 0.88) {
                    this.zoom += 0.03;
                    setTimeout(function () {
                        self.resetView()
                    }, 50);
                }
            }
        }//*/
        this.zoomSlider.render(this.zoom);
    };
    OU.activity.Crystals.prototype.zoomIn = function () {
        this.zoom += 0.2;
        this.render();
    };
    OU.activity.Crystals.prototype.zoomOut = function () {
        this.zoom -= 0.2;
        this.render();
    };
    OU.activity.Crystals.prototype.pinch = function ( e, s, x, y, dx, dy, me ) {
        var nz = ((e - s) / me.h) + me.zoom;
        nz = nz > 1 ? 1 : (nz < 0 ? 0 : nz);
        me.zoom = nz;
        me.render();
    };
    OU.activity.Crystals.prototype.setZoom = function ( zoom, graph ) {
        graph.zoom = zoom;
        graph.zoomSlider.render(zoom);
        graph.render();
    };
    OU.activity.Crystals.prototype.renderLoop = function () {
        var self = this;
        if (this.requireRender)
            this.performRender();
        setTimeout(function () {
            self.renderLoop();
        }, 40);
    };
    OU.activity.Crystals.prototype.render = function () {
        this.requireRender = true;
    };
    OU.activity.Crystals.prototype.performRender = function () {
        var bH = OU.controlHeight * 0.7,
            ctx = this.modelLayer.context;
        ctx.clearRect(0, 0, this.w, -this.h - bH); // clear background
        if (this.crystal.atoms===undefined)
            return;
        // establish the offset to lock the view central
        this.viewXOffset = 0;
        this.viewYOffset = 0;
        var topPt = new this.Point(
            this.crystal.x1 + (this.crystal.x2 - this.crystal.x1) / 2,
            this.crystal.y1 + (this.crystal.y2 - this.crystal.y1) / 2,
            this.crystal.z1 + (this.crystal.z2 - this.crystal.z1) / 2,
            this
        );
        this.viewXOffset = this.window.x + (this.window.w / 2) - topPt.in2d.x;
        this.viewYOffset = (-this.window.h / 2) - topPt.in2d.y;
        // Render elements
        if (this.editable)
            this.renderAxisGuide();
        this.requireRender = false;
        this.renderCrystal();
        this.renderControls();
        this.renderPitchYaw();
    };
    OU.activity.Crystals.prototype.renderControls = function () {
        var buttons = this.data.settings.showButtons;
        this.controlsLayer.clear();
        if (buttons.symbols)
            this.namesButton.render();
        if (buttons.spin)
            this.spinButton.render();
        if (buttons.animate)
            this.animateButton.render();
        this.zoomSlider.render();
    };
    OU.activity.Crystals.prototype.initControls = function () {
        var self = this, ctx = this.controlsLayer.context,
            bH = OU.controlHeight,
            buttons,
            bX,
            clickable = this.controlsLayer.events.clickable;
        buttons = this.data.settings.showButtons;
        bX = this.w - bH * 3;
        if (buttons.symbols) {
            this.namesButton = new OU.util.ControlButton({
                txt:'Symbols',
                x:bX,
                y:0,
                w:bH * 2,
                h:bH,
                layer:this.controlsLayer,
                onOffIndicator:true,
                onOff:this.namesOn,
                onClick:function () {
                    self.toggleNames();
                }
            });
            bX -= bH * 2;
        }
        if (buttons.spin) {
            this.spinButton = new OU.util.ControlButton({
                txt:'Spin',
                x:bX,
                y:0,
                w:bH * 2,
                h:bH,
                layer:this.controlsLayer,
                onOffIndicator:true,
                onOff:this.spinOn,
                onClick:function () {
                    self.toggleSpin();
                }
            });
            bX -= bH * 2;
        }
        this.zoomSlider = new OU.util.Slider({
            instance:'zoom' + this.instance,
            container:this,
            x:bH,
            y:0,
            w:bX + bH,
            h:bH,
            sliderHeight:bH / 2,
            drawContainer:true,
            callback:this.setZoom,
            callbackParam:this,
            frames:4,
            background:{
                // clear:true
            },
            context:ctx
        });
        clickable.push(this.zoomSlider);
        this.bgLabels();
    };
    OU.activity.Crystals.prototype.renderAxisGuide = function () {
        var ctx = this.modelLayer.context, mol = this.crystal;
        ctx.save();
        ctx.beginPath();
        ctx.strokeStyle = "#666";
        ctx.fillStyle = "#444";
        ctx.globalAlpha = 0.3;
        ctx.lineWidth = 1;
        // Axis
        this.moveTo(0, mol.y1, 0);
        this.lineTo(0, mol.y2, 0);
        this.moveTo(mol.x1, 0, 0);
        this.lineTo(mol.x2, 0, 0);
        this.moveTo(0, 0, mol.z1);
        this.lineTo(0, 0, mol.z2);
        ctx.stroke();
        this.text({
            txt:'X',
            x:mol.x2 * 1.1,
            y:0,
            z:0,
            w:100,
            h:40,
            align:'left',
            fontWeight:'bold'
        });
        this.text({
            txt:'Y',
            x:0,
            y:mol.y2 * 1.1,
            z:0,
            w:100,
            h:40,
            align:'left',
            fontWeight:'bold'
        });
        this.text({
            txt:'Z',
            x:0,
            y:0,
            z:mol.z2 * 1.1,
            w:100,
            h:40,
            align:'left',
            fontWeight:'bold'
        });
        ctx.restore();
    };
    OU.activity.Crystals.prototype.renderCrystal = function () {
        this.renderRange = {
            minX:this.window.w,
            maxX:0,
            minY:this.window.h,
            maxY:0,
            radius:0
        };
        var ctx = this.modelLayer.context;
        ctx.fillStyle = '#000';
        ctx.fillRect(0, 0, this.w, -this.h);
        if (this.crystal.tris.length > 0) {
            this.renderPoly();
        }
        else if (this.spaceFillOn) {
            this.renderSpaceFill();
        }
        else if (this.wireframeOn) {
            this.renderWireframe();
        }
        else {
            this.renderBallStick();
        }
    };
    OU.activity.Crystals.prototype.renderPoly = function () {
        var ctx = this.modelLayer.context, r, i, j, k, a, b, spb, spb2,
            sp, c = Math.PI * 2, eq = this.crystal,
            c, spa, spa3, spa2, spb3, spc, spc3, spc2, xx, yy, zz,
            shade, d1x, d1y, d1z, d2x, d2y, d2z, nx, ny, nz, xl, yl, zl, zmn = 1e20, zmx = -zmn, zdf
            , cl, cc
            , corners = [
                [0, 1, 2, 3, 0, 4, 5, 6, 7, 4],
                [1, 5],
                [2, 6],
                [3, 7]
            ]
            , ci, p0, p1, p02, p12;
        this.zOrder = [];
        this.triOrder = [];
        ctx.lineWidth = 0.5;
        ctx.strokeStyle = '#999';
        for (i = eq.atoms.length; i--;) {
            a = eq.atoms[i];
            if (a.pt===undefined)
                a.pt = new this.Point(a.x, a.y, a.z, this);
            else
                a.pt.calc();
            if (a.pt.in3d.z < zmn)zmn = a.pt.in3d.z;
            if (a.pt.in3d.z > zmx)zmx = a.pt.in3d.z;
            this.zOrder.push({
                idx:i,
                z:a.pt.in3d.z
            });
            if (a.links===undefined)
                a.links = [];
            if (a.duallinks===undefined)
                a.duallinks = [];
        }
        this.zOrder.sort(function ( a, b ) {
            return a.z - b.z;
        });
        xl = -0.5;// angle of shading light (xl,yl,zl)
        yl = 0;//.5;
        zl = 1 / Math.sqrt(2);
        xl = -zl;
        for (j = eq.tris.length; j--;) {
            xx = eq.tris[j][3];
            yy = eq.tris[j][4];
            zz = eq.tris[j][5];
            sp = new this.Point(xx, yy, zz, this).in3d;
            this.triOrder.push({
                idx:j,
                z:sp.z
            });
        }
        zdf = (zmx - zmn);
        zmx += zdf / 5 + 1;
        zmn -= zdf / 5 + 1;
        zdf = 1. / (zmx - zmn);
        this.triOrder.sort(function ( a, b ) {
            return a.z - b.z;
        });
        r = this.renderRange;
        r = 0.02 * (r.maxX - r.minX);
        for (i = this.triOrder.length; i--;) {
            j = this.triOrder[i].idx;
            a = eq.atoms[eq.tris[j][0]].pt.original;
            b = eq.atoms[eq.tris[j][1]].pt.original;
            c = eq.atoms[eq.tris[j][2]].pt.original;
            if ((a.x===b.x) && (a.y===b.y) && (a.z===b.z)) {
                this.ball(eq.atoms[eq.tris[j][1]], null, null, zmn, zdf);
            }
            else {
                spa = new this.Point(a.x, a.y, a.z, this);
                spb = new this.Point(b.x, b.y, b.z, this);
                spc = new this.Point(c.x, c.y, c.z, this);
                spa3 = spa.in3d;
                spb3 = spb.in3d;
                spc3 = spc.in3d;
                d1x = spb3.x - spa3.x;
                d1y = spb3.y - spa3.y;
                d1z = spb3.z - spa3.z;
                d2x = spc3.x - spa3.x;
                d2y = spc3.y - spa3.y;
                d2z = spc3.z - spa3.z;
                nx = d2z * d1y - d2y * d1z;
                ny = d2x * d1z - d2z * d1x;
                nz = d2y * d1x - d2x * d1y;
                shade = parseInt(-100 * (xl * nx + yl * ny + zl * nz) / Math.sqrt(nx * nx + ny * ny + nz * nz) + 100);
                cc = (((1 - (spc3.z - zmn) * zdf) * 100) | 0) / 100;
                if (cc < 0)cc = 0;
                if (shade > 198)cc = 1;
                if (cc > 1)cc = 1;
                shade = parseInt(shade * cc);
                cl = this.cols[eq.tris[j][6]][shade];
                ctx.fillStyle = cl;
                ctx.strokeStyle = cl;
                ctx.beginPath();
                spa2 = spa.in2d;
                spb2 = spb.in2d;
                spc2 = spc.in2d;
                ctx.lineWidth = 0.1;
                ctx.moveTo(spa2.x, spa2.y);
                ctx.lineTo(spb2.x, spb2.y);
                ctx.lineTo(spc2.x, spc2.y);
                ctx.stroke();
                ctx.fill();
            }
        }
        if (eq.boxCoords) {
            for (i = eq.boxCoords.length; i--;) {
                for (k = corners.length; k--;) {
                    ci = corners[k][corners[k].length - 1];
                    p0 = new this.Point(eq.boxCoords[i][ci][0]
                        , eq.boxCoords[i][ci][1]
                        , eq.boxCoords[i][ci][2]
                        , this);
                    for (var j = corners[k].length; j--;) {
                        c = corners[k][j];
                        p1 = new this.Point(eq.boxCoords[i][c][0], eq.boxCoords[i][c][1], eq.boxCoords[i][c][2], this);
                        ctx.strokeStyle = eq.boxCol;
                        ctx.beginPath();
                        p02 = p0.in2d;
                        p12 = p1.in2d;
                        ctx.lineWidth = 1.5;
                        ctx.moveTo(p02.x, p02.y);
                        ctx.lineTo(p12.x, p12.y);
                        ctx.stroke();
                        p0 = p1;
                    }
                }
            }
        }
    };
    OU.activity.Crystals.prototype.renderBallStick = function () {
        var ctx = this.modelLayer.context, r, r1, r2, i, j, a, b, b2, bO, p, t, xd, yd, spb, spb2,
            aO, a2, a3, dX, dY, dZ, d, sp, sp2, sdX, sdY, sdZ, c = Math.PI * 2, stickGradient,
            mol = this.crystal,
            z, rr = this.renderRange, aType, bType, f,
            zmn = 1e9, zmx = -zmn, zdf;
        this.zOrder = [];
        ctx.lineWidth = 0.5;
        ctx.strokeStyle = '#999';
        rr.minZ = 1000;
        rr.maxZ = -1000;
        for (i = mol.atoms.length; i--;) {
            a = mol.atoms[i];
            if (a.pt===undefined)
                a.pt = new this.Point(a.x, a.y, a.z, this);
            else
                a.pt.calc();
            if (a.pt.in3d.z < zmn)zmn = a.pt.in3d.z;
            if (a.pt.in3d.z > zmx)zmx = a.pt.in3d.z;
            z = a.pt.in3d.z;
            this.zOrder.push({
                idx:i,
                z:z
            });
            if (a.links===undefined)
                a.links = [];
            if (a.duallinks===undefined)
                a.duallinks = [];
            if (z > rr.maxZ)
                rr.maxZ = z;
            if (z < rr.minZ)
                rr.minZ = z;
        }
        zdf = (zmx - zmn);
        zmx += zdf / 5 + 1;
        zmn -= zdf / 5 + 1;
        zdf = 1. / (zmx - zmn);
        this.zOrder.sort(function ( a, b ) {
            return a.z - b.z;
        });
        for (i = this.zOrder.length; i--;) {
            a = mol.atoms[this.zOrder[i].idx];
            aO = a.pt.original;
            a2 = a.pt.in2d;
            a3 = a.pt.in3d;
            r1 = this.ball(a, null, null, zmn, zdf);
            for (j = a.links.length; j--;) {
                b = mol.atoms[a.links[j]];
                b2 = b.pt.in2d;
                bO = b.pt.original;
                if (b.pt.in3d.z <= a3.z) {
                    r2 = this.ballRadius(b);
                    r = r1 < r2 ? r1 : r2;
                    dX = bO.x - aO.x;
                    dY = bO.y - aO.y;
                    dZ = bO.z - aO.z;
                    d = Math.sqrt(dX * dX + dY * dY + dZ * dZ);
                    sdX = 0.9 * a.r * dX / d;
                    sdY = 0.9 * a.r * dY / d;
                    sdZ = 0.9 * a.r * dZ / d;
                    sp = new this.Point(aO.x + sdX, aO.y + sdY, aO.z + sdZ, this);
                    sp2 = sp.in2d;
                    ctx.beginPath();
                    stickGradient = ctx.createLinearGradient(a2.x, a2.y, b2.x, b2.y);
                    p = a.r / (a.r + b.r);
                    aType = OU.atoms[a.type];
                    bType = OU.atoms[b.type];
                    if (zmn) {
                        f = 1 - (((a.pt.in3d.z - zmn) * zdf * 100) | 0 ) / 100;
                        if (f > 0.7)f = 1;
                        stickGradient.addColorStop(p, 'rgba(' + (aType.r) + ',' + (aType.g) + ',' + (aType.b) + ',' + f + ')');
                        f = 1 - (((b.pt.in3d.z - zmn) * zdf * 100) | 0 ) / 100;
                        if (f > 0.7)f = 1;
                        stickGradient.addColorStop(p, 'rgba(' + (bType.r) + ',' + (bType.g) + ',' + (bType.b) + ',' + f + ')');
                    }
                    else {
                        f = (128 * (a.pt.in3d.z - rr.minZ) / (rr.maxZ - rr.minZ) ) | 0;
                        stickGradient.addColorStop(p, 'rgba(' + (aType.r - f) + ',' + (aType.g - f) + ',' + (aType.b - f) + ',1)');
                        f = (128 * (b.pt.in3d.z - rr.minZ) / (rr.maxZ - rr.minZ) ) | 0;
                        stickGradient.addColorStop(p, 'rgba(' + (aType.r - f) + ',' + (aType.g - f) + ',' + (aType.b - f) + ',1)');
                    }
                    ctx.strokeStyle = stickGradient;
                    ctx.moveTo(sp2.x, sp2.y);
                    ctx.lineTo(b2.x, b2.y);
                    ctx.lineWidth = r / 3;
                    ctx.stroke();
                    if (Math.abs(sp2.x - a2.x) < r1 * .99) {
                        ctx.beginPath();
                        ctx.fillStyle = stickGradient;
                        ctx.arc(sp2.x, sp2.y, r / 6, 0, c, false);
                        ctx.fill();
                    }
                }
            }
            for (j = a.duallinks.length; j--;) {
                b = mol.atoms[a.duallinks[j]];
                b2 = b.pt.in2d;
                bO = b.pt.original;
                if (b.pt.in3d.z <= a3.z) {
                    r2 = this.ballRadius(b);
                    r = r1 < r2 ? r1 : r2;
                    dX = bO.x - aO.x;
                    dY = bO.y - aO.y;
                    dZ = bO.z - aO.z;
                    d = Math.sqrt(dX * dX + dY * dY + dZ * dZ);
                    sdX = 0.8 * a.r * dX / d;
                    sdY = 0.8 * a.r * dY / d;
                    sdZ = 0.8 * a.r * dZ / d;
                    sp = new this.Point(aO.x + sdX, aO.y + sdY, aO.z + sdZ, this);
                    sp2 = sp.in2d;
                    dX = aO.x - bO.x;
                    dY = aO.y - bO.y;
                    dZ = aO.z - bO.z;
                    sdX = 0.8 * b.r * dX / d;
                    sdY = 0.8 * b.r * dY / d;
                    sdZ = 0.8 * b.r * dZ / d;
                    spb = new this.Point(bO.x + sdX, bO.y + sdY, bO.z + sdZ, this);
                    spb2 = spb.in2d;
                    t = b2.x==sp2.x ? 0 : ((Math.PI / 2) - Math.atan((b2.y - sp2.y) / (b2.x - sp2.x)));
                    xd = -Math.cos(t) / 4;
                    yd = Math.sin(t) / 4;
                    ctx.beginPath();
                    stickGradient = ctx.createLinearGradient(a2.x, a2.y, b2.x, b2.y);
                    p = a.r / (a.r + b.r);
                    aType = OU.atoms[a.type];
                    f = (128 * (a.pt.in3d.z - rr.minZ) / (rr.maxZ - rr.minZ) ) | 0;
                    stickGradient.addColorStop(p, 'rgba(' + (aType.r - f) + ',' + (aType.g - f) + ',' + (aType.b - f) + ',1)');
                    aType = OU.atoms[b.type];
                    f = (128 * (b.pt.in3d.z - rr.minZ) / (rr.maxZ - rr.minZ) ) | 0;
                    stickGradient.addColorStop(p, 'rgba(' + (aType.r - f) + ',' + (aType.g - f) + ',' + (aType.b - f) + ',1)');
                    ctx.strokeStyle = stickGradient;
                    ctx.moveTo(sp2.x - xd * r1, sp2.y - yd * r1);
                    ctx.bezierCurveTo(sp2.x - xd * r1 * 2,
                        sp2.y - yd * r1 * 2,
                        spb2.x - xd * r2 * 2,
                        spb2.y - yd * r2 * 2,
                        spb2.x - xd * r2,
                        spb2.y - yd * r2);
                    ctx.moveTo(sp2.x + xd * r1, sp2.y + yd * r1);
                    ctx.bezierCurveTo(sp2.x + xd * r1 * 2, sp2.y + yd * r1 * 2, spb2.x + xd * r2 * 2, spb2.y + yd * r2 * 2, spb2.x + xd * r2, spb2.y + yd * r2);
                    ctx.lineWidth = r / 3;
                    ctx.stroke();
                    if (Math.abs(sp2.x - a2.x) < r1 * .99) {
                        ctx.beginPath();
                        ctx.fillStyle = stickGradient;
                        ctx.arc(sp2.x - xd * r1, sp2.y - yd * r1, r / 6, 0, c, false);
                        ctx.arc(sp2.x + xd * r1, sp2.y + yd * r1, r / 6, 0, c, false);
                        ctx.fill();
                    }
                }
            }
        }
    };
    OU.activity.Crystals.prototype.rBallRadius = function ( a ) {
        var mol = this.crystal;
        var a3 = a.pt.in3d,
            dz = a3.z - (mol.z1 - ((mol.dz) * 60 * (1 - this.zoom))),
            rz = this.window.h, cX, eX;
        cX = (a3.x * 15) / (dz) * rz + this.viewXOffset;
        eX = ((a3.x + a.r) * 15) / (dz) * rz + this.viewXOffset;
        return Math.abs(cX - eX);//*/
    };
    OU.activity.Crystals.prototype.ballRadius = function ( a ) {
        return this.rBallRadius(a);
    };
    OU.activity.Crystals.prototype.ball = function ( a, ov, s, zmn, zdf ) {
        var override = ov || {},
            gradient, c = Math.PI * 2, ctx = this.modelLayer.context,
            x = override.x || a.pt.in2d.x,
            y = override.y || a.pt.in2d.y,
            scale = s || 1,
            radius = this.ballRadius(a) / scale,
            aType, f,
            rr = this.renderRange;
        ctx.beginPath();
        ctx.arc(x, y, radius, 0, c, false);
        aType = OU.atoms[a.type];
        if (zmn) {
            f = 1 - (((a.pt.in3d.z - zmn) * zdf * 100) | 0 ) / 100;
            if (f > 0.7)f = 1;
            ctx.fillStyle = 'rgba(' + (aType.r) + ',' + (aType.g) + ',' + (aType.b) + ',' + f + ')';
        }
        else {
            f = (128 * (a.pt.in3d.z - rr.minZ) / (rr.maxZ - rr.minZ) ) | 0;
            ctx.fillStyle = 'rgba(' + (aType.r - f) + ',' + (aType.g - f) + ',' + (aType.b - f) + ',1)';
        }
        ctx.fill();
        ctx.beginPath();
        ctx.arc(x, y, radius * .9, 0, c, false);
        gradient = ctx.createRadialGradient(x - radius / 2, y - radius / 2, 0, x, y, radius);
        gradient.addColorStop(0, 'rgba(255,255,255,0.5)');
        gradient.addColorStop(0.3, 'rgba(255,255,255,0.4)');
        gradient.addColorStop(0.8, 'rgba(255,255,255,0)');
        gradient.addColorStop(1, 'rgba(0,0,0,0)');
        ctx.fillStyle = gradient;
        ctx.fill();
        if (this.namesOn) {
            ctx.save();
            ctx.font = radius + 'px ' + OU.theme.font;
            ctx.fillStyle = OU.atoms[a.type].text;
            ctx.fillText(OU.atoms[a.type].name, x, y);
            ctx.restore();
        }
        if (radius > this.renderRange.radius)
            this.renderRange.radius = radius;
        return radius * scale;
    };
    OU.activity.Crystals.prototype.toggleNames = function () {
        this.namesOn = !this.namesOn;
        this.namesButton.params.onOff = this.namesOn;
        this.render();
    };
    OU.activity.Crystals.prototype.toggleSpin = function () {
        this.spinOn = !this.spinOn;
        this.spinButton.params.onOff = this.spinOn;
        this.render();
    };
    OU.activity.Crystals.prototype.bgLabels = function () {
        var bH = OU.controlHeight * 0.7,
            bgctx = this.bgLayer.context;
        bgctx.clearRect(0, this.h - bH, this.w, bH);
        bgctx.textAlign = 'center';
    };
    OU.activity.Crystals.prototype.renderPitchYaw = function () {
        if (!this.showPitchYaw)return;
        var yaw = Math.abs(this.rotation.y) % 360 | 0,
            pitch = Math.abs(this.rotation.x) % 360 | 0,
            bH = OU.controlHeight * 0.7,
            ctx = this.modelLayer.context;
        ctx.textAlign = 'left';
        ctx.font = 'bold ' + ((bH / 2) | 0) + 'px ' + OU.theme.font;
        ctx.fillStyle = '#aaa';
        ctx.fillText('Pitch: ' + pitch + ' \u00B0  Yaw: ' + yaw + ' \u00B0', this.w * .6, -this.h * .03 - bH * 1.5);
    };
    OU.activity.Crystals.prototype.moveTo = function ( x, y, z ) {
        var pt = new this.Point(x, y, z, this);
        this.modelLayer.context.moveTo(pt.in2d.x, pt.in2d.y);
    };
    OU.activity.Crystals.prototype.lineTo = function ( x, y, z ) {
        var pt = new this.Point(x, y, z, this);
        this.modelLayer.context.lineTo(pt.in2d.x, pt.in2d.y);
    };
    OU.activity.Crystals.prototype.text = function ( params ) {
        var pt = new this.Point(params.x, params.y, params.z, this);
        this.modelLayer.context.fillText(params.txt, pt.in2d.x, pt.in2d.y);
    };
    OU.activity.Crystals.prototype.Point = function ( x, y, z, graph, noRotate ) {
        this.graph = graph;
        this.noRotate = noRotate;
        this.original = {
            'x':x,
            'y':y,
            'z':z
        };
        OU.activity.Crystals.prototype.Point.prototype.calc = function () {
            var graph = this.graph, x = this.original.x, y = this.original.y, z = this.original.z,
                nz, ny, rz = graph.window.h, newX, newY, eq = graph.crystal, dz, rr = graph.renderRange;
            if (this.noRotate===undefined) {
                // rotate about Y axis
                nz = (z * graph.rotCY) - (x * graph.rotSY);
                x = (z * graph.rotSY) + (x * graph.rotCY);
                z = nz;
                // rotate about X axis
                ny = (y * graph.rotCX) - (z * graph.rotSX);
                z = (y * graph.rotSX) + (z * graph.rotCX);
                y = ny;
            }
            this.in3d = {
                'x':x,
                'y':y,
                'z':z
            };
            // Note 3D to 2D projection is performed following the method
            // described at http://en.wikipedia.org/wiki/3D_projection
            dz = z - (eq.z1 - ((eq.z2 - eq.z1) * 60 * (1 - graph.zoom)));
            newX = (rz * x * 15) / dz + graph.viewXOffset;
            newY = -((rz * y * 15) / dz) + graph.viewYOffset;
            this.in2d = { // Point in 2d relative to the window in pixels
                'x':newX,
                'y':newY
            };
            newY = -newY;
            if (newX > rr.maxX)
                rr.maxX = newX;
            if (newX < rr.minX)
                rr.minX = newX;
            if (newY > rr.maxY)
                rr.maxY = newY;
            if (newY < rr.minY)
                rr.minY = newY;
        };
        OU.activity.Crystals.prototype.Point.prototype.upd2Dto3D = function ( dx, dy ) { // calculate and update 3d co-ords, based on current 2d co-ords
            var newX = this.in2d.x = this.in2d.x + dx, newY = this.in2d.y = this.in2d.y + dy, g = this.graph,
                eq = g.crystal, rz = g.window.h, p3, nz, ny,
                dz = this.in3d.z - (eq.z1 - ((eq.z2 - eq.z1) * 60 * (1 - g.zoom))),
                r2d = Math.PI / 180,
                rotCX = Math.cos(-g.rotation.x * r2d),
                rotSX = Math.sin(-g.rotation.x * r2d),
                rotCY = Math.cos(-g.rotation.y * r2d),
                rotSY = Math.sin(-g.rotation.y * r2d);
            this.in3d.x = ((newX - g.viewXOffset) / (15 * rz)) * dz;
            this.in3d.y = ((g.viewYOffset - newY) / (15 * rz)) * dz;
            // note in3d.Z remains the same - we only move in current XY plane
            //Calculate Original XYZ from current in3d XYZ
            p3 = this.in3d;
            x = p3.x;
            y = p3.y;
            z = p3.z;
            ny = (y * rotCX) - (z * rotSX);
            z = (y * rotSX) + (z * rotCX);
            y = ny;
            nz = (z * rotCY) - (x * rotSY);
            x = (z * rotSY) + (x * rotCY);
            z = nz;
            this.original = {
                x:x,
                y:y,
                z:z
            }
        };
        this.calc();
    };
    OU.activity.Crystals.prototype.atomType = function ( s ) {
        var i;
        s = s.replace(/^\s+|\s+$/g, "");
        for (i = OU.atoms.length; i--;) {
            if (OU.atoms[i].name==s) {
                return i;
            }
        }
        return 0;
    };
    OU.activity.Crystals.prototype.Crystal = function ( params, parent ) {
        //TODO: ensure x2>x1, y2>y1, z2>z1 - if not, swap round values
        this.name = params.name || 'y=?';
        this.parent = parent;
        this.description = params.description;
        this.x1 = params.x1 || -10;
        this.x2 = params.x2 || 10;
        this.y1 = params.y1 || -5;
        this.y2 = params.y2 || 5;
        this.z1 = params.z1 || -10;
        this.z2 = params.z2 || 10;
        this.reset_x1 = this.x1;
        this.reset_x2 = this.x2;
        this.reset_y1 = this.y1;
        this.reset_y2 = this.y2;
        this.reset_z1 = this.z1;
        this.reset_z2 = this.z2;
        this.variables = params.variables || [];
        this.selection = params.selection;
        this.atoms = params.atoms;
        this.animations = params.animations;
        this.atomicRadiusFactor = params.atomicRadiusFactor;
        this.molFile = params.molfile;
        this.boxCoords = params.boxCoords;
        this.boxCol = params.boxCol;
        if (params.colour!==undefined)
            this.colour = params.colour;
        if (params.rgb!==undefined)
            this.rgb = params.rgb;
        OU.activity.Crystals.prototype.Crystal.prototype.parseMOL = function () {
            if (this.molFile===undefined)
                return;
            var i, ii, a, b, l, r, t, s, atom, lines = this.molFile.split("\n"),
                aCell = ["Si", "Mg", "Fe", "Al", "Wht", "Bwn", "Org"],
                coln = 0, tri, p1, p2, p3,
                sff = this.parent.config.spaceFillFactor,
                counts = lines[3],
                numAtoms = parseInt(counts.substr(0, 3)),
                numLinks = parseInt(counts.substr(3, 3)),
                numTris = parseInt(counts.substr(6, 3));
            this.x1 = 2000;
            this.x2 = -2000;
            this.y1 = 2000;
            this.y2 = -2000;
            this.z1 = 2000;
            this.z2 = -2000;
            this.atoms = [];
            for (i = 0; i < numAtoms; i++) {
                a = lines[4 + i];
                s = a.substr(31, 3);
                atom = {};
                atom.id = i + 1;
                atom.type = this.parent.atomType(s);
                atom.x = parseFloat(a.substr(0, 10));
                atom.y = parseFloat(a.substr(10, 10));
                atom.z = parseFloat(a.substr(20, 10));
                atom.r = OU.atoms[atom.type].radius * (this.atomicRadiusFactor || 5);
                atom.sfrs = (atom.r * sff) * (atom.r * sff);
                atom.name = s;
                atom.selected = false;
                if (atom.x < this.x1)
                    this.x1 = atom.x;
                if (atom.x > this.x2)
                    this.x2 = atom.x;
                if (atom.y < this.y1)
                    this.y1 = atom.y;
                if (atom.y > this.y2)
                    this.y2 = atom.y;
                if (atom.z < this.z1)
                    this.z1 = atom.z;
                if (atom.z > this.z2)
                    this.z2 = atom.z;
                atom.links = [];
                atom.duallinks = [];
                this.atoms.push(atom);
            }
            this.tris = [];
            this.x1 -= 1;
            this.x2 += 1;
            this.y1 -= 1;
            this.y2 += 1;
            this.z1 -= 10;
            this.z2 += 10;
            for (i = 0; i < numLinks; i++) {
                b = lines[4 + i + numAtoms];
                l = parseInt(b.substr(0, 3)) - 1;
                r = parseInt(b.substr(3, 3)) - 1;
                t = parseInt(b.substr(6, 3));
                if (t==1) {
                    this.atoms[l].links.push(r);
                    this.atoms[r].links.push(l);
                }
                else {
                    this.atoms[l].duallinks.push(r);
                    this.atoms[r].duallinks.push(l);
                }
            }
            for (i = 0; i < numTris; i++) {
                b = lines[4 + i + numAtoms + numLinks];
                tri = [];
                p1 = parseInt(b.substr(0, 3)) - 1;
                p2 = parseInt(b.substr(3, 3)) - 1;
                p3 = parseInt(b.substr(6, 3)) - 1;
                if (p1 < 1)p1 = 1;
                if (p2 < 1)p2 = 1;
                if (p3 < 1)p3 = 1;
                for (ii = 0; ii < aCell.length; ii++) {
                    if (this.atoms[p1 - 1].name.indexOf(aCell[ii]) >= 0
                        || this.atoms[p2 - 1].name.indexOf(aCell[ii]) >= 0
                        || this.atoms[p3 - 1].name.indexOf(aCell[ii]) >= 0) {
                        coln = ii;
                    }
                }
                tri.push(p1);
                tri.push(p2);
                tri.push(p3);
                tri.push((this.atoms[p1].x + this.atoms[p2].x + this.atoms[p3].x) / 3);
                tri.push((this.atoms[p1].y + this.atoms[p2].y + this.atoms[p3].y) / 3);
                tri.push((this.atoms[p1].z + this.atoms[p2].z + this.atoms[p3].z) / 3);
                tri.push(coln);
                this.tris.push(tri);
            }
        };
        OU.activity.Crystals.prototype.Crystal.prototype.deltas = function () {
            this.dx = Math.abs(this.x2 - this.x1);
            this.dy = Math.abs(this.y2 - this.y1);
            this.dz = Math.abs(this.z2 - this.z1);
        };
        this.parseMOL();
        if (params.highlightedAtoms!==undefined) {
            for (var i = params.highlightedAtoms.length; i--;) {
                this.atoms[params.highlightedAtoms[i] - 1].highlighted = true;
            }
        }
        this.deltas();
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.Crystals, OU.util.Activity);
