/**
 * @fileOverview  Renders an equation as a plane on a 3D graph
 *
 * @param {JSON array} data Data as included in the data/data.js file
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.HtmlBox');
OU.require('OU.util.Slider');
OU.require('OU.util.varControlBank');
OU.require('OU.util.Layer');

KEY_LEFT=37;
KEY_UP=38;
KEY_RIGHT=39;
KEY_DOWN=40;

/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.Equation3D = function ( data, instance, controller ) {
    OU.activity.Equation3D.prototype.canvasView = function () {
        this.config = {
            depth:5,
            fps:40 // 40ms = 25 fps
        };
        var bH = OU.controlHeight,self=this;
        // create Canvas Layers & Contexts
        this.bgLayer = new OU.util.Layer({
            container:this
        });
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        this.window = {
            x:0,
            y:0,
            w:this.w,
            h:this.h - bH
        };
        this.gridSpaces = OU.LocalStorage.load("OU.equation3d.quality") || 20; // Quality Value = number of intersection to render on the model
        this.PAD = this.h * .045;
        this.inRender = false;
        this.equation = {};
        this.controls = [];
        this.pyLayer = new OU.util.Layer({
            container:this,
            id:'pitchyaw'
        });
        var ctx = this.pyLayer.context;
        ctx.textAlign = 'right';
        ctx.font = 'bold ' + ((bH * .5) | 0) + 'px ' + OU.theme.font;
        ctx.fillStyle = '#AFDFE4';
        this.modelLayer = new OU.util.Layer({
            container:this,
            'id':'model',
            'h':this.h - bH,
            'hasEvents':true,
            pinch:this.pinch,
            pinchMe:this,
            keyPress: function(k) {
                self.keyPress(k);
            }
        });
        this.modelLayer.events.clickable.push(this);
        this.modelLayer.context.translate(this.window.x, this.window.y + this.window.h);
        this.controlsLayer = new OU.util.Layer({
            container:this,
            'y':this.h - bH,
            'h':bH,
            'id':'controls',
            hasEvents:true
        });
        this.initControls();
        this.midDrag = false;
        this.renderRange = {
            minX:0,
            maxX:0,
            minY:0,
            maxY:0
        };
        // Set equation to 1st and render
        this.setEquations();
        this.changeEquation(0); // OU.LocalStorage.load("OU.equation3d.eq") || 0);
        // Don't remember which equation as it conflicts with drop down menu
        //TODO fix drop down menu so that it matches remembered equation'
        this.equationSelector();
    };
    OU.activity.Equation3D.prototype.resize = function () {
        OU.activity.Equation3D.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight, sW;
        // create Canvas Layers & Contexts
        this.bgLayer.resize();
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        this.bgLabels();
        this.window = {
            x:0,
            y:0,
            w:this.w,
            h:this.h - bH
        };
        this.pyLayer.resize();
        this.modelLayer.resize({
            'h':this.h - bH
        });
        this.modelLayer.context.translate(this.window.x, this.window.y + this.window.h);
        this.controlsLayer.resize({
            'y':this.h - bH,
            'h':bH
        });
        sW = (this.w - this.controlsWidth) / 2;
        this.zoomSlider.resize({
            x:this.controlsWidth,
            w:sW,
            scrubX:this.zoomSlider.x + sW * .1,
            scrubWidth:sW * .8

        });
        this.qualitySlider.resize({
            x:this.controlsWidth + sW,
            w:sW,
            scrubWidth:sW * .8,
            scrubX:this.qualitySlider.x + sW * .1

        });
        this.controlBank.resize({
            layerOffset:this.h - bH
        });
        this.controlBank.render();
        this.resetButton.render();
        this.render();
        this.renderPitchYaw();
    };
    OU.activity.Equation3D.prototype.setEquations = function () {
        this.equations = [];
        for (var i = 0; i < this.data.equations.length; i++) {
            this.equations.push(new this.Equation(this.data.equations[i]));
        }
    };
    OU.activity.Equation3D.prototype.equationSelector = function () {
        if (this.equations.length < 2)
            return; // don't include if only 1 equation
        OU.obj[instance] = this; // push this into object array, so we can call it back from the page
        var eq, h = '<form>Equation: <select onchange="OU.obj.' + instance + '.changeEquation();" id="eqList' + this.instance + '">';
        for (eq = 0; eq < this.equations.length; eq++) {
            h += '<option value="' + eq + '">' + this.equations[eq].name + '</option>';
        }
        h += '</select></form>';
        this.eqSelector = new OU.util.HtmlBox({
            container:this,
            html:h,
            x:this.x + 0,
            y:this.y + this.h * .125,
            w:5,
            h:5,
            unclosable:true,
            handleScrolling:false,
            style:'overflow:visible'
        });
    };
    OU.activity.Equation3D.prototype.calcModel = function () {
        this.modelY = new Array();
        var v = {}, x = 0, z = 0, j,
        eq = this.equation,
        eqV = eq.variables,
        xS = eq.x1,
        xE = eq.x2,
        zS = eq.z1,
        zE = eq.z2,
        zD = eq.dz / this.gridSpaces,
        xD = eq.dx / this.gridSpaces;
        for (j = eqV.length; j--;) {
            if (eqV[j].val!==undefined)
                v[eqV[j].name] = eqV[j].val;
            else
                v[eqV[j].name] = eqV[j].init;
        }
        for (var x1 = xS; x1 <= xE; x1 += xD) {
            this.modelY[x] = new Array();
            z = 0;
            for (var z1 = zS; z1 <= zE; z1 += zD) {
                v.x = x1;
                v.z = z1;
                this.modelY[x][z] = eq.calcY(v);
                z++;
            }
            x++;
        }
    };
    OU.activity.Equation3D.prototype.changeEquation = function ( eqNum ) {
        if (eqNum===undefined) {
            var eqList = document.getElementById('eqList' + this.instance);
            eqNum = eqList.value;
        }
        this.equation = this.equations[eqNum || 0];
        var bH = OU.controlHeight,
        eq = this.equation, bgctx = this.bgLayer.context;
        bgctx.clearRect(0, 0, this.w, this.h);
        OU.LocalStorage.save("OU.equation3d.eq", eqNum);
        this.calcModel();
        this.zoom = 0.25;
        if (this.controls.length > 0)
            this.controls[0].closeControls();
        var clickable = this.controlsLayer.events.clickable;
        clickable.length = 0;
        clickable.push(this.zoomSlider);
        clickable.push(this.qualitySlider);
        clickable.push(this.resetButton);
        if (this.controlBank)
            this.controlBank.close();
        this.controlBank = new OU.util.varControlBank({
            x:bH * 2,
            y:0,
            w:this.w - (bH * 2),
            h:bH,
            layerOffset:this.h - bH,
            vars:eq.variables,
            valInButton:true,
            layer:this.controlsLayer,
            clickable:clickable,
            callback:this.varChange,
            parent:this
        });
        this.controlsWidth = bH * 2 + this.controlBank.w;
        this.resetView();
        this.bgLabels();
        this.resetButton.render();
    };
    OU.activity.Equation3D.prototype.varChange = function () {
        this.parent.calcModel();
        this.parent.render();
    };
    OU.activity.Equation3D.prototype.isHit = function ( x, y, pressed ) {
        var damper = 0.25;
        if (pressed && !this.midDrag) {
            this.dragStart = {
                'x':x,
                'y':y
            };
            this.midDrag = true;
        }
        else if (pressed && this.midDrag) {
            this.rotation.y = this.rotation.y - (x - this.dragStart.x) * damper;
            this.rotation.x = this.rotation.x - (y - this.dragStart.y) * damper;
            var r2d = Math.PI / 180;
            this.rotCX = Math.cos(this.rotation.x * r2d);
            this.rotSX = Math.sin(this.rotation.x * r2d);
            this.rotCY = Math.cos(this.rotation.y * r2d);
            this.rotSY = Math.sin(this.rotation.y * r2d);
            this.dragStart = {
                'x':x,
                'y':y
            };
            this.render();
            this.renderPitchYaw();
        }
        else {
            this.midDrag = false;
        }
    };
    OU.activity.Equation3D.prototype.keyPress = function ( k ) {
        var dX=0,dY=0;
        switch(k){
            case KEY_UP:
                dY=-10;
                break;
            case KEY_DOWN:
                dY=10;
                break;
            case KEY_LEFT:
                dX=-10;
                break;
            case KEY_RIGHT:
                dX=10;
                break;
        }
        this.rotation.y = this.rotation.y - (dX);
        this.rotation.x = this.rotation.x - (dY);

        var r2d = Math.PI / 180;
        this.rotCX = Math.cos(this.rotation.x * r2d);
        this.rotSX = Math.sin(this.rotation.x * r2d);
        this.rotCY = Math.cos(this.rotation.y * r2d);
        this.rotSY = Math.sin(this.rotation.y * r2d);
        this.render();
        this.renderPitchYaw();
    };
    OU.activity.Equation3D.prototype.spin = function () {
        var self = this;
        this.rotation.y += 0.5;
        this.render();
        setTimeout(function () {
            self.spin();
        }, 20);
    };
    OU.activity.Equation3D.prototype.setQuality = function ( quality, graph ) {
        graph.gridSpaces = quality * 40 + 20;
        OU.LocalStorage.save("OU.equation3d.quality", graph.gridSpaces);
        graph.qualitySlider.render((graph.gridSpaces - 20) / 40);
        graph.calcModel();
        graph.render();
    };
    OU.activity.Equation3D.prototype.resetView = function () {
        var r2d = Math.PI / 180, i, v, sW, tooSmall = true,
        numVars = this.equation.variables.length,
        bgctx = this.zoomSlider.context, bH = OU.controlHeight,
        self = this;
        this.rotation = {
            'x':330,
            'y':30
        };
        for (i = numVars; i--;) {
            v = this.equation.variables[i];
            v.val = v.init;
        }
        this.rotCX = Math.cos(this.rotation.x * r2d);
        this.rotSX = Math.sin(this.rotation.x * r2d);
        this.rotCY = Math.cos(this.rotation.y * r2d);
        this.rotSY = Math.sin(this.rotation.y * r2d);
        this.calcModel();
        this.render();
        this.renderPitchYaw();
        sW = (this.w - this.controlsWidth) / 2;
        this.zoomSlider.x = this.controlsWidth;
        this.zoomSlider.w = sW;
        this.zoomSlider.scrubX = this.zoomSlider.x + sW * .1;
        this.zoomSlider.scrubWidth = sW * .8;
        this.qualitySlider.x = this.controlsWidth + sW;
        this.qualitySlider.w = sW;
        this.qualitySlider.scrubWidth = sW * .8;
        this.qualitySlider.scrubX = this.qualitySlider.x + sW * .1;
        //bgctx.fillStyle = '#f00';
        bgctx.clearRect(0, 0, this.w, bH);
        //bgctx.clearRect(bH*2,this.h-bH,this.w-bH*4,bH);
        bgctx.font = 'bold ' + bH * .3 + 'px ' + OU.theme.font;
        bgctx.fillText('Zoom', this.zoomSlider.x + this.zoomSlider.w / 2, this.h - bH * .25);
        bgctx.fillText('Quality', this.qualitySlider.x + this.qualitySlider.w / 2, this.h - bH * .25);
        if (this.renderRange.minX < (this.window.w * .1)
            || this.renderRange.maxX > (this.window.w * .9)
            || this.renderRange.minY < (this.window.h * .1)
            || this.renderRange.maxY > (this.window.h * .9)) {
            tooSmall = false;
        }
        if (tooSmall) {
            this.zoom += 0.05;
            setTimeout(function () {
                self.resetView()
            }, 1);
        }
        this.controlBank.render();
        this.zoomSlider.render(this.zoom);
        this.qualitySlider.render((this.gridSpaces - 20) / 40);
        this.resetButton.render();
        bgctx.fillText('Zoom', this.zoomSlider.x + this.zoomSlider.w / 2, this.h - bH * .25);
        bgctx.fillText('Quality', this.qualitySlider.x + this.qualitySlider.w / 2, this.h - bH * .25);
    };
    OU.activity.Equation3D.prototype.zoomIn = function () {
        this.zoom += 0.2;
        this.render();
    };
    OU.activity.Equation3D.prototype.zoomOut = function () {
        this.zoom -= 0.2;
        this.render();
    };
    OU.activity.Equation3D.prototype.pinch = function ( e, s, x, y, dx, dy, me ) {
        var nz = ((e - s) / me.h) + me.zoom;
        nz = nz > 1 ? 1 : (nz < 0 ? 0 : nz);
        me.zoom = nz;
        me.render();
    };
    OU.activity.Equation3D.prototype.setZoom = function ( zoom, graph ) {
        graph.zoom = zoom;
        graph.zoomSlider.render(zoom);
        graph.render();
    };
    OU.activity.Equation3D.prototype.render = function () {
        var bH = OU.controlHeight, e = this.equation;
        this.modelLayer.context.clearRect(0, 0, this.w, -this.h - bH); // clear background
        // establish the offset to lock the view central
        this.viewXOffset = 0;
        this.viewYOffset = 0;
        var topPt = new this.Point(
            e.x1 + (e.x2 - e.x1) / 2,
            e.y1 + (e.y2 - e.y1) / 2,
            e.z1 + (e.z2 - e.z1) / 2,
            this
            );
        this.viewXOffset = this.window.x + (this.window.w / 2) - topPt.in2d.x;
        this.viewYOffset = (-this.window.h / 2) - topPt.in2d.y;
        // initial render range data to center screen
        this.renderRange = {
            minX:this.window.w,
            maxX:0,
            minY:this.window.h,
            maxY:0
        };
        // Render elements
        this.renderFrame();
        this.renderEquation3D();
        this.zoomSlider.render();
        this.qualitySlider.render();
    };
    OU.activity.Equation3D.prototype.renderFrame = function () {
        var ctx = this.modelLayer.context, x, y, z,
        e = this.equation;
        ctx.save();
        ctx.beginPath();
        ctx.strokeStyle = "#666";
        ctx.fillStyle = "#444";
        // Axis
        this.moveTo(0, e.y1, 0);
        this.lineTo(0, e.y2, 0);
        this.moveTo(e.x1, 0, 0);
        this.lineTo(e.x2, 0, 0);
        this.moveTo(0, 0, e.z1);
        this.lineTo(0, 0, e.z2);
        ctx.stroke();
        for (x = e.x1; x <= e.x2; x += e.dx / 10) {
            this.text({
                txt:((100 * x) | 0) / 100,
                x:x,
                y:0,
                z:0,
                w:100,
                h:20,
                align:'left'
            });
        }
        for (y = e.y1; y <= e.y2; y += e.dy / 10) {
            this.text({
                txt:((100 * y) | 0) / 100,
                x:0,
                y:y,
                z:0,
                w:100,
                h:20,
                align:'left'
            });
        }
        for (z = e.z1; z <= e.z2; z += e.dz / 10) {
            this.text({
                txt:((100 * z) | 0) / 100,
                x:0,
                y:0,
                z:z,
                w:100,
                h:20,
                align:'left'
            });
        }
        ctx.font = '24px bold';
        this.text({
            txt:'X',
            x:e.x2 * 1.1,
            y:0,
            z:0,
            w:100,
            h:40,
            align:'left',
            fontWeight:'bold'
        });
        this.text({
            txt:'Y',
            x:0,
            y:e.y2 * 1.1,
            z:0,
            w:100,
            h:40,
            align:'left',
            fontWeight:'bold'
        });
        this.text({
            txt:'Z',
            x:0,
            y:0,
            z:e.z2 * 1.1,
            w:100,
            h:40,
            align:'left',
            fontWeight:'bold'
        });
        ctx.restore();
    };
    OU.activity.Equation3D.prototype.renderEquation3D = function () {
        var ctx = this.modelLayer.context,
        pt = [],
        x = 0, x1, z1, xL, zL,
        z = 0, pt1, pt2, pt3, pt4,
        y = this.modelY,
        xD = -1,
        startZ,
        zD = -1,
        eq = this.equation,
        gs = this.gridSpaces;
        for (x1 = eq.x1; x1 <= eq.x2; x1 += eq.dx / gs) {
            pt[x] = [];
            z = 0;
            for (z1 = eq.z1; z1 <= eq.z2; z1 += eq.dz / gs) {
                pt[x][z] = new this.Point(x1, y[x][z], z1, this);
                z++;
            }
            x++;
        }
        ctx.lineWidth = 0.5;
        ctx.strokeStyle = '#ccc';
        // Always render from the back first - so determine the farthest corner and start there
        x = pt.length - 1;
        if (pt[0][0].in3d.z > pt[pt.length - 1][0].in3d.z) {
            x = 0;
            xD = 1;
        }
        startZ = pt[0].length - 1;
        if (pt[0][0].in3d.z > pt[0][pt[0].length - 1].in3d.z) {
            startZ = 0;
            zD = 1;
        }
        for (xL = 0; xL < pt.length - 1; xL++) {
            z = startZ;
            for (zL = 0; zL < pt[0].length - 1; zL++) {
                pt1 = pt[x][z];
                pt2 = pt[x][z + zD];
                pt3 = pt[x + xD][z + zD];
                pt4 = pt[x + xD][z];
                ctx.beginPath();
                ctx.moveTo(pt1.in2d.x, pt1.in2d.y);
                ctx.lineTo(pt2.in2d.x, pt2.in2d.y);
                ctx.lineTo(pt3.in2d.x, pt3.in2d.y);
                ctx.lineTo(pt4.in2d.x, pt4.in2d.y);
                ctx.closePath();
                if (eq.colour!==undefined) {
                    ctx.fillStyle = eq.colour;
                }
                else if (eq.rgb!==undefined) {
                    var yPerc = ((pt1.original.y) - eq.y1) / (eq.y2 - eq.y1);
                    if (yPerc > 1)
                        yPerc = 1;
                    var percRed = eq.rgb.r * yPerc | 0;
                    var percGreen = eq.rgb.g * yPerc | 0;
                    var percBlue = eq.rgb.b * yPerc | 0;
                    var spotCol = 'rgba(' + percRed + ',' + percGreen + ',' + percBlue + ',0.75)';
                    ctx.fillStyle = spotCol;
                }
                else {
                    ctx.fillStyle = '#00c';
                }
                ctx.fill();
                z += zD;
            }
            x += xD;
        }
        ctx.lineWidth = 0.75;
    };
    OU.activity.Equation3D.prototype.initControls = function () {
        var self = this, ctx = this.controlsLayer.context, bH = OU.controlHeight;
        this.zoomSlider = new OU.util.Slider({
            container:this,
            instance:'zoom' + this.instance,
            x:bH * 6,
            y:0,
            w:(this.w - (bH * 6)) / 2,
            h:bH,
            sliderHeight:bH / 2,
            drawContainer:true,
            callback:this.setZoom,
            callbackParam:this,
            frames:4,
            background:{
                color:'#e0e0e0'
            },
            context:ctx,
            tabIndex: 200
        });
        this.qualitySlider = new OU.util.Slider({
            container:this,
            instance:'qual' + this.instance,
            x:(bH * 6) + ((this.w - (bH * 6)) / 2),
            y:0,
            w:(this.w - (bH * 6)) / 2,
            h:bH,
            sliderHeight:bH / 2,
            drawContainer:true,
            callback:this.setQuality,
            callbackParam:this,
            frames:4,
            background:{
                color:'#e0e0e0'
            },
            context:ctx,
            tabIndex: 201
        });
        this.resetButton = new OU.util.ControlButton({
            txt:'Reset',
            x:0,
            y:0,
            w:bH * 2,
            h:bH,
            layer:this.controlsLayer,
            tabIndex:10,
            onClick:function () {
                self.resetView();
            }
        });
        this.bgLabels();
    };
    OU.activity.Equation3D.prototype.bgLabels = function () {
        var bH = OU.controlHeight,
        bgctx = this.bgLayer.context;
        //			bgctx.save();
        //				bgctx.fillStyle = '#E8EDF3';
        //				bgctx.fillRect(0,this.h-bH,this.w,bH);
        bgctx.clearRect(0, this.h - bH, this.w, bH);
        //				bgctx.restore();
        bgctx.beginPath();
        bgctx.moveTo(0, this.h - bH - 1);
        bgctx.lineTo(this.w, this.h - bH - 1);
        bgctx.strokeStyle = '#cccccc';
        bgctx.stroke();
        new OU.util.pageTitle({
            txt:'3D Equations',
            context:bgctx
        }
        );
        new OU.util.pageStrap({
            txt:this.equation.name,
            context:bgctx,
            col1:'rgba(170,204,255,0.5)',
            col2:'rgba(170,204,255,0.0)'
        }
        );
        bgctx.textAlign = 'center';
    };
    OU.activity.Equation3D.prototype.renderPitchYaw = function () {
        var yaw = Math.abs(this.rotation.y) % 360 | 0,
        pitch = Math.abs(this.rotation.x) % 360 | 0;
        this.pyLayer.clear();
        this.pyLayer.context.fillText('Pitch: ' + pitch + '\u00B0' + '  Yaw: ' + yaw + '\u00B0', this.w * .95, this.h * 0.12);
    };
    OU.activity.Equation3D.prototype.moveTo = function ( x, y, z ) {
        var pt = new this.Point(x, y, z, this);
        this.modelLayer.context.moveTo(pt.in2d.x, pt.in2d.y);
    };
    OU.activity.Equation3D.prototype.lineTo = function ( x, y, z ) {
        var pt = new this.Point(x, y, z, this);
        this.modelLayer.context.lineTo(pt.in2d.x, pt.in2d.y);
    };
    OU.activity.Equation3D.prototype.text = function ( params ) {
        var pt = new this.Point(params.x, params.y, params.z, this);
        this.modelLayer.context.fillText(params.txt, pt.in2d.x, pt.in2d.y);
    };
    /**
     * @class
     */
    OU.activity.Equation3D.prototype.Point = function ( x, y, z, graph ) {
        this.original = {
            'x':x,
            'y':y,
            'z':z
        };
        // rotate about Y axis
        var nz = (z * graph.rotCY) - (x * graph.rotSY);
        x = (z * graph.rotSY) + (x * graph.rotCY);
        z = nz;
        // rotate about X axis
        var ny = (y * graph.rotCX) - (z * graph.rotSX);
        z = (y * graph.rotSX) + (z * graph.rotCX);
        y = ny;
        this.in3d = {
            'x':x,
            'y':y,
            'z':z
        };
        // Note 3D to 2D projection is performed following the method
        // described at http://en.wikipedia.org/wiki/3D_projection
        var eq = graph.equation;
        var dz = z - (eq.z1 - ((eq.z2 - eq.z1) * 60 * (1 - graph.zoom)));
        var rz = graph.window.h;
        var newX = (x * 15) / (dz) * rz + graph.viewXOffset;
        var newY = -((y * 15) / (dz) * rz) + graph.viewYOffset;
        this.in2d = { // Point in 2d relative to the window in pixels
            'x':newX,
            'y':newY
        };
        newY = -newY;
        var rr = graph.renderRange;
        if (newX > rr.maxX)
            rr.maxX = newX;
        if (newX < rr.minX)
            rr.minX = newX;
        if (newY > rr.maxY)
            rr.maxY = newY;
        if (newY < rr.minY)
            rr.minY = newY;
    };
    /**
     * @class
     */
    OU.activity.Equation3D.prototype.Equation = function ( params ) {
        //TODO: ensure x2>x1, y2>y1, z2>z1 - if not, swap round values
        this.name = params.name || 'y=?';
        this.x1 = params.x1 || 0;
        this.x2 = params.x2 || 10;
        this.y1 = params.y1 || 0;
        this.y2 = params.y2 || 10;
        this.z1 = params.z1 || 0;
        this.z2 = params.z2 || 10;
        this.reset_x1 = this.x1;
        this.reset_x2 = this.x2;
        this.reset_y1 = this.y1;
        this.reset_y2 = this.y2;
        this.reset_z1 = this.z1;
        this.reset_z2 = this.z2;
        this.variables = params.variables || [];
        this.calcY = params.calcY;
        if (params.colour!==undefined)
            this.colour = params.colour;
        if (params.rgb!==undefined)
            this.rgb = params.rgb;
        OU.activity.Equation3D.prototype.Equation.prototype.deltas = function () {
            this.dx = Math.abs(this.x2 - this.x1);
            this.dy = Math.abs(this.y2 - this.y1);
            this.dz = Math.abs(this.z2 - this.z1);
        };
        this.deltas();
    };
    /**
     * Overides base method for accessible view
     */
    OU.activity.Equation3D.prototype.accessibleView = function () {
        var i, h = '<div id="equation3DAccessible">';
        h += '<h1>' + this.data.title + '</h1>';
        h += '<p>' + this.data.description + '</p>';
        for (i = 0; i < this.data.equations.length; i++) {
            var equation = this.data.equations[i];
            h += '<h2>' + equation.name + '</h2><p>' + equation.description + '</p>';
        }
        h += '</div>';
        document.body.innerHTML = '';
        var accessible = document.createElement('div');
        accessible.innerHTML = h;
        document.body.appendChild(accessible);
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.Equation3D, OU.util.Activity);
