/**
 * @fileOverview Virtual Microscope - HTML5 version
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.Slider');
OU.require('OU.util.Layer');
OU.require('OU.util.TileViewer');
/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.Microscope = function(data, instance, controller) {
    OU.activity.Microscope.prototype.canvasView = function() {
        var self = this, bH = OU.controlHeight, aspectRatio, thumbnail, w, h;
        // create Canvas Layers & Contexts
        this.bgLayer = new OU.util.Layer({
            container: this,
            id: 'bg'
        });
        if (this.data.backgroundColour !== undefined)
            this.bgLayer.context.gradRect({
                col1: this.data.backgroundColour,
                col2: this.data.backgroundColour
            }); // use specified background colour
        else
            this.bgLayer.context.gradRect(); // use default background
        this.scopeLayer = new OU.util.Layer({
            container: this,
            id: 'canvas',
            h: this.h - bH,
            hasEvents: true,
            pinch: this.pinch,
            pinchMe: this,
            keyPress: function(key) {
                self.keypress(key);
            }
        });
        if (this.data.thumbnail) {
            aspectRatio = this.data.width / this.data.height;
            if (aspectRatio > 1) {
                w = this.data.tileSize + 2;
                h = (w / aspectRatio + 2) | 0;
            }
            else {
                h = this.data.tileSize + 2;
                w = (h * aspectRatio + 2) | 0;
            }
            this.thumbLayer = new OU.util.Layer({
                container: this,
                id: 'thumbnail',
                x: this.w - w,
                w: w,
                y: this.h - bH - h,
                h: h,
                hasEvents: true
            });
            thumbnail = {
                layer: this.thumbLayer,
                x: 1,
                y: 1,
                w: w - 1,
                h: h - 1
            };
        }
        this.controlsBackLayer = new OU.util.Layer({
            container: this,
            id: 'oucontrols',
            y: this.h - bH,
            h: bH,
            hasEvents: true
        });
        this.controlsBackLayer.context.gradRect({
            col1: "#ddd",
            col2: "#ddd"
        });
        this.controlsLayer = new OU.util.Layer({
            container: this,
            id: 'oucontrols',
            y: this.h - bH,
            h: bH,
            hasEvents: true
        });
        this.inRotationView = false;
        this.showRulers = true;
        this.tileView = new OU.util.TileViewer({
            image: {
                w: this.data.width,
                h: this.data.height
            },
            tileSize: this.data.tileSize,
            useTileGroups: this.data.useTileGroups,
            imageBase: this.data.images,
            rotationImage: this.data.rotationImage,
            markerCallback: function(rot) {
                self.loadRotation(rot);
            },
            zoomLevels: this.data.zoomLevels,
            renderAngle: this.data.renderAngle,
            showRulers: this.showRulers,
            window: {
                w: this.w,
                h: this.h - bH
            },
            container: this,
            context: this.scopeLayer.context,
            zoomCallback: function(z) {
                self.setSlider(z);
            },
            thumbnail: thumbnail
        });
        this.scopeLayer.events.clickable.push(this.tileView);
        this.initControls();

        // Register the activity's Messenger API
        // pass an array of Parameters which define the API
        // each Parameter can have a get, set and monitor function
        this.defineActivityAPI([
            {
                name: "PositionPixels",
                getFunction: function() {
                    var position = self.tileView.getPositionPixels();
                    return {
                        x: position.x,
                        y: position.y,
                        zoom: position.zoom
                    };
                },
                setFunction: function(newPos) {
                    self.tileView.setPosition({
                        zoom: newPos.zoom,
                        xPixels: newPos.x,
                        yPixels: newPos.y
                    });
                    self.zoomSlider.sliderPos = newPos.zoom;
                    self.zoomSlider.render();
                },
                monitor: function(params) {
                    self.tileView.monitorPositionPixels = params.callback;
                }
            },
            {
                name: "PositionMM",
                getFunction: function() {
                    var position = self.tileView.getPositionMM();
                    return {
                        x: position.x,
                        y: position.y,
                        zoom: position.zoom
                    };
                },
                setFunction: function(newPos) {
                    self.tileView.setPosition({
                        zoom: newPos.zoom,
                        xMM: newPos.x,
                        yMM: newPos.y
                    });
                    self.zoomSlider.sliderPos = newPos.zoom;
                    self.zoomSlider.render();
                }
            },
            {
                name: "MeasureMM",
                getFunction: function() {
                    return self.tileView.getMeasureMM();
                },
                monitor: function(params) {
                    self.tileView.monitorMeasureMM = params.callback;
                }
            },
            {
                name: "Slide",
                getFunction: function() {
                    return self.tileView.imageIdx;
                },
                setFunction: function(newSlide) {
                    self.tileView.changeImage(newSlide);
                    self.tileView.render();
                    self.renderControls();
                }
            },
            {
                name: "FeatureState",
                getFunction: function() {
                    return {
                        rulers: self.showRulers,
                        measure: self.tileView.measureOn
                    };
                },
                setFunction: function(params) {
                    var features = {};
                    if (params.rulers !== undefined)
                        features.showRulers = self.showRulers = params.rulers;
                    if (params.measure !== undefined) {
                        features.measure = params.measure;
                        self.measureButton.state(params.measure);
                    }

                    self.tileView.setFeatures(features);
                    self.renderControls();
                }
            },
            {
                name: "viewURL",
                getFunction: function() {
                    return self.shareURL() + self.tileView.getDirectParams();
                }
            },
            {
                name: "Snapshot",
                getFunction: function(params) {
                    return self.scopeLayer.canvas.toDataURL(params.format, params.quality);
                }
            }
        ]);
        var URLVars = OU.getUrlVars();
        if (URLVars['rot'] > 0) {
            if (URLVars['deg']) {
                this._rotationStart = URLVars['deg'];
            }
            var rot = this.data.images[URLVars['s'] || 0].rotations[URLVars['rot'] - 1];
            rot.arrayIndex = URLVars['rot'] - 1;
            this.loadRotation(rot);
        }
    };
    OU.activity.Microscope.prototype.resize = function() {
        OU.activity.Microscope.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight, i, ctx = this.scopeLayer.context,
                aspectRatio, w, h,
                rightButtonsWidth = bH * 3,
                leftButtonsWidth = this.data.images.length * bH * 2;

        if (this._sharePopUp)
            this._sharePopUp.remove();
        this.bgLayer.resize();
        if (this.data.backgroundColour !== undefined)
            this.bgLayer.context.gradRect({
                col1: this.data.backgroundColour,
                col2: this.data.backgroundColour
            }); // use specified background colour
        else
            this.bgLayer.context.gradRect(); // use default background
        this.scopeLayer.resize({
            h: this.h - bH
        });
        ctx.font = '18px ' + OU.theme.font;
        ctx.lineWidth = 2;
        ctx.strokeStyle = '#c00';
        ctx.fillStyle = '#c00';
        ctx.textAlign = 'center';

        if (this.data.thumbnail) {
            aspectRatio = this.data.width / this.data.height;
            if (aspectRatio > 1) {
                w = this.data.tileSize + 2;
                h = (w / aspectRatio + 2) | 0;
            }
            else {
                h = this.data.tileSize + 2;
                w = (h * aspectRatio + 2) | 0;
            }
            this.thumbLayer.resize({
                x: this.w - w,
                w: w,
                y: this.h - bH - h,
                h: h
            });
        }

        this.controlsBackLayer.resize({
            y: this.h - bH,
            h: bH
        });
        this.controlsBackLayer.context.gradRect({
            col1: "#ddd",
            col2: "#ddd"
        });
        this.controlsLayer.resize({
            y: this.h - bH,
            h: bH
        });
        this.tileView.window = {
            w: this.w,
            h: this.h - bH
        };
        for (i = 0; i < this.data.images.length; i++) {
            this.viewButtons[i].resize({
                x: i * bH * 2,
                y: 0,
                w: bH * 2,
                h: bH
            });
            this.viewButtons[i].render();
        }
        this.measureButton.resize({
            x: this.w - rightButtonsWidth,
            y: 0,
            w: bH * 3,
            h: bH
        });
        this.measureButton.render();
        if (this.data.noShare === undefined) {
            rightButtonsWidth = rightButtonsWidth + bH * 3;
            this.shareButton.resize({
                x: this.w - rightButtonsWidth,
                y: 0,
                w: bH * 3,
                h: bH
            });
            this.shareButton.render();
        }
        this.zoomSlider.resize({
            x: leftButtonsWidth,
            y: 0,
            w: this.w - leftButtonsWidth - rightButtonsWidth,
            h: bH
        });
        this.tileView.render();
        if (this.inRotationView) {
            this.loadedRotation.remove();
            this.loadRotation(this.openRotation);
        }
    };
    OU.activity.Microscope.prototype.initControls = function() {
        var i, ctx = this.controlsLayer.context, bH = OU.controlHeight, self = this,
                URLVars = OU.getUrlVars(), slideNum = URLVars['s'] || 0,
                initZoom = URLVars['zoom'] === undefined ? 0 : parseFloat(URLVars['zoom']),
                clickable = this.controlsLayer.events.clickable,
                rightButtonsWidth = bH * 3,
                leftButtonsWidth = this.data.images.length * bH * 2;

        this.measureSize = new OU.util.DynText({
            txt: 'Measure',
            x: this.w - rightButtonsWidth,
            y: 0,
            w: bH * 3 - (24*OU.dpr + 20), // text will be button width, minus icon width&padding
            h: bH,
            context: this.controlsLayer.context,
            measureOnly: true
        });
        this.buttonFontSize = this.measureSize.font.size;
        clickable.length = 0;
        this.viewButtons = [];
        for (i = 0; i < this.data.images.length; i++) {
            this.viewButtons[i] = new OU.util.CheckBoxButton({
                txt: this.data.images[i].label,
                x: i * bH * 2,
                y: 0,
                w: bH * 2,
                h: bH,
                layer: this.controlsLayer,
                fontSize: this.buttonFontSize,
                onClick: function(p) {
                    self.changeImage(p);
                },
                onClickParam: i,
                state: i === slideNum ? true : false
            });
        }
        this.measureButton = new OU.util.CheckBoxButton({
            txt: 'Measure',
            x: this.w - rightButtonsWidth,
            y: 0,
            w: bH * 3,
            h: bH,
            layer: this.controlsLayer,
            fontSize: this.buttonFontSize,
            state: false,
            onClick: function() {
                self.toggleMeasure();
            }
        });
        this.tileView.measureOn = false;
        if (this.data.noShare === undefined) {
            rightButtonsWidth = rightButtonsWidth + bH * 3;
            this.shareButton = new OU.util.ControlButton({
                txt: 'Share',
                x: this.w - rightButtonsWidth,
                y: bH * .1,
                w: bH * 3,
                h: bH * .8,
                fontSize: this.buttonFontSize,
                layer: this.controlsLayer,
                onClick: function() {
                    self.share();
                }
            });
        }
        this.zoomSlider = new OU.util.Slider({
            container: this,
            sliderPos: initZoom,
            instance: 'zoom',
            x: leftButtonsWidth,
            y: 0,
            w: this.w - leftButtonsWidth - rightButtonsWidth,
            h: bH,
            sliderHeight: bH / 2,
            drawContainer: false,
            callback: function(val) {
                self.setZoom(val);
            },
            background: {
                clear: true
            },
            context: ctx
        });
        clickable.push(this.zoomSlider);
        this.renderSlider();
    };
    OU.activity.Microscope.prototype.changeImage = function(idx) {
        var i;
        this.tileView.changeImage(idx);
        for (i = 0; i < this.viewButtons.length; i++) {
            this.viewButtons[i].state(i === idx ? true : false);
        }
        this.renderControls();
    };
    OU.activity.Microscope.prototype.shareURL = function() {
        var match = /(.+:\/\/.+?\/).+\/html5Assets\/(.+)\//i.exec(window.location.href);
        if (match) { // found JISC project URL format
            return match[1] + 'rock_sample?asset=' + match[2] + '/index.html?';
        }
        match = /.+\??/i.exec(window.location.href);
        if (match) // strip any old parameters
            return match[0] + '?';
        // otherwise just return current URL
        return window.location.href + '?';
    };
    OU.activity.Microscope.prototype.share = function(rotationInfo) {
        var self = this, h, view = this.tileView.getDirectParams(), link;
        if (rotationInfo)
            view += rotationInfo;
        h = '<p>Share the current view using the link below.</p><p>'
                + '<input style="width:80%" id="shareLink" value="' + this.shareURL() + view + '"/>';
        this._sharePopUp = new OU.util.PopUpInfo({
            container: this,
            x: this.w * .1,
            y: 50,
            w: this.w * .5,
            h: 200,
            txt: h,
            zIndex:9200,
            onClose: function() {
                self._sharePopUp = null;
            }
        });
        link = document.getElementById('shareLink');
        link.select();
    };
    OU.activity.Microscope.prototype.toggleMeasure = function() {
        switch (this._measureState) {
            default:
            case 'off':
                this._measureState = 'measure';
                this.tileView.measureOn = true;
                this.tileView.vectorAngles = false;
                this.measureButton.state(true);
                this.measureButton.modify({txt: "Measure"});
                break;
            case 'measure':
                this._measureState = 'angle';
                this.tileView.measureOn = true;
                this.tileView.vectorAngles = true;
                this.measureButton.state(true);
                this.measureButton.modify({txt: "Angle"});
                break;
            case 'angle':
                this._measureState = 'off';
                this.tileView.measureOn = false;
                this.measureButton.state(false);
                this.measureButton.modify({txt: "Measure"});
                break;
        }
        this.renderControls();
        this.tileView.doRender = true;
    };
    OU.activity.Microscope.prototype.renderControls = function() {
        var i;
        this.controlsLayer.context.clearRect(0, 0, this.w, OU.controlHeight);
        this.measureButton.render();
        if (this.shareButton)
            this.shareButton.render();
        for (i = this.viewButtons.length; i--; ) {
            this.viewButtons[i].render();
        }
        this.renderSlider();
    };
    OU.activity.Microscope.prototype.renderSlider = function() {
        var self = this;
        this.zoomSlider.render();
        setTimeout(function() {
            self.renderSlider();
        }, 40);
    };
    OU.activity.Microscope.prototype.setZoom = function(val) { // Called when scrubBar is moved
        this.zoomSlider.render();
        this.tileView.scale(val);
    };
    OU.activity.Microscope.prototype.setSlider = function(val) { // Called when TileViewer changed via mousewheel, etc.
        this.zoomSlider.setTarget(val);
    };
    OU.activity.Microscope.prototype.keypress = function(keycode) {
        var ns = null;

        if (keycode === 65) { // 'A'
            ns = this.tileView.s + 0.1;
            ns = ns > 1 ? 1 : ns;
        }
        if (keycode === 90) { // 'Z'
            ns = this.tileView.s - 0.1;
            ns = ns < 0 ? 0 : ns;
        }
        if (ns) {
            this.setZoom(ns);
            this.zoomSlider.sliderPos = ns;
            this.zoomSlider.render();
        }

    };
    OU.activity.Microscope.prototype.pinch = function(end, start, x, y, dx, dy, me) { // called when pinch event detected
        var ns = ((end - start) / me.h) + me.tileView.s;
        ns = ns > 1 ? 1 : (ns < 0 ? 0 : ns);
        me.tileView.scale(ns, {
            x: x,
            y: y
        });
        me.zoomSlider.sliderPos = ns;
        me.zoomSlider.render();
    };
    OU.activity.Microscope.prototype.loadRotation = function(rot) {
        if (this.inRotationView)
            return;
        this.inRotationView = true;
        this.scopeLayer.events.touched = this.scopeLayer.events.pressed = false;
        this.openRotation = rot;
        rot.container = this;
        this.loadedRotation = new this.Rotation(rot);
    };
    /**
     * @class
     */
    OU.activity.Microscope.prototype.Rotation = function(params) {
        var rotation = this, bH = OU.controlHeight, sW;
        this.title = params.title || '';
        this.pplData = params.pplData || '';
        this.xplData = params.xplData || '';
        this.arrayIndex = params.arrayIndex || 0;
        this.container = params.container;
        //TODO - move to Top/Bottom arrangement if Portrait view
        if (this.container.w > this.container.h) {
            sW = this.container.h * .95 < (this.container.w / 2) * .95 ? this.container.h * .95 : (this.container.w / 2) * .95;
            sW = sW > 460 ? 460 : sW;
            this.slideWidth = params.slideWidth || sW;
            this.slideHeight = params.slideHeight || sW;
            this.orientation = 'landscape';
            this.leftX = (this.container.w / 2 - this.slideWidth) / 2;
            this.rightX = this.container.w / 2 + this.leftX;
            this.lefttopY = this.righttopY = (this.container.h - bH - this.slideHeight) / 2;
        }
        else {
            sW = this.container.w * .95 < (this.container.h / 2) * .95 ? this.container.w * .95 : (this.container.h / 2) * .95;
            sW = sW > 460 ? 460 : sW;
            this.slideWidth = params.slideWidth || sW;
            this.slideHeight = params.slideHeight || sW;
            this.orientation = 'portrait';
            this.rightX = this.leftX = (this.container.w - this.slideWidth) / 2;
            this.lefttopY = (this.container.h / 2 - this.slideWidth) / 2;
            this.righttopY = this.container.h / 2 + (this.container.h / 2 - this.slideWidth) / 2;
        }
        this.leftCenterX = this.leftX + this.slideWidth / 2;
        this.rightCenterX = this.rightX + this.slideWidth / 2;
        this.leftcenterY = this.lefttopY + this.slideHeight / 2;
        this.rightcenterY = this.righttopY + this.slideHeight / 2;
        this.numberOfImagesInFullRotation = params.numberOfImagesInFullRotation || 72;
        this.reuseCount = params.reuseCount || 2;
        this.degrees = ((this.container._rotationStart || 0) * Math.PI / 180) % (Math.PI * 2);
        /**
         * @private
         */
        OU.activity.Microscope.prototype.Rotation.prototype.init = function() {
            var i, imageLayer, ctx, bH = OU.controlHeight, scaleX, scaleY, scaleInPixels;
            this.bgLayer = new OU.util.Layer({// Background layer masks the microscope underneath
                zIndex: OU.POP_UP_LEVEL
            });
            this.bgLayer.context.fillStyle = '#fff';
            this.bgLayer.context.fillRect(0, 0, this.container.w, this.container.h);
            // Create 2 DIVs to contain the left and right image sets
            this.leftBox = document.createElement("div"),
                    this.rightBox = document.createElement("div");
            this.leftBox.style.width = this.rightBox.style.width = this.slideWidth + "px";
            this.leftBox.style.height = this.rightBox.style.height = this.slideHeight + "px";
            this.leftBox.style.left = this.leftX + "px";
            this.rightBox.style.left = this.rightX + "px";
            this.leftBox.style.top = this.lefttopY + "px";
            this.rightBox.style.top = this.righttopY + "px";
            this.leftBox.style.zIndex = this.rightBox.style.zIndex = OU.POP_UP_LEVEL + 1;
            this.leftBox.className = this.rightBox.className = "microscopeRotation";
            document.body.appendChild(this.leftBox);
            document.body.appendChild(this.rightBox);
            // Load the images into the divs
            this.segmentAngle = 2.0 * Math.PI / this.numberOfImagesInFullRotation;
            for (i = 0; i < this.numberOfImagesInFullRotation / this.reuseCount; i++) {
                // Add PPL image
                imageLayer = document.createElement("img");
                imageLayer.style.visibility = "hidden";
                imageLayer.src = this.container.dataDir + this.pplData + (((i + 1) < 10) ? "0" : "") + (i + 1) + ".jpg";
                imageLayer.width = this.slideWidth;
                imageLayer.height = this.slideHeight;
                imageLayer.style.zIndex = OU.POP_UP_LEVEL + 1 + i;
                this.leftBox.appendChild(imageLayer);
                // Add XPL image
                imageLayer = document.createElement("img");
                imageLayer.style.visibility = "hidden";
                imageLayer.src = this.container.dataDir + this.xplData + (((i + 1) < 10) ? "0" : "") + (i + 1) + ".jpg";
                imageLayer.width = this.slideWidth;
                imageLayer.height = this.slideHeight;
                imageLayer.style.zIndex = OU.POP_UP_LEVEL + 1 + i;
                this.rightBox.appendChild(imageLayer);
            }
            // Add a control layer to capture touch/mouse events & contain other visual elements
            this.controlLayer = new OU.util.Layer({
                zIndex: OU.POP_UP_LEVEL + this.numberOfImagesInFullRotation + 10,
                hasEvents: true
            });
            ctx = this.controlLayer.context;
            ctx.font = this.slideWidth * .05 + 'px ' + OU.theme.font;
            this.controlLayer.events.clickable.push(this);
            this.backButton = new OU.util.ControlButton({
                txt: 'Back',
                x: 0,
                y: this.container.h - bH,
                w: bH * 2,
                h: bH,
                layer: this.controlLayer,
                fontSize: this.buttonFontSize,
                onClick: function() {
                    rotation.close();
                }
            });
            this.shareButton = new OU.util.ControlButton({
                txt: 'Share',
                x: bH * 2.5,
                y: this.container.h - bH,
                w: bH * 2,
                h: bH,
                layer: this.controlLayer,
                fontSize: this.buttonFontSize,
                onClick: function() {
                    var degrees = (rotation.degrees / (Math.PI / 180)) | 0;
                    var rot = (rotation.arrayIndex || 0) + 1;
                    rotation.container.share('&rot=' + rot + "&deg=" + degrees);
                }
            });
            // Title
            ctx.textAlign = 'center';
            if (this.orientation === 'landscape') {
                ctx.fillStyle = 'rgba(256,256,256,0.65)';
                ctx.fillRect(0, this.container.h * .1 - 16, this.container.w, 32);
                ctx.fillStyle = '#222';
                ctx.fillText(this.title, this.container.w / 2, this.container.h * .05);
                ctx.fillText("plane polarised light", this.container.w / 4, this.container.h * .1);
                ctx.fillText("between crossed polars", this.container.w * .75, this.container.h * .1);
            }
            else {
                ctx.save();

                ctx.fillStyle = 'rgba(256,256,256,0.65)';
                ctx.fillRect(0, bH - 16, ctx.measureText("plane polarised light").width + 20, 32);
                ctx.fillRect(this.container.w - ctx.measureText(this.title).width - 20, bH - 16, ctx.measureText(this.title).width + 20, 32);
                ctx.fillRect(0, this.container.h * .5 + bH - 16, ctx.measureText("between crossed polars").width + 20, 32);
                ctx.fillStyle = '#222';
                ctx.textAlign = 'left';
                ctx.fillText(this.title, this.container.w - ctx.measureText(this.title).width - 10, bH);
                ctx.fillText("plane polarised light", 10, bH);
                ctx.fillText("between crossed polars", 10, this.container.h * .5 + bH);
                ctx.restore();
            }
            // scale
            scaleInPixels = 234 * (this.slideWidth / 460); // TODO this should be dynamic, provided by data
            if (this.orientation === 'landscape') {
                scaleY = this.container.h - bH - (this.lefttopY / 2);
                scaleX = this.container.w / 2 - scaleInPixels / 2;
            }
            else {
                scaleY = this.container.h / 2;
                scaleX = this.container.w - 10 - scaleInPixels;
            }
            ctx.save();
            ctx.font = '12px ' + OU.theme.font;
            ctx.beginPath();
            ctx.fillText('1mm', scaleX + scaleInPixels / 2, scaleY + 10);
            ctx.moveTo(scaleX, scaleY - 10);
            ctx.lineTo(scaleX, scaleY + 10);
            ctx.moveTo(scaleX, scaleY);
            ctx.lineTo(scaleX + scaleInPixels, scaleY);
            ctx.moveTo(scaleX + scaleInPixels, scaleY - 10);
            ctx.lineTo(scaleX + scaleInPixels, scaleY + 10);
            ctx.lineWidth = 2;
            ctx.strokeStyle = '#000';
            ctx.stroke();
            ctx.restore();
            // Crosshairs
            ctx.moveTo(this.leftX, this.lefttopY + this.slideHeight / 2);
            ctx.lineTo(this.leftX + this.slideWidth, this.lefttopY + this.slideHeight / 2);
            ctx.moveTo(this.leftX + this.slideWidth / 2, this.lefttopY);
            ctx.lineTo(this.leftX + this.slideWidth / 2, this.lefttopY + this.slideHeight);
            ctx.moveTo(this.rightX, this.righttopY + this.slideHeight / 2);
            ctx.lineTo(this.rightX + this.slideWidth, this.righttopY + this.slideHeight / 2);
            ctx.moveTo(this.rightX + this.slideWidth / 2, this.righttopY);
            ctx.lineTo(this.rightX + this.slideWidth / 2, this.righttopY + this.slideHeight);
            ctx.strokeStyle = '#fff';
            ctx.stroke();
            this.doRender = true;
        };
        OU.activity.Microscope.prototype.Rotation.prototype.renderCycle = function() {
            if (this.doRender)
                this.performRender();
            setTimeout(function() {
                rotation.renderCycle();
            }, 20);
        };
        OU.activity.Microscope.prototype.Rotation.prototype.render = function() {
            this.doRender = true;
        };
        OU.activity.Microscope.prototype.Rotation.prototype.share = function() {

        };
        OU.activity.Microscope.prototype.Rotation.prototype.performRender = function() {
            var i, leftLayer, rightLayer, rotationDegrees = this.degrees, temp,
                    r2d = Math.PI / 180, ctx = this.controlLayer.context,
                    nImgs = this.numberOfImagesInFullRotation / this.reuseCount,
                    lower = parseInt(this.degrees / this.segmentAngle),
                    upper = (lower + 1), // % nImgs,
                    upOpacity = (this.degrees - lower * this.segmentAngle) / this.segmentAngle;
            this.doRender = false;
            lower = lower % nImgs;
            upper = upper % nImgs;
            if (lower > upper) { // Swap order if required
                temp = upper;
                upper = lower;
                lower = temp;
                upOpacity = 1 - upOpacity;
            }
            // step through slides and set visibility,opacity & rotation accordingly
            for (i = nImgs; i--; ) {
                leftLayer = this.leftBox.childNodes[i];
                rightLayer = this.rightBox.childNodes[i];
                if (i === lower) {
                    if (leftLayer.style.visibility !== "visible")
                        leftLayer.style.visibility = rightLayer.style.visibility = "visible";
                    leftLayer.style.webkitTransform = rightLayer.style.webkitTransform = "rotate3d(0,0,1," + (rotationDegrees - this.segmentAngle * i) + "rad)";
                    leftLayer.style.MozTransform = rightLayer.style.MozTransform = "rotate(" + (rotationDegrees - this.segmentAngle * i) + "rad)";
                    leftLayer.style.opacity = rightLayer.style.opacity = 1;
                } else if (i === upper) {
                    if (leftLayer.style.visibility !== "visible")
                        rightLayer.style.visibility = leftLayer.style.visibility = "visible";
                    leftLayer.style.webkitTransform = rightLayer.style.webkitTransform = "rotate3d(0,0,1," + (rotationDegrees - this.segmentAngle * i) + "rad)";
                    leftLayer.style.MozTransform = rightLayer.style.MozTransform = "rotate(" + (rotationDegrees - this.segmentAngle * i) + "rad)";
                    leftLayer.style.opacity = rightLayer.style.opacity = upOpacity;
                }
                else {
                    if (leftLayer.style.visibility !== "hidden")
                        rightLayer.style.visibility = leftLayer.style.visibility = "hidden";
                }
            }
            if (this.orientation === 'landscape') {
                ctx.clearRect(this.leftX + this.slideWidth * .6, this.lefttopY + this.slideWidth * .9, this.rightX + (this.slideWidth * .4) - (this.leftX + this.slideWidth * .6), this.slideWidth * .1);
                ctx.fillText((this.degrees / r2d | 0) + '\u00B0', this.container.w / 2, this.lefttopY + this.slideWidth * .95);
            }
            else {
                ctx.clearRect(0, this.lefttopY + this.slideWidth * .95 - 10, this.container.w / 2, 20);
                ctx.fillText((this.degrees / r2d | 0) + '\u00B0', this.container.w / 4, this.lefttopY + this.slideWidth * .95);
            }
        };
        OU.activity.Microscope.prototype.Rotation.prototype.isHit = function(x, y, evState) {
            var dX, dY, oldTheta, newTheta;
            if (evState) {
                if (y < this.righttopY)
                    dY = y - this.leftcenterY;
                else
                    dY = y - this.rightcenterY;
                if (x < this.rightX)
                    dX = x - this.leftCenterX;
                else
                    dX = x - this.rightCenterX;
                this.distance = Math.sqrt(dX * dX + dY * dY);
                if (this.distance > this.slideWidth / 2)
                    return;
                if (this.inDrag) {
                    oldTheta = Math.atan2(this.startdY, this.startdX),
                            newTheta = Math.atan2(dY, dX);
                    this.degrees += newTheta - oldTheta;
                    this.degrees = (this.degrees + (Math.PI * 2)) % (Math.PI * 2);
                    this.startdY = dY;
                    this.startdX = dX;
                    this.render();
                }
                else {
                    this.startdY = dY;
                    this.startdX = dX;
                    this.inDrag = true;
                }
            }
            else {
                this.inDrag = false;
            }
        };
        OU.activity.Microscope.prototype.Rotation.prototype.remove = OU.activity.Microscope.prototype.Rotation.prototype.close = function() {
            OU.activity.Microscope.superClass_.remove.call(this); // call the superclass method
            this.controlLayer.remove();
            if (this.leftBox.parentNode !== null) {
                this.leftBox.parentNode.removeChild(this.leftBox);
                this.rightBox.parentNode.removeChild(this.rightBox);
            }
            this.bgLayer.remove();
            this.container.inRotationView = false;
        };
        this.init();
        this.renderCycle();
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.Microscope, OU.util.Activity);
