/**
 * @fileOverview Photo Reel Object
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.DynText');
OU.require('OU.util.Layer');
OU.require('OU.util.ImageLoader');
/** 
 * @class
 * @extends OU.util.Activity
 */
OU.activity.PhotoReel = function(data,instance,controller) {

    OU.activity.PhotoReel.prototype.canvasView = function() {
        var self=this;
        this.config = {
            imageLoadTimeout:10000, // allow max 10 seconds to load images

            // Settings for Controls
            overlayControls:false, // set to false to seperate controls from images

            // animation settings
            fps:40, // 40ms = 25 fps
            slideDuration: 10000, // each slide takes 10 seconds
            transPerc:0.2 // percentage of slideDuration to face in next slide (20%)
        };

        this.bgLayer = new OU.util.Layer({
            container:this
        });
        this.bgLayer.context.gradRect(); // draw background on backdrop layer

        this.imageLayer = new OU.util.Layer({
            container:this
        });

        // determine Proportions
        this.config.maxWidth = this.w;
        this.config.maxHeight= this.config.layerHeight = this.h;

        this.currentImage=0;
        this.slideStart = [];
        this.initImages();
        this.imageLoader = new OU.util.ImageLoader({
            container:this,
            data: this.data.images,
            onLoad: function() {
                self.start();
            }
        });
    };
    OU.activity.PhotoReel.prototype.resize = function() {
        OU.activity.PhotoReel.superClass_.resize.call(this); // call the parent class resize 
        this.bgLayer.resize();
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        this.imageLayer.resize();
        // determine Proportions
        this.config.maxWidth = this.w;
        this.config.maxHeight= this.config.layerHeight = this.h;
        this.initImages();
    };
    OU.activity.PhotoReel.prototype.render = function() {
        var self=this,slide = this.data.images[this.currentImage],
        bH = this.h*.1,
        now = new Date().getTime(),
        slideTime = now-this.slideStart[this.currentImage],
        slidePerc = slideTime/this.config.slideDuration,
        w = slide.sW+((slide.eW-slide.sW)*slidePerc),
        h = slide.sH+((slide.eH-slide.sH)*slidePerc),
        x = slide.oX+slide.sX+(slide.eX-slide.sX)*slidePerc,
        y = slide.oY+slide.sY+(slide.eY-slide.sY)*slidePerc,
        ctx = this.imageLayer.context;

        if(ctx==null)
            return;
        ctx.drawImage(slide.image,x,y,w,h);

        if(slidePerc>(1-this.config.transPerc)) { // start fading in next slide
            var nextImage = (this.currentImage+1)%this.data.images.length;
            var nextSlide = this.data.images[nextImage];

            if(this.slideStart[nextImage] === undefined || this.slideStart[nextImage] == null)
                this.slideStart[nextImage] = now;
            var nextSlideTime = now-this.slideStart[nextImage];
            var nextSlidePerc = nextSlideTime/this.config.slideDuration;
            w = nextSlide.sW+((nextSlide.eW-nextSlide.sW)*nextSlidePerc);
            h = nextSlide.sH+((nextSlide.eH-nextSlide.sH)*nextSlidePerc);
            x = nextSlide.oX+nextSlide.sX+(nextSlide.eX-nextSlide.sX)*nextSlidePerc;
            y = nextSlide.oY+nextSlide.sY+(nextSlide.eY-nextSlide.sY)*nextSlidePerc;

            // calc transparency of next slide
            ctx.save();
            var trans=(nextSlideTime/(this.config.slideDuration*this.config.transPerc));
            ctx.globalAlpha = trans;
            ctx.drawImage(nextSlide.image,x,y,w,h);
            ctx.restore();
        }
        if(slideTime>= this.config.slideDuration) {
            this.slideStart[this.currentImage] = null;
            this.currentImage = nextImage;
        }

        // render layer title
        if(this.data.showLabel !== undefined && this.data.showLabel && slide.label!='') {
            new OU.util.DynText({
                txt:slide.label,
                x:this.w*.05,
                y:this.h-(bH*.75),
                w:this.w*0.9,
                h:bH/2,
                background: {
                    RGB: '255,255,255',
                    radius: bH/4,
                    alpha: 0.5
                },
                context:this.imageLayer.context
            });
        }
        setTimeout(function() {
            self.render();
        },this.config.fps);
    };
    OU.activity.PhotoReel.prototype.start = function() {
        this.slideStart[0] = new Date().getTime();
        this.render();
    };
    OU.activity.PhotoReel.prototype.initImages = function() {
        var i,slide,minWsF,minHsF,e100sF,scale,dW,dH;
        this.loadedImagesCount=0;
        this.startedLoadingTime= new Date();
        for(i=0; i<this.data.images.length; i++) {
            slide = this.data.images[i];

            // calc tween metrics
            minWsF = this.w/slide.w;
            minHsF = this.h/slide.h;
            if(minWsF>minHsF) { // fit width
                e100sF = minWsF;
                slide.oX=0;
                slide.oY=(this.h-(slide.h*e100sF))/2;
            }
            else {
                e100sF = minHsF;
                slide.oX=(this.w-(slide.w*e100sF))/2;
                slide.oY=0;
            }


            if(slide.startPerc===undefined) {
                if(i%2 == 0) {
                    slide.startPerc = 0.7;
                    slide.endPerc = 1;
                    slide.sX = -((slide.w*(e100sF + (1-e100sF)*(0.2)))-slide.w)*Math.random();
                    slide.sY = -((slide.h*(e100sF + (1-e100sF)*(0.2)))-slide.h)*Math.random();
                    slide.eX=0;
                    slide.eY=0;
                }
                else {
                    slide.startPerc = 1;
                    slide.endPerc = 0.7;
                    slide.sX=0;
                    slide.sY=0;
                    slide.eX = -((slide.w*(e100sF + (1-e100sF)*(0.2)))-slide.w)*Math.random();
                    slide.eY = -((slide.h*(e100sF + (1-e100sF)*(0.2)))-slide.h)*Math.random();
                }
            }

            scale = e100sF + (1-e100sF)*(1-slide.startPerc);
            slide.sW = slide.w * scale;
            slide.sH = slide.h * scale;
            scale = e100sF + (1-e100sF)*(1-slide.endPerc);
            slide.eW = slide.w * scale;
            slide.eH = slide.h * scale;

            dW=Math.abs(slide.eW-slide.sW);
            dH=Math.abs(slide.eH-slide.sH);
            if(i%2 == 0) {
                slide.sX = -dW*Math.random();
                slide.sY = -dH*Math.random();
                slide.eX=0;
                slide.eY=0;
            }
            else {
                slide.sX=0;
                slide.sY=0;
                slide.eX = -dW*Math.random();
                slide.eY = -dH*Math.random();
            }
        }
    };
    OU.activity.PhotoReel.prototype.accessibleView = function() {
        var h = '<div id="accessibleView"><h1>'+this.data.title+'</h1>'
        + '<p>'+this.data.description+'</p><h2>This photo reel has the following images:</h2>';
        for(var i=0; i<this.data.images.length; i++)
            h += '<h3>'+this.data.images[i].label+'</h3>';

        h += '</div>';
        document.body.innerHTML='';
        var accessible = document.createElement('div');
        accessible.innerHTML = h;
        document.body.appendChild(accessible);
    };
    OU.base(this,data,instance,controller);
};
OU.inherits(OU.activity.PhotoReel,OU.util.Activity);
