/**
 * @fileOverview TabController - load and traverse multiple activities using Tabs
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */
OU.require('OU.util.Controller');
OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.Layer');
/**
 * @class Tab Controller, allows multiple activities to be loaded under each tab.
 * Tabs can be enabled/disabled at the start, and switched on and off via an API call.
 *
 * Extends the base controller class, which does all the real work of the controller
 *
 * @extends OU.util.Controller
 */
OU.activity.TabController = function(controller_data,instance,controller) {
    this.buttons = [];
    OU.activity.TabController.prototype.canvasView = function() {
        var self=this;
        OU.activity.TabController.superClass_.canvasView.call(this); // call the parent class canvasView
        this.tabHeight=OU.controlHeight;
        this.padding=5;

        this.controlLayer = new OU.util.Layer({
            dontRegister: this.baseController,
            x: this.x,
            y: this.y,
            w: this.w,
            h: this.tabHeight,
            container:this,
            hasEvents: true
        });
        this.resize();

        // Register the activity's API.
        // An array of Parameters which define the API,
        // each Parameter can have a get, set and monitor function
        this.defineActivityAPI([
        {
            name: "EnableTabs",
            /**
             * @returns {json object[]} states - array of current states in format [{tabIndex: <1-n>, enabled: boolean}]
             * Note: index numbers start at 1, not 0
             */
            getFunction: function() {
                var i,states = [];
                for(i=self.data.sections.length; i--;) {
                    states[i]= {
                        tabIndex: i+1,
                        enabled: self.data.sections[i].tabEnabled!==false
                    }
                }
                return states;
            },
            /**
             * @param {json object[]} newStates - array of new states in format [{tabIndex: <1-n>, enabled: boolean}]
             * Note: index numbers start at 1, not 0
             */
            setFunction: function(newStates) {
                var i;
                for(i=newStates.length; i--;) {
                    self.data.sections[newStates[i].tabIndex-1].tabEnabled = newStates[i].enabled;
                }
                self.render();
            }
        }
        ]);
    };
    /**
     *  Resizes the control layer and re-renders
     */
    OU.activity.TabController.prototype.resize = function() {
        OU.activity.TabController.superClass_.resize.call(this); // call the parent class resize

        this.controlLayer.resize({
            x: this.x,
            y: this.y,
            w: this.w,
            h: OU.controlHeight
        });

        this.render();
    };
    /**
     * Renders the tabs and outer bounds of the "section"
     */
    OU.activity.TabController.prototype.render = function() {
        var ctl= this.controlLayer,self=this,
        bg=this.bgLayer,
        bctx=bg.context,
        bH = OU.controlHeight,
        i,s,ns = this.numSections,
        x=0;

        ctl.clear();
        bg.clear();

        for (i=0; i<ns; i++) {
            s = this.data.sections[i];
            s.sectionId = i;
            if(i!=this.sectionNum) {
                if(this.buttons[i]) {
                    if(s.tabEnabled!==false)
                        this.buttons[i].enable();
                    else
                        this.buttons[i].disable();
                    this.buttons[i].render();
                }
                else {
                    this.buttons[i] = new OU.util.Button({
                        layer: ctl,
                        x: x,
                        y: 0,
                        w: 160*OU.dpr,
                        h: bH+8,
                        background: {
                            col: '#d9d9d9',
                            radius: 8,
                            borderCol: '#666'
                        },
                        visiblePadding: 0,
                        fontSize: 16*OU.dpr,
                        disabled: s.tabEnabled===false,
                        txt: s.name,
                        onClick: function(n) {
                            self.step(n);
                        },
                        onClickParam: i+1
                    });
                }
            }
            else {
                new OU.util.DynText({
                    context: ctl.context,
                    x: x,
                    y: 0,
                    w: 160*OU.dpr,
                    h: bH+8,
                    fontSize: 16*OU.dpr,
                    background: {
                        col: '#fff',
                        radius: 8,
                        borderCol: '#666'
                    },
                    visiblePadding: 0,
                    txt: s.name
                });

            }
            x=x+160*OU.dpr;
        }

        bctx.strokeStyle='#666';
        bctx.roundRect(12+160*this.sectionNum,0,136,bH+10,8);
        bctx.stroke();
        bctx.fillStyle='#fff';
        bctx.beginPath();
        bctx.rect(10+160*this.sectionNum,bH,140,12);
        bctx.fill();
        bctx.roundRectTLC(1,bH,this.w-2,this.h-bH-2,12);
        bctx.stroke();
        bctx.beginPath();
        bctx.rect(1+160*this.sectionNum,bH-3,158,10);
        bctx.fill();
    };
    OU.base(this,controller_data,instance,controller);
};
if(OU.util.Controller) {
    OU.inherits(OU.activity.TabController, OU.util.Controller);
}
else {
    OU.preOnLoad.push(function () {
        OU.inherits(OU.activity.TabController, OU.util.Controller);
    });
}