/**
 * @fileOverview OU.activity.Temperaturechart - S104 Chemical Equilibrium Activity
 *
 * @author Kevin Quick & Jon Linney
 * @author Refactored by Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.PopUpInfo');
OU.require('OU.util.NavButtons');
OU.require('OU.util.Layer');
OU.require('OU.util.ImageLoader');
OU.require('OU.util.Slider');
/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.Temperaturechart = function ( data, instance, controller ) {

    /* Bad practice - variables declared here are shared across all instances and therefore clash
     * //TODO refactor these variables inside methods 
     */
    var temperaturechart = this;
    var butWidth, ballRad, butGap, butHeight, ballX, ballY, clickable;
    OU.activity.Temperaturechart.prototype.canvasView = function () {
        var self = this;
        this.whichReaction = 0;
        this.totalPressure = this.data.reactions[this.whichReaction].reactionPressure;
        this.popUpAudio = 0;
        this.minTemperature = this.data.reactions[this.whichReaction].minTemperature;
        this.minFeedback = this.data.reactions[this.whichReaction].minFeedback;
        this.reactionTemperature = this.minTemperature;
        this.showLowPopup = 0;
        OU.LocalStorage.save("ou.temperaturechart.temperature.reaction", 0);
        this.config = {
            vAreaW:5000,
            vScale:1,
            thenum:10,
            change:0,
            myColor:["#0037ff", "#fffa00", "#ff2400", "#00f900"],
            myData:[20, 30, 20, 40],
            maxTemperature:this.data.maxTemperature,
            minTemperature:273,
            step:-1,
            stepCompleted:0,
            stopstartCount:0,
            imagesLoaded:0,
            fontSize1:10,
            fontSize2:7
        };
        this.bgLayer = new OU.util.Layer({
            container:this
        });
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        this.imageLayer = new OU.util.Layer({
            container:this
        });
        this.controlLayer = new OU.util.Layer({
            container:this,
            hasEvents:true
        });
        this.boundary = {
            x:this.w * .05,
            y:this.h * .05,
            w:this.w * .9,
            h:this.h * .9,
            a:0,
            aParticleRadius:0
        };
        this.boundary.a = this.boundary.w * .7 < this.boundary.h ? this.boundary.w * .7 : this.boundary.h;
        this.config.vScale = this.boundary.a / this.config.vAreaW;
        this.boundary.aParticleRadius = this.config.vParticleRadius * this.config.vScale;
        this.fbWidth = this.w;
        // define dimentions depending on window size
        butWidth = 0.429 * this.boundary.a * .7;
        ballRad = 0.429 * this.boundary.a * .08;
        butGap = 0.429 * this.boundary.a * .05;
        butHeight = ballRad * 2;
        this.boundary.x = this.w * .05 + (this.boundary.w - (this.boundary.a + 2 * butGap + 2 * ballRad + butWidth)) * .5;
        //
        ballX = 1.429 * this.boundary.a - 2 * ballRad + this.boundary.x;
        ballY = (this.boundary.y + this.boundary.a) - ballRad;
        this.butData = {
            butWidth:butWidth,
            butHeight:butHeight,
            butGap:butGap,
            ballRad:ballRad,
            ballX:ballX,
            ballY:ballY
        };
        clickable = this.controlLayer.events.clickable;
        // create buttons
        if (this.data.changeButton==1) {
            temperaturechart.changeButton = new temperaturechart.Button({
                txt:"Change",
                x:this.boundary.a + this.boundary.x + butGap,
                y:(this.boundary.y + this.boundary.a) - butHeight,
                w:butWidth,
                h:butHeight,
                padding:butHeight / 10,
                onClick:temperaturechart.buttonClicked,
                onClickParam:{
                    choice:3,
                    me:this
                },
                disabled:true
            });
        }
        this.temperatureSlider = new OU.util.Slider({
            container:this,
            x:this.boundary.a + this.boundary.x + butGap,
            y:(this.boundary.y + this.boundary.a) - 3 * butHeight,
            w:butWidth,
            h:butHeight,
            //sliderStyle: 'circle',
            drawContainer:false,
            showValue:true,
            range:{
                max:this.config.maxTemperature - 273,
                min:this.minTemperature - 273,
                nDecimals:0
            },
            callback:this.setTemperature,
            callbackParam:this,
            background:{
                clear:true
            },
            context:this.controlLayer.context
        });
        clickable.push(this.temperatureSlider);
        this.imageLoader = new OU.util.ImageLoader({
            container:this,
            data:this.data,
            onLoad:function () {
                self.config.imagesLoaded = 1;
                self.start();
            }
        });
    };
    OU.activity.Temperaturechart.prototype.drawPie = function () {
        var i, lastend = -Math.PI / 2, config = this.config, ctx = this.imageLayer.context,
            midPointX = this.boundary.x + (this.boundary.a / 2),
            midPointY = this.boundary.y + (this.boundary.a / 2);
        ctx.strokeStyle = "#000";
        for (i = 0; i < config.myData.length; i++) {
            ctx.fillStyle = config.myColor[i];
            ctx.beginPath();
            ctx.moveTo(midPointX, midPointY);
            ctx.arc(midPointX, midPointY, this.boundary.a * .35, lastend - Math.PI * 2 * (config.myData[i] / 360), lastend, false);
            ctx.lineTo(midPointX, midPointY);
            ctx.fill();
            ctx.stroke();
            lastend -= Math.PI * 2 * (config.myData[i] / 360);
        }
    };
    OU.activity.Temperaturechart.prototype.resize = function () {
        OU.activity.Temperaturechart.superClass_.resize.call(this); // call the parent class resize         
        var config = this.config, bd;
        if (this.popUp)
            this.popUp.close();
        if (this.config.imagesLoaded==1) {
            this.bgLayer.resize();
            this.bgLayer.context.gradRect(); // draw background on backdrop layer
            this.imageLayer.resize();
            this.boundary = {
                x:this.w * .05,
                y:this.h * .05,
                w:this.w * .9,
                h:this.h * .9,
                a:0,
                aParticleRadius:0
            };
            this.boundary.a = this.boundary.w * .7 < this.boundary.h ? this.boundary.w * .7 : this.boundary.h;
            config.vScale = this.boundary.a / config.vAreaW;
            this.boundary.aParticleRadius = config.vParticleRadius * config.vScale;
            this.fbWidth = this.w;
            bd = this.butData;
            // redefine dimentions depending on window size
            bd.butWidth = 0.429 * this.boundary.a * .7;
            bd.ballRad = 0.429 * this.boundary.a * .08;
            bd.butGap = 0.429 * this.boundary.a * .05;
            bd.butHeight = bd.ballRad * 2;
            this.boundary.x = this.w * .05 + (this.boundary.w - (this.boundary.a + 2 * bd.butGap + 2 * bd.ballRad + bd.butWidth)) * .5;
            bd.ballX = 1.429 * this.boundary.a - 2 * bd.ballRad + this.boundary.x;
            bd.ballY = (this.boundary.y + this.boundary.a) - bd.ballRad;
            if (this.data.changeButton==1) {
                temperaturechart.changeButton.resize({
                    x:this.boundary.a + this.boundary.x + bd.butGap,
                    y:(this.boundary.y + this.boundary.a) - bd.butHeight,
                    w:bd.butWidth,
                    h:bd.butHeight,
                    fontSize:200 * config.vScale
                });
            }
            this.temperatureSlider.resize({
                x:this.boundary.a + this.boundary.x + bd.butGap,
                y:(this.boundary.y + this.boundary.a) - bd.butHeight * 3,
                w:bd.butWidth,
                h:bd.butHeight
            });
            this.controlLayer.resize();
            this.drawControl();
            this.doneLive = false;
            this.doRender = true;
        }
    };
    OU.activity.Temperaturechart.prototype.start = function () {
        this.drawControl();
        this.doRender = true;
        this.renderLoop();
    };
    //change pressure when slider altered
    OU.activity.Temperaturechart.prototype.setTemperature = function ( temperature, pchart ) {
        pchart.reactionTemperature = pchart.minTemperature + temperature * (pchart.config.maxTemperature - pchart.minTemperature);
        pchart.temperatureSlider.render(temperature);
        pchart.drawControl();
    };
    // buttons do this
    OU.activity.Temperaturechart.prototype.buttonClicked = function ( p ) {
        var whichClicked = p.choice;
        if (whichClicked==3 && p.me.config.stepCompleted >= 1) {
            // make popup to change the equation
            if (p.me.popUpAudio==0) {
                p.me.popUpAudio = 1;
                p.me.controller.stepAudio(2);
            }
            p.me.makePopUp();
        }
        p.me.drawControl();
    };
    // create a button
    OU.activity.Temperaturechart.prototype.Button = function ( params ) {
        params.propTextHeight = .5;
        params.fontWeight = 'normal';
        params.layer = temperaturechart.controlLayer;
        return new OU.util.Button(params);
    };
    // if true call the render loop to change the screen
    OU.activity.Temperaturechart.prototype.renderLoop = function () {
        var self = this;
        // test for showing the low temp popup
        if (this.reactionTemperature < this.oldTemperature && this.reactionTemperature <= this.minTemperature) {
            // this.showLowPopup = 1;
            if (this.minFeedback!="") {
                this.makeLowPopUp();
            }
        }
        this.oldTemperature = this.reactionTemperature;
        this.drawScreen();
        setTimeout(function () {
            self.renderLoop();
        }, 40);
    };
    OU.activity.Temperaturechart.prototype.drawControl = function () {
        var grd, gx, gy, tx, ty, fs, bx, by, theWidth, ctxCon = this.controlLayer.context,
            bgCon = this.bgLayer.context, butData = this.butData, moleculeW = 250;
        this.temperature();
        this.controlLayer.clear();
        this.bgLayer.clear();
        // draw plinth
        bgCon.strokeStyle = "#666666";
        grd = bgCon.createLinearGradient(0, 0, 0, 175);
        grd.addColorStop(0, "#EEEEEE");
        grd.addColorStop(1, "#FFFFFF");
        bgCon.fillStyle = grd;
        bgCon.lineWidth = 1;
        bgCon.beginPath();
        bgCon.fillRect(this.boundary.x, this.boundary.y, this.boundary.a, this.boundary.a);
        bgCon.strokeRect(this.boundary.x, this.boundary.y, this.boundary.a, this.boundary.a);
        bgCon.closePath();
        bgCon.stroke();
        // draw slider triangle
        bgCon.save();
        bgCon.fillStyle = "#000";
        bgCon.globalAlpha = .15;
        bgCon.beginPath();
        bgCon.moveTo(this.boundary.a + this.boundary.x + butData.butGap + butData.butHeight * .5, (this.boundary.y + this.boundary.a) - 3 * butData.butHeight + butData.butHeight * .5);
        bgCon.lineTo(this.boundary.a + this.boundary.x + butData.butGap + butData.butWidth - butData.butHeight * .5, (this.boundary.y + this.boundary.a) - 3 * butData.butHeight + butData.butHeight * .25);
        bgCon.lineTo(this.boundary.a + this.boundary.x + butData.butGap + butData.butWidth - butData.butHeight * .5, (this.boundary.y + this.boundary.a) - 3 * butData.butHeight + butData.butHeight * .75);
        bgCon.closePath();
        bgCon.fill();
        bgCon.restore();
        ctxCon.strokeStyle = "#000";
        ctxCon.lineWidth = 10 * this.config.vScale;
        // render the button
        if (this.data.changeButton==1) {
            this.changeButton.render();
        }
        // text scale
        tx = this.boundary.a + this.boundary.x + butData.butGap;
        // draw the thermometer
        gx = this.boundary.a + this.boundary.x + butData.butGap + butData.butWidth * .5;
        gy = (this.boundary.y + this.boundary.a) - butData.butHeight * 6 - butData.butGap * 7,
            this.makeThermometer(this.reactionTemperature - 273, 1200 * this.config.vScale, gx, gy);
        // reset text format after guage drawn
        fs = 200 * this.config.vScale;
        ctxCon.font = fs + "px " + OU.theme.font;
        ctxCon.fillStyle = "#666";
        ty = (gy - (850 * this.config.vScale) * .5) + butData.butHeight;
        ctxCon.fillText("°C", tx, ty);
        ctxCon.strokeStyle = "#000";
        ctxCon.lineWidth = 1;
        ty = (this.boundary.y + this.boundary.a) - 4 * butData.butHeight,
            ctxCon.fillText("temperature", tx, ty);
        // draw particle titles
        if (this.data.reactions[this.whichReaction].menuChars[0]!="") {
            ctxCon.drawImage(this.data.menu[0].image, this.boundary.a + this.boundary.x + butData.butGap, this.boundary.y, butData.butHeight, butData.butHeight);
            ctxCon.beginPath();
            ctxCon.arc(this.boundary.a + this.boundary.x + butData.butGap + butData.ballRad, this.boundary.y + butData.ballRad, butData.ballRad - 1, 0, Math.PI * 2, true);
            ctxCon.closePath();
            ctxCon.stroke();
            bx = this.boundary.a + this.boundary.x + butData.butHeight + butData.butGap;
            by = this.boundary.y - this.config.vScale * moleculeW / 2 + butData.butHeight * .5;
            new OU.util.DynText({
                txt:this.data.reactions[this.whichReaction].menuChars[0],
                x:bx,
                y:by,
                w:this.config.vScale * 6 * butWidth,
                h:this.config.vScale * moleculeW,
                context:ctxCon,
                propTextHeight:1,
                align:'left',
                fontSize:this.config.vScale * moleculeW,
                fontWeight:'normal'
            });
            ctxCon.save();
            ctxCon.textBaseline = "middle";
            by = this.boundary.y + butData.butHeight * .5;
            bx = bx + butData.butHeight * 2.5;
            ctxCon.fillText(this.proportionA + "%", bx, by);
            ctxCon.restore();
        }
        if (this.data.reactions[this.whichReaction].menuChars[1]!="") {
            ctxCon.drawImage(this.data.menu[1].image, this.boundary.a + this.boundary.x + butData.butGap, this.boundary.y + butData.butHeight + butData.butGap, butData.butHeight, butData.butHeight);
            ctxCon.beginPath();
            ctxCon.arc(this.boundary.a + this.boundary.x + butData.butGap + butData.ballRad, this.boundary.y + butData.butHeight + butData.butGap + butData.ballRad, butData.ballRad - 1, 0, Math.PI * 2, true);
            ctxCon.closePath();
            ctxCon.stroke();
            bx = this.boundary.a + this.boundary.x + butData.butHeight + butData.butGap;
            by = this.boundary.y - this.config.vScale * moleculeW / 2 + butData.butHeight * 1.5 + butData.butGap;
            new OU.util.DynText({
                txt:this.data.reactions[this.whichReaction].menuChars[1],
                x:bx,
                y:by,
                w:this.config.vScale * 6 * butWidth,
                h:this.config.vScale * moleculeW,
                context:ctxCon,
                propTextHeight:1,
                align:'left',
                fontSize:this.config.vScale * moleculeW,
                fontWeight:'normal'
            });
            ctxCon.save();
            ctxCon.textBaseline = "middle";
            by = this.boundary.y + butData.butHeight * 1.5 + butData.butGap;
            bx = bx + butData.butHeight * 2.5;
            ctxCon.fillText(this.proportionB + "%", bx, by);
            ctxCon.restore();
        }
        if (this.data.reactions[this.whichReaction].menuChars[2]!="") {
            ctxCon.drawImage(this.data.menu[2].image, this.boundary.a + this.boundary.x + butData.butGap, this.boundary.y + butData.butHeight * 2 + butData.butGap * 2, butData.butHeight, butData.butHeight);
            ctxCon.beginPath();
            ctxCon.arc(this.boundary.a + this.boundary.x + butData.butGap + butData.ballRad, this.boundary.y + butData.butHeight * 2 + butData.butGap * 2 + butData.ballRad, butData.ballRad - 1, 0, Math.PI * 2, true);
            ctxCon.closePath();
            ctxCon.stroke();
            bx = this.boundary.a + this.boundary.x + butData.butHeight + butData.butGap;
            by = this.boundary.y - this.config.vScale * moleculeW / 2 + butData.butHeight * 2.5 + butData.butGap * 2.0;
            new OU.util.DynText({
                txt:this.data.reactions[this.whichReaction].menuChars[2],
                x:bx,
                y:by,
                w:this.config.vScale * 6 * butWidth,
                h:this.config.vScale * moleculeW,
                context:ctxCon,
                propTextHeight:1,
                align:'left',
                fontSize:this.config.vScale * moleculeW,
                fontWeight:'normal'
            });
            ctxCon.save();
            ctxCon.textBaseline = "middle";
            by = this.boundary.y + butData.butHeight * 2.5 + 2 * butData.butGap;
            bx = bx + butData.butHeight * 2.5;
            ctxCon.fillText(this.proportionC + "%", bx, by);
            ctxCon.restore();
        }
        if (this.data.reactions[this.whichReaction].menuChars[3]!="") {
            ctxCon.drawImage(this.data.menu[3].image, this.boundary.a + this.boundary.x + butData.butGap, this.boundary.y + butData.butHeight * 3 + butData.butGap * 3, butData.butHeight, butData.butHeight);
            ctxCon.beginPath();
            ctxCon.arc(this.boundary.a + this.boundary.x + butData.butGap + butData.ballRad, this.boundary.y + butData.butHeight * 3 + butData.butGap * 3 + butData.ballRad, butData.ballRad - 1, 0, Math.PI * 2, true);
            ctxCon.closePath();
            ctxCon.stroke();
            bx = this.boundary.a + this.boundary.x + butData.butHeight + butData.butGap;
            by = this.boundary.y - this.config.vScale * moleculeW / 2 + butData.butHeight * 3.5 + butData.butGap * 3.0;
            new OU.util.DynText({
                txt:this.data.reactions[this.whichReaction].menuChars[3],
                x:bx,
                y:by,
                w:this.config.vScale * 6 * butWidth,
                h:this.config.vScale * moleculeW,
                context:ctxCon,
                propTextHeight:1,
                align:'left',
                fontSize:this.config.vScale * moleculeW,
                fontWeight:'normal'
            });
            ctxCon.save();
            ctxCon.textBaseline = "middle";
            by = this.boundary.y + butData.butHeight * 3.5 + 3 * butData.butGap;
            bx = bx + butData.butHeight * 2.5;
            ctxCon.fillText(this.proportionD + "%", bx, by);
            ctxCon.restore();
        }
        theWidth = this.boundary.a * .7;
        tx = this.boundary.x + this.boundary.a * .5 - theWidth / 2;
        by = this.boundary.y + this.boundary.a - 1.5 * this.config.vScale * 300;
        new OU.util.DynText({
            txt:this.data.reactions[this.whichReaction].formula,
            x:tx,
            y:by,
            w:theWidth,
            h:this.config.vScale * 300,
            context:ctxCon,
            propTextHeight:1,
            align:'center',
            fontWeight:'normal'
        });
        this.temperatureSlider.render((this.reactionTemperature - this.minTemperature) / (this.config.maxTemperature - this.minTemperature));
    };
    OU.activity.Temperaturechart.prototype.drawScreen = function () {
        var ctx = this.imageLayer.context;
        this.imageLayer.clear();
        ctx.lineWidth = 1;
        this.temperature();
        this.drawPie();
    };
    OU.activity.Temperaturechart.prototype.makePopUp = function () {
        var self = this, infoText, fm,
            currentGroup = this.data.reactions[0].group,
            currentReaction = 1 * OU.LocalStorage.load('ou.temperaturechart.temperature.reaction'),
            cellWidthTotal = this.w * .6 * .8;
        this.controlLayer.events.flush();
        infoText = '<h1>Select a Chemical Equation</h1><form name="pieform"><table width="' + cellWidthTotal + '" border="0"><tr><td width="' + cellWidthTotal * .2 + '" bgcolor="#999999" class="cepressurechartLL">' + this.data.reactions[0].group + '</td><td width="' + cellWidthTotal * .8 + '" valign="top">';
        for (fm = 0; fm < this.data.reactions.length; fm++) {
            if (currentGroup!=this.data.reactions[fm].group) {
                infoText += '</td></tr>';
                infoText += '<tr><td width="' + cellWidthTotal * .2 + '" bgcolor="#999999" class="cepressurechartLL">' + this.data.reactions[fm].group + '</td><td width="' + cellWidthTotal * .8 + '" valign="top">';
                currentGroup = this.data.reactions[fm].group;
            }
            if (fm==currentReaction) {
                infoText += '<input type="radio" name="group1" id="radiofield' + fm + '" value="' + fm + '" checked="checked" onclick="OU.LocalStorage.save(\'ou.temperaturechart.temperature.reaction\',' + fm + ')" /><label for="radiofield' + fm + '"> ' + this.data.reactions[fm].formula + '</label>';
            }
            else {
                infoText += '<input type="radio" name="group1" id="radiofield' + fm + '" value="' + fm + '" onclick="OU.LocalStorage.save(\'ou.temperaturechart.temperature.reaction\',' + fm + ')" /><label for="radiofield' + fm + '"> ' + this.data.reactions[fm].formula + '</label>';
            }
            if (fm!=this.data.reactions.length - 1) {
                infoText += '<br/>';
            }
        }
        infoText += '</td></tr>';
        infoText += '</table>';
        infoText += '</form>';
        new OU.util.PopUpInfo({
            container:this,
            txt:infoText,
            x:this.w * .20,
            y:this.h * .15,
            w:this.w * .6,
            h:this.h * .7,
            onClose:function () {
                self.whichReaction = 1 * OU.LocalStorage.load('ou.temperaturechart.temperature.reaction');
                self.minFeedback = self.data.reactions[self.whichReaction].minFeedback;
                self.minTemperature = self.data.reactions[self.whichReaction].minTemperature;
                self.reactionTemperature = self.minTemperature;
                self.oldTemperature = self.reactionTemperature;
                self.temperatureSlider.range.min = self.minTemperature - 273;
                self.temperatureSlider.render(0);
                self.drawControl();
            }
        });
    };
    //
    OU.activity.Temperaturechart.prototype.makeLowPopUp = function () {
        this.controlLayer.events.flush();
        new OU.util.PopUpInfo({
            container:this,
            txt:'<h1>Alert!</h1>' + this.minFeedback + ' if the temperature is decreased further.',
            x:this.w * .20,
            y:this.h * .15,
            w:this.w * .6,
            h:this.h * .7
        });
    };
    // make thermometer
    OU.activity.Temperaturechart.prototype.makeThermometer = function ( temperature, targetHeight, centreX, centreY ) {
        var ctx = this.controlLayer.context;
        var theHeight = 100;
        var theRadius = 15;
        var overallHeight = theHeight + 4 * theRadius;
        var scale = targetHeight / overallHeight;
        theHeight = 100 * scale;
        theRadius = 15 * scale;
        var strokeWidth = 4 * scale;
        var scaleChunk = theHeight * .25;
        var theFontSize = 14 * scale;
        var scaleLineLength = 12 * scale;
        var zeroY = centreY + (scaleChunk * .5) + scaleChunk * 3;
        var maxY = centreY + (scaleChunk * .5);
        var yRange = zeroY - maxY;
        var maxT = 1500;
        var contentY = zeroY - yRange * temperature / maxT;
        if (contentY < centreY) contentY = centreY;
        var tempTextWidth;
        // top curve
        ctx.beginPath();
        ctx.arc(centreX, centreY, theRadius, 0, (Math.PI * 2) * .5, true);
        ctx.strokeStyle = "#888";
        ctx.fillStyle = "#ccc";
        ctx.lineCap = "butt";
        ctx.lineWidth = strokeWidth;
        ctx.fill();
        ctx.stroke();
        // edges
        ctx.beginPath();
        ctx.moveTo(centreX - theRadius, centreY);
        ctx.lineTo(centreX - theRadius, centreY + theHeight);
        ctx.moveTo(centreX + theRadius, centreY);
        ctx.lineTo(centreX + theRadius, centreY + theHeight);
        ctx.stroke();
        // bottom curve
        ctx.beginPath();
        ctx.arc(centreX, centreY + theHeight, theRadius, 0, (Math.PI * 2) * .5, false);
        ctx.fillStyle = "#ff0000";
        ctx.fill();
        ctx.stroke();
        // lower circle
        ctx.beginPath();
        ctx.arc(centreX, centreY + theHeight + (theRadius * 2), theRadius, 0, Math.PI * 2, true);
        ctx.fillStyle = "#ff0000";
        ctx.fill();
        ctx.stroke();
        // red join
        ctx.beginPath();
        ctx.moveTo(centreX, centreY + theHeight + theRadius - strokeWidth);
        ctx.lineTo(centreX, centreY + theHeight + theRadius + strokeWidth);
        ctx.strokeStyle = "#ff0000";
        ctx.stroke();
        // red content
        ctx.beginPath();
        ctx.fillRect(centreX - theRadius + strokeWidth * .5, contentY, (theRadius * 2) - strokeWidth, (centreY + theHeight) - contentY);
        // scale
        ctx.beginPath();
        ctx.moveTo(centreX - theRadius - scaleLineLength, centreY + (scaleChunk * .5));
        ctx.lineTo(centreX - theRadius, centreY + (scaleChunk * .5));
        ctx.moveTo(centreX - theRadius - scaleLineLength, centreY + (scaleChunk * .5) + scaleChunk);
        ctx.lineTo(centreX - theRadius, centreY + (scaleChunk * .5) + scaleChunk);
        ctx.moveTo(centreX - theRadius - scaleLineLength, centreY + (scaleChunk * .5) + scaleChunk * 2);
        ctx.lineTo(centreX - theRadius, centreY + (scaleChunk * .5) + scaleChunk * 2);
        ctx.moveTo(centreX - theRadius - scaleLineLength, centreY + (scaleChunk * .5) + scaleChunk * 3);
        ctx.lineTo(centreX - theRadius, centreY + (scaleChunk * .5) + scaleChunk * 3);
        ctx.strokeStyle = "#888";
        ctx.lineCap = "round";
        ctx.stroke();
        // grey background
        ctx.beginPath();
        ctx.fillStyle = "#cccccc";
        ctx.fillRect(centreX - theRadius + strokeWidth * .5, centreY, (theRadius * 2) - strokeWidth, theHeight);
        // red content
        ctx.beginPath();
        ctx.fillStyle = "#ff0000";
        ctx.fillRect(centreX - theRadius + strokeWidth * .5, contentY, (theRadius * 2) - strokeWidth, (centreY + theHeight) - contentY);
        // scale text
        ctx.fillStyle = "#888";
        ctx.font = theFontSize + "px " + OU.theme.font;
        tempTextWidth = ctx.measureText("1500");
        ctx.fillText("1500", (centreX - theRadius - scaleLineLength) - tempTextWidth.width - strokeWidth, centreY + (theFontSize - strokeWidth) * .5 + (scaleChunk * .5));
        tempTextWidth = ctx.measureText("1000");
        ctx.fillText("1000", (centreX - theRadius - scaleLineLength) - tempTextWidth.width - strokeWidth, centreY + (theFontSize - strokeWidth) * .5 + (scaleChunk * .5) + scaleChunk);
        tempTextWidth = ctx.measureText("500");
        ctx.fillText("500", (centreX - theRadius - scaleLineLength) - tempTextWidth.width - strokeWidth, centreY + (theFontSize - strokeWidth) * .5 + (scaleChunk * .5) + scaleChunk * 2);
        tempTextWidth = ctx.measureText("0");
        ctx.fillText("0", (centreX - theRadius - scaleLineLength) - tempTextWidth.width - strokeWidth, centreY + (theFontSize - strokeWidth) * .5 + (scaleChunk * .5) + scaleChunk * 3);
    };
    ////////////////////
    // Class for 'Effect of pressure' section of S104 Chemical Equilibrium program
    OU.activity.Temperaturechart.prototype.temperature = function () {
        //---------------------------------------------------------
        //declare and instantiate variables
        //---------------------------------------------------------
        var R = 8.31441;
        //---------------------------------------------------------
        //main method for calculating equilibrium quantities of reactants and products
        //at different temperatures
        var propA, propB, propC, propD;
        var partialPressure = new Array();
        var maxPartialPressure = new Array();
        var estimatedK, factor, iterations, totalPartialPressure;
        var reactions = this.data.reactions[this.whichReaction];
        var K = Math.exp(-1000 / (R * this.reactionTemperature) * reactions.deltaH) * Math.exp(reactions.deltaS / R);
        //calculate partial pressures
        maxPartialPressure[0] = reactions.stoichiometry[0] / (reactions.stoichiometry[0] + reactions.stoichiometry[1]) * this.totalPressure;
        maxPartialPressure[1] = reactions.stoichiometry[1] / (reactions.stoichiometry[0] + reactions.stoichiometry[1]) * this.totalPressure;
        maxPartialPressure[2] = reactions.stoichiometry[2] / (reactions.stoichiometry[2] + reactions.stoichiometry[3]) * this.totalPressure;
        maxPartialPressure[3] = reactions.stoichiometry[3] / (reactions.stoichiometry[2] + reactions.stoichiometry[3]) * this.totalPressure;
        //
        partialPressure[0] = 0.01 * maxPartialPressure[0];
        partialPressure[1] = 0.01 * maxPartialPressure[1];
        partialPressure[2] = 0.99 * maxPartialPressure[2];
        partialPressure[3] = 0.99 * maxPartialPressure[3];
        iterations = 0;
        //
        // Have to use a numerical method rather than solving directly for unknowns.
        // Use estimated value of K and iterate until within a specified tolerance.
        estimatedK = (Math.pow(partialPressure[2], reactions.stoichiometry[2]) * Math.pow(partialPressure[3], reactions.stoichiometry[3])) / (Math.pow(partialPressure[0], reactions.stoichiometry[0]) * Math.pow(partialPressure[1], reactions.stoichiometry[1]));
        //
        while (Math.abs(1 - estimatedK / K) > 0.0005 && iterations < 1000) {
            factor = Math.sqrt(Math.sqrt(estimatedK / K));
            partialPressure[0] = partialPressure[0] * factor;
            partialPressure[1] = partialPressure[1] * factor;
            partialPressure[2] = partialPressure[2] / factor;
            partialPressure[3] = partialPressure[3] / factor;
            //
            estimatedK = (Math.pow(partialPressure[2], reactions.stoichiometry[2]) * Math.pow(partialPressure[3], reactions.stoichiometry[3])) / (Math.pow(partialPressure[0], reactions.stoichiometry[0]) * Math.pow(partialPressure[1], reactions.stoichiometry[1]));
            iterations++;
        }
        // =================================================
        // // Proportion tests for pie chart amounts:
        // The total pressure:
        totalPartialPressure = partialPressure[0] + partialPressure[1] + partialPressure[2] + partialPressure[3];
        // Proportion of total pressure for A
        propA = partialPressure[0] / totalPartialPressure;
        // Proportion of total pressure for B
        propB = partialPressure[1] / totalPartialPressure;
        // Proportion of total pressure for C
        propC = partialPressure[2] / totalPartialPressure;
        // Proportion of total pressure for D
        propD = partialPressure[3] / totalPartialPressure;
        // Pie chart angle size for each pressure
        this.config.myData[0] = 360 * propA;
        this.config.myData[1] = 360 * propB;
        this.config.myData[2] = 360 * propC;
        this.config.myData[3] = 360 * propD;
        // =================================================
        propA = 100 * propA;
        propB = 100 * propB;
        propC = 100 * propC;
        propD = 100 - (propA + propB + propC);
        this.proportionA = propA.toFixed(2);
        this.proportionB = propB.toFixed(2);
        this.proportionC = propC.toFixed(2);
        this.proportionD = propD.toFixed(2);
    };
    OU.activity.Temperaturechart.prototype.step = function ( n ) { // called by the controller
        if (this.config.step!=n) {
            this.config.step = n;
            if (this.config.step==1) {
                this.config.stepCompleted = 1;
                this.changeButton.disabled = false;
                this.changeButton.render();
                this.drawControl();
            }
        }
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.Temperaturechart, OU.util.Activity);
