/**
 * @fileOverview visual angle activity
 *
 * @author Fiona Williams
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.Layer');
OU.require('OU.util.PopUpInfo');
OU.require('OU.util.Slider');
/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.VisualAngle = function ( data, instance, controller ) { //data and instance are passed in from index.html
    OU.activity.VisualAngle.prototype.canvasView = function () {
        var self = this;
        var bH = OU.controlHeight;
        
	this.showCycles = this.data.showCycles;
	this.numSliders = this.showCycles ? 3 : 2;
	
	// create Canvas Layers & Contexts
	
	this.lineCol = "#000";
	 if (this.data.graphLineColour !== undefined)
	 {
	     this.lineCol = this.data.graphLineColour; 
	 }
	 
	 this.backCol = "#FFF";
	 if (this.data.backgroundColour !== undefined)
	 {
	     this.backCol = this.data.backgroundColour;
	 }
	
        this.bgLayer = new OU.util.Layer({
            container:this, //container is the whole area and can contain other stuff
            id:'bg'
        });
	
        this.graphLayer = new OU.util.Layer({
            container:this,
            id:'graphLayer',
            x:this.w*0.05,
            y:2*bH + 0.02*this.h,
            w:this.w*0.8,
            h:this.h*0.4,
            hasEvents:true,
            pinch:this.pinch,
            pinchMe:this
        });
		

        this.controlsLayer = new OU.util.Layer({
            container:this,
            id:'controlsLayer',
            x:this.w*0.02,
            y:this.graphLayer.y+this.graphLayer.h,
            w:this.w*0.35,
            h:this.h*0.5,
            hasEvents:true,
            pinch:this.pinch,
            pinchMe:this
        });
	
        this.controlsLayer.context.gradRect(); // use default background

        this.textDiv = new OU.util.Div({
            x:this.controlsLayer.x+this.controlsLayer.w+this.w*0.01,
            y:this.controlsLayer.y,
            w:this.w*0.6,
            h:this.h*0.4,
            container:this
        });
        this.textDiv.div.innerHTML = this.data.instructions;		

        this.initControls();

// wait for image to load and then call resize(), which in turn calls render()	
	if (this.data.imageEye !== undefined)
	{
	    this.eyeImage  = new Image();
	    this.eyeImage.src = this.data.imageEye;
	    this.eyeImage.addEventListener("load", function() { self.resize();}, false);
	}
	else
	{
	    this.resize();	    
	}
	        
    };


    OU.activity.VisualAngle.prototype.initControls = function () {
        var ctx = this.controlsLayer.context, self = this,
        clickable = this.controlsLayer.events.clickable, cH = this.controlsLayer.h, cW = this.controlsLayer.w;
			
	this.sliders = new Array(this.numSliders);
        this.sliderLabels = ["Distance, D","Height, S","No. of cycles, n"];
        this.sliderIDs = ["distSlider","heightSlider","cyclesSlider"];
	this.sliderUnits = [" cm", " cm", ""];
	this.sliderMin = [10, 1.5, 0.5];
	this.sliderMax = [210, 10, 6.0];
	this.numDecimals = [0, 1, 1];
        for(i = 0; i < this.numSliders; i++){
            this.sliders[i] = new OU.util.Slider({
                container:this.controlsLayer,
                instance:'',
                drawContainer:false,
		//sliderHeight:OU.controlHeight*2,
                title:this.sliderLabels[i]+"",
                titlePosition:"above",
		valueUnits:this.sliderUnits[i],
                showValue:true,
		range:{
                    min:this.sliderMin[i],
                    max:this.sliderMax[i],
                    nDecimals:this.numDecimals[i]
                },
                callback:self.setSliders,
                callbackParam:{
                    i:this.sliderIDs[i],
                    self:this
                }, //identifies which slider
                background:{
                    clear:true
                },
                context:ctx
            });
            clickable.push(this.sliders[i]);
        }
	
	this.sliders[0].sliderPos = 0.5;
        this.sliders[1].sliderPos = 0.611;
	if (this.showCycles) this.sliders[2].sliderPos = 0.636;	
	

        this.helpBtn = new OU.util.Button({
            txt:"Help",
            padding:0,
            verticalPadding:0,
            layer:this.controlsLayer,
            onClick:function () {
                if (!self.helpDisp) {
                    self.helpDisp = true;
                    self.showHelp();
                }
            }
        });
 
    };

    OU.activity.VisualAngle.prototype.showHelp = function () {
        var self = this;
        new OU.util.PopUpInfo({
            container:this,
            txt:this.data.help,
            x:this.w*0.17,
            y:this.h*0.17,
            w:this.w*0.66,
            h:this.h*0.66,
            onClose:function () {
                self.helpDisp = false;
            }
        });
    };

    /**
     * The main renderer.
     */
    OU.activity.VisualAngle.prototype.render = function () {
	this.helpBtn.render();	
	 for(i = this.sliders.length; i--;){
            this.sliders[i].render();
        }
        this.drawGraphs();
    };

    OU.activity.VisualAngle.prototype.resize = function () {
        OU.activity.VisualAngle.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight, ctxS = this.controlsLayer.context, cH = this.controlsLayer.h,
        cW = this.controlsLayer.w;
	
        this.bgLayer.resize();
	
	this.graphLayer.resize({
            x:this.w*0.05,
            y:2*bH + 0.02*this.h,
            w:this.w*0.8,
            h:this.h*0.4
        });

        this.controlsLayer.resize({
            x:this.w*0.02,
            y:this.graphLayer.y+this.graphLayer.h + 0.05*this.h,
            w:this.w*0.35,
            h:this.h*0.5
        });
	
        if (this.data.backgroundColour!==undefined)
	// use specified background colour
	{
            this.bgLayer.context.gradRect({
                col1:this.data.backgroundColour,
                col2:this.data.backgroundColour
            });
	    
	    ctxS.gradRect({
                col1:this.data.backgroundColour,
                col2:this.data.backgroundColour
            });	    	    
	}
	else
        {
	    this.bgLayer.context.gradRect(); // use default background
	    ctxS.gradRect();
	}
     
	for(i = 0; i < this.sliders.length; i++){
            this.sliders[i].resize({
           //     container:this.controlsLayer,
           //     instance:'',
                x:0,
                y:0.1*cH + i*0.25*cH,
                w:cW*0.7,
                h:0.15*cH
            });
        }
 
        this.helpBtn.resize({
            y:cH*0.8, //this.sliders[2].y + this.sliders[2].h + cH*0.05,
            h:cH*0.15,
            w:cW*0.2,
            x:cW*0.75
        });
	
        this.textDiv.resize({
            x:this.controlsLayer.x+this.controlsLayer.w+this.w*0.01,
            y:this.controlsLayer.y,
            w:this.w*0.6,
            h:this.h*0.4
        });
	
		
        this.render();
        
    };
	
    OU.activity.VisualAngle.prototype.drawGraphs = function () {
        var ctx = this.graphLayer.context;
        var w = this.graphLayer.w;
        var h = this.graphLayer.h;
	// scale relative to h to keep aspect ratio correct
	var origx = h*0.75;
	var origy = h*0.5;
	// offsetx is the x position (w.r.t. origx) for the minimum D - i.e. diagram is not to scale - eye section is magnified
	var offsetx = h*0.6;
	var R = 0.55*h;      // the distance from the lens to the retina
	var xscale = (w - 1.35*h)/300;    // xscale is such that [w - (origx + offsetx)] represents 250
	var pixScale = h/165;    // image height is 165 pixels so this is scale of pixels
	
	ctx.fillStyle = this.backCol;
        ctx.fillRect(0, 0, w, h);
	ctx.strokeStyle = "#000";
	ctx.lineWidth = "1";
	ctx.fillStyle = "#000";
	
	// draw the background image if one is specified
	if (this.data.imageEye !== undefined)
	{
	    ctx.drawImage(this.eyeImage, 0, 0, h, h);	  
	}
	
	var minD = this.sliders[0].range.min;
	var maxD = this.sliders[0].range.max;
	var D = Math.round(minD + this.sliders[0].sliderPos*(maxD - minD));    // round to nearest integer otherwise get rounding discrepancy in calculations
	var minS = this.sliders[1].range.min;
	var maxS = this.sliders[1].range.max;	
	var S = minS + this.sliders[1].sliderPos*(maxS - minS);
		
	var xPic = offsetx + xscale*D;        
	var yPic = 0.5*S*pixScale*14;
	var yRetina = R*yPic/xPic;
	var alpha = Math.atan(S/(2*D)) * (360/Math.PI);    // 2*atan(S/(2D)) converted to degrees by 360/2Pi factor
	
	// update text elements in textDiv
	document.getElementById("S").innerHTML = (S/100).toPrecision(2);
	document.getElementById("2D").innerHTML = (2*D/100).toPrecision(3);
	var alphas = document.getElementsByClassName("alpha");
	for (i = alphas.length; i--;)
	{
	    alphas[i].innerHTML = alpha.toPrecision(3);
	}

	// draw the lines that show the visual angle
	ctx.beginPath();
	ctx.moveTo(origx - R, origy + yRetina);
	ctx.lineTo(origx + xPic, origy - yPic);
	ctx.moveTo(origx - R, origy - yRetina);
	ctx.lineTo(origx + xPic, origy + yPic);
	
	//draw the arc to show the angle alpha
	var screenAlpha = Math.atan(yPic/xPic);
	var radiusArc = 0.8*offsetx;
	ctx.moveTo(origx + radiusArc*Math.cos(screenAlpha), origy - radiusArc*Math.sin(screenAlpha));
	ctx.arc(origx, origy, 0.8*offsetx, -screenAlpha, screenAlpha, false);
	ctx.font = "16px serif";
	ctx.fillText("\u03B1", origx + radiusArc + 2, origy);

	// draw the arrow to show D
	ctx.moveTo(origx, 0.95*h);
	ctx.lineTo(origx + 6*pixScale, 0.95*h - 5*pixScale);
	ctx.moveTo(origx, 0.95*h);
	ctx.lineTo(origx + 6*pixScale, 0.95*h + 5*pixScale);
	ctx.moveTo(origx, 0.95*h);
	ctx.lineTo(origx + xPic, 0.95*h);
	ctx.moveTo(origx + xPic, 0.95*h);
	ctx.lineTo(origx + xPic - 6*pixScale, 0.95*h - 5*pixScale);
	ctx.moveTo(origx + xPic, 0.95*h);
	ctx.lineTo(origx + xPic - 6*pixScale, 0.95*h + 5*pixScale);
	ctx.stroke();
	ctx.font = "12px verdana,sans-serif";
	ctx.fillText("D", origx + xPic/2, 0.91*h);
	
	// draw the red arrow
	ctx.beginPath();
	ctx.strokeStyle = "#F00";
	ctx.lineWidth = "2";
	ctx.moveTo(origx - R, origy - yRetina + 1);
	ctx.lineTo(origx - R, origy + yRetina - 1);
	ctx.moveTo(origx - R, origy + yRetina - 1);
	ctx.lineTo(origx - R - 5*pixScale, origy + yRetina - 6*pixScale);
	ctx.moveTo(origx - R, origy + yRetina - 1);
	ctx.lineTo(origx - R + 5*pixScale, origy + yRetina - 6*pixScale);
	ctx.stroke();
	
	// draw the arrow to show S, and the diagram to show the number of cycles, if required		
	var arrowPos;
	var dW = xscale*40;    // width of the cycles display
	var dH = yPic*2;
	if (this.showCycles)
	{
	    var minN = this.sliders[2].range.min;
	    var maxN =  this.sliders[2].range.max;
	    var nCycles = minN + this.sliders[2].sliderPos*(maxN - minN);
	    var sFreq = nCycles/alpha;
	    document.getElementById("ncycles").innerHTML = nCycles.toFixed(1);
	    document.getElementById("spatial").innerHTML = sFreq.toPrecision(3);
	    
	    arrowPos = dW + 8;	    
	    // draw the diagram to show the number of cycles	
	    // draw a grey border around the display
	    ctx.strokeStyle = "rgb(162,162,162)";
	    ctx.lineWidth = "1";
	    ctx.strokeRect(origx + xPic, origy - yPic, dW, dH);
	    
	    var stripWidth = 0.5*dH/nCycles;
	    ctx.fillStyle = "rgb(0,0,0)";
	    var stripCount = 0;
	    for (var i = 0; i < dH; i += stripWidth)
	    {
		if ( stripCount % 2 < 1e-6)
		{
		    ctx.fillStyle = "#FFF";
		}
		else
		{
		    ctx.fillStyle = "#000";
		}
		
		var endPt = i + stripWidth;
		if (endPt > dH) endPt = dH;
		var wid = endPt - i;
		ctx.fillRect(origx + xPic, origy - yPic + i, dW, wid);
		stripCount++;
	    }	    
	    
	}
	else
	{	  
	    arrowPos = 4;
	}
	
	// draw the arrow to show S
	ctx.beginPath();
	ctx.strokeStyle = "#000";
	ctx.fillStyle = "rgb(0,0,0)";
	ctx.lineWidth = "2";
	ctx.moveTo(origx + xPic + arrowPos, origy + yPic);
	ctx.lineTo(origx + xPic + arrowPos, origy - yPic);
	ctx.moveTo(origx + xPic + arrowPos, origy - yPic);
	ctx.lineTo(origx + xPic + arrowPos - 5*pixScale, origy - yPic + 6*pixScale);
	ctx.moveTo(origx + xPic + arrowPos, origy - yPic);
	ctx.lineTo(origx + xPic + arrowPos + 5*pixScale, origy - yPic + 6*pixScale);
	ctx.stroke();
	ctx.font = "12px verdana,sans-serif";
	ctx.fillText("S", origx + xPic + arrowPos + 4, origy);
		
		
    };

    
    OU.activity.VisualAngle.prototype.renderSliders = function () {
        //render the slider
        var i, self = this;

        this.controlsLayer.clear(0,0,this.controlsLayer.w,this.controlsLayer.h);
        for(i = this.sliders.length; i--;){
            this.sliders[i].render();
        }       
        self.helpBtn.render();
    };
    
    OU.activity.VisualAngle.prototype.setSliders = function ( p, v ) { // Called when slider is moved
	v.self.renderSliders();
        v.self.drawGraphs();	
    };
   

    /*OU.activity.Cafewall.prototype.pinch = function ( e, s, x, y, dx, dy, me ) { // called when pinch event detected
        var ns = ((e - s) / me.h) + me.tileView.s;
        ns = ns > 1 ? 1 : (ns < 0 ? 0 : ns);
        me.tileView.scale(ns, {
            x:x,
            y:y
        });
        me.zoomSlider.sliderPos = ns;
        me.zoomSlider.render();
    };//*/
    /**
     * @class
     */
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.VisualAngle, OU.util.Activity);
