/**
 * @fileOverview LayerZoom activity
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.DynText');
OU.require('OU.util.Transitions');
OU.require('OU.util.NavButtons');
OU.require('OU.util.Slider');
OU.require('OU.util.Layer');
OU.require('OU.util.ImageLoader');
OU.require('OU.util.TileViewer');
/** 
 * @class
 * @extends OU.util.Activity
 */
OU.activity.LayerZoom = function(data,instance,controller) {

    OU.activity.LayerZoom.prototype.canvasView = function() {
        var bH = OU.controlHeight,self=this;

        this.config = {
            imageLoadTimeout:10000, // allow max 10 seconds to load images
            useNavButtons:true,

            minZoom: 0,
            maxZoom: 1,

            // Settings for Controls
            overlayControls:false, // set to false to seperate controls from images

            // animation settings
            autoScroll:false,
            fps:40 // 40ms = 25 fps
        };

        this.bgLayer = new OU.util.Layer({
            container:this
        });
        this.imageLayer = new OU.util.Layer({
            container:this,
            hasEvents: true
        });
        this.controlLayer = new OU.util.Layer({
            container:this,
            y: this.h-bH,
            h: bH,
            hasEvents: true,
            id: 'controls'
        });

        this.bgLayer.context.gradRect(); // draw background on backdrop layer

        if(this.data.hotspotThreshold!==undefined)
            this.renderThreshold = this.data.hotspotThreshold/100;
        else
            this.renderThreshold = 0.5;

        // determine Proportions
        this.config.maxWidth = this.w;
        if(this.config.overlayControls)
            this.config.layerHeight = this.h;
        else
            this.config.layerHeight = this.h-bH;
        this.config.maxHeight= this.config.layerHeight;
        this.layerNumber=0;
        this.inTrans=false;

        // initialise transitions
        this.transitions = new OU.util.Transitions(this.imageLayer.canvas,this.imageLayer.context);

        this.imageLayer.events.clickable.push(this);

        this.renderControls();

        this.imageLoader = new OU.util.ImageLoader({
            container: this,
            data: this.data.layers,
            onLoad: function() {
                self.initLayers();
            }
        });
    };
    OU.activity.LayerZoom.prototype.resize = function() {
        OU.activity.LayerZoom.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight,self=this,
        ctx = this.controlLayer.context,
        clickable=this.controlLayer.events.clickable;

        this.bgLayer.resize();
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        this.imageLayer.resize();
        this.controlLayer.resize({
            y: this.h-bH,
            h: bH
        });

        clickable.length=0;
        // Add nav buttons
        this.navButtons = new OU.util.NavButtons({
            x:this.w/2,
            y:0,
            w:bH,
            h:bH,
            bw:bH,
            pad:bH*.1,
            context:ctx,
            leftFunc:function() {
                self.prevLayer();
            },
            rightOn: false
        });
        clickable.push(this.navButtons);

        // Add the zoom Slider
        this.zoomSlider.resize({
            x:this.w/2+bH,
            y:0,
            w:this.w/2-bH,
            h:bH
        });
        clickable.push(this.zoomSlider);

        this.resetLayer(this.data.layers);
        this.initLayer(this.data.layers);

        this.layerTitle();
        this.zoomSlider.render(this.currentLayer.scaleFactor);
        this.renderLayer({
            layer:this.currentLayer
        });
    };
    OU.activity.LayerZoom.prototype.renderControls = function() {
        var bH = OU.controlHeight,self=this,
        ctx = this.controlLayer.context,
        events = this.controlLayer.events;

        // Add nav buttons
        this.navButtons = new OU.util.NavButtons({
            x:this.w/2,
            y:0,
            w:bH,
            h:bH,
            bw:bH,
            pad:bH*.1,
            context:ctx,
            leftFunc:function() {
                self.prevLayer();
            },
            rightOn: false
        });
        this.controlLayer.events.clickable.push(this.navButtons);

        // Add the zoom Slider
        this.zoomSlider = new OU.util.Slider({
            container:this,
            instance: 'zoom'+this.instance,
            x:this.w/2+bH,
            y:0,
            w:this.w/2-bH,
            h:bH,
            sliderHeight: bH/2,
            drawContainer:false,
            callback: this.setZoom,
            callbackParam: this,
            events: events,
            context: ctx,
            frames: 10,
            background: {
                clear: true,
                RGB: '255,255,255',
                alpha: 0.5,
                radius: bH/2,
                padding: bH/10
            }
        });
        this.controlLayer.events.clickable.push(this.zoomSlider);
    };
    OU.activity.LayerZoom.prototype.isHit = function(x,y) {
        var events = this.imageLayer.events,self=this,
        pressed = events.touched || events.pressed,
        zoomStep = 0.05,dx,dy,newZoom,now = new Date().getTime();

        if(!pressed) {
            this.midDrag=false;
            return;
        }

        if(!this.midDrag) {
            this.dragStart = {
                'x':x,
                'y':y
            };
            this.midDrag=true;
            this.dragStartTime = new Date().getTime();
            this.zoomSlider.halted=true;
        }
        else {
            dx = (x-this.dragStart.x);
            dy = (y-this.dragStart.y);
            if(Math.abs(dx)>5 || Math.abs(dy)>5) {
                this.currentLayer.x += dx;
                this.currentLayer.y += dy;
                this.dragStart = {
                    'x':x,
                    'y':y
                };
                this.dragStartTime = new Date().getTime();
                this.renderLayer({
                    layer:this.currentLayer
                });
                return;
            }
            else if(now-this.dragStartTime > 200) {
                newZoom = this.currentLayer.scaleFactor+zoomStep;
                if(newZoom>1) {
                    newZoom=1;
                }
                this.setZoom(newZoom,this);
            }
        }
        setTimeout(function() { // Cycle unless mouse/touch ended, as Events doesn't fire event for non-movement'
            self.isHit(x,y);
        },40);
    };
    OU.activity.LayerZoom.prototype.layerTitle = function() {
        var layer = this.currentLayer,bH=OU.controlHeight;
        new OU.util.DynText({
            txt:layer.label,
            x:-this.h*.025,
            y:bH*.1,
            w:this.w/2+this.h*.025,
            h:bH*.8,
            background: {
                clear:true,
                RGB: '255,255,255',
                alpha: 0.5,
                radius: this.h*.025
            },
            context:this.controlLayer.context
        });

        if(layer.parentLayer!==undefined)
            this.navButtons.leftOn=true;
        else
            this.navButtons.leftOn=false;
        this.navButtons.render();
    };
    OU.activity.LayerZoom.prototype.biggestRadius = function(params) {

        var x = params.x || this.imageLayer.canvas.width/2,
        y = params.y || this.imageLayer.canvas.height/2,
        w = params.w || this.imageLayer.canvas.width,
        h = params.h || this.imageLayer.canvas.height;

        var bigDist = OU.distance2D({
            x2: x,
            y2: y,
            x1: 0,
            y1: 0
        });
        var bL = OU.distance2D({
            x2: x,
            y2: y,
            x1: 0,
            y1: h
        });
        var bR = OU.distance2D({
            x2: x,
            y2: y,
            x1: w,
            y1: h
        });
        var tR = OU.distance2D({
            x2: x,
            y2: y,
            x1: w,
            y1: 0
        });
        if(bL>bigDist)
            bigDist=bL;
        if(bR>bigDist)
            bigDist=bR;
        if(tR>bigDist)
            bigDist=tR;
        return bigDist;
    };
    OU.activity.LayerZoom.prototype.holeOut = function() {
        if(this.holeRadius>this.bigRadius) {
            this.layerTitle();
            this.tempLayer.remove();
            this.inTrans=false;
            return;
        }
        var ctx = this.tempLayer.context,self=this;
        this.holeRadius = this.holeRadius*1.3;

        ctx.beginPath();
        ctx.arc(this.holeX,this.holeY,this.holeRadius,0,Math.PI * 2,false);
        ctx.closePath();
        ctx.fill();
        ctx.save();
        ctx.globalCompositeOperation = 'destination-out';
        if(this.holeRadius>10) {
            ctx.beginPath();
            ctx.arc(this.holeX,this.holeY,this.holeRadius*.9,0,Math.PI * 2,false);
            ctx.closePath();
        }
        ctx.fill();
        ctx.restore();
        setTimeout(function() {
            self.holeOut()
        },40);
    };
    OU.activity.LayerZoom.prototype.layerDown = function(newLayer) {
        if(this.inTrans)
            return;
        this.inTrans=true;
        var events = this.imageLayer.events;
        events.pressed=false;
        events.touched=false;
        this.zoomSlider.halted=true;

        this.holeX = events.x;
        this.holeY = events.y;

        this.bigRadius = this.biggestRadius({
            x:events.x,
            y:events.y
        });
        this.holeRadius=5;

        var layer = this.currentLayer;
        this.tempLayer = new OU.util.Layer({
            container:this,
            beforeDiv:this.instance+'controls'
        });
        this.tempLayer.context.drawImage(layer.image,layer.x,layer.y,layer.w,layer.h); // render image
        this.tempLayer.context.fillStyle='#fff';

        this.initLayer(newLayer);
        this.zoomSlider.render(newLayer.scaleFactor);
        this.renderLayer({
            layer:newLayer
        });

        this.holeOut(); // do the hole
    };
    OU.activity.LayerZoom.prototype.layerUp = function(newLayer) {
        if(this.inTrans)
            return;
        this.inTrans=true;
        if(newLayer.fileName===undefined || newLayer.fileName=='')
            return;
        var events = this.imageLayer.events,zoom=this;
        events.pressed=false;
        events.touched=false;
        this.zoomSlider.halted=true;

        this.transitions.fade(function() {
            zoom.initLayer(newLayer,true);
            zoom.zoomSlider.render(newLayer.scaleFactor);
            zoom.renderLayer({
                layer:newLayer
            });
            zoom.layerTitle();
            zoom.inTrans=false;
        } );
    };
    OU.activity.LayerZoom.prototype.prevLayer = function() {
        this.zoomSlider.setTarget(0);
    };
    OU.activity.LayerZoom.prototype.setZoom = function(zoomPerc,zoom) {
        var events = zoom.imageLayer.events;
        var layer = zoom.currentLayer;
        if(layer.parentLayer!== undefined && zoomPerc<.1) {
            zoom.layerUp(layer.parentLayer);
            return;
        }
        var dZoom = zoomPerc/layer.scaleFactor;
        layer.scaleFactor=zoomPerc;

        var newW = layer.origW * zoomPerc;
        var newH = layer.origH * zoomPerc;

        var viewportCentreX = zoom.w/2;
        var viewportCentreY  = zoom.h/2;

        if(events.touched || events.pressed) {
            viewportCentreX = events.x;
            viewportCentreY = events.y;
        }

        var zoomOffsetX = viewportCentreX - layer.x;
        var zoomOffsetY = viewportCentreY - layer.y;

        var dX = zoomOffsetX - (zoomOffsetX * dZoom);
        var dY = zoomOffsetY - (zoomOffsetY * dZoom);

        layer.x += dX;
        layer.y += dY;
        layer.w = newW;
        layer.h = newH;

        zoom.zoomSlider.render(layer.scaleFactor);
        zoom.renderLayer({
            layer:layer
        });
    };
    OU.activity.LayerZoom.prototype.resetLayer = function(layer) {
        layer.w = layer.origW;
        layer.h = layer.origH;
        layer.origW=layer.origH=undefined;
        if(layer.layers!==undefined) {
            for(var i=0; i<layer.layers.length; i++) {
                this.resetLayer(layer.layers[i]);
            }
        }
    };
    OU.activity.LayerZoom.prototype.initLayer = function(layer) { // Initialise the layer's image size - recurse through the full data set'
        // Resize the image to fit canvas

        var recurse=false;
        if(layer.origW===undefined) { // if this is the first init, set up orig values and switch on recursion
            layer.origW = layer.w;
            layer.origH = layer.h;
            recurse=true;
        }
        else {
            layer.w = layer.origW;
            layer.h = layer.origH;
        }
        // calc metrics
        var minWsF = this.w/layer.w;
        var minHsF = this.h/layer.h;
        var e100sF; // effective 100% scaleFactor
        if(minWsF>minHsF)
            e100sF = minWsF;
        else
            e100sF = minHsF;

        e100sF = e100sF>1?1:e100sF; // ensure initial setting is not greater than 100%

        layer.w = layer.origW*e100sF;
        layer.h = layer.origH*e100sF;
        layer.scaleFactor = layer.w/layer.origW;
        layer.x = (this.w-layer.w)/2;
        layer.y = (this.h-layer.h)/2;

        if(layer.layers!==undefined && recurse) {
            for(var i=0; i<layer.layers.length; i++) {
                layer.layers[i].parentLayer = layer;
                this.initLayer(layer.layers[i]);
            }
        }
    };
    OU.activity.LayerZoom.prototype.renderLayer = function(params) {
        var layer = params.layer;
        this.currentLayer = layer;

        var ctx = this.imageLayer.context;

        ctx.clearRect(0,0,this.w,this.h); // clear old image
        ctx.drawImage(layer.image,layer.x,layer.y,layer.w,layer.h); // render image

        // Render Hotspots
        if(layer.layers!== undefined && layer.scaleFactor>this.renderThreshold) {
            var hs = [];
            for(var i=0; i<layer.layers.length; i++) {
                hs[i] = new this.Hotspot(layer.layers[i],this);
            }
            var events = this.imageLayer.events;
            if((events.pressed || events.touched) && layer.scaleFactor>0.98) {
                for(i=0; i<layer.layers.length; i++) {
                    hs[i].isHit(events.x,events.y);
                }
            }
        }
    };
    OU.activity.LayerZoom.prototype.Hotspot = function(layer,zoom) {
        this.zoom=zoom;
        this.x = zoom.currentLayer.x + (layer.hotspot.x*zoom.currentLayer.scaleFactor);
        this.y = zoom.currentLayer.y + (layer.hotspot.y*zoom.currentLayer.scaleFactor);
        this.radius = layer.hotspot.radius*zoom.currentLayer.scaleFactor;
        this.layer = layer;
        this.showHotspot = layer.hotspot.showGuide===undefined?true:layer.hotspot.showGuide;

        OU.activity.LayerZoom.prototype.Hotspot.prototype.render = function() {
            if(this.showHotspot) {
                var ctx = this.zoom.imageLayer.context,c=Math.PI*2;
                ctx.save();
                ctx.fillStyle='#fff';
                ctx.beginPath();
                ctx.arc(this.x,this.y,10,0,c,false);
                ctx.closePath();
                ctx.fill();
                ctx.fillStyle='#c00';
                ctx.beginPath();
                ctx.arc(this.x,this.y,5,0,c,false);
                ctx.closePath();
                ctx.fill();
                ctx.strokeStyle='#444';
                ctx.beginPath();
                ctx.arc(this.x,this.y,this.radius,0,c,false);
                ctx.closePath();
                ctx.stroke();
                ctx.restore();
                this.HotspotLabel = new OU.util.DynText({
                    txt:this.layer.hotspot.label,
                    x:this.x-(this.radius*.75),
                    y:this.y+(this.radius/4),
                    w:this.radius*1.5,
                    h:this.radius/4,
                    context:ctx,
                    background: {
                        alpha:0.5,
                        RGB:'255,255,255',
                        shadow:true
                    },
                    fontFamily:'tahoma,arial,sans',
                    fontWeight:'bold'
                });
            }
        };
        OU.activity.LayerZoom.prototype.Hotspot.prototype.isHit = function(x,y) {
            var isHit=false;
            var dx = x-this.x;
            var dy = y-this.y;
            if(Math.sqrt((dx*dx)+(dy*dy))<this.radius)
                isHit=true; // mouse/touch is within circle

            if(isHit) {
                this.zoom.layerDown(this.layer);
            }
        };
        this.render();
    };
    OU.activity.LayerZoom.prototype.initLayers = function() {
        var self=this;
        this.initLayer(this.data.layers);

        this.currentLayer = this.data.layers;
        this.layerTitle();
        this.zoomSlider.render(this.currentLayer.scaleFactor);
        this.transitions.fade(function() {
            self.renderLayer({
                layer:self.currentLayer
            });
        });
    };
    OU.activity.LayerZoom.prototype.accessibleLayerInfo = function(layer,depth) {
        var h ='<h2>Layer '+depth+': '+layer.label+'</h2>';

        if(layer.layers!==undefined) {
            h += "<strong>This layer contains:</strong><div style='padding-left: 30px'>";
            for(var i=0; i<layer.layers.length; i++) {
                h += this.accessibleLayerInfo(layer.layers[i],depth+1);
            }
            h += "</div>";
        }
        return h;
    };
    OU.activity.LayerZoom.prototype.accessibleView = function() {
        var h = '<div id="accessibleView">';
        h += '<h1>'+this.data.title+'</h1>';
        h += '<p>'+this.data.description+'</p>';
        h += this.accessibleLayerInfo(this.data.layers,1);

        h += '</div>';
        document.body.innerHTML='';
        var accessible = document.createElement('div');
        accessible.innerHTML = h;
        document.body.appendChild(accessible);
    };
    OU.base(this,data,instance,controller);
};
OU.inherits(OU.activity.LayerZoom,OU.util.Activity);
