/**
 * @fileOverview  DiffractionGrating - Simulates a diffraction grating experiment
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Layer');
OU.require('OU.util.Slideshow');
OU.require('OU.util.ImageLoader');
OU.require('OU.util.Instruction');

/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.DiffractionGrating = function(data,instance,controller) {

    OU.activity.DiffractionGrating.prototype.canvasView = function() {
        var self=this;

        this.bgLayer = new OU.util.Layer({
            container:this
        });
        this.stringLayer = new OU.util.Layer({
            container:this,
            hasEvents:true
        });
        this.stringLayer.events.clickable.push(this);

        this.grateLayer = new OU.util.Layer({
            container:this,
            x:this.x,
            y: this.y,
            w: this.w,
            h: this.h*.5
        });//*/
        this.titleLayer = new OU.util.Layer({
            container:this,
            x:this.x,
            y:this.y,
            w: this.w,
            h: this.h*.1
        });

        this.imageLoader = new OU.util.ImageLoader({
            container: this,
            data: this.data,
            onLoad: function () {
                self.start();
            }
        });
        this.imageId=0;
    };
    OU.activity.DiffractionGrating.prototype.renderSpectrumTitle = function() {
        var ctx = this.titleLayer.context,
        w = this.w/3,
        h = this.h/2,
        x = (this.w - w)/2,
        y = 10*OU.dpr;//(this.h*.4 - h)/2;

        ctx.clearRect(0,0,this.w, this.h);
        new OU.util.DynText({
            context: ctx,
            txt: 'SPECTRUM DISPLAY',
            x: x,
            y: y,
            w: w,
            h: 25*OU.dpr,
            fontSize: 14*OU.dpr,
            background: {
                col: 'rgba(255,255,255,0.6)'
            },
            colour: '#222'
        });
    };
    OU.activity.DiffractionGrating.prototype.start = function() {

        this._imageReady=true;
        this.resize();

        new OU.util.Instruction({
            container: this,
            message: "Drag left or right to change the angle",
            point: {
                x: this.x+this.w/2,
                y: this.y+this.h/2 + 120
            },
            bgCol: '#000',
            colour: '#fff'
        });
    };
    OU.activity.DiffractionGrating.prototype.renderString = function() {
        var l=this.stringLayer, ctx = l.context,degrees= (180-this.imageId)/4, // 4 images per degree
        stringLength=(760/1800)*this.data.background.image.height*this.bgScale,
        degreeStr;
        l.clear();
        ctx.save();
        ctx.translate(this.stringCenterX,this.stringCenterY);
        ctx.rotate(degrees*Math.PI*2/360);
        ctx.beginPath();
        ctx.moveTo(0,0);
        ctx.lineTo(0,stringLength);
        ctx.stroke();
        ctx.beginPath();
        ctx.arc(0,stringLength,10*OU.dpr,0,Math.PI*2,false);
        ctx.fillStyle='#004';
        ctx.fill();
        degreeStr=""+Math.abs((degrees*100)|0)/100+"\u00B0";
        ctx.fillText(degreeStr || "?",0,stringLength+30*OU.dpr);
        ctx.restore();
    };
    OU.activity.DiffractionGrating.prototype.isHit = function(x, y, evState, ev) {
        var dX,yMultiplier;
        if (evState !== undefined)
            this.mouseDown = evState;

        if (this.mouseDown) {
            if (this.inDrag) {

                yMultiplier = (this.bgH/2)/(y-(this.bgY+this.bgH/2)); // move faster if being moved closer to the centre point
                yMultiplier=yMultiplier>20?20:yMultiplier;
                dX = 0.7*yMultiplier*(x - this.dragStartX)/OU.dpr;
                if(dX|0 != 0) {
                    this.dragStartX = x;
                    this.imageId += dX|0;
                    this.imageId = this.imageId<0?0:this.imageId>this.grateView.numImages-1?this.grateView.numImages-1:this.imageId;
                    this.grateView.move(this.imageId);
                    this.renderString();
                }
            }
            else {
                this.dragStartX = x;
                this.inDrag = true;
            }
        }
        else {
            this.inDrag = false;
        }
    };
    OU.activity.DiffractionGrating.prototype.resize = function() {
        OU.activity.DiffractionGrating.superClass_.resize.call(this); // call the parent class resize

        var bgImg = this.data.background.image,sctx=this.stringLayer.context;

        if(!this._imageReady)
            return;

        this.bgLayer.resize();
        this.stringLayer.resize();
        this.grateLayer.resize({
            x:this.x,
            y: this.y,
            w: this.w,
            h: this.h*.4
        });

        sctx.lineWidth=OU.dpr;
        var font = new OU.font({ size: 16*OU.dpr});
        sctx.font = font.str();
        sctx.textAlign = 'center';

        this.bgW = bgImg.width;
        this.bgH = bgImg.height;
        if(this.bgW>this.w) {
            this.bgH = this.bgH*(this.w/this.bgW);
            this.bgW = this.w;
        }
        if(this.bgH>this.h) {
            this.bgW = this.bgW*(this.h/this.bgH);
            this.bgH = this.h;
        }
        this.bgX = (this.w-this.bgW)/2;
        this.bgY = (this.h-this.bgH)/2;
        this.bgScale = this.bgW/bgImg.width;
        this.bgLayer.clear();
        this.bgLayer.context.drawImage(bgImg,this.bgX,this.bgY,this.bgW,this.bgH);
        this.stringCenterX = this.bgX+(619/1200)*bgImg.width*this.bgScale;
        this.stringCenterY = this.bgY+(827/1800)*bgImg.height*this.bgScale;

        if(this.grateView ) {
            this.grateView.resize({
                x: this.bgX,
                y: this.bgY,
                w: this.bgW,
                h: this.bgH*.4
            });
        }
        else {
            this.grateView = new OU.util.Slideshow({
                container: this,
                context: this.grateLayer.context,
                x:this.bgX,
                y: this.bgY,
                w: this.bgW,
                h: this.bgH*.4,
                images: this.data.grateSlideshow.images
            });
        }
        this.renderSpectrumTitle();
        this.renderString();
    };

    OU.base(this,data,instance,controller);
};
OU.inherits(OU.activity.DiffractionGrating,OU.util.Activity);
