/**
 * @fileOverview Drag2group - user drags a serious of items onto various group boxes
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.PopUpInfo');
OU.require('OU.util.Instruction');
OU.require('OU.util.Draggable');
OU.require('OU.util.Layer');
OU.require('OU.util.ImageLoader');

KEY_LEFT=37;
KEY_UP=38;
KEY_RIGHT=39;
KEY_DOWN=40;

/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.Drag2group = function ( data, instance, controller ) {
    OU.activity.Drag2group.prototype.canvasView = function () {
        var self = this, bH = OU.controlHeight;
        this.bgLayer = new OU.util.Layer({
            container:this
        });
        this.messageLayer = new OU.util.Layer({
            container:this
        });
        this.itemsLayer = new OU.util.Layer({
            container:this,
            hasEvents:true
        });
        this.itemsLayer.context.font = "bold 20px " + OU.theme.font;
        if (this.data.preload===true) {
            this.controlLayer = new OU.util.Layer({
                x:this.x + this.w / 2 - bH,
                y:this.y + this.h - bH,
                w:bH * 2,
                h:bH,
                container:this,
                hasEvents:true
            });
        }
        this.images = {};
        this.images.data = this.data;
        this.images.background = {
            img:this.data.background
        };
        this.imageLoader = new OU.util.ImageLoader({
            container:this,
            data:this.images,
            onLoad:function () {
                self.start();
                self.started = true;
            }
        });
    };
    OU.activity.Drag2group.prototype.resize = function () {
        OU.activity.Drag2group.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight;
        if (!this.started)
            return;
        this.bgLayer.resize();
        this.messageLayer.resize();
        this.itemsLayer.resize();
        if (this.data.preload===true) {
            this.controlLayer.resize({
                x:this.x + this.w / 2 - bH,
                y:this.y + this.h - bH,
                w:bH * 2,
                h:bH
            });
            this.doneButton.render();
        }
        this.initScale();
        this.doRender = true;
    };
    OU.activity.Drag2group.prototype.remove = function () {
        OU.activity.Drag2group.superClass_.remove.call(this); // call the superclass method
        this.doRender = false;
    };
    OU.activity.Drag2group.prototype.initScale = function () {
        var i, g, bg = this.images.background.image,
        fitW = this.w / bg.width,
        fitH = this.h / bg.height,
        s = this.scale = (fitW < fitH ? fitW : fitH),
        x = (this.w - bg.width * s) / 2,
        y = (this.h - bg.height * s) / 2;
        this.bgLayer.context.drawImage(bg, x, y, bg.width * s, bg.height * s);
        if (this.data._doneResize===undefined) {
            this.data._doneResize = true;
            for (i = this.groups.length; i--;) { // scale group dimensions to background
                g = this.groups[i];
                g._x = g.x;
                g._y = g.y;
                g._w = g.w;
                g._h = g.h;
                g.c = '#0f0';
            }
        }
        for (i = this.groups.length; i--;) { // scale group dimensions to background
            g = this.groups[i];
            g.x = x + g._x * s;
            g.y = y + g._y * s;
            g.w = g._w * s;
            g.h = g._h * s;
        //this.bgLayer.context.strokeStyle=g.c;
        //this.bgLayer.context.strokeRect(g.x,g.y,g.w,g.h);
        }
    };
    OU.activity.Drag2group.prototype.start = function () {
        var self = this, bH = OU.controlHeight, i, j, g, item;
        //Init data
        this.visibles = []; // onscreen items
        this.items = [];
        this.groups = [];
        for (j = this.data.groups.length; j--;) {
            g = this.data.groups[j];
            this.groups.push(g);
            for (i = g.items.length; i--;) {
                item = g.items[i];
                item.group = g;
                item.visible = false;
                item.w = this.itemsLayer.context.measureText(item.label, 0, 0).width * 1.5;
                item.h = 30;
                this.items.push(item);
            }
        }
        // randomise order of the items
        this.items.sort(function () {
            return ( Math.round(Math.random()) - 0.5 );
        });
        this.initScale();
        if (this.data.preload===true) {
            this.doneButton = new OU.util.ControlButton({
                txt:"I'm Done",
                x:0,
                y:0,
                w:bH * 2,
                h:bH,
                layer:this.controlLayer,
                onClick:function () {
                    self.feedback();
                }
            });
        }
        this.restart();
    };
    OU.activity.Drag2group.prototype.restart = function () {
        var targetX, targetY,shrinkArea,infoSize=2*OU.controlHeight;
        this._feedbackRendered = false;
        if(this.data.infoButtonLocation=='topRight')
            shrinkArea = {
                x: this.w-infoSize,
                y: OU.controlHeight,
                w: infoSize,
                h: infoSize
            };
        if(this.data.infoButtonLocation=='bottomRight')
            shrinkArea = {
                x: this.w-infoSize,
                y: this.h-infoSize,
                w: infoSize,
                h: infoSize
            };
        if(this.data.infoButtonLocation=='bottomLeft')
            shrinkArea = {
                x: OU.controlHeight,
                y: this.h-infoSize,
                w: infoSize,
                h: infoSize
            };
        if (this.data.preload===true) {
            this.popAllItems();
            if (this.data.instructions!==undefined) {
                new OU.util.Instruction({
                    container:this,
                    message:this.data.instructions,
                    shrinkArea: shrinkArea // defaults to topLeft if undefined
                });
            }
        }
        else {
            this.popped = 0;
            this.visibles.length = 0;
            this.popItem(); // add first item to the screen
            if (this.data.defaultStartingPosition!==undefined && this.data.defaultStartingPosition.x!=0) {
                targetX = (this.data.defaultStartingPosition.x * this.scale);
                targetY = (this.data.defaultStartingPosition.y * this.scale);
            }
            else {
                targetX = this.w / 2;
                targetY = 75;
            }
            if (this.data.instructions!==undefined) {
                new OU.util.Instruction({
                    container:this,
                    message:this.data.instructions,
                    duration:this.data.instructionDuration,
                    point:{
                        x:targetX,
                        y:targetY
                    },
                        shrinkArea: shrinkArea // defaults to topLeft if undefined
                });
            }
        }
        // start render loop
        this.doRender = true;
        this.renderLoop();
    };
    OU.activity.Drag2group.prototype.popAllItems = function () {
        var i, item, x, y;
        this.popped = 0;
        this.visibles.length = 0;
        for (i = this.items.length; i--;) {
            item = this.items[i];
            if (item.init!==undefined) {
                x = item.init.x * this.scale;
                y = item.init.y * this.scale;
            }
            else {
                x = (this.w - item.w) / 2;
                y = 10;
            }
            this.visibles.push(new this.Item({
                "events":this.itemsLayer.events,
                "item":item,
                "sortOrder":this.popped,
                "order":this.popped,
                "name":item.label,
                "color":'#fcfcfc',
                "x":x,
                "y":y,
                "w":item.w,
                "h":item.h,
                parent:this
            }));
        }
        this.popped++;
    };
    OU.activity.Drag2group.prototype.popItem = function () {
        var item = this.items[this.popped];
        if (item===undefined) {
            return;
        }
        this.visibles.push(new this.Item({
            "events":this.itemsLayer.events,
            "item":item,
            "sortOrder":this.popped,
            "order":this.popped,
            "name":item.label,
            "color":'#fcfcfc',
            "x":(this.w - item.w) / 2,
            "y":-20,
            "w":item.w,
            "h":item.h,
            parent:this
        }));
        this.popped++;
    };
    OU.activity.Drag2group.prototype.renderLoop = function () {
        var self = this;
        if (this.doRender)
            this.render();
        setTimeout(function () {
            self.renderLoop();
        }, 40);
    };
    OU.activity.Drag2group.prototype.distance = function ( item, group ) { // returns the distance between the centres of 2 objects
        var iX = item.x + (item.w / 2),
        iY = item.y + (item.h / 2),
        gX = group.x + (group.w / 2),
        gY = group.y + (group.h / 2),
        dx = gX - iX,
        dy = gY - iY;
        return Math.sqrt(dx * dx + dy * dy);
    };
    OU.activity.Drag2group.prototype.nearestGroup = function ( item ) {
        var closest = null, i, g, d, shortest = 99999999;
        for (i = this.groups.length; i--;) { // reset group counts
            g = this.groups[i];
            if (this.collide(item, g)) {
                d = this.distance(item, g);
                if (d < shortest) {
                    shortest = d;
                    closest = g;
                }
            }
        }
        return closest;
    };
    OU.activity.Drag2group.prototype.collide = function ( a, b ) {
        var ArX = a.x + a.w,
        AbY = a.y + a.h,
        BrX = b.x + b.w,
        BbY = b.y + b.h,
        collide = true;
        if (a.x > BrX)
            collide = false;
        if (ArX < b.x)
            collide = false;
        if (a.y > BbY)
            collide = false;
        if (AbY < b.y)
            collide = false;
        return collide;
    };
    OU.activity.Drag2group.prototype.message = function ( t, d ) {
        var ctx = this.messageLayer.context,
        now = new Date().getTime();
        if (t!==undefined) {
            this._message = t;
            this._messageStart = new Date().getTime();
            this._duration = d || 2000;
        }
        if (this._message===undefined)
            return;
        ctx.clearRect(0, 0, this.w, this.h);
        if (now - this._messageStart > this._duration)
            return;
        ctx.globalAlpha = 1 - (now - this._messageStart) / this._duration;
        new OU.util.DynText({
            txt:this._message,
            context:ctx,
            x:this.w / 4,
            y:this.h * .75,
            w:this.w / 2,
            h:this.h * .2,
            background:{
                col:'#ccc',
                radius:20
            }
        });
        this.doRender = true;
    };
    OU.activity.Drag2group.prototype.render = function () {
        // render dragged items
        var events = this.itemsLayer.events,
        visible, g, i, j, targetX, targetY, newX, newY, colW,
        inGroups = 0, shuffling = false;
        this.itemsLayer.clear();
        events.clickable.length = 0;
        for (i = this.groups.length; i--;) { // reset group counts
            g = this.groups[i];
            g.nItems = 0;
            g.rItems = 0;
            g.totalW = 0;
            g.totalH = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0];
            g.runningH = 0;
            g.columns = 0;
            g.column = 0;
            g.lastColumn = 0;
            g.colItems = 0;
        /* Debug - Show where groups are
             var ctx = this.itemsLayer.context;
             ctx.rect(g.x,g.y,g.w,g.h);
             ctx.fillStyle='#c00';
             ctx.fill();
             // */
        }
        for (i = this.visibles.length; i--;) {
            visible = this.visibles[i];
            // animate shuffle if out of place
            visible.g = null; //this.groups[this.groups.length-1];
            visible.itmN = 0;
            visible.colN = 0;
            if (visible.draggable.dragId==0 && !visible.hasFocus) { // item is not being dragged (or has Tab Focus), so shuffle to appropriate position
                for (j = this.groups.length; j--;) {
                    g = this.groups[j];
                    if (this.collide(g, visible)) {
                        g = this.nearestGroup(visible);
                        if (g.maxItems===undefined || g.maxItems < 1 || g.nItems < g.maxItems) {
                            visible.g = g;
                            visible.itmN = g.nItems;
                            g.nItems++;
                            //g.c='#f00';
                            inGroups++;
                        }
                        else {
                            //g.c='#000';
                            this.message('Only ' + g.maxItems + ' item' + (g.maxItems > 1 ? 's' : '') + ' allowed in each area');
                        }
                        j = 0;
                    }
                }
            }
            else {
                shuffling = true;
            }
        //            this.bgLayer.context.strokeStyle=g.c;
        //            this.bgLayer.context.strokeRect(g.x,g.y,g.w,g.h);
        }
        var maxH = -1;
        for (i = this.visibles.length; i--;) {
            visible = this.visibles[i];
            if (visible.h > maxH)
                maxH = visible.h;
            if (visible.draggable.dragId==0) {
                g = visible.g;
                if (g!=null) {
                    g.totalW = g.totalW + visible.w;
                    if (g.totalH[g.columns] + visible.h > g.h && visible.h < g.h) {
                        g.columns++;
                    }
                    g.totalH[g.columns] = g.totalH[g.columns] + visible.h;
                }
            }
        }
        //   var s=this.scale;
        for (i = this.visibles.length; i--;) {
            visible = this.visibles[i];
            if (visible.draggable.dragId==0 && !visible.hasFocus) {
                g = visible.g;
                if (g!=null) {
                    colW = g.w / (g.columns + 1);
                    if (g.runningH + visible.h > g.h && visible.h < g.h) {
                        g.column++;
                        g.runningH = 0;
                    }
                    targetX = g.x + (g.column * colW) + (colW - visible.w) / 2;
                    targetY = g.y + ((g.h - g.totalH[g.column]) / 2) + g.runningH;
                    g.runningH = g.runningH + visible.h + 10;
                    g.rItems++;
                }
                else {
                    if (this.data.defaultStartingPosition!==undefined && this.data.defaultStartingPosition.x!=0) {
                        targetX = (this.data.defaultStartingPosition.x * this.scale) - visible.w / 2;
                        targetY = (this.data.defaultStartingPosition.y * this.scale) - visible.h / 2;
                    }
                    else {
                        targetX = (this.w - visible.w) / 2;
                        targetY = 10 + visible.h / 2;
                    }
                }
                if (visible.x!=targetX || visible.y!=targetY) {
                    shuffling = true;
                    newX = visible.x + (targetX - visible.x) / 2;
                    newY = visible.y + (targetY - visible.y) / 2;
                    if (Math.abs(targetX - newX) < 0.2)
                        newX = targetX;
                    if (Math.abs(targetY - newY) < 0.2)
                        newY = targetY;
                    visible.move(newX, newY);
                }
            }
            visible.render();
        }
        if (this.data.preload!==true && inGroups==this.popped)
            this.popItem();
        if (!shuffling) {
            this.doRender = false;
        }
        /*
         if(inGroups<1)
         this.message(this.data.instructions);
         else
         this.message();
         */
        this.message();
        if (this.data.preload!==true && inGroups >= this.items.length)
            this.feedback();
    };
    OU.activity.Drag2group.prototype.feedback = function () {
        var h = '', i, item, incorrect = 0, dX, dY, dW, dH, fA = this.data.feedbackArea,
        ev = this.itemsLayer.events,
        self = this;
        if (this._feedbackRendered)
            return;
        this._feedbackRendered = true;
        for (i = this.visibles.length; i--;) {
            item = this.visibles[i];
            h = h + "<p><strong>" + item.item.label + "</strong><br/>";
            if (item.g===item.item.group) {
                h = h + item.item.feedback.correct + "</p>";
            }
            else {
                h = h + item.item.feedback.incorrect + "</p>";
                incorrect++;
            }
        }
        if (incorrect==0)
            h = "<h1 style='margin-top:0;' >Well done, you have it all correct.</h1>" + h;
        else
            h = "<h1 style='margin-top:0;'>Sorry, you don't have them all correct...</h1>" + h;
        if (fA!==undefined) {
            dX = fA.x * this.w;
            dY = fA.y * this.h;
            dW = fA.w * this.w;
            dH = fA.h * this.h;
        }
        else {
            dX = 0.1 * this.w;
            dY = 0.1 * this.h;
            dW = 0.8 * this.w;
            dH = 0.8 * this.h;
        }
        ev.flush();
        new OU.util.PopUpInfo({
            container:this,
            txt:h,
            x:dX,
            y:dY,
            w:dW,
            h:dH,
            onClose:function () {
                self.restart();
            }
        });
    };
    /**
     * @class
     * @extends Tabbable
     */
    OU.activity.Drag2group.prototype.Item = function ( params ) {
        this.events = params.events;
        this.item = params.item;
        this.order = params.order;
        this.sortOrder = params.sortOrder;
        this.parent = params.parent;
        this.container = this.parent.container;
        this.name = params.name;
        this.color = params.color,
        this.x = params.x;
        this.y = params.y;
        this.h = params.h * this.parent.scale;
        this.w = params.w * this.parent.scale;
        this.image = params.item.image;
        this.startX = this.x;
        this.startY = this.y;
        if (this.image!==undefined) {
            this.w = this.image.width * this.parent.scale;
            this.h = this.image.height * this.parent.scale;
        }
        OU.activity.Drag2group.prototype.Item.prototype.render = function () {
            var ctx = this.parent.itemsLayer.context,bg,
            sF = (this.g==null ? 1.4 : (this.draggable.dragId!=0 ? 1.4 : 1)),
            x = this.x - (this.w * (sF - 1) / 2),
            y = this.y - (this.h * (sF - 1) / 2),
            w = this.w * sF,
            h = this.h * sF;
            if (this.image!==undefined) {
                ctx.drawImage(this.image, x, y, w, h);
            }
            else {
                if(this.hasFocus) {
                    bg = {
                        "borderCol":"#444",
                        "col":'#c00',
                        "radius":h / 2
                    };
                }
                else {
                    bg = {
                        "borderCol":"#444",
                        "col":this.color,
                        "radius":h / 2
                    };
                }
                new OU.util.DynText({
                    "txt":this.name,
                    "x":x,
                    "y":y,
                    "w":w,
                    "h":h,
                    "context":ctx,
                    background:bg,
                    fontWeight:"bold",
                    fontSize:20*OU.dpr
                });
            }
            this.parent.itemsLayer.events.clickable.push(this.draggable);
        };
        OU.activity.Drag2group.prototype.Item.prototype.focus = function () {
            this.hasFocus = true;
            this.render();
            return false;
        };
        OU.activity.Drag2group.prototype.Item.prototype.blur = function () {
            this.hasFocus = false;
            this.render();
            return false;
        };
        OU.activity.Drag2group.prototype.Item.prototype.arrowKey = function (k) {
            switch(k){
                case KEY_UP:
                    this.move(this.x,this.y-10);
                    break;
                case KEY_DOWN:
                    this.move(this.x,this.y+10);
                    break;
                case KEY_LEFT:
                    this.move(this.x-10,this.y);
                    break;
                case KEY_RIGHT:
                    this.move(this.x+10,this.y);
                    break;
            }
            this.parent.doRender=true;
        };
        OU.activity.Drag2group.prototype.Item.prototype.startDrag = function ( p ) {
            //sort the items so that the latest dragged is at the end of the queue, therefore render last next time (on top)
            p.me.parent.visibles.sort(function ( a, b ) {
                return b.draggable.dragId - a.draggable.dragId;
            });
            p.me.parent.view = 'norm';
            p.me.parent.doRender = true;
        };
        OU.activity.Drag2group.prototype.Item.prototype.endDrag = function ( p ) {
            p.me.parent.doRender = true;
        };
        OU.activity.Drag2group.prototype.Item.prototype.newPos = function ( p ) {
            p.me.x = p.x;
            p.me.y = p.y;
            p.me.doRender = true;
        };
        OU.activity.Drag2group.prototype.Item.prototype.move = function ( x, y ) {
            this.x = x;
            this.y = y;
            this.draggable.x = x;
            this.draggable.y = y;
        };
        this.draggable = new OU.util.Draggable({
            "me":this,
            "events":this.parent.itemsLayer.events,
            "x":this.x,
            "y":this.y,
            "h":this.h,
            "w":this.w,
            "onStart":this.startDrag,
            "onEnd":this.endDrag,
            "onMove":this.newPos
        });

        OU.base(this, params);
    };
    OU.inherits(OU.activity.Drag2group.prototype.Item, OU.util.Tabbable);

    OU.activity.Drag2group.prototype.accessibleView = function () {
        clearInterval(this.renderCycle);
        var j, g, i, item, h = '<div id="accessibleView">';
        h += '<h1>Drag and drop activity</h1>';
        if (this.data.description!==undefined) {
            h += '<p>' + this.data.description + '</p>';
        }
        else {
            h += '<p>This acivity allows you to drag words or images related parts of a background image or diagram.</p>';
            h += '<p>This particular activity has the following groups and corresponding items within each.</p>';
            for (j = this.data.groups.length; j--;) {
                g = this.data.groups[j];
                h += "<h2>" + g.name + "</h2>";
                for (i = g.items.length; i--;) {
                    item = g.items[i];
                    h += "<h3>" + item.label + "</h3>";
                    if (item.feedback.correct)
                        h += "<p>Correct placement feedback: " + item.feedback.correct + "</p>";
                    if (item.feedback.incorrect)
                        h += "<p>Incorrect placement feedback: " + item.feedback.incorrect + "</p>";
                }
            }
        }
        h += '</div>';
        document.body.innerHTML = '';
        var accessible = document.createElement('div');
        accessible.innerHTML = h;
        document.body.appendChild(accessible);
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.Drag2group, OU.util.Activity);
