/**
 * @fileOverview Drag2Table - user drags a serious of items onto a table structure
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.PopUpInfo');
OU.require('OU.util.Instruction');
OU.require('OU.util.Draggable');
OU.require('OU.util.Layer');
OU.require('OU.util.ImageLoader');

KEY_LEFT=37;
KEY_UP=38;
KEY_RIGHT=39;
KEY_DOWN=40;

/**
 * CanvasTable - renders and manipulates a table object on canvas
 * 
 * @param {json objet} params - options:
 * <ul>
 * <li><strong>{object} container:</strong> (required) Calling object, typically a OU.util.Activity</li>
 * <li><strong>{OU.util.Layer} layer:</strong> (required) Layer to render the table to</li>
 * <li><strong>{json object} sections:</strong> (required) defines the structure, ie. sections, columns, rows, etc.</li>
 * <li><strong>{int} x:</strong> X co-ordinate</li>
 * <li><strong>{int} y:</strong> Y co-ordinate</li>
 * <li><strong>{int} w:</strong> Width</li>
 * <li><strong>{int} h:</strong> Height</li>
 * </ul>
 */
OU.util.CanvasTable = function(params) {
    if(params.container===undefined) {
        console.log("ERROR: CanvasTable requires a container");
        return;
    }
    if(params.layer===undefined) {
        console.log("ERROR: CanvasTable requires a layer");
        return;
    }
    if(params.sections===undefined) {
        console.log("ERROR: CanvasTable requires sections");
        return;
    }
    this.container = params.container;
    this.layer = params.layer;
    this.context = params.layer.context;
    this.x = params.x || 0;
    this.y = params.y || 0;
    this.w = params.w || this.container.w;
    this.h = params.h || this.container.h;
    this.sections = params.sections;
    this.borderColour = params.borderColour || "#000";
    this.bgColour = params.bgColour || "#fff";
    this.titleTextColour = params.titleTextColour || "#fff";
    this.titleBgColour = params.titleBgColour || "#000";
    this.hasInfo=params.hasInfo || false;
    this.totalItems=0;
    
    /**
     * Initialise the sections and use resize to render
     * @private
     */
    OU.util.CanvasTable.prototype.init = function() {
        var i;
        for(i=this.sections.length;i--;) {
            this.sections[i]._items=[];
        }
        
        this.resize();
    };
    /**
     * Renders the table
     */
    OU.util.CanvasTable.prototype.render = function() {
        var section,sectionNum,row,column,ctx=this.context,sectionOffset=this.x,
        minFont=1000,sectionText;
        
        ctx.strokeStyle=this.borderColour;
        ctx.fillStyle=this.bgColour;
        ctx.lineWidth=6;
        ctx.rect(sectionOffset,this.y,this.w,this.h);
        ctx.stroke();
        ctx.fill();
        for(sectionNum=0; sectionNum<this.sections.length;sectionNum++) {
            ctx.lineWidth=6;
            section = this.sections[sectionNum];
            ctx.rect(sectionOffset,this.y,this.columnWidth*section.columns,this.h); //this.totalRows*this.rowHeight);
            ctx.stroke();
            ctx.save();
            ctx.lineWidth=1;
            for(column=section.columns; column--;) {
                ctx.beginPath();
                ctx.moveTo(sectionOffset+column*this.columnWidth,this.y);
                ctx.lineTo(sectionOffset+column*this.columnWidth,this.y+this.totalRows*this.rowHeight);
                ctx.stroke();
                for(row=section.rows+1;row--;) {
                    ctx.beginPath();
                    ctx.moveTo(sectionOffset,this.y+row*this.rowHeight);
                    ctx.lineTo(sectionOffset+section.columns*this.columnWidth,this.y+row*this.rowHeight);
                    ctx.stroke();
                }
            }
            ctx.save();
            ctx.fillStyle=this.titleBgColour;
            ctx.fillRect(sectionOffset,this.y+this.h-this.titleHeight,this.columnWidth*section.columns,this.titleHeight);
            ctx.fillStyle=this.titleTextColour;
            // Calculate smallest font used to render all titles - do not render yet
            sectionText = new OU.util.DynText({
                context:ctx,
                txt: section.title,
                fontSize: this.titleHeight*.8,
                x: sectionOffset,
                y:this.y+this.h-this.titleHeight,
                w:this.columnWidth*section.columns,
                h:this.titleHeight,
                measureOnly:true
            });
            ctx.restore();
            if(sectionText.font.size < minFont)
                minFont=sectionText.font.size
            sectionOffset = sectionOffset + section.columns*this.columnWidth;
            ctx.restore();
        }
        // Now render all titles at the same size (the smallest needed to fit)
        sectionOffset=this.x;
        ctx.save();
        ctx.fillStyle=this.titleTextColour;
        for(sectionNum=0; sectionNum<this.sections.length;sectionNum++) {
            section = this.sections[sectionNum];
            section._x = sectionOffset;
            sectionText = new OU.util.DynText({
                context:ctx,
                txt: section.title,
                fontSize: minFont,
                x: sectionOffset,
                y:this.y+this.h-this.titleHeight,
                w:this.columnWidth*section.columns,
                h:this.titleHeight
            });
            sectionOffset = sectionOffset + section.columns*this.columnWidth;
        }
        ctx.restore();
    };
    /**
     * Resizes the table dimensions and re-calculates the section sizes, etc.
     * @param {json object} dims - optional new dims, default to current value unless specified:
     * <ul>
     * <li><strong>{int} x:</strong> X co-ordinate</li>
     * <li><strong>{int} y:</strong> Y co-ordinate</li>
     * <li><strong>{int} w:</strong> Width</li>
     * <li><strong>{int} h:</strong> Height</li>
     * </ul>
     */
    OU.util.CanvasTable.prototype.resize = function(dims) {
        var i,j,section,hasTitles=false;
        if(dims===undefined) dims={};
        this.x = dims.x || this.x;
        this.y = dims.y || this.y;
        this.w = dims.w || this.w;
        this.h = dims.h || this.h;
        
        this.totalColumns = 0;
        this.totalRows = 0;
        for(i=this.sections.length;i--;) {
            section = this.sections[i];
            this.totalColumns += section.columns;
            if(section.rows>this.totalRows)
                this.totalRows = section.rows;
            if(section.title && section.title!=="")
                hasTitles=true;
            for(j=section._items.length;j--;) {
                this.setTargetPosition(section._items[j]);
            }
        }
        this.columnWidth = this.w / this.totalColumns;
        if(hasTitles)
            this.titleHeight=OU.controlHeight;
        else 
            this.titleHeight=0;
        if(this.hasInfo) // if has feedback, then add another row for it
            this.rowHeight = (this.h-this.titleHeight) / (this.totalRows+1);
        else
            this.rowHeight = (this.h-this.titleHeight) / this.totalRows;
        
        this.render();
    };
    /**
     * Adds an item to the table, if it is over the table
     * @param {json object} item - and item that has x,y,w,h dimensions and a move() function
     */
    OU.util.CanvasTable.prototype.addItem = function(item) {
        var i,
        centreX = item.x+item.w/2,
        centreY = item.y+item.h/2,
        section;
        //        console.log("add item, X,Y:"+centreX+","+centreY);
        if(centreX<this.x || centreX>this.x+this.w || centreY<this.y || centreY>this.y+this.h)
            return false; //return false if item is not over the table
        
        //determine which section it is over
        for(i=0; i<this.sections.length;i++) {
            section = this.sections[i];
            if(centreX > section._x && centreX < section._x+section.columns*this.columnWidth) {
                this.addItemToSection(section,item);
                return true;
            }
        }
        return false;
    };
    /**
     * Sets the target position of an item based on the index number in a specific section
     * @param {json object} item - the item to position
     * @private
     */
    OU.util.CanvasTable.prototype.setTargetPosition = function(item) {
        var section = item.tableInfo.section,itemIndex = item.tableInfo.itemIndex,
        column = itemIndex % section.columns,
        row = (itemIndex-column) / section.columns;
        
        item.tableInfo.targetX=section._x + this.columnWidth*column + (this.columnWidth-item.w)/2;
        item.tableInfo.targetY=this.y + this.rowHeight*row + (this.rowHeight-item.h)/2;
    };
    /**
     * Adds an item to a table section
     * @param {json object} section - the section that the item is going in
     * @param {json object} item - the item to put in the section
     * @private
     */
    OU.util.CanvasTable.prototype.addItemToSection = function(section,item) {
        var itemIndex = section._items.length;
        section._items[itemIndex]=item;
        item.tableInfo = {
            section:section,
            itemIndex:itemIndex
        };
        this.totalItems++;
        this.setTargetPosition(item);
    };
    /**
     * Removes an item from a section and closes any gap left by adjusting the indexes of all the items left.
     * @param {json object} item - the item to remove
     */
    OU.util.CanvasTable.prototype.removeItem = function(item) {
        var i,section = item.tableInfo.section;
        section._items.splice(item.tableInfo.itemIndex,1);
        item.tableInfo=null;
        this.totalItems--;
        for(i=0;i<section._items.length;i++) {
            section._items[i].tableInfo.itemIndex=i;
            this.setTargetPosition(section._items[i]);
        }
        return true;
    };
    /**
     * Moves an item towards it's target position. Calls the items own move() function with a new value in the direction of the target
     * @param {json object} item - the item to move
     */
    OU.util.CanvasTable.prototype.moveItem = function(item) {
        var x,y,dX = item.tableInfo.targetX - item.x,
        dY = item.tableInfo.targetY - item.y;
        if(dX==0 && dY==0)
            return false; // already at target, so return false, meaning no movement
        if(Math.abs(dX)<2) {
            x=item.tableInfo.targetX;
        }
        else {
            x = item.x+dX/2;
        }
        if(Math.abs(dY)<2) {
            y=item.tableInfo.targetY;
        }
        else {
            y = item.y+dY/2;
        }
        item.move(x,y);
        return true;
    };
    this.init();
};
/**
 * @class Drag2Table - renders a table in the bottom half of the activity and places items in the top half on a scrolling backdrop.
 * The user is expected to drag the items into the correct sections of the table.
 * 
 * @extends OU.util.Activity
 */
OU.activity.Drag2Table = function ( data, instance, controller ) {
    OU.activity.Drag2Table.prototype.canvasView = function () {
        var self = this, bH = OU.controlHeight;
        this.bgLayer = new OU.util.Layer({
            container:this
        });
        this.messageLayer = new OU.util.Layer({
            container:this
        });
        this.tableLayer = new OU.util.Layer({
            container:this
        });

        this.table = new OU.util.CanvasTable({
            container:this,
            layer: this.tableLayer,
            sections: this.data.table.sections,
            x: this.x+this.w*0.05,
            y: this.y+this.h*0.5,
            w: this.w*0.9,
            h: this.h*0.475,
            borderColour:this.data.table.borderColour,
            bgColour:this.data.table.bgColour,
            titleBgColour:this.data.table.titleBgColour,
            titleTextColour:this.data.table.titleTextColour,
            hasInfo:this.data.table.hasInfo
        });
        this.itemsLayer = new OU.util.Layer({
            container:this,
            hasEvents:true
        });
        this.itemsLayer.context.font = "bold 20px " + OU.theme.font;
        if (this.data.preload===true) {
            this.controlLayer = new OU.util.Layer({
                x:this.x + this.w / 2 - bH,
                y:this.y + this.h - bH,
                w:bH * 2,
                h:bH,
                container:this,
                hasEvents:true
            });
        }
        this.imageLoader = new OU.util.ImageLoader({
            container:this,
            data:this.data,
            onLoad:function () {
                self.start();
                self.started = true;
            }
        });
    };
    /**
     * Resizes all elements
     * @private
     */
    OU.activity.Drag2Table.prototype.resize = function () {
        OU.activity.Drag2Table.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight;
        if (!this.started)
            return;
        this.bgLayer.resize();

        this.tableLayer.resize();
        this.table.resize({
            x: this.x+this.w*0.05,
            y: this.y+this.h*0.5,
            w: this.w*0.9,
            h: this.h*0.475
        });
        this.messageLayer.resize();
        this.itemsLayer.resize();
        if (this.data.preload===true) {
            this.controlLayer.resize({
                x:this.x + this.w / 2 - bH,
                y:this.y + this.h - bH,
                w:bH * 2,
                h:bH
            });
            this.doneButton.render();
        }
        this.initScale();
        this.doRender = true;
    };
    /**
     * Stops the animation loop
     */
    OU.activity.Drag2Table.prototype.remove = function () {
        OU.activity.Drag2Table.superClass_.remove.call(this); // call the superclass method
        this.doRender = false;
    };
    /**
     * Calculates the scale of the activity
     * @private
     */
    OU.activity.Drag2Table.prototype.initScale = function () {
        var bg = this.data.background.image;
        this.scale = (this.h/2) / bg.height;
    };
    /**
     * Starts the activity off after all images have been loaded.
     * @private
     */
    OU.activity.Drag2Table.prototype.start = function () {
        var i, item,x,y;
        //Init data
        this.visibles = []; // onscreen items
        this.items = [];
        this.initScale();
        this.popped=0;
        this.strictOrder = this.data.strictOrder || false;
        for (i = this.data.items.length; i--;) {
            item = this.data.items[i];
            item.visible = false;
            if(item.image) {
                item.w = item.image.width*this.scale;
                item.h = item.image.height*this.scale;
            }
            else {
                item.w = this.itemsLayer.context.measureText(item.label, 0, 0).width * 1.5;
                item.h = 30;
            }
            this.items.push(item);
            
            if (item.startPos!==undefined) {
                x = item.startPos.x * this.scale;
                y = item.startPos.y * this.scale;
            }
            else {
                x = (this.w - item.w) / 2;
                y = 10;
            }
            this.visibles.push(new this.Item({
                "events":this.itemsLayer.events,
                "item":item,
                "sortOrder":this.popped,
                "order":this.popped,
                "name":item.label,
                "color":'#fcfcfc',
                "x":x,
                "y":y,
                "w":item.w,
                "h":item.h,
                disabled:this.strictOrder,
                parent:this
            }));
            this.popped++;
            
        }
        this.restart();
        // start render loop
        this.doRender = true;
        this.renderLoop();
    };
    /**
     * Resets the activity and starts it off again.
     * @private
     */
    OU.activity.Drag2Table.prototype.restart = function () {
        var shrinkArea,infoSize=2*OU.controlHeight;
        this._feedbackRendered = false;
        this.backgroundX = 0;
        if(this.data.infoButtonLocation=='topRight')
            shrinkArea = {
                x: this.w-infoSize,
                y: OU.controlHeight,
                w: infoSize,
                h: infoSize
            };
        if(this.data.infoButtonLocation=='bottomRight')
            shrinkArea = {
                x: this.w-infoSize,
                y: this.h-infoSize,
                w: infoSize,
                h: infoSize
            };
        if(this.data.infoButtonLocation=='bottomLeft')
            shrinkArea = {
                x: OU.controlHeight,
                y: this.h-infoSize,
                w: infoSize,
                h: infoSize
            };
        if (this.data.instructions!==undefined) {
            new OU.util.Instruction({
                container:this,
                message:this.data.instructions,
                shrinkArea: shrinkArea // defaults to topLeft if undefined
            });
        }
    };
    /**
     * Performs the render loop to enable animations
     * @private
     */
    OU.activity.Drag2Table.prototype.renderLoop = function () {
        var self = this;
        if (this.doRender)
            this.render();
        setTimeout(function () {
            self.renderLoop();
        }, 40);
    };
    /**
     * Renders a message to screen that times out after a set duration
     * @param {string} text - (optional) the message (provided the first time the message is called)
     * @param {int} duration - (optional) time is ms that message stays on screen 
     * @private
     */
    OU.activity.Drag2Table.prototype.message = function ( text, duration ) {
        var ctx = this.messageLayer.context,
        now = new Date().getTime();
        if (text!==undefined) {
            this._message = text;
            this._messageStart = new Date().getTime();
            this._duration = duration || 2000;
        }
        if (this._message===undefined)
            return;
        ctx.clearRect(0, 0, this.w, this.h);
        if (now - this._messageStart > this._duration)
            return;
        ctx.globalAlpha = 1 - (now - this._messageStart) / this._duration;
        new OU.util.DynText({
            txt:this._message,
            context:ctx,
            x:this.w / 4,
            y:this.h * .75,
            w:this.w / 2,
            h:this.h * .2,
            background:{
                col:'#ccc',
                radius:20
            }
        });
        this.doRender = true;
    };
    /**
     * Renders the activity
     * @private
     */
    OU.activity.Drag2Table.prototype.render = function () {
        // render dragged items
        var events = this.itemsLayer.events,
        bg=this.data.background,
        visible, i;
        this.itemsLayer.clear();
        events.clickable.length = 0;
        this.doRender = false;
        
        this.bgLayer.clear();
        this.bgLayer.context.fillStyle=this.data.backgroundColor;
        this.bgLayer.context.fillRect(0,0,this.w,this.h);
        this.bgLayer.context.drawImage(bg.image,this.backgroundX,0,bg.image.width*this.scale,bg.image.height*this.scale);
        
        for (i = this.visibles.length; i--;) {
            visible = this.visibles[i];
            // animate shuffle if out of place

            if (visible.draggable.dragId==0 && !visible.hasFocus) { // item is not being dragged (or has Tab Focus), so shuffle to appropriate position
                if(visible.tableInfo) { // visible is in the Table,so will have a position
                    if(this.table.moveItem(visible))
                        this.doRender = true; // if item is in motion then enable shuffling
                }
                else { // Item is not being dragged and is not already in a table, so try to add it (will only be added if currently over the table
                    if(this.table.addItem(visible)) {
                        this.doRender = true;
                    }
                }
            }
            else { // item is being dragged therefore 
                if(visible.tableInfo) { // Item is now being dragged, but is also in a table, so remove it from the table
                    this.table.removeItem(visible);
                }
                this.doRender = true;
            }
            visible.render();
        }

        this.message();
        
    };

    /**
     * @class Item - a draggable item
     * @extends Tabbable
     */
    OU.activity.Drag2Table.prototype.Item = function ( params ) {
        this.events = params.events;
        this.item = params.item;
        this.order = params.order;
        this.sortOrder = params.sortOrder;
        this.parent = params.parent;
        this.container = this.parent.container;
        this.name = params.name;
        this.color = params.color,
        this.x = params.x;
        this.y = params.y;
        this.h = params.h * this.parent.scale;
        this.w = params.w * this.parent.scale;
        this.image = params.item.image;
        this.startX = this.x;
        this.startY = this.y;
        this.disabled=params.disabled || false;
        if (this.image!==undefined) {
            this.w = this.image.width * this.parent.scale;
            this.h = this.image.height * this.parent.scale;
        }
        OU.activity.Drag2Table.prototype.Item.prototype.render = function () {
            var ctx = this.parent.itemsLayer.context,bg,
            sF = (this.draggable.dragId!=0 ? 1.4 : 1)*this.parent.scale,
            x = this.x - (this.w * (sF - 1) / 2),
            y = this.y - (this.h * (sF - 1) / 2),
            w = this.w * sF,
            h = this.h * sF;
            if (this.image!==undefined) {
                ctx.drawImage(this.image, x, y, w, h);
            }
            else {
                if(this.hasFocus) {
                    bg = {
                        "borderCol":"#444",
                        "col":'#c00',
                        "radius":h / 2
                    };
                }
                else {
                    bg = {
                        "borderCol":"#444",
                        "col":this.color,
                        "radius":h / 2
                    };
                }
                new OU.util.DynText({
                    "txt":this.name,
                    "x":x,
                    "y":y,
                    "w":w,
                    "h":h,
                    "context":ctx,
                    background:bg,
                    fontWeight:"bold",
                    fontSize:20*OU.dpr
                });
            }
            this.parent.itemsLayer.events.clickable.push(this.draggable);
        };
        OU.activity.Drag2Table.prototype.Item.prototype.enable = function () {
            this.draggable.disabled=false;
        };
        OU.activity.Drag2Table.prototype.Item.prototype.disable = function () {
            this.draggable.disabled=true;
        };
        OU.activity.Drag2Table.prototype.Item.prototype.focus = function () {
            this.hasFocus = true;
            this.render();
            return false;
        };
        OU.activity.Drag2Table.prototype.Item.prototype.blur = function () {
            this.hasFocus = false;
            this.render();
            return false;
        };
        OU.activity.Drag2Table.prototype.Item.prototype.arrowKey = function (k) {
            switch(k){
                case KEY_UP:
                    this.move(this.x,this.y-10);
                    break;
                case KEY_DOWN:
                    this.move(this.x,this.y+10);
                    break;
                case KEY_LEFT:
                    this.move(this.x-10,this.y);
                    break;
                case KEY_RIGHT:
                    this.move(this.x+10,this.y);
                    break;
            }
            this.parent.doRender=true;
        };
        OU.activity.Drag2Table.prototype.Item.prototype.startDrag = function ( p ) {
            //sort the items so that the latest dragged is at the end of the queue, therefore render last next time (on top)
            p.me.parent.visibles.sort(function ( a, b ) {
                return b.draggable.dragId - a.draggable.dragId;
            });
            p.me.parent.view = 'norm';
            p.me.parent.doRender = true;
        };
        OU.activity.Drag2Table.prototype.Item.prototype.endDrag = function ( p ) {
            p.me.parent.doRender = true;
        };
        OU.activity.Drag2Table.prototype.Item.prototype.newPos = function ( p ) {
            p.me.x = p.x;
            p.me.y = p.y;
            p.me.doRender = true;
        };
        OU.activity.Drag2Table.prototype.Item.prototype.move = function ( x, y ) {
            this.x = x;
            this.y = y;
            this.draggable.x = x;
            this.draggable.y = y;
        };
        this.draggable = new OU.util.Draggable({
            "me":this,
            "events":this.parent.itemsLayer.events,
            "x":this.x,
            "y":this.y,
            "h":this.h,
            "w":this.w,
            "onStart":this.startDrag,
            "onEnd":this.endDrag,
            "onMove":this.newPos,
            disabled:this.disabled
        });

        OU.base(this, params);
    };
    OU.inherits(OU.activity.Drag2Table.prototype.Item, OU.util.Tabbable);

    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.Drag2Table, OU.util.Activity);
