/**
 * @fileOverview PeriodicTable -interactive Periodic Table
 *
 * @param {JSON array} data Data set
 *
 * @author Kevin Quick
 * @author Refactored by Nigel Clarke <nigel.clarke@pentahedra.com>
 *
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.PopUpInfo');
OU.require('OU.util.Instruction');
OU.require('OU.util.Layer');
/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.PeriodicTable = function ( data, instance, controller ) {
    OU.activity.PeriodicTable.prototype.canvasView = function () {
        var i, buttonY, buttonX, element, buttonColour, useW, useH, tempButtonWidth, tempHeight,
            tempWidth, elementButtonWidth, numPairs;
        this.config = {
            spacer:2,
            fBlockSpacer:40
        };
        this.bgLayer = new OU.util.Layer({
            container:this
        });
        this.bgLayer.context.gradRect();
        useW = this.w * .9;
        useH = this.h * .9;
        tempButtonWidth = (useW - (useW / 1000) * this.config.spacer * (19 - 1)) / 19;
        tempHeight = 10 * tempButtonWidth + 9 * (useH / 1000) * this.config.spacer + (useH / 1000) * this.config.fBlockSpacer;
        if (tempHeight > useH) {
            tempButtonWidth = (useH - 9 * (useH / 1000) * this.config.spacer - (useH / 1000) * this.config.fBlockSpacer) / 10;
            tempWidth = tempButtonWidth * 19 + 18 * (useH / 1000) * this.config.spacer;
            this.boundary = {
                x:(this.w - tempWidth) / 2,
                y:this.h * .05,
                w:tempWidth,
                h:useH,
                scale:tempWidth / 1000
            };
        }
        else {
            tempHeight = 10 * tempButtonWidth + 9 * (useW / 1000) * this.config.spacer + (useW / 1000) * this.config.fBlockSpacer;
            this.boundary = {
                x:this.w * .05,
                y:(this.h - tempHeight) / 2,
                w:useW,
                h:tempHeight,
                scale:useW / 1000
            };
        }
        this.elementsLayer = new OU.util.Layer({
            container:this,
            hasEvents:true
        });
        elementButtonWidth = (this.boundary.w - this.boundary.scale * this.config.spacer * (19 - 1)) / 19;
        numPairs = this.data.elements.length;
        this.buttons = [];
        for (i = 0; i < numPairs; i++) {
            element = this.data.elements[i];
            if (element.atomicNumber > 56 && element.atomicNumber < 72) {
                buttonX = (element.group + element.atomicNumber - 57) * (elementButtonWidth + this.boundary.scale * this.config.spacer) + this.boundary.x;
                buttonY = 8 * (elementButtonWidth + this.boundary.scale * this.config.spacer) + this.boundary.y + this.boundary.scale * this.config.fBlockSpacer;
            } else if (element.atomicNumber > 88 && element.atomicNumber < 104) {
                buttonX = (element.group + element.atomicNumber - 89) * (elementButtonWidth + this.boundary.scale * this.config.spacer) + this.boundary.x;
                buttonY = 9 * (elementButtonWidth + this.boundary.scale * this.config.spacer) + this.boundary.y + this.boundary.scale * this.config.fBlockSpacer;
            }
            else {
                buttonX = (element.group) * (elementButtonWidth + this.boundary.scale * this.config.spacer) + this.boundary.x;
                buttonY = (element.period) * (elementButtonWidth + this.boundary.scale * this.config.spacer) + this.boundary.y;
            }
            if (element.classification=="Nonmetal") {
                buttonColour = "226,58,100";
            } else if (element.classification=="Alkali Metal") {
                buttonColour = "174,207,108";
            } else if (element.classification=="Alkaline-Earth Metal") {
                buttonColour = "141,187,54";
            } else if (element.classification=="Semimetal") {
                buttonColour = "180,113,179";
            } else if (element.classification=="Halogen") {
                buttonColour = "228,108,140";
            } else if (element.classification=="Inert Gas") {
                buttonColour = "244,160,180";
            } else if (element.classification=="Transition Metal") {
                buttonColour = "56,100,186";
            } else if (element.classification=="Post-Transition Metal") {
                buttonColour = "147,56,145";
            } else if (element.classification=="Lanthanide") {
                buttonColour = "107,134,195";
            } else if (element.classification=="Actinide") {
                buttonColour = "145,167,213";
            }
            else {
                buttonColour = "200,200,200";
            }
            this.buttons[i] = new this.Button({
                txt:element.symbol,
                x:buttonX,
                y:buttonY,
                w:elementButtonWidth,
                h:elementButtonWidth,
                onClick:this.buttonClicked,
                onClickParam:{
                    choice:i,
                    me:this
                },
                colour:'#FFF',
                background:{
                    RGB:buttonColour
                },
                container:this
            });
        }
        this.render();
    };
    OU.activity.PeriodicTable.prototype.labels = function () {
        var i, buttonX, buttonY, labelButtonX, labelButtonY,
            elementButtonWidth = (this.boundary.w - this.boundary.scale * this.config.spacer * (19 - 1)) / 19,
            ctx = this.elementsLayer.context;
        buttonX = 3 * (elementButtonWidth + this.boundary.scale * this.config.spacer) + this.boundary.x;
        buttonY = 6 * (elementButtonWidth + this.boundary.scale * this.config.spacer) + this.boundary.y;
        ctx.fillStyle = "#6b86c3";
        ctx.fillRect(buttonX, buttonY, elementButtonWidth, elementButtonWidth);
        buttonY = 7 * (elementButtonWidth + this.boundary.scale * this.config.spacer) + this.boundary.y;
        ctx.fillStyle = "#91a7d5";
        ctx.fillRect(buttonX, buttonY, elementButtonWidth, elementButtonWidth);
        ctx.strokeStyle = "#CCCCCC";
        ctx.lineWidth = 1;
        ctx.textAlign = 'right';
        ctx.font = 'bold ' + this.boundary.scale * 16 + 'px ' + OU.theme.font;
        ctx.fillStyle = '#999';
        labelButtonX = 3 * (elementButtonWidth + this.boundary.scale * this.config.spacer) + this.boundary.x - 10 * this.boundary.scale;
        labelButtonY = 8 * (elementButtonWidth + this.boundary.scale * this.config.spacer) + this.boundary.y + this.boundary.scale * this.config.fBlockSpacer + elementButtonWidth / 2;
        ctx.fillText("Lanthanides", labelButtonX, labelButtonY);
        labelButtonY += elementButtonWidth + this.boundary.scale * this.config.spacer;
        ctx.fillText("Actinides", labelButtonX, labelButtonY);
        for (i = 0; i < 18; i++) {
            buttonX = (i + 1) * (elementButtonWidth + this.boundary.scale * this.config.spacer) + this.boundary.x;
            buttonY = this.boundary.y;
            ctx.beginPath();
            ctx.strokeRect(buttonX, buttonY, elementButtonWidth, elementButtonWidth / 2);
            ctx.closePath();
            ctx.stroke();
            ctx.textAlign = 'center';
            ctx.font = 'bold ' + (this.boundary.scale * 15) + 'px ' + OU.theme.font;
            ctx.fillStyle = '#BBBBBB';
            ctx.fillText(i + 1, buttonX + elementButtonWidth / 2, buttonY + this.boundary.scale * 12, elementButtonWidth);
        }
        for (i = 0; i < 7; i++) {
            buttonX = this.boundary.x;
            buttonY = (i + 1) * (elementButtonWidth + this.boundary.scale * this.config.spacer) + this.boundary.y;
            ctx.beginPath();
            ctx.strokeRect(buttonX, buttonY, elementButtonWidth / 2, elementButtonWidth);
            ctx.closePath();
            ctx.stroke();
            ctx.textAlign = 'center';
            ctx.font = 'bold ' + (this.boundary.scale * 15) + 'px ' + OU.theme.font;
            ctx.fillStyle = '#BBBBBB';
            ctx.fillText(i + 1, buttonX + elementButtonWidth / 4, buttonY + elementButtonWidth / 2);
        }
    };
    OU.activity.PeriodicTable.prototype.Button = function ( params ) {
        params.propTextHeight = .5;
        params.fontWeight = 'normal';
        if (params.background===undefined)
            params.background = {};
        params.background.borderCol = '#444';
        params.background.radius = 3;
        params.layer = params.container.elementsLayer;
        return new OU.util.Button(params);
    };
    OU.activity.PeriodicTable.prototype.buttonClicked = function ( p ) {
        var events = p.me.elementsLayer.events,
            infoText = '<h1>' + p.me.data.elements[p.choice].name + ' (' + p.me.data.elements[p.choice].symbol + ')</h1>';
        if (p.me.data.elements[p.choice].subClassification!='') {
            infoText += '<p>Classification: ' + p.me.data.elements[p.choice].classification + ' (' + p.me.data.elements[p.choice].subClassification + ')</p>';
        }
        else {
            infoText += '<p>Classification: ' + p.me.data.elements[p.choice].classification + '</p>';
        }
        infoText += '<p>Atomic Number: ' + p.me.data.elements[p.choice].atomicNumber + '</p>';
        infoText += '<p>Group, Period, Block: ' + p.me.data.elements[p.choice].group + ', ';
        infoText += p.me.data.elements[p.choice].period + ', ';
        infoText += p.me.data.elements[p.choice].tableBlock + '</p>';
        infoText += '<p>Electron Configuration: ' + p.me.data.elements[p.choice].electronConfig + '</p>';
        infoText += '<p>Relative Atomic Mass: ' + p.me.data.elements[p.choice].atomicMass + '</p>';
        infoText += '<p>Density (20 °C): ' + p.me.data.elements[p.choice].density + '</p>';
        infoText += '<p>Atomic Radius: ' + p.me.data.elements[p.choice].atomicRadius + '</p>';
        infoText += '<p>Oxidation States: ' + p.me.data.elements[p.choice].oxidationStates + '</p>';
        infoText += '<p>Electronegativity: ' + p.me.data.elements[p.choice].electronegativity + '</p>';
        infoText += '<p>Ionization Energy: ' + p.me.data.elements[p.choice].ionizationEnergy + '</p>';
        infoText += '<p>Melting Point: ' + p.me.data.elements[p.choice].meltingPoint + '</p>';
        infoText += '<p>Boiling Point: ' + p.me.data.elements[p.choice].boilingPoint + '</p>';
        events.pressed = false;
        events.touched = false;
        p.me.popUp = new OU.util.PopUpInfo({
            container:p.me,
            txt:infoText,
            x:p.me.w * .20,
            y:p.me.h * .12,
            w:p.me.w * .6,
            h:p.me.h * .76,
            onClose:function () {
            }
        });
    };
    OU.activity.PeriodicTable.prototype.resize = function () {
        OU.activity.PeriodicTable.superClass_.resize.call(this); // call the parent class resize 
        var i, element, buttonX, buttonY, useH, useW, tempButtonWidth, tempHeight, tempWidth,
            elementButtonWidth, numPairs;
        if (this.popUp!==undefined)
            this.popUp.close();
        this.bgLayer.resize();
        this.bgLayer.context.gradRect();
        this.elementsLayer.resize();
        useW = this.w * .9;
        useH = this.h * .9;
        tempButtonWidth = (useW - (useW / 1000) * this.config.spacer * (19 - 1)) / 19;
        tempHeight = 10 * tempButtonWidth + 9 * (useH / 1000) * this.config.spacer + (useH / 1000) * this.config.fBlockSpacer;
        if (tempHeight > useH) {
            tempButtonWidth = (useH - 9 * (useH / 1000) * this.config.spacer - (useH / 1000) * this.config.fBlockSpacer) / 10;
            tempWidth = tempButtonWidth * 19 + 18 * (useH / 1000) * this.config.spacer;
            this.boundary = {
                x:(this.w - tempWidth) / 2,
                y:this.h * .05,
                w:tempWidth,
                h:useH,
                scale:tempWidth / 1000
            };
        }
        else {
            tempHeight = 10 * tempButtonWidth + 9 * (useW / 1000) * this.config.spacer + (useW / 1000) * this.config.fBlockSpacer;
            this.boundary = {
                x:this.w * .05,
                y:(this.h - tempHeight) / 2,
                w:useW,
                h:tempHeight,
                scale:useW / 1000
            };
        }
        elementButtonWidth = (this.boundary.w - this.boundary.scale * this.config.spacer * (19 - 1)) / 19;
        numPairs = this.data.elements.length;
        for (i = 0; i < numPairs; i++) {
            element = this.data.elements[i];
            if (element.atomicNumber > 56 && element.atomicNumber < 72) {
                buttonX = (element.group + element.atomicNumber - 57) * (elementButtonWidth + this.boundary.scale * this.config.spacer) + this.boundary.x;
                buttonY = 8 * (elementButtonWidth + this.boundary.scale * this.config.spacer) + this.boundary.y + this.boundary.scale * this.config.fBlockSpacer;
            } else if (element.atomicNumber > 88 && element.atomicNumber < 104) {
                buttonX = (element.group + element.atomicNumber - 89) * (elementButtonWidth + this.boundary.scale * this.config.spacer) + this.boundary.x;
                buttonY = 9 * (elementButtonWidth + this.boundary.scale * this.config.spacer) + this.boundary.y + this.boundary.scale * this.config.fBlockSpacer;
            }
            else {
                buttonX = (element.group) * (elementButtonWidth + this.boundary.scale * this.config.spacer) + this.boundary.x;
                buttonY = (element.period) * (elementButtonWidth + this.boundary.scale * this.config.spacer) + this.boundary.y;
            }
            this.buttons[i].resize({
                x:buttonX,
                y:buttonY,
                w:elementButtonWidth,
                h:elementButtonWidth
            });
        }
        this.render();
    };
    OU.activity.PeriodicTable.prototype.render = function () {
        var i, numPairs = this.data.elements.length;
        this.elementsLayer.clear();
        this.labels();
        for (i = numPairs; i--;) {
            this.buttons[i].render();
        }
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.PeriodicTable, OU.util.Activity);
