/**
 * OU.activity.QuarkMachine Object
 * @fileOverview The Quark Fruit Machine
 *
 * @param {JSON array} data Data as included in the data/data.js file
 *
 * @author Martin Donnelly
 */
OU.require('OU.util.DynText', '../library/js/util/dynText.js');
OU.require('OU.util.PopUpInfo');
OU.require('OU.util.Layer', '../library/js/util/layer.js');
OU.require('OU.util.ImageLoader', '../library/js/util/imageloader.js');
/**
 * @class
 * @extends OU.util.Activity
 *  */
OU.activity.QuarkMachine = function ( data, instance, controller ) {
    OU.activity.QuarkMachine.prototype.canvasView = function () {
        this.clickable = new OU.util.TypedArray();
        this.aRGB = [];
        this.aColour = [];
        this.staticClicked = [];
        this.hasImages = false;
        this.quarkMode = this.data.quarkMode || 1;
        this.doRender = true;
        this.startB = -5.0;
        this.startC = -10.0;
        this.reelPosA = 0;
        this.reelPosB = 1;
        this.reelPosC = 2;
        this.doneSpun = true;
        this.subStep = 0;
        this.doSpinA = true;
        this.doSpinB = false;
        this.doSpinC = false;
        this.reelStepA = 0;
        this.reelStepB = 0;
        this.reelStepC = 0;
        this.unlocked = true;
        this.touched = false;
        this.touchReady = false;
        this.friction = 0.97;
        this.dragBox = {};
        this.infoDisp = false;
        switch(this.quarkMode) {
            case 3:
                this.reelMode = [1, 2, 0];
                break;
            case 2:
                this.reelMode = [2, 2, 2];
                break;
            case 1:
            default:
                this.reelMode = [1, 1, 1];
                break;
        }
        this.reelLayout = [
            [5, 0, 1],
            [0, 1, 2],
            [1, 2, 3],
            [2, 3, 4],
            [3, 4, 5],
            [4, 5, 0]
        ];
        this.active = ['#737328', '#665E26', '#5A4A25', '#4E3623', '#422222'];
        this.q = [];
        this.totCharge = 0;
        this.antimatter = 0;
        this.top = 0;
        // create Canvas & Context
        OU.initPage();
        this.bgLayer = new OU.util.Layer({

            container:this

        });
        this.quarkLayer = new OU.util.Layer({
            container:this,
            hasEvents:true
        });
        this.renderLoop();
    };
    /**
     * Called by the controller when the activity is removed. Turns off the renderer
     */
    OU.activity.QuarkMachine.prototype.remove = function () {
        OU.activity.QuarkMachine.superClass_.remove.call(this); // call the superclass method

        // stop the renderer
        this.doRender = false;
        //g'bye
    };
    /**
     * Called automatically when window is resized / rotated..
     */
    OU.activity.QuarkMachine.prototype.resize = function () {
        OU.activity.QuarkMachine.superClass_.resize.call(this); // call the parent class resize
        this.bgLayer.resize();
        this.bigBadGrad = null;
        this.backgroundG = null;
        this.resultBedG = null;
    };
    /**
     * Main loop. Handles user interaction
     */
    OU.activity.QuarkMachine.prototype.renderLoop = function () {
        var self = this,tx,ty, db = this.dragBox;
        if (!this.doRender)
            return;
        tx = this.quarkLayer.events.x;
        ty = this.quarkLayer.events.y;
        if (!this.quarkLayer.events.pressed && !this.quarkLayer.events.touched) {
            if (this.triggerSettings!=null) {
                if (this.triggerSettings.pos >= this.triggerSettings.fire - this.triggerSettings.h && this.unlocked) {
                    this.unlocked = false;
                    this.spinQuarks();
                }
                if (this.triggerSettings.pos > 0) this.triggerSettings.pos /= 2;
            }
            this.prevY = null;
        }
        else {
            if (this.unlocked) {
                if (((tx > db.x && tx < (db.x + db.w)) && (ty > db.y && ty < (db.y + db.h)))) {
                    this.touched = true;
                    if (this.prevY==null)
                        this.prevY = this.quarkLayer.events.y;
                    this.triggerSettings.pos = ty - this.triggerSettings.top * 0.97;
                    if (this.triggerSettings.pos >= this.triggerSettings.fire)
                        this.triggerSettings.pos = this.triggerSettings.fire;
                }
            }
        }
        this.render();
        if (this.doRender) {
            // this.otherrender();
            setTimeout(function () {
                self.renderLoop();
            }, 40);
        }
    };
    /**
     * Unlocks the reels and slider so that they can be spun.
     */
    OU.activity.QuarkMachine.prototype.unlock = function () {
        this.unlocked = true;
    };
    /**
     * Draws the background for the pull slider.
     * @param {JSON object} p Contains {x, y, w, h, ctx}
     */
    OU.activity.QuarkMachine.prototype.drawSlideBed = function ( p ) {
        var x, y, w, h, ctx, sW;
        x = p.x;
        y = p.y;
        w = p.w;
        h = p.h;
        ctx = p.ctx;
        sW = w / 2;
        ctx.strokeStyle = "rgba(0,0,0,0.75)";
        ctx.fillStyle = "#151515";
        ctx.roundRect(x + (sW / 2), y, sW, h, 5);
        ctx.fill();
        ctx.stroke();
        ctx.strokeStyle = "rgba(255,255,255,0.1)";
        ctx.roundRect(x + (sW / 2) - 2, y - 2, sW + 4, h + 4, 5);
        ctx.fill();
        ctx.stroke();
        ctx.save();
        ctx.translate(x + w, y);
        ctx.rotate(Math.PI / 2);
        new OU.util.DynText({
            txt:"Slide to spin",
            x:-1,
            y:1,
            w:h,
            h:w,
            fontSize:30,
            context:ctx,
            colour:"000000",
            align:'center',
            noBreak:true  });
        new OU.util.DynText({
            txt:"Slide to spin",
            x:0,
            y:0,
            w:h,
            h:w,
            fontSize:30,
            context:ctx,
            colour:(this.unlocked ? "rgb(128,128,128)" : "rgb(51,51,51)"),
            align:'center',
            noBreak:true  });
        ctx.restore();
    };
    /**
     * Draws the slider drag point
     * @param {JSON object} p Contains {x, y, w, h, ctx}
     */
    OU.activity.QuarkMachine.prototype.drawSlideDrag = function ( p ) {
        var x, y, w, h, ctx, sW, gO, hW, hH, hHb, a, b, c, d, e, f;
        x = p.x;
        y = p.y;
        w = p.w;
        h = p.h;
        ctx = p.ctx;
        sW = w / 2;
        gO = p.gO;
        if (!this.triggerSettings) {
            this.triggerSettings = {};
            this.triggerSettings.pos = 0;
            this.triggerSettings.top = gO.y;
            this.triggerSettings.h = h / 5;
            this.triggerSettings.fire = h - (h / 5);
        }
        y += this.triggerSettings.pos;
        hW = sW * 0.4;
        hH = (h / 5) * 0.4;
        hHb = hH / 3;
        a = x + (sW / 2) + ((sW - hW) / 2);
        b = y + (((h / 5) - hH) / 2);
        c = (hHb / 2);
        d = (hHb * 2);
        e = b + c;
        f = a + hW;
        ctx.strokeStyle = "rgba(128,128,128,0.75)";
        ctx.fillStyle = "#f9f9f9";
        ctx.roundRect(x + (sW / 2), y, sW, h / 5, 5);
        ctx.fill();
        ctx.stroke();
        ctx.fillStyle = "rgba(128,128,128,0.25)";
        ctx.fillRect(a, e, hW, c || 2);
        ctx.fillRect(a, e + hHb, hW, c || 2);
        ctx.fillRect(a, e + d, hW, c || 2);
        ctx.strokeStyle = "rgba(128,128,128,0.75)";
        ctx.beginPath();
        ctx.moveTo(f, e);
        ctx.lineTo(a, e);
        ctx.lineTo(a, b + c + c);
        ctx.moveTo(f, e + hHb);
        ctx.lineTo(a, e + hHb);
        ctx.lineTo(a, e + hHb + c);
        ctx.moveTo(f, e + d);
        ctx.lineTo(a, e + d);
        ctx.lineTo(a, e + d + c);
        ctx.stroke();
        this.dragBox.x = gO.x + x + (sW / 2);
        this.dragBox.y = gO.y + y;
        this.dragBox.w = sW;
        this.dragBox.h = h / 5;
    };
    /**
     * Draws the background for the results
     * @param {JSON object} p Contains {x, y, w, h, ctx}
     */
    OU.activity.QuarkMachine.prototype.drawResultBed = function ( p ) {
        var x, y, w, h, ctx, sW;
        x = p.x;
        y = p.y;
        w = p.w;
        h = p.h;
        ctx = p.ctx;
        sW = p.sW;
        if (!this.resultBedG) {
            this.resultBedG = ctx.createLinearGradient(x, y, x, y + h);
            this.resultBedG.addColorStop(0, '#ffffff');
            this.resultBedG.addColorStop(0.5, '#eeeeee');
            this.resultBedG.addColorStop(0.51, '#ffffff');
            this.resultBedG.addColorStop(1, '#eeeeee');
        }
        ctx.fillStyle = this.resultBedG;
        ctx.fillRect(x + 1, y + 1, w - 2, h - 2);
        ctx.strokeStyle = '#A0A0A0';
        ctx.beginPath();
        ctx.moveTo(1, y + (h / 2));
        ctx.lineTo(w - 1, y + (h / 2));
        ctx.moveTo(sW, y + 1);
        ctx.lineTo(sW, y + (h / 2));
        ctx.moveTo(2 * sW, y + 1);
        ctx.lineTo(2 * sW, y + (h / 2));
        ctx.stroke();
        // chisel out the box...
        ctx.strokeStyle = "rgba(0,0,0,0.75)";
        ctx.strokeRect(x - 1, y - 1, w + 2, h + 2);
        ctx.strokeStyle = "rgba(255,255,255,0.1)";
        ctx.strokeRect(x - 2, y - 2, w + 4, h + 4);
    };
    /**
     *
     */
    OU.activity.QuarkMachine.prototype.showInfo = function () {
        var self = this;
        new OU.util.PopUpInfo({
            container:this,
            txt:"<table width='100%' border='1'> <tr> <th colspan='2' width='50%'>Leptons</th> <th colspan='2' width='50%'>Quarks</th> </tr> <tr> <td width='25%'>electron</td> <td width='25%'>electron neutrino</td> <td width='25%'>up, charm, top quarks</td> <td width='25%'>down, strange, bottom quarks</td> </tr> <tr> <td>muon</td> <td>muon neutrino</td> <td>electric charge = +2e/3</td> <td>electric charge = -e/3</td> </tr> <tr> <td>tauon</td> <td>tauon neutrino</td> <td colspan='2'/> </tr> <tr> <td>electric charge = -e</td> <td>electric charge = 0</td> <td colspan='2'/> </tr> <tr> <td colspan='2'/> <th colspan='2'>Hadrons</th> </tr> <tr> <td colspan='2'/> <td>Baryons</td> <td>Mesons</td> </tr> <tr> <td colspan='2'/> <td>composed of three quarks</td> <td>composed of one quark and one antiquark</td> </tr> </table>",
            x:this.w * .05,
            y:this.h * .15,
            w:this.w * .9,
            h:this.h * .7,
            onClose:function () {
                self.infoDisp = false;
            }
        });
    };
    /**
     * The main renderer.
     */
    OU.activity.QuarkMachine.prototype.render = function () {
        var self = this, boxW, boxH, boxX, boxY, r, ctx, reelPos = [], i, gO = {x:0, y:0};
        ctx = this.quarkLayer.context;
        boxW = this.w * 0.9;
        boxH = boxW;
        if (boxH > this.h) {
            // wrong size.. redo.
            boxW = this.h * 0.9;
            boxH = boxW;
        }
        boxX = (this.w - boxW) / 2;
        boxY = (this.h - boxH) / 2;
        var sW = boxW / 3.75;
        var sH = boxH / 20;
        var reelH = sH * 12;
        var resultbed = {x:0, y:sH * 13, w:sW * 3, h:sH * 7, ctx:ctx, sW:sW};
        var slideBed = {x:sW * 3, y:0, h:reelH, w:boxW - (sW * 3), ctx:ctx};
        var lensBox = {x:0, y:4 * sH, w:(sW * 3), h:4 * sH  };
        reelPos[0] = 0;
        reelPos[1] = sW;
        reelPos[2] = sW * 2;
        if (!this.backgroundG) {
            this.backgroundG = ctx.createRadialGradient(this.w / 2, this.h / 2, 0, this.w / 2, this.h / 2, this.w);
            this.backgroundG.addColorStop(0, '#454545');
            this.backgroundG.addColorStop(1, '#030303');
        }
        ctx.fillStyle = this.backgroundG;
        ctx.fillRect(0, 0, this.w, this.h);
        ctx.strokeStyle = '#ff0000';
        // guide boxes
        this.infoB = new OU.util.Button({
            txt:"",
            padding:0,
            verticalPadding:0,
            layer:this.quarkLayer,
            background:this.bBack,
            x:this.w - 42, y:this.h - 42, w:32, h:32,
            onClick:function () {
                if (!self.infoDisp) {
                    self.infoDisp = true;
                    //console.log('hello');
                    self.showInfo();
                }
            },
            glyph:{
                type:'infoIcon',
                align:'center',
                w:30,
                h:30
            }
        });
        ctx.save();
        ctx.translate(boxX, boxY);
        gO.x = boxX;
        gO.y = boxY;
        var gb = false;
        if (gb) {
            ctx.strokeRect(0, 0, boxW, boxH);
            ctx.strokeRect(reelPos[0], 0, sW, boxH);
            ctx.strokeRect(reelPos[1], 0, sW, boxH);
            ctx.strokeRect(reelPos[2], 0, sW, boxH);
            ctx.strokeStyle = '#00ffff';
            for (var t = 0; t < 20; t++) {
                ctx.strokeRect(0, t * sH, boxW, sH);
            }
            ctx.strokeStyle = '#ffff00';
            ctx.strokeRect(0, 0, sW, reelH);
            ctx.strokeRect(sW, 0, sW, reelH);
            ctx.strokeRect(sW * 2, 0, sW, reelH);
            ctx.strokeStyle = '#ff00ff';
            ctx.strokeRect(resultbed.x, resultbed.y, resultbed.w, resultbed.h);
        }
        ctx.strokeStyle = "rgba(0,0,0,0.75)";
        ctx.strokeRect(0, -1, (sW * 3) - 1, reelH + 1);
        ctx.strokeStyle = "rgba(255,255,255,0.1)";
        ctx.strokeRect(-2, -2, (sW * 3) + 2, reelH + 4);
        // draw the lens
        this.drawResultBed(resultbed);
        this.drawSlideBed(slideBed);
        slideBed.gO = gO;
        this.drawSlideDrag(slideBed);
        if (this.doneSpun) {
            this.drawReel({x:reelPos[0], y:0, w:sW, h:reelH, r:sW, offset:this.reelPosA, mode:this.reelMode[0]});
            this.drawReel({x:reelPos[1], y:0, w:sW, h:reelH, r:sW, offset:this.reelPosB, mode:this.reelMode[1]});
            this.drawReel({x:reelPos[2], y:0, w:sW, h:reelH, r:sW, offset:this.reelPosC, mode:this.reelMode[2]});
            for (i = 0; i < 3; i++) {
                if (this.reelMode[i]==1) {
                    if (this.q[i] < 3) this.displayValues({x:i * sW, y:resultbed.y, h:resultbed.h / 2, w:sW, mode:2});
                    else this.displayValues({x:i * sW, y:resultbed.y, h:resultbed.h / 2, w:sW, mode:1});
                }
                else if (this.reelMode[i]==2) {
                    if (this.q[i] < 3) this.displayValues({x:i * sW, y:resultbed.y, h:resultbed.h / 2, w:sW, mode:4});
                    else this.displayValues({x:i * sW, y:resultbed.y, h:resultbed.h / 2, w:sW, mode:3});
                }
                else {
                }
            }
            this.displayCharge(resultbed);
        }
        else {
            if (1==this.subStep % 2) {
                if (this.reelStepA < 25) {
                    this.reelPosA += 1;
                    this.reelStepA += 1;
                }
                else {
                    if ((this.reelPosA % 6)!=this.q[0] % 6) this.reelPosA += 1;
                    else this.doSpinA = false;
                }
                if (this.reelStepB < 25) {
                    if (this.startB==0) {
                        this.reelPosB += 1;
                        this.reelStepB += 1;
                        this.doSpinB = true;
                    }
                    else this.startB++;
                }
                else {
                    if ((this.reelPosB % 6)!=this.q[1] % 6) this.reelPosB += 1;
                    else {
                        this.doSpinB = false;
                        if (this.quarkMode==3) this.doneSpun = true;
                    }
                }
                if (this.quarkMode!=3) {
                    if (this.reelStepC < 25) {
                        if (this.startC==0) {
                            this.reelPosC += 1;
                            this.reelStepC += 1;
                            this.doSpinC = true;
                        }
                        else this.startC++;
                    }
                    else {
                        if ((this.reelPosC % 6)!=this.q[2] % 6) this.reelPosC += 1;
                        else {
                            this.doneSpun = true;
                            this.doSpinC = false;
                        }
                    }
                }
            }
            if (this.doSpinA)
                this.drawReel({x:reelPos[0], y:0, w:sW, h:reelH, r:sW, offset:this.reelPosA, vo:this.subStep % 2, mode:this.reelMode[0]});
            else this.drawReel({x:reelPos[0], y:0, w:sW, h:reelH, r:sW, offset:this.reelPosA, mode:this.reelMode[0]});
            if (this.doSpinB) this.drawReel({x:reelPos[1], y:0, w:sW, h:reelH, r:sW, offset:this.reelPosB, vo:this.subStep % 2, mode:this.reelMode[1]});
            else this.drawReel({x:reelPos[1], y:0, w:sW, h:reelH, r:sW, offset:this.reelPosB, mode:this.reelMode[1]});
            if (this.doSpinC) this.drawReel({x:reelPos[2], y:0, w:sW, h:reelH, r:sW, offset:this.reelPosC, vo:this.subStep % 2, mode:this.reelMode[2]});
            else this.drawReel({x:reelPos[2], y:0, w:sW, h:reelH, r:sW, offset:this.reelPosC, mode:this.reelMode[2]});
            if (!this.doneSpun) {
            }
            else {
                setTimeout(function () {
                    self.unlock();
                }, 1000)
            }
            this.subStep++;
        }
        if (!this.bigBadGrad) {
            this.bigBadGrad = ctx.createLinearGradient(0, 0, 0, reelH);
            this.bigBadGrad.addColorStop(0, "rgba(0,0,0,0.9)");
            this.bigBadGrad.addColorStop(0.10, "rgba(0,0,0,0.6)");
            this.bigBadGrad.addColorStop(0.27, 'rgba(255,255,255,0)');
            this.bigBadGrad.addColorStop(0.34, "rgba(128,128,128,0.6)");
            this.bigBadGrad.addColorStop(0.34, "rgba(209,255,255,0.12)");
            this.bigBadGrad.addColorStop(0.50, "rgba(209,255,255,0.26)");
            this.bigBadGrad.addColorStop(0.50, "rgba(156,211,211,0.16)");
            this.bigBadGrad.addColorStop(0.67, 'rgba(156,211,211,0.26)');
            this.bigBadGrad.addColorStop(0.67, "rgba(128,128,128,0.6)");
            this.bigBadGrad.addColorStop(0.73, 'rgba(255,255,255,0)');
            this.bigBadGrad.addColorStop(0.9, "rgba(0,0,0,0.6)");
            this.bigBadGrad.addColorStop(1, "rgba(0,0,0,0.9)");
        }
        ctx.fillStyle = this.bigBadGrad;
        ctx.fillRect(0, 0, sW * 3, reelH);
        ctx.strokeStyle = "rgba(156,211,211,0.56)";
        ctx.strokeRect(lensBox.x - 1, lensBox.y + 1, lensBox.w, lensBox.h);
        ctx.restore();
    };
    /**
     * Called to randomise the spin results. Starts of the whole spin process.
     */
    OU.activity.QuarkMachine.prototype.spinQuarks = function () {
        var i = 0, n = 24, qval = 0, clickable = this.quarkLayer.events.clickable;
        var random = function ( n ) {
            return Math.floor(Math.random() * n)
        };
        this.unlocked = false;
        this.totCharge = 0;
        this.antimatter = 0;
        clickable.length = 0;
        /* Quick Note:

         in the original application charge was calculated using -2.0 / 3.0 etc.

         This was causing stupid rounding errors when it came to selecting the correct total charge glyph to display.

         To get around this, the denominator is tossed and the numerators are added toether giving us plain integers.

         The original methods are left here commented out to show what it was doing.
         */
        switch(this.quarkMode) {
            case 2:
                this.antimatter = 1;
            case 1:
                /* Martin's crazy random generator
                 * removed as it favours lower numbers in the range, and therefore takes ages to get the full set of ansers out
                for (i = 0; i <= random(n); i++) this.q[0] = (6 + i) % 6;
                for (i = 0; i <= random(n); i++) this.q[1] = (6 + i) % 6;
                for (i = 0; i <= random(n); i++) this.q[2] = (6 + i) % 6;
                //*/
                this.q[0] = random(6);
                this.q[1] = random(6);
                this.q[2] = random(6);

                for (i = 0; i < 3; i++) {
                    if (this.q[i] < 3) {
                        // up, charm & top quarks.
                        //qval = this.antimatter ? -2.0 / 3.0 : 2.0 / 3.0;
                        qval = this.antimatter ? -2.0 : 2.0;
                        this.totCharge += qval;
                    }
                    else {
                        // down, strange & bottom quarks.
                        //qval = this.antimatter ? 1.0 / 3.0 : -1.0 / 3.0;
                        qval = this.antimatter ? 1.0 : -1.0;
                        this.totCharge += qval;
                    }
                }
                break;
            case 3:
                /* Martin's crazy random generator
                for (i = 0; i <= random(n); i++) this.q[0] = (6 + i) % 6;
                for (i = 0; i <= random(n); i++) this.q[1] = (6 + i) % 6;
                //*/
                this.q[0] = random(6);
                this.q[1] = random(6);
                if (this.q[0] < 3) {
                    // up, charm & top quarks.
                    //qval = 2.0 / 3.0;
                    qval = 2.0;
                    this.totCharge += qval;
                }
                else {
                    // down, strange & bottom quarks.
                    //qval = -1.0 / 3.0;
                    qval = -1.0;
                    this.totCharge += qval;
                }
                if (this.q[1] < 3) {
                    // up, charm & top quarks.
                    //qval = -2.0 / 3.0;
                    qval = -2.0;
                    this.totCharge += qval;
                }
                else {
                    // down, strange & bottom quarks.
                    //qval = 1.0 / 3.0;
                    qval = 1.0;
                    this.totCharge += qval;
                }
                break;
            default:
        }
        console.log(this.q[0] + ", " + this.q[1] + ", " + this.q[2]);
        // let 'em rip!
        this.doSpinA = true;
        this.reelStepA = 0;
        this.reelStepB = 0;
        this.reelStepC = 0;
        this.startB = -5.0;
        this.startC = -10.0;
        this.doneSpun = false;
    };
    /**
     * Displays the resulting values
     * @param {JSON object} params Contains {x, y, w, h}
     */
    OU.activity.QuarkMachine.prototype.displayValues = function ( params ) {
        if (params===undefined)
            params = {};
        var x, y, w, h , mode, ctx = this.quarkLayer.context, s, signs = ["", "-", "+", "+", "-"], vals = ["", "e", "2e", "e", "2e"];
        w = params.w;
        h = params.h;
        x = params.x;
        y = params.y;
        mode = params.mode || 0;
        s = w / 3;
        ctx.fillStyle = "#000000";
        ctx.fillRect(x + s, y + s - 2, s, 2);
        new OU.util.DynText({
            txt:(signs[mode]),
            x:x,
            y:y,
            w:s,
            h:h,
            fontSize:30,
            context:ctx,
            colour:'rgba(21,21,21,0.9)',
            align:'right',
            fontWeight:900,
            noBreak:true  });
        new OU.util.DynText({
            txt:vals[mode],
            x:x,
            y:y,
            w:w,
            h:h / 2,
            fontSize:30,
            context:ctx,
            colour:'rgba(21,21,21,0.9)',
            align:'center',
            fontWeight:900,
            noBreak:true  });
        new OU.util.DynText({
            txt:"3",
            x:x,
            y:y + (h / 2),
            w:w,
            h:h / 2,
            fontSize:30,
            context:ctx,
            colour:'rgba(21,21,21,0.9)',
            align:'center',
            fontWeight:900,
            noBreak:true  });
    };
    /**
     * Displays the resulting charge
     * @param {JSON object} params Contains {x, y, w, h}
     */
    OU.activity.QuarkMachine.prototype.displayCharge = function ( params ) {
        if (params===undefined)
            params = {};
        var x, y, h, w, r, ctx = this.quarkLayer.context, v;
        x = params.x;
        y = params.y;
        w = params.w;
        h = params.h;
        switch(this.totCharge) {
            case -6:
                v = '-2';
                break;
            case 6: // 2
                v = '2';
                break;
            case 3: // 1
                v = '1';
                break;
            case -3: //-1
                v = '-1';
                break;
            case 0:
            default:
                v = '0';
                break;
        }
        new OU.util.DynText({
            txt:("Charge = " + v + "e" ),
            x:x,
            y:y + (h / 2),
            w:w,
            h:h / 2,
            fontSize:30,
            context:ctx,
            colour:'rgba(21,21,21,0.9)',
            align:'center',
            fontWeight:900,
            noBreak:true  });
    };
    /**
     * Displays a reel, called once for each displayed reel
     * @param {JSON object} params Contains {x, y, w, h,offset,vo,mode}
     */
    OU.activity.QuarkMachine.prototype.drawReel = function ( params ) {
        var x, y, h, w, mid, midb , r, ctx = this.quarkLayer.context, cv, offset, vo, mode, rSec, rBox, lBox, startpos, cBox, sp, t;
        x = params.x;
        y = params.y;
        w = params.w;
        h = params.h;
        offset = params.offset || 0;
        vo = params.vo || 0;
        mode = params.mode || 0;
        w -= 2;
        // silver background
        ctx.save();
        ctx.translate(x, y);
        ctx.fillStyle = "rgba(200,200,200,1)";
        ctx.fillRect(0, 0, w, h);
        ctx.save();
        ctx.beginPath();
        ctx.moveTo(0, 0);
        ctx.lineTo(w, 0);
        ctx.lineTo(w, h);
        ctx.lineTo(0, h);
        ctx.clip();
        ctx.fillStyle = "#ffffff";
        ctx.fillRect(3, y, w - 6, h);
        rSec = h / 2;
        rBox = (w / 6) * 4;
        lBox = (w - rBox) / 2;
        startpos = 0 - (rSec / 2);
        cBox = ((rSec - rBox) / 2);
        cv = this.reelLayout[(offset + 6 ) % 6];
        for (t = 0; t < 3; t++) {
            sp = startpos + cBox + (vo * (rSec / 2));
            mid = {x:lBox, y:sp, r:rBox, f:"#999999", c:true};
            midb = {x:lBox, y:2 + sp, r:rBox, c:false};
            if (mode==1) {
                switch(cv[2 - t]) {
                    case 0:
                        ctx.u(mid);
                        ctx.u(midb);
                        break;
                    case 1:
                        ctx.c(mid);
                        ctx.c(midb);
                        break;
                    case 2:
                        ctx.t(mid);
                        ctx.t(midb);
                        break;
                    case 3:
                        ctx.d(mid);
                        ctx.d(midb);
                        break;
                    case 4:
                        ctx.s(mid);
                        ctx.s(midb);
                        break;
                    case 5:
                    default:
                        ctx.b(mid);
                        ctx.b(midb);
                        break;
                }
            }
            else if (mode==2) {
                switch(cv[2 - t]) {
                    case 0:
                        ctx.antiU(mid);
                        ctx.antiU(midb);
                        break;
                    case 1:
                        ctx.antiC(mid);
                        ctx.antiC(midb);
                        break;
                    case 2:
                        ctx.antiT(mid);
                        ctx.antiT(midb);
                        break;
                    case 3:
                        ctx.antiD(mid);
                        ctx.antiD(midb);
                        break;
                    case 4:
                        ctx.antiS(mid);
                        ctx.antiS(midb);
                        break;
                    case 5:
                    default:
                        ctx.antiB(mid);
                        ctx.antiB(midb);
                        break;
                }
            }
            else {
                // turned off do nothing
            }
            startpos += rSec;
        }
        // end bit
        ctx.restore();
        ctx.restore();
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.QuarkMachine, OU.util.Activity);

