/**
 * @fileOverview ThumbController - load and traverse multiple activities using Thumbnails
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */
OU.require('OU.util.Controller');
OU.require('OU.util.DynText');
OU.require('OU.util.Layer');
OU.require('OU.util.Div');
OU.require('OU.util.Instruction');
OU.require('OU.util.ImageLoader');
/**
 * @class
 * @extends OU.util.Controller
 */
OU.activity.ThumbController = function ( controller_data, instance, controller ) {
    OU.activity.ThumbController.prototype.canvasView = function () {
        OU.activity.ThumbController.superClass_.canvasView.call(this); // call the parent class canvasView
        var self = this;
        this.thumbs = [];
        this.controlHeight = (this.data.controlHeight || 120) * OU.dpr;
        this.controlX = this.x;
        this.footerHeight = this.controlHeight;
        this.controlLayer = new OU.util.Layer({
            dontRegister:this.baseController,
            x:0,
            y:this.y + this.h - this.footerHeight,
            w:this.w,
            h:this.footerHeight,
            zIndex:OU.POP_UP_LEVEL - 1,
            container:this,
            hasEvents:true
        });
        this.imageLoader = new OU.util.ImageLoader({
            container:this,
            data:this.data,
            onLoad:function () {
                self.start();
            }
        });
    };
    OU.activity.ThumbController.prototype.start = function () {
        var i,x=0,thumb;
    
        for (i = 0; i < this.data.sections.length; i++) {
            this.thumbs.push(thumb = new OU.util.ThumbControllerThumb({
                container:this,
                ctx:this.controlLayer.context,
                id:i+1,
                x:x,
                y:0,
                section: this.data.sections[i] 
            }));
        
            x = x + thumb.w;
        }
        this.totalW=x;
        this.controlLayer.events.clickable.push(this);
        this.render();
    };        
    OU.activity.ThumbController.prototype.resize = function () {
        OU.activity.ThumbController.superClass_.resize.call(this); // call the parent class resize
        this.controlLayer.resize({
            x:0,
            y:this.y + this.h - this.footerHeight,
            w:this.w,
            h:this.footerHeight
        });
        this.render();
    };
    OU.activity.ThumbController.prototype.thumbHit = function ( x, y ) {
        for (var i =this.thumbs.length; i--;) {
            if(this.thumbs[i].isHit(x,y))
                return;
        }
    };
    OU.activity.ThumbController.prototype.isHit = function ( x, y, pressed ) {
        var damper = 1,newX;
        if (pressed)
            this.lastHit = 0;
        if (pressed && !this.midDrag) {
            this.dragStart = {
                'x':x,
                'y':y
            };
            this.midDrag = true;
            this.tap = true;
            this.tapX = x;
            this.tapY = y;
        }
        else if (pressed && this.midDrag) {
            newX = this.controlX + (x - this.dragStart.x) * damper;
            if (newX < this.x + this.w - this.totalW)
                newX = this.x + this.w - this.totalW;
            if (newX > this.x)
                newX = this.x;
            
            this.dragStart = {
                'x':x,
                'y':y
            };
            this.tap = false;
            this.controlX=newX;
            this.render();
        }
        else {
            if (this.tap) {
                this.thumbHit(this.tapX, this.tapY);
            }
            this.midDrag = false;
            this.tap = false;
        }
    };
    OU.activity.ThumbController.prototype.render = function () {
        var ctl = this.controlLayer,
        ctx = ctl.context,i,h=this.controlLayer.h;
        ctl.clear();
        this.leftArrow=(this.controlX<this.x+20);
        this.rightArrow=(this.controlX < this.x + this.w - this.controlLayer.w-20);

        for (i = this.thumbs.length; i--;) {
            this.thumbs[i].render();
        }
        ctx.save();
        ctx.fillStyle='rgba(200,0,0,0.8)';
        ctx.strokeStyle='#fff';
        ctx.lineWidth=4;
        ctx.lineCap='round';
        if(this.controlX<0) {
            ctx.beginPath();
            ctx.moveTo(25,0);
            ctx.lineTo(25,h*.16);
            ctx.lineTo(5,h*.08);
            ctx.fill();
            ctx.stroke();
        }
        if(this.controlX > this.x + this.w - this.totalW) {
            ctx.beginPath();
            ctx.moveTo(this.w-25,0);
            ctx.lineTo(this.w-25,h*.16);
            ctx.lineTo(this.w-5,h*.08);
            ctx.fill();
            ctx.stroke();
        }
        ctx.restore();
    };
    OU.base(this, controller_data, instance, controller);
}; 

if(OU.util.Controller) {
    OU.inherits(OU.activity.ThumbController, OU.util.Controller);
}
else {
    OU.preOnLoad.push(function () {
        OU.inherits(OU.activity.ThumbController, OU.util.Controller);
    });
}
OU.util.ThumbControllerThumb = function (params) {
    OU.util.ThumbControllerThumb.prototype.init = function (params) {
        this.image = params.section.image;
        this.container=params.container;
        this.section = params.section;
        this.ctx=  params.ctx;
        this.sectionId=params.id;
        this.x=params.x;
        this.y=params.y;
        if (this.image) {
            this.w = this.image.width*(this.container.controlHeight/this.image.height);
        }
        else {
            this.w = 160*OU.dpr;
        }
        this.h=this.container.controlHeight;               
    };
    OU.util.ThumbControllerThumb.prototype.ensureVisible = function () {
        var x = this.x+this.container.controlX;
         if(x<0) { // thumbnail is off the left of the screen, so move to the right
             this.container.controlX = -this.x;
             this.container.render();
         }
         if(x+this.w>=this.container.w) { // thumbnail is off the right, so move left
             this.container.controlX = -(this.x+this.w-this.container.w);
             this.container.render();
         }
    };
    OU.util.ThumbControllerThumb.prototype.isHit = function (x,y) {
        var offsetX = x-this.container.controlX;
        if(offsetX>=this.x && offsetX <= this.x+this.w && y>=this.y && y<=this.y+this.h) {
            this.hit();
            return true;
        }
        return false;
    };
    OU.util.ThumbControllerThumb.prototype.hit = function () {
        this.container.step(this.sectionId);
    };
    OU.util.ThumbControllerThumb.prototype.render = function () {
        var x = this.x+this.container.controlX,
        y=this.y,
        w=this.w,h=this.h,ctx=this.ctx,fg,bg;
            
        ctx.clearRect(x,y,w,h);
        if(this.hasFocus) {
            ctx.save();
            ctx.fillStyle='#c00';
            ctx.roundRect(x,y,w,h,5);
            ctx.restore();
        }
            
        if(x+w>=0 && x<this.container.w) {
            if (this.image) {
                ctx.drawImage(this.image, x, y, w, h);
            }
            else {
                ctx.fillRect(x, y, w, h);
            }
            if(this.hasFocus) {
                bg = {
                    radius:8,
                    RGB:'200,0,0',
                    alpha:0.8
                };
                fg='#fff';
            }
            else {
                bg = {
                    radius:8,
                    RGB:'255,255,255',
                    alpha:0.7
                };
                fg='#444';
            }
            new OU.util.DynText({
                txt:this.section.name===undefined ? this.sectionId : this.section.name,
                context:ctx,
                x:x + 5,
                y:h-30*OU.dpr,
                w:w,
                h:20*OU.dpr,
                colour:fg,
                align:'center',
                background:bg
            });
        }
    };
    OU.util.ThumbControllerThumb.prototype.focus = function () {
        this.hasFocus = true;
        this.ensureVisible();
        this.render();
        return false;
    };
    OU.util.ThumbControllerThumb.prototype.blur = function () {
        this.hasFocus = undefined;
        this.render();
        return false;
    };        
    OU.base(this, params);
    this.init(params);
};
OU.inherits(OU.util.ThumbControllerThumb, OU.util.Tabbable);
    

