/**
 * @fileOverview Breathing square illusion
 *
 * @author Greg Black
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.Layer');
OU.require('OU.util.PopUpInfo');
OU.require('OU.util.Slider');
/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.BreathingSquare = function ( data, instance, controller ) { //data and instance are passed in from index.html
    OU.activity.BreathingSquare.prototype.canvasView = function () {
        var bH =  OU.controlHeight;
        // create Canvas Layers & Contexts
        this.initialSliderPos = 0.5;
        this.theta = 0.0;
        this.dTheta = 0.05;
        this.radiansPerDegree = Math.atan(1.0)/45.0;
        this.outerSquareColour = this.data.outerSquareColour;
        this.innerSquareColour = this.data.innerSquareColour;

        this.bgLayer = new OU.util.Layer({
            container:this, //container is the whole area and can contain other stuff
            id:'bg'
        });
        //set colour of container layer (if specified in data.js) or use a default
        if (this.data.backgroundColour !== undefined)
            this.bgLayer.context.gradRect({
                col1:this.data.backgroundColour,
                col2:this.data.backgroundColour
            }); // use specified background colour
        else
            this.bgLayer.context.gradRect(); // use default background

        this.illusionLayer = new OU.util.Layer({
            container:this,
            id:'illusionLayer',
            x:this.w*.1,
            y:this.h*.1+bH,
            w:this.h*.7,
            h:this.h*.7,
            hasEvents:true,
            pinch:this.pinch,
            pinchMe:this
        });

        this.sliderLayer = new OU.util.Layer({
            container:this,
            id:'slidersLayer',
            x:this.w*.1,
            y:this.illusionLayer.y+this.illusionLayer.h,
            w:this.illusionLayer.w,
            h:this.h*.2,
            hasEvents:true,
            pinch:this.pinch,
            pinchMe:this
        });
        this.sliderLayer.context.gradRect(); // use default background

        var ctx = this.illusionLayer.context;
        if (this.data.illusionLayer!==undefined){
            ctx.stroke();
            ctx.fillStyle = this.baseColour;
            ctx.fillRect(0, 0, this.illusionLayer.w, this.illusionLayer.h);
        }
        else{
            ctx.stroke();
            ctx.fillStyle = this.data.baseColour;
            ctx.fillRect(0, 0, this.illusionLayer.w, this.illusionLayer.h);
        }
        this.textDiv = new OU.util.Div({
            container:this
        });
        this.textDiv.div.innerHTML = this.data.instructions;
        this.initialRedWidth = this.illusionLayer.w/5;
        this.greenWidth = this.illusionLayer.w/2;
        this.redWidth = this.initialSliderPos*this.initialRedWidth*2.4;

        this.initControls();
        this.render();
        this.renderLoop();
    };

    OU.activity.BreathingSquare.prototype.step = function () {
        var dt;
        this.theta = this.theta + this.dTheta;
        dt = this.theta / this.radiansPerDegree;
        if(dt >= 360.0){
            dt -= 360.0;
            this.theta = dt * this.radiansPerDegree;
        }
    };

    OU.activity.BreathingSquare.prototype.initControls = function () {
        var ctx = this.sliderLayer.context, bH = OU.controlHeight, self = this,
        clickable = this.sliderLayer.events.clickable;
        this.slider = new OU.util.Slider({
            container:this.sliderLayer,
            instance:'',
            y:this.h*.65+bH,
            w:this.illusionLayer.w,
            h:this.h*.3,
            sliderHeight:bH / 2,
            sliderPos:this.initialSliderPos,
            drawContainer:false,
            title:"",
            showValue:false,
            callback:self.setSquares,
            callbackParam:{
                i:1,
                self:this
            }, //identifies which slider was clicked
            background:{
                clear:true
            },
            context:ctx
        });
        clickable.push(this.slider);

        this.resetBtn = new OU.util.Button({
            txt:"Reset",
            padding:0,
            verticalPadding:0,
            layer:this.sliderLayer,
            onClick:function () {
                self.resetParameters();
            }
        });
        this.helpBtn = new OU.util.Button({
            txt:"Help",
            padding:0,
            verticalPadding:0,
            layer:this.sliderLayer,
            onClick:function () {
                if (!self.helpDisp) {
                    self.helpDisp = true;
                    self.showHelp();
                }
            }
        });
        this.resize();
        this.renderSlider();
    };

    OU.activity.BreathingSquare.prototype.showHelp = function () {
        var self = this;
        new OU.util.PopUpInfo({
            container:this,
            txt:this.data.help,
            x:this.w*.17,
            y:this.h*.17,
            w:this.w*.66,
            h:this.h*.66,
            onClose:function () {
                self.helpDisp = false;
            }
        });
    };

    /**
     * The main renderer.
     */
    OU.activity.BreathingSquare.prototype.render = function () {
        this.drawIllusion();
    };

    OU.activity.BreathingSquare.prototype.remove = function () {
        OU.activity.BreathingSquare.superClass_.remove.call(this); // call the parent class resize
        this.dead=true;
    };
    OU.activity.BreathingSquare.prototype.resize = function () {
        OU.activity.BreathingSquare.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight, ctxS = this.sliderLayer.context, slH = this.sliderLayer.h,
        slW = this.sliderLayer.w;
        this.bgLayer.resize();
        if (this.data.backgroundColour!==undefined)
            this.bgLayer.context.gradRect({
                col1:this.data.backgroundColour,
                col2:this.data.backgroundColour
            }); // use specified background colour
        else
            this.bgLayer.context.gradRect(); // use default background

        this.illusionLayer.resize({
            x:this.w*.1,
            y:this.h*.1+bH,
            w:this.h*.7,
            h:this.h*.7
        });
        this.sliderLayer.resize({
            x:this.w*.1,
            y:this.illusionLayer.y+this.illusionLayer.h,
            w:this.illusionLayer.w,
            h:this.h*.2
        });

        if (this.data.panelColour!==undefined)
            ctxS.gradRect({
                col1:this.data.panelColour,
                col2:this.data.panelColour
            }); // use specified background colour
        else
            ctxS.gradRect(); // use default background

        this.slider.resize({
            y:slW/25,
            w:this.sliderLayer.w,
            h:slH/4
        });

        ctxS.font = '18px ' + OU.theme.font;
        ctxS.lineWidth = 2;
        ctxS.strokeStyle = '#c00';
        ctxS.fillStyle = '#c00';
        ctxS.textAlign = 'center';

        this.resetBtn.resize({
            y:this.sliderLayer.h*.5,
            h:this.sliderLayer.h*.3,
            w:this.sliderLayer.w*.25,
            x:this.sliderLayer.w*.25
        });
        this.helpBtn.resize({
            y:this.sliderLayer.h*.5,
            h:this.sliderLayer.h*.3,
            w:this.sliderLayer.w*.2,
            x:this.sliderLayer.w*.5
        });
        this.textDiv.resize({
            x:this.illusionLayer.x+this.illusionLayer.w+this.w*.05,
            y:this.h*0.2,
            w:this.w*0.3,
            h:this.h/2
        });
        this.resetBtn.render();
        this.helpBtn.render();
    };

    OU.activity.BreathingSquare.prototype.renderLoop = function () {
        var self=this;
        if(!this.dead) {
            this.step();
            this.drawIllusion();
            setTimeout(function() {
                self.renderLoop();
            },40);
        }
    };

    OU.activity.BreathingSquare.prototype.drawIllusion = function () {
        var ctx = this.illusionLayer.context;
        var h = this.illusionLayer.h;
        var w = h; //square layer
        var i;
        // Draw background in light colour
        ctx.fillStyle = "#FFFFFF";
        ctx.fillRect(0, 0, w-1, h-1);

        var rWidth = w/3 * this.slider.sliderPos*1.4;
        var gWidth = w/1.8;
        var x1 = -gWidth/2;
        var x2 = gWidth/2;

        var xpts = [w/2 - (x1*Math.cos(this.theta) - x1*Math.sin(this.theta)),
        w/2 - (x1*Math.cos(this.theta) - x2*Math.sin(this.theta)),
        w/2 - (x2*Math.cos(this.theta) - x2*Math.sin(this.theta)),
        w/2 - (x2*Math.cos(this.theta) - x1*Math.sin(this.theta))
        ];

        var ypts = [w/2 - (x1*Math.sin(this.theta) + x1*Math.cos(this.theta)),
        w/2 - (x1*Math.sin(this.theta) + x2*Math.cos(this.theta)),
        w/2 - (x2*Math.sin(this.theta) + x2*Math.cos(this.theta)),
        w/2 - (x2*Math.sin(this.theta) + x1*Math.cos(this.theta))
        ];

        ctx.fillStyle = this.innerSquareColour;
        ctx.beginPath();
        ctx.moveTo(xpts[0], ypts[0]);
        ctx.lineTo(xpts[1], ypts[1]);
        ctx.lineTo(xpts[2], ypts[2]);
        ctx.lineTo(xpts[3], ypts[3]);
        ctx.closePath();
        ctx.fill();

        var rx = [(w/4), (w/4), 3*(w/4), 3*(w/4)];
        var ry = [(w/4), 3*(w/4), 3*(w/4), (w/4)];
        ctx.fillStyle = this.outerSquareColour;
        for(i = 4; i--;){
            ctx.fillRect(rx[i]  - (rWidth/2), ry[i]  - (rWidth/2), rWidth, rWidth);
        }

        // Draw a black border
        ctx.strokeStyle = "#000000";
        ctx.strokeRect(0, 0, w - 1, h - 1);
    };

    OU.activity.BreathingSquare.prototype.resetParameters = function () {
        this.slider.sliderPos = this.initialSliderPos;
        this.drawIllusion();
    };

    OU.activity.BreathingSquare.prototype.renderSlider = function () {
        //render the slider
        var self = this, s = this.slider;

        this.sliderLayer.clear(0,0,s.w,s.h);
        s.render();

        setTimeout(function () {
            self.renderSlider();
        }, 40);
    };

    OU.activity.BreathingSquare.prototype.setSquares = function ( p, v ) { // Called when slider is moved
        v.self.redWidth = v.self.slider.sliderPos*v.self.initialRedWidth*2.4;
        v.self.drawIllusion();
    };

    /*OU.activity.Cafewall.prototype.pinch = function ( e, s, x, y, dx, dy, me ) { // called when pinch event detected
        var ns = ((e - s) / me.h) + me.tileView.s;
        ns = ns > 1 ? 1 : (ns < 0 ? 0 : ns);
        me.tileView.scale(ns, {
            x:x,
            y:y
        });
        me.zoomSlider.sliderPos = ns;
        me.zoomSlider.render();
    };//*/

    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.BreathingSquare, OU.util.Activity);
