/**
 * @fileOverview Graph2D - renders a equation on a 2D graph
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.Layer');
OU.require('OU.util.Instruction');
/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.Graph2D = function ( data, instance, controller ) {
    OU.activity.Graph2D.prototype.canvasView = function () {
        var bH = OU.controlHeight;
        this.config = {
            depth:5,
            fps:40 // 40ms = 25 fps
        };
        // create Canvas Layers & Contexts
        this.bgLayer = new OU.util.Layer({
            container:this
        });
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        this.PAD = this.h * .045;
        this.modelLayer = new OU.util.Layer({
            container:this,
            'id':'model',
            'h':this.h - bH
        });
        this.controlsLayer = new OU.util.Layer({
            container:this,
            'id':'controls',
            hasEvents:true,
            pinch:this.pinch,
            pinchMe:this
        });
        this.inRender = false;
        this.dataset = function () {
        };
        this.bestFitOn = this.data.settings.straightLineFit || false;
        // Set dataset to 1st and render
        this.setDataset();
        this.changeDataset(OU.LocalStorage.load("OU.graph2d.dataset") || 0);
        this.doRender = true;
        this.renderCycle();
        this.datasetSelector();
    };
    OU.activity.Graph2D.prototype.resize = function () {
        OU.activity.Graph2D.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight, bX = bH * 3;
        this.bgLayer.resize();
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        this.PAD = this.h * .045;
        this.modelLayer.resize({
            'h':this.h - bH
        });
        this.controlsLayer.resize();
        this.xScale = (this.w - this.PAD * 2) / this.dataset.dx;
        this.yScale = ((this.h * .9) - this.PAD * 2) / this.dataset.dy;
        this.controlsLayer.context.clearRect(0, 0, this.w, this.h);
        this.initControls();
        this.renderFrame();
        if (this.dsSelector!==undefined) {
            this.dsSelector.resize({
                x:this.x + this.w - 400,
                y:this.y + this.h * .025,
                w:5,
                h:5
            });
        }
        if (this.data.settings.buttons.straightLine) {
            this.straightLineButton.resize({
                x:this.w - bX - bH * 2,
                y:this.h - bH,
                w:bH * 4,
                h:bH
            });
            bX = bX - bH * 4;
        }
        this.doRender = true;
    };
    OU.activity.Graph2D.prototype.remove = function () {
        OU.activity.Graph2D.superClass_.remove.call(this); // call the superclass method
        this.doRender = false;
    };
    OU.activity.Graph2D.prototype.isHit = function ( x, y, evState ) {
        var ds = this.dataset, dX, dY, edX, edY, bH = OU.controlHeight, PAD = this.PAD;
        if (evState) {
            if (this.inDrag) {
                dX = x - this.dragStartX;
                dY = y - this.dragStartY;
                edX = dX / (this.w - 2 * PAD) * (ds.x2 - ds.x1);
                edY = dY / (this.h - bH - 2 * PAD) * (ds.y2 - ds.y1);
                ds.x1 = ds.x1 - edX;
                ds.x2 = ds.x2 - edX;
                ds.y1 = ds.y1 + edY;
                ds.y2 = ds.y2 + edY;
                this.dragStartX = x;
                this.dragStartY = y;
                this.renderFrame();
                this.doRender = true;
            }
            else {
                this.dragStartX = x;
                this.dragStartY = y;
                this.inDrag = true;
            }
        }
        else {
            this.inDrag = false;
        }
    };
    OU.activity.Graph2D.prototype.pinch = function ( e, s, x, y, dX, dY, me ) {
        var ds = me.dataset,
            mX = ds.x1 + (ds.x2 - ds.x1) / 2,
            mY = ds.y1 + (ds.y2 - ds.y1) / 2;
        dX = 1 - (dX - 1) * .5;
        dY = 1 - (dY - 1) * .5;
        ds.x1 = mX - (mX - ds.x1) * dX;
        ds.x2 = mX + (ds.x2 - mX) * dX;
        ds.y1 = mY - (mY - ds.y1) * dY;
        ds.y2 = mY + (ds.y2 - mY) * dY;
        ds.x1 = ds.x1 > ds.x2 - 1 ? ds.x2 - 1 : ds.x1; // ensure we never flip scale
        ds.y1 = ds.y1 > ds.y2 - 1 ? ds.y2 - 1 : ds.y1;
        ds.deltas();
        me.xScale = (me.w - me.PAD * 2) / ds.dx;
        me.yScale = ((me.h * .9) - me.PAD * 2) / ds.dy;
        me.renderFrame();
        me.doRender = true;
    };
    OU.activity.Graph2D.prototype.setDataset = function () {
        this.datasets = [];
        for (var i = 0; i < this.data.datasets.length; i++) {
            this.datasets.push(new this.Dataset(this.data.datasets[i]));
        }
    };
    OU.activity.Graph2D.prototype.datasetSelector = function () {
        var i, ds, n = this.datasets.length, bH = OU.controlheight,
            h = '<form>Dataset:<select onchange="OU.obj.' + instance + '.changeDataset();" id="eqList' + this.instance + '">';
        if (n < 2)
            return; // don't include if only 1 dataset
        for (i = 0; i < n; i++) {
            ds = this.datasets[i];
            h += '<option value="' + i + '">' + ds.name + '</option>';
        }
        h += '</select></form>';
        this.dsSelector = new OU.util.HtmlBox({
            container:this,
            html:h,
            x:this.x + bH * 5,
            y:this.y + this.PAD * 3,
            w:5,
            h:5,
            unclosable:true,
            handleScrolling:false,
            style:'overflow:visible'
        });
    };
    OU.activity.Graph2D.prototype.changeDataset = function ( ds ) {
        if (ds===undefined) {
            var eqList = document.getElementById('eqList' + this.instance);
            ds = eqList.value;
        }
        this.dataset = this.datasets[ds || 0];
        OU.LocalStorage.save("OU.graph2d.dataset", ds);
        // Calculate the viewport
        this.xScale = (this.w - this.PAD * 2) / this.dataset.dx;
        this.yScale = ((this.h * .9) - this.PAD * 2) / this.dataset.dy;
        this.controlsLayer.context.clearRect(0, 0, this.w, this.h);
        this.initControls();
        this.renderFrame();
        this.resetView();
    };
    OU.activity.Graph2D.prototype.resetView = function () {
        var ds = this.dataset, bH = OU.controlHeight;
        ds.reset();
        this.xScale = (this.w - this.PAD * 2) / ds.dx;
        this.yScale = ((this.h - bH) - this.PAD * 2) / ds.dy;
        this.renderFrame();
        this.doRender = true;
    };
    OU.activity.Graph2D.prototype.renderCycle = function () {
        var self = this;
        if (this.doRender)
            this.render();
        setTimeout(function () {
            self.renderCycle();
        }, 20);
    };
    OU.activity.Graph2D.prototype.render = function () {
        var ctx = this.modelLayer.context, x, y, i, j, set,
            bH = OU.controlHeight, circ = Math.PI * 2, failed = false,
            n, sumXY = 0, sumX = 0, sumX2 = 0, sumY = 0, a, b,
            ds = this.dataset,
            xS = this.xScale,
            yS = this.yScale,
            r = this.w / 200,
            PAD = this.PAD,
            yOff = this.h - bH - PAD;
        ctx.clearRect(0, 0, this.w, this.h - bH);
        ctx.save();
        ctx.lineWidth = 1;
        for (i = ds.sets.length; i--;) {
            set = ds.sets[i];
            n = set.points.length;
            sumXY = 0;
            sumX = 0;
            sumX2 = 0;
            sumY = 0;
            for (j = n; j--;) {
                x = set.points[j].x;
                y = set.points[j].y;
                if (!this.isNum(x)) {
                    if (x[0]=='=') {
                        x = parseFloat(OU.LocalStorage.load('ou.savehtml.' + x.substr(1)));
                    }
                }
                if (!this.isNum(y)) {
                    if (y[0]=='=') {
                        y = OU.LocalStorage.load('ou.savehtml.' + y.substr(1));
                        y = parseFloat(y);
                    }
                }
                if (x===undefined || y===undefined || isNaN(x) || isNaN(y)) {
                    failed = true;
                }
                else {
                    if (this.bestFitOn) {
                        sumXY = sumXY + (x * y);
                        sumX = sumX + x;
                        sumX2 = sumX2 + x * x;
                        sumY = sumY + y;
                    }
                    ctx.fillStyle = set.fillStyle || "#f44";
                    ctx.strokeStyle = set.strokeStyle || "#444";
                    if (x >= ds.x1 && x <= ds.x2 && y >= ds.y1 && y <= ds.y2) {
                        x = PAD + (x - ds.x1) * xS;
                        y = yOff - (y - ds.y1) * yS;
                        if (set.style=='circle') {
                            ctx.beginPath();
                            ctx.arc(x, y, r, 0, circ, false);
                            ctx.fill();
                            ctx.stroke();
                        }
                        else if (set.style=='rect') {
                            ctx.beginPath();
                            ctx.fillRect(x - r, y - r, r * 2, r * 2);
                            ctx.stroke();
                        }
                    }
                }
            }
            if (!failed && this.bestFitOn) {
                a = (n * sumXY - sumX * sumY) / (n * sumX2 - sumX * sumX);
                b = (sumY - a * sumX) / n;
                x = ds.x1;
                y = x * a + b;
                if (y < ds.y1) {
                    y = ds.y1;
                    x = (y - b) / a;
                }
                else if (y > ds.y2) {
                    y = ds.y2;
                    x = (y - b) / a;
                }
                x = PAD + (x - ds.x1) * xS;
                y = yOff - (y - ds.y1) * yS;
                ctx.beginPath();
                ctx.moveTo(x, y);
                x = ds.x2;
                y = x * a + b;
                if (y < ds.y1) {
                    y = ds.y1;
                    x = (y - b) / a;
                }
                else if (y > ds.y2) {
                    y = ds.y2;
                    x = (y - b) / a;
                }
                x = PAD + (x - ds.x1) * xS;
                y = yOff - (y - ds.y1) * yS;
                ctx.lineTo(x, y);
                ctx.strokeStyle = set.fillStyle || '#44f';
                ctx.stroke();
            }
        }
        ctx.restore();
        if (failed && !this.failureMsgDone) {
            this.failureMsgDone = true;
            new OU.util.Instruction({
                container:this,
                message:this.data.incompleteMessage || 'Some of the values are missing, please ensure you have completed the relevant table.'
            });
        }
        this.renderControls();
        this.doRender = false;
    };
    OU.activity.Graph2D.prototype.isNum = function ( n ) {
        return !isNaN(parseFloat(n)) && isFinite(n);
    };
    OU.activity.Graph2D.prototype.renderFrame = function () {
        var ctx = this.bgLayer.context,
            bH = OU.controlHeight,
            ds = this.dataset,
            xAxis, yAxis, i, step, val,
            xS = this.xScale,
            yS = this.yScale,
            PAD = this.PAD;
        ctx.gradRect();
        ctx.save();
        ctx.fillStyle = '#fff';
        ctx.fillRect(PAD, PAD, this.w - 2 * PAD, this.h - bH - (2 * PAD));
        ctx.strokeStyle = "#000";
        ctx.lineWidth = 0.25;
        ctx.fillStyle = "#444";
        yAxis = -ds.x1 * xS + PAD;
        yAxis = yAxis < PAD ? PAD : (yAxis > this.w - PAD ? this.w - PAD : yAxis);
        ctx.beginPath();
        ctx.moveTo(yAxis, PAD);
        ctx.lineTo(yAxis, this.h - bH - PAD);
        ctx.closePath();
        ctx.stroke();
        step = (ds.dy / 20);
        ctx.textAlign = 'right';
        //      ctx.fillText("x:"+ds.x1+" to "+ds.x2+"    y:"+ds.y1+" to "+ds.y2+"    dX:"+ds.dx+"    dY:"+ds.dy,this.w-10,10);
        for (i = ds.y2; i >= ds.y1; i = i - step) {
            ctx.beginPath();
            ctx.moveTo(yAxis, PAD + (ds.y2 - i) * yS);
            ctx.lineTo(yAxis + 10, PAD + (ds.y2 - i) * yS);
            ctx.closePath();
            ctx.stroke();
            val = ((i * 100) | 0) / 100;
            ctx.fillText(val, yAxis - 5, PAD + (ds.y2 - i) * yS);
        }
        xAxis = ds.y2 * yS + PAD;
        xAxis = xAxis < PAD ? PAD : (xAxis > this.h - PAD - bH ? this.h - PAD - bH : xAxis);
        ctx.beginPath();
        ctx.moveTo(PAD, xAxis);
        ctx.lineTo(this.w - PAD, xAxis);
        ctx.closePath();
        ctx.stroke();
        step = (ds.dx / 20);
        ctx.textAlign = 'center';
        for (i = ds.x1; i <= ds.x2; i = i + step) {
            ctx.beginPath();
            ctx.moveTo(PAD + (i - ds.x1) * xS, xAxis);
            ctx.lineTo(PAD + (i - ds.x1) * xS, xAxis - 10);
            ctx.closePath();
            ctx.stroke();
            val = ((i * 100) | 0) / 100;
            ctx.fillText(val, PAD + (i - ds.x1) * xS, xAxis + 10);
        }
        ctx.restore();
    };
    OU.activity.Graph2D.prototype.initControls = function () {
        var ctx = this.controlsLayer.context, bH = OU.controlHeight,
            clickable = this.controlsLayer.events.clickable,
            self = this, bX = bH * 3;
        clickable.length = 0;
        this.resetButton = new OU.util.ControlButton({
            txt:'Reset',
            x:0,
            y:this.h - bH,
            w:bH * 2,
            h:bH,
            layer:this.controlsLayer,
            onClick:function () {
                self.resetView();
            }
        });
        if (this.data.settings.buttons.straightLine) {
            this.straightLineButton = new OU.util.ControlButton({
                txt:'Straight Line Fit',
                x:this.w - bX - bH * 2,
                y:this.h - bH,
                w:bH * 4,
                h:bH,
                layer:this.controlsLayer,
                onOffIndicator:true,
                onOff:this.bestFitOn,
                onClick:function () {
                    self.toggleStraightLine();
                }
            });
            bX = bX - bH * 4;
        }
        clickable.push(this);
    };
    OU.activity.Graph2D.prototype.toggleStraightLine = function () {
        this.bestFitOn = !this.bestFitOn;
        this.straightLineButton.params.onOff = this.bestFitOn;
        this.render();
    };
    OU.activity.Graph2D.prototype.renderControls = function () {
        var ctx = this.controlsLayer.context;
        this.controlsLayer.clear();
        this.resetButton.render();
        if (this.data.settings.buttons.straightLine) {
            this.straightLineButton.render();
        }
        new OU.util.DynText({
            txt:this.dataset.name,
            x:-this.h * .03,
            y:this.PAD,
            w:this.w / 4,
            h:this.PAD * 1.5,
            context:ctx,
            background:{
                col:'#afdffb',
                radius:this.h * .03
            }
        });
    };
    OU.activity.Graph2D.prototype.Dataset = function ( params ) {
        var t;
        this.name = params.name || 'y=?';
        this.x1 = params.x1 || 0;
        this.x2 = params.x2 || 10;
        this.y1 = params.y1 || 0;
        this.y2 = params.y2 || 10;
        this.z1 = params.z1 || 0;
        this.z2 = params.z2 || 10;
        if (this.x1 > this.x2) {
            t = this.x2;
            this.x2 = this.x1;
            this.x1 = t;
        }
        if (this.y1 > this.y2) {
            t = this.y2;
            this.y2 = this.y1;
            this.y1 = t;
        }
        this.reset_x1 = this.x1;
        this.reset_x2 = this.x2;
        this.reset_y1 = this.y1;
        this.reset_y2 = this.y2;
        this.sets = params.sets || [];
        if (params.colour!==undefined)
            this.colour = params.colour;
        if (params.rgb!==undefined)
            this.rgb = params.rgb;
        OU.activity.Graph2D.prototype.Dataset.prototype.deltas = function () {
            this.dx = Math.abs(this.x2 - this.x1);
            this.dy = Math.abs(this.y2 - this.y1);
        };
        OU.activity.Graph2D.prototype.Dataset.prototype.reset = function () {
            this.x1 = this.reset_x1;
            this.x2 = this.reset_x2;
            this.y1 = this.reset_y1;
            this.y2 = this.reset_y2;
            this.deltas();
        };
        this.deltas();
    };
    OU.activity.Graph2D.prototype.accessibleView = function () {
        var i, h = '<div id="equation3DAccessible">';
        h += '<h1>' + this.data.title + '</h1>';
        h += '<p>' + this.data.description + '</p>';
        for (i = 0; i < this.data.dataset.length; i++) {
            var equation = this.data.dataset[i];
            h += '<h2>' + equation.name + '</h2><p>' + equation.description + '</p>';
        }
        h += '</div>';
        document.body.innerHTML = '';
        var accessible = document.createElement('div');
        accessible.innerHTML = h;
        document.body.appendChild(accessible);
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.Graph2D, OU.util.Activity);
