/**
 * @fileOverview Necker cube illusion
 *
 * @author Greg Black & Will Rawes
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.Layer');
OU.require('OU.util.PopUpInfo');
OU.require('OU.util.Slider');
/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.NeckerCube = function ( data, instance, controller ) { //data and instance are passed in from index.html
    OU.activity.NeckerCube.prototype.canvasView = function () {
        var self = this;
        var bH = OU.controlHeight;
        // create Canvas Layers & Contexts
        
        //viewport of the cube
        /** @constant */
        this.VIEWWINDOW = {
            x:-2,
            y:-2,
            w:4,
            h:4
        };
        /** @constant */
        this.DEFAULT_TILT = 35;
        /** @constant */ 
        this.RADIANS_PER_DEGREE = Math.atan(1.0)/45.0;
        /** @constant */ 
        this.DTHETA = 0.05;

        this.tilt = this.DEFAULT_TILT*this.RADIANS_PER_DEGREE;
        this.cosTilt = Math.cos(this.tilt);
        this.sinTilt = Math.sin(this.tilt);
        this.theta = 0.0;
        this.cosTheta = Math.cos(this.theta);
        this.sinTheta = Math.sin(this.theta);
        
        /** @constant */
        this.SIDEVIEW  = 1;
        /** @constant */
        this.TOPVIEW   = 2;
        /** @constant */
        this.BOTHVIEWS = 3;
        /** @constant */
        this.WIREFRAME  = 1;
        /** @constant */
        this.FLATSHADE  = 2;
        /** @constant */
        this.WIRESHADE  = 3;
        
        this.whichView = this.SIDEVIEW;
        this.renderMode = this.WIREFRAME; //WireShade;

        this.isRunning = false;
  
        this.bgLayer = new OU.util.Layer({
            container:this, //container is the whole area and can contain other stuff
            id:'bg'
        });
        //set colour of container layer (if specified in data.js) or use a default
        if (this.data.backgroundColour !== undefined)
            this.bgLayer.context.gradRect({
                col1:this.data.backgroundColour,
                col2:this.data.backgroundColour
            }); // use specified background colour
        else
            this.bgLayer.context.gradRect(); // use default background

        this.illusionLayer = new OU.util.Layer({
            container:this,
            id:'illusionLayer',
            x:this.w*.05,
            y:this.h*.1+bH,
            w:this.h*.7,
            h:this.h*.7,
            hasEvents:true,
            pinch:this.pinch,
            pinchMe:this
        });
        this.ctx = this.illusionLayer.context;
        
        //helper functions to draw polygons and lines
        this.ctx.fillPolygon = function( x, y, n ){
            self.ctx.beginPath();
            self.ctx.moveTo(x[n-1],y[n-1]);
            for(var i = n - 1; i--;){
                self.ctx.lineTo(x[i],y[i]);
            }
            self.ctx.stroke();
            self.ctx.fill();
        };
        this.ctx.drawLine = function( x0, y0, x1, y1 ){
            self.ctx.moveTo(x0,y0);
            self.ctx.lineTo(x1,y1);
        };
        
        this.h = this.illusionLayer.h;
        this.w = this.h; //square layer
        this.d = {
            width:this.w, //illusionLayer.w,
            height:this.h //illusionLayer.h
        };
        this.ctx.strokeStyle = "#000";
        this.ctx.fillStyle = "#000";
        this.ctx.fillPolygon([10,20,100], [30,20,50], 3);

        // Allocate and initialize edge array
        this.vertex = [];
        this.vertex[0] = new this.Vertex(-1.0, -1.0, -1.0);
        this.vertex[1] = new this.Vertex(-1.0, -1.0,  1.0);
        this.vertex[2] = new this.Vertex(-1.0,  1.0, -1.0);
        this.vertex[3] = new this.Vertex(-1.0,  1.0,  1.0);
        this.vertex[4] = new this.Vertex( 1.0, -1.0, -1.0);
        this.vertex[5] = new this.Vertex( 1.0, -1.0,  1.0);
        this.vertex[6] = new this.Vertex( 1.0,  1.0, -1.0);
        this.vertex[7] = new this.Vertex( 1.0,  1.0,  1.0);

        // Allocate and initialize face array
        this.face = [];
        this.face[0] = new this.Face(0, 1, 3, 2);
        this.face[1] = new this.Face(0, 4, 5, 1);
        this.face[2] = new this.Face(0, 2, 6, 4);
        this.face[3] = new this.Face(1, 5, 7, 3);
        this.face[4] = new this.Face(2, 3, 7, 6);
        this.face[5] = new this.Face(4, 6, 7, 5);

        // Allocate and initialize edge array
        this.edge = [];
        this.edge[0] = new this.Edge(0, 1);
        this.edge[1] = new this.Edge(0, 2);
        this.edge[2] = new this.Edge(0, 4);
        this.edge[3] = new this.Edge(1, 3);
        this.edge[4] = new this.Edge(1, 5);
        this.edge[5] = new this.Edge(2, 3);
        this.edge[6] = new this.Edge(2, 6);
        this.edge[7] = new this.Edge(3, 7);
        this.edge[8] = new this.Edge(4, 5);
        this.edge[9] = new this.Edge(4, 6);
        this.edge[10] = new this.Edge(5, 7);
        this.edge[11] = new this.Edge(6, 7);

        this.p = [{},{},{},{},{},{}];
        this.q = [{},{},{},{},{},{}];
        this.n = [{},{},{},{},{},{}];
        
        this.xpoints = [];
        this.ypoints = [];
        this.zpoints = [];

        this.xBigPoints = [];
        this.yBigPoints = [];
        this.zBigPoints = [];
        
        for(i = 4; i--;){
            this.xBigPoints[i] = [];
            this.yBigPoints[i] = [];
            this.zBigPoints[i] = [];
        }

        this.optionsLayer = new OU.util.Layer({
            container:this,
            id:'optionsLayer',
            x:this.w*.6,
            y:this.illusionLayer.y,
            w:this.w*.4,
            h:this.h*.3,
            hasEvents:true,
            pinch:this.pinch,
            pinchMe:this
        });
        this.optionsLayer.context.gradRect(); // use default background
        
        this.sliderLayer = new OU.util.Layer({
            container:this,
            id:'slidersLayer',
            x:this.w*.6,
            y:this.illusionLayer.y+this.optionsLayer.h,
            w:this.w*.4,
            h:this.h*.4,
            hasEvents:true,
            pinch:this.pinch,
            pinchMe:this
        });
        this.sliderLayer.context.gradRect(); // use default background

        var ctx = this.illusionLayer.context;
        if (this.data.illusionLayer !== undefined){
            ctx.stroke();
            ctx.fillStyle = this.baseColour;
            ctx.fillRect(0, 0, this.illusionLayer.w, this.illusionLayer.h);
        }
        else{
            ctx.stroke();
            ctx.fillStyle = this.data.baseColour;
            ctx.fillRect(0, 0, this.illusionLayer.w, this.illusionLayer.h);
        }

        this.initControls();
        this.render();
        this.resize();
    };

    /** 
     * @class
     * @returns {Object} The x,y,z coordinates of a vertex.
     */
    OU.activity.NeckerCube.prototype.Vertex = function ( a, b, c ) {
        return {
            x:a,
            y:b,
            z:c
        };
    };
    
    /** 
     * @class
     * @returns {Object} The x,y,z coordinates of the corners of a square face.
     */
    OU.activity.NeckerCube.prototype.Face = function ( a, b, c, d ) {
        return {
            v1:a,
            v2:b,
            v3:c,
            v4:d
        };
    };
    
    /** 
     * @class
     * @returns {Object} The x,y,z coordinates of an edge.
     */
    OU.activity.NeckerCube.prototype.Edge = function ( a, b ) {
        return {
            v1:a,
            v2:b
        };
    };
    
    OU.activity.NeckerCube.prototype.initControls = function () {
        var i, ctx = this.sliderLayer.context, bH = OU.controlHeight, self = this,
        clickable = this.sliderLayer.events.clickable, olW = this.optionsLayer.w,
        olH = this.optionsLayer.h, slW = this.sliderLayer.w, slH = this.sliderLayer.h;

        this.viewButtons = [];
        this.viewButtonLabels = ["Side view", "Top view", "Both"];
        this.viewButtonIDs = ["svOption", "tvOption", "bothOption"];
        for(i = 0; i < 3; i++){
            this.viewButtons[i] = new OU.util.CheckBoxButton({
                txt:this.viewButtonLabels[i],
                x:0,
                y:i*olH/4,
                w:olW*.4,
                h:olH/4,
                layer:this.optionsLayer,
                onClick:function ( p ) {
                    self.setView(p);
                },
                onClickParam:i,//this.viewButtonIDs[i],
                state:false
            });
        }
        this.viewButtons[0].state(true);

        this.styleButtons = new Array(3);
        this.styleButtonLabels = ["Wire frame", "Flat shaded", "Both"];
        this.styleButtonIDs = ["wfOption", "fsOption", "bothOption"];
        for(i = 0; i < 3; i++){
            this.styleButtons[i] = new OU.util.CheckBoxButton({
                txt:this.styleButtonLabels[i],
                x:olW/2,
                y:i*olH/4,
                w:olW*.4,
                h:olH/4,
                layer:this.optionsLayer,
                onClick:function ( p ) {
                    self.setStyle(p);
                },
                onClickParam:i,//this.styleButtonIDs[i],
                state:false
            });
        }
        this.styleButtons[0].state(true);

        
        this.sliders = new Array(3);
        this.sliderLabels = ["Angle","Tilt","Speed"];
        this.sliderIDs = ["angleSlider","tiltSlider","speedSlider"];
        for(i = 0; i < 3; i++){
            this.sliders[i] = new OU.util.Slider({
                container:this.slidersLayer,
                instance:'',
                sliderHeight:bH,
                drawContainer:false,
                title:this.sliderLabels[i]+"",
                titlePosition:"above",
                showValue:false,
                callback:self.setSliders,
                callbackParam:{
                    i:this.sliderIDs[i],
                    self:this
                }, //identifies which button was clicked
                background:{
                    clear:true
                },
                context:ctx
            });
            clickable.push(this.sliders[i]);
        }
        this.sliders[1].sliderPos = 0.35;
        this.sliders[2].sliderPos = 0.5;
        self.angleSlider=this.sliders[0];
        self.tiltSlider=this.sliders[1];
        self.speedSlider=this.sliders[2];

        this.motorBtn = new OU.util.Button({
            txt:"Start motor",
            padding:0,
            verticalPadding:0,
            layer:this.sliderLayer,
            onClick:function () {
                self.toggleMotor();
            }
        });
        this.helpBtn = new OU.util.Button({
            txt:"Help",
            padding:0,
            verticalPadding:0,
            layer:this.sliderLayer,
            onClick:function () {
                if (!self.helpDisp) {
                    self.helpDisp = true;
                    self.showHelp();
                }
            }
        });
        this.resize();
        this.renderSlider();
        this.renderLoop();
    };

    OU.activity.NeckerCube.prototype.showHelp = function () {
        var self = this;
        new OU.util.PopUpInfo({
            container:this,
            txt:this.data.help,
            x:this.w*.17,
            y:this.h*.17,
            w:this.w*.66,
            h:this.h*.66,
            onClose:function () {
                self.helpDisp = false;
            }
        });
    };

    /**
     * The main renderer.
     */
    OU.activity.NeckerCube.prototype.render = function () {
        this.drawIllusion();
    };

    OU.activity.NeckerCube.prototype.resize = function () {
        OU.activity.NeckerCube.superClass_.resize.call(this); // call the parent class resize
        var i, bH = OU.controlHeight, ctxS = this.sliderLayer.context, slH = this.sliderLayer.h,
        slW = this.sliderLayer.w, olW = this.optionsLayer.w, olH = this.optionsLayer.h;
        this.bgLayer.resize();
        if (this.data.backgroundColour!==undefined)
            this.bgLayer.context.gradRect({
                col1:this.data.backgroundColour,
                col2:this.data.backgroundColour
            }); // use specified background colour
        else
            this.bgLayer.context.gradRect(); // use default background

        this.illusionLayer.resize({
            x:this.w*.05,
            y:this.h*.1+bH,
            w:this.h*.7,
            h:this.h*.7
        });
        
        this.optionsLayer.resize({
            x:this.w*.6,
            y:this.illusionLayer.y,
            w:this.w*.4,
            h:this.h*.3
        });

        this.sliderLayer.resize({
            x:this.w*.6,
            y:this.illusionLayer.y+this.optionsLayer.h,
            w:this.w*.4,
            h:this.h*.4
        });
        
        for(i = 0; i < 3; i++){
            this.sliders[i].resize({
                container:this.slidersLayer,
                instance:'',
                x:0,
                y:i*slH*0.2,
                w:slW*.9,
                h:slH*.15
            });
        }
        if (this.data.panelColour!==undefined)
            ctxS.gradRect({
                col1:this.data.panelColour,
                col2:this.data.panelColour
            }); // use specified background colour
        else
            ctxS.gradRect(); // use default background

        ctxS.font = '18px ' + OU.theme.font;
        ctxS.lineWidth = 2;
        ctxS.strokeStyle = '#c00';
        ctxS.fillStyle = '#c00';
        ctxS.textAlign = 'center';

        this.motorBtn.resize({
            x:this.sliderLayer.w*.15,
            y:this.sliderLayer.h*.7,
            w:this.sliderLayer.w*.4,
            h:this.sliderLayer.h*.15
        });
        this.helpBtn.resize({
            x:this.sliderLayer.w*.55,
            y:this.sliderLayer.h*.7,
            w:this.sliderLayer.w*.2,
            h:this.sliderLayer.h*.15
        });
        this.motorBtn.render();
        this.helpBtn.render();
        for(i = this.viewButtons.length; i--;){
            this.viewButtons[i].resize({
                x:0,
                y:i*olH/4,
                w:olW*.4,
                h:olH/4
            });
            this.viewButtons[i].render();
        }
        for(i = this.styleButtons.length; i--;){
            this.styleButtons[i].resize({
                x:olW/2,
                y:i*olH/4,
                w:olW*.4,
                h:olH/4
            });
            this.styleButtons[i].render();
        }
        this.drawIllusion();
    };

    OU.activity.NeckerCube.prototype.renderLoop = function () {
        var self = this;
        this.step();
        
        this.drawIllusion();
        if(!this.isRunning) return;
        setTimeout(function() {
            self.renderLoop();
        },70-this.speedSlider.sliderPos*50);
    };

    OU.activity.NeckerCube.prototype.xform = function (v, d ) {
        var  w, x, y, z;
        var  dx, dy, vs;

        // Default rotation about the x-axis
        x = v.x;
        y = v.y * this.cosTilt - v.z * this.sinTilt;
        z = v.y * this.sinTilt + v.z * this.cosTilt;

        // Default rotation about the z-axis
        w = x;
        x = x * this.cosTilt - y * this.sinTilt;
        y = w * this.sinTilt + y * this.cosTilt;

        // Rotate by theta radians around y-axis
        w = z;
        z = z * this.cosTheta - x * this.sinTheta;
        x = w * this.sinTheta + x * this.cosTheta;

        // Keep viewport square by using smaller dimension plus offsets
        if(d.width < d.height){
            vs = d.width;
            dx = 0.0;
            dy = (d.height - d.width) / 2;
        }
        else{
            vs = d.height;
            dx = (d.width - d.height) / 2;
            dy = 0.0;
        }

        // Map window to square viewport:  v = (w-w0)(v1-v0)/(w1-w0) + v0
        // z maps to -y in top view
        return this.Vertex(
            (x - (this.VIEWWINDOW.x)) * vs / (this.VIEWWINDOW.w) + dx,
            (y - (this.VIEWWINDOW.y)) * vs / (this.VIEWWINDOW.h) + dy,
            (z - (this.VIEWWINDOW.y)) * vs / (this.VIEWWINDOW.h) + dy
        );
    };

    OU.activity.NeckerCube.prototype.drawIllusion = function () {
        var ctx = this.illusionLayer.context;
        var i, j;
        this.d = {
            width:this.illusionLayer.w,
            height:this.illusionLayer.h
        };
        ctx.beginPath();
        ctx.fillStyle = "#fff";
        ctx.fillRect(0, 0, this.d.width, this.d.height);
 
        // Adjust width dimension for split view
        if(this.whichView == this.BOTHVIEWS){
            this.d.width /= 2;
        }

        if((this.renderMode & this.FLATSHADE) != 0){
            // Calculate faces in face array
            for(i = 6; i--;){
                // Transform vertices
                this.v1 = this.xform(this.vertex[this.face[i].v1], this.d);
                this.v2 = this.xform(this.vertex[this.face[i].v2], this.d);
                this.v3 = this.xform(this.vertex[this.face[i].v3], this.d);
                this.v4 = this.xform(this.vertex[this.face[i].v4], this.d);
                this.xBigPoints[0][i] = (this.v1.x);
                this.yBigPoints[0][i] = (this.v1.y);
                this.zBigPoints[0][i] = (this.v1.z);
                this.xBigPoints[1][i] = (this.v2.x);
                this.yBigPoints[1][i] = (this.v2.y);
                this.zBigPoints[1][i] = (this.v2.z);
                this.xBigPoints[2][i] = (this.v3.x);
                this.yBigPoints[2][i] = (this.v3.y);
                this.zBigPoints[2][i] = (this.v3.z);
                this.xBigPoints[3][i] = (this.v4.x);
                this.yBigPoints[3][i] = (this.v4.y);
                this.zBigPoints[3][i] = (this.v4.z);

                // Compute face normal
                this.p[i].x = this.v2.x - this.v1.x;
                this.p[i].y = this.v2.y - this.v1.y;
                this.p[i].z = this.v2.z - this.v1.z;
                this.q[i].x = this.v3.x - this.v2.x;
                this.q[i].y = this.v3.y - this.v2.y;
                this.q[i].z = this.v3.z - this.v2.z;

                // Reverse sign of n.y because z maps to -y
                this.n[i].x = (this.p[i].y * this.q[i].z) - (this.p[i].z * this.q[i].y);
                this.n[i].y = (this.p[i].x * this.q[i].z) - (this.p[i].z * this.q[i].x);
                this.n[i].z = (this.p[i].x * this.q[i].y) - (this.p[i].y * this.q[i].x);
                this.length = 2.0 * Math.sqrt(
                    (this.n[i].x * this.n[i].x)
                    + (this.n[i].y * this.n[i].y)
                    + (this.n[i].z * this.n[i].z)
                );
                this.n[i].x /= this.length;
                this.n[i].y /= this.length;
                this.n[i].z /= this.length;
            }
            // now paint faces in face array
            for(i = 6; i--;){
                for(j = 4; j--;){
                    this.xpoints[j] = this.xBigPoints[j][i];
                    this.ypoints[j] = this.yBigPoints[j][i];
                    this.zpoints[j] = this.zBigPoints[j][i];
                }
                var redValue;
                if(((this.whichView & this.SIDEVIEW)) && (this.n[i].z > 0.0)){
                    redValue = ((this.n[i].z+0.5)*255)|0;
                    ctx.fillStyle = "rgba("+redValue+",0,0,1)"; //#f00";
                    ctx.strokeStyle = "rgba("+redValue+",0,0,1)"; //#f00";
                    ctx.fillPolygon( this.xpoints, this.ypoints, 4 );
                }
                if(this.whichView == this.BOTHVIEWS){
                    for(j = 4; j--;){
                        this.xpoints[j] += this.d.width;
                    }
                }
                if(((this.whichView & this.TOPVIEW) != 0) && (this.n[i].y > 0.0)){
                    redValue = ((this.n[i].y+0.5)*255)|0;
                    ctx.fillStyle = "rgba("+redValue+",0,0,1)";
                    ctx.strokeStyle = "rgba("+redValue+",0,0,1)";
                    ctx.fillPolygon(this.xpoints, this.zpoints, 4);
                }
            }
        }

        if((this.renderMode & this.WIREFRAME) != 0){
            // Draw edges in edge array
            ctx.strokeStyle = "#000";
            for(i = 12; i--;){
                this.v1 = this.xform(this.vertex[this.edge[i].v1], this.d);
                this.v2 = this.xform(this.vertex[this.edge[i].v2], this.d);
                if((this.whichView & this.SIDEVIEW) != 0){
                    ctx.drawLine(
                        (this.v1.x), (this.v1.y),
                        (this.v2.x), (this.v2.y)
                    );
                }
                if(this.whichView == this.BOTHVIEWS){
                    this.v1.x += this.d.width;
                    this.v2.x += this.d.width;
                }
                if((this.whichView & this.TOPVIEW) != 0){
                    ctx.drawLine(
                        (this.v1.x), (this.v1.z),
                        (this.v2.x), (this.v2.z)
                    );
                }
            }
        }
        ctx.stroke();
    };

    OU.activity.NeckerCube.prototype.toggleMotor = function () {
        this.isRunning=!this.isRunning;
        if(this.isRunning) this.motorBtn.params.txt = "Stop motor";
        else this.motorBtn.params.txt = "Start motor";
        this.motorBtn.render();
        this.renderLoop();
    };

    OU.activity.NeckerCube.prototype.renderSlider = function () {
        //render the slider
        var i, self = this;

        this.sliderLayer.clear(0,0,this.sliderLayer.w,this.sliderLayer.h);
        for(i = this.sliders.length; i--;){
            this.sliders[i].render();
        }
        setTimeout(function () {
            self.renderSlider();
        }, 40);
        //self.squareBtn.render();
        self.motorBtn.render();
        self.helpBtn.render();
    };
    
    OU.activity.NeckerCube.prototype.step = function () {
        var dt;
        this.theta += this.DTHETA;
        dt = this.theta / this.RADIANS_PER_DEGREE;
        if(dt >= 360.0){
            dt -= 360.0;
            this.theta = dt * this.RADIANS_PER_DEGREE;
        }
        this.cosTheta = Math.cos(this.theta);
        this.sinTheta = Math.sin(this.theta);
        this.angleSlider.sliderPos = dt/360;
    };

    OU.activity.NeckerCube.prototype.setSliders = function ( p, v ) { // Called when slider is moved
        var id = v.i;
        switch(id){
            case "angleSlider":
                //set theta
                var theta = v.self.angleSlider.sliderPos*360;
                v.self.theta = theta*v.self.RADIANS_PER_DEGREE;
                v.self.cosTheta = Math.cos(v.self.theta);
                v.self.sinTheta = Math.sin(v.self.theta);
                break;
            case "tiltSlider":
                //set tilt
                var tilt = v.self.tiltSlider.sliderPos*90;
                v.self.tilt = tilt*v.self.RADIANS_PER_DEGREE;
                v.self.cosTilt = Math.cos(v.self.tilt);
                v.self.sinTilt = Math.sin(v.self.tilt);
                break;
            case "speedSlider":
                //set speed
                break;
            default:
                break;
        }
        if(!v.self.isRunning) v.self.drawIllusion();
    };
    
    OU.activity.NeckerCube.prototype.setView = function ( idx ) { // Called when slider is moved
        this.viewType = idx;
        var i;
        for(i = this.viewButtons.length; i--;){
            this.viewButtons[i].state(i==idx ? true : false);
        }
        for(i = this.viewButtons.length; i--;) {
            this.viewButtons[i].render();
        }
        switch(idx){
            case 0:
                this.whichView = this.SIDEVIEW;
                break;
            case 1:
                this.whichView = this.TOPVIEW;
                break;
            case 2:
                this.whichView = this.BOTHVIEWS;
                break;
        }
        if(!this.isRunning) this.drawIllusion();
    };
    
    OU.activity.NeckerCube.prototype.setStyle = function ( idx ) { // Called when slider is moved
        this.styleType = idx;
        var i;
        for(i = this.viewButtons.length; i--;){
            this.styleButtons[i].state(i==idx ? true : false);
        }
        for(i = this.viewButtons.length; i--;) {
            this.styleButtons[i].render();
        }
        switch(idx){
            case 0:
                this.renderMode = this.WIREFRAME;
                break;
            case 1:
                this.renderMode = this.FLATSHADE;
                break;
            case 2:
                this.renderMode = this.WIRESHADE;
                break;
        }
        if(!this.isRunning) this.drawIllusion();
    };

    /*OU.activity.Cafewall.prototype.pinch = function ( e, s, x, y, dx, dy, me ) { // called when pinch event detected
        var ns = ((e - s) / me.h) + me.tileView.s;
        ns = ns > 1 ? 1 : (ns < 0 ? 0 : ns);
        me.tileView.scale(ns, {
            x:x,
            y:y
        });
        me.zoomSlider.sliderPos = ns;
        me.zoomSlider.render();
    };//*/

    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.NeckerCube, OU.util.Activity);
