/**
 * @fileOverview square wave activity
 *
 * @author Fiona Williams
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.Layer');
OU.require('OU.util.PopUpInfo');

/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.SquareWave = function ( data, instance, controller ) { //data and instance are passed in from index.html
    OU.activity.SquareWave.prototype.canvasView = function () {
        //var self = this
        var bH = OU.controlHeight;
        // create Canvas Layers & Contexts
		
	this.harmonics = [];
        for (var i = 0; i < this.data.harmonics.length; i++) {
            var item  = this.data.harmonics[i];
            this.harmonics.push(item);
        }
	
	this.selectedHarmonics = new Array(this.harmonics.length);
	for (var i = 0; i < this.selectedHarmonics.length; i++)
	{
	    this.selectedHarmonics[i] = false;
	}
	
	this.lineCol = "#000";
	 if (this.data.graphLineColour !== undefined)
	 {
	     this.lineCol = this.data.graphLineColour; 
	 }
	 
	 this.backCol = "#FFF";
	 if (this.data.backgroundColour !== undefined)
	 {
	     this.backCol = this.data.backgroundColour;
	 }
	
	
	
        this.bgLayer = new OU.util.Layer({
            container:this, //container is the whole area and can contain other stuff
            id:'bg'
        });
	
        this.controlsLayer = new OU.util.Layer({
            container:this,
            id:'controlsLayer',
            x:this.w*0.05,
            y:2*bH + 0.02*this.h,
            w:this.w*0.25,
            h:this.h*0.65,
            hasEvents:true,
            pinch:this.pinch,
            pinchMe:this
        });
	
	this.harmonicsLayer = new OU.util.Layer({
            container:this,
            id:'harmonicsLayer',
            x:this.controlsLayer.x + this.controlsLayer.w,
            y:this.controlsLayer.y,
            w:this.w*0.25,
            h:this.controlsLayer.h,
            hasEvents:true,
            pinch:this.pinch,
            pinchMe:this
        });
	
	this.xscale = (this.harmonicsLayer.w)/200;   // use same scale for all graphs
	
        this.graphLayer = new OU.util.Layer({
            container:this,
            id:'graphLayer',
            x:this.harmonicsLayer.x + this.harmonicsLayer.w + this.w*0.05,
            y:this.controlsLayer.y,
            w:this.w*0.35,
            h:this.h*0.6,
            hasEvents:true,
            pinch:this.pinch,
            pinchMe:this
        });
	
        this.controlsLayer.context.gradRect(); // use default background

        this.textDiv = new OU.util.Div({
            x:this.controlsLayer.x,
            y:this.controlsLayer.y + this.controlsLayer.h + this.h*0.05,
            w:this.w*0.9,
            h:this.h*0.3,
            container:this
        });
        this.textDiv.div.innerHTML = this.data.instructions;
		
        this.initControls();
        this.resize();    // sets all sizes and calls render()
    };


    OU.activity.SquareWave.prototype.initControls = function () {
        var self = this, cH = this.controlsLayer.h, cW = this.controlsLayer.w;
	
	this.checkButtons = new Array(this.harmonics.length);
	var gap = cH/(this.harmonics.length);
	var ht = 0.75*gap;
        for(var i = 0; i < this.harmonics.length; i++){
            this.checkButtons[i] = new OU.util.CheckBoxButton({
                txt:this.harmonics[i].title,
                x:0.05*cW,
                y:i*gap,
                w:0.9*cW,
                h:ht,
                layer:this.controlsLayer,
		state:false,
                onClick:function ( p ) {
                    self.chooseCheckBox(p);
                },
                onClickParam:i                
            });
        }	

    };

    /**
     * The main renderer.
     */
    OU.activity.SquareWave.prototype.render = function () {	
	for (var i=0; i< this.checkButtons.length; i++)
	{
	    this.checkButtons[i].render();
	}
	
        this.drawGraphs();
	this.drawHarmonics();
    };

    OU.activity.SquareWave.prototype.resize = function () {
        OU.activity.SquareWave.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight, ctxS = this.controlsLayer.context, cH = this.controlsLayer.h,
        cW = this.controlsLayer.w;
	
        this.bgLayer.resize();
			
        this.controlsLayer.resize({
            x:this.w*0.05,
            y:2*bH + 0.02*this.h,
            w:this.w*0.25,
            h:this.h*0.65
        });
	
	this.harmonicsLayer.resize({
	    x:this.controlsLayer.x + this.controlsLayer.w,
            y:this.controlsLayer.y,
            w:this.w*0.25,
            h:this.controlsLayer.h
	})
	this.xscale = (this.harmonicsLayer.w)/200; 
	
	this.graphLayer.resize({
            x:this.harmonicsLayer.x + this.harmonicsLayer.w + this.w*0.05,
            y:this.controlsLayer.y,
            w:this.w*0.35,
            h:this.h*0.6
        });
			
        if (this.data.backgroundColour!==undefined)
	// use specified background colour
	{
            this.bgLayer.context.gradRect({
                col1:this.data.backgroundColour,
                col2:this.data.backgroundColour
            });
	    
	    ctxS.gradRect({
                col1:this.data.backgroundColour,
                col2:this.data.backgroundColour
            });	    	    
	}
	else
        {
	    this.bgLayer.context.gradRect(); // use default background
	    ctxS.gradRect();
	}

	var gap = cH/this.harmonics.length;
	var ht = 0.75*gap;
	for(i = 0; i < this.checkButtons.length; i++){
            this.checkButtons[i].resize({
		x:0.05*cW,
                y:i*gap,
                w:0.9*cW,
                h:ht
            });
        }  
	
        this.textDiv.resize({
           x:this.controlsLayer.x,
            y:this.controlsLayer.y + this.controlsLayer.h + this.h*0.05,
            w:this.w*0.9,
            h:this.h*0.3
        });
	     	
        this.render();
        
    };
    
    OU.activity.SquareWave.prototype.drawAxes = function () {
	var ctx = this.graphLayer.context;
        var w = this.graphLayer.w;
        var h = this.graphLayer.h;
	var origx = w*0.08;    
	var origy = 0.75*h;
	ctx.fillStyle = this.backCol;
        ctx.fillRect(0, 0, w, h);
		
	//draw the axes
	ctx.strokeStyle = this.lineCol;
	ctx.fillStyle = this.lineCol;
	
	var Aht = 0.18*h;
	ctx.beginPath();
	ctx.moveTo(origx, origy - Aht);
	ctx.lineTo(origx, origy + Aht);
	ctx.lineTo(w, origy + Aht);
	ctx.stroke();
	ctx.font = "12px verdana,sans-serif";
	ctx.textBaseline = "top";
	size = ctx.measureText("sum").width;
	ctx.rotate(-Math.PI/2);
	ctx.fillText("sum", -origy + Aht - size, origx - 14);
	ctx.rotate(Math.PI/2);	
    }

    OU.activity.SquareWave.prototype.drawHarmonics = function () {
	var ctx = this.harmonicsLayer.context;
	var w = this.harmonicsLayer.w;
	var h = this.harmonicsLayer.h;
	var gap = h/this.harmonics.length;
	var rectHt = 0.75*gap; 
		
	ctx.fillStyle = this.backCol;
        ctx.fillRect(0, 0, w, h);
	
	ctx.strokeStyle = "rgb(162,162,162)";	
	for (var i = 0; i < this.harmonics.length; i++)
	{	    
	    var yorig = i*gap;
	    var A = this.harmonics[i].amp;
	    var freq = (this.harmonics[i].relativeFreq)/60;
	    var fscale = 2*freq/this.xscale;	
	    
	    ctx.strokeRect(0, yorig, w - 1, rectHt);
	    
	     for (var j = 0; j < w; j=j+1)
	    {	
		intensity = A*Math.sin(Math.PI*fscale*j);     // a value between -1 and +1
		var colourVal = 127 + Math.floor(127*intensity);       
		ctx.fillStyle = 'rgb(' + colourVal + ',' + colourVal + ',' + colourVal + ')';
		ctx.fillRect(j, yorig, 1, rectHt);		
	    }	    	    	    
	}	
    };
	
	
    OU.activity.SquareWave.prototype.drawGraphs = function () {
	
	this.drawAxes();
        var ctx = this.graphLayer.context;
        var w = this.graphLayer.w;
        var h = this.graphLayer.h;
	var h2 = Math.floor(0.5*h);
	var origx = Math.floor(w*0.1);     
	var origy = Math.floor(0.75*h);   // origy refers to the lower graph
	var A = Math.floor(0.15*h);
	var graphWidth = w - origx;
	
	var intensity;
		
	// clear the graph areas
	var Aht = Math.floor(1.2*A);
	ctx.fillStyle = this.backCol;
        ctx.fillRect(origx-1, origy - Aht, graphWidth + 2, 2*Aht - 1);	
	ctx.fillRect(origx-1, 0, graphWidth + 2, h2);
	
	ctx.strokeStyle = this.lineCol;
	ctx.beginPath();
	ctx.moveTo(origx,origy);
					    
	    for (var i = 0; i < graphWidth; i=i+1)
	    {	
		intensity = this.getResultant(i); 
		var colourVal = 127 + Math.floor(127*intensity);       
		ctx.fillStyle = 'rgb(' + colourVal + ',' + colourVal + ',' + colourVal + ')';
		ctx.fillRect(origx + i,2,1,h2-2);
		ctx.lineTo(origx + i, origy - A*intensity);
	    }
	 	
	 ctx.stroke();
	 
	// draw a grey border around the display
	ctx.strokeStyle = "rgb(162,162,162)";
	ctx.strokeRect(origx, 1, graphWidth-1, h2);
			
    };
    
    OU.activity.SquareWave.prototype.getResultant = function(xpos){
	var sum = 0;
	for (var i=0; i < this.selectedHarmonics.length; i++)
	{
	    var freq = (this.harmonics[i].relativeFreq)/60;
	    var fscale = 2*freq/this.xscale;
	    var A = this.harmonics[i].amp;
	    
	    if (this.selectedHarmonics[i])
	    {
		sum = sum + A*Math.sin(Math.PI*fscale*xpos);
	    }
	}
	return sum;
	
    }
    

    OU.activity.SquareWave.prototype.chooseCheckBox = function ( idx ) { // Called when slider is moved
        this.selectedHarmonics[idx] = !(this.selectedHarmonics[idx]);
	this.checkButtons[idx].state(this.selectedHarmonics[idx]);
	this.checkButtons[idx].render();
	this.drawGraphs();
    };


    /*OU.activity.Cafewall.prototype.pinch = function ( e, s, x, y, dx, dy, me ) { // called when pinch event detected
        var ns = ((e - s) / me.h) + me.tileView.s;
        ns = ns > 1 ? 1 : (ns < 0 ? 0 : ns);
        me.tileView.scale(ns, {
            x:x,
            y:y
        });
        me.zoomSlider.sliderPos = ns;
        me.zoomSlider.render();
    };//*/
    /**
     * @class
     */
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.SquareWave, OU.util.Activity);
