/**
 * OU.activity.chemicalBalancer Object
 * @fileOverview Chemical Equation Balancer Activity
 * @param {JSON array} data Data as included in the data/data.js file
 *
 * @author Martin Donnelly
 */
OU.require('OU.util.DynText', '../library/js/util/dynText.js');
OU.require('OU.util.Layer', '../library/js/util/layer.js');
OU.require('OU.util.Spinner', '../library/js/util/spinner.js');
/**
 * @constructor
 * @param data
 * @param instance
 * @param controller
 */
OU.activity.ChemicalBalancer = function ( data, instance, controller ) {
    OU.activity.ChemicalBalancer.prototype.canvasView = function () {
        this.elements = [
            [1, 'H', 'Hydrogen'],
            [2, 'He', 'Helium'],
            [3, 'Li', 'Lithium'],
            [4, 'Be', 'Beryllium'],
            [5, 'B', 'Boron'],
            [6, 'C', 'Carbon'],
            [7, 'N', 'Nitrogen'],
            [8, 'O', 'Oxygen'],
            [9, 'F', 'Fluorine'],
            [10, 'Ne', 'Neon'],
            [11, 'Na', 'Sodium'],
            [12, 'Mg', 'Magnesium'],
            [13, 'Al', 'Aluminium'],
            [14, 'Si', 'Silicon'],
            [15, 'P', 'Phosphorus'],
            [16, 'S', 'Sulfur'],
            [17, 'Cl', 'Chlorine'],
            [18, 'Ar', 'Argon'],
            [19, 'K', 'Potassium'],
            [20, 'Ca', 'Calcium'],
            [21, 'Sc', 'Scandium'],
            [22, 'Ti', 'Titanium'],
            [23, 'V', 'Vanadium'],
            [24, 'Cr', 'Chromium'],
            [25, 'Mn', 'Manganese'],
            [26, 'Fe', 'Iron'],
            [27, 'Co', 'Cobalt'],
            [28, 'Ni', 'Nickel'],
            [29, 'Cu', 'Copper'],
            [30, 'Zn', 'Zinc'],
            [31, 'Ga', 'Gallium'],
            [32, 'Ge', 'Germanium'],
            [33, 'As', 'Arsenic'],
            [34, 'Se', 'Selenium'],
            [35, 'Br', 'Bromine'],
            [36, 'Kr', 'Krypton'],
            [37, 'Rb', 'Rubidium'],
            [38, 'Sr', 'Strontium'],
            [39, 'Y', 'Yttrium'],
            [40, 'Zr', 'Zirconium'],
            [41, 'Nb', 'Niobium'],
            [42, 'Mo', 'Molybdenum'],
            [43, 'Tc', 'Technetium'],
            [44, 'Ru', 'Ruthenium'],
            [45, 'Rh', 'Rhodium'],
            [46, 'Pd', 'Palladium'],
            [47, 'Ag', 'Silver'],
            [48, 'Cd', 'Cadmium'],
            [49, 'In', 'Indium'],
            [50, 'Sn', 'Tin'],
            [51, 'Sb', 'Antimony'],
            [52, 'Te', 'Tellurium'],
            [53, 'I', 'Iodine'],
            [54, 'Xe', 'Xenon'],
            [55, 'Cs', 'Cesium'],
            [56, 'Ba', 'Barium'],
            [57, 'La', 'Lanthanum'],
            [58, 'Ce', 'Cerium'],
            [59, 'Pr', 'Praseodymium'],
            [60, 'Nd', 'Neodymium'],
            [61, 'Pm', 'Promethium'],
            [62, 'Sm', 'Samarium'],
            [63, 'Eu', 'Europium'],
            [64, 'Gd', 'Gadolinium'],
            [65, 'Tb', 'Terbium'],
            [66, 'Dy', 'Dysprosium'],
            [67, 'Ho', 'Holmium'],
            [68, 'Er', 'Erbium'],
            [69, 'Tm', 'Thulium'],
            [70, 'Yb', 'Ytterbium'],
            [71, 'Lu', 'Lutetium'],
            [72, 'Hf', 'Hafnium'],
            [73, 'Ta', 'Tantalum'],
            [74, 'W', 'Tungsten'],
            [75, 'Re', 'Rhenium'],
            [76, 'Os', 'Osmium'],
            [77, 'Ir', 'Iridium'],
            [78, 'Pt', 'Platinum'],
            [79, 'Au', 'Gold'],
            [80, 'Hg', 'Mercury'],
            [81, 'Tl', 'Thallium'],
            [82, 'Pb', 'Lead'],
            [83, 'Bi', 'Bismuth'],
            [84, 'Po', 'Polonium'],
            [85, 'At', 'Astatine'],
            [86, 'Rn', 'Radon'],
            [87, 'Fr', 'Francium'],
            [88, 'Ra', 'Radium'],
            [89, 'Ac', 'Actinium'],
            [90, 'Th', 'Thorium'],
            [91, 'Pa', 'Protactinium'],
            [92, 'U', 'Uranium'],
            [93, 'Np', 'Neptunium'],
            [94, 'Pu', 'Plutonium'],
            [95, 'Am', 'Americium'],
            [96, 'Cm', 'Curium'],
            [97, 'Bk', 'Berkelium'],
            [98, 'Cf', 'Californium'],
            [99, 'Es', 'Einsteinium'],
            [100, 'Fm', 'Fermium'],
            [101, 'Md', 'Mendelevium'],
            [102, 'No', 'Nobelium'],
            [103, 'Lr', 'Lawrencium'],
            [104, 'Rf', 'Rutherfordium'],
            [105, 'Db', 'Dubnium'],
            [106, 'Sg', 'Seaborgium'],
            [107, 'Bh', 'Bohrium'],
            [108, 'Hs', 'Hassium'],
            [109, 'Mt', 'Meitnerium'],
            [110, 'Ds', 'Darmstadtium'],
            [111, 'Rg', 'Roentgenium'],
            [112, 'Cn', 'Copernicium'],
            [113, 'Uut', 'Ununtrium'],
            [114, 'Uuq', 'Ununquadium'],
            [115, 'Uup', 'Ununpentium'],
            [116, 'Uuh', 'Ununhexium'],
            [117, 'Uus', 'Ununseptium'],
            [118, 'Uuo', 'Ununoctium']
        ];
        this.displayMode = this.data.displayMode || 2;
        this.currentEquation = this.data.equation;
        this.solution = this.data.solution;
        this.showHint = this.data.tries || 5;
        this.maxVal = this.data.max || 20;
        this.clickable = new OU.util.TypedArray();
        this.score = 0;
        this.totalScore = 0;
        this.doRender = true;
        this.currentChem = [];
        this.hitBoxes = [];
        this.fontSize = 60;
        this.font = null;
        this.charged = false;
        this.leftCluster = [];
        this.rightCluster = [];
        this.balanceLeft = [];
        this.balanceRight = [];
        this.leftString = "";
        this.rightString = "";
        this.arrows = ["\u2192", "\u21d2", "\u21e2", "\u21e8"];
        this.totalUse = 0;
        this.tbh = 40 * OU.dpr;
        this.changeLimit = 0;
        this.changeCount = 0;
        this.allowedAttempts = 0;
        this.currentAttempts = 0;
        this.dontsave = false;
        this.saved = false;
        this.balanced = false;
        this.congratulated = false;
        this.fixBackground();
        // create Canvas & Context
//        this.bgLayer = new OU.util.Layer({
//            container:this,
//            id:'b_back'
//        });
        /*this.displayLayer = new OU.util.Layer({
         container:this,
         hasEvents:true,
         id:'b_front'

         });*/
        if (this.controller.bgLayer!=undefined) {
            this.bgLayer = this.controller.bgLayer;
        }
        else {
            this.bgLayer = new OU.util.Layer({
                container:this,
                id:'b_back'
            });
        }
        if (this.controller.displayLayer!=undefined) {
            this.displayLayer = this.controller.displayLayer;
        }
        else {
            this.displayLayer = new OU.util.Layer({
                container:this,
                hasEvents:true,
                id:'b_front'

            });
        }
        this.fontface = "league,san-serif";
        this.wasteoftime = false;
        this.defineAreas();
        /*      this.currentLevel = 0;
         this.checkSaveData();
         this.processEquation();
         this.balanceBuilder();
         this.discoverChangeLimit();
         */
    };
    OU.activity.ChemicalBalancer.prototype.fixBackground = function () {
        var cInner = this.controller, cOuter = cInner.controller;
        //    cInner.setBackground(OU.preLoadImg[1]);
    };
    /**
     * Removes the current tutor pop up
     */
    OU.activity.ChemicalBalancer.prototype.killTutorPopup = function () {
        if (this.tutorLayer!=undefined) {
            // kill it with fire!
            this.tutorLayer.remove();
            this.tutorLayer = undefined;
        }
    };
    /**
     * Displays the tutor popup
     * @param p The json settings object
     */
    OU.activity.ChemicalBalancer.prototype.tutorPopup = function ( p ) {
        var x = p.x, y = p.y, w = p.w, h = p.h, txt = p.txt, tx = p.tx, k = p.k;
        this.tutorLayer = new OU.util.Layer({
            container:this,
            hasEvents:false,
            x:x - 10, y:y - 10, w:w + 20, h:h + 20
        });
        x = 10;
        y = 10;
        w = w - 10;
        h = h - 10;
        var l = this.tutorLayer, ctx = l.context;
        ctx.drawImage(OU.preLoadImg[30 + k], x, y, w, h);
        h = h * 0.9;
        if (k <= 1)  y = y + (h * 0.1);
        var txta = new OU.util.DynText({
            txt:txt,
            x:x,
            y:y - (1 * OU.dpr),
            w:w,
            h:h,
            fontSize:h / 3,
            context:ctx,
            colour:"#000",
            align:'center',
            noBreak:false,
            fontFamily:this.fontface });
        delete txta;
        txta = null;
        // fontFamily:'helvetica,arial,sans' });
        var txta = new OU.util.DynText({
            txt:txt,
            x:x,
            y:y,
            w:w,
            h:h,
            fontSize:h / 3,
            context:ctx,
            colour:"#fff",
            align:'center',
            noBreak:false,
            fontFamily:this.fontface });
        delete txta;
        txta = null;
        // fontFamily:'helvetica,arial,sans' });
    };
    /**
     * Removes the spinner control
     */
    OU.activity.ChemicalBalancer.prototype.remove = function () {
        OU.activity.ChemicalBalancer.superClass_.remove.call(this); // call the superclass method
        this.elements = null;
        // check for rogue spinners...
        //var e = document.getElementById('spinner');
        if (this.spinner!==undefined && this.spinner!=null) {
            this.spinner.removeSpinner();
            delete this.spinner;
            this.spinner = undefined;
        }
    };
    /**
     * Defines the layout depending on the rotation of the device
     */
    OU.activity.ChemicalBalancer.prototype.defineAreas = function () {
        var h;
        h = this.h - this.tbh;
        this.areas = {
            feedback:{
                x:0,
                y:this.tbh,
                w:this.w,
                h:h * 0.7
            },
            equation:{
                x:0,
                y:h * 0.7 + this.tbh,
                w:this.w,
                h:h * 0.3
            }
        };
    };
    /**
     * Displays the hint button
     */

    OU.activity.ChemicalBalancer.prototype.showHintButton = function () {
        var l = this.displayLayer, area = this.areas.equation, self = this;
        this.hintButton = new OU.util.Button({
            x:(area.w - 160) / 2,
            y:area.y + area.h - 50,
            w:160,
            h:40,
            container:l,
            align:"center",
            txt:"Solve",
            padding:1,
            verticalPadding:1,
            layer:l,
            onClick:function () {
                self.solve();
            },
            background:{col:"#FF8624", radius:3, borderCol:OU.theme.buttonBorderCol},
            onOffIndicator:false,
            onOff:false
        });
        l.events.clickable.push(this.hintButton);
    };
    /**
     * Solves the equation for the user
     */
    OU.activity.ChemicalBalancer.prototype.discoverChangeLimit = function () {
        var t, i = 0;
        for (t = 0; t < this.currentChem.length; t++) {
            if (this.currentChem[t]!==null) {
                if (this.solution[i] > 1) this.changeLimit++;
                i++;
            }
        }
        this.allowedAttempts = this.changeLimit;
    };
    /**
     * Solves the equation for the user
     */
    OU.activity.ChemicalBalancer.prototype.solve = function () {
        this.dontsave = true;
        var t, i = 0;
        for (t = 0; t < this.currentChem.length; t++) {
            if (this.currentChem[t]!==null) {
                this.currentChem[t].m = this.solution[i];
                i++;
            }
        }
        this.REprocessEquation();
        this.balanceBuilder();
        this.redraw();
    };
    /**
     * the callback from the spinner control
     * @param {int} inval the value selected by the user
     */

    OU.activity.ChemicalBalancer.prototype.spinnerReturn = function ( inval ) {
        if (this.spinner!==undefined) {
            // kill the spinner.
            this.spinner.removeSpinner();
            delete this.spinner;
            this.spinner = undefined;
        }
        this.currentChem[this.itemClicked].m = inval;
        this.REprocessEquation();
        this.balanceBuilder();
        this.totalUse += 1;
        this.currentAttempts++;
        this.redraw();
        //if (this.totalUse >= this.showHint) this.showHintButton();
    }
    /**
     * Handles the touch event
     * @param {object} p Contains {x,y,context,w,h,execute,item}
     */
    OU.activity.ChemicalBalancer.prototype.Marker = function ( p ) {
        this.mX = p.x;
        this.mY = p.y;
        this.ctx = p.context;
        this.mW = p.w;
        this.mH = p.h;
        this.execute = p.execute;
        this.item = p.item;
        this.clickable = this.ctx.canvas.events.clickable;
        /**
         * Checks the location of the hit on the screen
         * @param x
         * @param y
         * @param pressed
         * @param events
         */
        //     console.log("+preishit");
        if (OU.activity.ChemicalBalancer.prototype.Marker.prototype.isHit==undefined) {
            //  console.log("+creating ishit");
            OU.activity.ChemicalBalancer.prototype.Marker.prototype.isHit = function ( x, y, pressed, events ) {
                //     console.log("+ishit");
                var self = this;
                if ((pressed ) && ((x >= self.mX) && (x <= self.mX + self.mW) ) && ((y >= self.mY) && ( y <= self.mY + self.mH))) {
                    events.flush();
                    if (!this.spinner) self.execute(self.item);
                }
            };
        }
    };
    /**
     * Called automatically when window is resized / rotated..
     */
    OU.activity.ChemicalBalancer.prototype.resize = function () {
        OU.activity.ChemicalBalancer.superClass_.resize.call(this); // call the parent class resize
        if (this.spinner!==undefined) {
            // kill the spinner.
            this.spinner.removeSpinner();
            this.spinner = undefined;
        }
//        this.bgLayer.resize();
//       // //   this.tbh = this.h * 0.059;
//        if (this.displayMode!==1) {
//            var self = this;
//            self.bgLayer.context.drawImage(OU.preLoadImg[1], 0, 0, self.bgLayer.w, self.bgLayer.h);
//        }
        this.displayLayer.resize();
        this.redraw();
    };
    /**
     * Redraws the display
     */
    OU.activity.ChemicalBalancer.prototype.redraw = function () {
        this.bgLayer.context.drawImage(OU.preLoadImg[1], 0, 0, this.bgLayer.w, this.bgLayer.h);
        this.displayLayer.clear();
        this.defineAreas();
        //    this.displayEquation();
        //     this.displayBalance();
        if (this.displayMode!==1) {
            this.displaySection();
            this.displayLevel();
            this.displayTopBar();
        }
        if (this.balanced) {
            this.win();
        }
        if (this.wasteoftime) {
            var l = this.displayLayer, ctx = l.context;
            ctx.save();
            ctx.strokeStyle = "#00ffff";
            ctx.fillStyle = "#ffff00";
            ctx.font = '12px Courier';
            ctx.beginPath();
            ctx.moveTo(this.w / 2, 0);
            ctx.lineTo(this.w / 2, this.h);
            ctx.moveTo(0, this.h / 2);
            ctx.lineTo(this.w, this.h / 2);
            ctx.stroke();
            ctx.restore();
        }
    };
    /**
     * Re-processes the equation when a value has been changed
     */



    OU.activity.ChemicalBalancer.prototype.REprocessEquation = function () {
        var runner = [];
        this.leftString = this.rebuilder(this.leftCluster);
        this.rightString = this.rebuilder(this.rightCluster);
        runner = this.runnerBuilder(this.leftCluster, runner);
        runner = this.runnerBuilder(this.rightCluster, runner);
        this.currentChem = runner;
    };
    /**
     * Process the equation for the first time
     */
    OU.activity.ChemicalBalancer.prototype.processEquation = function () {
        var start = new Date().getTime();
        var left, right, wa = this.currentEquation.breaker('=');
        left = wa[0].breaker(' + ');
        right = wa[1].breaker(' + ');
        this.leftCluster = this.parseClusters(left);
        this.rightCluster = this.parseClusters(right);
        this.REprocessEquation();
        var end = new Date().getTime();
        var time = end - start;
    };
    /**
     * Displays the small balance bar
     * Now renders using an svg loaded via the pre-loader
     * @param {object} p Contains {x,y,,w,h,ctx,l,r}
     */
    OU.activity.ChemicalBalancer.prototype.displayBalanceBar = function ( p ) {
        var x = p.x, y = p.y, h = p.h, w = h * 1.82, o = (p.w - w) / 2, ctx = p.ctx, l = p.l, r = p.r, i = 17;
        if (l > r)
            i = 18;
        else if (l < r)
            i = 19;
        ctx.clearRect(x, y, w, h);
        //ctx.drawImage(OU.preLoadImg[7], 10, this.tbh + 10, bw, bw);
        ctx.drawImage(OU.preLoadImg[i], x + o, y, w, h);
        if (this.wasteoftime) {
            //  var l = this.displayLayer, ctx = l.context;
            ctx.save();
            ctx.strokeStyle = "#FF00FF";
            ctx.fillStyle = "#ffff00";
            ctx.font = '12px Courier';
            ctx.strokeRect(x + o, y, w, h);
            ctx.beginPath();
            ctx.moveTo(x + o, y);
            ctx.lineTo(x + o + w, y + h);
            ctx.stroke();
            ctx.fillText("w = h * 1.82 : " + Math.round(w) + "px", x + o + 5, y + 8);
            ctx.restore();
        }
    };
    /**
     * Displays the win animation using a css3 transform
     */
    OU.activity.ChemicalBalancer.prototype.showWin = function () {
        var self = this;
        this.winLayer.canvas.setAttribute("class", "winin");
        setTimeout(function () {
            self.unshowWin();
        }, 2000);
    };
    /**
     * Hides the win animation
     */
    OU.activity.ChemicalBalancer.prototype.unshowWin = function () {
        var self = this;
        this.winLayer.canvas.setAttribute("class", "winout");
        this.displaySection();
        setTimeout(function () {
            self.killWin();
        }, 390);
    };
    /**
     * removes the animation layer and forces a resize check on the screen
     */
    OU.activity.ChemicalBalancer.prototype.killWin = function () {
        this.winLayer.remove();
        delete this.winLayer;
        this.winLayer = undefined;
        this.resize();
    };
    /**
     * sets up the win animation
     * @return {*}
     */
    OU.activity.ChemicalBalancer.prototype.win = function () {
        var l, ctx , area = this.areas.feedback, self = this, clickable, w, h, x, y , cInner = this.controller, cOuter = cInner.controller, d = {};
        // 27
        // y'know what, lets create a new layer for this..
        if (this.tutorLayer!==undefined) this.killTutorPopup();
        if (this.congratulated) return null;
        this.congratulated = true;
        this.winLayer = new OU.util.Layer({
            // container:this,
            container:this.displayLayer.container,
            hasEvents:false,
            instance:this.instance + '.winning'
            /*,
             x:this.x,
             y:this.y,
             h:this.h,
             w:this.w */
        });
        if (!this.dontsave) {
            d.lost = this.allowedAttempts - this.currentAttempts;
            OU.saveData.completedSections[cOuter.sectionNum - 1][cInner.sectionNum] = d;
            OU.util.exportSaveData();
        }
        this.winLayer.canvas.setAttribute("class", "win");
        l = this.winLayer;
        ctx = l.context;
        w = 532 * OU.dpr;
        h = 91 * OU.dpr;
        x = (this.w - w) / 2;
        y = (this.h - h) / 2;
        ctx.drawImage(OU.preLoadImg[27], x, y, w, h);
        // this.winLayer.canvas.class = "winout";
        setTimeout(function () {
            self.showWin();
        }, 750);
    };
    /**
     * Displays the top blue bar
     */
    OU.activity.ChemicalBalancer.prototype.displayTopBar = function () {
        var l = this.displayLayer, ctx = l.context, area = this.areas.feedback, self = this, clickable = l.events.clickable;
        ;
        var bx, by = this.tbh * 0.1, bh = this.tbh * 0.9, bw;
        //this.tbh = this.h * 0.059;
        var gBack = ctx.createLinearGradient(0, 0, 0, this.tbh);
        gBack.addColorStop(0, '#b5e3f2');
        gBack.addColorStop(0.03, '#50bbdf');
        gBack.addColorStop(0.97, '#50bbdf');
        gBack.addColorStop(1, '#295e6e');
        ctx.fillStyle = gBack;
        ctx.fillStyle = "rgba(96,214,247,0.33)";
        ctx.fillRect(0, 0, area.w, this.tbh);
        var space = ((25 * 3) + 14 + (4 * 3)) * OU.dpr;
        var vpos = (this.tbh - (24 * OU.dpr)) / 2;
        var hl = (this.w - space) / 2;
        var hs = (25 * OU.dpr) + (4 * OU.dpr);
        ctx.drawImage(OU.preLoadImg[26], this.w - (35 * OU.dpr), (2 * OU.dpr), 29 * OU.dpr, 35 * OU.dpr);
        if (this.currentAttempts - this.allowedAttempts < 1)
            ctx.drawImage(OU.preLoadImg[25], hl, vpos, (25 * OU.dpr), 24 * OU.dpr);
        else
            ctx.drawImage(OU.preLoadImg[24], hl, vpos, (25 * OU.dpr), 24 * OU.dpr);
        if (this.currentAttempts - this.allowedAttempts < 2)
            ctx.drawImage(OU.preLoadImg[25], hl + hs, vpos, (25 * OU.dpr), 24 * OU.dpr);
        else
            ctx.drawImage(OU.preLoadImg[24], hl + hs, vpos, (25 * OU.dpr), 24 * OU.dpr);
        if (this.currentAttempts - this.allowedAttempts < 3)
            ctx.drawImage(OU.preLoadImg[25], hl + (hs * 2), vpos, (25 * OU.dpr), 24 * OU.dpr);
        else
            ctx.drawImage(OU.preLoadImg[24], hl + (hs * 2), vpos, (25 * OU.dpr), 24 * OU.dpr);
        if (this.currentAttempts - this.allowedAttempts < 3)
            ctx.drawImage(OU.preLoadImg[22], hl + (hs * 3), vpos, (15 * OU.dpr), 24 * OU.dpr);
        else {
            ctx.drawImage(OU.preLoadImg[23], hl + (hs * 3), vpos, (15 * OU.dpr), 24 * OU.dpr);
            //self.solve();
            clickable.push(new this.Marker(
                {

                    x:hl + (hs * 3),
                    y:vpos,
                    w:(15 * OU.dpr),
                    h:24 * OU.dpr,
                    context:ctx,
                    isclicked:false,
                    execute:function () {
                        self.solve();
                    }
                }));
//- ((this.w / 5) * 0.16)
            var params = {x:((hl + (hs * 3)) + (7.5 * OU.dpr)) - ((this.w / 5) * 0.125),
                y:vpos + (24 * OU.dpr),
                w:(this.w / 5),
                h:(this.w / 5) / 1.7,
                txt:"Are you stuck? Hit the light bulb to see the solution to the current equation", k:0};
            this.tutorPopup(params);
        }
        if (this.backButton==undefined || this.backButton==null) {
            this.backButton = new OU.util.Button({
                x:10,
                y:by / 2,
                w:160,
                h:bh,
                container:l,
                align:"center",
                txt:"Back",
                padding:1,
                verticalPadding:1,
                layer:l,
                onClick:function () {
                    self.gotoMenu();
                },
                background:{col:"#50bbdf", radius:3, borderCol:OU.theme.buttonBorderCol},
                onOffIndicator:false,
                onOff:false
            });
        }
        else {
            // this.backButton.render();
        }
    };
    OU.activity.ChemicalBalancer.prototype.displayBalanceinArray = function ( a, e ) {
        for (var i = 0, l = a.length; i < l; i++) {
            if (a[ i ][0]===e) {
                return i;
            }
        }
        return -1;
    };
    /**
     * Draws a balance pair
     * @param ctx current display context
     * @param o offset
     * @param i current index
     * @param q screen quadrant
     * @param lt left text
     * @param rt right text
     * @param lv left value
     * @param rv right value
     * @param f font family
     * @param ft font height
     */
    OU.activity.ChemicalBalancer.prototype.drawBalancePair = function ( ctx, o, i, q, lt, rt, lv, rv, f, ft ) {
        var txt = [];
        var col = "#000140";
        if (this.displayMode===1)
            var pc = 'rgba(21,21,21,0.9)';
        else
            var pc = '#ffffff';
        var fh = 30 * OU.dpr;
        fh = ft * 0.7;
        // var fontface = "HelveticaNeue-CondensedBold";
        ctx.strokeRect(q * 2, o + (i * ft), q / 2, ft);
        ctx.strokeRect(q * 2 + (q / 2), o + (i * ft), q / 2, ft);
        ctx.strokeRect(q * 5, o + (i * ft), q / 2, ft);
        ctx.strokeRect(q * 5 + (q / 2), o + (i * ft), q / 2, ft);
        txt.push(new OU.util.DynText({
            txt:lt,
            x:q * 2,
            y:(o + (i * ft)) - (1 * OU.dpr),
            w:q / 2,
            h:ft,
            fontSize:fh,
            context:ctx,
            colour:"rgba(0,0,0,0.5)",
            align:'center',
            noBreak:true,
            fontFamily:f }));
        txt.push(new OU.util.DynText({
            txt:lt,
            x:q * 2,
            y:o + (i * ft) + (1 * OU.dpr),
            w:q / 2,
            h:ft,
            fontSize:fh,
            context:ctx,
            colour:"rgba(255,255,255,0.5)",
            align:'center',
            noBreak:true,
            fontFamily:f }));
        txt.push(new OU.util.DynText({
            txt:lt,
            x:q * 2,
            y:o + (i * ft),
            w:q / 2,
            h:ft,
            fontSize:fh,
            context:ctx,
            colour:col,
            align:'center',
            noBreak:true,
            fontFamily:f }));
        //////////
        txt.push(new OU.util.DynText({
            txt:lv.toString(),
            x:q * 2 + (q / 2),
            y:o + (i * ft) - (1 * OU.dpr),
            w:q / 2,
            h:ft,
            fontSize:fh,
            context:ctx,
            colour:"rgba(0,0,0,0.5)",
            align:'center',
            noBreak:true,
            fontFamily:f }));
        txt.push(new OU.util.DynText({
            txt:lv.toString(),
            x:q * 2 + (q / 2),
            y:o + (i * ft) + (1 * OU.dpr),
            w:q / 2,
            h:ft,
            fontSize:fh,
            context:ctx,
            colour:"rgba(255,255,255,0.5)",
            align:'center',
            noBreak:true,
            fontFamily:f }));
        txt.push(new OU.util.DynText({
            txt:lv.toString(),
            x:q * 2 + (q / 2),
            y:o + (i * ft),
            w:q / 2,
            h:ft,
            fontSize:fh,
            context:ctx,
            colour:col,
            align:'center',
            noBreak:true,
            fontFamily:f }));
        ////////////
        txt.push(new OU.util.DynText({
            txt:rv.toString(),
            x:q * 5,
            y:o + (i * ft) - (1 * OU.dpr),
            w:q / 2,
            h:ft,
            fontSize:fh,
            context:ctx,
            colour:"rgba(0,0,0,0.5)",
            align:'center',
            noBreak:true,
            fontFamily:f }));
        txt.push(new OU.util.DynText({
            txt:rv.toString(),
            x:q * 5,
            y:o + (i * ft) + (1 * OU.dpr),
            w:q / 2,
            h:ft,
            fontSize:fh,
            context:ctx,
            colour:"rgba(255,255,255,0.5)",
            align:'center',
            noBreak:true,
            fontFamily:f }));
        txt.push(new OU.util.DynText({
            txt:rv.toString(),
            x:q * 5,
            y:o + (i * ft),
            w:q / 2,
            h:ft,
            fontSize:fh,
            context:ctx,
            colour:col,
            align:'center',
            noBreak:true,
            fontFamily:f }));
        ////////////////
        txt.push(new OU.util.DynText({
            txt:rt,
            x:q * 5 + (q / 2),
            y:o + (i * ft) - (1 * OU.dpr),
            w:q / 2,
            h:ft,
            fontSize:fh,
            context:ctx,
            colour:"rgba(0,0,0,0.5)",
            align:'center',
            noBreak:true,
            fontFamily:f  }));
        txt.push(new OU.util.DynText({
            txt:rt,
            x:q * 5 + (q / 2),
            y:o + (i * ft) + (1 * OU.dpr),
            w:q / 2,
            h:ft,
            fontSize:fh,
            context:ctx,
            colour:"rgba(255,255,255,0.5)",
            align:'center',
            noBreak:true,
            fontFamily:f  }));
        txt.push(new OU.util.DynText({
            txt:rt,
            x:q * 5 + (q / 2),
            y:o + (i * ft),
            w:q / 2,
            h:ft,
            fontSize:fh,
            context:ctx,
            colour:col,
            align:'center',
            noBreak:true,
            fontFamily:f  }));
        for (var i = txt.length; i--;) {
            delete txt[i];
            //         txt[i]=undefined;
        }
        txt = null;
    };
    /**
     * Displays the balance of each side
     */


    OU.activity.ChemicalBalancer.prototype.displayBalance = function () {
        var ctx = this.displayLayer.context, q = this.w / 8, o, l = this.balanceLeft.length, i, offset, area = this.areas.feedback, dc;
        var gBack, balanced = true;
        // var ft = 40 * OU.dpr;
        var ft = this.balanceLeft.length > 6 ? (area.h * 0.90) / this.balanceLeft.length : (area.h * 0.90) / 5;
        if (this.displayMode===1) {
            gBack = ctx.createLinearGradient(0, area.y, 0, area.y + area.h)
            gBack.addColorStop(0, 'rgba(200,215,220,1)');
            gBack.addColorStop(0.37, 'rgba(227,234,237,1)');
            gBack.addColorStop(1, 'rgba(242,245,246,1)');
            ctx.fillStyle = gBack;
            ctx.fillRect(area.x, area.y, area.w, area.h);
            ctx.strokeStyle = "rgb(21,21,21)";
        }
        else
            ctx.strokeStyle = "#ffffff";
        // check for charge:
        dc = (Math.abs(this.balanceLeft[this.displayBalanceinArray(this.balanceLeft, "chg")][1]) > 0 || Math.abs(this.balanceRight[this.displayBalanceinArray(this.balanceRight, "chg")][1]) > 0);
        offset = dc ? ((area.h - this.tbh) - ((l + 1) * ft)) / 2 : ((area.h - this.tbh) - ((l - 1) * ft)) / 2;
        offset = this.tbh + (area.h - (area.h * 0.90)) / 2;
        for (i = 1; i < l; i++) {
            o = this.displayBalanceinArray(this.balanceRight, this.balanceLeft[i][0]);
            if (this.balanceLeft[i][1]!=this.balanceRight[o][1]) {
                balanced = false;
            }
            this.drawBalancePair(ctx, offset, i - 1, q, this.balanceLeft[i][0], this.balanceRight[o][0], this.balanceLeft[i][1], this.balanceRight[o][1], this.fontface, ft);
            this.displayBalanceBar({
                x:(q * 4) - (q / 2), y:offset + ((i - 1) * ft), w:q, h:ft, ctx:ctx, l:this.balanceLeft[i][1], r:this.balanceRight[o][1]

            });
        }
        if (this.charged) {
            this.drawBalancePair(ctx, offset, i, q, 'Charge', 'Charge', (parseInt(this.balanceLeft[0][1], 10) > 0 ? "+" + this.balanceLeft[0][1] : this.balanceLeft[0][1]) || 0, (parseInt(this.balanceRight[0][1], 10) > 0 ? "+" + this.balanceRight[0][1] : this.balanceRight[0][1]) || 0, this.fontface, ft);
            this.displayBalanceBar({
                x:(q * 4) - (q / 2), y:offset + ((i) * ft), w:q, h:ft, ctx:ctx, l:this.balanceLeft[0][1], r:this.balanceRight[0][1]

            });
            if (this.balanceLeft[0][1]!=this.balanceRight[0][1]) {
                balanced = false;
            }
        }
        this.balanced = balanced;
        /*
         they don't like these numbers but they are useful for debugging
         new OU.util.DynText({
         txt:this.balanceLeft.total,
         x:0,
         y:(area.h / 2) - 30,
         w:300,
         h:60,
         fontSize:60,
         context:ctx,
         colour:'rgba(21,21,21,0.9)',
         align:'center',
         noBreak:true  });
         // display right amount
         new OU.util.DynText({
         txt:this.balanceRight.total,
         x:this.w - 300,
         y:(area.h / 2) - 30,
         w:300,
         h:60,
         fontSize:60,
         context:ctx,
         colour:'rgba(21,21,21,0.9)',
         align:'center',
         noBreak:true  });

         */
    };
    /**
     * Displays the current level
     */
    OU.activity.ChemicalBalancer.prototype.displayLevel = function () {
        var self = this, l = this.displayLayer, ctx = l.context, cInner = this.controller, cOuter = cInner.controller, offset, offsetY, offsetX, area = this.areas.feedback;
        offset = (area.h - (area.h * 0.90)) / 2;
        offsetY = this.tbh + offset;
        offsetX = offset;
        var bw = (this.w * 0.15);
        ctx.clearRect(0, this.tbh + 10, bw, bw);
        ctx.drawImage(OU.preLoadImg[6 + cOuter.sectionNum], offsetX, offsetY, bw, bw);
        if (this.wasteoftime) {
            ctx.save();
            ctx.strokeStyle = "#FF00FF";
            ctx.fillStyle = "#ffff00";
            ctx.font = '12px Courier';
            ctx.strokeRect(offsetX, offsetY, bw, bw);
            ctx.beginPath();
            ctx.moveTo(offsetX, offsetY);
            ctx.lineTo(offsetX + bw, offsetY + bw);
            ctx.stroke();
            ctx.restore();
        }
    };
    /**
     * Forces the outer controller to the menu
     */
    OU.activity.ChemicalBalancer.prototype.gotoMenu = function () {
        this.controller.controller.setSection(0);
    };
    /**
     * Makes the outer controller go to the previous page
     */
    OU.activity.ChemicalBalancer.prototype.previousSection = function () {
        var os = this.controller.sectionNum;
        this.controller.prevSection();
        if (os!=this.controller.sectionNum) {
            this.displaySection();
        }
    };
    /**
     * Makes the outer controller go to the next page
     */
    OU.activity.ChemicalBalancer.prototype.nextSection = function () {
        var os = this.controller.sectionNum;
        this.controller.nextSection();
        if (os!=this.controller.sectionNum) {
            this.displaySection();
        }
    };
    /**
     * Displays the current section number
     */
    OU.activity.ChemicalBalancer.prototype.displaySection = function () {

        // lets draw the controller section :)
        var self = this, l = this.displayLayer, ctx = l.context, runner, sectionText = [], clickable = l.events.clickable, offset, offsetY, offsetX, area = this.areas.feedback;
        var bw = (this.w * 0.15), bwh = bw / 2;
        var cInner = this.controller, cOuter = cInner.controller;
        ctx.clearRect((this.w - bw - 10) - 30, this.tbh + 10, bw + 40, bw * 2);
        var dt = (this.controller.sectionNum + 1) + ' / ' + this.controller.sections.length;
        var fontface = this.fontface;
        offset = (area.h - (area.h * 0.90)) / 2;
        offsetY = this.tbh + offset;
        offsetX = this.w - offset - bw;
        sectionText.push(new OU.util.DynText({
            txt:dt,
            x:offsetX,
            y:offsetY - (1 * OU.dpr),
            w:bw,
            h:bwh,
            fontSize:bwh,
            context:ctx,
            colour:'rgba(0,0,0,0.5)',
            align:'center',
            measureOnly:false,
            noBreak:true,
            fontFamily:fontface}));
        sectionText.push(new OU.util.DynText({
            txt:dt,
            x:offsetX,
            y:offsetY + (1 * OU.dpr),
            w:bw,
            h:bwh,
            fontSize:bwh,
            context:ctx,
            colour:'rgba(255,255,255,0.5)',
            align:'center',
            measureOnly:false,
            noBreak:true,
            fontFamily:fontface}));
        sectionText.push(new OU.util.DynText({
            txt:dt,
            x:offsetX,
            y:offsetY,
            w:bw,
            h:bwh,
            fontSize:bwh,
            context:ctx,
            colour:'#000140',
            align:'center',
            measureOnly:false,
            noBreak:true,
            fontFamily:fontface}));
        if (this.controller.sectionNum===0) {
            ctx.save();
            ctx.globalAlpha = 0.2;
            ctx.drawImage(OU.preLoadImg[4], offsetX, offsetY + bwh, bwh, bwh);
            ctx.restore();
        }
        else {
            ctx.drawImage(OU.preLoadImg[3], offsetX, offsetY + bwh, bwh, bwh);
            clickable.push(new this.Marker(
                {

                    x:offsetX,
                    y:offsetY + bwh,
                    w:bwh,
                    h:bwh,
                    context:ctx,
                    isclicked:false,
                    execute:function () {
                        self.previousSection();
                    }
                }));
        }
        if (this.controller.sectionNum===this.controller.sections.length - 1) {
            ctx.drawImage(OU.preLoadImg[6], offsetX + bwh, offsetY + bwh, bwh, bwh);
        }
        else {
            ctx.drawImage(OU.preLoadImg[5], offsetX + bwh, offsetY + bwh, bwh, bwh);
            clickable.push(new this.Marker(
                {

                    x:offsetX + bwh,
                    y:offsetY + bwh,
                    w:bwh,
                    h:bwh,
                    context:ctx,
                    isclicked:false,
                    execute:function () {
                        self.nextSection();
                    }
                }));
        }
        if (typeof OU.saveData.completedSections[cOuter.sectionNum - 1][cInner.sectionNum]=='object') {

            //  ctx.clearRect(this.w - bw - 10, this.tbh + 10, bw, bw);
            //   ctx.drawImage(OU.preLoadImg[5], this.w - bw - 10 + bwh, this.tbh + 10 + bwh, bwh, bwh);
            sectionText.push(new OU.util.DynText({
                txt:'COMPLETED',
                x:offsetX,
                y:offsetY + bw - (1 * OU.dpr),
                w:bw,
                h:bwh,
                fontSize:bwh,
                context:ctx,
                colour:'rgba(0,0,0,0.5)',
                align:'center',
                measureOnly:false,
                noBreak:true,
                fontFamily:fontface}));
            sectionText.push(new OU.util.DynText({
                txt:'COMPLETED',
                x:offsetX,
                y:offsetY + bw + (1 * OU.dpr),
                w:bw,
                h:bwh,
                fontSize:bwh,
                context:ctx,
                colour:'rgba(255,255,255,0.5)',
                align:'center',
                measureOnly:false,
                noBreak:true,
                fontFamily:fontface}));
            sectionText.push(new OU.util.DynText({
                txt:'COMPLETED',
                x:offsetX,
                y:offsetY + bw,
                w:bw,
                h:bwh,
                fontSize:bwh,
                context:ctx,
                colour:'#000140',
                align:'center',
                measureOnly:false,
                noBreak:true,
                fontFamily:fontface}));
        }
        for (var i = sectionText.length; i--;) {
            delete sectionText[i];
            //         txt[i]=undefined;
        }
        sectionText = null;
    };
    /**
     * Displays the equation
     */
    OU.activity.ChemicalBalancer.prototype.displayEquation = function () {
        var self = this, l = this.displayLayer, area = this.areas.equation, ctx = l.context, runner, clickable = l.events.clickable, gBack, i, fs;
        var rightText, goesTo, leftText, leftTest, rightTest, lX, bW, rX, cb, lttb, rttb, maxH;
        var tBlack = 'rgba(0,0,0,0.50)', fontface;
        clickable.length = 0;
        ctx.clearRect(0, 0, this.w, this.h);
        runner = this.currentChem;
        if (this.displayMode===1) {
            gBack = ctx.createLinearGradient(0, area.y, 0, area.y + area.h);
            gBack.addColorStop(0, 'rgba(96,108,136,1)');
            gBack.addColorStop(1, 'rgba(63,76,107,1)');
            ctx.fillStyle = gBack;
            ctx.fillRect(area.x, area.y, area.w, area.h);
        }
        cb = area.w / 2;
        bW = cb * 0.85;
        lX = (cb - bW) / 2;
        rX = lX + cb;
        fontface = this.fontface;
        if (OU.IS_IPHONE)
            maxH = area.h;
        else
        // maxH = Math.floor(area.h / 3.3333);
            maxH = area.h * 0.55;
        // shadow
        leftTest = new OU.util.DynText({
            txt:this.leftString,
            x:lX + 2,
            y:area.y + 2,
            w:bW,
            h:area.h,
            fontSize:maxH,
            context:ctx,
            colour:tBlack,
            align:'right',
            measureOnly:true,
            noBreak:true,
            fontFamily:fontface});
        rightTest = new OU.util.DynText({
            txt:this.rightString,
            x:lX + 2,
            y:area.y + 2,
            w:bW,
            h:area.h,
            fontSize:maxH,
            context:ctx,
            colour:tBlack,
            align:'right',
            measureOnly:true,
            noBreak:true,
            fontFamily:fontface});
        fs = leftTest.font.size > rightTest.font.size ? rightTest.font.size : leftTest.font.size;
        //////
        new OU.util.DynText({
            txt:this.leftString,
            // x:lX + (2 * OU.dpr),
            x:lX,
            y:area.y - (1 * OU.dpr),
            w:bW,
            h:area.h,
            fontSize:fs,
            context:ctx,
            colour:"rgba(0,0,0,0.5)",
            align:'right',
            noBreak:true,
            fontFamily:fontface});
        new OU.util.DynText({
            txt:this.leftString,
            // x:lX + (2 * OU.dpr),
            x:lX,
            y:area.y + (1 * OU.dpr),
            w:bW,
            h:area.h,
            fontSize:fs,
            context:ctx,
            colour:"rgba(255,255,255,0.5)",
            align:'right',
            noBreak:true,
            fontFamily:fontface});
        //////
        new OU.util.DynText({
            txt:this.rightString,
            x:rX + (0 * OU.dpr),
            y:area.y - (1 * OU.dpr),
            w:bW,
            h:area.h,
            fontSize:fs,
            context:ctx,
            colour:"rgba(0,0,0,0.5)",
            align:'left',
            noBreak:true,
            fontFamily:fontface});
        new OU.util.DynText({
            txt:this.rightString,
            x:rX + (0 * OU.dpr),
            y:area.y + (1 * OU.dpr),
            w:bW,
            h:area.h,
            fontSize:fs,
            context:ctx,
            colour:"rgba(255,255,255,0.5)",
            align:'left',
            noBreak:true,
            fontFamily:fontface});
        new OU.util.DynText({
            txt:this.arrows[1],
            x:lX + bW + (1 * OU.dpr),
            y:area.y + (1 * OU.dpr),
            w:rX - (lX + bW),
            h:area.h,
            fontSize:fs,
            context:ctx,
            colour:tBlack,
            align:'center',
            noBreak:true,
            fontFamily:fontface});
        leftText = new OU.util.DynText({
            txt:this.leftString,
            x:lX,
            y:area.y,
            w:bW,
            h:area.h,
            fontSize:fs,
            context:ctx,
            colour:'#c7c9d4',
            align:'right',
            noBreak:true,
            fontFamily:fontface});
        rightText = new OU.util.DynText({
            txt:this.rightString,
            x:rX,
            y:area.y,
            w:bW,
            h:area.h,
            fontSize:fs,
            context:ctx,
            colour:'#c7c9d4',
            align:'left',
            noBreak:true,
            fontFamily:fontface});
        goesTo = new OU.util.DynText({
            txt:this.arrows[1],
            x:lX + bW,
            y:area.y,
            w:rX - (lX + bW),
            h:area.h,
            fontSize:fs,
            context:ctx,
            colour:'#ffffff',
            align:'center',
            noBreak:true,
            fontFamily:fontface});
        this.font = leftText.font;
        this.fontSize = leftText.font.size;
        ctx.strokeStyle = '#ff0000';
        // fill the hitboxes..
        lttb = leftText.textBoxes;
        rttb = rightText.textBoxes;
        this.hitBoxes.length = 0;
        for (i = 0; i < lttb.length; i++)
            this.hitBoxes.push(lttb[i]);
        this.hitBoxes.push(goesTo.textBoxes[0]);
        for (i = 0; i < rttb.length; i++)
            this.hitBoxes.push(rttb[i]);
        for (i = this.hitBoxes.length - 1; i >= 0; i--) {
            if (runner[i]!==null) {
                clickable.push(new this.Marker(
                    {

                        x:this.hitBoxes[i].x,
                        y:this.hitBoxes[i].y,
                        w:this.hitBoxes[i].w,
                        h:this.hitBoxes[i].h,
                        item:i,
                        context:ctx,
                        isclicked:false,
                        execute:function ( inval ) {
                            self.itemclicked(inval);
                        }
                    }));
            }
        }
        // if (this.totalUse >= this.showHint) this.showHintButton();
        if (this.wasteoftime) {
            var l = this.displayLayer, ctx = l.context;
            ctx.save();
            ctx.strokeStyle = "#FF00FF";
            ctx.fillStyle = "#ffff00";
            ctx.font = '12px Courier';
            ctx.strokeRect(0, area.y, 0 + area.w, area.y + area.h);
            ctx.strokeRect(lX + 2, area.y + 2, bW, area.h);
            ctx.strokeRect(rX + (2 * OU.dpr), area.y + (2 * OU.dpr), bW, area.h);
            ctx.strokeRect(lX + bW, area.y, rX - (lX + bW), area.h);
            ctx.beginPath();
            ctx.moveTo(0, area.y);
            ctx.lineTo(0 + area.w, area.y + area.h);
            ctx.stroke();
            ctx.beginPath();
            ctx.moveTo(lX + 2 + bW, area.y + 2);
            ctx.lineTo(lX + 2, area.y + 2 + area.h);
            ctx.stroke();
            ctx.beginPath();
            ctx.moveTo(rX + (2 * OU.dpr) + bW, area.y + (2 * OU.dpr));
            ctx.lineTo(rX + (2 * OU.dpr), area.y + (2 * OU.dpr) + area.h);
            ctx.stroke();
            ctx.beginPath();
            ctx.moveTo(lX + bW + rX - (lX + bW), area.y);
            ctx.lineTo(lX + bW, area.y + area.h);
            ctx.stroke();
            ctx.fillText("30% of h : " + Math.round(area.h) + "px", 10, area.y + 20);
            ctx.fillText("textbox w 42.5% of w : " + Math.round(bW) + "px", 10, area.y + 20 + 12);
            ctx.fillText("max font height 55% of textbox.h : " + Math.round(maxH) + "px", 10, area.y + 20 + 24);
            ctx.fillText("left  53.75% of w: " + Math.round(rX) + "px", rX + (2 * OU.dpr) + 10, area.y + 20);
            ctx.restore();
        }
    };
    /**
     * Displays the spinner control for the equation species clicked on
     * @param inval
     */
    OU.activity.ChemicalBalancer.prototype.itemclicked = function ( inval ) {
        if (this.spinner!==undefined && this.spinner!==null) return null;
        if (this.tutorLayer!==undefined) this.killTutorPopup();
        var ctx = this.displayLayer.context, hbX, hbY, hbW, hbH, self = this, f = this.font;
        var area = this.areas.equation;
        // lets remove the click layer...
        self.itemClicked = inval;
        hbX = this.hitBoxes[inval].x;
        // hbY = this.hitBoxes[inval].y;
        hbW = this.hitBoxes[inval].w;
        //hbH = this.hitBoxes[inval].h;
        hbH = area.h * 0.85;
        // hbY = (this.areas.equation.y + (this.areas.equation.h / 2)) - (hbH );
        hbY = area.y + ((area.h - hbH) / 2);
        f.size = 40 * OU.dpr;
        var params = {
            x:hbX,
            //y:hbY - 50,
            y:hbY,
            w:hbW,
            h:hbH,
            max:this.maxVal,
            //maxH:this.areas.equation.h - 5,
            maxH:hbH,
            context:ctx,
            layer:this.displayLayer,
            font:f,
            currentVal:this.currentChem[inval].m,
            onReturn:function ( inval ) {
                self.spinnerReturn(inval);
            }, container:this.displayLayer.container
        };
        self.spinner = new OU.util.Spinner(params);
    };
    /**
     * Builds the running total array.
     * @param inval
     * @param r
     */
    OU.activity.ChemicalBalancer.prototype.runnerBuilder = function ( inval, r ) {
        for (var i = 0; i < inval.length; i++) {
            if (r.length > 0) r.push(null);
            r.push(inval[i]);
        }
        return r;
    };
    /**
     * Validates elements against the elements array to ensure they are correct Chemicals
     * @param  inval
     */
    OU.activity.ChemicalBalancer.prototype.validateElement = function ( inval ) {
        for (var i = 0, found = false; i < this.elements.length; i++)
            if (this.elements[i][1]===inval)
                return {el:this.elements[i][2], l:i};
        return found;
    };
    /**
     * Parses each cluster of a side
     * @param inval
     */
    OU.activity.ChemicalBalancer.prototype.parseClusters = function ( inval ) {
        var i, ws, r, c = [];
        for (i = 0; i < inval.length; i++) {
            ws = inval[i];
            r = this.clusterBreaker(ws);
            r.m = 1;
            c.push(r);
        }
        return c;
    };
    /**
     * Breaks up each cluster into it's component parts
     * @param  inval
     */
    OU.activity.ChemicalBalancer.prototype.clusterBreaker = function ( inval ) {
        var self = this, cluster = [], ts, ws, p, ms, nc, sub, sup, bracket, chem, v, bracketList = [], _char, _char1, _char1b;
        p = 0;
        ms = "";
        sub = "";
        sup = "";
        bracket = false;
        ws = inval;
        while (p < ws.length) {
            _char = ws.charCodeAt(p);
            if (_char >= 65 && _char <= 90) {
                ms += ws[p];
                _char1 = ws.charCodeAt(p + 1);
                if (_char1 >= 97 && _char1 <= 122)
                    ms += ws[p + 1];
                if (ms.length > 1) p++;
                _char1 = ws.charCodeAt(p + 1);
                if (_char1 >= 48 && _char1 <= 57) {
                    nc = 1;
                    //_char1b = ws.charCodeAt(p + nc);
                    while (_char1b = ws.charCodeAt(p + nc), _char1b >= 48 && _char1b <= 57) {
                        sub += ws[p + nc];
                        nc++;
                        // _char1b = ws.charCodeAt(p + nc);
                    }
                    // we have a number..
                    p += (nc - 1);
                }
                if (ws.charCodeAt(p + 1)===123) {
                    // we have a curly bracket
                    p++;
                    nc = 1;
                    ts = '';
                    self.charged = true;
                    while (ws.charCodeAt(p + nc)!==125) {
                        ts += ws[p + nc];
                        nc++;
                    }
                    if (ts.length > 0) sup = ts;
                    if (sup==="-") sup = "-1";
                    p += nc;
                }
                v = self.validateElement(ms);
                sup = sup.length > 0 ? sup : 0;
                sub = sub.length > 0 ? sub : 1;
                chem = {el:ms, sub:sub, sup:sup, plus:false, bracket:bracket, n:v};
            }
            else {
                _char = ws.charCodeAt(p);
                if (_char===40 || _char===91) {
                    // we have ( or [
                    bracketList.push(_char===40 ? 41 : 93);
                    bracket = _char;
                    nc = 0;
                    p++;
                    while (bracketList.length!==0) {
                        _char1 = ws.charCodeAt(p + nc);
                        if (_char1===bracketList[bracketList.length - 1]) bracketList.pop();
                        if (_char1===40 || _char1===91) {
                            bracketList.push(_char1===40 ? 41 : 93);
                        }
                        if (bracketList.length > 0) {
                            ms += ws[p + nc];
                            nc++;
                        }
                    }
                    chem = self.clusterBreaker(ms);
                    chem.newBracket = bracket;
                    p += (nc + 1);
                    _char = ws.charCodeAt(p);
                    if (_char >= 48 && _char <= 57) {
                        nc = 0;
                        while (_char1 = ws.charCodeAt(p + nc), _char1 >= 48 && _char1 <= 57) {
                            sub += ws[p + nc];
                            nc++;
                        }
                        // we have a number..
                        chem.sub = sub;
                        // p += (nc);
                    }
                }
                else if (ws.charCodeAt(p)===123) {
                    // we have a curly bracket
                    p++;
                    nc = 0;
                    ts = '';
                    self.charged = true;
                    while (ws.charCodeAt(p + nc)!==125) {
                        ts += ws[p + nc];
                        nc++;
                    }
                    if (ts.length > 0) sup = ts;
                    p += nc;
                    if (ts==="-") ts = "e";
                    chem = {el:ts, sub:"", sup:"", plus:false, bracket:bracket, n:"electron"};
                }
            }
            cluster.push(chem);
            p++;
            ms = "";
            sub = "";
            sup = "";
            bracket = false;
        }
        return cluster;
    };
    OU.activity.ChemicalBalancer.prototype.weighSideinArray = function ( a, e ) {
        for (var i = 0, l = a.length; i < l; i++) if (a[ i ][0]===e) return i;
        return -1;
    };
    /**
     * Weighs each side to calculate how many chemicals it has
     * @param side
     * @param balance
     * @param multi
     */
    OU.activity.ChemicalBalancer.prototype.weighSide = function ( side, balance, multi ) {
        var cluster, m, curW, j, c, v, tc, bm, masterM = multi || 1, i, cM = 0;
        for (i = 0; i < side.length; i++) {
            // clusters...
            cluster = side[i];
            bm = cluster.bm || 1;
            m = (parseInt(cluster.m, 10) || 1) * masterM * bm;
            // find charge
            for (j = 0, cM = 0; j < cluster.length; j++) cM = cluster[j].sup || cM;
            // balance charge.
            v = this.weighSideinArray(balance, 'chg');
            if (v==-1) balance.push(['chg', m * cM]);
            else balance[v][1] += (m * cM);
            for (j = 0; j < cluster.length; j++) {
                c = cluster[j];
                if (typeOf(c)=="object") {
                    curW = (m ) * (c.sub || 1) > 0 ? (m ) * (c.sub || 1) : 1;
                    v = this.weighSideinArray(balance, c.el);
                    if (v==-1) balance.push([c.el, curW]);
                    else balance[v][1] += curW;
                }
                else {
                    tc = [];
                    tc[0] = c;
                    //tc[0].m = c.sub;
                    tc[0].m = 1;
                    tc[0].bm = c.sub;
                    this.weighSide(tc, balance, m);
                }
            }
        }
    };
    OU.activity.ChemicalBalancer.prototype.balanceBuildertotalize = function ( a ) {
        for (var t = 0, i = 0, l = a.length; i < l; i++)
            t += a[ i ][1];
        return t;
    };
    /**
     * Balances the sides
     */
    OU.activity.ChemicalBalancer.prototype.balanceBuilder = function () {
        this.balanceLeft = [];
        this.balanceRight = [];
        this.weighSide(this.leftCluster, this.balanceLeft);
        this.weighSide(this.rightCluster, this.balanceRight);
        this.balanceLeft.total = this.balanceBuildertotalize(this.balanceLeft);
        this.balanceRight.total = this.balanceBuildertotalize(this.balanceRight);
    };
    /**
     * Rebuilds the display string after it has been broken down
     * @param inval
     * @param depth
     */
    OU.activity.ChemicalBalancer.prototype.rebuilder = function ( inval, depth ) {
        var self = this, i, hArray = [], cur, d = depth || 0;
        // this should take an array
        if (d > 5) throw "loopy";
        for (i = 0; i < inval.length; i++) {
            cur = inval[i];
            if (typeOf(cur)=="array") {

                // array concat
                if (d==0 && hArray.length > 0)  hArray.push(' + ');
                if (cur.m!=undefined && cur.m > 1) hArray.push(cur.m);
                hArray.push(self.rebuilder(cur, d + 1));
                if (cur.sub!=undefined && cur.sub > 1) hArray.push("<sub>" + cur.sub + "</sub>");
            }
            if (typeOf(cur)=="object") {
                hArray.push(cur.el);
                if (cur.sub!="" && cur.sub > 1) hArray.push("<sub>" + cur.sub + "</sub>");
                if (cur.sup!="") hArray.push(this.makeSuperScript(cur.sup));
            }
        }
        if (depth > 1) {
            hArray.splice(0, 0, this.makeBracket(inval.newBracket, 1));
            hArray.push(this.makeBracket(inval.newBracket, 2));
        }
        //  return h;
        return hArray.join("");
    };
    /**
     * Selects and inserts the correct bracket
     * @param inval
     * @param mode
     */
    OU.activity.ChemicalBalancer.prototype.makeBracket = function ( inval, mode ) {
        //  var makeBracket = function ( inval, mode ) {
        if (inval==40 && mode==1) return "(";
        if (inval==40 && mode==2) return ")";
        if (inval==91 && mode==1) return "[";
        if (inval==91 && mode==2) return "]";
    };
    /**
     * Displays a superscript element with the charge symbol the correct way round
     * @param inval
     */
    /* pushing this out */
    OU.activity.ChemicalBalancer.prototype.makeSuperScript = function ( inval ) {
        var ws = "<sup>", v = parseInt(inval, 10), sign;
        sign = v && v / Math.abs(v)==-1 ? "-" : "+";
        v = Math.abs(v);
        ws += (v==1 ? sign : v + sign) + "</sup>";
        return ws;
    };
    OU.activity.ChemicalBalancer.prototype.checkSaveData = function () {
        if (OU.saveData!=undefined && OU.saveData!=null) return null;
        var saveData = OU.LocalStorage.load('ou.chembal.app') || null;
        if (saveData==undefined) {
            this.buildSaveData();
        }
        else {
            OU.saveData = JSON.parse(saveData);
        }
        OU.util.exportSaveData = function () {

            // quick rebuild of the completed levels..
            var i, c, j, u = 1;
            for (i = 0; i < OU.saveData.completedLevels.length; i++) {
                c = 0;
                for (j = 0; j < OU.saveData.completedSections[i].length; j++) {
                    if (typeof OU.saveData.completedSections[i][j]=='object') c++;
                }
                OU.saveData.completedLevels[i] = c;
                if (OU.saveData.completedLevels[i]==OU.saveData.completedSections[i].length) u++;
            }
            OU.saveData.unlocked = u;
            OU.LocalStorage.save('ou.chembal.app', JSON.stringify(OU.saveData));
        };
        OU.util.loadSaveData = function () {
            var saveData = OU.LocalStorage.load('ou.chembal.app') || null;
            if (saveData!==undefined) OU.saveData = JSON.parse(saveData);
        };
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.ChemicalBalancer, OU.util.Activity);


