/**
 * OU.activity.chemicalBalancer Object
 * @fileOverview Chemical Equation Balancer Activity
 * @param {JSON array} data Data as included in the data/data.js file
 *
 *
 * @author Martin Donnelly
 *
 * Special version for a guided tutorial
 *
 */
OU.require('OU.util.DynText', '../library/js/util/dynText.js');
OU.require('OU.util.Layer', '../library/js/util/layer.js');
OU.require('OU.util.Spinner', '../library/js/util/spinner.js');
/**
 * @constructor
 * @param data
 * @param instance
 * @param controller
 */
OU.activity.ChemicalBalancerTutor = function ( data, instance, controller ) {
    OU.activity.ChemicalBalancerTutor.prototype.canvasView = function () {
        this.elements = [
        [1, 'H', 'Hydrogen'],
        [2, 'He', 'Helium'],
        [3, 'Li', 'Lithium'],
        [4, 'Be', 'Beryllium'],
        [5, 'B', 'Boron'],
        [6, 'C', 'Carbon'],
        [7, 'N', 'Nitrogen'],
        [8, 'O', 'Oxygen'],
        [9, 'F', 'Fluorine'],
        [10, 'Ne', 'Neon'],
        [11, 'Na', 'Sodium'],
        [12, 'Mg', 'Magnesium'],
        [13, 'Al', 'Aluminium'],
        [14, 'Si', 'Silicon'],
        [15, 'P', 'Phosphorus'],
        [16, 'S', 'Sulfur'],
        [17, 'Cl', 'Chlorine'],
        [18, 'Ar', 'Argon'],
        [19, 'K', 'Potassium'],
        [20, 'Ca', 'Calcium'],
        [21, 'Sc', 'Scandium'],
        [22, 'Ti', 'Titanium'],
        [23, 'V', 'Vanadium'],
        [24, 'Cr', 'Chromium'],
        [25, 'Mn', 'Manganese'],
        [26, 'Fe', 'Iron'],
        [27, 'Co', 'Cobalt'],
        [28, 'Ni', 'Nickel'],
        [29, 'Cu', 'Copper'],
        [30, 'Zn', 'Zinc'],
        [31, 'Ga', 'Gallium'],
        [32, 'Ge', 'Germanium'],
        [33, 'As', 'Arsenic'],
        [34, 'Se', 'Selenium'],
        [35, 'Br', 'Bromine'],
        [36, 'Kr', 'Krypton'],
        [37, 'Rb', 'Rubidium'],
        [38, 'Sr', 'Strontium'],
        [39, 'Y', 'Yttrium'],
        [40, 'Zr', 'Zirconium'],
        [41, 'Nb', 'Niobium'],
        [42, 'Mo', 'Molybdenum'],
        [43, 'Tc', 'Technetium'],
        [44, 'Ru', 'Ruthenium'],
        [45, 'Rh', 'Rhodium'],
        [46, 'Pd', 'Palladium'],
        [47, 'Ag', 'Silver'],
        [48, 'Cd', 'Cadmium'],
        [49, 'In', 'Indium'],
        [50, 'Sn', 'Tin'],
        [51, 'Sb', 'Antimony'],
        [52, 'Te', 'Tellurium'],
        [53, 'I', 'Iodine'],
        [54, 'Xe', 'Xenon'],
        [55, 'Cs', 'Cesium'],
        [56, 'Ba', 'Barium'],
        [57, 'La', 'Lanthanum'],
        [58, 'Ce', 'Cerium'],
        [59, 'Pr', 'Praseodymium'],
        [60, 'Nd', 'Neodymium'],
        [61, 'Pm', 'Promethium'],
        [62, 'Sm', 'Samarium'],
        [63, 'Eu', 'Europium'],
        [64, 'Gd', 'Gadolinium'],
        [65, 'Tb', 'Terbium'],
        [66, 'Dy', 'Dysprosium'],
        [67, 'Ho', 'Holmium'],
        [68, 'Er', 'Erbium'],
        [69, 'Tm', 'Thulium'],
        [70, 'Yb', 'Ytterbium'],
        [71, 'Lu', 'Lutetium'],
        [72, 'Hf', 'Hafnium'],
        [73, 'Ta', 'Tantalum'],
        [74, 'W', 'Tungsten'],
        [75, 'Re', 'Rhenium'],
        [76, 'Os', 'Osmium'],
        [77, 'Ir', 'Iridium'],
        [78, 'Pt', 'Platinum'],
        [79, 'Au', 'Gold'],
        [80, 'Hg', 'Mercury'],
        [81, 'Tl', 'Thallium'],
        [82, 'Pb', 'Lead'],
        [83, 'Bi', 'Bismuth'],
        [84, 'Po', 'Polonium'],
        [85, 'At', 'Astatine'],
        [86, 'Rn', 'Radon'],
        [87, 'Fr', 'Francium'],
        [88, 'Ra', 'Radium'],
        [89, 'Ac', 'Actinium'],
        [90, 'Th', 'Thorium'],
        [91, 'Pa', 'Protactinium'],
        [92, 'U', 'Uranium'],
        [93, 'Np', 'Neptunium'],
        [94, 'Pu', 'Plutonium'],
        [95, 'Am', 'Americium'],
        [96, 'Cm', 'Curium'],
        [97, 'Bk', 'Berkelium'],
        [98, 'Cf', 'Californium'],
        [99, 'Es', 'Einsteinium'],
        [100, 'Fm', 'Fermium'],
        [101, 'Md', 'Mendelevium'],
        [102, 'No', 'Nobelium'],
        [103, 'Lr', 'Lawrencium'],
        [104, 'Rf', 'Rutherfordium'],
        [105, 'Db', 'Dubnium'],
        [106, 'Sg', 'Seaborgium'],
        [107, 'Bh', 'Bohrium'],
        [108, 'Hs', 'Hassium'],
        [109, 'Mt', 'Meitnerium'],
        [110, 'Ds', 'Darmstadtium'],
        [111, 'Rg', 'Roentgenium'],
        [112, 'Cn', 'Copernicium'],
        [113, 'Uut', 'Ununtrium'],
        [114, 'Uuq', 'Ununquadium'],
        [115, 'Uup', 'Ununpentium'],
        [116, 'Uuh', 'Ununhexium'],
        [117, 'Uus', 'Ununseptium'],
        [118, 'Uuo', 'Ununoctium']
        ];
        this.displayMode = this.data.displayMode || 2;
        this.currentEquation = this.data.equation;
        this.solution = this.data.solution;
        this.showHint = this.data.tries || 5;
        this.maxVal = this.data.max || 20;
        this.clickable = new OU.util.TypedArray();
        this.aRGB = [];
        this.aColour = [];
        this.score = 0;
        this.totalScore = 0;
        this.doRender = true;
        this.currentChem = [];
        this.hitBoxes = [];
        this.fontSize = 60;
        this.font = null;
        this.charged = false;
        this.leftCluster = [];
        this.rightCluster = [];
        this.balanceLeft = [];
        this.balanceRight = [];
        this.leftString = "";
        this.rightString = "";
        this.arrows = ["\u2192", "\u21d2", "\u21e2", "\u21e8"];
        this.totalUse = 0;
        this.tbh = 40 * OU.dpr;
        this.changeLimit = 0;
        this.changeCount = 0;
        this.allowedAttempts = 0;
        this.currentAttempts = 0;
        this.dontsave = false;
        this.saved = false;
        this.balanced = false;
        // create Canvas & Context
        this.bgLayer = new OU.util.Layer({
            container:this
        });
        this.displayLayer = new OU.util.Layer({
            container:this,
            hasEvents:true
        });
        /*     this.tutorLayer = new OU.util.Layer({
         container:this,
         hasEvents:false
         });
         */
        this.fontface = "league,san-serif";
        this.defineAreas();
        this.currentLevel = 0;
        this.checkSaveData();
        this.processEquation();
        this.balanceBuilder();
        this.discoverChangeLimit();
        this.tutorSteps = [
        {
            x:0,
            y:0,
            txt:'To begin, touch the symbol for Sodium "Na" below'
        },

        {
            x:0,
            y:0,
            txt:'Now slide your finger up the slider to select the number 2'
        },

        {
            x:0,
            y:0,
            txt:'Now tap the center or outside the spinner to select the value'
        },

        {
            x:0,
            y:0,
            txt:'You will see that the symbol for Sodium is now multiplied by 2'
        },

        {
            x:0,
            y:0,
            txt:'You will also see that the value has changed here too'
        },

        {
            x:0,
            y:0,
            txt:'And the balance bar has now balanced out showing that you have balanced that molecule'
        },

        {
            x:0,
            y:0,
            txt:'Well done. You have successfully balanced both molecules. Click the right hand arrow above to try your luck with more'
        }
        ];
        this.tutor = 0;
    //  this.redraw();
    };
    OU.activity.ChemicalBalancerTutor.prototype.killTutorPopup = function () {
        if (this.tutorLayer) {
            this.tutorLayer.remove();
            this.tutorLayer = null;
        }
    };
    OU.activity.ChemicalBalancerTutor.prototype.tutorStepAlpha = function () {
        this.killTutorPopup();
        this.tutor = 3;
        this.resize();
    /*
         setTimeout(function () {
         self.tutorStepFour();
         }, 7000);

         */
    };
    OU.activity.ChemicalBalancerTutor.prototype.tutorStepBeta = function () {
        this.killTutorPopup();
        var self = this, params = {
            x:this.tutorSteps[4].x,
            y:this.tutorSteps[4].y,
            w:(this.w / 5),
            h:(this.w / 5) / 1.7,
            txt:this.tutorSteps[5].txt,
            k:0
        };
        this.tutorPopup(params);
        this.tutor = 5;
        setTimeout(function () {
            self.tutorStepGamma();
        }, 5000);
    };
    OU.activity.ChemicalBalancerTutor.prototype.tutorStepGamma = function () {
        this.killTutorPopup();
        this.tutor = 9;
        this.REprocessEquation();
        this.balanceBuilder();
        this.redraw();
    };
    OU.activity.ChemicalBalancerTutor.prototype.tutorStepDelta = function () {
        this.killTutorPopup();
        this.tutorPopup({
            x:this.tutorSteps[6].x,
            y:this.tutorSteps[6].y,
            w:this.tutorSteps[6].w,
            h:this.tutorSteps[6].h,
            txt:this.tutorSteps[6].txt,
            k:1
        });
    };
    OU.activity.ChemicalBalancerTutor.prototype.tutorPopup = function ( p ) {
        var x = p.x, y = p.y, w = p.w, h = p.h, txt = p.txt, tx = p.tx, k = p.k;
        this.tutorLayer = new OU.util.Layer({
            container:this,
            hasEvents:false,
            x:x - 10,
            y:y - 10,
            w:w + 20,
            h:h + 20
        });
        x = 10;
        y = 10;
        w = w - 10;
        h = h - 10;
        var ctx = this.tutorLayer.context;
//        ctx.drawImage(OU.preLoadImg[30 + k], x, y, w, h);
        h = h * 0.9;
        if (k <= 1)  y = y + (h * 0.1);
        new OU.util.DynText({
            txt:txt,
            x:x,
            y:y - (1 * OU.dpr),
            w:w,
            h:h,
            fontSize:h / 3,
            context:ctx,
            colour:"#000",
            align:'center',
            noBreak:false,
            fontFamily:this.fontface
        });

        new OU.util.DynText({
            txt:txt,
            x:x,
            y:y,
            w:w,
            h:h,
            fontSize:h / 3,
            context:ctx,
            colour:"#fff",
            align:'center',
            noBreak:false,
            fontFamily:this.fontface
        });
    };
    OU.activity.ChemicalBalancerTutor.prototype.remove = function () {

        // check for rogue spinners...
        //var e = document.getElementById('spinner');
        if (this.spinner!==undefined && this.spinner!=null) {
            this.spinner.removeSpinner();
            this.spinner = null;
        }
    };
    /**
     * Defines the layout depending on the rotation of the device
     */
    OU.activity.ChemicalBalancerTutor.prototype.defineAreas = function () {
        this.areas = {
            feedback:{
                x:0,
                y:0,
                w:this.w,
                h:this.h * 0.7
            },
            equation:{
                x:0,
                y:this.h * 0.7,
                w:this.w,
                h:this.h * 0.3
            }
        };
    };
    /**
     * Displays the hint button
     */

    OU.activity.ChemicalBalancerTutor.prototype.showHintButton = function () {
        var l = this.displayLayer, area = this.areas.equation, self = this;
        this.hintButton = new OU.util.Button({
            x:(area.w - 160) / 2,
            y:area.y + area.h - 50,
            w:160,
            h:40,
            container:l,
            align:"center",
            txt:"Solve",
            padding:1,
            verticalPadding:1,
            layer:l,
            onClick:function () {
                self.solve();
            },
            background:{
                col:"#FF8624",
                radius:3,
                borderCol:OU.theme.buttonBorderCol
                },
            onOffIndicator:false,
            onOff:false
        });
        l.events.clickable.push(this.hintButton);
    };
    /**
     * Solves the equation for the user
     */
    OU.activity.ChemicalBalancerTutor.prototype.discoverChangeLimit = function () {
        var t, i = 0;
        for (t = 0; t < this.currentChem.length; t++) {
            if (this.currentChem[t]!==null) {
                if (this.solution[i] > 1) this.changeLimit++;
                i++;
            }
        }
        this.allowedAttempts = this.changeLimit;
    };
    /**
     * Solves the equation for the user
     */
    OU.activity.ChemicalBalancerTutor.prototype.solve = function () {
        this.dontsave = true;
        var t, i = 0;
        for (t = 0; t < this.currentChem.length; t++) {
            if (this.currentChem[t]!==null) {
                this.currentChem[t].m = this.solution[i];
                i++;
            }
        }
        this.REprocessEquation();
        this.balanceBuilder();
        this.redraw();
    };
    /**
     * the callback from the spinner control
     * @param {int} inval the value selected by the user
     */

    OU.activity.ChemicalBalancerTutor.prototype.spinnerReturn = function ( inval ) {
        if (this.spinner!==undefined) {
            // kill the spinner.
            this.spinner.removeSpinner();
            delete this.spinner;
            this.spinner = undefined;
        }
        this.killTutorPopup();
        this.currentChem[this.itemClicked].m = inval;
        this.REprocessEquation();
        this.balanceBuilder();
        this.totalUse += 1;
        this.currentAttempts++;
        this.redraw();
    //if (this.totalUse >= this.showHint) this.showHintButton();
    }
    /**
     * Handles the touch event
     * @param {object} p Contains {x,y,context,w,h,execute,item}
     */
    OU.activity.ChemicalBalancerTutor.prototype.Marker = function ( p ) {
        this.mX = p.x;
        this.mY = p.y;
        this.ctx = p.context;
        this.mW = p.w;
        this.mH = p.h;
        this.execute = p.execute;
        this.item = p.item;
        this.clickable = this.ctx.canvas.events.clickable;
        /**
         * Checks the location of the hit on the screen
         * @param x
         * @param y
         * @param pressed
         * @param events
         */
        OU.activity.ChemicalBalancerTutor.prototype.Marker.prototype.isHit = function ( x, y, pressed, events ) {
            var self = this;
            if ((pressed ) && ((x >= self.mX) && (x <= self.mX + self.mW) ) && ((y >= self.mY) && ( y <= self.mY + self.mH))) {
                events.flush();
                if (!this.spinner) self.execute(self.item);
            }
        };
    };
    /**
     * Called automatically when window is resized / rotated..
     */
    OU.activity.ChemicalBalancerTutor.prototype.resize = function () {
        OU.activity.ChemicalBalancerTutor.superClass_.resize.call(this); // call the parent class resize
        if (this.spinner!==undefined) {
            // kill the spinner.
            this.spinner.removeSpinner();
            this.spinner = null;
        }
        this.bgLayer.resize();
        //   this.tbh = this.h * 0.059;
        if (this.displayMode!==1) {
            var self = this;
            self.bgLayer.context.drawImage(OU.preLoadImg[1], 0, 0, self.bgLayer.w, self.bgLayer.h);
        }
        this.displayLayer.resize();
        this.redraw();
    };
    OU.activity.ChemicalBalancerTutor.prototype.redraw = function () {
            console.log('define areas');
        this.defineAreas();
            console.log('display eq');
        this.displayEquation();
            console.log('display balance');
        this.displayBalance();
        if (this.displayMode!==1) {
            this.displaySection();
            this.displayLevel();
            this.displayTopBar();
        }
        if (this.balanced) {
            this.win();
        }
    };
    /**
     * Re-processes the equation when a value has been changed
     */



    OU.activity.ChemicalBalancerTutor.prototype.REprocessEquation = function () {
        var runner = [];
        this.leftString = this.rebuilder(this.leftCluster);
        this.rightString = this.rebuilder(this.rightCluster);
        runner = this.runnerBuilder(this.leftCluster, runner);
        runner = this.runnerBuilder(this.rightCluster, runner);
        this.currentChem = runner;
    };
    /**
     * Process the equation for the first time
     */
    OU.activity.ChemicalBalancerTutor.prototype.processEquation = function () {
        var start = new Date().getTime();
        var left, right, wa = this.currentEquation.breaker('=');
        left = wa[0].breaker(' + ');
        right = wa[1].breaker(' + ');
        this.leftCluster = this.parseClusters(left);
        this.rightCluster = this.parseClusters(right);
        this.REprocessEquation();
        var end = new Date().getTime();
        var time = end - start;
    };
    /**
     * Displays the small balance bar
     * @param {object} p Contains {x,y,,w,h,ctx,l,r}
     */
    OU.activity.ChemicalBalancerTutor.prototype.displayBalanceBar = function ( p ) {
        var x = p.x, y = p.y, h = p.h, w = h * 1.82, o = (p.w - w) / 2, ctx = p.ctx, l = p.l, r = p.r, i = 17;
        if (l > r)
            i = 18;
        else if (l < r)
            i = 19;
        ctx.clearRect(x, y, w, h);
        //ctx.drawImage(OU.preLoadImg[7], 10, this.tbh + 10, bw, bw);
        ctx.drawImage(OU.preLoadImg[i], x + o, y, w, h);
    };
    OU.activity.ChemicalBalancerTutor.prototype.showWin = function () {
        var self = this;
        this.winLayer.canvas.setAttribute("class", "winin");
        setTimeout(function () {
            self.unshowWin();
        }, 2000);
    };
    OU.activity.ChemicalBalancerTutor.prototype.unshowWin = function () {
        var self = this;
        this.winLayer.canvas.setAttribute("class", "winout");
        this.displaySection();
        setTimeout(function () {
            self.killWin();
        }, 390);
    };
    OU.activity.ChemicalBalancerTutor.prototype.killWin = function () {
        this.winLayer.remove();
    // console.log("remove!");
    };
    OU.activity.ChemicalBalancerTutor.prototype.win = function () {
        var l, ctx , area = this.areas.feedback, self = this, clickable, w, h, x, y , cInner = this.controller, cOuter = cInner.controller, d = {};
        // 27
        // y'know what, lets create a new layer for this..
        if (this.tutor!=9) return null;
        this.winLayer = new OU.util.Layer({
            // container:this,
            container:this.displayLayer.container,
            hasEvents:false,
            instance:this.instance + '.winning'
        /*,
             x:this.x,
             y:this.y,
             h:this.h,
             w:this.w */
        });
        if (!this.dontsave) {
            d.lost = this.allowedAttempts - this.currentAttempts;
            OU.saveData.completedSections[cOuter.sectionNum - 1][cInner.sectionNum] = d;
            OU.util.exportSaveData();
        }
        this.winLayer.canvas.setAttribute("class", "win");
        l = this.winLayer;
        ctx = l.context;
        w = 532 * OU.dpr;
        h = 91 * OU.dpr;
        x = (this.w - w) / 2;
        y = (this.h - h) / 2;
        ctx.drawImage(OU.preLoadImg[27], x, y, w, h);
        // this.winLayer.canvas.class = "winout";
        this.tutor = 10;
        setTimeout(function () {
            self.showWin();
        }, 750);
    };
    OU.activity.ChemicalBalancerTutor.prototype.displayTopBar = function () {
        var l = this.displayLayer, ctx = l.context, area = this.areas.feedback, self = this, clickable = l.events.clickable;
        ;
        var bx, by = this.tbh * 0.1, bh = this.tbh * 0.9, bw;
        //this.tbh = this.h * 0.059;
        var gBack = ctx.createLinearGradient(0, 0, 0, this.tbh);
        gBack.addColorStop(0, '#b5e3f2');
        gBack.addColorStop(0.03, '#50bbdf');
        gBack.addColorStop(0.97, '#50bbdf');
        gBack.addColorStop(1, '#295e6e');
        ctx.fillStyle = gBack;
        ctx.fillRect(area.x, area.y, area.w, this.tbh);
        var space = ((25 * 3) + 14 + (4 * 3)) * OU.dpr;
        var vpos = (this.tbh - (24 * OU.dpr)) / 2;
        var hl = (this.w - space) / 2;
        var hs = (25 * OU.dpr) + (4 * OU.dpr);
        ctx.drawImage(OU.preLoadImg[26], this.w - (35 * OU.dpr), (2 * OU.dpr), 29 * OU.dpr, 35 * OU.dpr);
        if (this.currentAttempts - this.allowedAttempts < 1)
            ctx.drawImage(OU.preLoadImg[25], hl, vpos, (25 * OU.dpr), 24 * OU.dpr);
        else
            ctx.drawImage(OU.preLoadImg[24], hl, vpos, (25 * OU.dpr), 24 * OU.dpr);
        if (this.currentAttempts - this.allowedAttempts < 2)
            ctx.drawImage(OU.preLoadImg[25], hl + hs, vpos, (25 * OU.dpr), 24 * OU.dpr);
        else
            ctx.drawImage(OU.preLoadImg[24], hl + hs, vpos, (25 * OU.dpr), 24 * OU.dpr);
        if (this.currentAttempts - this.allowedAttempts < 3)
            ctx.drawImage(OU.preLoadImg[25], hl + (hs * 2), vpos, (25 * OU.dpr), 24 * OU.dpr);
        else
            ctx.drawImage(OU.preLoadImg[24], hl + (hs * 2), vpos, (25 * OU.dpr), 24 * OU.dpr);
        if (this.currentAttempts - this.allowedAttempts < 3)
            ctx.drawImage(OU.preLoadImg[22], hl + (hs * 3), vpos, (15 * OU.dpr), 24 * OU.dpr);
        else {
            ctx.drawImage(OU.preLoadImg[23], hl + (hs * 3), vpos, (15 * OU.dpr), 24 * OU.dpr);
            //self.solve();
            clickable.push(new this.Marker({
                x:hl + (hs * 3),
                y:vpos,
                w:(15 * OU.dpr),
                h:24 * OU.dpr,
                context:ctx,
                isclicked:false,
                execute:function () {
                    self.solve();
                }
            }));
        }
        if (this.backButton==undefined || this.backButton==null) {
            this.backButton = new OU.util.Button({
                x:10,
                y:by / 2,
                w:160,
                h:bh,
                container:l,
                align:"center",
                txt:"Back",
                padding:1,
                verticalPadding:1,
                layer:l,
                onClick:function () {
                    self.gotoMenu();
                },
                background:{
                    col:"#50bbdf",
                    radius:3,
                    borderCol:OU.theme.buttonBorderCol
                    },
                onOffIndicator:false,
                onOff:false
            });
        }
        else {
        //  this.backButton.redraw();
        }
    };
    /**
     * Displays the balance of each side
     */
    OU.activity.ChemicalBalancerTutor.prototype.displayBalance = function () {
        /**
         * Used to locate a specific element (e) withing an array (a)
         * @param a
         * @param e
         */
        var inArray = function ( a, e ) {
            for (var i = 0, l = a.length; i < l; i++) {
                if (a[ i ][0]===e) {
                    return i;
                }
            }
            return -1;
        };
        /**
         * Draws a balance pair
         * @param ctx
         * @param o
         * @param i
         * @param q
         * @param lt
         * @param rt
         * @param lv
         * @param rv
         * @param f
         * @param ft
         */
        var drawBalancePair = function ( ctx, o, i, q, lt, rt, lv, rv, f, ft, self ) {
            var pc = self.displayMode===1?'rgba(21,21,21,0.9)':'#ffffff',
            fh = ft * 0.7;
            ctx.strokeRect(q * 2, o + (i * ft), q / 2, ft);
            ctx.strokeRect(q * 2 + (q / 2), o + (i * ft), q / 2, ft);
            ctx.strokeRect(q * 5, o + (i * ft), q / 2, ft);
            ctx.strokeRect(q * 5 + (q / 2), o + (i * ft), q / 2, ft);
            new OU.util.DynText({
                txt:lt,
                x:q * 2,
                y:o + (i * ft),
                w:q / 2,
                h:ft,
                fontSize:fh,
                context:ctx,
                colour:pc,
                align:'center',
                noBreak:true,
                fontFamily:f
            });
            new OU.util.DynText({
                txt:lv.toString(),
                x:q * 2 + (q / 2),
                y:o + (i * ft),
                w:q / 2,
                h:ft,
                fontSize:fh,
                context:ctx,
                colour:pc,
                align:'center',
                noBreak:true,
                fontFamily:f
            });
            new OU.util.DynText({
                txt:rv.toString(),
                x:q * 5,
                y:o + (i * ft),
                w:q / 2,
                h:ft,
                fontSize:fh,
                context:ctx,
                colour:pc,
                align:'center',
                noBreak:true,
                fontFamily:f
            });
            new OU.util.DynText({
                txt:rt,
                x:q * 5 + (q / 2),
                y:o + (i * ft),
                w:q / 2,
                h:ft,
                fontSize:fh,
                context:ctx,
                colour:pc,
                align:'center',
                noBreak:true,
                fontFamily:f
            });
            if (self.tutor==3 && lt=="Na") {
                self.tutorSteps[3].x = q * 2 + (q / 2) + 15;
                self.tutorSteps[3].y = o + (i * ft) + ft + 10;
                self.killTutorPopup();
            }
        };


        var self = this, ctx = this.displayLayer.context, q = this.w / 8, o, l = this.balanceLeft.length, i, offset, area = this.areas.feedback, dc;
        var gBack, balanced = true;
        // var ft = 40 * OU.dpr;
        var ft = this.balanceLeft.length > 5 ? (area.h * 0.95) / this.balanceLeft.length : (area.h * 0.95) / 5;
        if (this.displayMode===1) {
            gBack = ctx.createLinearGradient(0, area.y, 0, area.y + area.h)
            gBack.addColorStop(0, 'rgba(200,215,220,1)');
            gBack.addColorStop(0.37, 'rgba(227,234,237,1)');
            gBack.addColorStop(1, 'rgba(242,245,246,1)');
            ctx.fillStyle = gBack;
            ctx.fillRect(area.x, area.y, area.w, area.h);
            ctx.strokeStyle = "rgb(21,21,21)";
        }
        else
            ctx.strokeStyle = "#ffffff";
        // check for charge:
        dc = (Math.abs(this.balanceLeft[inArray(this.balanceLeft, "chg")][1]) > 0 || Math.abs(this.balanceRight[inArray(this.balanceRight, "chg")][1]) > 0);
        offset = dc ? (area.h - ((l + 1) * ft)) / 2 : (area.h - ((l - 1) * ft)) / 2;
        for (i = 1; i < l; i++) {
            o = inArray(this.balanceRight, this.balanceLeft[i][0]);
            if (this.balanceLeft[i][1]!=this.balanceRight[o][1]) {
                balanced = false;
            }
            drawBalancePair(ctx, offset, i - 1, q, this.balanceLeft[i][0], this.balanceRight[o][0], this.balanceLeft[i][1], this.balanceRight[o][1], this.fontface, ft, this);
            this.displayBalanceBar({
                x:(q * 4) - (q / 2),
                y:offset + ((i - 1) * ft),
                w:q,
                h:ft,
                ctx:ctx,
                l:this.balanceLeft[i][1],
                r:this.balanceRight[o][1]

            });
            if (i==1 && this.tutor==3) {
                //this.tutorSteps[4].x = (q * 4) - (q / 2)+q;
                this.tutorSteps[4].x = (q * 4) - 15;
                this.tutorSteps[4].y = offset + ((i - 1) * ft) + ft + 5;
            }
        }
        if (this.charged) {
            drawBalancePair(ctx, offset, i, q, 'Charge', 'Charge', (parseInt(this.balanceLeft[0][1], 10) > 0 ? "+" + this.balanceLeft[0][1] : this.balanceLeft[0][1]) || 0, (parseInt(this.balanceRight[0][1], 10) > 0 ? "+" + this.balanceRight[0][1] : this.balanceRight[0][1]) || 0, this.fontface, ft, this);
            this.displayBalanceBar({
                x:(q * 4) - (q / 2),
                y:offset + ((i) * ft),
                w:q,
                h:ft,
                ctx:ctx,
                l:this.balanceLeft[0][1],
                r:this.balanceRight[0][1]

            });
            if (this.balanceLeft[0][1]!=this.balanceRight[0][1]) {
                balanced = false;
            }
        }
        this.balanced = balanced;
        /*
         they don't like these numbers but they are useful for debugging
         new OU.util.DynText({
         txt:this.balanceLeft.total,
         x:0,
         y:(area.h / 2) - 30,
         w:300,
         h:60,
         fontSize:60,
         context:ctx,
         colour:'rgba(21,21,21,0.9)',
         align:'center',
         noBreak:true  });
         // display right amount
         new OU.util.DynText({
         txt:this.balanceRight.total,
         x:this.w - 300,
         y:(area.h / 2) - 30,
         w:300,
         h:60,
         fontSize:60,
         context:ctx,
         colour:'rgba(21,21,21,0.9)',
         align:'center',
         noBreak:true  });

         */
        if (this.tutor==3) {
            var params = {
                x:this.tutorSteps[3].x,
                y:this.tutorSteps[3].y,
                w:(this.w / 5),
                h:(this.w / 5) / 1.7,
                txt:this.tutorSteps[4].txt,
                k:0
            };
            this.tutorPopup(params);
            this.tutor = 4;
        }
        if (this.tutor==4 && (this.balanceLeft[1][1]==this.balanceRight[1][1])) {
            setTimeout(function () {
                self.tutorStepBeta();
            }, 5000);
        }
    };
    OU.activity.ChemicalBalancerTutor.prototype.displayLevel = function () {
        var self = this, l = this.displayLayer, ctx = l.context, runner, clickable = l.events.clickable, cInner = this.controller, cOuter = cInner.controller;
        var bw = (this.w * 0.15);
        ctx.clearRect(0, this.tbh + 10, bw, bw);
        ctx.drawImage(OU.preLoadImg[6 + cOuter.sectionNum], 10, this.tbh + 10, bw, bw);
    };
    OU.activity.ChemicalBalancerTutor.prototype.gotoMenu = function () {
        this.controller.controller.setSection(0);
    };
    OU.activity.ChemicalBalancerTutor.prototype.previousSection = function () {
        this.controller.prevSection();
    };
    OU.activity.ChemicalBalancerTutor.prototype.nextSection = function () {
        this.controller.nextSection();
    };
    OU.activity.ChemicalBalancerTutor.prototype.displaySection = function () {

        // lets draw the controller section :)
        var self = this, l = this.displayLayer, ctx = l.context, runner, clickable = l.events.clickable;
        var bw = (this.w * 0.15), bwh = bw / 2;
        var cInner = this.controller, cOuter = cInner.controller
        ctx.clearRect(this.w - bw - 10, this.tbh + 10, bw, bw);
        var dt = (this.controller.sectionNum + 1) + ' / ' + this.controller.sections.length;
        var fontface = this.fontface;
        var sectionText = new OU.util.DynText({
            txt:dt,
            x:this.w - bw - 10,
            y:this.tbh + 10,
            w:bw,
            h:bwh,
            fontSize:bwh,
            context:ctx,
            colour:'#fff',
            align:'center',
            measureOnly:false,
            noBreak:true,
            fontFamily:fontface
        });
        if (this.controller.sectionNum===0) {
            ctx.save();
            ctx.globalAlpha = 0.2;
            ctx.drawImage(OU.preLoadImg[4], this.w - bw - 10, this.tbh + 10 + bwh, bwh, bwh);
            ctx.restore();
        }
        else {
            ctx.drawImage(OU.preLoadImg[3], this.w - bw - 10, this.tbh + 10 + bwh, bwh, bwh);
            clickable.push(new this.Marker(
            {

                x:this.w - bw - 10,
                y:this.tbh + 10 + bwh,
                w:bwh,
                h:bwh,
                context:ctx,
                isclicked:false,
                execute:function () {
                    self.previousSection();
                }
            }));
        }
        if (this.controller.sectionNum===this.controller.sections.length - 1) {
            ctx.drawImage(OU.preLoadImg[6], this.w - bw - 10 + bwh, this.tbh + 10 + bwh, bwh, bwh);
        }
        else {
            ctx.drawImage(OU.preLoadImg[5], this.w - bw - 10 + bwh, this.tbh + 10 + bwh, bwh, bwh);
            clickable.push(new this.Marker(
            {

                x:this.w - bw - 10 + bwh,
                y:this.tbh + 10 + bwh,
                w:bwh,
                h:bwh,
                context:ctx,
                isclicked:false,
                execute:function () {
                    self.nextSection();
                }
            }));
        }
        if (typeof OU.saveData.completedSections[cOuter.sectionNum - 1][cInner.sectionNum]=='object') {

            //  ctx.clearRect(this.w - bw - 10, this.tbh + 10, bw, bw);
            //   ctx.drawImage(OU.preLoadImg[5], this.w - bw - 10 + bwh, this.tbh + 10 + bwh, bwh, bwh);
            var sectionText = new OU.util.DynText({
                txt:'COMPLETED',
                x:this.w - bw - 10,
                y:this.tbh + 10 + bw,
                w:bw,
                h:bwh,
                fontSize:bwh,
                context:ctx,
                colour:'#fff',
                align:'center',
                measureOnly:false,
                noBreak:true,
                fontFamily:fontface
            });
            if (this.tutor==10) {
                this.tutorSteps[6].x = (this.w - bw - 10) - (this.w / 4) + (bw / 2) + 15;
                this.tutorSteps[6].y = (this.tbh + 10 + bw) + bwh;
                this.tutorSteps[6].w = (this.w / 4);
                this.tutorSteps[6].h = (this.w / 4) / 1.7;
                setTimeout(function () {
                    self.tutorStepDelta();
                }, 500);
            }
        }
    };
    /**
     * Displays the equation
     */
    OU.activity.ChemicalBalancerTutor.prototype.displayEquation = function () {
        var self = this, l = this.displayLayer, area = this.areas.equation, ctx = l.context, runner, clickable = l.events.clickable, gBack, i, fs;
        var rightText, goesTo, leftText, leftTest, rightTest, lX, bW, rX, cb, lttb, rttb, maxH;
        var tBlack = 'rgba(0,0,0,0.50)', fontface;
        clickable.length = 0;
        ctx.clearRect(0, 0, this.w, this.h);
        runner = this.currentChem;
        if (this.displayMode===1) {
            gBack = ctx.createLinearGradient(0, area.y, 0, area.y + area.h);
            gBack.addColorStop(0, 'rgba(96,108,136,1)');
            gBack.addColorStop(1, 'rgba(63,76,107,1)');
            ctx.fillStyle = gBack;
            ctx.fillRect(area.x, area.y, area.w, area.h);
        }
        cb = area.w / 2;
        bW = cb * 0.85;
        lX = (cb - bW) / 2;
        rX = lX + cb;
        fontface = this.fontface;
        if (OU.IS_IPHONE)
            maxH = area.h;
        else
            maxH = Math.floor(area.h / 3.3333);
        // shadow
        leftTest = new OU.util.DynText({
            txt:this.leftString,
            x:lX + 2,
            y:area.y + 2,
            w:bW,
            h:area.h,
            fontSize:maxH,
            context:ctx,
            colour:tBlack,
            align:'right',
            measureOnly:true,
            noBreak:true,
            fontFamily:fontface
        });
        rightTest = new OU.util.DynText({
            txt:this.rightString,
            x:lX + 2,
            y:area.y + 2,
            w:bW,
            h:area.h,
            fontSize:maxH,
            context:ctx,
            colour:tBlack,
            align:'right',
            measureOnly:true,
            noBreak:true,
            fontFamily:fontface
        });
        fs = leftTest.font.size > rightTest.font.size ? rightTest.font.size : leftTest.font.size;
        new OU.util.DynText({
            txt:this.leftString,
            x:lX + (2 * OU.dpr),
            y:area.y + (2 * OU.dpr),
            w:bW,
            h:area.h,
            fontSize:fs,
            context:ctx,
            colour:tBlack,
            align:'right',
            noBreak:true,
            fontFamily:fontface
        });
        new OU.util.DynText({
            txt:this.rightString,
            x:rX + (2 * OU.dpr),
            y:area.y + (2 * OU.dpr),
            w:bW,
            h:area.h,
            fontSize:fs,
            context:ctx,
            colour:tBlack,
            align:'left',
            noBreak:true,
            fontFamily:fontface
        });
        new OU.util.DynText({
            txt:this.arrows[1],
            x:lX + bW + (2 * OU.dpr),
            y:area.y + (2 * OU.dpr),
            w:rX - (lX + bW),
            h:area.h,
            fontSize:fs,
            context:ctx,
            colour:tBlack,
            align:'center',
            noBreak:true,
            fontFamily:fontface
        });
        leftText = new OU.util.DynText({
            txt:this.leftString,
            x:lX,
            y:area.y,
            w:bW,
            h:area.h,
            fontSize:fs,
            context:ctx,
            colour:'#ffffff',
            align:'right',
            noBreak:true,
            fontFamily:fontface
        });
        rightText = new OU.util.DynText({
            txt:this.rightString,
            x:rX,
            y:area.y,
            w:bW,
            h:area.h,
            fontSize:fs,
            context:ctx,
            colour:'#ffffff',
            align:'left',
            noBreak:true,
            fontFamily:fontface
        });
        goesTo = new OU.util.DynText({
            txt:this.arrows[1],
            x:lX + bW,
            y:area.y,
            w:rX - (lX + bW),
            h:area.h,
            fontSize:fs,
            context:ctx,
            colour:'#ffffff',
            align:'center',
            noBreak:true,
            fontFamily:fontface
        });
        this.font = leftText.font;
        this.fontSize = leftText.font.size;
        ctx.strokeStyle = '#ff0000';
        // fill the hitboxes..
        lttb = leftText.textBoxes;
        rttb = rightText.textBoxes;
        this.hitBoxes.length = 0;
        for (i = 0; i < lttb.length; i++)
            this.hitBoxes.push(lttb[i]);
        this.hitBoxes.push(goesTo.textBoxes[0]);
        for (i = 0; i < rttb.length; i++)
            this.hitBoxes.push(rttb[i]);
        for (i = this.hitBoxes.length - 1; i >= 0; i--) {
            if (runner[i]!==null) {
                clickable.push(new this.Marker(
                {

                    x:this.hitBoxes[i].x,
                    y:this.hitBoxes[i].y,
                    w:this.hitBoxes[i].w,
                    h:this.hitBoxes[i].h,
                    item:i,
                    context:ctx,
                    isclicked:false,
                    execute:function ( inval ) {
                        self.itemclicked(inval);
                    }
                }));
            }
        }
        // if (this.totalUse >= this.showHint) this.showHintButton();
        if (this.tutor==0) {
            // first step..
            var params = {
                x:this.hitBoxes[0].x,
                y:this.hitBoxes[0].y - (((this.w / 5) / 1.7) + 10),
                w:(this.w / 5),
                h:(this.w / 5) / 1.7,
                txt:this.tutorSteps[this.tutor].txt,
                k:3
            };
            this.tutorPopup(params);
        }
        if ((this.tutor==2 && this.balanceLeft[1][1]==2)) {
            // first step..
            this.killTutorPopup();
            var params = {
                x:this.hitBoxes[0].x,
                y:this.hitBoxes[0].y - (((this.w / 5) / 1.7) + 10),
                w:(this.w / 5),
                h:(this.w / 5) / 1.7,
                txt:this.tutorSteps[3].txt,
                k:3
            };
            this.tutorPopup(params);
            setTimeout(function () {
                self.tutorStepAlpha()();
            }, 3500);
        }
        else if ((this.tutor==1 ) || (this.tutor==2 && this.balanceLeft[1][1]!=2)) {
            // naughty lets go back to the beginning..
            this.tutor = 0;
            this.resize();
        }
    };
    /**
     * Displays the spinner control for the equation species clicked on
     * @param inval
     */
    OU.activity.ChemicalBalancerTutor.prototype.itemclicked = function ( inval ) {
        if (this.spinner!==undefined && this.spinner!==null) return null;
        var ctx = this.displayLayer.context, hbX, hbY, hbW, hbH, self = this, f = this.font;
        var area = this.areas.equation;
        // lets remove the click layer...
        self.itemClicked = inval;
        if (self.itemClicked==0 && self.tutor==0) {
            self.tutor = 1;
            self.killTutorPopup();
        }
        hbX = this.hitBoxes[inval].x;
        // hbY = this.hitBoxes[inval].y;
        hbW = this.hitBoxes[inval].w;
        //hbH = this.hitBoxes[inval].h;
        hbH = area.h * 0.85;
        // hbY = (this.areas.equation.y + (this.areas.equation.h / 2)) - (hbH );
        hbY = area.y + ((area.h - hbH) / 2);
        f.size = 40 * OU.dpr;
        var params = {
            x:hbX,
            //y:hbY - 50,
            y:hbY,
            w:hbW,
            h:hbH,
            max:this.maxVal,
            //maxH:this.areas.equation.h - 5,
            maxH:hbH,
            context:ctx,
            layer:this.displayLayer,
            font:f,
            currentVal:this.currentChem[inval].m,
            onReturn:function ( inval ) {
                self.spinnerReturn(inval);
            },
            container:this.displayLayer.container
        };
        if (self.itemClicked==0 && self.tutor==1) {
            params.tutor = 1;
            params.parent = self;
        }
        self.spinner = new OU.util.Spinner(params);
        if (this.tutor==1) {
            // first step..
            self.tutorSteps[2].x = hbX - 15;
            self.tutorSteps[2].y = hbY - (((self.w / 5) / 1.7) + 10);
            self.tutorSteps[2].w = (self.w / 5);
            self.tutorSteps[2].h = (self.w / 5) / 1.7;
            self.tutorSteps[2].k = 3;
            var params = {
                x:self.tutorSteps[2].x,
                y:self.tutorSteps[2].y,
                w:self.tutorSteps[2].w,
                h:self.tutorSteps[2].h,
                txt:self.tutorSteps[this.tutor].txt,
                k:3
            };
            this.tutorPopup(params);
        }
    };
    /**
     * Builds the running total array.
     * @param inval
     * @param r
     */
    OU.activity.ChemicalBalancerTutor.prototype.runnerBuilder = function ( inval, r ) {
        for (var i = 0; i < inval.length; i++) {
            if (r.length > 0) r.push(null);
            r.push(inval[i]);
        }
        return r;
    };
    /**
     * Validates elements against the elements array to ensure they are correct Chemicals
     * @param  inval
     */
    OU.activity.ChemicalBalancerTutor.prototype.validateElement = function ( inval ) {
        for (var i = 0, found = false; i < this.elements.length; i++)
            if (this.elements[i][1]===inval)
                return {
                    el:this.elements[i][2],
                    l:i
                };
        return found;
    };
    /**
     * Parses each cluster of a side
     * @param inval
     */
    OU.activity.ChemicalBalancerTutor.prototype.parseClusters = function ( inval ) {
        var i, ws, r, c = [];
        for (i = 0; i < inval.length; i++) {
            ws = inval[i];
            r = this.clusterBreaker(ws);
            r.m = 1;
            c.push(r);
        }
        return c;
    };
    /**
     * Breaks up each cluster into it's component parts
     * @param  inval
     */
    OU.activity.ChemicalBalancerTutor.prototype.clusterBreaker = function ( inval ) {
        var self = this, cluster = [], ts, ws, p, ms, nc, sub, sup, bracket, chem, v, bracketList = [], _char, _char1, _char1b;
        p = 0;
        ms = "";
        sub = "";
        sup = "";
        bracket = false;
        ws = inval;
        while (p < ws.length) {
            _char = ws.charCodeAt(p);
            if (_char >= 65 && _char <= 90) {
                ms += ws[p];
                _char1 = ws.charCodeAt(p + 1);
                if (_char1 >= 97 && _char1 <= 122)
                    ms += ws[p + 1];
                if (ms.length > 1) p++;
                _char1 = ws.charCodeAt(p + 1);
                if (_char1 >= 48 && _char1 <= 57) {
                    nc = 1;
                    //_char1b = ws.charCodeAt(p + nc);
                    while (_char1b = ws.charCodeAt(p + nc), _char1b >= 48 && _char1b <= 57) {
                        sub += ws[p + nc];
                        nc++;
                    // _char1b = ws.charCodeAt(p + nc);
                    }
                    // we have a number..
                    p += (nc - 1);
                }
                if (ws.charCodeAt(p + 1)===123) {
                    // we have a curly bracket
                    p++;
                    nc = 1;
                    ts = '';
                    self.charged = true;
                    while (ws.charCodeAt(p + nc)!==125) {
                        ts += ws[p + nc];
                        nc++;
                    }
                    if (ts.length > 0) sup = ts;
                    if (sup==="-") sup = "-1";
                    p += nc;
                }
                v = self.validateElement(ms);
                sup = sup.length > 0 ? sup : 0;
                sub = sub.length > 0 ? sub : 1;
                chem = {
                    el:ms,
                    sub:sub,
                    sup:sup,
                    plus:false,
                    bracket:bracket,
                    n:v
                };
            }
            else {
                _char = ws.charCodeAt(p);
                if (_char===40 || _char===91) {
                    // we have ( or [
                    bracketList.push(_char===40 ? 41 : 93);
                    bracket = _char;
                    nc = 0;
                    p++;
                    while (bracketList.length!==0) {
                        _char1 = ws.charCodeAt(p + nc);
                        if (_char1===bracketList[bracketList.length - 1]) bracketList.pop();
                        if (_char1===40 || _char1===91) {
                            bracketList.push(_char1===40 ? 41 : 93);
                        }
                        if (bracketList.length > 0) {
                            ms += ws[p + nc];
                            nc++;
                        }
                    }
                    chem = self.clusterBreaker(ms);
                    chem.newBracket = bracket;
                    p += (nc + 1);
                    _char = ws.charCodeAt(p);
                    if (_char >= 48 && _char <= 57) {
                        nc = 0;
                        while (_char1 = ws.charCodeAt(p + nc), _char1 >= 48 && _char1 <= 57) {
                            sub += ws[p + nc];
                            nc++;
                        }
                        // we have a number..
                        chem.sub = sub;
                    // p += (nc);
                    }
                }
                else if (ws.charCodeAt(p)===123) {
                    // we have a curly bracket
                    p++;
                    nc = 0;
                    ts = '';
                    self.charged = true;
                    while (ws.charCodeAt(p + nc)!==125) {
                        ts += ws[p + nc];
                        nc++;
                    }
                    if (ts.length > 0) sup = ts;
                    p += nc;
                    if (ts==="-") ts = "e";
                    chem = {
                        el:ts,
                        sub:"",
                        sup:"",
                        plus:false,
                        bracket:bracket,
                        n:"electron"
                    };
                }
            }
            cluster.push(chem);
            p++;
            ms = "";
            sub = "";
            sup = "";
            bracket = false;
        }
        return cluster;
    };
    /**
     * Weighs each side to calculate how many chemicals it has
     * @param side
     * @param balance
     * @param multi
     */
    OU.activity.ChemicalBalancerTutor.prototype.weighSide = function ( side, balance, multi ) {
        var cluster, m, curW, j, c, v, tc, bm, masterM = multi || 1, i, cM = 0;
        var inArray = function ( a, e ) {
            for (var i = 0, l = a.length; i < l; i++) if (a[ i ][0]===e) return i;
            return -1;
        };
        for (i = 0; i < side.length; i++) {
            // clusters...
            cluster = side[i];
            bm = cluster.bm || 1;
            m = (parseInt(cluster.m, 10) || 1) * masterM * bm;
            // find charge
            for (j = 0, cM = 0; j < cluster.length; j++) cM = cluster[j].sup || cM;
            // balance charge.
            v = inArray(balance, 'chg');
            if (v==-1) balance.push(['chg', m * cM]);
            else balance[v][1] += (m * cM);
            for (j = 0; j < cluster.length; j++) {
                c = cluster[j];
                if (typeOf(c)=="object") {
                    curW = (m ) * (c.sub || 1) > 0 ? (m ) * (c.sub || 1) : 1;
                    v = inArray(balance, c.el);
                    if (v==-1) balance.push([c.el, curW]);
                    else balance[v][1] += curW;
                }
                else {
                    tc = [];
                    tc[0] = c;
                    //tc[0].m = c.sub;
                    tc[0].m = 1;
                    tc[0].bm = c.sub;
                    this.weighSide(tc, balance, m);
                }
            }
        }
    };
    /**
     * Balances the sides
     */
    OU.activity.ChemicalBalancerTutor.prototype.balanceBuilder = function () {
        var totalize = function ( a ) {
            for (var t = 0, i = 0, l = a.length; i < l; i++)
                t += a[ i ][1];
            return t;
        };
        this.balanceLeft = [];
        this.balanceRight = [];
        this.weighSide(this.leftCluster, this.balanceLeft);
        this.weighSide(this.rightCluster, this.balanceRight);
        this.balanceLeft.total = totalize(this.balanceLeft);
        this.balanceRight.total = totalize(this.balanceRight);
    };
    /**
     * Rebuilds the display string after it has been broken down
     * @param inval
     * @param depth
     */
    OU.activity.ChemicalBalancerTutor.prototype.rebuilder = function ( inval, depth ) {
        var self = this, i, h = "", cur, d = depth || 0;
        /**
         * Selects and inserts the correct bracket
         * @param inval
         * @param mode
         */
        var makeBracket = function ( inval, mode ) {
            if (inval==40 && mode==1) return "(";
            if (inval==40 && mode==2) return ")";
            if (inval==91 && mode==1) return "[";
            if (inval==91 && mode==2) return "]";
        };
        /**
         * Displays a superscript element with the charge symbol the correct way round
         * @param inval
         */
        var makeSuperScript = function ( inval ) {
            var ws = "<sup>", v = parseInt(inval, 10), sign;
            sign = v && v / Math.abs(v)==-1 ? "-" : "+";
            v = Math.abs(v);
            ws += (v==1 ? sign : v + sign) + "</sup>";
            return ws;
        };
        // this should take an array
        if (d > 5) throw "loopy";
        for (i = 0; i < inval.length; i++) {
            cur = inval[i];
            if (typeOf(cur)=="array") {
                if (d==0 && h.length > 0) h += ' + ';
                if (cur.m!=undefined && cur.m > 1) h += cur.m;
                h += self.rebuilder(cur, d + 1);
                if (cur.sub!=undefined && cur.sub > 1) h += "<sub>" + cur.sub + "</sub>";
            }
            if (typeOf(cur)=="object") {
                h += cur.el;
                if (cur.sub!="" && cur.sub > 1) h += "<sub>" + cur.sub + "</sub>";
                if (cur.sup!="") h += makeSuperScript(cur.sup);
            }
        }
        if (depth > 1) h = makeBracket(inval.newBracket, 1) + h + makeBracket(inval.newBracket, 2);
        return h;
    };
    OU.activity.ChemicalBalancerTutor.prototype.checkSaveData = function () {
        if (OU.saveData!=undefined && OU.saveData!=null) return null;
        var saveData = OU.LocalStorage.load('ou.chembal.app') || null;
        if (saveData==undefined) {
            this.buildSaveData();
        }
        else {
            OU.saveData = JSON.parse(saveData);
        }
        OU.util.exportSaveData = function () {

            // quick rebuild of the completed levels..
            var i, c, j, u = 1;
            for (i = 0; i < OU.saveData.completedLevels.length; i++) {
                c = 0;
                for (j = 0; j < OU.saveData.completedSections[i].length; j++) {
                    if (typeof OU.saveData.completedSections[i][j]=='object') c++;
                }
                OU.saveData.completedLevels[i] = c;
                if (OU.saveData.completedLevels[i]==OU.saveData.completedSections[i].length) u++;
            }
            OU.saveData.unlocked = u;
            OU.LocalStorage.save('ou.chembal.app', JSON.stringify(OU.saveData));
        };
        OU.util.loadSaveData = function () {
            var saveData = OU.LocalStorage.load('ou.chembal.app') || null;
            if (saveData!==undefined) OU.saveData = JSON.parse(saveData);
        };
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.ChemicalBalancerTutor, OU.util.Activity);


