/**
 * @fileOverview Pairs - Allows user to match together pairs of images, then given feedback and further info on content
 *
 * @param {JSON array} data Data set
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.PopUpInfo');
OU.require('OU.util.Instruction');
OU.require('OU.util.Draggable');
OU.require('OU.util.Layer');
OU.require('OU.util.ImageLoader');
/**
 * @class Pairs activity, requires user to match up pairs of items using a dragging technique.
 * Items can either be graphics or text based. Further information can be provided once all matches are made.
 * 
 * @extends OU.util.Activity
 * @param {object} data - data.js content
 * @param {string} instance - unique instance name
 * @param {OU.util.Controller} controller - (optional) reference of parent controller
 */
OU.activity.Pairs = function(data, instance, controller) {
    OU.activity.Pairs.prototype.canvasView = function() {
        var self = this;
        this.bgLayer = new OU.util.Layer({
            container: this
        });
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        this.imageLayer = new OU.util.Layer({
            container: this,
            hasEvents: true
        });
        this.boundary = {
            x: 5,
            y: 5,
            w: this.w - 10,
            h: this.h - OU.controlHeight * 2 - 5
        };
        this.fbWidth = this.w;
        this.controlLayer = new OU.util.Layer({
            y: this.h - OU.controlHeight * 2,
            h: OU.controlHeight * 2,
            container: this,
            hasEvents: true
        });
        this.imageLoader = new OU.util.ImageLoader({
            container: this,
            data: this.data,
            onLoad: function() {
                self.start();
            }
        });
    };
    /**
     * Resizes the activity.
     */
    OU.activity.Pairs.prototype.resize = function() {
        OU.activity.Pairs.superClass_.resize.call(this); // call the parent class resize
        this.bgLayer.resize();
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        this.imageLayer.resize();
        this.boundary = {
            x: 5,
            y: 5,
            w: this.w - 10,
            h: this.h - OU.controlHeight * 2 - 5
        };
        this.fbWidth = this.w;
        this.controlLayer.resize({
            y: this.h - OU.controlHeight * 2,
            h: OU.controlHeight * 2
        });
        this.doneLive = false;
        this.doRender = true;
    };
    /**
     * Starts the activity once all images are loaded.
     * @private
     */
    OU.activity.Pairs.prototype.start = function() {
        var i, item;
        // resize items to images if aplicable
        this.initItems();
        for (i = this.items.length; i--; ) {
            item = this.items[i];
            if (item.image) {
                item.draggable.w = item.w = item.image.width;
                item.draggable.h = item.h = item.image.height;
            }
        }
        if (this.data.instructions !== undefined) {
            new OU.util.Instruction({
                container: this,
                message: this.data.instructions
            });
        }
        // start render loop
        this.doRender = true;
        this.renderLoop();
    };
    /**
     * Initialises the items.
     * @private
     */
    OU.activity.Pairs.prototype.initItems = function() {
        var i, cX = this.w / 2 - 25, cY = this.h / 2 - 25, pair,
        ctx = this.imageLayer.context, lineWidth,
        numPairs = this.data.pairs.length,
        ev = this.imageLayer.events,
        dTheta = 720 / numPairs, theta = 0, v, d2r = Math.PI / 180;
        this.items = [];
        this.energy = 1;
        ctx.font = '14px ' + OU.theme.font;
        for (i = 0; i < numPairs; i++) {
            pair = this.data.pairs[ i ];
            theta = theta + dTheta;
            v = Math.random() * 10 + 2;
            lineWidth = ctx.measureText(pair.left.name, 50, 50).width;
            this.items.push(new this.Item({
                pairs: this,
                id: i * 2,
                "side": "left",
                "events": ev,
                "data": pair.left,
                "leftPaired": pair.leftPaired,
                "pair": pair,
                "x": cX,
                "y": cY,
                "w": lineWidth + 40,
                "h": 50,
                "v": {
                    x: Math.cos(theta * d2r) * v,
                    y: Math.sin(theta * d2r) * v
                }
            }));
            lineWidth = ctx.measureText(pair.right.name, 50, 50).width;
            this.items.push(new this.Item({
                pairs: this,
                id: i * 2 + 1,
                "side": "right",
                "events": ev,
                "data": pair.right,
                "rightPaired": pair.rightPaired,
                "pair": pair,
                "x": cX,
                "y": cY,
                "w": lineWidth + 40,
                "h": 50,
                "v": {
                    x: Math.cos(theta * d2r) * v,
                    y: Math.sin(theta * d2r) * v
                }
            }));
        }
    };
    /**
     * Removes the activity, stops animation.
     */
    OU.activity.Pairs.prototype.remove = function() {
        OU.activity.Pairs.superClass_.remove.call(this); // call the parent class resize
        this.doRender = false;
        this.kill = true;
    };
    /**
     * Performs the render loop to animate the items
     * @private
     */
    OU.activity.Pairs.prototype.renderLoop = function() {
        var self = this;
        if (!this.kill) {
            if (this.doRender)
                this.render();
            setTimeout(function() {
                self.renderLoop();
            }, 40);
        }
    };
    /**
     * Renders all items
     * @private
     */
    OU.activity.Pairs.prototype.render = function() {
        var i, item, ctx, numItems = this.items.length, inDrag, unmatched = 0;
        this.imageLayer.clear();
        ctx = this.imageLayer.context;
        if (!ctx)
            return;
        ctx.fillStyle = '#ccc';
        ctx.textAlign = 'center';
        this.collisions = 0;
        for (i = numItems; i--; ) {
            item = this.items[i];
            if (item.draggable.dragId === 0) { // item is not being dragged, so move according to motion
                item.motion();
            }
            if (item.lockleft < 0 && item.lockright < 0)
                unmatched++;
        }
        this.energy = this.collisions / numItems;
        inDrag = false;
        for (i = numItems; i--; ) {
            item = this.items[i];
            if (item.draggable.dragId) // if item in drag, render it last
                inDrag = item;
            else
                this.items[i].render();
        }
        if (inDrag)
            inDrag.render();
        if (unmatched === 0) {
            if (this.data.dontAllowBadPairs) {
                //   this.infoStage();
                this.resetButton();
            }
            else {
                this.doneButton();
            }

        }
        else if (this.doneLive) {
            this.doneLive = false;
            this.controlLayer.events.clickable.removeType('doneButton');
            this.controlLayer.context.clearRect(0, 0, this.fbWidth, this.h * .1);
        }
    };
    /**
     * Break apart any incorrectly matched pairs.
     * @private
     */
    OU.activity.Pairs.prototype.breakBad = function() {
        var i, a, b;
        for (i = this.items.length; i--; ) {
            a = this.items[i];
            if (a.lockright >= 0) {
                b = this.items[a.lockright];
                if (a.pair !== b.pair)
                    a.unlock();
            }
        }
    };
    /**
     * Show feedback if allowing bad matches
     * @private
     */
    OU.activity.Pairs.prototype.feedback = function() {
        var i, good = 0, bad = 0, a, b, bH = OU.controlHeight, self = this;
        for (i = this.items.length; i--; ) {
            a = this.items[i];
            if (a.lockright >= 0) {
                b = this.items[a.lockright];
                if (a.pair === b.pair)
                    good++;
                else
                    bad++;
            }
        }
        this.controlLayer.events.clickable.length = 0;
        this.controlLayer.clear();
        if (bad > 0) {
            new OU.util.ControlButton({
                layer: this.controlLayer,
                txt: "You have " + bad + " bad matches... break them up",
                x: this.fbWidth / 4,
                y: 0,
                w: this.fbWidth / 2,
                h: bH,
                onClick: function() {
                    self.breakBad();
                }
            });
        }
        else {
            new OU.util.DynText({
                txt: "Correct... you can now click the pairs to access more information.",
                x: 0,
                y: 0,
                w: this.fbWidth - bH * 3,
                h: bH,
                context: this.controlLayer.context
            });
            if (this.data.loadActivityOnComplete) {
                new OU.util.ControlButton({
                    layer: this.controlLayer,
                    txt: "Continue",
                    x: this.fbWidth - bH * 3,
                    y: 0,
                    w: bH * 2,
                    h: bH,
                    onClick: function() {
                        self.controller.overrideSection(self.data.loadActivityOnComplete);
                    }
                });
            }
            else {
                new OU.util.ControlButton({
                    layer: this.controlLayer,
                    txt: "Restart",
                    x: this.fbWidth - bH * 3,
                    y: 0,
                    w: bH * 2,
                    h: bH,
                    onClick: function() {
                        self.init();
                    }
                });
            }
            this.infoStage();
        }
    };
    /**
     * After completion, disable drag for all items and switch on isHit to load further info instead
     * @private
     */
    OU.activity.Pairs.prototype.infoStage = function() {
        var i, item, clickable = this.imageLayer.events.clickable;
        for (i = this.items.length; i--; ) {
            item = this.items[i];
            item.draggable.disabled = true;
            clickable.push(item);
        }
    };
    /**
     * Render a "reset" button
     * @private
     */
    OU.activity.Pairs.prototype.resetButton = function() {
        var bH = OU.controlHeight, self = this;      
        this.controlLayer.events.clickable.length = 0;
        new OU.util.ControlButton({
            layer: this.controlLayer,
            txt: "Reset",
            x: (this.fbWidth - bH * 3)/2,
            y: 0,
            w: bH * 3,
            h: bH,
            onClick: function() {
                self.init();
            }
        });
    };
    /**
     * Render a "done" button
     * @private
     */
    OU.activity.Pairs.prototype.doneButton = function() {
        var bH = OU.controlHeight, self = this;
        if (this.doneLive)
            return;
        this.doneLive = true;
        this.controlLayer.events.clickable.length = 0;
        new OU.util.ControlButton({
            layer: this.controlLayer,
            txt: "I'm done... how did I do?",
            x: this.fbWidth / 4,
            y: 0,
            w: this.fbWidth / 2,
            h: bH,
            onClick: function() {
                self.feedback();
            }
        });
    };
    /**
     * @class Item class handles the items' behaviour
     * @private
     */
    OU.activity.Pairs.prototype.Item = function(params) {
        this.pairs = params.pairs;
        this.events = params.events;
        this.id = params.id;
        this.side = params.side;
        this.image = params.data.image;
        if (params.leftPaired)
            this.imagePaired = params.leftPaired.image;
        if (params.rightPaired)
            this.imagePaired = params.rightPaired.image;
        this.pair = params.pair;
        this.data = params.data;
        this.bgcolor = params.data.bgcolor || '#fff',
        this.x = params.x;
        this.y = params.y;
        this.h = params.h;
        this.w = params.w;
        this.v = params.v;
        this.startX = this.x;
        this.startY = this.y;
        this.lockleft = -1;
        this.lockright = -1;
        /**
         * Renders the item depending on whether it's an icon or text
         * @private
         */
        OU.activity.Pairs.prototype.Item.prototype.render = function() {
            var ctx = this.pairs.imageLayer.context, sF = 1, hasImage, b;
            if (this.lockleft === -1 & this.lockright === -1) { // not locked
                if (this.image) {
                    ctx.drawImage(this.image, this.x, this.y);
                }
                else {
                    if (this.draggable.dragId !== 0) { // in motion render state
                        sF = 1.4;
                    }
                    ctx.roundRect(this.x - (this.w * (sF - 1) / 2), this.y - (this.h * (sF - 1) / 2), this.w * sF, this.h * sF, 5);
                    ctx.strokeStyle = '#444';
                    if (this.side === 'left')
                        ctx.fillStyle = this.bgcol || this.pairs.data.leftBgcol || '#fff';
                    else
                        ctx.fillStyle = this.bgcol || this.pairs.data.rightBgcol || '#fff';
                    ctx.fill();
                    ctx.stroke();
                    if (this.side === 'left')
                        ctx.fillStyle = this.col || this.pairs.data.leftCol || '#444';
                    else
                        ctx.fillStyle = this.col || this.pairs.data.rightCol || '#444';
                    ctx.fillText(this.data.name, this.x + (this.w) / 2, this.y + (this.h) / 2);
                }
                return;
            }

            if (this.lockleft >= 0) { // right hand item is rendered by the left, so return
                return;
            }
            // left hand item of a locked pair, so render both
            b = this.pairs.items[this.lockright];
            hasImage = false;
            if (this.image || b.image) {
                hasImage = true;
            }
            if (hasImage) {
                if (this.image) {
                    if (this.imagePaired)
                        ctx.drawImage(this.imagePaired, this.x, this.y);
                    else
                        ctx.drawImage(this.image, this.x, this.y);
                }
                else {
                    ctx.roundRect(this.x, this.y, this.w, this.h, 5);
                    ctx.strokeStyle = '#444';
                    ctx.fillStyle = this.pairs.data.joinedBgcol || this.bgcol || this.pairs.data.leftBgcol || '#fff';
                    ctx.fill();
                    ctx.stroke();
                    ctx.fillStyle = this.pairs.data.joinedCol || this.col || this.pairs.data.leftcol || '#444';
                    ctx.fillText(this.data.name, this.x + this.w / 2, this.y + this.h / 2);
                }
                if (b.image) {
                    if (b.imagePaired)
                        ctx.drawImage(b.imagePaired, this.x + this.w, this.y);
                    else
                        ctx.drawImage(b.image, this.x + this.w, this.y);                    
                }
                else {
                    ctx.roundRect(this.x + this.w, this.y, b.w, b.h, 5);
                    ctx.strokeStyle = '#444';
                    ctx.fillStyle = this.pairs.data.joinedBgcol || this.bgcol || this.pairs.data.leftBgcol || '#fff';
                    ctx.fill();
                    ctx.stroke();
                    ctx.fillStyle = this.pairs.data.joinedCol || this.col || this.pairs.data.leftcol || '#444';
                    ctx.fillText(b.data.name, this.x + this.w + b.w / 2, this.y + b.h / 2);
                }
            }
            else {
                ctx.roundRect(this.x, this.y, this.w + b.w, this.h, 5);
                ctx.strokeStyle = '#444';
                ctx.fillStyle = this.pairs.data.joinedBgcol || this.bgcol || this.pairs.data.leftBgcol || '#fff';
                ctx.fill();
                ctx.stroke();
                ctx.fillStyle = this.pairs.data.joinedCol || this.col || this.pairs.data.leftcol || '#444';
                ctx.fillText(this.data.name + ' ' + b.data.name, this.x + (this.w + b.w) / 2, this.y + (this.h * sF) / 2);
            }
        };
        /**
         * Recieves Mouse & Touch events and determines if this item has been "hit".
         * If it has then we load up the further info for this pair in a pop-up.
         * Note: isHit does not handle the dragging of an item, that is done usine a OU.util.Draggable object.
         * @private
         */
        OU.activity.Pairs.prototype.Item.prototype.isHit = function(x, y, m) {
            var ev = this.pairs.imageLayer.events;
            if (!m)
                return;
            if (x < this.x || x > this.x + this.w)
                return;
            if (y < this.y || y > this.y + this.h)
                return;
            ev.pressed = false;
            ev.touched = false;
            new OU.util.PopUpInfo({
                container: this.pairs,
                txt: this.pair.info
            });
        };
        /**
         * Called when this item starts to be dragged.
         * Unlocks the item if it is already paired
         * @private
         */
        OU.activity.Pairs.prototype.Item.prototype.startDrag = function(p) {
            //sort the items so that the latest dragged is at the end of the queue, therefore render last next time (on top)
            p.me.unlock(); // unlock if paired
        };
        /**
         * Called when Item is dropped.
         * Checks to see if it should be paired.
         * @private
         */
        OU.activity.Pairs.prototype.Item.prototype.endDrag = function(p) {
            p.me.lockable(); // check for new pairing
            p.me.pairs.energy = 1;
        };
        /**
         * Called mid drag to move the item.
         * @private
         */
        OU.activity.Pairs.prototype.Item.prototype.newPos = function(p) {
            p.me.x = p.x;
            p.me.y = p.y;
            p.me.pairs.doRender = true;
        };
        /**
         * Unlocks (unpairs) the item
         * @private
         */
        OU.activity.Pairs.prototype.Item.prototype.unlock = function() {
            if (this.lockright > -1)
                this.pairs.items[this.lockright].lockleft = -1;
            if (this.lockleft > -1)
                this.pairs.items[this.lockleft].lockright = -1;
            this.lockleft = -1;
            this.lockright = -1;
        };
        /**
         * Determines if an item can be paired with another.
         * IE. if it collides with another unmatched item.
         * If so, then pair them up.
         * @private
         */
        OU.activity.Pairs.prototype.Item.prototype.lockable = function() {
            var i, b;
            for (i = this.pairs.items.length; i--; ) {
                b = this.pairs.items[i];
                if (this !== b && this.collide(b) && (!this.pairs.data.dontAllowBadPairs || this.pair === b.pair)) {
                    if (b.lockright < 0 && b.lockleft < 0) {
                        if (this.pairs.data.forceLeftRight === true) {
                            if (this.side === 'left' && b.side === 'right') {
                                this.lockright = b.id;
                                b.lockleft = this.id;
                            }
                            else if (this.side === 'right' && b.side === 'left') {
                                this.lockleft = b.id;
                                b.lockright = this.id;
                            }
                        }
                        else {
                            this.lockleft = b.id;
                            b.lockright = this.id;
                        }
                    }
                    i = 0; // found one to lock, so end loop
                }
            }
        };
        /**
         * Returns the dimensions of a item.
         * If the item is the left hand side of a pair, then it returns the dimensions of both items, as a pair.
         * @private
         */
        OU.activity.Pairs.prototype.Item.prototype.dims = function() {
            if (this.lockright >= 0) {
                return {
                    w: this.w + this.pairs.items[this.lockright].w,
                    h: this.h + this.pairs.items[this.lockright].h
                };
            }
            else {
                return {
                    w: this.w,
                    h: this.h
                };
            }
        };
        /**
         * Determines if this item collides with another
         * @param {OU.activity.Pairs.prototype.Item} b - the other item
         * @private
         */
        OU.activity.Pairs.prototype.Item.prototype.collide = function(b) {
            var spacer = 1.1,
            aD = this.dims();
            if (b.draggable.dragId || b.lockleft >= 0) // don't collide with items being dragged or left locked
                return false;
            if (this.x > b.x + b.w * spacer)
                return false;
            if (this.x + aD.w * spacer < b.x)
                return false;
            if (this.y > b.y + b.h * spacer)
                return false;
            if (this.y + aD.h * spacer < b.y)
                return false;
            return true;
        };
        /**
         * Calculates the motion of this item.
         * Items are repulsed by each other, so that they spread out in the space given.
         * @private
         */
        OU.activity.Pairs.prototype.Item.prototype.motion = function() {
            var i, nX, nY, rX, rY, b, dX, dY, buddy, dims,
            cX = this.pairs.w / 2 - 25, cY = this.pairs.h * .45 - 25;
            if (this.lockleft < 0) {
                // reduce momentum
                dX = this.v.x * .9;
                dY = this.v.y * .9;
                dX = dX > 10 ? 10 : dX;
                dY = dY > 10 ? 10 : dY;
                dX = dX < -10 ? -10 : dX;
                dY = dY < -10 ? -10 : dY;
                // Collide repulsion
                for (i = this.pairs.items.length; i--; ) {
                    b = this.pairs.items[i];
                    if (this !== b && this.collide(b)) {
                        rX = this.x - b.x > 0 ? ((this.w + b.w) / 2 - (this.x - b.x)) / 50 : (-((this.w + b.w) / 2) + (this.x - b.x)) / 50;
                        rY = this.y - b.y > 0 ? ((this.h + b.h) / 2 - (this.y - b.y)) / 50 : (-((this.h + b.h) / 2) + (this.y - b.y)) / 50;
                        dX = dX + rX * (1 - this.pairs.energy / 2);
                        dY = dY + rY * (1 - this.pairs.energy / 2);
                        b.addDelta(-rX * (1 - this.pairs.energy / 3), -rY * (1 - this.pairs.energy / 2)); // bump the colliding item the other way
                        i = 0; // only collide with 1 item per cycle => end loop
                        this.pairs.collisions++;
                    }
                }
                if (this.pairs.energy < 0.1) { // if overall energy>0 then gravitate back to center
                    dX = dX + this.pairs.energy * (cX - this.x) / 500;
                    dY = dY + this.pairs.energy * (cY - this.y) / 500;
                }
                // Move item according to Motion with boundary
                nX = this.x + dX;
                nY = this.y + dY;
                b = this.pairs.boundary;
                dims = this.dims();
                if (nX < b.x)
                    nX = b.x;
                if (nX > b.x + b.w - dims.w)
                    nX = b.x + b.w - dims.w;
                if (nY < b.y)
                    nY = b.y;
                if (nY > b.y + b.h - dims.h)
                    nY = b.y + b.h - dims.h;
                this.move(nX, nY);
                this.v.x = dX; // store current movement for next cycle
                this.v.y = dY;
                if (this.lockright >= 0) {
                    // item is paired, so move locked item to right of it
                    buddy = this.pairs.items[this.lockright];
                    buddy.move(this.x + this.w, this.y);
                }
            }
        };
        /**
         * Move the item by dX,dY
         * @param {double} dx - amount to move x position
         * @param {double} dy - amount to move y position
         * @private
         */
        OU.activity.Pairs.prototype.Item.prototype.addDelta = function(dx, dy) {
            this.v.x = this.v.x + dx;
            this.v.y = this.v.y + dy;
        };
        /**
         * Move the item to a new X,Y position
         * @param {double} x - new x position
         * @param {double} y - new y position
         * @private
         */
        OU.activity.Pairs.prototype.Item.prototype.move = function(x, y) {
            this.x = x;
            this.y = y;
            this.draggable.x = x;
            this.draggable.y = y;
        };
        // Initialise a Draggable object and associate it with this Item
        this.draggable = new OU.util.Draggable({
            "me": this,
            "events": this.pairs.imageLayer.events,
            "x": this.x,
            "y": this.y,
            "h": this.h,
            "w": this.w,
            "onStart": this.startDrag,
            "onEnd": this.endDrag,
            "onMove": this.newPos
        });
        this.events.clickable.push(this.draggable);
    };
    /**
     * Renders an accessible version of the activity.
     */
    OU.activity.Pairs.prototype.renderAccessible = function() {
        var numPairs = this.data.pairs.length, pair, i, name, h;
        clearInterval(this.renderCycle);
        h = '<div id="accessibleView">';
        h += '<h1>' + this.data.title + '</h1>';
        h += '<p>' + this.data.description + '</p>';
        h += '<p>This matching activity contains the following pairs:</p>';
        for (i = 0; i < numPairs; i++) {
            pair = this.data.pairs[ i ];
            name = pair.left.name ? pair.left.name + ' : ' : '' + pair.right.name ? pair.right.name : '';
            if (name !== undefined)
                h += '<h2>' + name + '</h2>';
            if (pair.info !== undefined)
                h += '<p>' + pair.info + '</p>';
        }
        h += '</div>';
        document.body.innerHTML = '';
        var accessible = document.createElement('div');
        accessible.innerHTML = h;
        document.body.appendChild(accessible);
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.Pairs, OU.util.Activity);
