/**
 * @fileOverview Particles - S104 Chemical Equilibrium Activity
 *
 * @param {JSON array} data Data set
 *
 * @author Kevin Quick & Jon Linney
 * @author Re-factored by Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.PopUpInfo');
OU.require('OU.util.NavButtons');
OU.require('OU.util.Layer');
OU.require('OU.util.ImageLoader');
/** 
 * @class
 * @extends OU.util.Activity
 */
OU.activity.Particles = function(data,instance,controller) {

    var particles = this,butWidth, ballRad, butGap, butHeight, ballX, ballY;

    OU.activity.Particles.prototype.canvasView = function() {
        var self=this;

        this.config = {
            vAreaW:5000,
            vScale:1,
            vParticleRadius:200,
            speedFactor:1*OU.LocalStorage.load('ou.particles.speedFactor') || 1,
            totalParticles:0,
            maxParticles:this.data.maxParticles,
            // equilibrium maximum and minimum ball count
            eqMin:(16/4)-1,
            eqMax:(16/4)+1,
            actualBallsR1:0,
            actualBallsR2:0,
            actualBallsP1:0,
            actualBallsP2:0,
            step:-1,
            stepCompleted:0,
            stopstartCount:0,
            imagesLoaded:0
        };

        this.bgLayer = new OU.util.Layer({
            container:this
        });
        this.bgLayer.context.gradRect(); // draw background on backdrop layer

        this.imageLayer = new OU.util.Layer({
            container:this
        });

        this.controlLayer = new OU.util.Layer({
            container:this,
            hasEvents:true
        });

        this.boundary = {
            x:this.w*.05,
            y:this.h*.05,
            w:this.w*.9,
            h:this.h*.9,
            a:0,
            aParticleRadius:0
        };
        
        this.boundary.a = this.boundary.w*.7<this.boundary.h?this.boundary.w*.7:this.boundary.h;
        
        this.config.vScale= this.boundary.a/this.config.vAreaW;
        this.boundary.aParticleRadius=this.config.vParticleRadius*this.config.vScale;

        this.fbWidth = this.w;

        // define dimentions depending on window size
        butWidth = 0.429*this.boundary.a*.7;
        ballRad=0.429*this.boundary.a*.1;
        butGap = 0.429*this.boundary.a*.05;
        butHeight = ballRad*2;
        this.boundary.x=this.w*.05+(this.boundary.w-(this.boundary.a+2*butGap+2*ballRad+butWidth))*.5;
        //var ballX=this.boundary.a+this.boundary.x+butGap;
        ballX=1.429*this.boundary.a-2*ballRad+this.boundary.x;
        ballY =(this.boundary.y+this.boundary.a)-ballRad;

        this.butData = {
            butWidth:butWidth,
            butHeight:butHeight,
            butGap:butGap,
            ballRad:ballRad,
            ballX:ballX,
            ballY:ballY
        };

        particles.reactant1Button = new particles.Button({
            txt:this.data.menu[0].name,
            x:this.boundary.a+this.boundary.x+butGap,
            y:(this.boundary.y+this.boundary.a)-butHeight*7-butGap*8,
            w:butWidth,
            h:butHeight,
            disabled:true,
            onClick:particles.buttonClicked,
            onClickParam: {
                choice:0,
                me:this
            }
        });

        particles.reactant2Button = new particles.Button({
            txt:this.data.menu[1].name,
            x:this.boundary.a+this.boundary.x+butGap,
            y:(this.boundary.y+this.boundary.a)-butHeight*6-butGap*7,
            w:butWidth,
            h:butHeight,
            disabled:true,
            onClick:particles.buttonClicked,
            onClickParam: {
                choice:1,
                me:this
            }
        });

        particles.product1Button = new particles.Button({
            txt:this.data.menu[2].name,
            x:this.boundary.a+this.boundary.x+butGap,
            y:(this.boundary.y+this.boundary.a)-butHeight*5-butGap*6,
            w:butWidth,
            h:butHeight,
            disabled:true,
            onClick:particles.buttonClicked,
            onClickParam: {
                choice:2,
                me:this
            }
        });

        particles.product2Button = new particles.Button({
            txt:this.data.menu[3].name,
            x:this.boundary.a+this.boundary.x+butGap,
            y:(this.boundary.y+this.boundary.a)-butHeight*4-butGap*5,
            w:butWidth,
            h:butHeight,
            disabled:true,
            onClick:particles.buttonClicked,
            onClickParam: {
                choice:3,
                me:this
            }
        });

        particles.plusButton = new particles.Button({
            txt:this.data.menu[4].name,
            x:this.boundary.a+this.boundary.x+butGap,
            y:(this.boundary.y+this.boundary.a)-butHeight*3-butGap*3,
            w:butWidth,
            h:butHeight,

            onClick:particles.buttonClicked,
            onClickParam: {
                choice:4,
                me:this
            }
        });
        particles.minusButton = new particles.Button({
            txt:this.data.menu[5].name,
            x:this.boundary.a+this.boundary.x+butGap,
            y:(this.boundary.y+this.boundary.a)-butHeight*2-butGap*2,
            w:butWidth,
            h:butHeight,

            onClick:particles.buttonClicked,
            onClickParam: {
                choice:5,
                me:this
            }

        });
        particles.stopButton = new particles.Button({
            txt:this.data.menu[6].name,
            x:this.boundary.a+this.boundary.x+butGap,
            y:(this.boundary.y+this.boundary.a)-butHeight,
            w:butWidth,
            h:butHeight,
                
            onClick:particles.buttonClicked,
            onClickParam: {
                choice:6,
                me:this
            }
        });

        this.imageLoader = new OU.util.ImageLoader({
            container: this,
            data: this.data,
            onLoad: function() {
                self.config.imagesLoaded=1;
                self.start();
            }
        });
    };

    OU.activity.Particles.prototype.resize = function() {
        OU.activity.Particles.superClass_.resize.call(this); // call the parent class resize   
        var config= this.config,bd;

        if(this.config.imagesLoaded==1){
            if(this.popUp!==undefined)
                this.popUp.close();

            this.bgLayer.resize();
            this.bgLayer.context.gradRect(); // draw background on backdrop layer

            this.imageLayer.resize();

            this.boundary = {
                x:this.w*.05,
                y:this.h*.05,
                w:this.w*.9,
                h:this.h*.9,
                a:0,
                aParticleRadius:0
            };
            this.boundary.a = this.boundary.w*.7<this.boundary.h?this.boundary.w*.7:this.boundary.h;

            config.vScale= this.boundary.a/config.vAreaW;
            this.boundary.aParticleRadius=config.vParticleRadius*config.vScale;

            this.fbWidth = this.w;

            bd=this.butData;
            // redefine dimentions depending on window size
            bd.butWidth = 0.429*this.boundary.a*.7;
            bd.ballRad=0.429*this.boundary.a*.1;
            bd.butGap = 0.429*this.boundary.a*.05;
            bd.butHeight = bd.ballRad*2;
            this.boundary.x=this.w*.05+(this.boundary.w-(this.boundary.a+2*bd.butGap+2*bd.ballRad+bd.butWidth))*.5;
            //bd.ballX=this.boundary.a+this.boundary.x+bd.butGap;
            bd.ballX=1.429*this.boundary.a-2*bd.ballRad+this.boundary.x;
            bd.ballY=(this.boundary.y+this.boundary.a)-bd.ballRad;

            particles.reactant1Button.resize({
                x:this.boundary.a+this.boundary.x+bd.butGap,
                y:(this.boundary.y+this.boundary.a)-bd.butHeight*7-bd.butGap*8,
                w:bd.butWidth,
                h:bd.butHeight
            });
            particles.reactant2Button.resize({
                x:this.boundary.a+this.boundary.x+bd.butGap,
                y:(this.boundary.y+this.boundary.a)-bd.butHeight*6-bd.butGap*7,
                w:bd.butWidth,
                h:bd.butHeight
            });
            particles.product1Button.resize({
                x:this.boundary.a+this.boundary.x+bd.butGap,
                y:(this.boundary.y+this.boundary.a)-bd.butHeight*5-bd.butGap*6,
                w:bd.butWidth,
                h:bd.butHeight
            });
            particles.product2Button.resize({
                x:this.boundary.a+this.boundary.x+bd.butGap,
                y:(this.boundary.y+this.boundary.a)-bd.butHeight*4-bd.butGap*5,
                w:bd.butWidth,
                h:bd.butHeight
            });
            particles.plusButton.resize({
                x:this.boundary.a+this.boundary.x+bd.butGap,
                y:(this.boundary.y+this.boundary.a)-bd.butHeight*3-bd.butGap*3,
                w:bd.butWidth,
                h:bd.butHeight
            });
            particles.minusButton.resize({
                x:this.boundary.a+this.boundary.x+bd.butGap,
                y:(this.boundary.y+this.boundary.a)-bd.butHeight*2-bd.butGap*2,
                w:bd.butWidth,
                h:bd.butHeight
            });
            particles.stopButton.resize({
                x:this.boundary.a+this.boundary.x+bd.butGap,
                y:(this.boundary.y+this.boundary.a)-bd.butHeight,
                w:bd.butWidth,
                h:bd.butHeight
            });
            this.controlLayer.resize();
            this.drawControl();
            this.doneLive=false;
            this.doRender=true;
        }
    };

    OU.activity.Particles.prototype.start = function() {
        var i;

        this.boundary.a = this.boundary.w*.7<this.boundary.h?this.boundary.w*.7:this.boundary.h;

        // initialize the particles
        this.items = [ ];
        for (i=this.data.menu[0].initialParticles; i--;) {
            this.addParticle(0);
        }
        for (i=this.data.menu[1].initialParticles; i--;) {
            this.addParticle(1);
        }
        for (i=this.data.menu[2].initialParticles; i--;) {
            this.addParticle(2);
        }
        for (i=this.data.menu[3].initialParticles; i--;) {
            this.addParticle(3);
        }
        this.drawControl();
        this.doRender=true;
        this.renderLoop();
    };
    OU.activity.Particles.prototype.addParticle= function(whichType) {
        var i=this.items.length-1;
        this.items.push( new this.Item( {
            particles: this,
            id: i,
            radius: this.config.vParticleRadius,
            x: this.config.vParticleRadius*2+(Math.random()*(this.config.vAreaW-(this.config.vParticleRadius*4))), // xpos
            y: this.config.vParticleRadius*2+(Math.random()*(this.config.vAreaW-(this.config.vParticleRadius*4))), // y pos
            vx: Math.random()*100-50, // x velocity
            vy: Math.random()*100-50, // y velocity
            m: 35, // mass
            t: whichType, // type of particle 0=R1;1=R2;2=P1;3=P2
            contx: this.imageLayer.context
        }));
        this.config.totalParticles+=1;
        if(whichType==0){
            this.config.actualBallsR1+=1;
        }else if(whichType==1){
            this.config.actualBallsR2+=1;
        }else if(whichType==2){
            this.config.actualBallsP1+=1;
        }else{
            this.config.actualBallsP2+=1;
        }
    };
    // buttons do this
    // 0=reactant1;1=reactant2;2=product1;3=product2;4=plus;5=minus;6=stop
    OU.activity.Particles.prototype.buttonClicked = function(p) {
        var whichClicked=p.choice;
        // testing the ability to change audio file
        if (whichClicked==1){
            // reactant 2 button
            if(p.me.config.step==0 && p.me.config.totalParticles<p.me.config.maxParticles){
                // add a reactant 2 particle
                p.me.addParticle(1);
                if(p.me.config.totalParticles==p.me.config.maxParticles){
                    p.me.reactant2Button.visibleParams.colour="#"+p.me.data.disabledColour;
                    p.me.reactant2Button.disabled = true;
                    p.me.config.stepCompleted=1;
                }
            }
        } else if (whichClicked==3){
            // product 2 button
            if(p.me.config.step==7 && p.me.config.totalParticles<p.me.config.maxParticles){
                // add a reactant 2 particle
                p.me.addParticle(3);
                if(p.me.config.totalParticles==p.me.config.maxParticles){
                    p.me.product2Button.visibleParams.colour="#"+p.me.data.disabledColour;
                    p.me.product2Button.disabled = true;
                    p.me.config.stepCompleted=1;
                }
            }
        } else if (whichClicked==4){
            // plus button
            if (p.me.config.speedFactor<3){
                p.me.config.speedFactor+=.4;
                OU.LocalStorage.save('ou.particles.speedFactor',p.me.config.speedFactor);
            }
        } else if (whichClicked==5){
            // minus button
            if (p.me.config.speedFactor>0.4){
                p.me.config.speedFactor-=.4;
                OU.LocalStorage.save('ou.particles.speedFactor',p.me.config.speedFactor);
            }

        } else if (whichClicked==6){
            // stop/start button
            p.me.config.stopstartCount+=1;
            if (p.me.doRender==true){
                p.me.doRender=false;
                p.me.stopButton.textParams.txt="Start";
            }else{
                p.me.doRender=true;
                p.me.stopButton.textParams.txt="Stop";
            }

            if(p.me.config.step==2 && p.me.config.stopstartCount>=2){
                p.me.config.stepCompleted=1;
            } else if(p.me.config.step==4 && p.me.config.stopstartCount>=20){
                p.me.config.stepCompleted=1;
            } else if(p.me.config.step==9 && p.me.config.stopstartCount>=8){
                p.me.config.stepCompleted=1;
            }
        }
        p.me.drawControl();
    };
    OU.activity.Particles.prototype.Button = function(params) {
        params.propTextHeight=.5;
        params.fontWeight='normal';
        params.events = particles.controlLayer.events;
        params.layer = particles.controlLayer;
        return new OU.util.Button(params);
    };
    OU.activity.Particles.prototype.Item = function( params ) {
        this.particles = params.particles;
        this.id = params.id;
        this.radius = params.radius;
        this.x = params.x;
        this.y = params.y;
        this.vx = params.vx;
        this.vy = params.vy;
        this.m = params.m;
        this.t = params.t;
        this.contx = params.contx;
        this.flash = 0;
        this.flashX = 0;
        this.flashY = 0;
    };
    // called to check if a ball has hit a side
    OU.activity.Particles.prototype.checkWalls = function(w) {
        var tempBallRadius = this.boundary.aParticleRadius;
        var config = this.config;
        if (this.items[w].x<config.vParticleRadius) {
            this.items[w].x = config.vParticleRadius+(config.vParticleRadius-this.items[w].x);
            this.items[w].vx *= -1;
        } else if (this.items[w].x*config.vScale>this.boundary.a-tempBallRadius) {
            this.items[w].x = ((this.boundary.a-tempBallRadius)-(this.items[w].x*config.vScale-(this.boundary.a-tempBallRadius)))/config.vScale;
            //this.items[w].x = (this.boundary.a-2*tempBallRadius)/config.vScale;
            this.items[w].vx *= -1;
        }
        if (this.items[w].y<config.vParticleRadius) {
            this.items[w].y = config.vParticleRadius+(config.vParticleRadius-this.items[w].y);
            this.items[w].vy *= -1;
        } else if (this.items[w].y*config.vScale>this.boundary.a-tempBallRadius) {
            this.items[w].y = ((this.boundary.a-tempBallRadius)-(this.items[w].y*config.vScale-(this.boundary.a-tempBallRadius)))/config.vScale;
            //this.items[w].y = (this.boundary.a-2*tempBallRadius)/config.vScale;
            this.items[w].vy *= -1;
        }
    };

    // called to check if a ball has hit another
    OU.activity.Particles.prototype.checkCollision = function(aa, bb) {
        var config = this.config,sa,sb,s1,s2,xy,currentSpeed,midX,midY,
        a=this.items[aa],b=this.items[bb],
        dx = b.x-a.x,
        dy = b.y-a.y,
        dist = Math.sqrt(dx*dx+dy*dy),
        overlap = 2*this.config.vParticleRadius - dist;
        
        if(overlap > 0){
            sa=Math.sqrt(a.vx*a.vx + a.vy*a.vy);
            sb=Math.sqrt(b.vx*b.vx + b.vy*b.vy);
            s1=(sa * (a.m - b.m) + 2 * sb * b.m) / (a.m + b.m);
            s2=(sb * (b.m - a.m) + 2 * sa * a.m) / (b.m + a.m);
            xy = [(b.x-a.x)/dist, (b.y-a.y)/dist];
            a.vx = -xy[0];
            a.vy = -xy[1];
            b.vx = xy[0];
            b.vy = xy[1];
            currentSpeed = Math.sqrt(a.vx*a.vx + a.vy*a.vy);
            a.vx = a.vx / currentSpeed * s1;
            a.vy = a.vy / currentSpeed * s1;
            currentSpeed = Math.sqrt(b.vx*b.vx + b.vy*b.vy);
            b.vx = b.vx / currentSpeed * s2;
            b.vy = b.vy / currentSpeed * s2;
            a.x += -xy[0] * overlap/2;
            a.y += -xy[1] * overlap/2;
            xy = [(a.x-b.x)/dist, (a.y-b.y)/dist];
            b.x += -xy[0] * overlap/2;
            b.y += -xy[1] * overlap/2;
            midX=a.x+(b.x-a.x)/2;
            midY=a.y+(b.y-a.y)/2;

            // swap types depending on the balls that collide - 0=Blue (); 1=yellow (R2); 2=red (P1); 3=green (P2)
            if (a.t==0 && b.t==1){
                if (config.actualBallsR1>config.eqMin || config.actualBallsR2>config.eqMin || config.actualBallsP1<config.eqMax || config.actualBallsP2<config.eqMax){
                    a.t=3;
                    b.t=2;
                    a.flash=1;
                    a.flashX=midX;
                    a.flashY=midY;
                    config.actualBallsR1--;
                    config.actualBallsR2--;
                    config.actualBallsP1++;
                    config.actualBallsP2++;
                }
            } else if (a.t==1 && b.t==0){
                if (config.actualBallsR1>config.eqMin || config.actualBallsR2>config.eqMin || config.actualBallsP1<config.eqMax || config.actualBallsP2<config.eqMax){
                    a.t=2;
                    b.t=3;
                    a.flash=1;
                    a.flashX=midX;
                    a.flashY=midY;
                    config.actualBallsR1--;
                    config.actualBallsR2--;
                    config.actualBallsP1++;
                    config.actualBallsP2++;
                }
            } else if (a.t==2 && b.t==3){
                if (config.actualBallsP1>config.eqMin || config.actualBallsP2>config.eqMin || config.actualBallsR1<config.eqMax || config.actualBallsR2<config.eqMax){
                    a.t=0;
                    b.t=1;
                    a.flash=1;
                    a.flashX=midX;
                    a.flashY=midY;
                    config.actualBallsR1++;
                    config.actualBallsR2++;
                    config.actualBallsP1--;
                    config.actualBallsP2--;
                }
            } else if (a.t==3 && b.t==2){
                if (config.actualBallsP1>config.eqMin || config.actualBallsP2>config.eqMin || config.actualBallsR1<config.eqMax || config.actualBallsR2<config.eqMax){
                    a.t=1;
                    b.t=0;
                    a.flash=1;
                    a.flashX=midX;
                    a.flashY=midY;
                    config.actualBallsR1++;
                    config.actualBallsR2++;
                    config.actualBallsP1--;
                    config.actualBallsP2--;
                }
            }
        }
    };

    OU.activity.Particles.prototype.renderLoop = function() {
        var self=this;
        if(this.doRender){
            this.render();
        }
        this.drawScreen();
        setTimeout(function() {
            self.renderLoop();
        },40);
    };
    OU.activity.Particles.prototype.render = function() {
        var l,a,b;
        for (l = 0; l<this.config.totalParticles; l++) {
            this.items[l].x += this.items[l].vx*this.config.speedFactor;
            this.items[l].y += this.items[l].vy*this.config.speedFactor;
            this.checkWalls(l);
        }
        // check through each of the balls to see if they have hit another ball
        // select a ball...
        for (a = this.config.totalParticles; a--;) {
            for (b=a; b--;) {
                this.checkCollision(a,b);
            }
        }
    // update the screen...
    //this.drawScreen();
    };

    OU.activity.Particles.prototype.drawControl = function() {
        var grd;

        this.controlLayer.clear();
        this.bgLayer.clear();
        var ctxCon = this.controlLayer.context;
        var bgCon = this.bgLayer.context;
        var butData=this.butData;

        // draw plinth
        bgCon.strokeStyle="#666666";
        //ctx.fillStyle="#ffffff";
        grd=bgCon.createLinearGradient(0,0,0,175);
        grd.addColorStop(0,"#EEEEEE");
        grd.addColorStop(1,"#FFFFFF");
        bgCon.fillStyle=grd;
        bgCon.lineWidth=1;
        bgCon.beginPath();
        bgCon.fillRect(this.boundary.x,this.boundary.y,this.boundary.a,this.boundary.a);
        bgCon.strokeRect(this.boundary.x,this.boundary.y,this.boundary.a,this.boundary.a);
        bgCon.closePath();
        bgCon.stroke();
        //console.log(butData.butHeight);

        //  y:(this.boundary.y+this.boundary.a)-butHeight*7-butGap*8,
        // draw the button balls
        ctxCon.drawImage(this.data.menu[0].image,butData.ballX,butData.ballY-butData.butHeight*7-butData.butGap*8+butData.ballRad,butData.butHeight,butData.butHeight);
        ctxCon.drawImage(this.data.menu[1].image,butData.ballX,butData.ballY-butData.butHeight*6-butData.butGap*7+butData.ballRad,butData.butHeight,butData.butHeight);
        ctxCon.drawImage(this.data.menu[2].image,butData.ballX,butData.ballY-butData.butHeight*5-butData.butGap*6+butData.ballRad,butData.butHeight,butData.butHeight);
        ctxCon.drawImage(this.data.menu[3].image,butData.ballX,butData.ballY-butData.butHeight*4-butData.butGap*5+butData.ballRad,butData.butHeight,butData.butHeight);

        ctxCon.strokeStyle="#000";
        ctxCon.lineWidth=10*this.config.vScale;

        ctxCon.beginPath();
        ctxCon.arc(butData.ballX+butData.ballRad,butData.ballY-butData.butHeight*7-butData.butGap*8+butData.ballRad*2,butData.ballRad-1,0,Math.PI*2,true);
        ctxCon.closePath();
        ctxCon.stroke();
        ctxCon.beginPath();
        ctxCon.arc(butData.ballX+butData.ballRad,butData.ballY-butData.butHeight*6-butData.butGap*7+butData.ballRad*2,butData.ballRad-1,0,Math.PI*2,true);
        ctxCon.closePath();
        ctxCon.stroke();
        ctxCon.beginPath();
        ctxCon.arc(butData.ballX+butData.ballRad,butData.ballY-butData.butHeight*5-butData.butGap*6+butData.ballRad*2,butData.ballRad-1,0,Math.PI*2,true);
        ctxCon.closePath();
        ctxCon.stroke();
        ctxCon.beginPath();
        ctxCon.arc(butData.ballX+butData.ballRad,butData.ballY-butData.butHeight*4-butData.butGap*5+butData.ballRad*2,butData.ballRad-1,0,Math.PI*2,true);
        ctxCon.closePath();
        ctxCon.stroke();

        // ctx = this.imageLayer.context;
        this.reactant1Button.render();
        this.reactant2Button.render();
        this.product1Button.render();
        this.product2Button.render();
        this.minusButton.render();
        this.plusButton.render();
        this.stopButton.render();

    };

    // redraw the screen
    OU.activity.Particles.prototype.drawScreen = function() {
        var i,item,vS=this.config.vScale,b=this.boundary;
        this.imageLayer.clear();
        //
        var ctx = this.imageLayer.context;

        ctx.lineWidth=15*vS;

        // draw the moved particles
        for (i=this.config.totalParticles; i--;) {
            // draw collision flash
            item=this.items[i];
            if (item.flash==1){
                ctx.fillStyle="#FF6666";
                ctx.beginPath();
                ctx.arc(item.flashX*vS+b.x,item.flashY*vS+b.y,b.aParticleRadius*.7,0,Math.PI*2,true);
                ctx.closePath();
                ctx.fill();
                this.items[i].flash=0;
            } else {
                ctx.strokeStyle="#000000";
            }
            ctx.drawImage(this.data.menu[item.t].image,item.x*vS+b.x-b.aParticleRadius,item.y*vS+b.y-b.aParticleRadius,2*b.aParticleRadius,2*b.aParticleRadius);
            ctx.beginPath();
            ctx.arc(item.x*vS+b.x,item.y*vS+b.y,b.aParticleRadius-1,0,Math.PI*2,true);
            ctx.closePath();
            ctx.stroke();
        }

        ctx.lineWidth=1;

    };
    OU.activity.Particles.prototype.makePopup = function(infoText) {

        var events=this.controlLayer.events, winX, winY, winH, winW;
        if(this.controller!==undefined) {
            winX=window.innerWidth*.20;
            winY=window.innerHeight*.15;
            winW=window.innerWidth*.6;
            winH=window.innerHeight*.7;
        }
        else {
            winX=this.w*.20;
            winY=this.h*.15;
            winW=this.w*.6;
            winH=this.h*.7;
        }
        events.pressed=false;
        events.touched=false;
        //events.disableSwipe();
        events.clickable.push(particles.popUp = new OU.util.PopUpInfo({
            container: this,
            txt:infoText,
            x:winX,
            y:winY,
            w:winW,
            h:winH,
            onClose: function() {
            //events.enableSwipe();
            }
        }));
    };

    // step from the controller
    OU.activity.Particles.prototype.step = function(n) {
        var infoText;
        //this.product2Button.visibleParams.txt="step = "+n;
        //this.product2Button.render();
        if(this.config.step!=n) {
            // check step completed before moving on
            if(this.config.stepCompleted==1){
                this.config.step=n;
                this.config.stepCompleted=0;
                // if step completed next audio
                if(this.config.step==1){
                    this.config.stopstartCount=0;
                    this.controller.stepAudio(2);
                }else if(this.config.step==3){
                    this.config.stopstartCount=0;
                    this.controller.stepAudio(3);
                }else if(this.config.step==5){
                    this.config.stopstartCount=0;
                    this.controller.stepAudio(4);
                }else if(this.config.step==8){
                    this.config.stopstartCount=0;
                    this.controller.stepAudio(2);
                }else if(this.config.step==10){
                    this.config.stopstartCount=0;
                    infoText='<p>You should have found that there were the same number of molecules of the first product at equilibrium as there were when you started by adding the reactants.</p><p>The equilibrium mixture is the same whether you start with reactants or products.</p>';
                    this.makePopup(infoText);
                }
            }
            // next audio file start
            if(n==2||n==4||n==9){
                this.config.step=n;
            } else if (n==6) {
                this.config.step=n;
                this.config.stopstartCount=0;
                this.config.stepCompleted=0;
                infoText='<p>At equilibrium, molecules continue to react, but overall the total amounts of products and reactant stay fixed.</p>';
                this.makePopup(infoText);
            } else if (n==0) {
                this.config.step=n;
                this.config.stopstartCount=0;
                this.config.stepCompleted=0;
                this.reactant2Button.visibleParams.colour="#"+this.data.enabledColour;
                this.reactant2Button.disabled = false;
                this.reactant2Button.render();
            } else if (n==7) {
                this.config.step=n;
                this.config.stopstartCount=0;
                this.config.stepCompleted=0;
                this.product2Button.visibleParams.colour="#"+this.data.enabledColour;
                this.product2Button.disabled = false;
                this.product2Button.render();
            }
        }
    };

    OU.base(this,data,instance,controller);
};
OU.inherits(OU.activity.Particles,OU.util.Activity);
