/**
 * @fileOverview OU.activity.Pressurechart - S104 Chemical Equilibrium Activity
 *
 * @param {JSON array} data Data set
 *
 * @author Kevin Quick & Jon Linney
 * @author Refactored by Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.PopUpInfo');
OU.require('OU.util.NavButtons');
OU.require('OU.util.Layer');
OU.require('OU.util.ImageLoader');
OU.require('OU.util.Slider');
/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.Pressurechart = function ( data, instance, controller ) {
    var pressurechart = this, temp1, temp2, butWidth, ballRad, butGap, butHeight, ballX, ballY, clickable;
    OU.obj[ instance ] = this;
    OU.activity.Pressurechart.prototype.canvasView = function () {
        var self = this;
        this.data = data || {};
        this.controller = controller;
        this.instance = instance || 'p1';
        this.whichReaction = 0;
        this.totalPressure = 1;
        this.popUpAudio = 0;
        OU.LocalStorage.save("ou.pressurechart.pressure.reaction", 0);
        this.config = {
            vAreaW:5000,
            vScale:1,
            thenum:10,
            change:0,
            myColor:["#0037ff", "#fffa00", "#ff2400", "#00f900"],
            myData:[20, 30, 20, 40],
            maxPressure:this.data.maxPressure,
            minPressure:this.data.minPressure,
            step:-1,
            stepCompleted:0,
            stopstartCount:0,
            imagesLoaded:0,
            fontSize1:10,
            fontSize2:7
        };
        this.bgLayer = new OU.util.Layer({
            container:this
        });
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        this.imageLayer = new OU.util.Layer({
            container:this
        });
        this.controlLayer = new OU.util.Layer({
            container:this,
            hasEvents:true
        });
        this.boundary = {
            x:this.w * .05,
            y:this.h * .05,
            w:this.w * .9,
            h:this.h * .9,
            a:0,
            aParticleRadius:0
        };
        temp1 = this.boundary.w * .7;
        temp2 = this.boundary.h;
        if (temp1 < temp2) {
            this.boundary.a = temp1;
        }
        else {
            this.boundary.a = temp2;
        }
        this.config.vScale = this.boundary.a / this.config.vAreaW;
        this.boundary.aParticleRadius = this.config.vParticleRadius * this.config.vScale;
        this.fbWidth = this.w;
        // define dimentions depending on window size
        butWidth = 0.429 * this.boundary.a * .7;
        ballRad = 0.429 * this.boundary.a * .08;
        butGap = 0.429 * this.boundary.a * .05;
        butHeight = ballRad * 2;
        this.boundary.x = this.w * .05 + (this.boundary.w - (this.boundary.a + 2 * butGap + 2 * ballRad + butWidth)) * .5;
        //
        ballX = 1.429 * this.boundary.a - 2 * ballRad + this.boundary.x;
        ballY = (this.boundary.y + this.boundary.a) - ballRad;
        this.butData = {
            butWidth:butWidth,
            butHeight:butHeight,
            butGap:butGap,
            ballRad:ballRad,
            ballX:ballX,
            ballY:ballY
        };
        clickable = this.controlLayer.events.clickable;
        if (this.data.changeButton==1) {
            pressurechart.changeButton = new pressurechart.Button({
                txt:"Change",
                x:this.boundary.a + this.boundary.x + butGap,
                y:(this.boundary.y + this.boundary.a) - butHeight,
                w:butWidth,
                h:butHeight,
                onClick:pressurechart.buttonClicked,
                onClickParam:{
                    choice:3,
                    me:this
                },
                disabled:true
            });
        }
        this.pressureSlider = new OU.util.Slider({
            container:this,
            x:this.boundary.a + this.boundary.x + butGap,
            y:(this.boundary.y + this.boundary.a) - 3 * butHeight,
            w:butWidth,
            h:butHeight,
            //sliderStyle: 'circle',
            drawContainer:false,
            callback:this.setPressure,
            callbackParam:this,
            background:{
                clear:true
            },
            context:this.controlLayer.context
        });
        clickable.push(this.pressureSlider);
        this.imageLoader = new OU.util.ImageLoader({
            container:this,
            data:this.data,
            onLoad:function () {
                self.config.imagesLoaded = 1;
                self.start();
            }
        });
    };
    OU.activity.Pressurechart.prototype.drawPie = function () {
        var i, lastend = -Math.PI / 2, config = this.config,
            ctx = this.imageLayer.context, midPointX = this.boundary.x + (this.boundary.a / 2),
            midPointY = this.boundary.y + (this.boundary.a / 2);
        ctx.strokeStyle = "#000";
        for (i = 0; i < config.myData.length; i++) {
            ctx.fillStyle = config.myColor[i];
            ctx.beginPath();
            ctx.moveTo(midPointX, midPointY);
            ctx.arc(midPointX, midPointY, this.boundary.a * .35, lastend - Math.PI * 2 * (config.myData[i] / 360), lastend, false);
            ctx.lineTo(midPointX, midPointY);
            ctx.fill();
            ctx.stroke();
            lastend -= Math.PI * 2 * (config.myData[i] / 360);
        }
    };
    OU.activity.Pressurechart.prototype.resize = function () {
        OU.activity.Pressurechart.superClass_.resize.call(this); // call the parent class resize 
        var config = this.config, temp1, temp2, bd;
        if (this.popUp!==undefined)
            this.popUp.close();
        if (this.config.imagesLoaded==1) {
            this.bgLayer.resize();
            this.bgLayer.context.gradRect(); // draw background on backdrop layer
            this.imageLayer.resize();
            this.boundary = {
                x:this.w * .05,
                y:this.h * .05,
                w:this.w * .9,
                h:this.h * .9,
                a:0,
                aParticleRadius:0
            };
            temp1 = this.boundary.w * .7;
            temp2 = this.boundary.h;
            if (temp1 < temp2) {
                this.boundary.a = temp1;
            }
            else {
                this.boundary.a = temp2;
            }
            config.vScale = this.boundary.a / config.vAreaW;
            this.boundary.aParticleRadius = config.vParticleRadius * config.vScale;
            this.fbWidth = this.w;
            bd = this.butData;
            // redefine dimentions depending on window size
            bd.butWidth = 0.429 * this.boundary.a * .7;
            bd.ballRad = 0.429 * this.boundary.a * .08;
            bd.butGap = 0.429 * this.boundary.a * .05;
            bd.butHeight = bd.ballRad * 2;
            this.boundary.x = this.w * .05 + (this.boundary.w - (this.boundary.a + 2 * bd.butGap + 2 * bd.ballRad + bd.butWidth)) * .5;
            bd.ballX = 1.429 * this.boundary.a - 2 * bd.ballRad + this.boundary.x;
            bd.ballY = (this.boundary.y + this.boundary.a) - bd.ballRad;
            if (this.data.changeButton==1) {
                pressurechart.changeButton.resize({
                    x:this.boundary.a + this.boundary.x + bd.butGap,
                    y:(this.boundary.y + this.boundary.a) - bd.butHeight,
                    w:bd.butWidth,
                    h:bd.butHeight,
                    fontSize:200 * config.vScale
                });
            }
            this.pressureSlider.resize({
                x:this.boundary.a + this.boundary.x + bd.butGap,
                y:(this.boundary.y + this.boundary.a) - bd.butHeight * 3,
                w:bd.butWidth,
                h:bd.butHeight
            });
            this.controlLayer.resize();
            this.drawControl();
            this.doneLive = false;
            this.doRender = true;
        }
    };
    OU.activity.Pressurechart.prototype.start = function () {
        this.drawControl();
        this.doRender = true;
        this.renderLoop();
    };
    //change pressure when slider altered
    OU.activity.Pressurechart.prototype.setPressure = function ( pressure, pchart ) {
        pchart.totalPressure = pchart.config.minPressure + pressure * (pchart.config.maxPressure - pchart.config.minPressure);
        pchart.pressureSlider.render(pressure);
        pchart.drawControl();
    };
    // buttons do this
    OU.activity.Pressurechart.prototype.buttonClicked = function ( p ) {
        var whichClicked = p.choice;
        if (whichClicked==3 && p.me.config.stepCompleted >= 3) {
            // make popup to change the equation
            if (p.me.popUpAudio==0) {
                p.me.popUpAudio = 1;
                p.me.controller.stepAudio(2);
            }
            p.me.makePopUp();
        }
        p.me.drawControl();
    };
    OU.activity.Pressurechart.prototype.Button = function ( params ) {
        params.propTextHeight = .5;
        params.fontWeight = 'normal';
        params.layer = pressurechart.controlLayer;
        return new OU.util.Button(params);
    };
    // if true call the render loop to change the screen
    OU.activity.Pressurechart.prototype.renderLoop = function () {
        var self = this;
        this.drawScreen();
        setTimeout(function () {
            self.renderLoop();
        }, 40);
    };
    //
    OU.activity.Pressurechart.prototype.drawControl = function () {
        var grd, gx, gy, tx, ty, butData = this.butData, fs, bx, by, theWidth,
            ctxCon = this.controlLayer.context, bgCon = this.bgLayer.context, moleculeW = 250;
        this.pressure();
        this.controlLayer.clear();
        this.bgLayer.clear();
        // draw plinth
        bgCon.strokeStyle = "#666666";
        grd = bgCon.createLinearGradient(0, 0, 0, 175);
        grd.addColorStop(0, "#EEEEEE");
        grd.addColorStop(1, "#FFFFFF");
        bgCon.fillStyle = grd;
        bgCon.lineWidth = 1;
        bgCon.beginPath();
        bgCon.fillRect(this.boundary.x, this.boundary.y, this.boundary.a, this.boundary.a);
        bgCon.strokeRect(this.boundary.x, this.boundary.y, this.boundary.a, this.boundary.a);
        bgCon.closePath();
        bgCon.stroke();
        // draw slider triangle
        bgCon.save();
        bgCon.fillStyle = "#000";
        bgCon.globalAlpha = .15;
        bgCon.beginPath();
        bgCon.moveTo(this.boundary.a + this.boundary.x + butData.butGap + butData.butHeight * .5, (this.boundary.y + this.boundary.a) - 3 * butData.butHeight + butData.butHeight * .5);
        bgCon.lineTo(this.boundary.a + this.boundary.x + butData.butGap + butData.butWidth - butData.butHeight * .5, (this.boundary.y + this.boundary.a) - 3 * butData.butHeight + butData.butHeight * .25);
        bgCon.lineTo(this.boundary.a + this.boundary.x + butData.butGap + butData.butWidth - butData.butHeight * .5, (this.boundary.y + this.boundary.a) - 3 * butData.butHeight + butData.butHeight * .75);
        bgCon.closePath();
        bgCon.fill();
        bgCon.restore();
        ctxCon.strokeStyle = "#000";
        ctxCon.lineWidth = 10 * this.config.vScale;
        // render the button
        if (this.data.changeButton==1) {
            this.changeButton.render();
        }
        // text scale
        tx = this.boundary.a + this.boundary.x + butData.butGap;
        // draw the guage
        gx = this.boundary.a + this.boundary.x + butData.butGap + butData.butWidth * .5;
        gy = (this.boundary.y + this.boundary.a) - butData.butHeight * 5 - butData.butGap * 5,
            this.makeGuage(this.totalPressure, 900 * this.config.vScale, gx, gy);
        // reset text format after guage drawn
        fs = 200 * this.config.vScale;
        ctxCon.font = fs + "px " + OU.theme.font;
        ctxCon.fillStyle = "#666";
        ty = gy - (850 * this.config.vScale) * .5;
        ctxCon.fillText("atm", tx, ty);
        ctxCon.strokeStyle = "#000";
        ctxCon.lineWidth = 1;
        ty = (this.boundary.y + this.boundary.a) - 4 * butData.butHeight,
            ctxCon.fillText("pressure", tx, ty);
        // draw particle titles
        if (this.data.reactions[this.whichReaction].menuChars[0]!="") {
            ctxCon.drawImage(this.data.menu[0].image, this.boundary.a + this.boundary.x + butData.butGap, this.boundary.y, butData.butHeight, butData.butHeight);
            ctxCon.beginPath();
            ctxCon.arc(this.boundary.a + this.boundary.x + butData.butGap + butData.ballRad, this.boundary.y + butData.ballRad, butData.ballRad - 1, 0, Math.PI * 2, true);
            ctxCon.closePath();
            ctxCon.stroke();
            bx = this.boundary.a + this.boundary.x + butData.butHeight + butData.butGap;
            by = this.boundary.y - this.config.vScale * moleculeW / 2 + butData.butHeight * .5;
            new OU.util.DynText({
                txt:this.data.reactions[this.whichReaction].menuChars[0],
                x:bx,
                y:by,
                w:this.config.vScale * 6 * butWidth,
                h:this.config.vScale * moleculeW,
                context:ctxCon,
                propTextHeight:1,
                align:'left',
                fontSize:this.config.vScale * moleculeW,
                fontWeight:'normal'
            });
            ctxCon.save();
            ctxCon.textBaseline = "middle";
            by = this.boundary.y + butData.butHeight * .5;
            bx = bx + butData.butHeight * 2.5;
            ctxCon.fillText(this.proportionA + "%", bx, by);
            ctxCon.restore();
        }
        if (this.data.reactions[this.whichReaction].menuChars[1]!="") {
            ctxCon.drawImage(this.data.menu[1].image, this.boundary.a + this.boundary.x + butData.butGap, this.boundary.y + butData.butHeight + butData.butGap, butData.butHeight, butData.butHeight);
            ctxCon.beginPath();
            ctxCon.arc(this.boundary.a + this.boundary.x + butData.butGap + butData.ballRad, this.boundary.y + butData.butHeight + butData.butGap + butData.ballRad, butData.ballRad - 1, 0, Math.PI * 2, true);
            ctxCon.closePath();
            ctxCon.stroke();
            bx = this.boundary.a + this.boundary.x + butData.butHeight + butData.butGap;
            by = this.boundary.y - this.config.vScale * moleculeW / 2 + butData.butHeight * 1.5 + butData.butGap;
            new OU.util.DynText({
                txt:this.data.reactions[this.whichReaction].menuChars[1],
                x:bx,
                y:by,
                w:this.config.vScale * 6 * butWidth,
                h:this.config.vScale * moleculeW,
                context:ctxCon,
                propTextHeight:1,
                align:'left',
                fontSize:this.config.vScale * moleculeW,
                fontWeight:'normal'
            });
            ctxCon.save();
            ctxCon.textBaseline = "middle";
            by = this.boundary.y + butData.butHeight * 1.5 + butData.butGap;
            bx = bx + butData.butHeight * 2.5;
            ctxCon.fillText(this.proportionB + "%", bx, by);
            ctxCon.restore();
        }
        if (this.data.reactions[this.whichReaction].menuChars[2]!="") {
            ctxCon.drawImage(this.data.menu[2].image, this.boundary.a + this.boundary.x + butData.butGap, this.boundary.y + butData.butHeight * 2 + butData.butGap * 2, butData.butHeight, butData.butHeight);
            ctxCon.beginPath();
            ctxCon.arc(this.boundary.a + this.boundary.x + butData.butGap + butData.ballRad, this.boundary.y + butData.butHeight * 2 + butData.butGap * 2 + butData.ballRad, butData.ballRad - 1, 0, Math.PI * 2, true);
            ctxCon.closePath();
            ctxCon.stroke();
            bx = this.boundary.a + this.boundary.x + butData.butHeight + butData.butGap;
            by = this.boundary.y - this.config.vScale * moleculeW / 2 + butData.butHeight * 2.5 + butData.butGap * 2.0;
            new OU.util.DynText({
                txt:this.data.reactions[this.whichReaction].menuChars[2],
                x:bx,
                y:by,
                w:this.config.vScale * 6 * butWidth,
                h:this.config.vScale * moleculeW,
                context:ctxCon,
                propTextHeight:1,
                align:'left',
                fontSize:this.config.vScale * moleculeW,
                fontWeight:'normal'
            });
            ctxCon.save();
            ctxCon.textBaseline = "middle";
            by = this.boundary.y + butData.butHeight * 2.5 + 2 * butData.butGap;
            bx = bx + butData.butHeight * 2.5;
            ctxCon.fillText(this.proportionC + "%", bx, by);
            ctxCon.restore();
        }
        if (this.data.reactions[this.whichReaction].menuChars[3]!="") {
            ctxCon.drawImage(this.data.menu[3].image, this.boundary.a + this.boundary.x + butData.butGap, this.boundary.y + butData.butHeight * 3 + butData.butGap * 3, butData.butHeight, butData.butHeight);
            ctxCon.beginPath();
            ctxCon.arc(this.boundary.a + this.boundary.x + butData.butGap + butData.ballRad, this.boundary.y + butData.butHeight * 3 + butData.butGap * 3 + butData.ballRad, butData.ballRad - 1, 0, Math.PI * 2, true);
            ctxCon.closePath();
            ctxCon.stroke();
            bx = this.boundary.a + this.boundary.x + butData.butHeight + butData.butGap;
            by = this.boundary.y - this.config.vScale * moleculeW / 2 + butData.butHeight * 3.5 + butData.butGap * 3.0;
            new OU.util.DynText({
                txt:this.data.reactions[this.whichReaction].menuChars[3],
                x:bx,
                y:by,
                w:this.config.vScale * 6 * butWidth,
                h:this.config.vScale * moleculeW,
                context:ctxCon,
                propTextHeight:1,
                align:'left',
                fontSize:this.config.vScale * moleculeW,
                fontWeight:'normal'
            });
            ctxCon.save();
            ctxCon.textBaseline = "middle";
            by = this.boundary.y + butData.butHeight * 3.5 + 3 * butData.butGap;
            bx = bx + butData.butHeight * 2.5;
            ctxCon.fillText(this.proportionD + "%", bx, by);
            ctxCon.restore();
        }
        theWidth = this.boundary.a * .7;
        tx = this.boundary.x + this.boundary.a * .5 - theWidth / 2;
        by = this.boundary.y + this.boundary.a - 1.5 * this.config.vScale * 300;
        new OU.util.DynText({
            txt:this.data.reactions[this.whichReaction].formula,
            x:tx,
            y:by,
            w:theWidth,
            h:this.config.vScale * 300,
            context:ctxCon,
            propTextHeight:1,
            align:'center',
            fontWeight:'normal'
        });
        this.pressureSlider.render((this.totalPressure - this.config.minPressure) / (this.config.maxPressure - this.config.minPressure));
    };
    OU.activity.Pressurechart.prototype.drawScreen = function () {
        this.imageLayer.clear();
        var ctx = this.imageLayer.context;
        ctx.lineWidth = 1;
        this.pressure();
        this.drawPie();
    };
    OU.activity.Pressurechart.prototype.makePopUp = function () {
        var self = this;
        var infoText = '<h1>Select a Chemical Equation</h1>';
        var currentGroup = this.data.reactions[0].group;
        var currentReaction = 1 * OU.LocalStorage.load('ou.pressurechart.pressure.reaction');
        var cellWidthTotal = this.w * .6 * .8;
        infoText += '<form name="pieform">';
        infoText += '<table width="' + cellWidthTotal + '" border="0"><tr><td width="' + cellWidthTotal * .2 + '" bgcolor="#999999" class="cepressurechartLL">' + this.data.reactions[0].group + '</td><td width="' + cellWidthTotal * .8 + '" valign="top">';
        for (var fm = 0; fm < this.data.reactions.length; fm++) {
            if (currentGroup!=this.data.reactions[fm].group) {
                infoText += '</td></tr>';
                infoText += '<tr><td width="' + cellWidthTotal * .2 + '" bgcolor="#999999" class="cepressurechartLL">' + this.data.reactions[fm].group + '</td><td width="' + cellWidthTotal * .8 + '" valign="top">';
                currentGroup = this.data.reactions[fm].group;
            }
            if (fm==currentReaction) {
                infoText += '<input type="radio" name="group1" id="radiofield' + fm + '" value="' + fm + '" checked="checked" onclick="OU.LocalStorage.save(\'ou.pressurechart.pressure.reaction\',' + fm + ')"/><label for="radiofield' + fm + '"> ' + this.data.reactions[fm].formula + '</label>';
            }
            else {
                infoText += '<input type="radio" name="group1" id="radiofield' + fm + '" value="' + fm + '"  onclick="OU.LocalStorage.save(\'ou.pressurechart.pressure.reaction\',' + fm + ')"/><label for="radiofield' + fm + '"> ' + this.data.reactions[fm].formula + '</label>';
            }
            if (fm!=this.data.reactions.length - 1) {
                infoText += '<br/>';
            }
        }
        infoText += '</td></tr>';
        infoText += '</table>';
        infoText += '</form>';
        //
        var events = this.controlLayer.events;
        events.pressed = false;
        events.touched = false;
        new OU.util.PopUpInfo({
            container:this,
            txt:infoText,
            x:this.w * .20,
            y:this.h * .15,
            w:this.w * .6,
            h:this.h * .7,
            onClose:function () {
                self.whichReaction = 1 * OU.LocalStorage.load('ou.pressurechart.pressure.reaction');
                self.totalPressure = 1;
                self.drawControl();
            }
        });
    };
    // make guage
    OU.activity.Pressurechart.prototype.makeGuage = function ( pressure, targetHeight, centreX, centreY ) {
        var ctx = this.controlLayer.context;
        var theRadius = 50;
        var scale = targetHeight / (theRadius * 2);
        theRadius = theRadius * scale;
        var innerRadius = theRadius - 22 * scale;
        var angle, x, y, x2, y2;
        var dash = 3 * scale;
        var flag = 0;
        var label = 0;
        var tail = 10 * scale;
        var triangle1 = 1 * scale;
        var triangle2 = 3 * scale;
        var centreRad = 4 * scale;
        var fontSize1 = 10 * scale;
        var fontSize2 = 7 * scale;
        var needleTravel = 2 * Math.PI - Math.PI * (30 / 180);
        ctx.save();
        ctx.fillStyle = "#fff";
        // white background
        ctx.beginPath();
        ctx.arc(centreX, centreY, theRadius - 2, 0, Math.PI * 2, false);
        ctx.fill();
        ctx.fillStyle = "#000";
        // outer grey rim
        ctx.beginPath();
        ctx.arc(centreX, centreY, theRadius - 2, 0, Math.PI * 2, false);
        ctx.strokeStyle = "#777";
        ctx.lineWidth = 4 * scale;
        ctx.stroke();
        // outer black rim
        ctx.beginPath();
        ctx.arc(centreX, centreY, theRadius, 0, Math.PI * 2, false);
        ctx.strokeStyle = "#000";
        ctx.lineWidth = 1 * scale;
        ctx.stroke();
        // dial arcs
        ctx.beginPath();
        ctx.arc(centreX, centreY, innerRadius, 0, (Math.PI * 2) * (324 - 90 - 30 + 33) / 360, false);
        ctx.stroke();
        ctx.beginPath();
        ctx.arc(centreX, centreY, innerRadius, 0, (Math.PI * 2) * .75, true);
        ctx.stroke();
        // dial dashes and numbers
        for (var i = 0; i < 360; i = i + 33) {
            angle = Math.PI * (i - 180 + 30) / 180;
            x = innerRadius * Math.sin(angle);
            y = innerRadius * Math.cos(angle);
            x2 = dash * Math.sin(angle);
            y2 = dash * Math.cos(angle);
            //
            //if (label>0) {
            ctx.beginPath();
            ctx.moveTo(centreX + x, centreY + y);
            ctx.lineTo(centreX + x + x2, centreY + y + y2);
            ctx.stroke();
            //}
            if (flag==0) {
                flag = 1;
                x2 = 3 * dash * Math.sin(angle);
                y2 = 3 * dash * Math.cos(angle);
                ctx.textAlign = "center";
                ctx.textBaseline = "middle";
                ctx.font = fontSize1 + "px " + OU.theme.font;
                if (label > 0) {
                    ctx.fillText(5 - label, centreX + x + x2, centreY + y + y2);
                }
                label += 1;
            }
            else {
                flag = 0;
            }
        }
        // atm caption
        ctx.font = fontSize2 + "px " + OU.theme.font;
        ctx.fillText("atm", centreX, centreY - innerRadius * .5);
        // pointer
        angle = Math.PI * 2 - (needleTravel * pressure / 5 - Math.PI);
        x = innerRadius * Math.sin(angle);
        y = innerRadius * Math.cos(angle);
        y2 = triangle1 * Math.sin(angle);
        x2 = triangle1 * Math.cos(angle);
        //
        ctx.beginPath();
        ctx.moveTo(centreX + x, centreY + y);
        ctx.lineTo(centreX + x2, centreY - y2);
        ctx.lineTo(centreX - x2, centreY + y2);
        ctx.lineTo(centreX + x, centreY + y);
        ctx.fill();
        ctx.stroke();
        // pointer tail
        x = tail * Math.sin(angle);
        y = tail * Math.cos(angle);
        ctx.beginPath();
        ctx.moveTo(centreX, centreY);
        y2 = triangle2 * Math.sin(angle);
        x2 = triangle2 * Math.cos(angle);
        ctx.lineTo(centreX - x + x2, centreY - y - y2);
        ctx.lineTo(centreX - x - x2, centreY - y + y2);
        ctx.lineTo(centreX, centreY);
        ctx.fill();
        ctx.stroke();
        // centre circles
        ctx.beginPath();
        ctx.arc(centreX, centreY, centreRad, 0, Math.PI * 2, false);
        ctx.fill();
        ctx.beginPath();
        ctx.arc(centreX, centreY, centreRad * .5, 0, Math.PI * 2, false);
        ctx.fillStyle = "#ffaa00";
        ctx.fill();
        // change text styles back to default...
        ctx.textAlign = "start";
        ctx.restore();
    };
    // Class for 'Effect of pressure' section of S104 Chemical Equilibrium program
    OU.activity.Pressurechart.prototype.pressure = function () {
        //---------------------------------------------------------
        //declare and instantiate variables
        //---------------------------------------------------------
        var R = 8.31441;
        //---------------------------------------------------------
        //main method for calculating equilibrium quantities of reactants and products
        //at different pressures
        var propA, propB, propC, propD;
        var partialPressure = new Array();
        var maxPartialPressure = new Array();
        var estimatedK, factor, iterations, totalPartialPressure;
        var reactions = this.data.reactions[this.whichReaction];
        var K = Math.exp(-1000 / (R * reactions.reactionTemperature) * reactions.deltaH) * Math.exp(reactions.deltaS / R);
        //calculate partial pressures
        maxPartialPressure[0] = reactions.stoichiometry[0] / (reactions.stoichiometry[0] + reactions.stoichiometry[1]) * this.totalPressure;
        maxPartialPressure[1] = reactions.stoichiometry[1] / (reactions.stoichiometry[0] + reactions.stoichiometry[1]) * this.totalPressure;
        maxPartialPressure[2] = reactions.stoichiometry[2] / (reactions.stoichiometry[2] + reactions.stoichiometry[3]) * this.totalPressure;
        maxPartialPressure[3] = reactions.stoichiometry[3] / (reactions.stoichiometry[2] + reactions.stoichiometry[3]) * this.totalPressure;
        //
        partialPressure[0] = 0.01 * maxPartialPressure[0];
        partialPressure[1] = 0.01 * maxPartialPressure[1];
        partialPressure[2] = 0.99 * maxPartialPressure[2];
        partialPressure[3] = 0.99 * maxPartialPressure[3];
        iterations = 0;
        //
        // Have to use a numerical method rather than solving directly for unknowns.
        // Use estimated value of K and iterate until within a specified tolerance.
        estimatedK = (Math.pow(partialPressure[2], reactions.stoichiometry[2]) * Math.pow(partialPressure[3], reactions.stoichiometry[3])) / (Math.pow(partialPressure[0], reactions.stoichiometry[0]) * Math.pow(partialPressure[1], reactions.stoichiometry[1]));
        //
        while (Math.abs(1 - estimatedK / K) > 0.0005 && iterations < 1000) {
            factor = Math.sqrt(Math.sqrt(estimatedK / K));
            partialPressure[0] = partialPressure[0] * factor;
            partialPressure[1] = partialPressure[1] * factor;
            partialPressure[2] = partialPressure[2] / factor;
            partialPressure[3] = partialPressure[3] / factor;
            //
            estimatedK = (Math.pow(partialPressure[2], reactions.stoichiometry[2]) * Math.pow(partialPressure[3], reactions.stoichiometry[3])) / (Math.pow(partialPressure[0], reactions.stoichiometry[0]) * Math.pow(partialPressure[1], reactions.stoichiometry[1]));
            iterations++;
            //
            if (iterations >= 1000) {
                console.log("Seemed to get stuck");
            }
        }
        // =================================================
        // // Proportion tests for pie chart amounts:
        // The total pressure:
        totalPartialPressure = partialPressure[0] + partialPressure[1] + partialPressure[2] + partialPressure[3];
        // Proportion of total pressure for A
        propA = partialPressure[0] / totalPartialPressure;
        // Proportion of total pressure for B
        propB = partialPressure[1] / totalPartialPressure;
        // Proportion of total pressure for C
        propC = partialPressure[2] / totalPartialPressure;
        // Proportion of total pressure for D
        propD = partialPressure[3] / totalPartialPressure;
        // Pie chart angle size for each pressure
        this.config.myData[0] = 360 * propA;
        this.config.myData[1] = 360 * propB;
        this.config.myData[2] = 360 * propC;
        this.config.myData[3] = 360 * propD;
        // =================================================
        propA = 100 * propA;
        propB = 100 * propB;
        propC = 100 * propC;
        propD = 100 - (propA + propB + propC);
        this.proportionA = propA.toFixed(2);
        this.proportionB = propB.toFixed(2);
        this.proportionC = propC.toFixed(2);
        this.proportionD = propD.toFixed(2);
    };
    // step from the controller
    OU.activity.Pressurechart.prototype.step = function ( n ) {
        if (this.config.step!=n) {
            this.config.step = n;
            if (this.config.step==3) {
                this.config.stepCompleted = 3;
                this.changeButton.disabled = false;
                this.changeButton.render();
                this.drawControl();
            }
        }
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.Pressurechart, OU.util.Activity);
