/**
 * GWarming - renders a graph of an iterative model
 *
 * @author Nigel Clarke
 * @author Will Rawes
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.Layer');
OU.require('OU.util.Slider');
OU.require('OU.util.Instruction');
/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.GWarming = function ( data, instance, controller ) {
    OU.activity.GWarming.prototype.canvasView = function () {
        var bH = OU.controlHeight, self = this, i, v;
        this.config = {
            depth:5,
            fps:120 // 40ms=25 fps
        };
        // create Canvas Layers & Contexts
        this.bgLayer = new OU.util.Layer({
            container:this
        });
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        this.PAD = this.h * .045;
        this.yTics = {};
        this.xTics = {};
        this.buttons = [];
        this.extraHtml = [];
        this.running = false;
        this.modelLayer = new OU.util.Layer({
            container:this,
            'id':'model',
            'h':this.h - bH
        });
        this.infoLayer = new OU.util.Layer({
            container:this,
            'id':'info',
            'h':this.h - bH
        });
        this.controlsLayer = new OU.util.Layer({
            container:this,
            'id':'controls',
            hasEvents:true
        });
        this.inRender = false;
        this.dataset = function () {
        };
        this.bestFitOn = this.data.settings.straightLineFit || false;
        this.sliderUpdate = this.data.settings.sliderUpdate!==false;
        this.first = true;
        this.very1st = true;
        this.resizeGraph = true;
        // Set dataset to 1st and render
        this.setGraphSize(bH);
        this.setDataset();
        this.changeDataset(OU.LocalStorage.load("OU.GWarming.dataset") || 0);
        this.setScales(bH, this.dataset);
        v = this.data.settings.audio;
        if (v) {
            this.audioH = "<audio id='_aud' controls='controls' width='" + this.w * 5 + "' height='40'>";
            for (i = v.sources.length; i--;) {
                this.audioH = this.audioH + "<source src='data/" + v.sources[i].src + "' type='" + v.sources[i].type + "' />";
            }
            this.audioH = this.audioH + "</audio>";
            this.audioBox = new OU.util.HtmlBox({
                html:self.audioH,
                container:this
            });
            this.audio = document.getElementById('_aud');
            this.audio.play();
        }
        this.resize();
        this.doRender = true;
        this.renderCycle();
        this.datasetSelector();
    };
    OU.activity.GWarming.prototype.setGraphSize = function ( bH ) {
        this.graphArea = {
            x:this.w * .5,
            y:this.PAD + bH * 1,
            w:this.w * .5 - this.PAD,
            h:this.h - bH - this.PAD * 2 - bH * 2
        };
    };
    OU.activity.GWarming.prototype.resize = function () {
        var i, bH = OU.controlHeight, bX = bH * 3, self = this
            , ds = this.dataset;
        this.bgLayer.resize();
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        this.PAD = this.h * .045;
        this.modelLayer.resize({
            'h':this.h - bH
        });
        this.infoLayer.resize();
        this.controlsLayer.resize();
        this.infoLayer.clear();
        this.controlsLayer.clear();
        //        this.initControls();
        if (this.dsSelector!==undefined) {
            this.dsSelector.resize({
                x:this.x + this.w - 400,
                y:this.y + this.h * .025,
                w:5,
                h:5
            });
        }
        if (this.data.settings.buttons.straightLineFit) {
            this.straightLineButton.resize({
                x:this.w - bX - bH * 2 - 200,
                y:this.h - bH,
                w:bH * 4,
                h:bH
            });
            bX = bX - bH * 4;
        }
        if (this.runB) {
            this.runB.resize({
                x:this.w * 0.5,
                y:this.h - bH * 0.8,
                h:bH * 0.6,
                w:bH * 2
            });
        }
        for (i = this.dataset.variables.length; i--;) {
            this.dataset.variables[i].slider.resize({
                w:this.w * 0.35
            });
        }
        for (i = this.buttons.length; i--;) {
            this.buttons[i].resize({
                x:this.w * 0.5 + (i + 1) * bH * 2.2,
                y:this.h - bH * 0.8,
                h:bH * 0.6,
                w:bH * 2.
            });
            bX = bX - bH * 4;
        }
        this.setGraphSize(bH);
        this.setScales(bH, this.dataset);
        if (this.infoText) {
            this.infoText.resize({
                x:0,
                y:(bH * ds.variables.length + this.PAD * 4),
                w:self.w * 0.41,
                h:1
            });
        }
        if (this.extraText) {
            this.extraText.resize({
                x:self.w * 0.7,
                y:self.h - 60,
                w:self.w * 0.25,
                h:1
            });
        }
        this.resizeGraph = true;
        if (this.audioBox) {
            this.audioBox.resize({
                x:0,
                y:self.h * 0.87,
                w:self.w * 0.9,
                h:self.h * 0.08
            });
        }
        this.doRender = true;
        OU.resizeCloseButton(this);
    };
    OU.activity.GWarming.prototype.remove = function () {
        OU.activity.GWarming.superClass_.remove.call(this); // call the superclass method
        this.doRender = false;
    };
    OU.activity.GWarming.prototype.setDataset = function () {
        var ds;
        this.datasets = [];
        for (var i = 0; i < this.data.datasets.length; i++) {
            ds = this.data.datasets[i];
            ds.container = this;
            this.datasets.push(new this.Dataset(ds));
        }
    };
    OU.activity.GWarming.prototype.datasetSelector = function () {
        var i, ds, n = this.datasets.length, bH = OU.controlheight,
            h = '<form>Dataset:<select onchange="OU.obj.' + instance + '.changeDataset();" id="eqList' + this.instance + '">';
        if (n < 2)return; // don't include if only 1 dataset
        for (i = 0; i < n; i++) {
            ds = this.datasets[i];
            h += '<option value="' + i + '">' + ds.name + '</option>';
        }
        h += '</select></form>';
        this.dsSelector = new OU.util.HtmlBox({
            html:h,
            x:this.x + bH * 5,
            y:this.y + this.PAD * 3,
            w:5,
            h:5,
            unclosable:true,
            handleScrolling:false,
            style:'overflow:visible',
            container:this
        });
    };
    OU.activity.GWarming.prototype.changeDataset = function ( ds ) {
        if (ds===undefined) {
            var eqList = document.getElementById('eqList' + this.instance);
            ds = eqList.value;
        }
        this.dataset = this.datasets[ds || 0];
        OU.LocalStorage.save("OU.GWarming.dataset", ds);
        // Calculate the viewport
        this.xScale = (this.w - this.PAD * 2) / (this.dataset.dx || 1);
        this.yScale = ((this.h * .9) - this.PAD * 2) / (this.dataset.dy || 1);
        this.setYMin = this.dataset.setYMin;
        this.setYMax = this.dataset.setYMax;
        this.yZero = this.dataset.yZero || 0;
        this.controlsLayer.context.clearRect(0, 0, this.w, this.h);
        this.initControls();
        this.resetView();
    };
    OU.activity.GWarming.prototype.resetView = function () {
        var ds = this.dataset, bH = OU.controlHeight;
        ds.reset();
        this.setScales(bH, ds);
        this.doRender = true;
    };
    OU.activity.GWarming.prototype.setScales = function ( bH, ds ) {
        this.xScale = this.graphArea.w / ds.dx;
        this.yScale = this.graphArea.h / ds.dy;
    };
    OU.activity.GWarming.prototype.renderCycle = function () {
        var self = this;
        if (this.doRender) {
            if (this.first!==false || this.resizeGraph) {
                this.render();
                this.renderControls();
            }
            this.first = false;
            this.resizeGraph = false;
        }
        setTimeout(function () {
            self.renderCycle();
        }, 100);
    };
    OU.activity.GWarming.prototype.xPt = function ( v ) {
        return this.xOff + (v - this.xMin) * this.xScale;
    };
    OU.activity.GWarming.prototype.yPt = function ( v ) {
        return this.yOff - (v - this.yMin) * this.yScale;
    };
    OU.activity.GWarming.prototype.render = function () {
        var ctx = this.modelLayer.context, x, y, i, j, lastX, lastY,
            bH = OU.controlHeight, circ = Math.PI * 2, failed = false,
            n, sumXY = 0, sumX = 0, sumX2 = 0, sumY = 0, a, b,
            ds = this.dataset, v,
            r = this.w / 200,
            drawLine = ds.drawLine,
            drawPoints = ds.drawPoints,
            lineCol, pointsCol, lineTitles, k
            ;
        this.xOff = this.graphArea.x;
        this.yOff = this.graphArea.y + this.graphArea.h;
        if (this.first===true)ds.calcPoints();
        if (!this.sliderUpdate)this.first = false;
        this.xS = this.xScale;
        this.yS = this.yScale;
        this.gxMin = this.xPt(ds.x1);
        this.gxMax = this.xPt(ds.x2);
        this.gyMin = this.yPt(ds.y1);
        this.gyMax = this.yPt(ds.y2);
        this.renderFrame();
        ctx.clearRect(0, 0, this.w, this.h - bH);
        ctx.save();
        ctx.lineWidth = 1;
        n = ds.points.length;
        if (ds.plotTime) {
            if (this.gTime < n) {
                this.gTime++;
                n = this.gTime;
            }
        }
        sumXY = 0;
        sumX = 0;
        sumX2 = 0;
        sumY = 0;
        ctx.font = "12pt " + OU.theme.font;
        if (ds.points[0].yVals) {
            pointsCol = (ds.pointsCol || ['#d33', '#33d', '#3d3', '#dd3']);
            lineCol = (ds.lineCol || ['#d33', '#33d', '#3d3', '#dd3']);
            lineTitles = ds.lineTitles || [" ", " ", " ", " ", " ", " ", " ", " "];
            for (k = ds.points[0].yVals.length; k--;) {
                lastX = null;
                ctx.fillStyle = pointsCol[k];
                ctx.strokeStyle = lineCol[k];
                ctx.lineWidth = ds.lineWidth;
                ctx.beginPath();
                ctx.moveTo(this.gxMin + 100 + k * 180, this.gyMax - 15);
                ctx.lineTo(this.gxMin + 130 + k * 180, this.gyMax - 15);
                ctx.fillText(lineTitles[k], this.gxMin + 135 + k * 180, this.gyMax - 15);
                ctx.stroke();
                for (j = n; j--;) {
                    x = ds.points[j].x;
                    y = ds.points[j].yVals[k];
                    if (x===undefined || y===undefined || isNaN(x) || isNaN(y)) {
                        failed = true;
                    }
                    else {
                        if (this.bestFitOn && k===0) {
                            sumXY = sumXY + (x * y);
                            sumX = sumX + x;
                            sumX2 = sumX2 + x * x;
                            sumY = sumY + y;
                        }
                        if (x >= ds.x1 && x <= ds.x2 && y >= ds.y1 && y <= ds.y2) {
                            x = this.xPt(x);
                            y = this.yPt(y);
                            if (j < n - 1) {
                                if (drawLine===true) {
                                    ctx.beginPath();
                                    ctx.moveTo(lastX, lastY);
                                    ctx.lineTo(x, y);
                                    ctx.stroke();
                                }
                            }
                            if (drawPoints===true) {
                                ctx.strokeStyle = '#000';
                                ctx.lineWidth = 1;
                                if (ds.style=='rect') {
                                    ctx.beginPath();
                                    ctx.fillRect(x - r, y - r, r * 2, r * 2);
                                    ctx.stroke();
                                }
                                else {
                                    ctx.beginPath();
                                    ctx.arc(x, y, r, 0, circ, false);
                                    ctx.fill();
                                    ctx.stroke();
                                }
                            }
                            lastX = x;
                            lastY = y;
                        }
                    }
                }
            }
        }
        else {
            pointsCol = (ds.pointsCol || '#d33');
            lineCol = (ds.lineCol || '#d33');
            lastX = null;
            for (j = n; j--;) {
                x = ds.points[j].x;
                y = ds.points[j].y;
                if (x===undefined || y===undefined || isNaN(x) || isNaN(y)) {
                    failed = true;
                }
                else {
                    if (this.bestFitOn) {
                        sumXY = sumXY + (x * y);
                        sumX = sumX + x;
                        sumX2 = sumX2 + x * x;
                        sumY = sumY + y;
                    }
                    ctx.fillStyle = pointsCol;
                    ctx.strokeStyle = lineCol;
                    ctx.lineWidth = ds.lineWidth;
                    if (x >= ds.x1 && x <= ds.x2 && y >= ds.y1 && y <= ds.y2) {
                        x = this.xPt(x);
                        y = this.yPt(y);
                        if (lastX!==undefined) {
                            if (drawLine===true) {
                                ctx.beginPath();
                                ctx.moveTo(lastX, lastY);
                                ctx.lineTo(x, y);
                                ctx.stroke();
                            }
                        }
                        if (drawPoints===true) {
                            ctx.strokeStyle = '#000';
                            ctx.lineWidth = 1;
                            if (ds.style=='rect') {
                                ctx.beginPath();
                                ctx.fillRect(x - r, y - r, r * 2, r * 2);
                                ctx.stroke();
                            }
                            else {
                                ctx.beginPath();
                                ctx.arc(x, y, r, 0, circ, false);
                                ctx.fill();
                                ctx.stroke();
                            }
                        }
                        lastX = x;
                        lastY = y;
                    }
                }
            }
        }
        if (!failed && this.bestFitOn) {
            a = (n * sumXY - sumX * sumY) / (n * sumX2 - sumX * sumX);
            b = (sumY - a * sumX) / n;
            x = ds.x1;
            y = x * a + b;
            if (y < ds.y1) {
                y = ds.y1;
                x = (y - b) / a;
            } else if (y > ds.y2) {
                y = ds.y2;
                x = (y - b) / a;
            }
            x = this.xPt(x);
            y = this.yPt(y);
            ctx.beginPath();
            ctx.moveTo(x, y);
            x = ds.x2;
            y = x * a + b;
            if (y < ds.y1) {
                y = ds.y1;
                x = (y - b) / a;
            } else if (y > ds.y2) {
                y = ds.y2;
                x = (y - b) / a;
            }
            x = this.xPt(x);
            y = this.yPt(y);
            ctx.lineTo(x, y);
            ctx.strokeStyle = ds.fillStyle || '#44f';
            ctx.stroke();
        }
        ctx.restore();
        if (failed) {
            new OU.util.Instruction({
                container:this,
                message:this.data.incompleteMessage || 'Some of the values are missing, please ensure you have completed the relevant table.'
            });
        }
        this.doRender = false;
        if (ds.plotTime) {
            this.recalc = false;
            var self = this;
            if (this.gTime < ds.points.length) {
                if (this.running)setTimeout(function () {
                    self.render();
                    self.renderControls();
                }, (ds.plotTime / ds.points.length) | 100);
                if (ds.extraRun) {
                    if (this.very1st===false)ds.extraRun.fn(this);
                }
            }
            else {
                ds.afterRun(this);
                this.running = false;
            }
        }
        else this.running = false;
        if (this.running) {
            for (i = this.dataset.variables.length; i--;) {
                v = this.dataset.variables[i];
                v.slider.disabled = true;
            }
        }
        else {
            for (i = this.dataset.variables.length; i--;) {
                v = this.dataset.variables[i];
                v.slider.timer = null;
                v.slider.disabled = (v.disabled===true) | false;
            }
        }
    };
    OU.activity.GWarming.prototype.isNum = function ( n ) {
        return !isNaN(parseFloat(n)) && isFinite(n);
    };
    OU.activity.GWarming.prototype.renderFrame = function () {
        var ctx = this.bgLayer.context,
            ds = this.dataset,
            xAxis, yAxis, i, j, val,
            xS = this.xScale,
            yS = this.yScale,
            xOff = this.graphArea.x;
        ctx.gradRect();
        ctx.save();
        ctx.fillStyle = '#fff';
        ctx.fillRect(this.xPt(this.xMin), this.graphArea.y, this.graphArea.w, this.graphArea.h);
        ctx.textAlign = 'right';
        ctx.lineWidth = 0.7;
        yAxis = -ds.x1 * xS + this.graphArea.x;
        yAxis = yAxis < xOff ? xOff : (yAxis > this.graphArea.x + this.graphArea.w ? this.graphArea.x + this.graphArea.w : yAxis);
        ctx.fillStyle = "#000";
        ctx.font = "12pt " + OU.theme.font;
        for (i = this.yTics.min; i <= this.yTics.max; i += this.yTics.interval) {
            var yy = this.yPt(i);
            ctx.beginPath();
            ctx.strokeStyle = "#000";
            ctx.moveTo(this.gxMin - 3, yy);
            ctx.lineTo(this.gxMin, yy);
            ctx.closePath();
            ctx.stroke();
            ctx.lineWidth = 0.7;
            ctx.strokeStyle = (i===0 ? "#000" : "#777");
            ctx.beginPath();
            ctx.moveTo(this.gxMin, yy);
            ctx.lineTo(this.gxMax, yy);
            ctx.closePath();
            ctx.stroke();
            val = ((i * 100) | 0) / 100;
            ctx.fillText(val, this.gxMin - 10, yy + 2);
            if (i < this.yTics.max) {
                for (j = i + this.yTics.subInterval; j < i + this.yTics.interval; j += this.yTics.subInterval) {
                    yy = this.yPt(j);
                    ctx.strokeStyle = "#aaa";
                    ctx.lineWidth = 0.3;
                    ctx.beginPath();
                    ctx.moveTo(this.gxMin, yy);
                    ctx.lineTo(this.gxMax, yy);
                    ctx.closePath();
                    ctx.stroke();
                }
            }
        }
        xAxis = ds.y2 * yS + this.graphArea.x;
        xAxis = xAxis < this.graphArea.y ? this.graphArea.y : (xAxis > this.graphArea.y + this.graphArea.h ? this.graphArea.y + this.graphArea.h : xAxis);
        ctx.textAlign = 'center';
        for (i = this.xTics.min; i <= this.xTics.max; i += this.xTics.interval) {
            var xx = this.xPt(i);
            ctx.beginPath();
            ctx.strokeStyle = "#000";
            ctx.moveTo(xx, this.gyMin);
            ctx.lineTo(xx, this.gyMin + 3);
            ctx.closePath();
            ctx.stroke();
            ctx.lineWidth = 0.7;
            ctx.strokeStyle = (i===0 ? "#000" : "#777");
            ctx.beginPath();
            ctx.moveTo(xx, this.gyMin);
            ctx.lineTo(xx, this.gyMax);
            ctx.closePath();
            ctx.stroke();
            val = ( ds.xAxisLabels ? ds.xAxisLabels[i | 0] : ((i * 100) | 0) / 100 );
            ctx.fillStyle = "#000";
            ctx.fillText(val, xx, this.gyMin + 12);
            if (i < this.xTics.max) {
                for (j = i + this.xTics.subInterval; j < i + this.xTics.interval; j += this.xTics.subInterval) {
                    xx = this.xPt(j);
                    ctx.lineWidth = 0.3;
                    ctx.strokeStyle = "#aaa";
                    ctx.beginPath();
                    ctx.moveTo(xx, this.gyMin);
                    ctx.lineTo(xx, this.gyMax);
                    ctx.closePath();
                    ctx.stroke();
                }
            }
        }
        ctx.fillStyle = "#000";
        ctx.textAlign = 'left';
        ctx.fillText(ds.yAxisTitle, this.gxMin, this.graphArea.y - 15);
        ctx.textAlign = 'center';
        ctx.fillText(ds.xAxisTitle, this.gxMin + this.graphArea.w / 2, this.graphArea.y + this.graphArea.h + 28);
        ctx.stroke();
        ctx.strokeStyle = "#000";
        ctx.lineWidth = 0.7;
        ctx.fillStyle = "#444";
        ctx.beginPath();
        ctx.moveTo(this.gxMin, this.graphArea.y);
        ctx.lineTo(this.gxMin, this.graphArea.y + this.graphArea.h);
        ctx.lineTo(this.gxMin + this.graphArea.w, this.graphArea.y + this.graphArea.h);
        ctx.lineTo(this.gxMin + this.graphArea.w, this.graphArea.y);
        ctx.lineTo(this.gxMin, this.graphArea.y);
        ctx.closePath();
        ctx.stroke();
        ctx.restore();
    };
    OU.activity.GWarming.prototype.calcTics = function ( mn, mx ) {
        ///  ticInterval calculation thanks to M.A.Brown
        var dif = mx - mn, n = Math.floor(Math.log(dif) / Math.log(10.0)),
            a = dif * Math.pow(10.0, -n), ticInterval, dt = 5;
        if (a < 2.0) {
            ticInterval = 0.2;
            dt = 2;
        } else if (a < 5.0) ticInterval = 0.5;
        else ticInterval = 1.0;
        ticInterval = ticInterval * Math.pow(10.0, n);
        return {
            interval:ticInterval,
            min:Math.floor(mn / ticInterval) * ticInterval,
            max:Math.ceil(mx / ticInterval) * ticInterval,
            subInterval:ticInterval / dt
        }
    };
    OU.activity.GWarming.prototype.setVariable = function ( p, v ) {
        var ths = v.parent;
        if (ths.running===true)return;
        if (ths.data.settings.extraFunction)ths.data.settings.extraFunction(ths, v, p);
        v.value = v.min + (v.max - v.min) * p;
        v.slider.sliderPos = (v.value - v.min) / (v.max - v.min);
        ths.renderControls();
        if (!(ths.sliderUpdate===false)) {
            ths.gTime = 0;
            ths.running = true;
            ths.first = true;
            ths.doRender = true;
        }
    };
    OU.activity.GWarming.prototype.initControls = function () {
        var i, v, ctx = this.controlsLayer.context,
            ictx = this.infoLayer.context, bH = OU.controlHeight,
            clickable = this.controlsLayer.events.clickable,
            self = this, bX = bH * 3, ds = this.dataset, y;
        clickable.length = 0;
        for (i = ds.variables.length; i--;) {
            v = ds.variables[i];
            v.parent = this;
            v.sts = this.data.settings;
            v.slider = new OU.util.Slider({
                instance:this.instance + '_var_' + i,
                titlePosition:v.titlePosition,
                valueUnits:v.units,
                title:v.title,
                x:0,
                y:bH * i + this.PAD * 3,
                w:(v.orientation ? 300 : this.graphArea.x - bH * 4),
                h:(v.orientation ? 200 : bH),
                sliderHeight:(v.sts.sliderHeight ? bH * v.sts.sliderHeight : bH / 2),
                drawContainer:false,
                container:this,
                callback:this.setVariable,
                callbackParam:v,
                frames:1,
                fps:5,
                background:{
                    clear:true
                },
                context:ctx,
                orientation:v.orientation || 'horizontal',
                showValue:true,
                range:{
                    min:v.min,
                    max:v.max,
                    nDecimals:v.nDecimals
                },
                sliderPos:(v.init - v.min) / (v.max - v.min)
            });
            if (v.latitude) {
                //Render normal slider but with degrees N/S units
                v.slider.renderHorizontal = function ( sliderPos ) {
                    var sXl = this.x + this.w * .1,
                        sXr = this.x + this.w * .9,
                        sY = this.y + this.h / 2,
                        sH = this.sliderHeight,
                        sW = this.sliderWidth,
                        ctx = this.context, val,
                        mh = sH / 10;
                    // draw scrub
                    ctx.save();
                    ctx.fillStyle = '#CCCCCC';
                    if (this.params.background.color!==undefined)
                        ctx.fillStyle = this.params.background.colour;
                    if (this.params.background.RGB!=undefined) {
                        if (this.params.background.alpha!=undefined)
                            ctx.fillStyle = 'rgba(' + this.params.background.RGB + ',' + this.params.background.alpha + ')';
                        else
                            ctx.fillStyle = 'rgba(' + this.params.background.RGB + ',1)';
                    }
                    if (this.params.drawContainer!==false) {
                        ctx.roundRect(sXl - sW, sY - sH, sXr - sXl + (sW * 2), sH * 2, 3);
                        ctx.fill();
                    }
                    else {
                        ctx.clearRect(sXl - sW, sY - sH, sXr - sXl + (sW * 2), sH * 2);
                    }
                    if (this.disabled)ctx.globalAlpha = 0.5;
                    ctx.lineWidth = 1;
                    ctx.strokeStyle = "#888";
                    ctx.beginPath();
                    ctx.moveTo(sXl + 0.5, sY - (8 * mh));
                    ctx.lineTo(sXl + 0.5, sY - (4 * mh));
                    ctx.moveTo(sXl + 0.5, sY + (8 * mh));
                    ctx.lineTo(sXl + 0.5, sY + (4 * mh));
                    ctx.moveTo(sXr + 0.5, sY - (8 * mh));
                    ctx.lineTo(sXr + 0.5, sY - (4 * mh));
                    ctx.moveTo(sXr + 0.5, sY + (8 * mh));
                    ctx.lineTo(sXr + 0.5, sY + (4 * mh));
                    ctx.moveTo(sXl + 0.5, sY);
                    ctx.lineTo(sXr, sY);
                    ctx.closePath();
                    ctx.stroke();
                    if (this.disabled)
                        ctx.globalAlpha = 0.25;
                    // draw slider
                    if (this.sliderStyle=='circle') {
                        ctx.save();
                        ctx.fillStyle = '#333333';
                        ctx.strokeStyle = "#333333";
                        ctx.beginPath();
                        ctx.arc(sXl + ( this.sliderPos * (sXr - sXl)), sY, this.h / 6, 0, Math.PI * 2, false);
                        ctx.fill();
                        ctx.closePath();
                        ctx.restore();
                        ctx.fillStyle = '#888888';
                        ctx.beginPath();
                        ctx.arc(sXl + ( this.sliderPos * (sXr - sXl)), sY, this.h / 10, 0, Math.PI * 2, false);
                        ctx.fill();
                        ctx.fillStyle = '#ffffff';
                        ctx.beginPath();
                        ctx.arc(sXl + ( this.sliderPos * (sXr - sXl)), sY, this.h / 24, 0, Math.PI * 2, false);
                        ctx.fill();
                    }
                    else {
                        ctx.roundRect(sXl - (sW / 2) + ( this.sliderPos * (sXr - sXl)), sY - sH / 2, sW, sH, sW / 2);
                        ctx.fillStyle = "rgba( 144, 144, 144, 0.8 )";
                        ctx.strokeStyle = "#000";
                        ctx.stroke();
                        ctx.fill();
                    }
                    ctx.restore();
                    if (this.title!==undefined || this.showValue) {
                        ctx.save();
                        ctx.font = (this.h / 2) + 'px ' + OU.theme.font;
                        if (this.title!==undefined) {
                            if (this.titlePosition=='above') {
                                new OU.util.DynText({
                                    context:ctx,
                                    txt:this.title,
                                    notDynamic:true,
                                    fontSize:this.h * .4,
                                    x:this.x + this.w * .05,
                                    y:this.y,
                                    w:this.scrubWidth,
                                    h:this.h / 2,
                                    align:'left',
                                    padding:0
                                });
                            }
                            else {
                                new OU.util.DynText({
                                    context:ctx,
                                    txt:this.title,
                                    notDynamic:true,
                                    fontSize:this.h * .4,
                                    x:this.x,
                                    y:this.y,
                                    w:this.w * .1,
                                    h:this.h
                                });
                            }
                        }
                        if (this.showValue) {
                            val = this.range.min + this.sliderPos * this.range.dx;
                            val = val.nDecimals(this.range.nDecimals===undefined ? 3 : this.range.nDecimals);
                            if (this.valueUnits!==undefined) {
                                this.dispVal = Math.abs(val) + (val < 0 ? "° S" : "° N");
                                if (self.data.settings.extraFunction)self.data.settings.extraFunction(self, this, val);
                                val = this.dispVal;
                            }
                            this.sldrTxt = new OU.util.DynText({
                                context:ctx,
                                notDynamic:true,
                                fontSize:this.h * .4,
                                align:'left',
                                txt:val,
                                x:this.scrubX + this.scrubWidth * 1.05,
                                y:this.y,
                                w:this.w * .3,
                                h:this.h
                            });
                        }
                        ctx.restore();
                    }
                };
                this.setLat0 = v.slider.renderHorizontal;
            }
            if (v.disabled)v.slider.disabled = true;
            else clickable.push(v.slider);
        }
        if (ds.instructions!==undefined) {
            y = (bH * ds.variables.length + this.PAD * 4);
            if (ds.afterRun) {
                if (!this.infoText) {
                    var self = this;
                    this.infoText = new OU.util.HtmlBox({
                        html:"<span style='font-size:13px;'>" + ds.instructions + "</span>",
                        txt:ds.instructions,
                        x:0,
                        y:y,
                        w:self.w * 0.45,
                        h:1,
                        unclosable:true,
                        handleScrolling:false,
                        style:'overflow:visible',
                        container:this
                    });
                    if (ds.extraRun) {
                        this.extraText = new OU.util.HtmlBox({
                            html:" ",
                            x:self.w * 0.74,
                            y:self.h * 0.88,
                            w:self.w * 0.25,
                            h:1,
                            unclosable:true,
                            handleScrolling:false,
                            style:'overflow:visible',
                            container:this
                        });
                    }
                }
            }
            else {
                this.instDT = new OU.util.DynText({
                    txt:ds.instructions,
                    align:'left',
                    x:0,
                    y:y,
                    w:this.graphArea.x - bH,
                    h:this.h - bH - y,
                    context:ictx
                });
            }
        }
        bX = this.w - bH * 11;
        if (this.data.settings.buttons.straightLineFit) {
            this.straightLineButton = new OU.util.ControlButton({
                txt:'Straight Line Fit',
                x:bX,
                y:this.h - bH,
                w:bH * 4,
                h:bH,
                layer:this.controlsLayer,
//                context:ctx,
                onOffIndicator:true,
                onOff:this.bestFitOn,
                onClick:function () {
                    self.toggleStraightLine();
                }
            });
            bX += bH * 4;
//            clickable.push(this.straightLineButton);
        }
        if (this.sliderUpdate===false) {
            this.runB = new OU.util.Button({
                txt:"Run",
                h:bH * 0.7,
                layer:this.controlsLayer,
//                context:ctx,
//                context:ctx,
                onClick:function () {
                    self.gTime = 0;
                    self.running = true;
                    self.first = true;
                    self.render();
                }
            });
//            clickable.push(this.runB);
        }
        if (this.data.settings.extraHtml) {
            if (!this.extraHtml[0]) {
                for (i = this.data.settings.extraHtml.length; i--;) {
                    this.extraHtml[i] = new OU.util.HtmlBox({
                        html:self.data.settings.extraHtml[i].html,
                        x:self.w * self.data.settings.extraHtml[i].xf,
                        y:self.w * self.data.settings.extraHtml[i].yf,
                        w:self.w * self.data.settings.extraHtml[i].wf,
                        h:self.w * self.data.settings.extraHtml[i].hf,
                        unclosable:true,
                        handleScrolling:false,
                        style:'overflow:visible;',
                        container:this
                    });
                }
            }
        }
        bX += (this.runB ? this.runB.w : 0) + 100;
        if (this.data.settings.buttons.extras) {
            for (i = this.data.settings.buttons.extras.length; i--;) {
                this.buttons[i] = new OU.util.Button({
                    txt:this.data.settings.buttons.extras[i].name,
                    h:bH * 0.7,
                    onOffIndicator:this.data.settings.buttons.extras[i].onOff,
                    hasOnOff:this.data.settings.buttons.extras[i].onOff,
                    layer:this.controlsLayer,
//                context:ctx,
                    onClick:function ( t ) {
                        t.params.onOff = !t.params.onOff;
                        t.render();
                        t.action(self);
                    }
                });
                this.buttons[i].onClickParam = this.buttons[i];
                this.buttons[i].action = this.data.settings.buttons.extras[i].action;
                bX += this.buttons[i].w + 5;
//                clickable.push(this.buttons[i]);
            }
        }
        new OU.util.DynText({
            txt:this.dataset.name,
            x:-this.h * .03,
            y:this.PAD,
            w:this.w / 4,
            h:this.PAD * 1.5,
            context:ictx,
            background:{
                col:'#afdffb',
                radius:this.h * .03
            }
        });
        this.resize();
    };
    OU.activity.GWarming.prototype.toggleStraightLine = function () {
        this.bestFitOn = !this.bestFitOn;
        this.straightLineButton.params.onOff = this.bestFitOn;
        this.render();
    };
    OU.activity.GWarming.prototype.renderControls = function () {
        var i;
        this.controlsLayer.clear();
        if (this.data.settings.buttons.straightLineFit)this.straightLineButton.render();
        for (i = this.buttons.length; i--;)this.buttons[i].render();
        for (i = this.dataset.variables.length; i--;)this.dataset.variables[i].slider.render();
        if (this.sliderUpdate===false)this.runB.render();
    };
    OU.activity.GWarming.prototype.Dataset = function ( params ) {
        var t;
        this.name = params.name || 'y=?';
        this.container = params.container;
        this.x1 = params.x1 || 0;
        this.x2 = params.x2 || 10;
        this.startY = params.startY || 10;
        this.setYMin = params.setYMin;
        this.setYMax = params.setYMax;
        this.xAxisTitle = params.xAxisTitle || "";
        this.yAxisTitle = params.yAxisTitle || "";
        this.xAxisLabels = params.xAxisLabels;
        this.yZero = params.yZero;
        this.x2 = params.x2 || 10;
        this.variables = params.variables || {};
        this.instructions = params.instructions;
        this.couplings = params.couplings;
        this.gType = params.gType;
        this.xInterval = params.xInterval;
        this.buttons = [];
        this.extraHTML = [];
        this.afterRun = params.afterRun || null;
        this.extraRun = params.extraRun || null;
        if (params.setMap)params.setMap(params.container);
        this.plotTime = params.plotTime;
        this.drawLine = (params.drawLine===false ? false : true);
        this.lineCol = params.lineCol;
        this.lineWidth = params.lineWidth || 2;
        this.lineTitles = params.lineTitles;
        this.drawPoints = (params.drawPoints===false ? false : true);
        this.pointsCol = params.pointsCol;
        if (this.x1 > this.x2) {
            t = this.x2;
            this.x2 = this.x1;
            this.x1 = t;
        }
        this.reset_x1 = this.x1;
        this.reset_x2 = this.x2;
        this.reset_y1 = this.y1;
        this.reset_y2 = this.y2;
        if (params.colour!==undefined)
            this.colour = params.colour;
        if (params.rgb!==undefined)
            this.rgb = params.rgb;
        OU.activity.GWarming.prototype.Dataset.prototype.calcPoints = function () {
            var i, params = {}, v, y, c = this.container, j;
            for (i = this.variables.length; i--;) {
                v = this.variables[i];
                params[v.varName] = v.value;
            }
            params.x1 = this.x1;
            params.x2 = this.x2;
            params.xInterval = this.xInterval;
            params.startY = this.startY;
            if (c.very1st===true && this.afterRun) {
                this.points = [
                    {
                        x:0,
                        y:0
                    }
                ];
                c.very1st = false;
            }
            else this.points = document.OUGWModel(params, this.couplings, this.gType);
            c.very1st = false;
            this.y1 = 1.e12;
            this.y2 = -this.y1;
            if (this.points[0].yVals) {
                for (i = this.points.length; i--;) {
                    for (j = this.points[i].yVals.length; j--;) {
                        if (c.isNum(c.yZero))this.points[i].yVals[j] -= c.yZero;
                        y = this.points[i].yVals[j];
                        if (y >= this.y2)this.y2 = y;
                        if (y <= this.y1)this.y1 = y;
                    }
                }
            }
            else {
                for (i = this.points.length; i--;) {
                    if (c.isNum(c.yZero))this.points[i].y -= c.yZero;
                    y = this.points[i].y;
                    if (y >= this.y2) this.y2 = y;
                    if (y <= this.y1) this.y1 = y;
                }
            }
            if (c.isNum(c.setYMin)) {
                this.y1 = c.setYMin;
                this.y2 = c.setYMax;
            }
            if (this.y1==this.y2) {
                this.y1 -= 1;
                this.y2 += 1;
            }
            c.yTics = c.calcTics(this.y1, this.y2);
            this.y1 = c.yTics.min;
            this.y2 = c.yTics.max;
            c.xTics = c.calcTics(this.x1, this.x2);
            if (this.xAxisLabels)c.xTics.interval = 1;
            if (this.xInterval) {
                c.xTics.min = this.x1;
                c.xTics.max = this.x2;
                c.xTics.interval = this.xInterval;
                c.xTics.subInterval = 30;
            }
            this.dx = Math.abs(this.x2 - this.x1);
            this.dy = Math.abs(this.y2 - this.y1);
            c.xScale = (c.graphArea.w) / this.dx;
            c.yScale = (c.graphArea.h) / this.dy;
            c.xMin = this.x1;
            c.yMin = this.y1;
        };
        OU.activity.GWarming.prototype.Dataset.prototype.init = function () {
            var i, v;
            for (i = this.variables.length; i--;) {
                v = this.variables[i];
                v.value = v.init;
            }
        };
        OU.activity.GWarming.prototype.Dataset.prototype.reset = function () {
            this.x1 = this.reset_x1;
            this.x2 = this.reset_x2;
            this.y1 = this.reset_y1;
            this.y2 = this.reset_y2;
            this.init();
        };
        this.init();
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.GWarming, OU.util.Activity);
if (!document.OUGWModel)document.OUGWModel = function ( vars, couplings, gType ) {
    var co2line = true;
    //   Reading starting Feb 1990; alterations to call Slingo
    //   This version developed for the Open University in April/May 1996.
    //   and the ability to change the oxygen/nitrogen concentration in
    //   the convection scheme was re-written.
    //                                        Dept of Meteorology
    //                                        k.p.shine@reading.ac.uk
    // and climate processes in general is
    // era", which must of the present study is based around is
    // with a convective adjustment. J.Atmos.Sci., 21, 361-385
    // feedback (via the maintenance of a profile of fixed relative humidity)
    // Manabe S. and R.T.Wetherald, 1967: Thermal equilibrium of the
    // J.Atmos.Sci., 24, 241-259
    // A further classic radiative-convective study, in which
    // Hansen J., D.Johnson, A.Lacis, S.Lebedeff, P. Lee, D.Rind and
    // carbon dioxide, Science, 213, 957-966
    var nlev = 11, nlp1 = nlev + 1, nl4 = nlev * 4, nlt2 = nlev * 2
        , ncld = 8, nlnc = nlev * ncld, maxcld = 4, mcld2 = maxcld * 2
        , mxgas = 3
        , stateGW = 0
        , cld = []
        , cld1 = []
        , cld2 = []
        , curtis = []
        , cltran
        , data = []
        , fluxnt = []
        , fr = []
        , hold = []
        , store = []
        , sumab = []
        , sumpr = []
        , tirflx = []
        , totflx = []
        , tran = []
        , trpfl = []
        , tswflx = []
        , co2m
        , gas = []
        , p = []
        , t = []
        , dp = []
        , tinit
        , ocean, xmas, delta, delta2, water, iwat
        , tsd
        , icld = []
        , icld1 = []
        , icld2 = []
        , ifr = []
        , ich
        , pctime, radmean, convmean
        , pcstore, imxst
        , data2 = []
        , solcon
        , cosmu
        , flxsol = []
        , wc = []
        , em1 = 0.0, em2 = 0.0, em3 = 0.0
        , told = 0.0, acc = 0.000010
        , b1, delt, deltt, swin, stime,
        dyln, rg, rgice, rgnoice, fracice,
        co2, o3, c1, c2, s1, s2, otot, ps,
        plal, trp, crep, fmax, tmax, sbc, ratio, percent
        , i, i50, j, maxit, mn, iclvl, issn, ieq,
        ial, immr, irad, k1, i10, lht, n, istep, nmi, ij1,
        ij2, i1, k2, k3, k4, k, imm, iend, incld, iice, ifin,
        iwrite, iwr
        , tau0
        , tau
        , ta
        , ivol
        , om = ["", " no water vapour feedback", " water vapour feedback activated"]
        , oi = ["", " no ice albedo feedback", " ice albedo feedback activated"]
        , data0 = [
            [0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
            [0.0, 5000, 208.227, 6e-06, 10000, 4.57e-06],
            [0.0, 15000, 210.812, 6e-06, 10000, 7.54e-07],
            [0.0, 25000, 221.395, 1.79528e-05, 10000, 2.83e-07],
            [0.0, 35000, 236.018, 0.000104868, 10000, 1.66e-07],
            [0.0, 45000, 247.579, 0.000362274, 10000, 1.2e-07],
            [0.0, 55000, 257.214, 0.00108677, 10000, 9.24e-08],
            [0.0, 65000, 265.52, 0.00213967, 10000, 7.73e-08],
            [0.0, 75000, 272.848, 0.00374079, 10000, 6.63e-08],
            [0.0, 85000, 279.422, 0.00600486, 10000, 5.86e-08],
            [0.0, 95000, 285.398, 0.00904161, 10000, 5.33e-08],
            [0.0, 100000, 288.196, 0.0108826, 1, 5.33e-08]
        ]
        , ebmCO2 = 350.0
        , ebmLatitude = 0.0
        , ebmMonth = 0
        , volcanoDecay = 0;

    function setGW( n, /*long*/ ivalue ) {
        var i
            , value = ( ivalue) / 1000.0
            , solcon0 = 1368.0;
        switch(n) {
            case 1:
                solcon = solcon0 * value;
                break;
            case 2:
                co2 = value;
                break;
            case 3:
                if (ivalue!=1000) {
                    for (i = 1; i <= nlev; i++) data[3][i] = value * data0[i][3];
                    iwat = 1;
                }
                else iwat = 0;
                break;
            case 4:
                tau0 = 0.13 * value;
                ivol = 0 /* TEST IME NOV96 */;
                break;
            case 5:
                rgnoice = value;
                break;
            case 6:
                fracice = value;
                break;
            case 7:
                cld1[1][1] = value / 100.0;
                break;
            case 8:
                if (iwat==0 && immr > 0)
                    if (value > 0.5) immr = 2;
                    else immr = 1;
                break;
            case 9:
                if (iice > 0)
                    if (value > 0.5) iice = 2;
                    else iice = 1;
                break;
            case 10:
                delta = 1.0 + value * 0.21;
                delta2 = 1.0 / delta;
                break;
            case 11:
                volcanoDecay = ivalue;
                break;
            case 21:
                ebmCO2 = ivalue;
                break;
            case 22:
                ebmLatitude = ivalue;
                break;
            case 23:
                ebmMonth = ivalue;
                break;
        }
        return 0;
    }

    function getGW( n ) {
        var value = 0.0, i;
        switch(n) {
            case 0:
                value = data[2][nlev];
                break;
            case 1:
                value = solcon;
                break;
            case 2:
                value = co2;
                break;
            case 3:
                for (i = 1; i <= nlev; i++) value += data[3][i] * data[4][i];
                value /= 9.8;
                value *= 1.33;
                break;
            case 4:
                value = tau / 0.13;
                break;
            case 5:
                value = rgnoice;
                break;
            case 6:
                value = fracice;
                break;
            case 7:
                value = cld1[1][1];
                break;
            case 8:
                value = immr;
                break;
            case 9:
                value = iice;
                break;
            case 10:
                value = delta;
                break;
            case 11:
                value = totflx[1][1];
                break;
            case 20:
                value = tempAtCO2();
                break;
            case 21:
                value = tempAtLat();
                break;
        }
        return parseInt(value * 1000.0 + 0.5);
    }

    function tempAtCO2() {
        var ic0 = 0
            , iLat = 0
            , x = 0.0
            , T = [
                [-18.85, -11.75, -8.46, -6.44, -5.0],
                [-13.83, -7.06, -4.01, -2.09, -0.75],
                [-6.66, -0.5, 2.25, 4.02, 5.27     ],
                [0.17, 5.7, 8.19, 9.83, 11.0       ],
                [5.43, 10.43, 12.73, 14.26, 15.38  ],
                [11.45, 15.79, 17.88, 19.28, 20.35 ],
                [16.14, 19.98, 21.9, 23.22, 24.25  ],
                [18.58, 22.17, 24.01, 25.28, 26.3  ],
                [18.79, 22.35, 24.19, 25.45, 26.47 ],
                [20.38, 23.1, 24.75, 25.95, 26.82  ],
                [19.32, 21.98, 23.64, 24.83, 25.7  ],
                [17.66, 20.33, 22.0, 23.2, 24.06   ],
                [15.25, 17.94, 19.63, 20.85, 21.71 ],
                [12.26, 15.00, 16.72, 17.96, 18.82 ],
                [8.54, 11.36, 13.13, 14.4, 15.27   ],
                [-0.9, 2.23, 4.14, 5.5, 6.44       ],
                [-12.11, -8.64, -6.54, -5.05, -4.00],
                [-19.97, -16.61, -14.46, -12.85, -11.64]
            ];
        if (ebmCO2 < 245.0 || ebmCO2 > 655.0) return 3.3;
        x = (ebmCO2 - 250.0) / 100.0 - 1.0;
        if (x > 1.0) {
            ic0 = 1;
            x = x - 1.0;
        }
        iLat = parseInt((ebmLatitude + 90.0 ) / 10.0);
        return  -x * (x - 1.0) * (x - 2.0) * T[iLat][ic0] / 6.0
            + (x * x - 1.0) * (x - 2.0) * T[iLat][ic0 + 1] / 2.0
            - x * (x + 1.0) * (x - 2.0) * T[iLat][ic0 + 2] / 2.0
            + x * (x * x - 1.0) * T[iLat][ic0 + 3] / 6.0;
    }

    function tempAtLat() {
        var iL0 = 0, iL1 = 0;
        var x = 0.0;
        var T = [
            [-17.73, -21.61, -24.18, -23.94, -19.66, -10, 0.89, 5.35, 3.23, -2.16, -7.82, -13.12],
            [-14.08, -17.82, -19.73, -18.64, -13.48, -4.22, 5.35, 9.47, 7.71, 2.16, -3.93, -9.43],
            [-9.25, -12.33, -12.86, -9.99, -3.17, 5.23, 12.22, 14.74, 12.62, 7.07, 0.78, -4.71],
            [-2.36, -5.19, -5.34, -2.3, 3.38, 9.88, 15.38, 18.09, 17.33, 13.47, 7.88, 2.21],
            [3.09, 0.85, 1.22, 4.04, 8.63, 13.62, 17.88, 20.25, 19.99, 17.14, 12.49, 7.29],
            [9.34, 8.49, 9.62, 12.22, 15.55, 18.76, 21.29, 22.57, 22.04, 19.65, 15.99, 12.1],
            [15.97, 15.92, 16.9, 18.51, 20.18, 21.69, 22.8, 23.38, 22.95, 21.49, 19.37, 17.25],
            [21.43, 21.1, 21.05, 21.23, 21.54, 21.91, 22.3, 22.69, 22.93, 22.92, 22.59, 22.02],
            [22.11, 21.71, 21.48, 21.45, 21.58, 21.8, 22.11, 22.47, 22.8, 22.99, 22.93, 22.61],
            [23.32, 23.76, 24.17, 24.38, 24.27, 23.88, 23.35, 22.89, 22.63, 22.59, 22.72, 22.98],
            [23.37, 23.95, 24.14, 23.81, 22.96, 21.89, 20.9, 20.36, 20.38, 20.91, 21.72, 22.6],
            [23.23, 23.47, 22.94, 21.68, 20.02, 18.5, 17.58, 17.55, 18.32, 19.68, 21.16, 22.41],
            [21.07, 21.31, 20.69, 19.34, 17.6, 16.04, 15.06, 14.97, 15.72, 17.13, 18.74, 20.14],
            [15.97, 16.87, 17.29, 17.12, 16.38, 15.33, 14.23, 13.36, 12.96, 13.11, 13.78, 14.82],
            [12.09, 13.17, 13.53, 13.16, 12.28, 11.21, 10.15, 9.24, 8.7, 8.68, 9.32, 10.58],
            [11.15, 8.55, 2.8, -2.68, -5.6, -6.81, -7.39, -7.9, -7.35, -4.98, 0.26, 7.57],
            [-1.62, -6.76, -14.4, -20.43, -22.95, -23.96, -24.42, -24.82, -23.92, -20.37, -13.39, -4.42],
            [-10.85, -16.92, -25.82, -31.84, -34.23, -35.08, -35.59, -35.88, -35.24, -31.5, -22.96, -13.62]
        ];
        if (ebmLatitude < -88.0) ebmLatitude = -88.0;
        else if (ebmLatitude > 88.0) ebmLatitude = 88.0;
        x = (90.0 - ebmLatitude) / 10.0;
        iL0 = parseInt(x);
        iL1 = iL0 + 1;
        return (x - iL0) * T[iL1][ebmMonth] + ( iL1 - x) * T[iL0][ebmMonth];
    }

    function statusGW() {
        return stateGW;
    }

    function advanceGW( n ) {
        var t = 0.0, lastStep = istep + (n / 2);
        if (lastStep > imxst) lastStep = imxst;
        while (istep < lastStep /* && stateGW >= 0 */) t = makeStep();
        return parseInt(t * 1000.0 + 0.5);
    }

    function initGW() {
        istep = 0;
        maxit = 0;
        stateGW = 0;
        b1 = 9.80665 * 86400.0 / (1004.0 * 100000.0);
        delt = 2.0;
        deltt = delt * 86400.0;
        imxst = 25000;
        iwrite = 365;
        ocean = 1.e3 * 70.0 * 4000.0;
        mn = nlev - 1;
        ifin = 0;
        for (j = 1; j <= nlev; j++) {
            for (i = 1; i <= 5; i++) {
                data[i][j] = data0[j][i];
            }
        }
        tinit = data[2][nlev];
        tsd = data[2][nlev];
        solcon = 1368.0;
        co2 = 300.0;
        water = 1.0;
        iwat = 0;
        rgice = 0.55;
        rgnoice = 0.087;
        fracice = 0.1;
        tau0 = 0.0;
        ivol = 0;
        ta = 365.0 * 2.0;
        volcanoDecay = 0;
        xmas = 0.0;
        delta = 1.0;
        delta2 = 1.0;
        cld1[1][1] = 0.46;
        cld1[3][1] = 53.0;
        cld1[2][1] = 1.0 - Math.exp(-1.66 * 50.0 * 0.001 * cld1[3][1]);
        icld1[1][1] = 4;
        icld1[2][1] = 4;
        iclvl = 1;
        cosmu = 0.5;
        dyln = 0.5;
        immr = 1;
        iice = 1;
    }

    function precalc1() {
        if (iclvl > 0) {
            cfn(2);
            if (iclvl < 3) k1 = iclvl * 2;
            else if (iclvl==3) k1 = 8;
        }
        else {
            fr[1] = 1.0;
            k1 = 1;
        }
        swin = solcon * cosmu * dyln;
        i10 = 0;
        i50 = 0;
        iwr = 0;
        otot = 0.0;
    }

    function makeStep() {
        if (istep < imxst) istep++;
        else stateGW = -2;
        if (istep==1) precalc1();
        calc();
        tstep();
        connew();
        for (i = 1; i <= nlev; i++) {
            if (Math.abs(totflx[1][i]) > Math.abs(fmax)) fmax = totflx[1][i];
            if (Math.abs(totflx[2][i]) > Math.abs(tmax)) tmax = totflx[2][i];
        }
        radmean = 0.0;
        for (imm = 1; i <= (nlev - 1); i++) radmean = radmean - totflx[2][imm] * data[4][imm];
        radmean = radmean / data[1][nlev];
        convmean = b1 * (totflx[1][nlev] - totflx[1][1]);
        if (istep==1 && Math.abs(totflx[1][1]) > 25.0) {
            stateGW = -1;
            // return data[2][nlev];
        }
        stateGW = 0;
        //   Check for convergence in three ways:
        //               (ii) d/dt of surface temperature close to zero
        //   ALL three must be satisfied. (TOA=top of atmosphere)
        //   For the volcanic aerosol runs (IVOL=1) do not test
        if (ivol==0) {
            if (Math.abs(otot - totflx[1][1]) < 0.001
                && Math.abs(told - data[2][nlev]) < 0.001
                && Math.abs(totflx[1][1]) < 0.01) stateGW = 1;
        }
        else if (istep >= 3650) {
            ivol = 0;
        }
        told = data[2][nlev];
        otot = totflx[1][1];
        return told;
    }

    function calc() {
        i10 = i10 + 1;
        i50 = i50 + 1;
        iwr = iwr + 1;
        if (i50==11 || ivol==1) i50 = 1;
        if (i10==11 || ivol==1) i10 = 1;
        if (iwr==iwrite) iwr = 1;
        stime = (istep - 1) * delt;
        if (volcanoDecay==1) tau = tau0 * Math.exp(-stime / ta);
        else tau = tau0;
        if (i10!=1) return;
        if (immr==2) mmr();
        if (iice==2) {
            fracice = 0.1 - 0.002 * (data[2][nlev] - tinit);
            if (fracice < 0.0) fracice = 0.0;
            rg = (1.0 - fracice) * rgnoice + fracice * rgice;
        }
        else rg = (1.0 - fracice) * rgnoice + fracice * rgice;
        for (i = 1; i <= nlev; i++) {
            nmi = nlev - i + 1;
            for (j = 1; j <= 5; j++) {
                if (j==1 || j==4) data2[j][nmi] = delta * data[j][i];
                else if (j==3 || j==5) data2[j][nmi] = delta2 * data[j][i];
                else data2[j][nmi] = data[j][i];
            }
        }
        for (ij1 = 1; ij1 <= 5; ij1++) {
            for (ij2 = 1; ij2 <= nlev; ij2++) {
                tirflx[ij1][ij2] = 0.0;
                if (ij1 <= 2 && i50==1) {
                    tswflx[ij1][ij2] = 0.0;
                    flxsol[ij2][ij1] = 0.0;
                }
            }
        }
        if (i50==1) {
            if (dyln >= 0.001) {
                for (i = 1; i <= k1; i++) {
                    for (j = 1; j <= nlev; j++) {
                        if (icld2[j][i]==0) wc[j] = 0.0;
                        else wc[j] = 1.5 * cld2[j][i][2] / 8.6;
                    }
                    if (fr[i] < 0.00010) continue;
                    landh();
                    for (i1 = 1; i1 <= nlev; i1++) {
                        nmi = nlev - i1 + 1;
                        tswflx[1][nmi] = tswflx[1][nmi] + flxsol[i1][1] * dyln * fr[i];
                        tswflx[2][nmi] = tswflx[2][nmi] + flxsol[i1][2] * dyln * fr[i];
                    }
                }
                plal = (-tswflx[1][1] + swin) / swin;
            }
        }
        co2m = co2 * 1.0e-6 * (44.0 / 28.964);
        for (k4 = 1; k4 <= nlev; k4++) {
            nmi = nlev + 1 - k4;
            p[nmi] = delta * data[1][k4];
            t[nmi] = data[2][k4];
            dp[nmi] = delta * data[4][k4];
            gas[1][nmi] = delta2 * data[3][k4];
            gas[2][nmi] = delta2 * co2m;
            gas[3][nmi] = delta2 * data[5][k4];
        }
        for (k2 = 1; k2 <= k1; k2++) {
            for (k3 = 1; k3 <= nlev; k3++) {
                if (k1!=1) {
                    icld[k3] = icld2[k3][k2];
                    cld[k3] = cld2[k3][k2][1];
                }
                else {
                    icld[k3] = 0;
                    cld[k3] = 0.0;
                }
                for (k4 = 1; k4 <= 5; k4++) fluxnt[k4][k3] = 0.0;
            }
            allter();
            for (i1 = 1; i1 <= nlev; i1++) {
                nmi = nlev + 1 - i1;
                tirflx[1][i1] = tirflx[1][i1] + fluxnt[4][nmi] * fr[k2];
                tirflx[2][i1] = tirflx[2][i1] + fluxnt[5][nmi] * fr[k2];
            }
        }
        //  rates into total fluxes and heating rates
        for (i = 1; i <= nlev; i++) {
            totflx[1][i] = tswflx[1][i] - tirflx[1][i];
            totflx[2][i] = tswflx[2][i] + tirflx[2][i];
        }
        fmax = 0.0;
        tmax = 0.0;
    }

    function mmr() {
        var rhumidity, t, s, svp2, i;
        for (i = 1; i <= nlev; i++) {
            rhumidity = 0.77 * ((data[1][i] / 1.0e5) - 0.02) / 0.75;
            t = data[2][i];
            if (t <= 253.0)
                s = 21.87456 * (t - 273.16) / (t - 7.66);
            else
                s = 17.26939 * (t - 273.16) / (t - 35.86);
            svp2 = 6.108 * Math.exp(s);
            data[3][i] = 0.622 * rhumidity * svp2 / ((data[1][i] / 100.0) - svp2);
            if (data[3][i] < 6.0e-6) data[3][i] = 6.0e-6;
            if (rhumidity < 0.0) data[3][i] = 6.0e-6;
        }
    }

    function connew() {
        var crit = []
            , temp = []
            , t, f2, rhs, delf, olt, tchange
            , spt, sp, sdtp
            , i, n
            , a, a1 = 0
            , dd = []
            , dp = []
            , ifl = 0;
        if (ifl==0) {
            a = 5.6698e-8 * deltt * 9.80665 / 1004.0;
            a1 = 6.5e-3 * 287.0 / (9.80665 * 2.0);
            for (i = 1; i < nlev; i++) {
                dd[i] = data[4][nlev - i];
                t = data[1][nlev - i] / data[1][nlev + 1 - i];
                dp[i] = Math.log(t);
            }
            ifl = 1;
        }
        for (i = 1; i <= nlev; i++) temp[i] = data[2][nlev + 1 - i];
        tchange = 999.0;
        while (tchange >= 0.00001) {
            sp = 0.0;
            olt = temp[1];
            for (n = 1; n < nlev; n++) {
                for (i = 1; i <= nlev - 1; i++) crit[i] = a1 * (temp[i] + temp[i + 1]) * dp[i];
                if ((temp[n] + crit[n]) <= temp[n + 1]) break;
                for (i = 1; i <= n; i++) temp[i + 1] = temp[i] + crit[i];
            }
            for (i = n + 1; i < nlev; i++) {
                if ((temp[i] + crit[i]) > temp[i + 1]) {
                    temp[i] = (dd[i - 1] * temp[i] + dd[i] * (temp[i + 1] - crit[i])) / (dd[i - 1] + dd[i]);
                    temp[i + 1] = temp[i] + crit[i];
                }
            }
            tchange = Math.abs(olt - temp[1]);
        }
        for (i = 1; i <= nlev; i++) data[2][i] = temp[nlev + 1 - i];
    }

    function tstep() {
        var adam, bash, i;
        if (i50==1) {
            adam = 2.0;
            bash = 0.0;
        }
        else {
            adam = 3.0;
            bash = 1.0;
        }
        for (i = 1; i < nlev; i++) {
            data[2][i] = data[2][i] + delt * 0.5 * (adam * totflx[2][i] - bash * hold[2][i]);
            hold[2][i] = totflx[2][i];
        }
        tsd += delt * 0.5 * 86400.0 * (adam * (totflx[1][1] / ocean) - bash * hold[1][1]);
        hold[1][1] = totflx[1][1] / ocean;
        data[2][nlev] = tsd;
    }

    //   **Radiation Scheme**
    // of Edinburgh 1981). It is a highly parameterised 3 band radiation
    // anything like state-of-the-art!
    // with user-specified emittance.
    //                                                   KPS 26/4/90
    // To make consistent with TRPRAD it is assumed that data is input
    // This is the opposite convention to earlier (than 6.1) versions. The
    // "invert" the arrays                            KPS 1/5/90
    //     SUMAB,SUMPR,CLD,ICLD,TRP,TRPFL,NLEV,MAXGAS)
    //  :              SUMAB,SUMPR,CLD,ICLD,TRP,TRPFL,NLEV,MXGAS)
    function allter() {
        var a, aaa, al, alpha, amo, ap, au, b, beta, c
            , co2, d, h2o, p1, p2, temp, trp, w1, w2, x
            , nmi, i, iii, j, j2, jjj, k, kkk, l, n, nn;
        for (i = 1; i <= 6; i++)
            for (j = 1; j <= nlev; j++)
                for (k = 1; k <= nlev; k++) tran[i][j][k] = 1.0;
        for (i = 1; i <= nlev; i++) {
            nmi = nlev + 1 - i;
            fluxnt[4][i] = 0.0;
            temp = t[nmi];
            h2o = gas[1][nmi];
            co2 = gas[2][nmi];
            store[1][1][i] = (0.180510 * temp - 16.12530) / ((4.8312e-6 * temp - 4.2288e-3) * temp + 0.9999910);
            store[1][2][i] = (((-1.92906e-8 * temp + 1.75525e-5) * temp - 2.15774e-3) * temp - 0.1052150) * temp + 19.26210;
            store[1][3][i] = (((-4.34081e-8 * temp + 5.43817e-5) * temp - 1.742578e-2) * temp + 2.2267910) * temp - 102.2570;
            a = dp[nmi] * h2o / 98.06650;
            b = a * p[nmi];
            c = temp / 263.00;
            d = c * c * c * c;
            store[2][1][i] = a * d;
            store[3][1][i] = b * d;
            w1 = ((5.827506e-7 * temp + 2.61259e-4) * temp - 0.008953380);
            w2 = ((4.20654e-5 * temp - 3.71686e-2) * temp + 10.84860) /
                ((2.82154e-5 * temp - 1.04436e-2) * temp + 0.999945);
            store[2][3][i] = (w1 + h2o * w2 / 0.6220) * b / 1.013e5;
            store[2][4][i] = a * c * c * c;
            store[3][4][i] = store[2][4][i] * p[nmi];
            store[2][5][i] = dp[nmi] * co2 / 98.06650;
            store[3][5][i] = store[2][5][i] * p[nmi];
            amo = gas[3][nmi] * dp[nmi] / (9.806650 * 2.15e-2);
            ap = p[nmi] / 100000.0;
            au = (4.10 * amo) / (1.00 + 9.50 * amo);
            al = (0.84670 * amo) * (1.90 - amo) / (1.0 + 2.0 * amo);
            if (amo > 0.650) alpha = (1.0 - ap) * au + ap * al;
            else {
                if (ap > 0.0150) {
                    beta = Math.sqrt((ap - 0.0150) / 0.2350);
                    if (ap > 0.250) alpha = 0.66670 * (1.750 - ap) * al;
                    else alpha = Math.pow(au, 1.0 - beta) * Math.pow(al, beta);
                }
                else alpha = (1.0850 - 0.0850 * ap) * au;
            }
            store[2][6][i] = 1.6670 * amo * Math.pow(ap, alpha);
        }
        nn = nlev - 1;
        for (k = 1; k <= nn; k++) {
            cltran = 1.0;
            for (j = 1; j <= 6; j++) {
                sumab[j] = 0.0;
                sumpr[j] = 0.0;
            }
            for (l = k; l <= nn; l++) evalTransmittance(k, l);
        }
        for (j = 1; j <= 3; j++) {
            for (k = 1; k <= nlev; k++) {
                for (n = 1; n <= nlev; n++) {
                    if (n==nlev) {
                        if (n==k) curtis[j][k][n] = 1.0;
                        else curtis[j][k][n] = tran[j][k][nlev - 1];
                    }
                    else {
                        if (n==k) curtis[j][k][n] = 1.0 - tran[j][k][k];
                        else if (n==(k - 1)) curtis[j][k][n] = tran[j][k - 1][k - 1] - 1.0;
                        else if (n < k) curtis[j][k][n] = tran[j][n][k - 1] - tran[j][n + 1][k - 1];
                        else curtis[j][k][n] = tran[j][k][n - 1] - tran[j][k][n];
                    }
                }
            }
        }
        for (j = 1; j <= 3; j++) {
            for (k = 1; k <= nlev; k++) {
                nmi = nlev + 1 - k;
                fluxnt[j][nmi] = 0.0;
                for (n = 1; n <= nlev; n++) fluxnt[j][nmi] = curtis[j][k][n] * store[1][j][n] + fluxnt[j][nmi];
                fluxnt[4][nmi] = fluxnt[j][nmi] + fluxnt[4][nmi];
            }
        }
        //   and then they are summed to give the total heating rate;
        fluxnt[5][1] = 0.0;
        for (i = 2; i <= nlev; i++) {
            fluxnt[5][i] = 9.80665 * 86400.0 * (fluxnt[4][i - 1] - fluxnt[4][i]) / (1004.0 * dp[i]);
        }
    }

    function evalTransmittance( k, l ) {
        // evaluates transmiitance between layer k and layer l
        var j, nmi, p1, p2, x, aaa;
        for (j = 1; j <= 6; j++) {
            if (j==2) continue;
            sumab[j] = store[2][j][l] + sumab[j];
            if (j==3 || j==6) continue;
            sumpr[j] = store[3][j][l] + sumpr[j];
        }
        p1 = sumpr[1] / (sumab[1] * 1.e5);
        p2 = (
            ((0.9870010 * p1 + 0.2395750) * p1 + 2.531447e-6) /
                ((0.1276890 * p1 + 0.985242) * p1 + 0.11399));
        x = log10(1.66 * sumab[1] * p2);
        if (x > 1.0) fail(1, x);
        else {
            if (x < -5.1279) tran[1][k][l] = 0.9655 * cltran;
            else tran[1][k][l] =
                (((6.799046e-3 * x + 3.871804e-2) * x - 0.13924380) * x + 0.1474555) * cltran;
        }
        aaa = -1.66 * sumab[3];
        if (aaa < -180.0) tran[3][k][l] = 0.0;
        else tran[3][k][l] = Math.exp((-1.66) * sumab[3]);
        p1 = sumpr[4] / (sumab[4] * 1.e5);
        p2 = ((0.9870010 * p1 + 0.2395750) * p1 + 2.531447e-6) / ((0.1276890 * p1
            + 0.9852420) * p1 + 0.113990);
        x = log10(1.66 * sumab[4] * p2);
        if (x > 1.75) fail(4, x);
        else if (x < -4.250) tran[4][k][l] = 1.0;
        else tran[4][k][l] = ((((6.59638e-4 * x + 6.74892e-3) * x + 0.01357920) * x
                - 0.06289620) * x - 0.2864020) * x + 0.676140;
        p1 = sumpr[5] / (sumab[5] * 1.e5);
        p2 = ((0.98486360 * p1 + 0.2231050) * p1 + 2.697552e-6) /
            ((0.09256111 * p1 + 0.9874070) * p1 + 0.12829350);
        x = 1.66 * sumab[5] * p2;
        tran[5][k][l] = 0.19375 / (0.19735 + Math.pow(x, 0.43));
        x = log10(sumab[6]);
        if (x > 0.5) fail(6, x);
        else if (x < -2.750) tran[6][k][l] = 0.993;
        else tran[6][k][l] = ( (0.3486390 * x - 0.00849660) * x + 0.36114140) /
                ((0.417180 * x + 0.346470) * x + 0.8401930);
        tran[3][k][l] = cltran * tran[3][k][l] * (0.7 + 0.3 * tran[6][k][l]);
        tran[2][k][l] = cltran * tran[4][k][l] * tran[5][k][l];
        nmi = nlev + 1 - l;
        if (icld[nmi]==1) {
            for (j = 1; j <= 3; j++) tran[j][k][l] = (1.0 - cld[nmi]) * tran[j][k][l];
            cltran = cltran * (1.0 - cld[nmi]);
        }
    }

    //   Subroutine to calculate the fractional area for each;
    //   ICLD and CLD which contain the info on the cloud config.;
    //   MODE = 2 is LW and SW;
    function cfn( mode ) {
        var xn1, xn2, xn3
            , i, i1, i2, i3, ia, ib = 2, ic = 7, j;
        for (i = 1; i <= 8; i++) {
            for (j = 1; j <= nlev; j++) {
                cld2[j][i][1] = 0.0;
                cld2[j][i][2] = 0.0;
                icld2[j][i] = 0;
            }
        }
        xn1 = cld1[1][1];
        xn2 = cld1[1][2];
        xn3 = cld1[1][3];
        fr[1] = (1.0 - xn1) * (1.0 - xn2) * (1.0 - xn3);
        fr[2] = (1.0 - xn3) * (1.0 - xn2) * xn1;
        fr[3] = (1.0 - xn3) * xn2 * xn1;
        fr[4] = (1.0 - xn3) * xn2 * (1.0 - xn1);
        fr[5] = xn1 * xn2 * xn3;
        fr[6] = (1.0 - xn1) * xn2 * xn3;
        fr[7] = xn1 * xn3 * (1.0 - xn2);
        fr[8] = (1.0 - xn1) * (1.0 - xn2) * xn3;
        for (ia = 1; ia <= iclvl; ia++) {
            if (iclvl==1) {
                ib = 2;
                ic = 2;
            }
            else if (iclvl==2) {
                if (ia==1) {
                    ib = 2;
                    ic = 3;
                }
                else if (ia==2) {
                    ib = 3;
                    ic = 4;
                }
            }
            else {
                if (ia==1) {
                    ib = 2;
                    ic = 7;
                }
                else if (ia==2) {
                    ib = 3;
                    ic = 6;
                }
                else if (ia==3) {
                    ib = 5;
                    ic = 8;
                }
            }
            for (i = ib; i <= ic; i++) {
                if (ia==1 && (i==4 || i==6)) continue;
                i1 = icld1[1][ia];
                i2 = icld1[2][ia];
                for (i3 = i1; i3 <= i2; i3++) {
                    icld2[i3][i] = 1;
                    if (mode <= 2) cld2[i3][i][1] = cld1[2][ia];
                    if (mode >= 2) cld2[i3][i][2] = cld1[3][ia];
                }
            }
        }
    }

    function fail( iband, x ) {
        //                console.log(" function allter fails: in band "+iband+", abs = "+String(x));
        if (iband > 0) System.exit(0);
    }

    // atmospheres after Lacis, A.A.  and J.E.Hansen,J.Atmos.Sci,1974.;
    // scattering by volcanic aerosols and for variation of the;
    // nitrogen and oxygen content;
    var nfits = 6, nlm1 = nlev - 1;

    function landh() {
        var ref = []
            , tran = []
            , rab = []
            , tab = []
            , rabst = []
            , tabst = []
            , flux = 0.0;
        for (var i = 0; i < nlev + 1; i++) {
            ref[i] = [];
            tran[i] = [];
        }
        var tauloc, a2, omg, u, t, et, xm, ra, rast, ta, tast, rb, sv, tb, dn1, u1, d1 = 0, r1, w1, y1
            , nch = 0, llev, lspec
            , flo3 = []
            , o3inc, reff, rlgh, totrl, totdn, totup
            , surflx, o3tau, sumo3, flxacc
            , dpd = [
                [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0],
                /* ime*/
                [0.0, /*ime*/ 7.e-5, 0.005, 0.041, 0.416, 4.75, 72.5  ],
                [0.0, /*ime*/ 0.647, 0.107, 0.104, 0.073, 0.044, 0.025]
            ]
            , a1 = Math.sqrt(273.0) / (10.0 * 9.80665 * 101300.0)
            , a3 = 9.80665 * 86400.0 / 1004.0
            , a4 = 1.e2 / (9.80665 * 2.144);
        nch = 0;
        for (llev = nlev; llev >= 2; llev--) {
            w1 = data2[3][llev];
            if (wc[llev] > 0.001) {
                sv = svp(data2[2][llev]);
                w1 = 0.622 * sv / (data2[1][llev] - sv);
                if (w1 < 6.0e-6)w1 = 6.0e-6;
            }
            y1 = a1 * w1 * data2[4][llev] * data2[1][llev] / (Math.sqrt(data2[2][llev]));
            flxsol[llev][1] = 0.0;
            for (lspec = 2; lspec <= nfits; lspec++) {
                ref[1][lspec] = rg;
                tran[1][lspec] = 0.0;
                tauloc = wc[llev] + dpd[1][lspec] * y1;
                omg = wc[llev] / tauloc;
                if (wc[llev] > 0.001) {
                    nch = 1;
                    u = Math.sqrt((1.0 - 0.85 * omg) / (1.0 - omg));
                    t = tauloc * Math.sqrt(3.0 * (1.0 - omg) * (1.0 - 0.85 * omg));
                    if (t <= 75.0) {
                        et = Math.exp(t);
                        a2 = et * Math.pow(u + 1.0, 2.0) - Math.pow(u - 1.0, 2.0) / et;
                        ref[llev][lspec] = ((u + 1.0) * (u - 1.0) * (et - 1.0 / et)) / a2;
                        tran[llev][lspec] = 4.0 * u / a2;
                    }
                    else {
                        ref[llev][lspec] = (u - 1.0) / (u + 1.0);
                        tran[llev][lspec] = 0.0;
                    }
                }
                else {
                    ref[llev][lspec] = 0.0;
                    if (nch==0) {
                        xm = 35.0 / (Math.sqrt(1224.0 * Math.pow(cosmu, 2.0) + 1.0));
                        tran[llev][lspec] = Math.exp(-xm * tauloc);
                    }
                    else {
                        tran[llev][lspec] = Math.exp(-1.66 * tauloc);
                    }
                }
            }
        }
        flxsol[1][1] = 0.0;
        for (lspec = 2; lspec <= nfits; lspec++) {
            ra = ref[nlev][lspec];
            rast = ra;
            ta = tran[nlev][lspec];
            tast = ta;
            for (llev = nlm1; llev >= 1; llev--) {
                rb = ref[llev][lspec];
                tb = tran[llev][lspec];
                dn1 = 1.0 / (1.0 - rast * rb);
                rab[llev] = ra + ta * rb * tast * dn1;
                tab[llev] = ta * tb * dn1;
                rabst[llev] = rb + tb * rast * tb * dn1;
                tabst[llev] = tb * tast * dn1;
                ra = rab[llev];
                ta = tab[llev];
                rast = rabst[llev];
                tast = tabst[llev];
            }
            r1 = ref[1][lspec];
            for (llev = 1; llev <= nlev; llev++) {
                if (llev < nlm1) d1 = tab[llev + 1] / (1.0 - r1 * rabst[llev + 1]);
                else if (llev==nlm1)   d1 = tran[nlev][lspec] / (1.0 - r1 * ref[nlev][lspec]);
                else if (llev==nlev)   d1 = 1.0;
                u1 = d1 * r1;
                flux = dpd[2][lspec] * solcon * cosmu * (d1 - u1);
                flxsol[llev][1] = flxsol[llev][1] + flux;
                if (llev!=nlev) {
                    r1 = ref[llev + 1][lspec] + r1 * Math.pow(tran[llev + 1][lspec], 2.0) / (1.0 - ref[llev + 1][lspec] * r1);
                }
            }
        }
        nch = 0;
        o3tau = 0.0;
        sumo3 = 0.0;
        for (llev = 2; llev <= nlev; llev++) {
            flo3[llev] = 0.0;
            sumo3 = sumo3 + a4 * data2[4][llev] * data2[5][llev];
            if (wc[llev] > 0.001) {
                nch = 1;
                o3tau = o3tau + wc[llev];
            }
        }
        flo3[1] = 0.0;
        if (nch==0) {
            rlgh = (data2[1][1] / 1.e5) * 0.219 / (1.0 + 0.816 * cosmu);
            rlgh = rlgh + 0.27 * tau;
            reff = rlgh + (1.0 - rlgh) * (1.0 - 0.144) * rg / (1.0 - 0.144 * rg);
        }
        else {
            rlgh = 0.13 * o3tau / (1.0 + 0.13 * o3tau);
            reff = rlgh + (1.0 - rlgh) * (1.0 - rlgh) * rg / (1.0 - rlgh * rg);
        }
        xm = 35.0 / Math.sqrt(1224.0 * cosmu * cosmu + 1.0);
        totdn = 0.0;
        totup = (xm + 1.9) * sumo3;
        for (llev = nlev; llev >= 1; llev--) {
            if (wc[llev] > 0.001) flo3[llev] = flo3[llev + 1];
            else {
                flo3[llev] = solcon * cosmu * ((1.0 - o3ab(totdn)) - reff * (1.0 - o3ab(totup)));
                o3inc = a4 * data2[4][llev] * data2[5][llev];
                totdn = totdn + xm * o3inc;
                totup = totup - 1.9 * o3inc;
            }
        }
        if (nch==0) {
            totrl = (data2[1][1] / 1.e5) * 0.28 / (1.0 + 6.43 * cosmu);
            totrl = totrl + 0.27 * tau;
            reff = (1.0 - totrl) * (1.0 - rg) / (1.0 - 0.0685 * rg);
        }
        else {
            totrl = 0.13 * o3tau / (1.0 + 0.13 * o3tau);
            reff = (1.0 - totrl) * (1.0 - rg) / (1.0 - totrl * rg);
        }
        surflx = solcon * cosmu * (0.647 - o3ab(xm * sumo3)) * reff;
        flxacc = surflx;
        for (llev = 2; llev <= nlev; llev++) {
            flxacc = flxacc + (flo3[llev] - flo3[llev - 1]);
            flxsol[llev][1] = flxsol[llev][1] + flxacc;
        }
        flxsol[1][1] = flxsol[1][1] + surflx;
        for (llev = 2; llev <= nlev; llev++) {
            flxsol[llev][2] = a3 * (flxsol[llev][1] - flxsol[llev - 1][1]) / data2[4][llev];
        }
        flxsol[1][2] = 0.0;
    }

    // Calculates the saturation vapour pressure given T, using Murray's;
    // SVP returned in Pascals;
    function svp( t ) {
        if (t >= 253.0)
            return 611.0 * Math.exp(17.27 * (t - 273.16) / (t - 35.86));
        else
            return 611.0 * Math.exp(21.87 * (t - 273.16) / (t - 7.66));
    }

    //  Lacis and Hansen's ozone absorption functions
    function o3ab( x ) {
        return (0.02118 * x) / (1.0 + x * (0.042 + 0.000323 * x))
            + ((1.082 * x) / Math.pow((1.0 + 138.6 * x), 0.805) )
            + ((0.0658 * x) / (1.0 + Math.pow(103.6 * x, 3.0)) );
    }

    // Some machines don't seem to be able to cope with real underflow.
    function expk( x ) {
        if (x < -180.0) return 0.0;
        else return Math.exp(x);
    }

    function log10( x ) {
        return Math.log(x) / Math.log(10.);
    }

    function setArrays() {
        var i, j, k;
        for (i = 0; i < nlev + 1; i++) {
            cld[i] = 0.0;
            p[i] = 0.0;
            t[i] = 0.0;
            dp[i] = 0.0;
            icld[i] = 0.0;
            wc[i] = 0.0;//*/
            cld2[i] = [];
            icld2[i] = [];
            for (j = 0; j < 8 + 1; j++) {
                icld2[i][j] = 0.0;
                cld2[i][j] = [];
                for (k = 0; k < 2 + 1; k++)cld2[i][j][k] = 0.0;
            }
            flxsol[i] = [];
            for (j = 0; j < 2 + 1; j++)flxsol[i][j] = 0.0;
        }
        for (i = 0; i < 3 + 1; i++) {
            cld1[i] = [];
            for (j = 0; j < 3 + 1; j++) {
                cld1[i][j] = 0.0;
            }//*/
            trpfl[i] = 0.0;
            curtis[i] = [];
            for (j = 0; j < nlev + 1; j++) {
                curtis[i][j] = [];
                for (k = 0; k < nlev + 1; k++)curtis[i][j][k] = 0.0;
            }
            store[i] = [];
            for (j = 0; j < 6 + 1; j++) {
                store[i][j] = [];
                for (k = 0; k < nlev + 1; k++)store[i][j][k] = 0.0;
            }
        }
        for (i = 0; i < 5 + 1; i++) {
            data[i] = [];
            fluxnt[i] = [];
            tirflx[i] = [];
            data2[i] = [];
            for (j = 0; j < nlev + 1; j++) {
                data[i][j] = 0.0;
                fluxnt[i][j] = 0.0;
                tirflx[i][j] = 0.0;
                data2[i][j] = 0.0;
            }
        }
        for (i = 0; i < 8 + 1; i++) {
            fr[i] = 0.0;
            ifr[i] = [];
            for (j = 0; j < 3 + 1; j++) {
                ifr[i][j] = 0.0;
            }
        }
        for (i = 0; i < 2 + 1; i++) {
            hold[i] = [];
            totflx[i] = [];
            tswflx[i] = [];
            for (j = 0; j < nlev + 1; j++) {
                hold[i][j] = 0.0;
                totflx[i][j] = 0.0;
                tswflx[i][j] = 0.0;
            }
            icld1[i] = [];
            for (j = 0; j < 3 + 1; j++) {
                icld1[i][j] = 0.0;
            }
        }
        for (i = 0; i < 6 + 1; i++) {
            sumab[i] = 0.0;
            sumpr[i] = 0.0;
            tran[i] = [];
            for (j = 0; j < nlev + 1; j++) {
                tran[i][j] = [];
                for (k = 0; k < nlev + 1; k++)tran[i][j][k] = 0.0;
            }
        }
        for (i = 0; i < mxgas + 1; i++) {
            gas[i] = [];
            for (j = 0; j < nlev + 1; j++) {
                gas[i][j] = 0.0;
            }//*/
        }
    }

    setArrays();
    var N = [];
    initGW();
    switch(gType) {
        case 'm':
            setGW(1, vars.sc * 1000. / 1368.);
            setGW(2, vars.co2 * 1000);
            setGW(3, (vars.h2o / 30.55) * 1000);
            setGW(4, vars.ac * 1000);
            setGW(5, vars.al * 1000);
            setGW(6, vars.icsn * 1000);
            setGW(7, vars.cc * 100000);
            for (var i = 0; i < couplings.length; i++)setGW(couplings[i * 2], couplings[i * 2 + 1]);
            var N = [];
            var temp = (getGW(0) / 1000.0 - 273.12);
            N.push({
                x:0,
                y:temp,
                h2o:getGW(3) / 1000.,
                aer:getGW(4) / 1000.,
                isn:getGW(6) / 1000.
            });
            for (var ii = 1; ii < 41; ii++) {
                advanceGW(174);
                temp = (getGW(0) / 1000.0 - 273.12);
                N.push({
                    x:ii / 2,
                    y:temp,
                    h2o:getGW(3) / 1000.,
                    aer:getGW(4) / 1000.,
                    isn:getGW(6) / 1000.
                });
            }
            break;
        case 'l':
            if (co2line) {
                var lat = -85, ii, ys;
                for (var ii = 18; ii--;) {
                    ys = [];
                    setGW(21, vars.co2);
                    setGW(22, lat);
                    ys.push(getGW(20) / 1000.);
                    setGW(21, 377);
                    setGW(22, lat);
                    ys.push(getGW(20) / 1000.);
                    N.push({
                        x:-lat,
                        y:ys[0],
                        yVals:ys
                    });
                    lat += 10;
                }
            }
            else {
                var ii, ys;
                for (ii = 13; ii--;) {
                    ys = [];
                    setGW(22, vars.lat);
                    setGW(23, ii % 12);
                    ys.push(getGW(21) / 1000.);
                    setGW(22, 55);
                    setGW(23, ii % 12);
                    ys.push(getGW(21) / 1000.);
                    N.push({
                        x:ii,
                        y:ys[0],
                        yVals:ys
                    });
                }
            }
            break;
        case 's':
            var ii, ys;
            for (ii = 12; ii--;) {
                ys = [];
                setGW(22, vars.lat);
                setGW(23, (ii % 12) / 1.);
                ys.push(getGW(21) / 1000.);
                setGW(22, 55);
                setGW(23, (ii % 12) / 1.);
                ys.push(getGW(21) / 1000.);
                N.push({
                    x:ii * 1.,
                    y:ys[0],
                    yVals:ys
                });
            }
            break;
    }
    return N;
};
