/**
 * @fileOverview OU.activity.Landscape - Data model for a landscape Water model
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.HtmlBox');
OU.require('OU.util.Slider');
OU.require('OU.util.varControlBank');
OU.require('OU.util.Layer');
OU.require('OU.util.VarHist');
OU.ICON_NONE = 0;
OU.ICON_SEA = 1;
OU.ICON_TREE = 2;
OU.ICON_HUT = 3;
/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.Landscape = function ( data, instance, controller ) {
    OU.activity.Landscape.prototype.canvasView = function () {
        var bH = OU.controlHeight;
        this.config = {
            depth:5,
            fps:40 // 40ms = 25 fps
        };
        // create Canvas Layers & Contexts
        this.bgLayer = new OU.util.Layer({
            container:this
        });
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        this.window = {
            x:0,
            y:0,
            w:this.w,
            h:this.h - bH
        };
        this.gridSpaces = OU.LocalStorage.load("OU.landscape.quality") || 60; // Quality Value = number of intersection to render on the model
        this.PAD = this.h * .045;
        this.inRender = false;
        this.equation = function () {
        };
        this.controls = [];
        this.varLayer = new OU.util.Layer({
            container:this,
            'id':'varHist'
        });
        this.modelLayer = new OU.util.Layer({
            container:this,
            'id':'model',
            'h':this.h - bH,
            'hasEvents':true,
            pinch:this.pinch,
            pinchMe:this
        });
        this.modelLayer.events.clickable.push(this);
        this.modelLayer.context.translate(this.window.x, this.window.y + this.window.h);
        this.modelLayer.context.lineWidth = 1;
        this.controlsLayer = new OU.util.Layer({
            container:this,
            'y':this.h - bH,
            'h':bH,
            'id':'controls',
            hasEvents:true
        });
        this.initControls();
        this.icons = new this.Icons();
        this.midDrag = false;
        this.renderRange = {
            minX:0,
            maxX:0,
            minY:0,
            maxY:0
        };
        // Set equation to 1st and render
        this.setEquations();
        this.changeEquation(OU.LocalStorage.load("OU.landscape.eq") || 0);
        this.equationSelector();
    };
    OU.activity.Landscape.prototype.resize = function () {
        OU.activity.Landscape.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight;
        // create Canvas Layers & Contexts
        this.bgLayer.resize();
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        this.bgLabels();
        this.window = {
            x:0,
            y:0,
            w:this.w,
            h:this.h - bH
        };
        this.modelLayer.resize({
            'h':this.h - bH
        });
        this.varLayer.resize({
        });
        this.modelLayer.context.translate(this.window.x, this.window.y + this.window.h);
        this.controlsLayer.resize({
            'y':this.h - bH,
            'h':bH
        });
        this.zoomSlider.resize({
            x:bH * 6,
            y:0,
            w:(this.w - (bH * 6)) / 2,
            h:bH
        });
        this.startCycleButton.resize({
            x:this.w - bH * 4.5,
            y:0,
            w:bH * 2,
            h:bH
        });
        this.stopCycleButton.resize({
            x:this.w - bH * 2.5,
            y:0,
            w:bH * 2,
            h:bH
        });
        this.controlBank.resize({
            layerOffset:this.h - bH
        });
        this.controlBank.render();
        this.resetButton.render();
        this.startCycleButton.render();
        this.stopCycleButton.render();
        this.popPerc.resize({
            x:this.w * .6,
            y:bH / 2,
            w:this.w * .2 - 10,
            h:bH * 1.5
        });
        this.forestPerc.resize({
            x:this.w * .8,
            y:bH / 2,
            w:this.w * .2 - 10,
            h:bH * 1.5
        });
        this.render();
    };
    OU.activity.Landscape.prototype.setEquations = function () {
        this.equations = [];
        for (var i = 0; i < this.data.equations.length; i++) {
            this.equations.push(new this.Equation(this.data.equations[i]));
        }
    };
    OU.activity.Landscape.prototype.equationSelector = function () {
        if (this.equations.length < 2)
            return; // don't include if only 1 equation
        var eq, h = '<form>Landscape: <select onchange="OU.obj.' + instance + '.changeEquation();" id="eqList' + this.instance + '">';
        for (eq = 0; eq < this.equations.length; eq++) {
            h += '<option value="' + eq + '">' + this.equations[eq].name + '</option>';
        }
        h += '</select></form>';
        this.eqSelector = new OU.util.HtmlBox({
            container:this,
            html:h,
            x:this.x + 0,
            y:this.y + this.h * .125,
            w:5,
            h:5,
            unclosable:true,
            handleScrolling:false,
            style:'overflow:visible'
        });
    };
    OU.activity.Landscape.prototype.cycleModel = function () {
        var self = this, cell,
            v = {}, x = 0, z = 0, j,
            pop = 0, forest = 0, total = 0,
            eq = this.equation,
            eqV = eq.variables,
            xS = eq.x1,
            xE = eq.x2,
            zS = eq.z1,
            zE = eq.z2,
            zD = eq.dz / this.gridSpaces,
            xD = eq.dx / this.gridSpaces;
        for (j = eqV.length; j--;) {
            if (eqV[j].val!==undefined)
                v[eqV[j].name] = eqV[j].val;
            else
                v[eqV[j].name] = eqV[j].init;
        }
        for (var x1 = xS; x1 <= xE; x1 += xD) {
            z = 0;
            for (var z1 = zS; z1 <= zE; z1 += zD) {
                cell = this.modelY[x][z];
                total++;
                if (cell.icon==OU.ICON_TREE) {
                    this.growTrees(x, z);
                    forest++;
                }
                else if (cell.icon==OU.ICON_HUT) {
                    this.expandVillage(x, z);
                    cell.villageLevel = cell.villageLevel - 0.1;
                    if (cell.villageLevel <= 0) {
                        if (!this.surrounding(x, z, OU.ICON_TREE)) {
                            cell.icon = OU.ICON_NONE;
                            cell.villageLevel = -30;
                        }
                    }
                    pop++;
                }
                else {
                    cell.treeLevel = cell.treeLevel + Math.random() * 0.08;
                }
                if (cell.icon!=OU.ICON_SEA) {
                    if (cell.treeLevel >= 1)
                        cell.icon = OU.ICON_TREE;
                    if (cell.treeLevel <= 0 && cell.villageLevel >= 1)
                        cell.icon = OU.ICON_HUT;
                }
                z++;
            }
            x++;
        }
        this.cycleDay++;
        this.popPerc.update(((1000 * pop / total) | 0) / 10);
        this.forestPerc.update(((1000 * forest / total) | 0) / 10);
        this.render();
        if (this.cycleOn && this.cycleDay < 300) {
            setTimeout(function () {
                self.cycleModel();
            }, 40);
        }
    };
    OU.activity.Landscape.prototype.growTrees = function ( x, z ) {
        this.viral(x, z, function ( cell ) {
            cell.treeLevel = cell.treeLevel + 0.08;
            if (cell.treeLevel >= 1)
                cell.treeLevel = 1;
        });
    };
    OU.activity.Landscape.prototype.expandVillage = function ( x, z ) {
        this.viral(x, z, function ( cell ) {
            if (cell.icon!=OU.ICON_HUT) {
                cell.villageLevel = cell.villageLevel + 0.4;
                cell.treeLevel = cell.treeLevel - 0.15;
            }
        });
    };
    OU.activity.Landscape.prototype.viral = function ( x, z, f ) {
        if (x > 0) {
            f(this.modelY[x - 1][z]);
            if (z > 0)
                f(this.modelY[x - 1][z - 1]);
            if (this.modelY[x - 1][z + 1]!==undefined)
                f(this.modelY[x - 1][z + 1]);
        }
        if (z > 0)
            f(this.modelY[x][z - 1]);
        if (this.modelY[x][z + 1]!==undefined)
            f(this.modelY[x][z + 1]);
        if (this.modelY[x + 1]!==undefined) {
            f(this.modelY[x + 1][z]);
            if (z > 0)
                f(this.modelY[x + 1][z - 1]);
            if (this.modelY[x + 1][z + 1]!==undefined)
                f(this.modelY[x + 1][z + 1]);
        }
    };
    OU.activity.Landscape.prototype.surrounding = function ( x, z, t ) {
        if (x > 0) {
            if (this.modelY[x - 1][z].icon==t)
                return true;
            if (z > 0)
                if (this.modelY[x - 1][z - 1].icon==t)
                    return true;
            if (this.modelY[x - 1][z + 1]!==undefined)
                if (this.modelY[x - 1][z + 1].icon==t)
                    return true;
        }
        if (z > 0)
            if (this.modelY[x][z - 1].icon==t)
                return true;
        if (this.modelY[x][z + 1]!==undefined)
            if (this.modelY[x][z + 1].icon==t)
                return true;
        if (this.modelY[x + 1]!==undefined) {
            if (this.modelY[x + 1][z].icon==t)
                return true;
            if (z > 0)
                if (this.modelY[x + 1][z - 1].icon==t)
                    return true;
            if (this.modelY[x + 1][z + 1]!==undefined)
                if (this.modelY[x + 1][z + 1].icon==t)
                    return true;
        }
        return false;
    };
    OU.activity.Landscape.prototype.calcModel = function () {
        var v = {}, x = 0, z = 0, j, cell,
            eq = this.equation,
            eqV = eq.variables,
            pop = 0, forest = 0, total = 0, bH = OU.controlHeight,
            xS = eq.x1,
            xE = eq.x2,
            zS = eq.z1,
            zE = eq.z2,
            zD = eq.dz / this.gridSpaces,
            xD = eq.dx / this.gridSpaces;
        this.modelY = new Array();
        this.cycleDay = 0;
        for (j = eqV.length; j--;) {
            if (eqV[j].val!==undefined)
                v[eqV[j].name] = eqV[j].val;
            else
                v[eqV[j].name] = eqV[j].init;
        }
        for (var x1 = xS; x1 <= xE; x1 += xD) {
            this.modelY[x] = new Array();
            z = 0;
            for (var z1 = zS; z1 <= zE; z1 += zD) {
                v.x = x1;
                v.z = z1;
                cell = this.modelY[x][z] = {};
                total++;
                cell.y = eq.calcY(v);
                if (cell.y <= eq.seaLevel) {
                    cell.y = eq.seaLevel;
                    cell.icon = OU.ICON_SEA;
                }
                else {
                    cell.treeLevel = 0;
                    cell.villageLevel = 0;
                    cell.icon = OU.ICON_NONE;
                    if (cell.y > eq.seaLevel + .5) {
                        if (Math.random() < 0.3) {
                            cell.icon = OU.ICON_TREE;
                            cell.treeLevel = 1;
                            forest++;
                        }
                        else if (Math.random() < 0.3) {
                            cell.icon = OU.ICON_HUT;
                            cell.villageLevel = 1;
                            pop++;
                        }
                    }
                }
                z++;
            }
            x++;
        }
        this.popPerc = new OU.util.VarHist({
            context:this.varLayer.context,
            title:'Population %',
            x:this.w * .6,
            y:bH / 2,
            w:this.w * .2 - 10,
            h:bH * 1.5,
            min:0,
            max:100,
            val:((1000 * pop / total) | 0) / 10
        });
        this.forestPerc = new OU.util.VarHist({
            context:this.varLayer.context,
            title:'Forest %',
            x:this.w * .8,
            y:bH / 2,
            w:this.w * .2 - 10,
            h:bH * 1.5,
            min:0,
            max:100,
            val:((1000 * forest / total) | 0) / 10
        });
    };
    OU.activity.Landscape.prototype.changeEquation = function ( eqNum ) {
        if (eqNum===undefined) {
            var eqList = document.getElementById('eqList' + this.instance);
            eqNum = eqList.value;
        }
        this.equation = this.equations[eqNum || 0];
        var bH = OU.controlHeight,
            eq = this.equation;
        OU.LocalStorage.save("OU.landscape.eq", eqNum);
        this.calcModel();
        this.zoom = 0.25;
        if (this.controls.length > 0)
            this.controls[0].closeControls();
        var clickable = this.controlsLayer.events.clickable;
        clickable.length = 0;
        clickable.push(this.zoomSlider);
        clickable.push(this.qualitySlider);
        clickable.push(this.resetButton);
        clickable.push(this.restartCycleButton);
        clickable.push(this.startCycleButton);
        clickable.push(this.stopCycleButton);
        if (this.controlBank)
            this.controlBank.close();
        this.controlBank = new OU.util.varControlBank({
            x:bH * 2,
            y:0,
            w:this.w - (bH * 2),
            h:bH,
            layerOffset:this.h - bH,
            vars:eq.variables,
            valInButton:true,
            layer:this.controlsLayer,
            clickable:clickable,
            callback:this.varChange,
            parent:this
        });
        this.controlsWidth = bH * 2 + this.controlBank.w;
        this.calcModel();
        this.resetView();
        this.bgLabels();
    };
    OU.activity.Landscape.prototype.varChange = function () {
        this.parent.calcModel();
        this.parent.render();
    };
    OU.activity.Landscape.prototype.isHit = function ( x, y, pressed ) {
        var damper = 0.25;
        if (pressed && !this.midDrag) {
            this.dragStart = {
                'x':x,
                'y':y
            };
            this.midDrag = true;
        }
        else if (pressed && this.midDrag) {
            this.rotation.y = this.rotation.y - (x - this.dragStart.x) * damper;
            //            this.rotation.x = this.rotation.x - (y-this.dragStart.y)*damper;
            var r2d = Math.PI / 180;
            this.rotCX = Math.cos(this.rotation.x * r2d);
            this.rotSX = Math.sin(this.rotation.x * r2d);
            this.rotCY = Math.cos(this.rotation.y * r2d);
            this.rotSY = Math.sin(this.rotation.y * r2d);
            this.dragStart = {
                'x':x,
                'y':y
            };
            this.render();
        }
        else {
            this.midDrag = false;
        }
    };
    OU.activity.Landscape.prototype.spin = function () {
        var self = this;
        this.rotation.y += 0.5;
        this.render();
        setTimeout(function () {
            self.spin();
        }, 20);
    };
    OU.activity.Landscape.prototype.resetView = function () {
        var r2d = Math.PI / 180, i, v, sW, tooSmall = true,
            numVars = this.equation.variables.length,
            bgctx = this.bgLayer.context, bH = OU.controlHeight,
            self = this;
        this.rotation = {
            'x':330,
            'y':30
        };
        for (i = numVars; i--;) {
            v = this.equation.variables[i];
            v.val = v.init;
        }
        this.rotCX = Math.cos(this.rotation.x * r2d);
        this.rotSX = Math.sin(this.rotation.x * r2d);
        this.rotCY = Math.cos(this.rotation.y * r2d);
        this.rotSY = Math.sin(this.rotation.y * r2d);
        this.render();
        sW = (this.w - this.controlsWidth) / 2;
        this.zoomSlider.x = this.controlsWidth;
        this.zoomSlider.w = sW;
        this.zoomSlider.scrubX = this.zoomSlider.x + sW * .1;
        this.zoomSlider.scrubWidth = sW * .8;
        bgctx.clearRect(bH * 2, this.h - bH, this.w - bH * 4, bH);
        bgctx.font = 'bold ' + bH * .3 + 'px ' + OU.theme.font;
        bgctx.fillText('Zoom', this.zoomSlider.x + this.zoomSlider.w / 2, this.h - bH * .25);
        if (this.renderRange.minX < (this.window.w * .1)
            || this.renderRange.maxX > (this.window.w * .9)
            || this.renderRange.minY < (this.window.h * .1)
            || this.renderRange.maxY > (this.window.h * .9)) {
            tooSmall = false;
        }
        if (tooSmall) {
            this.zoom += 0.05;
            setTimeout(function () {
                self.resetView()
            }, 1);
        }
        this.controlBank.render();
        this.zoomSlider.render(this.zoom);
    };
    OU.activity.Landscape.prototype.zoomIn = function () {
        this.zoom += 0.2;
        this.render();
    };
    OU.activity.Landscape.prototype.zoomOut = function () {
        this.zoom -= 0.2;
        this.render();
    };
    OU.activity.Landscape.prototype.pinch = function ( e, s, x, y, dx, dy, me ) {
        var nz = ((e - s) / me.h) + me.zoom;
        nz = nz > 1 ? 1 : (nz < 0 ? 0 : nz);
        me.zoom = nz;
        me.render();
    };
    OU.activity.Landscape.prototype.setZoom = function ( zoom, graph ) {
        graph.zoom = zoom;
        graph.zoomSlider.render(zoom);
        graph.render();
    };
    OU.activity.Landscape.prototype.render = function () {
        var bH = OU.controlHeight;
        this.modelLayer.context.clearRect(0, 0, this.w, -this.h - bH); // clear background
        // establish the offset to lock the view central
        this.viewXOffset = 0;
        this.viewYOffset = 0;
        var topPt = new this.Point(
            this.equation.x1 + (this.equation.x2 - this.equation.x1) / 2,
            this.equation.y1 + (this.equation.y2 - this.equation.y1) / 2,
            this.equation.z1 + (this.equation.z2 - this.equation.z1) / 2,
            this
        );
        this.viewXOffset = this.window.x + (this.window.w / 2) - topPt.in2d.x;
        this.viewYOffset = (-this.window.h / 2) - topPt.in2d.y;
        // initial render range data to center screen
        this.renderRange = {
            minX:this.window.w,
            maxX:0,
            minY:this.window.h,
            maxY:0
        };
        // Render elements
        this.renderLandscape();
        this.zoomSlider.render();
    };
    OU.activity.Landscape.prototype.renderFrame = function () {
        var ctx = this.modelLayer.context;
        ctx.save();
        ctx.beginPath();
        ctx.strokeStyle = "#666";
        ctx.fillStyle = "#444";
        // Axis
        this.moveTo(0, this.equation.y1, 0);
        this.lineTo(0, this.equation.y2, 0);
        this.moveTo(this.equation.x1, 0, 0);
        this.lineTo(this.equation.x2, 0, 0);
        this.moveTo(0, 0, this.equation.z1);
        this.lineTo(0, 0, this.equation.z2);
        ctx.stroke();
        for (var x = this.equation.x1; x <= this.equation.x2; x += this.equation.dx / 10) {
            var xVal = ((100 * x) | 0) / 100;
            this.text({
                txt:xVal,
                x:x,
                y:0,
                z:0,
                w:100,
                h:20,
                align:'left'
            });
        }
        for (var y = this.equation.y1; y <= this.equation.y2; y += this.equation.dy / 10) {
            var yVal = ((100 * y) | 0) / 100;
            this.text({
                txt:yVal,
                x:0,
                y:y,
                z:0,
                w:100,
                h:20,
                align:'left'
            });
        }
        for (var z = this.equation.z1; z <= this.equation.z2; z += this.equation.dz / 10) {
            var zVal = ((100 * z) | 0) / 100;
            this.text({
                txt:zVal,
                x:0,
                y:0,
                z:z,
                w:100,
                h:20,
                align:'left'
            });
        }
        ctx.font = '24px bold';
        this.text({
            txt:'X',
            x:this.equation.x2 * 1.1,
            y:0,
            z:0,
            w:100,
            h:40,
            align:'left',
            fontWeight:'bold'
        });
        this.text({
            txt:'Y',
            x:0,
            y:this.equation.y2 * 1.1,
            z:0,
            w:100,
            h:40,
            align:'left',
            fontWeight:'bold'
        });
        this.text({
            txt:'Z',
            x:0,
            y:0,
            z:this.equation.z2 * 1.1,
            w:100,
            h:40,
            align:'left',
            fontWeight:'bold'
        });
        ctx.restore();
    };
    OU.activity.Landscape.prototype.renderLandscape = function () {
        var ctx = this.modelLayer.context, bH = OU.controlHeight,
            pt = new Array(), x = 0, z = 0, eq = this.equation, gs = this.gridSpaces,
            x1, z1, xD = -1, zD = -1, startZ, xL, zL, pt1, pt2, pt3, pt4, cell,
            iX, iY, iW, iH;
        for (x1 = eq.x1; x1 <= eq.x2; x1 += eq.dx / gs) {
            pt[x] = new Array();
            z = 0;
            for (z1 = eq.z1; z1 <= eq.z2; z1 += eq.dz / gs) {
                pt[x][z] = new this.Point(x1, this.modelY[x][z].y, z1, this);
                z++;
            }
            x++;
        }
        // Always render from the back first - so determine the farthest corner and start there
        x = pt.length - 1;
        if (pt[0][0].in3d.z > pt[pt.length - 1][0].in3d.z) {
            x = 0;
            xD = 1;
        }
        startZ = pt[0].length - 1;
        if (pt[0][0].in3d.z > pt[0][pt[0].length - 1].in3d.z) {
            startZ = 0;
            zD = 1;
        }
        for (xL = 0; xL < pt.length - 1; xL++) {
            z = startZ;
            for (zL = 0; zL < pt[0].length - 1; zL++) {
                cell = this.modelY[x][z];
                pt1 = pt[x][z];
                pt2 = pt[x][z + zD];
                pt3 = pt[x + xD][z + zD];
                pt4 = pt[x + xD][z];
                ctx.beginPath();
                ctx.moveTo(pt1.in2d.x, pt1.in2d.y);
                ctx.lineTo(pt2.in2d.x, pt2.in2d.y);
                ctx.lineTo(pt3.in2d.x, pt3.in2d.y);
                ctx.lineTo(pt4.in2d.x, pt4.in2d.y);
                ctx.closePath();
                if (pt1.original.y > eq.seaLevel) {
                    ctx.strokeStyle = ctx.fillStyle = '#090';
                }
                else {
                    ctx.strokeStyle = ctx.fillStyle = '#00f';
                }
                ctx.fill();
                ctx.stroke();
                if (pt1.in2d.x < pt4.in2d.x) {
                    iX = pt2.in2d.x;
                    iY = pt1.in2d.y;
                }
                else {
                    iX = pt4.in2d.x;
                    iY = pt1.in2d.y;
                }
                if (Math.abs(pt2.in2d.x - pt4.in2d.x) > Math.abs(pt1.in2d.x - pt2.in2d.x)) {
                    iW = Math.abs(pt2.in2d.x - pt4.in2d.x);
                }
                else {
                    iW = Math.abs(pt1.in2d.x - pt3.in2d.x);
                }
                if (Math.abs(pt1.in2d.y - pt3.in2d.y) > Math.abs(pt2.in2d.y - pt4.in2d.y)) {
                    iH = Math.abs(pt1.in2d.y - pt3.in2d.y);
                }
                else {
                    iH = Math.abs(pt2.in2d.y - pt4.in2d.y);
                }
                this.icons.render({
                    parent:this,
                    type:cell.icon,
                    x:iX + iW / 2,
                    y:iY + iH / 2,
                    w:iW,
                    h:iH
                });
                z += zD;
            }
            x += xD;
        }
        ctx.save();
        ctx.font = '20px ' + OU.theme.font;
        ctx.textAlign = 'center';
        ctx.fillStyle = '#666';
        ctx.fillText('Day ' + this.cycleDay, this.w - bH * 2.5, -bH);
        ctx.restore();
        this.varLayer.clear();
        this.popPerc.render();
        this.forestPerc.render();
        /*
         this.icons.render({
         parent: this,
         type: OU.ICON_HUT,
         x:100,
         y:-100,
         w:20,
         h:20
         });
         */
    };
    OU.activity.Landscape.prototype.initControls = function () {
        var self = this, ctx = this.controlsLayer.context, bH = OU.controlHeight;
        // initialise the sliders
        this.zoomSlider = new OU.util.Slider({
            container:this,
            instance:'zoom' + this.instance,
            x:bH * 6,
            y:0,
            w:(this.w - (bH * 6)) / 2,
            h:bH,
            sliderHeight:bH / 2,
            drawContainer:false,
            callback:this.setZoom,
            callbackParam:this,
            frames:4,
            background:{
                clear:true
            },
            context:ctx
        });
        this.resetButton = new OU.util.ControlButton({
            txt:'Reset',
            x:0,
            y:0,
            w:bH * 2,
            h:bH,
            layer:this.controlsLayer,
            onClick:function () {
                self.resetView();
            }
        });
        this.restartCycleButton = new OU.util.ControlButton({
            txt:'Restart',
            x:this.w - bH * 6.5,
            y:0,
            w:bH * 2,
            h:bH,
            layer:this.controlsLayer,
            onClick:function () {
                self.cycleOn = false;
                self.calcModel();
                self.render();
            }
        });
        this.startCycleButton = new OU.util.ControlButton({
            txt:'Play',
            x:this.w - bH * 4.5,
            y:0,
            w:bH * 2,
            h:bH,
            layer:this.controlsLayer,
            onClick:function () {
                self.cycleOn = true;
                self.cycleModel();
            }
        });
        this.stopCycleButton = new OU.util.ControlButton({
            txt:'Stop',
            x:this.w - bH * 2.5,
            y:0,
            w:bH * 2,
            h:bH,
            layer:this.controlsLayer,
            onClick:function () {
                self.cycleOn = false;
            }
        });
        this.bgLabels();
    };
    OU.activity.Landscape.prototype.bgLabels = function () {
        var bH = OU.controlHeight,
            bgctx = this.bgLayer.context;
        bgctx.clearRect(0, this.h - bH, this.w, bH);
        bgctx.beginPath();
        bgctx.moveTo(0, this.h - bH - 1);
        bgctx.lineTo(this.w, this.h - bH - 1);
        bgctx.strokeStyle = '#000';
        bgctx.stroke();
        bgctx.textAlign = 'right';
        bgctx.font = 'bold ' + ((bH / 2) | 0) + 'px ' + OU.theme.font;
        bgctx.fillStyle = '#666';
        new OU.util.DynText({
            txt:this.equation.name,
            x:-this.h * .03,
            y:this.PAD,
            w:this.w / 4,
            h:this.h * .06,
            context:bgctx,
            background:{
                col:'#afdffb',
                radius:this.h * .03
            }
        });
        bgctx.textAlign = 'center';
    };
    OU.activity.Landscape.prototype.moveTo = function ( x, y, z ) {
        var pt = new this.Point(x, y, z, this);
        this.modelLayer.context.moveTo(pt.in2d.x, pt.in2d.y);
    };
    OU.activity.Landscape.prototype.lineTo = function ( x, y, z ) {
        var pt = new this.Point(x, y, z, this);
        this.modelLayer.context.lineTo(pt.in2d.x, pt.in2d.y);
    };
    OU.activity.Landscape.prototype.text = function ( params ) {
        var pt = new this.Point(params.x, params.y, params.z, this);
        this.modelLayer.context.fillText(params.txt, pt.in2d.x, pt.in2d.y);
    };
    OU.activity.Landscape.prototype.Point = function ( x, y, z, graph ) {
        this.original = {
            'x':x,
            'y':y,
            'z':z
        };
        // rotate about Y axis
        var nz = (z * graph.rotCY) - (x * graph.rotSY);
        x = (z * graph.rotSY) + (x * graph.rotCY);
        z = nz;
        // rotate about X axis
        var ny = (y * graph.rotCX) - (z * graph.rotSX);
        z = (y * graph.rotSX) + (z * graph.rotCX);
        y = ny;
        this.in3d = {
            'x':x,
            'y':y,
            'z':z
        };
        // Note 3D to 2D projection is performed following the method
        // described at http://en.wikipedia.org/wiki/3D_projection
        var eq = graph.equation;
        var dz = z - (eq.z1 - ((eq.z2 - eq.z1) * 60 * (1 - graph.zoom)));
        var rz = graph.window.h;
        var newX = (x * 15) / (dz) * rz + graph.viewXOffset;
        var newY = -((y * 15) / (dz) * rz) + graph.viewYOffset;
        this.in2d = { // Point in 2d relative to the window in pixels
            'x':newX,
            'y':newY
        };
        newY = -newY;
        var rr = graph.renderRange;
        if (newX > rr.maxX)
            rr.maxX = newX;
        if (newX < rr.minX)
            rr.minX = newX;
        if (newY > rr.maxY)
            rr.maxY = newY;
        if (newY < rr.minY)
            rr.minY = newY;
    };
    OU.activity.Landscape.prototype.Equation = function ( params ) {
        //TODO: ensure x2>x1, y2>y1, z2>z1 - if not, swap round values
        this.name = params.name || 'y=?';
        this.x1 = params.x1 || 0;
        this.x2 = params.x2 || 10;
        this.y1 = params.y1 || 0;
        this.y2 = params.y2 || 10;
        this.z1 = params.z1 || 0;
        this.z2 = params.z2 || 10;
        this.reset_x1 = this.x1;
        this.reset_x2 = this.x2;
        this.reset_y1 = this.y1;
        this.reset_y2 = this.y2;
        this.reset_z1 = this.z1;
        this.reset_z2 = this.z2;
        this.seaLevel = params.seaLevel || 0;
        this.variables = params.variables || [];
        this.calcY = params.calcY;
        if (params.colour!==undefined)
            this.colour = params.colour;
        if (params.rgb!==undefined)
            this.rgb = params.rgb;
        OU.activity.Landscape.prototype.Equation.prototype.deltas = function () {
            this.dx = Math.abs(this.x2 - this.x1);
            this.dy = Math.abs(this.y2 - this.y1);
            this.dz = Math.abs(this.z2 - this.z1);
        };
        this.deltas();
    };
    OU.activity.Landscape.prototype.Icons = function () {
        OU.activity.Landscape.prototype.Icons.prototype.render = function ( p ) {
            switch(p.type) {
                default:
                case OU.ICON_NONE:
                    break;
                case OU.ICON_TREE:
                    this.tree(p);
                    break;
                case OU.ICON_HUT:
                    this.hut(p);
                    break;
            }
        };
        OU.activity.Landscape.prototype.Icons.prototype.tree = function ( p ) {
            var parent = p.parent, ctx = parent.modelLayer.context,
                x = p.x, y = p.y, w = p.w, h = p.h;
            ctx.save();
            ctx.beginPath();
            ctx.lineWidth = 0.5;
            ctx.moveTo(x - w * .15, y);
            ctx.lineTo(x + w * .15, y);
            ctx.lineTo(x, y - h);
            ctx.fillStyle = '#8b3534';
            ctx.fill();
            ctx.beginPath();
            ctx.moveTo(x + w * .2, y - h);
            ctx.bezierCurveTo(x, y, x - w, y - h * 2, x, y - h * 1.5);
            ctx.bezierCurveTo(x, y - h * 3, x + w, y - h / 2, x + w * .2, y - h / 2);
            ctx.fillStyle = '#3a793e';
            ctx.strokeStyle = '#56eb5f';
            ctx.fill();
            ctx.stroke();
            ctx.restore();
        };
        OU.activity.Landscape.prototype.Icons.prototype.hut = function ( p ) {
            var parent = p.parent, ctx = parent.modelLayer.context,
                x = p.x, y = p.y, w = p.w, h = p.h;
            ctx.save();
            ctx.beginPath();
            ctx.lineWidth = 2;
            ctx.moveTo(x - w * .3, y);
            ctx.lineTo(x - w * .3, y - h / 2);
            ctx.lineTo(x + w * .3, y - h / 2);
            ctx.lineTo(x + w * .3, y);
            ctx.bezierCurveTo(x + w * .3, y + h / 4, x - w * .3, y + h / 4, x - w * .3, y);
            ctx.fillStyle = '#9b9551';
            ctx.fill();
            ctx.beginPath();
            ctx.moveTo(x - w * .4, y - h / 2);
            ctx.bezierCurveTo(x - w * .4, y - h * .25, x + w * .4, y - h * .25, x + w * .4, y - h / 2);
            ctx.strokeStyle = '#49440b';
            ctx.stroke();
            ctx.lineTo(x, y - h);
            ctx.fillStyle = '#f6eb6b';
            ctx.fill();
            ctx.restore();
        };
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.Landscape, OU.util.Activity);
