/**
 * @fileOverview LayerTileZoom - "Tiled" image viewer - with Layers of images
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.DynText');
OU.require('OU.util.Slider');
OU.require('OU.util.NavButtons');
OU.require('OU.util.Layer');
OU.require('OU.util.TileViewer');
OU.require('OU.util.ImageLoader');
/** 
 * @class
 * @extends OU.util.Activity
 */
OU.activity.LayerTileZoom = function(data,instance,controller) {

    OU.activity.LayerTileZoom.prototype.canvasView = function() {

        var bH=OU.controlHeight;

        this.bgLayer = new OU.util.Layer({
            container:this,
            id:'bg'
        });
        this.bgLayer.context.gradRect();

        if(this.controller!==undefined)
            this.dataDir=this.controller.activities[this.instance].data;
        else
            this.dataDir='data/';

        this.depth=0;
        this.imageLayers = [];
        this.imageLayer = this.imageLayers[this.depth] = new OU.util.Layer({
            container:this,
            id:'canvas'+this.instance,
            h: this.h-bH,
            hasEvents:true,
            pinch: this.pinch,
            pinchMe: this,
            zIndex: OU.CONTROL_LEVEL-1
        });
        this.controlsLayer = new OU.util.Layer({
            container:this,
            id: 'oucontrols'+this.instance,
            y: this.h-bH,
            h: bH,
            hasEvents:true,
            zIndex: OU.CONTROL_LEVEL
        });

        this.start();

    };
    OU.activity.LayerTileZoom.prototype.resize = function() {
         OU.activity.LayerTileZoom.superClass_.resize.call(this); // call the parent class resize
        var bH=OU.controlHeight,i,
        clickable = this.controlsLayer.events.clickable,
        ctx = this.controlsLayer.context,
        self=this,layer=this.currentImage;

        this.bgLayer.resize();
        this.bgLayer.context.gradRect();

        for(i=this.imageLayers.length; i--;) {
            this.imageLayers[i].resize({
                h: this.h-bH
            });
        }
        this.controlsLayer.resize({
            y: this.h-bH,
            h: bH
        });

        this.resetLayer(this.data.layers.images[0]);
        this.initLayer(this.data.layers.images[0]);

        if(this.tileView!=null) {
            this.tileView.w = {
                w:this.w,
                h:this.h-bH
            };
        }
        else {
            this.imageLayer.context.drawImage(layer.image,layer.x,layer.y,layer.w,layer.h); // render image
        }
        clickable.length=0;

        // Add nav buttons
        this.navButtons = new OU.util.NavButtons({
            x:this.w/2,
            y:0,
            w:bH,
            h:bH,
            bw:bH,
            pad:bH*.1,
            context:ctx,
            leftFunc:function() {
                self.prevLayer();
            },
            rightOn: false
        });
        clickable.push(this.navButtons);

        this.zoomSlider.resize({
            x:this.w/2 + bH,
            y:0,
            w:this.w/2 - bH,
            h:bH
        });
        clickable.push(this.zoomSlider);
        this.renderControls();
        if(this.tileView!=null)
            this.tileView.render();
    };
    OU.activity.LayerTileZoom.prototype.start = function() {
        var bH = OU.controlHeight, layer=this.data.layers.images;

        this.initLayer(layer[0]);
        this.tileView = new OU.util.TileViewer({
            imageBase: layer,
            zoomLevels: layer[0].zoomLevels,
            window: {
                w:this.w,
                h:this.h-bH
            },
            context: this.imageLayer.context,
            container: this,
            useGoogleMapsFileConvention:this.data.GoogleFileConvention,
            zoomCallback: this.setSlider,
            markerCallback: this.downLayer,
            crop: layer[0].crop,
            fitScreen:true,
            requireDouble:true
        });
        this.tileView.measureOn = false;
        this.imageLayer.events.clickable.push(this.tileView);
        this.currentImage=this.data.layers.images[0];
        this.initControls();
    };
    OU.activity.LayerTileZoom.prototype.resetLayer = function(layer) {
        layer.w = layer.origW;
        layer.h = layer.origH;
        layer.origW = undefined;
        layer.origH = undefined;
        if(layer.layers!==undefined) {
            for(var i=0; i<layer.layers.length; i++) {
                layer.layers[i].parentLayer = layer;
                this.initLayer(layer.layers[i]);
            }
        }
    };
    OU.activity.LayerTileZoom.prototype.initLayer = function(layer) { // Initialise the layer's image size - recurse through the full data set'
        // Resize the image to fit canvas

        var recurse=false,bH=OU.controlHeight;
        if(layer.origW===undefined) { // if this is the first init, set up orig values and switch on recursion
            layer.origW = layer.w;
            layer.origH = layer.h;
            recurse=true;
        }
        else {
            layer.w = layer.origW;
            layer.h = layer.origH;
        }
        if(layer.fileName!==undefined) {
            // calc metrics
            var minWsF = this.w/layer.w;
            var minHsF = (this.h-bH)/layer.h;
            var e100sF; // effective 100% scaleFactor
            if(minWsF<minHsF)
                e100sF = minWsF;
            else
                e100sF = minHsF;

            e100sF = e100sF>1?1:e100sF; // ensure initial setting is not greater than 100%

            layer.w = layer.origW*e100sF;
            layer.h = layer.origH*e100sF;
            layer.scaleFactor = layer.w/layer.origW;
            layer.x = (this.w-layer.w)/2;
            layer.y = ((this.h-bH)-layer.h)/2;
        }
        if(layer.layers!==undefined && recurse) {
            for(var i=0; i<layer.layers.length; i++) {
                layer.layers[i].parentLayer = layer;
                this.initLayer(layer.layers[i]);
            }
        }
    };
    OU.activity.LayerTileZoom.prototype.initControls = function() {
        var ctx = this.controlsLayer.context,bH=OU.controlHeight,self=this,
        clickable = this.controlsLayer.events.clickable;
        clickable.length=0;

        // Add nav buttons
        this.navButtons = new OU.util.NavButtons({
            x:this.w/2,
            y:0,
            w:bH,
            h:bH,
            bw:bH,
            pad:bH*.1,
            context:ctx,
            leftFunc:function() {
                self.prevLayer();
            },
            rightOn: false
        });
        clickable.push(this.navButtons);

        this.zoomSlider = new OU.util.Slider({
            container:this,
            sliderPos: 0,
            instance: 'zoom'+this.instance,
            x:this.w/2 + bH,
            y:0,
            w:this.w/2 - bH,
            h:bH,
            sliderHeight: bH/2,
            drawContainer:false,
            callback: this.setZoom,
            callbackParam: this,
            background: {
                clear:true
            },
            context: ctx
        });
        clickable.push(this.zoomSlider);
        this.zoomSlider.render();
        this.layerTitle();
    };
    OU.activity.LayerTileZoom.prototype.renderControls = function() {
        this.controlsLayer.context.clearRect(0,0,this.w,OU.controlHeight);
        this.layerTitle();
        if(this.currentImage.parentLayer!==undefined)
            this.navButtons.leftOn=true;
        else
            this.navButtons.leftOn=false;
        this.navButtons.render();
        if(this.zoomSlider.disabled==false)
            this.zoomSlider.render();
    };
    OU.activity.LayerTileZoom.prototype.biggestRadius = function(params) {
        var x = params.x || this.w/2,
        y = params.y || this.h/2,
        w = params.w || this.w,
        h = params.h || this.h,
        bigDist = OU.distance2D({
            x2: x,
            y2: y,
            x1: 0,
            y1: 0
        }),
        bL = OU.distance2D({
            x2: x,
            y2: y,
            x1: 0,
            y1: h
        }),
        bR = OU.distance2D({
            x2: x,
            y2: y,
            x1: w,
            y1: h
        }),
        tR = OU.distance2D({
            x2: x,
            y2: y,
            x1: w,
            y1: 0
        });
        if(bL>bigDist)
            bigDist=bL;
        if(bR>bigDist)
            bigDist=bR;
        if(tR>bigDist)
            bigDist=tR;
        return bigDist*1.1;
    };
    OU.activity.LayerTileZoom.prototype.prevLayer = function() {
        this.currentImage=this.currentImage.parentLayer;
        this.imageLayer.remove();
        this.depth--;
        this.imageLayer = this.imageLayers[this.depth];
        this.imageLayer.canvas.style.zIndex = OU.CONTROL_LEVEL-1;
        this.tileView = this.imageLayer.tileView;
        this.tileView.render();
        this.zoomSlider.disabled=false;
        this.zoomSlider.sliderPos=this.tileView.s;
        this.renderControls();
    };
    OU.activity.LayerTileZoom.prototype.holeOut = function() {
        if(this.holeRadius>this.bigRadius) {
            this.imageLayer.canvas.style.zIndex=OU.DATA_LEVEL+this.depth;
            this.imageLayer.tileView = this.tileView;
            this.imageLayer = this.tmpImageLayer;
            this.depth++;
            this.imageLayers[this.depth]=this.imageLayer;
            this.tileView = this.tmpTileView || null;
            this.imageLayer.canvas.style.zIndex = OU.CONTROL_LEVEL-1;
            this.renderControls();
            this.inTrans=false;
            return;
        }
        var ctx = this.imageLayer.context,self=this;
        this.holeRadius = this.holeRadius*1.3;

        ctx.beginPath();
        ctx.arc(this.holeX,this.holeY,this.holeRadius,0,Math.PI * 2,false);
        ctx.closePath();
        ctx.fill();
        ctx.save();
        ctx.globalCompositeOperation = 'destination-out';
        if(this.holeRadius>10) {
            ctx.beginPath();
            ctx.arc(this.holeX,this.holeY,this.holeRadius*.9,0,Math.PI * 2,false);
            ctx.closePath();
        }
        ctx.fill();
        ctx.restore();
        setTimeout(function() {
            self.holeOut()
        },40);
    };
    OU.activity.LayerTileZoom.prototype.downLayer = function(layer) {
        var bH=OU.controlHeight,ctx,viewer=this.container;

        if(viewer.inTrans)
            return;
        viewer.inTrans=true;
        var events = viewer.imageLayer.events;
        events.pressed=false;
        events.touched=false;
        viewer.zoomSlider.halted=true;

        viewer.holeX = events.x;
        viewer.holeY = events.y;

        viewer.bigRadius = viewer.biggestRadius({
            x:events.x,
            y:events.y
        });
        viewer.holeRadius=5;

        viewer.tmpImageLayer = new OU.util.Layer({
            container:viewer,
            id:'canvas',
            h: viewer.h-bH,
            hasEvents:true,
            pinch: viewer.pinch,
            pinchMe: viewer,
            crop: layer.crop,
            zIndex: OU.CONTROL_LEVEL-2
        });
        ctx = viewer.tmpImageLayer.context;

        viewer.tileView.stop();

        layer.parentLayer = viewer.currentImage;
        viewer.currentImage=layer;

        if(layer.fileName===undefined) { // another tileview layer
            viewer.zoomSlider.disabled=false;
            viewer.tmpTileView = new OU.util.TileViewer({
                imageBase: layer,
                zoomLevels: layer.zoomLevels,
                window: {
                    w:viewer.w,
                    h:viewer.h-bH
                },
                context: ctx,
                useGoogleMapsFileConvention:viewer.data.GoogleFileConvention,
                zoomCallback: viewer.setSlider,
                fitScreen:true,
                requireDouble:true
            });
            viewer.tmpImageLayer.events.clickable.push(viewer.tmpTileView);
        }
        else { // assume image
            viewer.zoomSlider.disabled=true;
            layer.image = new Image();
            layer.image.src = viewer.dataDir+layer.fileName;
            layer.image.onload =function() {
                ctx.drawImage(layer.image,layer.x,layer.y,layer.w,layer.h); // render image
            }
        }
        //        viewer.renderControls();
        viewer.holeOut();
    };
    OU.activity.LayerTileZoom.prototype.layerTitle = function() {
        var layer = this.currentImage,bH=OU.controlHeight;
        new OU.util.DynText({
            txt:layer.label,
            x:-this.h*.025,
            y:bH*.1,
            w:this.w/2+this.h*.025,
            h:bH*.8,
            background: {
                clear:true,
                RGB: '255,255,255',
                alpha: 0.5,
                radius: bH/2
            },
            context:this.controlsLayer.context
        });
    };
    OU.activity.LayerTileZoom.prototype.setZoom = function(val,viewer) { // Called when scrubBar is moved
        viewer.zoomSlider.render();
        if(viewer.tileView!=null) {
            viewer.tileView.scale(val);
        }
    };
    OU.activity.LayerTileZoom.prototype.setSlider = function(val,viewer) { // Called when TileViewer changed via mousewheel, etc.
        viewer.zoomSlider.setTarget(val);
    };
    OU.activity.LayerTileZoom.prototype.pinch = function(e,s,x,y,dx,dy,viewer) { // called when pinch event detected
        var ns = ((e-s)/viewer.h)+viewer.tileView.s;
        ns = ns>1?1:(ns<0?0:ns);
        viewer.tileView.scale(ns,{
            x:x,
            y:y
        });
        viewer.zoomSlider.sliderPos=ns;
        viewer.zoomSlider.render();
    };
    OU.base(this,data,instance,controller);
};
OU.inherits(OU.activity.LayerTileZoom,OU.util.Activity);
