/**
 * @fileOverview ModelGraph - renders a graph of an iterative model
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 * @author Will Rawes
 *
 * modified Martin Donnelly
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.Layer');
OU.require('OU.util.Slider');
OU.require('OU.util.Instruction');
/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.ModelGraph = function ( data, instance, controller ) {
    OU.activity.ModelGraph.prototype.canvasView = function () {
        var bH = OU.controlHeight, self = this;
        this.config = {
            depth:5,
            fps:120 // 40ms = 25 fps
        };
        this.bgLayer = new OU.util.Layer({
            container:this
        });
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        this.PAD = this.h * .045;
        this.yTics = {};
        this.xTics = {};
        this.modelLayer = new OU.util.Layer({
            container:this,
            'id':'model',
            'h':this.h - bH
        });
        this.infoLayer = new OU.util.Layer({
            container:this,
            'id':'info',
            'h':this.h - bH
        });
        this.controlsLayer = new OU.util.Layer({
            container:this,
            'id':'controls',
            hasEvents:true
        });
        this.graphArea = {
            x:this.w * .4,
            //y: this.PAD,
            y:this.h * 0.07,
            w:this.w * .6 - this.PAD,
            h:this.h - bH - this.PAD * 2
        };
        this.inRender = false;
        this.dataset = function () {
        };
        this.bestFitOn = this.data.settings.straightLineFit || false;
        //     this.displaySolution = this.data.settings.solution ! null || false;
        this.sliderUpdate = this.data.settings.sliderUpdate===false ? false : true;
        // Set dataset to 1st and render
        this.setDataset();
        this.changeDataset(OU.LocalStorage.load("OU.GWarming.dataset") || 0);
        //            this.doRender = true;
        this.render();//Cycle();
        this.answered = false;
        this.myAnswer = false;
        this.resized = false;
        this.datasetSelector();
    };
    OU.activity.ModelGraph.prototype.resize = function () {
        OU.activity.ModelGraph.superClass_.resize.call(this); // call the parent class resize
        var bH = OU.controlHeight, bX = bH * 3;
        this.resized = true;
        this.bgLayer.resize();
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        this.PAD = this.h * .045;
        this.graphArea = {
            x:this.w * .4,
            y:this.h * 0.07,
            w:this.w * .6 - this.PAD,
            h:this.h - bH - this.PAD * 2
        };
        this.modelLayer.resize({
            'h':this.h - bH
        });
        this.infoLayer.resize();
        this.controlsLayer.resize();
        this.infoLayer.clear();
        this.controlsLayer.clear();
        this.initControls();
        if (this.dsSelector!==undefined) {
            this.dsSelector.resize({
                x:this.x + this.w - 400,
                y:this.y + this.h * .025,
                w:5,
                h:5
            });
        }
        if (this.data.settings.buttons.straightLine) {
            this.straightLineButton.resize({
                x:this.w - bX - bH * 2,
                y:this.h - bH,
                w:bH * 4,
                h:bH
            });
            bX = bX - bH * 4;
        }
        if (this.data.settings.buttons.solve) {
            this.solveButton.resize({
                x:this.graphArea.x,
                y:this.h - bH,
                w:bH * 4,
                h:bH
            });
            bX = bX - bH * 4;
        }
        if (this.data.settings.buttons.solve && this.answered) {
            this.displayResult();
        }
        this.render();//
    };
    OU.activity.ModelGraph.prototype.remove = function () {
        OU.activity.ModelGraph.superClass_.remove.call(this); // call the superclass method
        this.doRender = false;
    };
    OU.activity.ModelGraph.prototype.setDataset = function () {
        var ds;
        this.datasets = [];
        for (var i = 0; i < this.data.datasets.length; i++) {
            ds = this.data.datasets[i];
            ds.container = this;
            this.datasets.push(new this.Dataset(ds));
        }
    };
    OU.activity.ModelGraph.prototype.datasetSelector = function () {
        var i, ds, n = this.datasets.length, bH = OU.controlheight,
            h = '<form>Dataset:<select onchange="OU.obj.' + instance + '.changeDataset();" id="eqList' + this.instance + '">';
        if (n < 2)
            return; // don't include if only 1 dataset
        for (i = 0; i < n; i++) {
            ds = this.datasets[i];
            h += '<option value="' + i + '">' + ds.name + '</option>';
        }
        h += '</select></form>';
        this.dsSelector = new OU.util.HtmlBox({
            container:this,
            html:h,
            x:this.x + bH * 5,
            y:this.y + this.PAD * 3,
            w:5,
            h:5,
            unclosable:true,
            handleScrolling:false,
            style:'overflow:visible'
        });
    };
    OU.activity.ModelGraph.prototype.changeDataset = function ( ds ) {
        if (ds===undefined) {
            var eqList = document.getElementById('eqList' + this.instance);
            ds = eqList.value;
        }
        this.dataset = this.datasets[ds || 0];
        OU.LocalStorage.save("OU.GWarming.dataset", ds);
        // Calculate the viewport
        this.xScale = (this.w - this.PAD * 2) / this.dataset.dx;
        this.yScale = ((this.h * .9) - this.PAD * 2) / this.dataset.dy;
        this.setYMin = this.dataset.setYMin;
        this.setYMax = this.dataset.setYMax;
        this.yZero = this.dataset.yZero;
        this.controlsLayer.context.clearRect(0, 0, this.w, this.h);
        this.initControls();
        this.resetView();
    };
    OU.activity.ModelGraph.prototype.resetView = function () {
        var ds = this.dataset, bH = OU.controlHeight;
        ds.reset();
        this.xScale = (this.w - this.PAD * 2) / ds.dx;
        this.yScale = ((this.h - bH) - this.PAD * 2) / ds.dy;
        //        this.renderFrame();
        //        this.doRender = true;
        this.render();
    };
    this.first = true;
    OU.activity.ModelGraph.prototype.renderCycle = function () {
        var self = this;
        if (this.doRender) {
            if (this.first) {
                if (!this.sliderUpdate)this.first = false;
                this.render();
            }
            if (this.resized) {
                this.resized = false;
            }
            else {
                this.myAnswer = null;
            }
            if (this.data.settings.buttons.solve && this.answered) {
                this.displayResult();
            }
        }
        this.renderControls();
        setTimeout(function () {
            self.renderCycle();
        }, 100);
    };
    OU.activity.ModelGraph.prototype.xPt = function ( v ) {
        return this.xOff + (v - this.xMin) * this.xS;
    };
    OU.activity.ModelGraph.prototype.yPt = function ( v ) {
        return this.yOff - (v - this.yMin) * this.yS;
    };
    OU.activity.ModelGraph.prototype.render = function () {
        var ctx = this.modelLayer.context, x, y, i, j, lastX, lastY,
            bH = OU.controlHeight, circ = Math.PI * 2, failed = false,
            n, sumXY = 0, sumX = 0, sumX2 = 0, sumY = 0, a, b,
            ds = this.dataset,
            r = this.w / 200;
        this.renderControls();
        this.xOff = this.graphArea.x;
        this.yOff = this.graphArea.y + this.graphArea.h;
        ds.calcPoints();
        this.xS = ds.xScale;
        this.yS = ds.yScale;
        this.xMin = ds.x1;
        this.yMin = ds.y1;
        this.gxMin = this.xPt(ds.x1);
        this.gxMax = this.xPt(ds.x2);
        this.gyMin = this.yPt(ds.y1);
        this.gyMax = this.yPt(ds.y2);
        this.renderFrame();
        ctx.clearRect(0, 0, this.w, this.h - bH);
        ctx.save();
        ctx.lineWidth = 1;
        n = ds.points.length;
        sumXY = 0;
        sumX = 0;
        sumX2 = 0;
        sumY = 0;
        for (j = n; j--;) {
            x = ds.points[j].x;
            y = ds.points[j].y;
            if (x===undefined || y===undefined || isNaN(x) || isNaN(y)) {
                failed = true;
            }
            else {
                if (this.bestFitOn) {
                    sumXY = sumXY + (x * y);
                    sumX = sumX + x;
                    sumX2 = sumX2 + x * x;
                    sumY = sumY + y;
                }
                ctx.fillStyle = ds.fillStyle || "#f44";
                ctx.strokeStyle = "#100";
                ctx.lineWidth = 1;
                if (x >= ds.x1 && x <= ds.x2 && y >= ds.y1 && y <= ds.y2) {
                    x = this.xPt(x);
                    y = this.yPt(y);
                    if (lastX!==undefined) {
                        ctx.beginPath();
                        ctx.moveTo(lastX, lastY);
                        ctx.lineTo(x, y);
                        ctx.stroke();
                    }
                    if (ds.style=='rect') {
                        ctx.beginPath();
                        ctx.fillRect(x - r, y - r, r * 2, r * 2);
                        ctx.stroke();
                    }
                    else {
                        ctx.beginPath();
                        ctx.arc(x, y, r, 0, circ, false);
                        ctx.fill();
                        ctx.stroke();
                    }//*/
                    lastX = x;
                    lastY = y;
                }
            }
        }
        if (!failed && this.bestFitOn) {
            a = (n * sumXY - sumX * sumY) / (n * sumX2 - sumX * sumX);
            b = (sumY - a * sumX) / n;
            x = ds.x1;
            y = x * a + b;
            if (y < ds.y1) {
                y = ds.y1;
                x = (y - b) / a;
            }
            else if (y > ds.y2) {
                y = ds.y2;
                x = (y - b) / a;
            }
            x = this.xPt(x);
            y = this.yPt(y);
            ctx.beginPath();
            ctx.moveTo(x, y);
            x = ds.x2;
            y = x * a + b;
            if (y < ds.y1) {
                y = ds.y1;
                x = (y - b) / a;
            }
            else if (y > ds.y2) {
                y = ds.y2;
                x = (y - b) / a;
            }
            x = this.xPt(x);
            y = this.yPt(y);
            ctx.lineTo(x, y);
            ctx.strokeStyle = ds.fillStyle || '#44f';
            ctx.stroke();
        }
        ctx.restore();
        if (failed) {
            new OU.util.Instruction({
                container:this,
                message:this.data.incompleteMessage || 'Some of the values are missing, please ensure you have completed the relevant table.'
            });
        }
        this.doRender = false;
    };
    OU.activity.ModelGraph.prototype.isNum = function ( n ) {
        return !isNaN(parseFloat(n)) && isFinite(n);
    };
    OU.activity.ModelGraph.prototype.renderFrame = function () {
        var ctx = this.bgLayer.context,
            ds = this.dataset,
            xAxis, yAxis, i, j, val,
            xS = ds.xScale,
            yS = ds.yScale,
            PAD = this.PAD,
            xOff = this.graphArea.x;
        ctx.gradRect();
        ctx.save();
        ctx.fillStyle = '#fff';
        ctx.fillRect(this.xPt(this.xMin)/*this.graphArea.x*/, this.graphArea.y, this.graphArea.w, this.graphArea.h);
        ctx.textAlign = 'right';
        ctx.lineWidth = 0.7;
        yAxis = -ds.x1 * xS + this.graphArea.x;
        yAxis = yAxis < xOff ? xOff : (yAxis > this.graphArea.x + this.graphArea.w ? this.graphArea.x + this.graphArea.w : yAxis);
        for (i = this.yTics.min; i <= this.yTics.max; i += this.yTics.interval) {
            var yy = this.yPt(i);
            ctx.beginPath();
            ctx.strokeStyle = "#000";
            ctx.moveTo(this.gxMin - 5, yy);
            ctx.lineTo(this.gxMin, yy);
            ctx.closePath();
            ctx.stroke();
            ctx.lineWidth = 0.7;
            ctx.strokeStyle = "#777";
            ctx.beginPath();
            ctx.moveTo(this.gxMin, yy);
            ctx.lineTo(this.gxMax, yy);
            ctx.closePath();
            ctx.stroke();
            val = ((i * 100) | 0) / 100;
            ctx.fillStyle = "#000";
            ctx.fillText(val, yAxis - 10, yy);
            for (j = i + this.yTics.subInterval; j < i + (i < this.yTics.max ? this.yTics.interval : 0); j += this.yTics.subInterval) {
                yy = this.yPt(j);
                ctx.strokeStyle = "#aaa";
                ctx.lineWidth = 0.3;
                ctx.beginPath();
                ctx.moveTo(this.gxMin, yy);
                ctx.lineTo(this.gxMax, yy);
                ctx.closePath();
                ctx.stroke();
            }
        }
        xAxis = ds.y2 * yS + this.graphArea.x;
        xAxis = xAxis < this.graphArea.y ? this.graphArea.y : (xAxis > this.graphArea.y + this.graphArea.h ? this.graphArea.y + this.graphArea.h : xAxis);
        ctx.textAlign = 'center';
        for (i = this.xTics.min; i <= this.xTics.max; i += this.xTics.interval) {
            var xx = this.xPt(i);
            ctx.beginPath();
            ctx.strokeStyle = "#000";
            ctx.moveTo(xx, this.gyMin);
            ctx.lineTo(xx, this.gyMin + 5);
            ctx.closePath();
            ctx.stroke();
            ctx.lineWidth = 0.7;
            ctx.strokeStyle = "#777";
            ctx.beginPath();
            ctx.moveTo(xx, this.gyMin);
            ctx.lineTo(xx, this.gyMax);
            ctx.closePath();
            ctx.stroke();
            val = ((i * 100) | 0) / 100;
            ctx.fillStyle = "#000";
            ctx.fillText(val, xx, this.gyMin + 10);
            for (j = i + this.xTics.subInterval; j < i + this.xTics.interval; j += this.xTics.subInterval) {
                xx = this.xPt(j);
                ctx.lineWidth = 0.3;
                ctx.strokeStyle = "#aaa";
                ctx.beginPath();
                ctx.moveTo(xx, this.gyMin);
                ctx.lineTo(xx, this.gyMax);
                ctx.closePath();
                ctx.stroke();
            }
        }
        ctx.strokeStyle = "#000";
        ctx.lineWidth = 0.7;
        ctx.fillStyle = "#444";
        ctx.beginPath();
        ctx.moveTo(yAxis, this.graphArea.y);
        ctx.lineTo(yAxis, this.graphArea.y + this.graphArea.h);
        ctx.lineTo(yAxis + this.graphArea.w, this.graphArea.y + this.graphArea.h);
        ctx.lineTo(yAxis + this.graphArea.w, this.graphArea.y);
        ctx.lineTo(yAxis, this.graphArea.y);
        ctx.closePath();
        ctx.stroke();
        ctx.restore();
        if (ds.xAxisLabel!==undefined) {
            new OU.util.DynText({
                txt:ds.xAxisLabel,
                container:this,
                context:ctx,
                align:'center',
                x:this.graphArea.x,
                y:this.graphArea.y + this.graphArea.h + 20,
                w:this.graphArea.w,
                h:30
            });
        }
        if (ds.yAxisLabel!==undefined) {
            ctx.save();
            ctx.translate(this.graphArea.x, this.graphArea.y + this.graphArea.h);
            ctx.rotate(-Math.PI / 2);
            new OU.util.DynText({
                txt:ds.yAxisLabel,
                container:this,
                context:ctx,
                align:'center',
                x:0,
                y:-60,
                w:this.graphArea.h,
                h:30
            });
            ctx.restore();
        }
    };
    OU.activity.ModelGraph.prototype.calcTics = function ( mn, mx ) {
        ///  ticInterval calculation thanks to M.A.Brown
        var dif = mx - mn, n = Math.floor(Math.log(dif) / Math.log(10.0)),
            a = dif * Math.pow(10.0, -n), ticInterval, dt = 5;
        if (a < 2.0) {
            ticInterval = 0.2;
            dt = 2;
        } else if (a < 5.0) ticInterval = 0.5;
        else ticInterval = 1.0;
        ticInterval = ticInterval * Math.pow(10.0, n);
        return {
            interval:ticInterval,
            min:Math.floor(mn / ticInterval) * ticInterval,
            max:Math.ceil(mx / ticInterval) * ticInterval,
            subInterval:ticInterval / dt
        }
    };
    OU.activity.ModelGraph.prototype.setVariable = function ( p, v ) {
        v.value = v.min + (v.max - v.min) * p;
        v.parent.render();
        //        v.parent.doRender = true;
    };
    OU.activity.ModelGraph.prototype.initControls = function () {
        var i, v, ctx = this.controlsLayer.context,
            ictx = this.infoLayer.context, bH = OU.controlHeight,
            clickable = this.controlsLayer.events.clickable,
            self = this, bX = bH * 3, ds = this.dataset, y, dH;
        clickable.length = 0;
        for (i = 0; i < ds.variables.length; i++) {
            v = ds.variables[i];
            v.parent = this;
            v.slider = new OU.util.Slider({
                container:this,
                instance:this.instance + '_var_' + i,
                titlePosition:v.titlePosition,
                valueUnits:v.units,
                title:v.title,
                x:bH,
                y:bH * i + this.PAD * 3,
                w:this.graphArea.x - bH * 4,
                h:bH,
                fps:60,
                sliderHeight:bH / 2,
                drawContainer:false,
                callback:this.setVariable,
                callbackParam:v,
                frames:1,
                background:{
                    color:'#d0d0d0'
                },
                context:ctx,
                showValue:true,
                disableTab:v.disabled,
                range:{
                    min:v.min,
                    max:v.max,
                    nDecimals:v.nDecimals
                },
                sliderPos:(v.init - v.min) / (v.max - v.min)
            });
            if (v.disabled)
                v.slider.disabled = true;
            else
                clickable.push(v.slider);
        }
        // Moved the calculation into the instruction displayer
        y = (bH * ds.variables.length * 0.7 + this.PAD * 4);
        dH = (this.h - bH - y) / 3;
        if (ds.instructions!==undefined) {
            //y=(bH*ds.variables.length + this.PAD*4);
            new OU.util.DynText({
                txt:ds.instructions,
                align:'left',
                x:0,
                y:y,
                w:this.graphArea.x - bH / 3,
                // h: dH*2,
                h:1.1 * dH * (this.data.settings.buttons.solve ? 2 : 3),
                context:ictx
            });
        }
        if (this.data.settings.buttons.straightLine) {
            this.straightLineButton = new OU.util.ControlButton({
                txt:'Straight Line Fit',
                x:this.w - bX - bH * 2,
                y:this.h - bH,
                w:bH * 4,
                h:bH,
                layer:this.controlsLayer,
                onOffIndicator:true,
                onOff:this.bestFitOn,
                onClick:function () {
                    self.toggleStraightLine();
                }
            });
            bX = bX - bH * 4;
        }
        if (this.data.settings.buttons.solve) {
            this.solveButton = new OU.util.ControlButton({
                txt:'Solve',
                x:this.graphArea.x,
                y:this.h - bH,
                w:bH * 4,
                h:bH,
                layer:this.controlsLayer,
                onOffIndicator:false,
                onOff:this.bestFitOn,
                onClick:function () {
                    self.calcSolution();
                }
            });
            bX = bX - bH * 4;
        }
        if (this.sliderUpdate===false) {
            this.selectionDone = new OU.util.ControlButton({
                txt:"Run",
                x:bH * 12,
                y:this.h - bH,
                w:bH * 2,
                h:bH,
                layer:this.controlsLayer,
                onClick:function () {
                    self.first = true;
                    self.render();
                }
            });
        }
        new OU.util.pageTitle({
                txt:this.dataset.name,
                context:ictx
            }
        );
    };
    OU.activity.ModelGraph.prototype.toggleStraightLine = function () {
        this.bestFitOn = !this.bestFitOn;
        this.straightLineButton.params.onOff = this.bestFitOn;
        this.render();
    };
    OU.activity.ModelGraph.prototype.calcSolution = function () {
        var m = this.data,
            i, v, smin, smax;
        var correct = true;
        for (i = 0; i < this.dataset.variables.length; i++) {
            v = this.dataset.variables[i].value;
            smin = parseFloat(this.dataset.variables[i].marginLower);
            smax = parseFloat(this.dataset.variables[i].marginUpper);
            if (smin > smax) {
                var t = smin;
                smin = smax;
                smax = t;
            }
            if ((v > smin) && (v < smax)) {
                // ok
            }
            else {
                correct = false;
            }
        }
        this.myAnswer = correct;
        this.displayResult(correct);
    };
    OU.activity.ModelGraph.prototype.renderControls = function () {
        var i, v;
        this.controlsLayer.clear();
        if (this.data.settings.buttons.straightLine) {
            this.straightLineButton.render();
        }
        if (this.data.settings.buttons.solve) {
            this.solveButton.render();
        }
        if (this.sliderUpdate===false) {
            this.selectionDone.render();
        }
        for (i = 0; i < this.dataset.variables.length; i++) {
            //console.log(i);
            v = this.dataset.variables[i];
            v.slider.render();
        }
    };
    OU.activity.ModelGraph.prototype.Dataset = function ( params ) {
        var t;
        this.name = params.name || 'y=?';
        this.container = params.container;
        this.x1 = params.x1 || 0;
        this.x2 = params.x2 || 10;
        this.xAxisLabel = params.xAxisLabel;
        this.yAxisLabel = params.yAxisLabel;
        this.startY = params.startY || 10;
        this.setYMin = params.setYMin;
        this.setYMax = params.setYMax;
        this.yZero = params.yZero;
        this.x2 = params.x2 || 10;
        this.variables = params.variables || {};
        this.instructions = params.instructions;
        this.solutionText = params.solutionText;
        this.model = params.model || function () {
        };
        this.xInterval = params.xInterval || 1;
        if (this.x1 > this.x2) {
            t = this.x2;
            this.x2 = this.x1;
            this.x1 = t;
        }
        this.reset_x1 = this.x1;
        this.reset_x2 = this.x2;
        this.reset_y1 = this.y1;
        this.reset_y2 = this.y2;
        if (params.colour!==undefined)
            this.colour = params.colour;
        if (params.rgb!==undefined)
            this.rgb = params.rgb;
        OU.activity.ModelGraph.prototype.Dataset.prototype.calcPoints = function () {
            var i, params = {}, v, y, c = this.container;
            for (i = this.variables.length; i--;) {
                v = this.variables[i];
                params[v.varName] = v.value;
            }
            params.x1 = this.x1;
            params.x2 = this.x2;
            params.xInterval = this.xInterval;
            params.startY = this.startY;
            this.points = this.model(params);
            this.y1 = 1.e12;
            this.y2 = -this.y1;
            for (i = this.points.length; i--;) {
                if (c.isNum(c.yZero))this.points[i].y -= c.yZero;
                y = this.points[i].y;
                if (y > this.y2)
                    this.y2 = y;
                if (y < this.y1)
                    this.y1 = y;
            }
            if (c.isNum(c.setYMin)) {
                this.y1 = c.setYMin;
                this.y2 = c.setYMax;
            }
            c.yTics = c.calcTics(this.y1, this.y2);
            this.y1 = c.yTics.min;
            this.y2 = c.yTics.max;
            c.xTics = c.calcTics(this.x1, this.x2);
            this.dx = Math.abs(this.x2 - this.x1);
            this.dy = Math.abs(this.y2 - this.y1);
            this.xScale = (c.graphArea.w) / this.dx;
            this.yScale = (c.graphArea.h) / this.dy;
        };
        OU.activity.ModelGraph.prototype.Dataset.prototype.init = function () {
            var i, v;
            for (i = this.variables.length; i--;) {
                v = this.variables[i];
                v.value = v.init;
            }
        };
        OU.activity.ModelGraph.prototype.Dataset.prototype.reset = function () {
            this.x1 = this.reset_x1;
            this.x2 = this.reset_x2;
            this.y1 = this.reset_y1;
            this.y2 = this.reset_y2;
            this.init();
        };
        this.init();
    };
    OU.activity.ModelGraph.prototype.displayResult = function ( result ) {
        var ictx = this.infoLayer.context, bH = OU.controlHeight,
            ds = this.dataset, y, dH, txt, col;
        this.infoLayer.clear();
        // Moved the calculation into the instruction displayer
        y = (bH * ds.variables.length + this.PAD * 4);
        dH = (this.h - bH - y) / 3;
        if (ds.instructions!==undefined) {
            //y=(bH*ds.variables.length + this.PAD*4);
            new OU.util.DynText({
                txt:ds.instructions,
                align:'left',
                x:0,
                y:y,
                w:this.graphArea.x - bH,
                // h: dH*2,
                h:dH * (this.data.settings.buttons.solve ? 2 : 3),
                context:ictx
            });
        }
        new OU.util.pageTitle({
            txt:this.dataset.name,
            context:ictx
        });
        if (this.myAnswer!=null) {
            if (this.myAnswer) { // correct
                txt = ds.solutionText[0];
                col = '#66cc00';
            }
            else { // incorrect
                txt = ds.solutionText[1];
                col = '#cc3333';
            }
            new OU.util.DynText({
                txt:txt,
                align:'left',
                x:0,
                y:y + (dH * 2),
                w:this.graphArea.x - bH,
                h:dH,
                context:ictx,
                background:{
                    col:col,
                    radius:this.h * .03
                }
            });
            this.answered = true;
        }
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.ModelGraph, OU.util.Activity);
