/**
 * @fileOverview Network - Displayed information in a network structure
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.Draggable');
OU.require('OU.util.Layer');
OU.require('OU.util.TabSlide');
OU.require('OU.util.ImageLoader');
/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.Network = function ( data, instance, controller ) {
    OU.activity.Network.prototype.canvasView = function () {
        var bH = OU.controlHeight, self = this;
        this.config = {
            fps:40
        };
        OU.obj[this.instance] = this; // push into object array so it can call itself from the HTML content
        // create Canvas & Context
        this.centre = undefined;
        this.bgLayer = new OU.util.Layer({
            container:this
        });
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        this.linkLayer = new OU.util.Layer({
            container:this
        });
        this.linkLayer.context.globalAlpha = 0.25;
        this.secondaryLayer = new OU.util.Layer({
            container:this
        });
        this.secondaryLayer.context.globalAlpha = 0.5;
        this.secondaryLinkLayer = new OU.util.Layer({
            container:this
        });
        this.secondaryLinkLayer.context.lineWidth = 3;
        this.imageLayer = new OU.util.Layer({
            container:this,
            hasEvents:true
        });
        this.groupsLayer = new OU.util.Layer({
            h:bH,
            container:this,
            hasEvents:true
        });
        this.boundary = {
            x:bH * 1.5,
            y:bH * 1.5,
            w:this.w - bH * 2,
            h:this.h - bH * 2
        };
        this.fbWidth = this.w;
        this.imageLoader = new OU.util.ImageLoader({
            container:this,
            data:this.data,
            onLoad:function () {
                self.start();
            }
        });
    };
    OU.activity.Network.prototype.resize = function () {
        OU.activity.Network.superClass_.resize.call(this); // call the parent class resize
        this.init();
    };
    OU.activity.Network.prototype.initItems = function () {
        var network = this, i, cX = this.w / 2 - 25, cY = this.h / 2 - 25, n, group, node, html,
            j, k, al, endNodeIdx, endNode, nodeCount = 0, bW, gctx, w, mw, gO,
            numGroups = this.data.groups.length,
            ctx = this.imageLayer.context,
            sctx = this.secondaryLayer.context,
            lineWidth, ev = this.imageLayer.events,
            dTheta, theta = 0, v, d2r = Math.PI / 180,
            lastNode = parseInt(OU.LocalStorage.load('OU.network.centre'));
        if (isNaN(lastNode))
            lastNode = 0;
        this.items = [ ];
        this.energy = 1;
        ctx.font = '14px ' + OU.theme.font;
        sctx.font = '14px ' + OU.theme.font;
        // Count Nodes
        for (i = 0; i < numGroups; i++) {
            group = this.data.groups[ i ];
            nodeCount = nodeCount + group.nodes.length;
        }
        // Add Nodes
        for (i = 0; i < numGroups; i++) {
            group = this.data.groups[ i ];
            dTheta = 360 / group.nodes.length;
            for (n = 0; n < group.nodes.length; n++) {
                node = group.nodes[n];
                node.group = group;
                if (node.links===undefined)
                    node.links = [];
                if (node.linkRels===undefined)
                    node.linkRels = [];
                theta = theta + dTheta;
                v = 30;
                lineWidth = ctx.measureText(node.label, 50, 50).width;
                this.items.push(new this.Item({
                    "network":this,
                    "id":node.id,
                    "events":ev,
                    "data":node,
                    "type":"node",
                    "x":cX,
                    "y":cY,
                    "w":lineWidth * 1.2,
                    "h":25,
                    "v":{
                        x:Math.cos(theta * d2r) * v,
                        y:Math.sin(theta * d2r) * v
                    }
                }));
                node = this.items[this.items.length - 1];
                node.links.push(nodeCount + i);
                node.linkRels.push("");
                if (lastNode==node.id) {
                    this.centre = node;
                }
            }
        }
        // Add nodes for each group
        for (i = 0; i < this.data.groups.length; i++) {
            group = this.data.groups[ i ];
            lineWidth = ctx.measureText(group.name, 50, 50).width;
            this.items.push(new this.Item({
                "network":this,
                "id":nodeCount + i,
                "events":ev,
                "type":"group",
                "data":{
                    label:group.name,
                    links:[],
                    group:group
                },
                "x":cX,
                "y":cY,
                "w":lineWidth * 1.2,
                "h":50,
                "v":{
                    x:0,
                    y:0
                }
            }));
            node = this.items[this.items.length - 1];
            if (lastNode==node.id) {
                this.centre = node;
            }
        }
        // add reciprocal links between nodes
        for (i = this.items.length; i--;) {
            node = this.items[i];
            for (j = node.links.length; j--;) {
                endNode = this.items[node.links[j]];
                al = false;
                for (k = endNode.links.length; k--;) {
                    if (endNode.links[k]==i) {
                        al = true;
                        k = 0;
                    }
                }
                if (!al) {
                    endNode.links.push(i); // add recriprocal link if not already there
                    endNode.linkRels.push(node.data.linkRels[j] || '');
                }
            }
            for (j = node.data.links.length; j--;) {
                endNodeIdx = this.getNodeByID(node.data.links[j].id);
                if (endNodeIdx!==false) {
                    node.links.push(endNodeIdx); // add link from this node to target
                    node.linkRels.push(node.data.links[j].rel || '');
                    endNode = this.items[endNodeIdx];
                    al = false;
                    for (k = endNode.links.length; k--;) {
                        if (endNode.links[k]==i) {
                            al = true;
                            k = 0;
                        }
                    }
                    if (!al) {
                        endNode.links.push(i); // add recriprocal link if not already there
                        endNode.linkRels.push(node.data.links[j].rel || '');
                    }
                }
            }
        }
        if (this.centre===undefined) {
            this.centre = this.items[nodeCount]; // start with the 1st group's node
        }
        gctx = this.groupsLayer.context;
        gctx.font = "14px " + OU.theme.font;
        mw = 0;
        for (i = 0; i < numGroups; i++) {
            group = this.data.groups[ i ];
            w = gctx.measureText(group.name, 50, 50).width + 40;
            mw = w > mw ? w : mw;
        }
        bW = this.w - 100 / numGroups;
        bW = bW < mw ? bW : mw;
        gO = (this.w - (numGroups * bW)) / 2;
        this.groupButtons = [];
        for (i = 0; i < numGroups; i++) {
            group = this.data.groups[ i ];
            this.groupButtons[i] = new OU.util.Button({
                layer:this.groupsLayer,
                fontSize:14,
                txt:group.name,
                x:gO + i * bW,
                y:0,
                w:bW,
                h:OU.controlHeight,
                colour:group.textcol,
                background:{
                    col:group.colour
                },
                onClickParam:nodeCount + i,
                onClick:function ( i ) {
                    network.centre = network.items[i];
                    network.centre.tab.toTop();
                    network.doRender = true;
                    OU.LocalStorage.save('OU.network.centre', network.centre.id);
                }
            });
        }
        // Add info layers
        for (i = network.items.length; i--;) {
            node = network.items[i];
            html = "<div class='right'><h1>Connections</h1>";
            html = html + network.nodeLinksHTML(node, true);
            if (node.data.info!==undefined)
                html = html + "</div><div class='left'>" + node.data.info + "</div>";
            else
                html = html + "</div><div class='left'>Click on the items within the network view to see further information.</div>";
            node.tab = new OU.util.TabSlide({
                container:network,
                y:OU.controlHeight * 1.5,
                title:node.data.label,
                tabCol:node.data.group.colour,
                textCol:node.data.group.textcol,
                html:html
            });
        }
        network.centre.tab.toTop();
        OU.LocalStorage.save('OU.network.centre', network.centre.id);
    };
    OU.activity.Network.prototype.newCentre = function ( id ) {
        for (var i = this.items.length; i--;) {
            if (this.items[i].id==id) {
                this.centre = this.items[i];
                this.centre.tab.toTop();
                return;
            }
        }
        OU.LocalStorage.save('OU.network.centre', this.centre.id);
    };
    OU.activity.Network.prototype.nodeLinksHTML = function ( node, incLinks ) {
        var i, sn, links = [], lnk, cg = "", h = "";
        for (i = node.links.length; i--;) {
            sn = this.items[node.links[i]];
            if (sn.type=="node") {
                links.push({
                    group:sn.data.group.name,
                    title:sn.data.label,
                    rel:node.linkRels[i],
                    id:sn.id
                });
            }
        }
        links.sort(function ( a, b ) {
            return b.group < a.group ? -1 : (b.group > a.group ? 1 : 0);
        });
        for (i = links.length; i--;) {
            lnk = links[i];
            if (lnk.group!=cg) {
                if (cg!="")
                    h = h + "</ul>";
                cg = lnk.group;
                h = h + "<h4>" + cg + "</h4><ul>";
            }
            if (incLinks)
                h = h + "<li><a href='javascript:OU.obj." + this.instance + ".newCentre(" + lnk.id + ");'><strong>" + lnk.title + "</strong></a>";
            else
                h = h + "<li><strong>" + lnk.title + "</strong>";
            if (lnk.rel!==undefined && lnk.rel!='')
                h = h + "<br/>" + lnk.rel;
            h = h + "</li>";
        }
        h = h + "</ul>";
        return h;
    };
    OU.activity.Network.prototype.getNodeByID = function ( id ) {
        for (var k = this.items.length; k--;) {
            if (this.items[k].data.id==id) {
                return k;
            }
        }
        return false;
    };
    OU.activity.Network.prototype.start = function () {
        var i, item;
        // resize items to images if aplicable
        this.initItems();
        for (i = this.items.length; i--;) {
            item = this.items[i];
            if (item.image) {
                item.draggable.w = item.w = item.image.width;
                item.draggable.h = item.h = item.image.height;
            }
        }
        // start render loop
        this.doRender = true;
        this.renderLoop();
    };
    OU.activity.Network.prototype.renderLoop = function () {
        var self = this;
        if (this.doRender)
            this.render();
        setTimeout(function () {
            self.renderLoop();
        }, this.config.fps);
    };
    OU.activity.Network.prototype.render = function () {
        var i, j, item, item2, numItems = this.items.length, b,
            lctx = this.linkLayer.context,
            sctx = this.secondaryLayer.context,
            slctx = this.secondaryLinkLayer.context,
            ctx = this.imageLayer.context;
        this.linkLayer.clear();
        this.secondaryLayer.clear();
        this.secondaryLinkLayer.clear();
        this.imageLayer.clear();
        ctx.fillStyle = '#ccc';
        ctx.textAlign = 'center';
        sctx.textAlign = 'center';
        this.collisions = 0;
        for (i = numItems; i--;) {
            item = this.items[i];
            if (item.draggable.dragId==0) { // item is not being dragged, so move according to motion
                item.motion();
            }
        }
        this.energy = this.collisions / numItems;
        for (i = numItems; i--;) {
            item = this.items[i];
            item.rendered = false;
            item.draggable.disabled = true;
        }
        for (i = this.centre.links.length; i--;) {
            item = this.items[this.centre.links[i]];
            if (item.type=="node") {
                item.tSF = 1;
                item.render(ctx, this.centre.type=='node', lctx);
            }
        }
        if (this.centre.type=='node') {
            for (i = this.centre.links.length; i--;) {
                item = this.items[this.centre.links[i]];
                if (item.type=="node") {
                    for (j = item.links.length; j--;) {
                        item2 = this.items[item.links[j]];
                        if (item2.type!="group") {
                            if (item2!=this.centre) {
                                item2.tSF = item2.rendered ? item2.tSF : 0.6;
                                item2.render(sctx);
                            }
                        }
                    }
                }
            }
        }
        this.centre.tSF = 1;
        this.centre.render(ctx, true, slctx);
    };
    OU.activity.Network.prototype.Item = function ( params ) {
        this.network = params.network;
        this.events = params.events;
        this.id = params.id;
        this.image = params.data.image;
        this.type = params.type;
        this.data = params.data;
        this.bgcolor = params.data.bgcolor || '#fff',
            this.x = params.x;
        this.y = params.y;
        this.h = params.h;
        this.w = params.w;
        this.v = params.v;
        this.startX = this.x;
        this.startY = this.y;
        this.sF = this.tSF = 1;
        this.links = [];
        this.linkRels = [];
        OU.activity.Network.prototype.Item.prototype.render = function ( ctx, renderLinks, lctx ) {
            var j, tn;
            ctx.beginPath();
            this.draggable.disabled = false;
            if (renderLinks && this.links!==undefined) {
                for (j = this.links.length; j--;) {
                    tn = this.network.items[this.links[j]];
                    if (tn.type=="node") {
                        lctx.beginPath();
                        lctx.moveTo(this.x + this.w / 2, this.y + this.h / 2);
                        if (this.x < tn.x)
                            lctx.bezierCurveTo(this.x + this.w / 2, this.y + this.h / 2, tn.x - tn.w, tn.y + tn.h / 2, tn.x + (1 - tn.sF) * tn.w / 2, tn.y + tn.h / 2);
                        else
                            lctx.bezierCurveTo(this.x + this.w / 2, this.y + this.h / 2, tn.x + tn.w * 2, tn.y + tn.h / 2, tn.x + tn.w - (1 - tn.sF) * tn.w / 2, tn.y + tn.h / 2);
                        lctx.strokeStyle = tn.data.group.colour || '#000';
                        lctx.stroke();
                    }
                }
            }
            if (this.rendered)
                return;
            this.rendered = true;
            if (this.image) {
                ctx.drawImage(this.image, this.x, this.y);
            }
            else {
                ctx.beginPath();
                ctx.roundRect(this.x - (this.w * (this.sF - 1) / 2), this.y - (this.h * (this.sF - 1) / 2), this.w * this.sF, this.h * this.sF, 5);
                if (this.type=="node") {
                    ctx.strokeStyle = '#444';
                    ctx.fillStyle = this.data.group.colour || '#fff';
                    ctx.lineWidth = 1;
                }
                else { // group
                    ctx.strokeStyle = this.data.group.colour || '#444';
                    ctx.fillStyle = '#fff';
                    ctx.lineWidth = 3;
                }
                ctx.fill();
                ctx.stroke();
                if (this.type=="node") {
                    ctx.fillStyle = this.data.group.textcol || '#444';
                }
                else {
                    ctx.fillStyle = this.data.group.colour || '#444';
                }
                ctx.font = '' + (14 * this.sF) + 'px ' + OU.theme.font;
                ctx.fillText(this.data.label, this.x + (this.w) / 2, this.y + (this.h) / 2);
            }
        };
        OU.activity.Network.prototype.Item.prototype.isHit = function ( x, y, m ) {
            var ev = this.network.imageLayer.events;
            if (!m)
                return;
            if (x < this.x || x > this.x + this.w)
                return;
            if (y < this.y || y > this.y + this.h)
                return;
            ev.flush();
            new OU.util.PopUpInfo({
                container:this,
                txt:this.pair.info
            });
        };
        OU.activity.Network.prototype.Item.prototype.startDrag = function ( p ) {
            //sort the items so that the latest dragged is at the end of the queue, therefore render last next time (on top)
            p.me.network.centre = p.me;
            p.me.network.centre.tab.toTop();
            OU.LocalStorage.save('OU.network.centre', p.me.network.centre.id);
        };
        OU.activity.Network.prototype.Item.prototype.endDrag = function ( p ) {
            p.me.network.energy = 1;
        };
        OU.activity.Network.prototype.Item.prototype.newPos = function ( p ) {
            p.me.x = p.x;
            p.me.y = p.y;
            p.me.network.doRender = true;
        };
        OU.activity.Network.prototype.Item.prototype.dims = function () {
            if (this.lockright >= 0) {
                return {
                    w:this.w + this.network.items[this.lockright].w,
                    h:this.h + this.network.items[this.lockright].h
                };
            }
            else {
                return {
                    w:this.w,
                    h:this.h
                };
            }
        };
        OU.activity.Network.prototype.Item.prototype.collide = function ( b ) {
            if (!b.rendered || !this.rendered)
                return false;
            var spacer = 1.1,
                aD = this.dims();
            if (b.draggable.dragId || b.lockleft >= 0) // don't collide with items being dragged or left locked
                return false;
            if (this.x > b.x + b.w * spacer)
                return false;
            if (this.x + aD.w * spacer < b.x)
                return false;
            if (this.y > b.y + b.h * spacer)
                return false;
            if (this.y + aD.h * spacer < b.y)
                return false;
            return true;
        };
        OU.activity.Network.prototype.Item.prototype.motion = function () {
            var i, nX, nY, rX, rY, b, dX, dY, dims,
                cX = this.network.w / 2 - 25, cY = this.network.h * .45 - 25;
            // Adjust Scale Factor
            if (this.draggable.dragId!=0) {
                this.sF = 1.4
            }
            else {
                this.sF = this.sF + (this.tSF - this.sF) / 4;
            }
            // reduce momentum
            dX = this.v.x * .9;
            dY = this.v.y * .9;
            dX = dX > 10 ? 10 : dX;
            dY = dY > 10 ? 10 : dY;
            dX = dX < -10 ? -10 : dX;
            dY = dY < -10 ? -10 : dY;
            // Collide repulsion
            for (i = this.network.items.length; i--;) {
                b = this.network.items[i];
                if (this!==b && this.collide(b)) {
                    rX = this.x - b.x > 0 ? ((this.w + b.w) / 2 - (this.x - b.x)) / 50 : (-((this.w + b.w) / 2) + (this.x - b.x)) / 50;
                    rY = this.y - b.y > 0 ? ((this.h + b.h) / 2 - (this.y - b.y)) / 50 : (-((this.h + b.h) / 2) + (this.y - b.y)) / 50;
                    dX = dX + rX * (1 - this.network.energy / 2);
                    dY = dY + rY * (1 - this.network.energy / 2);
                    b.addDelta(-rX * (1 - this.network.energy / 3), -rY * (1 - this.network.energy / 2)); // bump the colliding item the other way
                    i = 0; // only collide with 1 item per cycle => end loop
                    this.network.collisions++;
                }
            }
            if (this.network.energy < 0.1) { // if overall energy>0 then gravitate back to center
                dX = dX + this.network.energy * (cX - this.x) / 500;
                dY = dY + this.network.energy * (cY - this.y) / 500;
            }
            // Move item according to Motion with boundary
            nX = this.x + dX;
            nY = this.y + dY;
            b = this.network.boundary;
            dims = this.dims();
            if (nX < b.x)
                nX = b.x;
            if (nX > b.x + b.w - dims.w)
                nX = b.x + b.w - dims.w;
            if (nY < b.y)
                nY = b.y;
            if (nY > b.y + b.h - dims.h)
                nY = b.y + b.h - dims.h;
            if (this==this.network.centre) { // move "centre" node to centre of screen
                nX = nX - (nX - (this.network.w / 2 - this.w / 2)) / 4;
                nY = nY - (nY - (this.network.h / 2 - this.h / 2)) / 4;
            }
            this.move(nX, nY);
            this.v.x = dX; // store current movement for next cycle
            this.v.y = dY;
        };
        OU.activity.Network.prototype.Item.prototype.addDelta = function ( dx, dy ) {
            this.v.x = this.v.x + dx;
            this.v.y = this.v.y + dy;
        };
        OU.activity.Network.prototype.Item.prototype.move = function ( x, y ) {
            this.x = x;
            this.y = y;
            this.draggable.x = x;
            this.draggable.y = y;
        };
        if (this.events!=null) {
            this.draggable = new OU.util.Draggable({
                "me":this,
                "events":this.events,
                "x":this.x,
                "y":this.y,
                "h":this.h,
                "w":this.w,
                "onStart":this.startDrag,
                "onEnd":this.endDrag,
                "onMove":this.newPos
            });
            this.events.clickable.push(this.draggable);
        }
    };
    OU.activity.Network.prototype.initAccessible = function () {
        var i, cX = this.w / 2 - 25, cY = this.h / 2 - 25, n, group, node,
            j, k, al, endNodeIdx, endNode, nodeCount = 0,
            numGroups = this.data.groups.length,
            dTheta, theta = 0, v, d2r = Math.PI / 180;
        this.items = [ ];
        // Count Nodes
        for (i = 0; i < numGroups; i++) {
            group = this.data.groups[ i ];
            nodeCount = nodeCount + group.nodes.length;
        }
        // Add Nodes
        for (i = 0; i < numGroups; i++) {
            group = this.data.groups[ i ];
            dTheta = 360 / group.nodes.length;
            for (n = 0; n < group.nodes.length; n++) {
                node = group.nodes[n];
                node.group = group;
                if (node.links===undefined)
                    node.links = [];
                if (node.linkRels===undefined)
                    node.linkRels = [];
                theta = theta + dTheta;
                v = 30;
                this.items.push(new this.Item({
                    "network":this,
                    "id":node.id,
                    "events":null,
                    "data":node,
                    "type":"node",
                    "x":cX,
                    "y":cY,
                    "w":50, // irrelevant
                    "h":25,
                    "v":{
                        x:Math.cos(theta * d2r) * v,
                        y:Math.sin(theta * d2r) * v
                    }
                }));
                node = this.items[this.items.length - 1];
                node.links.push(nodeCount + i);
                node.linkRels.push("");
            }
        }
        // Add nodes for each group
        for (i = 0; i < this.data.groups.length; i++) {
            group = this.data.groups[ i ];
            this.items.push(new this.Item({
                "network":this,
                "id":nodeCount + i,
                "events":null,
                "type":"group",
                "data":{
                    label:group.name,
                    links:[],
                    group:group
                },
                "x":cX,
                "y":cY,
                "w":50,
                "h":50,
                "v":{
                    x:0,
                    y:0
                }
            }));
            node = this.items[this.items.length - 1];
        }
        // add reciprocal links between nodes
        for (i = this.items.length; i--;) {
            node = this.items[i];
            for (j = node.links.length; j--;) {
                endNode = this.items[node.links[j]];
                al = false;
                for (k = endNode.links.length; k--;) {
                    if (endNode.links[k]==i) {
                        al = true;
                        k = 0;
                    }
                }
                if (!al) {
                    endNode.links.push(i); // add recriprocal link if not already there
                    endNode.linkRels.push(node.data.linkRels[j] || '');
                }
            }
            for (j = node.data.links.length; j--;) {
                endNodeIdx = this.getNodeByID(node.data.links[j].id);
                if (endNodeIdx!==false) {
                    node.links.push(endNodeIdx); // add link from this node to target
                    node.linkRels.push(node.data.links[j].rel || '');
                    endNode = this.items[endNodeIdx];
                    al = false;
                    for (k = endNode.links.length; k--;) {
                        if (endNode.links[k]==i) {
                            al = true;
                            k = 0;
                        }
                    }
                    if (!al) {
                        endNode.links.push(i); // add recriprocal link if not already there
                        endNode.linkRels.push(node.data.links[j].rel || '');
                    }
                }
            }
        }
    };
    OU.activity.Network.prototype.accessibleView = function () {
        var i, node, h;
        this.initAccessible();
        h = '<div id="accessibleView">';
        if (this.data.title)
            h += '<h1>' + this.data.title + '</h1>';
        else
            h += "<h1>Connected information viewer</h1>";
        if (this.data.description)
            h += '<p>' + this.data.description + '</p>';
        else
            h += '<p>This activity shows a series of items of information that are connected through one or more connection types. The following is a list of the information and the connectivity between it within this data.</p>';
        for (i = this.items.length; i--;) {
            node = this.items[i];
            h = h + "<div style='clear:right'><hr/>";
            if (node.data.info)
                h = h + "<div class='left'>" + node.data.info + "</div>";
            h = h + "<div class='right'>";
            h = h + this.nodeLinksHTML(node);
            if (node.data.info!==undefined)
                h = h + "</div>";
            else
                h = h + "</div>";
            h = h + "</div>";
        }
        h += '</div>';
        document.body.innerHTML = '';
        var accessible = document.createElement('div');
        accessible.innerHTML = h;
        document.body.appendChild(accessible);
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.Network, OU.util.Activity);
