/**
 * @fileOverview Queue - Allows user to order elements in a queue and gives feedback against a possible order
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.Button');
OU.require('OU.util.DynText');
OU.require('OU.util.Draggable');
OU.require('OU.util.Layer');

KEY_LEFT=37;
KEY_UP=38;
KEY_RIGHT=39;
KEY_DOWN=40;

/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.Queue = function(data,instance,controller) {
    OU.activity.Queue.prototype.canvasView = function() {

        // randomise order of the items
        this.data.items.sort( function( ) {
            return ( Math.round( Math.random() ) - 0.5 );
        } );

        this.bgLayer = new OU.util.Layer({
            container:this
        });
        this.bgLayer.context.gradRect(); // draw background on backdrop layer

        this.imageLayer = new OU.util.Layer({
            container:this,
            hasEvents:true
        });

        this.order = this.data.order;
        this.items = [];

        this.listArea = {
            x:this.w*.05,
            y: this.h*.2,
            w: this.w*.5,
            h:this.h*.7,
            padding: this.w/40,
            itemHeight: ((this.h*.7)-(this.w/20))/this.data.items.length
        };
        this.descriptionArea = {
            x:this.w*.6,
            y: this.h*.2,
            w: this.w*.35,
            h:this.h*.5,
            padding: this.w/40
        };
        this.view='norm';

        for ( var i = 0; i < this.data.items.length; i++ ) {
            var color = "#ccd";
            this.items.push( new this.Item( {
                "queue": this,
                "events": this.events,
                "id": this.data.items[ i ].id,
                "sortOrder": i,
                "order": this.data.items[ i ].order,
                "name": this.data.items[ i ].name,
                "color": color,
                "x": this.listArea.x+this.listArea.padding,
                "y": this.listArea.y+this.listArea.padding +(i * this.listArea.itemHeight),
                "w": this.listArea.w-(2*this.listArea.padding),
                "h": this.listArea.itemHeight*.9
            } ) );
        }
        this.renderBackdrop();
        this.doRender=true;
        this.renderLoop();
    };
    OU.activity.Queue.prototype.resize = function() {
        OU.activity.Queue.superClass_.resize.call(this); // call the parent class resize
        var i;

        this.bgLayer.resize();
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        this.imageLayer.resize();
        this.listArea = {
            x:this.w*.05,
            y: this.h*.2,
            w: this.w*.5,
            h:this.h*.7,
            padding: this.w/40,
            itemHeight: ((this.h*.7)-(this.w/20))/this.data.items.length
        };
        this.descriptionArea = {
            x:this.w*.6,
            y: this.h*.2,
            w: this.w*.35,
            h:this.h*.5,
            padding: this.w/40
        };
        for (i=this.data.items.length; i--; ) {
            this.items[i].setFocus(false);
            this.items[i].resize({
                "x": this.listArea.x+this.listArea.padding,
                "y": this.listArea.y+this.listArea.padding +(i * this.listArea.itemHeight),
                "w": this.listArea.w-(2*this.listArea.padding),
                "h": this.listArea.itemHeight*.9
            });
        }
        this.renderBackdrop();
        this.doRender=true;
    };
    OU.activity.Queue.prototype.renderLoop = function() {
        var self=this;
        if(this.doRender)
            this.render();
        setTimeout(function() {
            self.renderLoop();
        },40);
    };
    OU.activity.Queue.prototype.renderBackdrop = function() {
        var ctx = this.bgLayer.context;
        // draw the title
        new OU.util.DynText( {
            "txt": ( this.data.title ) ? this.data.title : "",
            "x": 0,
            "y": this.h*.05,
            "w": this.w,
            "h": this.h*.1,
            "context": ctx,
            "propTextHeight": 0.9,
            "fontWeight": "bold"
        } );

        // draw instructions
        new OU.util.DynText( {
            "txt": "Change the order by dragging the items above",
            "x" : this.listArea.x,
            "y": this.listArea.y+this.listArea.h,
            "w": this.listArea.w,
            "h": this.h*.05,
            "context": ctx,
            "fontWeight": "bold",
            "colour": "#999"
        });
        // draw descriptions
        ctx.gradRect( {
            x:this.descriptionArea.x,
            y:this.descriptionArea.y,
            w:this.descriptionArea.w,
            h:this.descriptionArea.h,
            radius:this.descriptionArea.padding/2,
            col2: '#fff',
            col1:'#ccc'
        });
        // draw list area
        ctx.gradRect( {
            x:this.listArea.x,
            y:this.listArea.y,
            w:this.listArea.w,
            h:this.listArea.h,
            radius:this.listArea.padding/2,
            col2: '#eef',
            col1:'#99c'
        });
    };
    OU.activity.Queue.prototype.render = function() {
        var ctx = this.imageLayer.context,self=this,
        feedback,matches,i,shuffling,targetX,targetY,newX,newY;

        ctx.clearRect(0,0,this.w,this.h); // clear background

        if(this.view!='feedback') {
            new OU.util.DynText( {
                "txt": this.data.description,
                "x" : this.descriptionArea.x+this.descriptionArea.padding,
                "y": this.descriptionArea.y+this.descriptionArea.padding,
                "w": this.descriptionArea.w-(2*this.descriptionArea.padding),
                "h": this.descriptionArea.h-(2*this.descriptionArea.padding),
                "context": ctx,
                "fontWeight": "bold",
                "colour": "#444",
                align:'left'
            });
            if(this.checkButton) {
                this.checkButton.render({
                    x:this.descriptionArea.x+this.descriptionArea.w/4,
                    y:this.descriptionArea.y+this.descriptionArea.h+this.descriptionArea.padding,
                    w:this.descriptionArea.w/2,
                    h:this.descriptionArea.padding*1.5
                });
            }
            else {
                this.checkButton = new OU.util.Button({
                    layer: this.imageLayer,
                    txt:"Check your answer",
                    x:this.descriptionArea.x+this.descriptionArea.w/4,
                    y:this.descriptionArea.y+this.descriptionArea.h+this.descriptionArea.padding,
                    w:this.descriptionArea.w/2,
                    h:this.descriptionArea.padding*1.5,
                    onClick: function() {
                        self.view='feedback';
                        self.render();
                    }
                });
            }
        }
        else {
            matches=0;
            for ( i = 0; i < this.items.length; i++ ) {
                if(this.items[i].order==this.items[i].sortOrder+1)
                    matches++;
            }
            if(matches==this.items.length)
                feedback = this.data.matchedFeedback;
            else
                feedback = this.data.unmatchedFeedback;

            new OU.util.DynText( {
                "txt": feedback,
                "x" : this.descriptionArea.x+this.descriptionArea.padding,
                "y": this.descriptionArea.y+this.descriptionArea.padding,
                "w": this.descriptionArea.w-(2*this.descriptionArea.padding),
                "h": this.descriptionArea.h-(2*this.descriptionArea.padding),
                "context": ctx,
                "fontWeight": "bold",
                "colour": "#449",
                align:'left'
            });
        }
        shuffling = false;
        for (i = 0; i < this.items.length; i++ ) {
            // animate shuffle if out of place
            if(this.items[i].draggable.dragId == 0 && !this.items[i].hasFocus) { // item is not being dragged, so shuffle to appropriate position
                targetX=this.listArea.x+this.listArea.padding;
                targetY=this.listArea.y+this.listArea.padding +(this.items[i].sortOrder * this.listArea.itemHeight);

                if(this.items[i].x!=targetX || this.items[i].y!=targetY) {
                    shuffling=true;
                    newX = this.items[i].x + (targetX-this.items[i].x)/2;
                    newY = this.items[i].y + (targetY-this.items[i].y)/2;
                    if(Math.abs(targetX-newX)<0.2)
                        newX=targetX;
                    if(Math.abs(targetY-newY)<0.2)
                        newY=targetY;
                    this.items[i].move(newX,newY);
                }
            }
            else {
                shuffling=true;
            }
            // render item
            this.items[i].render();
        }

        if(!shuffling) {
            this.doRender=false;
        }
    };
    /**
     * @class
     * @extends OU.util.Tabbable
     */
    OU.activity.Queue.prototype.Item = function( params ) {
        this.queue = params.queue;
        this.events = params.events;
        this.id = params.id;
        this.order = params.order;
        this.sortOrder = params.sortOrder;
        this.name = params.name;
        this.color = params.color,
        this.x = params.x;
        this.y = params.y;
        this.h = params.h;
        this.w = params.w;
        this.startX = this.x;
        this.startY = this.y;
        OU.activity.Queue.prototype.Item.prototype.resize = function(p) {
            this.x = p.x;
            this.y = p.y;
            this.h = p.h;
            this.w = p.w;
            this.draggable.x = p.x;
            this.draggable.y = p.y;
            this.draggable.h = p.h;
            this.draggable.w = p.w;
            this.startX = this.x;
            this.startY = this.y;
        };
        OU.activity.Queue.prototype.Item.prototype.render = function( ) {
            var ctx = this.queue.imageLayer.context,sF=this.draggable.dragId!=0?1.2:1,
            x = this.x-(this.w*(sF-1)/2),
            y = this.y-(this.h*(sF-1)/2),
            w = this.w*sF,
            h = this.h*sF;
            new OU.util.DynText( {
                "txt": this.name,
                "x": x,
                "y": y,
                "w": w,
                "h": h,
                "context": ctx,
                background: {
                    "borderCol": "#444",
                    "col": this.hasFocus?"#c00":this.color,
                    "radius": h/2
                },
                fontWeight: "bold",
                fontSize: 18*sF
            } );
            ctx.gripPattern({
                x: w*.925+x-h/4,
                y:y+h/4,
                w: h/2,
                h: h/2,
                col:'#fff'
            });
            this.queue.imageLayer.events.clickable.push(this.draggable);
        };
        OU.activity.Queue.prototype.Item.prototype.startDrag = function( p ) {
            //sort the items so that the latest dragged is at the end of the queue, therefore render last next time (on top)
            p.me.queue.items.sort( function (a,b) {
                return a.draggable.dragId - b.draggable.dragId;
            });
            p.me.queue.view='norm';
            p.me.queue.doRender=true;
        };
        OU.activity.Queue.prototype.Item.prototype.endDrag = function( p ) {
            p.me.queue.doRender=true;
        };
        OU.activity.Queue.prototype.Item.prototype.newPos = function( p ) {
            p.me.x = p.x;
            p.me.y = p.y;

            // Shuffle sort order based on new position
            p.me.queue.items.sort( function (a,b) {
                return a.y - b.y;
            });
            for(var i=0; i<p.me.queue.items.length; i++)
                p.me.queue.items[i].sortOrder=i;
            p.me.queue.items.sort( function (a,b) {
                if(a.hasFocus)
                    return 1;
                return a.draggable.dragId - b.draggable.dragId;
            });
            p.me.queue.doRender=true;
        };
        OU.activity.Queue.prototype.Item.prototype.move = function( x,y ) {
            this.x=x;
            this.y=y;
            this.draggable.x=x;
            this.draggable.y=y;
        };
        this.draggable = new OU.util.Draggable( {
            "me":this,
            "events": this.queue.imageLayer.events,
            "x": this.x,
            "y": this.y,
            "h": this.h,
            "w": this.w,
            "onStart": this.startDrag,
            "onEnd": this.endDrag,
            "onMove": this.newPos
        } );
        OU.activity.Queue.prototype.Item.prototype.arrowKey = function (k) {
            switch(k){
                case KEY_UP:
                    this.move(this.x,this.y-10);
                    break;
                case KEY_DOWN:
                    this.move(this.x,this.y+10);
                    break;
                case KEY_LEFT:
                    this.move(this.x-10,this.y);
                    break;
                case KEY_RIGHT:
                    this.move(this.x+10,this.y);
                    break;
            }
            this.newPos({
                x:this.x,
                y:this.y,
                me:this
            });
            this.queue.view='norm';
            this.queue.doRender=true;
        };
        OU.activity.Queue.prototype.Item.prototype.focus = function () {
            this.hasFocus=true;
            this.queue.items.sort( function (a,b) {
                if(a.hasFocus)
                    return -1;
                return 0;
            });

            this.render();
        };
        OU.activity.Queue.prototype.Item.prototype.blur = function () {
            this.draggable.dragId=0;
            this.hasFocus=false;
            this.render();
        };
        OU.base(this, params);
    };
    OU.inherits(OU.activity.Queue.prototype.Item, OU.util.Tabbable);
    OU.activity.Queue.prototype.accessibleView = function() {
        clearInterval(this.renderCycle);

        var h = '<div id="accessibleView"><h1>'+this.data.title+'</h1><p>'+this.data.description+'</p>';

        for(var i=0; i<this.data.items.length; i++) {
            h += '<h2>'+this.data.items[i].name+'</h2>';
        }
        if(this.data.matchedFeedback != undefined) {
            h += "<h3>If the order is correct the feedback is:</h3>";
            if(typeof this.data.matchedFeedback == 'string') {
                h += '<p>'+this.data.matchedFeedback+'</p>';
            }
            else { // must be a string array
                for(var j=0; j<this.data.matchedFeedback.length; j++)
                    h += '<p>'+this.data.matchedFeedback[j]+'</p>';
            }
            h += "<h3>If the order is incorrect the feedback is:</h3>";
            if(typeof this.data.unmatchedFeedback == 'string') {
                h += '<p>'+this.data.unmatchedFeedback+'</p>';
            }
            else { // must be a string array
                for(var j=0; j<this.data.unmatchedFeedback.length; j++)
                    h += '<p>'+this.data.unmatchedFeedback[j]+'</p>';
            }
            // randomise order of the items
            this.data.items.sort( function(a,b) {
                return a.order-b.order;
            } );
            h += "<h3>The correct order is:</h3><p>";
            for(i=0; i<this.data.items.length; i++) {
                h += this.data.items[i].name+' : ';
            }
            h += '</p>';
        }
        else if(this.data.unmatchedFeedback != undefined) {
            h += "<h3>Feedback after your choice is:</h3>";
            if(typeof this.data.unmatchedFeedback == 'string') {
                h += '<p>'+this.data.unmatchedFeedback+'</p>';
            }
            else { // must be a string array
                for(var j=0; j<this.data.unmatchedFeedback.length; j++)
                    h += '<p>'+this.data.unmatchedFeedback[j]+'</p>';
            }
        }

        h += '</div>';
        document.body.innerHTML='';
        var accessible = document.createElement('div');
        accessible.innerHTML = h;
        document.body.appendChild(accessible);
    };

    OU.base(this,data,instance,controller);
};
OU.inherits(OU.activity.Queue,OU.util.Activity);
