/**
 * @fileOverview Add sound waves together and play
 *
 *  e.g. data:
 *
 *  html:"text",
 *  help:"text",
 *  answer:"text",
 ****
 *  type:"difference",
 *
 *  // top graph/sound
 *  freq:1000,
 *
 *  // bottom graph with sound range of frequencies set by a slider:
 *  minFreq:1001,
 *  maxFreq:1027,
 *
 *  showSum:true,
 *
 *  leftRight:true, // send audio to left or right channels
 *
 ****
 *  type: "harmonics",
 *
 *   fundamental:256,//frequency
 *   number:8,//no. harmonics with amplitudes to sum/display
 *
 ****
 *  type: "sum",
 *
 *   fundamental:100,//frequency
 *   number:3, // no. graphs to show (includes sum of previous)
 *
 ****
 * @author Will Rawes
 ****
 * RIFFWAVE js v0.02 - Audio encoder for HTML5 <audio>
 * elements. Copyright (C) 2011 Pedro Ladaria <pedro.ladaria
 * at Gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License version 2 as published by the Free
 * Software Foundation. The full license is available at
 * http://www.gnu.org/licenses/gpl.html
 *
 * This program is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE. See the GNU General Public License
 * for more details.
 *
 *
 * Changelog:
 *
 * 0.01 - First release 0.02 - New faster base64 encoding
 ****
 */

OU.require('OU.util.Layer');
OU.require('OU.util.Button');
OU.require('OU.util.Slider');
OU.require('OU.util.Div');
/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.Sounds = function ( data, instance, controller ) {
    /**
     *
     */
    OU.activity.Sounds.prototype.canvasView = function () {
        var i,j,self=this;
        OU.slf=this;
        OU.loadCSS("data/style.css");

        this.data.functions= [
            {
                id: "setSldr",
                fn: function(self,nm, v){
                    var n=nm.replace("A","");
                    self["a"+n]=v;
                    var aud=document.getElementById('aud'+n);
                    if(aud)aud.volume=v;
                    self[nm].render();
                    self[nm+"Div"].div.innerHTML="<span style='display:block;-webkit-transform:rotate(-90deg);-moz-transform:rotate(-90deg);-o-transform:rotate(-90deg);-ms-transform:rotate(-90deg);transform:rotate(-90deg);position:relative;left:-130%;width:550%;top:-650%;l:"
                        +(n%2 ? 50 : 50)
                        +"%;font-size:70%;'>"
                        +(n==1?"Fundamental":(""+(n)+"<sup>"+((n==2?"nd":(n==3?"rd":"th")))+"</sup> harmonic"))+" "
                        +((v*100+0.5)|0)+"%</span>";
                    self["draw"](self);
                }
            },
            {id: "setFreq", fn: function(self,f){
                self.dFreq=f;
                self.draw(self);
            }
            },
            {id: "render", fn: function(self){
                self.draw(self);
            }
            }
        ];
        switch(self.data.type){

            case "difference":
                this.data.canvas= [{id: "c1", x: 0.02, y: 0.01, w: 0.7, h: 0.4}];
                if(this.data.help){
                    this.data.help=this.data.help.replace(/__FREQ__/g,this.data.freq).replace(new RegExp("__MINFREQ__","g"),this.data.minFreq)+"<br/><br/>Press the 'Note frequency' button when you can hear a difference between both sounds.";
                }
                this.data.divs= [
                    {id: "audioTags", html: "", x: 0.66, y: 0.3, w: 0.5},
                    {id: "h2",html: "",x: 0.04, y: 0.5, w: 1}
                ];
                self.draw=function(self){
                    var w=self.c1.w/3,x0=w*0.2,i,j,k
                    ,a=[],f=[]
                    ,sum=self.data.showSum;
                    if(sum){
                        a=[[1],[1],[1,1]];
                        f=[[self.data.freq],[self.dFreq],[self.data.freq,self.dFreq]];
                    }else{
                        a=[[1],[1]];
                        f=[[self.data.freq],[self.dFreq]];
                    }

                    var h=self.c1.h/(sum?5:(self.data.leftRight?4.5:4))
                    , g=self.c1.g,v
                    , y0=h
                    , x, xs, imx=500, ic,di,dj;
                    self.c1.clear();
                    g.fillStyle='#000';
                    g.font=((h/4)|0)+'px'+' sans-serif';
                    h/=a.length;
                    for(i=0;i<a.length;i++){
                        sum=(i==2);
                        ic=0;
                        g.fillText((sum?"Sum":""+f[i][0]+" Hz"),2,y0-h*1.3);
                        x=x0;
                        if(sum){
							g.strokeStyle='#aaa';
                            g.lineWidth=0.5;
                            di=1/120;
                            imx=2000;
                            xs=w/165/4;
                        }else{
							g.strokeStyle='#000';
                            g.lineWidth=1;
                            di=1/6000;
                            imx=500;
                            xs=w/165;
                        };
						g.beginPath();
                        g.moveTo(x,y0);
                        for(j=0;j<imx;j++){
                            ic+=di;
                            v=0;
                            for(k=a[i].length;k--;)v+=a[i][k]*Math.sin(f[i][k]*ic);
                            g.lineTo(x,y0+h*v);
                            x+=xs;
                        }
                        g.stroke();
                        g.closePath();
                        x=x0;
                        g.textAlign='center';
						g.strokeStyle='#000';
						g.beginPath();
                        g.moveTo(x,y0);
						imx=500;
						xs=w/165;
						dj=75;
                        for(j=0;j<imx-dj;j+=dj){
                            g.lineTo(x,y0);
                            g.lineTo(x,y0+h/10);
                            g.moveTo(x,y0);
                            g.fillText(""+j*(sum?400:2)/75,x,y0+h/2.3);
                            x+=75*xs;
                        }
                        g.lineTo(x,y0);
                        g.stroke();
                        g.closePath();
                        g.textAlign='right';
                        g.fillText("ms",x,y0+h/2.3);
                        g.textAlign='left';
                        y0+=h*4;
                    }
                };
                this.showDif=function(self){
                    var s=self.h2.div.innerHTML.replace(/\<br\>/g,"<br/>")
                    ,p=s.indexOf("<br/><br/>Difference heard at ");
                    if(p>0)s=s.substring(0,p);
                    if(OU.LocalStorage.load("notedFreq"+self.data.freq))s+="<br/><br/>Difference heard at "+OU.LocalStorage.load("notedFreq"+self.data.freq)+" Hz.";
                    self.h2.div.innerHTML=s;
                }
                this.playNote=function(self,freq,btn,side){
                    var i,j;
                    if(navigator.appName.indexOf("icro")>0){
                        if(btn.params.txt!="Pause"){
                            var aud1=document.getElementById('aud1'),aud2;
                            aud1.volume=1;
                            if(freq[1] || (side==3)){
                                aud2=document.getElementById('aud2');
                                aud2.volume=1;
                            }
                            btn.params.txt="Pause";
                            aud1.oncanplaythrough=function(){
                                aud1.play();
                                if(aud2)aud2.play();
                                setTimeout(function(){
                                    btn.params.txt="Play";
                                    btn.render();
                                },5000);
                                btn.params.txt="Pause";
                                btn.render();
                            }
                            var f1,f2;
                            if(side==1){
                               f1=freq[0]+"L";
                            }else if(side==2){
                               f1=freq[0]+"R";
                            }else if(side==3){
                               f1=freq[0]+"L";
                               f2=freq[1]+"R";
                            }else{
                                f1=freq[0];
                                f2=freq[1];
                            }
                            if(aud2)aud2.src="data/audio/"+f2+".mp3";
                            aud1.src="data/audio/"+f1+".mp3";
                        }else{
                            btn.params.txt="Play";
                            document.getElementById('aud1').pause();
                            document.getElementById('aud2').pause();
                        }
                        btn.render();
                    }else{
                        var sam=[], t=0, n=freq.length
                        , f=[freq[0]*self.pitch*2*Math.PI];
                        if(freq[1])f[1]=freq[1]*self.pitch*2*Math.PI;
                        var dt=1/44100., l=5/dt // 5 seconds (44.1 KHz)
                        ,di=1,v
                        ,mx=-1e9, mn=-mx,i2
                        ,side1,side2;
                        if(side){
                            side1=side;
                            side2=2-side;
                            l*=2;
                            di=2;
                        }
                        if(side==1){
                            for(i=l;i-=di;){
                                t+=dt;v=Math.sin(f[0]*t);
                                sam[i]=v;
                                sam[i+1]=128;
                            }
                            mx=1;mn=-1;
                        }else if(side==2){
                            for(i=l;i-=di;){
                                t+=dt;v=Math.sin(f[0]*t);
                                sam[i]=128;
                                sam[i+1]=v;
                            }
                            mx=1;mn=-1;
                        }else if(side==3){
                            for(i=l;i-=di;){
                                t+=dt;
                                sam[i]=Math.sin(f[0]*t);
                                sam[i+1]=Math.sin(f[1]*t);
                            }
                            mx=1;mn=-1;
                            di=1;
                        }else{
                            for(i=l;i-=di;){
                                t+=dt;v=0;for(j=n;j--;)v+=Math.sin(f[j]*t);if(v>mx)mx=v;if(v<mn)mn=v;
                                sam[i]=v;
                            }
                        }
                        var df=250/(mx-mn+0.01);
                        for(i=(side==2?1:0);i<l;i+=di)sam[i]=parseInt((sam[i]-mn)*df);// Normalise wav
                        if(!self.wave)self.wave=new self.RIFFWAVE();
                        self.wave.header.sampleRate=44100; // 44.1Khz (1 second)
                        self.wave.header.numChannels=(side?2:1); // 1 channel (mono)
                        self.wave.Make(sam);
                        self.RIFFAudio.src=self.wave.dataURI;
                        self.RIFFAudio.play();
                    }
                }
                this.data.sliders= [
                    {name: "fSlider", x: 0.01, y: 0.42, h: 0.05, w: 0.9, sliderPos: 0,
                        orientation: "horizontal", action: function(v,self){
                            var df=(v*(self.data.maxFreq-self.data.minFreq)|0)
                            f=self.data.minFreq+(self.data.df?((df/self.data.df)|0)*self.data.df:df);
                            self["fSliderDiv"].div.innerHTML="";
                            self.setFreq(self,f);
                            this.render();
                        }
                    }
                ];
                this.data.buttons= [
                    {txt: "Play", x: 0.72, y: 0.05
                    , action: function(self){
                        self.playNote(self,[self.data.freq],this,(self.data.leftRight?1:null));
                        this.render();
                    }
                    },
                    {txt: "Play", x: 0.72, y: (self.data.showSum?0.16:0.25)
                    , action: function(self){
                        self.playNote(self,[self.dFreq],this,(self.data.leftRight?2:null));
                        this.render();
                    }
                    },
                    {txt: "Note frequency", x: 0.72, y: 0.77
                    , action: function(self){
                        OU.LocalStorage.save("notedFreq"+self.data.freq,""+self.dFreq);
                        self.showDif(self);
                        this.render();
                    }
                    }
                ];
                if(this.data.showSum||this.data.leftRight){
                    this.data.buttons[this.data.buttons.length]={txt: "Play", x: 0.72, y: (self.data.leftRight?0.34:0.27)
                    , action: function(self){
                        self.playNote(self,[self.data.freq,self.dFreq],this,(self.data.leftRight?3:null));
                        this.render();
                    }
                    };
                }
                this.initFunction=function(self){
                    self.fSliderDiv.div.innerHTML="";
                    self.fSlider.sliderPos=0;
                    self.pitch=1;
                    self.audioTags.div.innerHTML="<audio id='aud1' ></audio><audio id='aud2' ></audio>";
                    self.dFreq=this.data.minFreq;
                    self.amps=[[1,0],[0,1]];//Amplitudes
                    self.freqs=[this.data.freq,self.dFreq];//Frequencies
                    self.h2.div.innerHTML=this.data.html;
                    self.showDif(self);
                }
            break;

            case "harmonics":
                this.data.canvas= [{id: "c1", x: 0.53, y: 0.01, w: 0.45, h: 0.45}];
                var sel="";
                if(this.data.sounds){
                    sel="<select onchange='OU.slf.setSound(OU.slf,this);' style='font-size:90%;height:400%;'>";
                    for(var i=0;i<this.data.sounds.length;i++)sel+="<option value='"+i+"'>"+this.data.sounds[i].name+"</option>";
                    this.setSound=function(self, t){
                        var k=parseInt(t.value);
                        var amps=self.data.sounds[k].amps;
                        self.pitch=self.data.sounds[k].pitch;
                        for( var i=amps.length;i--;){
                            var s=self["A"+(i+1)];
                            s.sliderPos=amps[i];
                            self.setSldr(self,"A"+(i+1),amps[i]);
                        }
                    }
                    sel+="</select><div>&#160;</div>";
                }
                this.data.divs= [
                    {id: "audioTags", html: "", x: 0.6, y: 0.3, w: 0.5},
                    {id: "h2",html:sel+this.data.html,x: 0.04, y: 0.56, w: 1}
                ];
                this.initFunction=function(self){
                    var f=self.data.fundamental, tgs="",i;
                    for(i=1;i<=self.data.number;i++){
                        self["f"+i]=f;
                        self["a"+i]=0.5;
                        tgs+="<audio id='aud"+i+"'><source src='data/audio/"+f
                                +".mp3'/><source src='data/audio/"+f
                                +".ogg'/></audio><br/>";
                        f+=self.data.fundamental;
                        self["A"+i].animate=function(){ // switch off animation for speed
                            if(this.sliderPos!=this.target){
                                this.sliderPos=this.target;
                                this.callback(this.sliderPos,this.callbackParam);
                            }
                        };
                        self.setSldr(self,"A"+i,0.5);
                    }
                    self.audioTags.div.innerHTML=tgs;
                    self.pitch=1;
                };
                this.draw=function(self){
                    var w=self.c1.w/3, h=self.c1.h/18
                    ,g=self.c1.g, i, j, v
                    , y0=w*1.5, xs=w/160, imx=500, ic=imx/8000, di=1/9000;
                    self.c1.clear();
                    g.strokeStyle='#000';
                    g.beginPath();
                    g.lineWith=0.7;
                    g.moveTo(w*3,y0);
                    for(i=imx;i--;){
                        v=0;
                        for(j=1;j<=self.data.number;j++)v+=self["a"+j]*Math.sin(self["f"+j]*ic);
                        ic-=di;
                        g.lineTo(i*xs,y0+h*v);
                    }
                    g.stroke();

                };
                this.playNote=function(self){
                    var a=[], i, j;
                    for(i=self.data.number;i--;)a[i]=127/12*self["a"+(i+1)];
                    var sam=[], t=0, n=a.length, f=440*self.pitch*Math.PI;
                    var l=3*44100// 1 second (44.1 KHz)
                    , dt=1/44100.*f, mx=-1e9, mn=-mx;
                    for(i=l;i--;){
                        t+=dt;
                        sam[i]=0;
                        for(j=n;j--;)sam[i]+=a[j]*Math.sin(j*t+t);
                        if(sam[i]>mx)mx=sam[i];
                        if(sam[i]<mn)mn=sam[i];
                    }
                    var df=250/(mx-mn+0.01);
                    for(i=l;i--;)sam[i]=parseInt((sam[i]-mn)*df);// Normalise wav

                    if(!self.wave)self.wave=new self.RIFFWAVE();
                    self.wave.header.sampleRate=44100; // 44.1Khz (1 second)
                    self.wave.header.numChannels=1; // 1 channel (mono)
                    self.wave.Make(sam);
                    self.RIFFAudio.src=self.wave.dataURI;
                    self.RIFFAudio.play();
                }
                this.data.buttons=[
                    {txt: "Play", x: 0.585, y: 0.5
                    , action: function(self){
                        if(navigator.appName.indexOf("icro")<0)
                            self.playNote(self,[0.1, 0.2, 0.5, 0.1])
                        else{
                            if(this.params.txt!="Pause"){
                                this.params.txt="Pause";
                                for( var i=self.data.number;i--;)document.getElementById('aud'+(i+1)).play();
                            }else{
                                this.params.txt="Play";
                                for( var i=self.data.number;i--;)document.getElementById('aud'+(i+1)).pause();
                            }
                        }
                        this.render();
                    }
                    }
                ];
                this.data.sliders= [];
                var w=0.5/self.data.number;
                for(var i=0;i<self.data.number;i++){
                    this.data.sliders[i]={name: "A"+(i+1), x: 0.01+i*w, y: 0.18, h: 0.35, w: 0.037, sliderPos: 0.5,
                        orientation: "vertical", action: function(v,self){
                            self.setSldr(self,this.name,v);
                        }
                    }
                }
            break;

            case "sum":
                this.data.canvas= [{id: "c1", x: 0.01, y: 0.01, w: 0.7, h: 0.5}];
                var sel="";
                this.data.divs= [
                    {id: "audioTags", html: "", x: 0.6, y: 0.3, w: 0.5},
                    {id: "h2",html:this.data.html,x: 0.04, y: 0.56, w: 1}
                ];
                this.initFunction=function(self){
                    var f=self.data.fundamental, tgs="",i;
                    for(i=1;i<=self.data.number;i++){
                        self["f"+i]=f;
                        self["a"+i]=0.5;
                        tgs+="<audio id='aud"+i+"'><source src='data/audio/"+f
                                +".mp3'/><source src='data/audio/"+f
                                +".ogg'/></audio><br/>";
                        f+=self.data.fundamental;
                    }
                    self.audioTags.div.innerHTML=tgs;
                    self.pitch=1;
                };
                this.draw=function(self){
                    var n=self.data.number
                    ,w=self.c1.w/3, h=self.c1.h/1.5/n
                    ,g=self.c1.g, i, j, k, v
                    , y0=h*0.9, xs=w/160, imx=500, ic=imx/8000, di=1/9000;
                    self.c1.clear();
                    g.fillStyle='#000';
                    g.font=((w/10)|0)+'px'+' sans-serif';
                    g.strokeStyle='#000';
                    g.beginPath();
                    g.lineWith=1;
                    for(k=0;k<n;k++){
                        g.fillText(""+(k==n-1 && n!=1?"Sum":(k+1)*self.data.fundamental+" Hz"),0,y0-h/2);
                        g.moveTo(w*3,y0);
                        for(i=imx;i--;){
                            v=0;
                            if(k==n-1)for(j=1;j<=n;j++)v+=0.5*Math.sin(self["f"+j]*ic)/n;
                            else v=0.5*Math.sin(self["f"+(k+1)]*ic);
                            ic-=di;
                            g.lineTo(i*xs,y0+h*v);
                        }
                        y0+=h*1.5;
                    }
                    g.stroke();

                };
                this.playNote=function(self,a){
                    var i, j;
                    var sam=[], t=0, n=a.length, f=self.data.fundamental*self.pitch*2*Math.PI;
                    var l=3*44100// 1 second (44.1 KHz)
                    , dt=1/44100.*f, mx=-1e9, mn=-mx;
                    for(i=l;i--;){
                        t+=dt;
                        sam[i]=0;
                        for(j=n;j--;)sam[i]+=a[j]*Math.sin(j*t+t);
                        if(sam[i]>mx)mx=sam[i];
                        if(sam[i]<mn)mn=sam[i];
                    }
                    var df=250/(mx-mn+0.01);
                    for(i=l;i--;)sam[i]=parseInt((sam[i]-mn)*df);// Normalise wav

                    if(!self.wave)self.wave=new self.RIFFWAVE();
                    self.wave.header.sampleRate=44100; // 44.1Khz (1 second)
                    self.wave.header.numChannels=1; // 1 channel (mono)
                    self.wave.Make(sam);
                    self.RIFFAudio.src=self.wave.dataURI;
                    self.RIFFAudio.play();
                }
                this.data.buttons=[];
                for(var i=0;i<this.data.number;i++){
                    this.data.buttons[i]={
                        txt: "Play", x: 0.75, y: 0.07+i*(0.5/this.data.number)
                        , action: function(self){
                            var amps=[];
                            if(self.data.number==1)amps=[1];
                            else for(var i=0;i<self.data.number-1;i++)amps[i]=(i==this.num?1:(this.num==self.data.number-1?1:0));

                            if(navigator.appName.indexOf("icro")<0)self.playNote(self,amps);
                            else{
                                if(this.params.txt!="Pause"){
                                    this.params.txt="Pause";
                                    var aud,i,n=self.data.number-1;
                                    if(n==0)n=1;
                                    for(i=n;i--;){
                                        aud=document.getElementById('aud'+(i+1));
                                        aud.volume=amps[i];
                                        aud.play();
                                    }
                                    var btn=this;
                                    setTimeout(function(){btn.params.txt="Play";btn.render();},3000);
                                }else{
                                    this.params.txt="Play";
                                    var n=self.data.number-1;
                                    if(n==0)n=1;
                                    for(var i=n;i--;)document.getElementById('aud'+(i+1)).pause();
                                }
                            }
                            this.render();
                        }
                    };
                }
                this.data.sliders= [];
            break;

        }
        if(this.data.answer){
            this.data.buttons[this.data.buttons.length]={txt: "Answer", x: 0.72, y: 0.72
                , action: function(){
                    OU.alert(self.data.answer);
                    this.render();
                }
            }
        }
        if(this.data.help){
            if(!OU.helpB){
                OU.helpB=document.createElement("a");
                document.body.appendChild(OU.helpB);
            }
            OU.helpB.setAttribute("class",'infoButton');
            OU.helpB.setAttribute("tabIndex",1);
            OU.helpB.setAttribute("href","javascript:OU.alert(self.data.help);");
            OU.helpB.innerHTML="<b><i>&#160;&#160;i&#160;&#160;</i></b>";
            OU.helpB.style.display='block';
/*            this.data.buttons[this.data.buttons.length]={txt: "?", x: 0.95, y: 0.001
                , action: function(){
                    OU.alert(self.data.help);
                    this.render();
                }
            }//*/
        }else if(OU.helpB)OU.helpB.style.display='none';

        self.FastBase64={
            chars: "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=",
            encLookup: [],
            Init: function(){
                for( var i=0;i<4096;i++)this.encLookup[i]=this.chars[i>>6]+this.chars[i&0x3F];
            },
            Encode: function(src){
                var len=src.length;
                var dst='';
                var i=0;
                while(len>2){
                    n=(src[i]<<16)|(src[i+1]<<8)|src[i+2];
                    dst+=this.encLookup[n>>12]+this.encLookup[n&0xFFF];
                    len-=3;
                    i+=3;
                }
                if(len>0){
                    var n1=(src[i]&0xFC)>>2;
                    var n2=(src[i]&0x03)<<4;
                    if(len>1)n2|=(src[++i]&0xF0)>>4;
                    dst+=this.chars[n1];
                    dst+=this.chars[n2];
                    if(len==2){
                        var n3=(src[i++]&0x0F)<<2;
                        n3|=(src[i]&0xC0)>>6;
                        dst+=this.chars[n3];
                    }
                    if(len==1)dst+='=';
                    dst+='=';
                }
                return dst;
            } // end Encode
        }

        self.FastBase64.Init();

        self.RIFFWAVE=function(data){

            this.data=[]; // Byte array containing audio samples
            this.wav=[]; // Array containing the generated wave
            // file
            this.dataURI=''; // http://en.wikipedia.org/wiki/Data_URI_scheme

            this.header={ // OFFS SIZE NOTES
            chunkId: [0x52, 0x49, 0x46, 0x46], // 0 4 "RIFF" =
            // 0x52494646
            chunkSize: 0, // 4 4 36+SubChunk2Size =
            // 4+(8+SubChunk1Size)+(8+SubChunk2Size)
            format: [0x57, 0x41, 0x56, 0x45], // 8 4 "WAVE" =
            // 0x57415645
            subChunk1Id: [0x66, 0x6d, 0x74, 0x20], // 12 4 "fmt " =
            // 0x666d7420
            subChunk1Size: 16, // 16 4 16 for PCM
            audioFormat: 1, // 20 2 PCM = 1
            numChannels: 1, // 22 2 Mono = 1, Stereo = 2, etc.
            sampleRate: 8000, // 24 4 8000, 44100, etc
            byteRate: 0, // 28 4
            // SampleRate*NumChannels*BitsPerSample/8
            blockAlign: 0, // 32 2 NumChannels*BitsPerSample/8
            bitsPerSample: 8, // 34 2 8 bits = 8, 16 bits = 16,
            // etc...
            subChunk2Id: [0x64, 0x61, 0x74, 0x61], // 36 4 "data" =
            // 0x64617461
            subChunk2Size: 0
            // 40 4 data size =
            // NumSamples*NumChannels*BitsPerSample/8
            };

            function u32ToArray(i){
                return [i&0xFF, (i>>8)&0xFF, (i>>16)&0xFF,(i>>24)&0xFF];
            }

            function u16ToArray(i){
                return [i&0xFF, (i>>8)&0xFF];
            }

            this.Make=function(data){
                if(data instanceof Array)this.data=data;
                this.header.byteRate=(this.header.sampleRate*this.header.numChannels*this.header.bitsPerSample)>>3;
                this.header.blockAlign=(this.header.numChannels*this.header.bitsPerSample)>>3;
                this.header.subChunk2Size=this.data.length;
                this.header.chunkSize=36+this.header.subChunk2Size;

                this.wav=this.header.chunkId.concat(
                    u32ToArray(this.header.chunkSize),
                    this.header.format,this.header.subChunk1Id,
                    u32ToArray(this.header.subChunk1Size),
                    u16ToArray(this.header.audioFormat),
                    u16ToArray(this.header.numChannels),
                    u32ToArray(this.header.sampleRate),
                    u32ToArray(this.header.byteRate),
                    u16ToArray(this.header.blockAlign),
                    u16ToArray(this.header.bitsPerSample),
                    this.header.subChunk2Id,
                    u32ToArray(this.header.subChunk2Size),
                    this.data
                );
                this.dataURI='data:audio/wav;base64,'+self.FastBase64.Encode(this.wav);
            };
            if(data instanceof Array)this.Make(data);
        }; // end RIFFWAVE

        if(!self.RIFFAudio)self.RIFFAudio=new Audio();

        this.begoneWrong = false;
        this.bgLayer = new OU.util.Layer({
            container:this,
            hasEvents:true
        });
        this.bgLayer.context.gradRect(); // draw background on backdrop layer
        this.cCan=new OU.util.Layer({
            container:this,
            hasEvents:true
        });
        this.margin = (this.data.margin || 20)*OU.dpr;
        this.margin += 20*OU.dpr;
        this.cans=[];
        this.cans["bCan"]=this.bgLayer;
        this["bCan"]=this.cans["bCan"];
        for (i=this.data.functions.length;i--;){
            var nm=this.data.functions[i].id;
//            if(nm==="init")nm="initFunction";
            if(nm==="render")nm="renderFunction";
            if(nm==="timed")nm="timedFunction";
            this[nm]=this.data.functions[i].fn;
            if(nm==="timedFunction"){
                self.interval=this.data.functions[i].interval;
            }
        }
        this.divs=[];
        for (i=0;i<this.data.divs.length;i++) {
            this.divs[i] = new OU.util.Div({
                x:0.01,
                y:0.01,
                w:1,
                h:1,
                container:this,
                style:'overflow: visible;'
            });
            this.divs[i]._x=this.data.divs[i].x;
            this.divs[i]._y=this.data.divs[i].y;
            this.divs[i]._w=this.data.divs[i].w;
            this.divs[i]._h=(this.data.divs[i].h || 0.01);
            this.divs[i].div.innerHTML=this.data.divs[i].html;
            this.divs[i].setVisible=function(b){
                this.div.style.display=(b?"block":"none");
            }
            this[this.data.divs[i].id]=this.divs[i];
        }
        for (i=0;i<this.data.canvas.length;i++) {
            this.cans[i] = new OU.util.Layer({
                x:0.01,
                y:0.01,
                w:1,
                h:1,
                container:this,
                hasEvents:true
            });
            this.cans[i]._x=this.data.canvas[i].x;
            this.cans[i]._y=this.data.canvas[i].y;
            this.cans[i]._w=this.data.canvas[i].w;
            this.cans[i]._h=this.data.canvas[i].h;
            this.cans[i].g=this.cans[i].context;
            this.cans[i].vis=true;
            this.cans[i].setVisible=function(b){
                this.vis=b;
                this.canvas.style.display=(b?"block":"none");
            }
            this[this.data.canvas[i].id]=this.cans[i];
        }
        this.buttons = [];
        this.bis = [];
        for (i=0;i<this.data.buttons.length; i++) {
            this.buttons[i] = new OU.util.Button({
                txt:this.data.buttons[i].txt,
                verticalPadding:2,
                x:0.01,
                y:0.01,
                w:1,
                h:1,
                disabled:this.data.buttons[i].disabled,
                layer:this.cCan,
                container:this,
                background:this.bBack,
                ii:i,
                onClick:this.data.buttons[i].action,
                onClickParam:self
            });
            this.buttons[i].dontShow = false;
            this.buttons[i].id = this.data.buttons[i].txt;
            this.buttons[i].num = i;
            this[this.data.buttons[i].txt] = this.buttons[i];
            this.buttons[i]._x = this.data.buttons[i].x;
            this.buttons[i]._y = this.data.buttons[i].y;
            this.buttons[i].pageNum = this.data.buttons[i].pageNum || [];
        }
        this.sliders=[];
        for (i = 0; i < this.data.sliders.length; i++) {
            var actn=this.data.sliders[i].action;
            this.sliders[i] = new OU.util.Slider({
                container:this.cCan,
                instance:'',
                x:.1,
                y:.1,
                w:.1,
                h:.1,
                orientation:this.data.sliders[i].orientation,
                sliderHeight:.1,
                sliderPos:this.data.sliders[i].sliderPos||0.5,
                drawContainer:false,
                title:"",
                oshowValue:true,
                range:{min:0,max:100,nDecimals:1},
                valueUnits:" ",
                callback:this.data.sliders[i].action,
                callbackParam:self,
                background:{clear:true},
                context:this.cCan.context
            });
            this.sliders[i].ths=self;
            this.sliders[i]._x=this.data.sliders[i].x;
            this.sliders[i]._y=this.data.sliders[i].y;
            this.sliders[i]._w=this.data.sliders[i].w*0.9;
            this.sliders[i]._h=this.data.sliders[i].h;
            this.sliders[i].range.min=this.data.sliders[i].min;
            this.sliders[i].range.max=this.data.sliders[i].max;
            this.sliders[i].getValue=function(){return this.sliderPos*(this.range.max-this.range.min)+this.range.min;};
            this[this.data.sliders[i].name]=this.sliders[i];
            this.sliders[i].name=this.data.sliders[i].name;
            this.cCan.events.clickable.push(this.sliders[i]);
            var j=this.data.divs.length+i;
            this.divs[j] = new OU.util.Div({
                x:0.01,
                y:0.01,
                w:0.1,
                h:0.1,
                container:this,
                style:'text-align:center;overflow: visible;'
            });
            this.divs[j]._x=this.data.sliders[i].x;
            this.divs[j]._w=this.data.sliders[i].w/1.2;
            this.divs[j]._h=0.01;
            this.divs[j].div.innerHTML=this.data.sliders[i].name;
            this[this.data.sliders[i].name+"Div"]=this.divs[j];
        }
        if(this["timedFunction"]){
            if(!OU.isRunning)setInterval(
                function(){
                    if(OU.isRunning && self.timedFunction)self.timedFunction();
                }
            ,self.interval || 80);

        }
        if(this["initFunction"])this["initFunction"](this);

        self.showDiv=function(dv,b){
            if(self.dv)self.dv.div.style.display=(b?"block":"none");
        }
        self.error=[];
        this.resize();
    };


    /**
     *
     */
    OU.activity.Sounds.prototype.render = function () {
//        OU.activity.Sounds.superClass_.render.call(this);
        for(var i=this.buttons.length;i--;)this.buttons[i].render();
    }
    /**
     *
     */
    OU.activity.Sounds.prototype.remove = function () {
        OU.activity.Sounds.superClass_.remove.call(this); // call the parent class resize
        if(OU.alertDiv){
            document.body.removeChild(OU.alertDiv);
            OU.alertDiv=null;
        }
    }
    OU.activity.Sounds.prototype.resize = function () {
        OU.activity.Sounds.superClass_.resize.call(this); // call the parent class resize

        var self=this,scl=window.innerWidth*0.95
        ,i,j,bpn;
        if(scl>window.innerHeight*1.02)scl=window.innerHeight;
        self.scale=scl;

        if(!this.bgLayer)return;
        this.bgLayer.resize({x:0,y:this.margin*1.8,w:this.w,h:this.h});
        this.bgLayer.context.gradRect();

        for (i = this.cans.length; i--;) {
            if(this.cans[i].vis)this.cans[i].resize({
                x:this.cans[i]._x*scl+this.margin*1,
                y:this.cans[i]._y*scl+this.margin*2,
                w:this.cans[i]._w*scl,
                h:this.cans[i]._h*scl,
                container:this
            });
        }
        this.cCan.resize();

        for (i = 0;i<this.divs.length; i++) {
            if(this.divs[i]){
                this.divs[i].resize({
                    x:this.divs[i]._x*scl+this.margin*0,
                    y:this.divs[i]._y*scl+this.margin*2,
                    w:this.divs[i]._w*scl,
                    h:this.divs[i]._h*scl,
                    container:this
                });
                this.divs[i].div.style.fontSize=""+((scl/44)|0)+"px";
            }
        }

        for (i = this.sliders.length; i--;) {
            this.sliders[i].resize({
                x:this.sliders[i]._x*scl+this.margin,
                y:this.sliders[i]._y*scl+this.margin*0,
                w:this.sliders[i]._w*scl,
                h:this.sliders[i]._h*scl,
                container:this["c1"]
            });
            this.sliders[i].render();
            var j = i+this.data.divs.length;
            if(this.divs[j]){
                this.divs[j].resize({
                    x:this.divs[j]._x*scl+this.margin*0.5,
                    y:this.margin*2+parseInt(this.sliders[i].y)-(this.data.sliders[i].orientation=="vertical"?this.sliders[i].w*1.2:this.sliders[i].h/2.3),
                    w:this.divs[j]._w*scl,
                    h:this.divs[j]._h*scl,
                    container:this
                });
                this.divs[j].div.style.fontSize=""+((scl/44)|0)+"px";
            }
        }

        for (i = this.buttons.length; i--;) {
            for (j = this.buttons[i].pageNum.length; j--;)bpn=(this.pageNum===this.buttons[i].pageNum[j]);
            bpn = true;
            if (this.buttons[i].dontShow===true)bpn = false;
            if (bpn) {
                pos = {x:this.buttons[i]._x, y:this.buttons[i]._y};
                this.buttons[i].resize({
                    x:pos.x*scl+this.margin,
                    y:pos.y*scl+this.margin*0,
                    h:scl *(this.data.btnScale||1)*0.05,
                    w:scl *((this.data.btnScale||1)*this.buttons[i].id.length / 40 )
                    ,container:this
                });
            }else{
                this.buttons[i].resize({
                    x:1,
                    y:-500, //this.buttons[i]._y * s,
                    h:scl / 31,
                    w:scl / 10 * this.buttons[i].id.length / 1 + 18
                    ,container:this
                });
            }
        }
        OU.alert=function(s,cnf){
            s=s.replace(/\n/g,"<br/>");
            if(!OU.alertDiv){
                OU.alertDiv=document.createElement("div");
                document.body.appendChild(OU.alertDiv);
                OU.alertDiv.setAttribute("style","position:absolute;padding:20px;border:solid 1px #000;border-radius:4px;top:20%;left:20%;width:50%;background:#fff;font-size:100%;z-index:9999999;");
            }
            OU.alertDiv.style.display='block';
            OU.alertDiv.innerHTML="<p style='font-size:90%;'>"+s+"</p><br/><p style='text-align:center;'><br/>"
            +(cnf?"<a href='#' onclick='OU.alertDiv.style.display=\"none\";OU.confirmFn();' class='btn'>&#160;&#160;OK&#160;&#160;</a>&#160;&#160;&#160;"
            +"<a href='#' onclick='OU.alertDiv.style.display=\"none\";' class='btn'>&#160;&#160;Cancel&#160;&#160;</a> "
            :
            "<a href='#' onclick='OU.alertDiv.style.display=\"none\";' class='btn'>&#160;&#160;OK&#160;&#160;</a> ")
            +"</p>";
        }
        OU.confirm=function(s,fn){
            OU.confirmFn=fn;
            OU.alert(s,true);
        }
        if(this.renderFunction)this.renderFunction(self);
        this.render();

    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.Sounds, OU.util.Activity);
