/**
 * @fileOverview Time Graph
 *
 * @author Nigel Clarke <nigel.clarke@pentahedra.com>
 */

OU.require('OU.util.DynText');
OU.require('OU.util.PopUpInfo');
OU.require('OU.util.Draggable');
OU.require('OU.util.ViewSlider');
OU.require('OU.util.Layer');
OU.require('OU.util.ImageLoader');
/**
 * @class
 * @extends OU.util.Activity
 */
OU.activity.Timegraph = function ( data, instance, controller ) {
    OU.activity.Timegraph.prototype.canvasView = function () {
        this.config = {
            imageLoadTimeout:60000, // 60 seconds
            fps:40
        };
        this.bgLayer = new OU.util.Layer({
            container:this
        });
        this.graphLayer = new OU.util.Layer({
            container:this,
            hasEvents:true
        });
        this.controlLayer = new OU.util.Layer({
            container:this,
            y:this.h * .9,
            h:this.h * .1,
            hasEvents:true
        });
        this.initConstants();
        this.initData();
    };
    OU.activity.Timegraph.prototype.resize = function () {
        OU.activity.Timegraph.superClass_.resize.call(this); // call the parent class resize 
        this.cacheX = undefined;
        this.init();
    };
    OU.activity.Timegraph.prototype.start = function () {
        this.eventLayer = new OU.util.Layer({
            container:this,
            hasEvents:true,
            y:this.data.graphY,
            h:this.h * .9 - this.data.graphY
        });
        this.renderBackdrop();
        this.renderControls(this.controlLayer);
        this.graphLayer.events.moveRight = this.slideRight;
        this.graphLayer.events.moveLeft = this.slideLeft;
        this.render();
    };
    OU.activity.Timegraph.prototype.renderControls = function ( layer ) {
        var ctx = layer.context, events = layer.events;
        this.vpH = this.h * .9 - this.data.graphY;
        this.graphH = this.vpH;
        // init slider
        this.slider = new OU.util.ViewSlider({
            "context":ctx,
            events:events,
            "x":0,
            "y":0,
            "w":this.w,
            "h":this.h * .1,
            map:{
                w:this.range,
                h:this.vpH,
                vpW:this.w,
                vpH:this.vpH
            },
            background:{
                clear:true
            },
            "callback":this.move,
            container:this
        });
        this.dragEvents = new OU.util.Draggable({
            "me":this,
            "events":this.eventLayer.events,
            "x":0,
            "y":this.data.graphY,
            "h":this.vpH,
            "w":this.range,
            "onStart":this.startDrag,
            "onEnd":this.endDrag,
            "onMove":this.moveDrag
        });
    };
    OU.activity.Timegraph.prototype.startDrag = function () {
    };
    OU.activity.Timegraph.prototype.endDrag = function () {
    };
    OU.activity.Timegraph.prototype.moveDrag = function ( offsets ) { // Called by a drag event
        var graph = offsets.me;
        graph.graphX = offsets.x;
        graph.graphY = offsets.y - graph.data.graphY;
        if (graph.graphX > 0)
            graph.graphX = 0;
        if (graph.graphX < -(graph.range - graph.w))
            graph.graphX = -(graph.range - graph.w);
        if (graph.graphY > 0)
            graph.graphY = 0;
        if (graph.graphY < -(graph.graphH - graph.vpH))
            graph.graphY = -(graph.graphH - graph.vpH);
        graph.dragEvents.x = graph.graphX;
        graph.dragEvents.y = graph.graphY + graph.data.graphY;
        graph.slider.render({
            x:-graph.graphX,
            y:-graph.graphY
        });
        graph.render();
    };
    OU.activity.Timegraph.prototype.move = function ( offsets ) { // Called by the Slider
        var graph = offsets.me;
        graph.graphX = -offsets.x;
        graph.graphY = -offsets.y;
        graph.dragEvents.x = -offsets.x;
        graph.dragEvents.y = -offsets.y + graph.data.graphY;
        graph.render();
    };
    OU.activity.Timegraph.prototype.renderBackdrop = function () {
        var ctx = this.bgLayer.context, title = this.data.title, numBars = this.data.numBars, barH = this.data.barH, i,
            bCols = ['#16a3f4', '#9fc2d7', '#6db7e1', '#4a9ccb'];
        ctx.save();
        ctx.gradRect();
        new OU.util.DynText({
            "txt":title ? title : "",
            "x":0,
            "y":0,
            "w":this.w,
            "h":this.h * .1,
            "context":ctx,
            "propTextHeight":0.6
        });
        for (i = numBars; i--;) {
            ctx.fillStyle = bCols[numBars - i - 1];
            ctx.fillRect(0, this.h * .1 + (i * barH), this.w, barH - 1);
        }
        ctx.lineWidth = 0.25;
        ctx.strokeStyle = '#000';
        ctx.beginPath();
        ctx.moveTo(0, this.h * .9 | 0);
        ctx.lineTo(this.w, this.h * .9 | 0);
        ctx.closePath();
        ctx.stroke();
        ctx.restore();
    };
    OU.activity.Timegraph.prototype.render = function () {
        var interval = this.data.interval, scrW = this.w, i = 0, year, month, dayTxt, weekDay, lX, w, unitWidth,
            vBarDate = new Date(this.data.startDate),
            day = 1,
            rightX = 0, vpPropW = 1, vpPropH = 1,
            lineTop = this.data.lineY,
            lineBottom = this.h * .9,
            ctx = this.graphLayer.context,
            EvCtx = this.eventLayer.context,
            bH = this.data.barH,
            mTxt = this.constants.monthText,
            dTxt = this.constants.dayText,
            startMS = this.data.startDate.valueOf(),
            leftX = (this.scale * ((vBarDate.valueOf() - startMS) / 1000) + this.graphX) | 0;
        //debugTimer.start();
        if (this.graphX!=this.cacheX) {
            ctx.clearRect(0, 0, this.w, this.h);
            ctx.save();
            ctx.fillStyle = 'rgba(255,255,255,0.25)';
            ctx.font = ((bH / 2) | 0) + 'px ' + OU.theme.font;
            ctx.textAlign = 'center';
            do {
                switch(interval) {
                    default:
                    case 'year':
                        year = vBarDate.getFullYear();
                        vBarDate.incYear(this.step);
                        rightX = (this.scale * ((vBarDate.valueOf() - startMS) / 1000) + this.graphX) | 0;
                        if (rightX > 0) {
                            unitWidth = rightX - leftX;
                            ctx.fillStyle = '#000';
                            ctx.fillText(year, leftX + unitWidth / 2, lineTop + bH * .5);
                        }
                        break;
                    case 'month':
                        month = mTxt[vBarDate.getMonth()];
                        vBarDate.incMonth(this.step);
                        rightX = (this.scale * ((vBarDate.valueOf() - startMS) / 1000) + this.graphX) | 0;
                        if (rightX > 0) {
                            unitWidth = rightX - leftX;
                            ctx.fillStyle = '#000';
                            ctx.fillText(month, leftX + unitWidth / 2, lineTop + bH * .5);
                        }
                        break;
                    case 'day':
                        day = vBarDate.getDay();
                        dayTxt = vBarDate.getDate();
                        vBarDate.incDay(this.step);
                        rightX = (this.scale * ((vBarDate.valueOf() - startMS) / 1000) + this.graphX) | 0;
                        if (rightX > 0) {
                            weekDay = dTxt[day];
                            unitWidth = rightX - leftX;
                            ctx.fillStyle = '#000';
                            ctx.fillText(dayTxt, leftX + unitWidth / 2, lineTop + bH * .5);
                            ctx.fillStyle = '#fff';
                            ctx.fillText(weekDay, leftX + unitWidth / 2, lineTop + bH * 1.5);
                        }
                        break;
                }
                if (rightX > 0) {
                    ctx.beginPath();
                    if (day==6 || day==1)
                        ctx.strokeStyle = '#449';
                    else
                        ctx.strokeStyle = '#ccc';
                    ctx.moveTo(leftX, lineTop);
                    ctx.lineTo(leftX, lineBottom);
                    ctx.stroke();
                    ctx.closePath();
                }
                leftX = rightX;
            } while (leftX < scrW);
            this.unitWidth = unitWidth;
            //debugTimer.save('bars');
            ctx.font = ((bH / 2) | 0) + 'px ' + OU.theme.font;
            ctx.strokeStyle = '#fff';
            if (interval=='month' || interval=='day') { // Add years
                vBarDate = new Date(this.data.startDate);
                vBarDate.setMonth(0, 1);
                leftX = this.scale * ((vBarDate.valueOf() - startMS) / 1000) + this.graphX;
                do {
                    year = vBarDate.getFullYear();
                    vBarDate.incYear();
                    rightX = this.scale * ((vBarDate.valueOf() - startMS) / 1000) + this.graphX;
                    if (rightX > 0) {
                        lX = leftX < 0 ? 0 : leftX;
                        w = rightX - lX;
                        if (lX + w > scrW)
                            w = scrW - lX;
                        ctx.fillStyle = '#000';
                        ctx.fillText(year, lX + w / 2, this.h * .1 + bH * .5);
                        ctx.beginPath();
                        ctx.moveTo(lX + w, this.h * .1);
                        ctx.lineTo(lX + w, this.h * .1 + bH);
                        ctx.closePath();
                        ctx.stroke();
                    }
                    leftX = rightX;
                } while (leftX < scrW);
            }
            if (interval=='day') { // Add Months
                vBarDate = new Date(this.data.startDate);
                vBarDate.setDate(1);
                leftX = this.scale * ((vBarDate.valueOf() - startMS) / 1000) + this.graphX;
                do {
                    month = mTxt[vBarDate.getMonth()];
                    vBarDate.incMonth();
                    rightX = this.scale * ((vBarDate.valueOf() - startMS) / 1000) + this.graphX;
                    if (rightX > 0) {
                        lX = leftX < 0 ? 0 : leftX;
                        w = rightX - lX;
                        if (lX + w > scrW)
                            w = scrW - lX;
                        ctx.fillStyle = '#000';
                        ctx.fillText(month, lX + w / 2, this.h * .1 + bH * 1.5);
                        ctx.beginPath();
                        ctx.moveTo(lX + w, this.h * .1 + bH);
                        ctx.lineTo(lX + w, this.h * .1 + bH + bH);
                        ctx.closePath();
                        ctx.stroke();
                    }
                    leftX = rightX;
                } while (leftX < scrW);
            }
            this.cacheX = this.graphX;
            //debugTimer.save('years+months');
        }
        var events = this.data.events;
        this.clearLayoutEvents(events);
        this.eventLayer.events.clickable.length = 0;
        this.eventLayer.events.clickable.push(this.dragEvents);
        //debugTimer.save('clear');
        if (!this.slider.hSet) { // Only need to do this on the first render
            var tDims = this.layoutEvents(EvCtx, events, bH / 2);
            this.range = tDims.w + tDims.x;
            this.dragEvents.w = this.range;
            vpPropH = (this.slider.map.vpH - bH) / (tDims.h);
            vpPropW = (this.slider.map.vpW - bH) / (this.range);
            if (this.range > this.w || vpPropW < 1 || vpPropH < 1) {
                this.graphH = tDims.h + bH / 2;
                this.slider.map.w = this.range;
                this.slider.map.h = this.graphH;
                this.dragEvents.h = tDims.h;
                this.slider.hSet = true;
                this.slider.render({
                    x:-this.graphX,
                    y:-this.graphY,
                    propWidth:vpPropW,
                    propHeight:vpPropH
                });
            }
        }
        //debugTimer.save('layout events');
        EvCtx.clearRect(0, 0, this.w, this.h);
        this.renderEvents(EvCtx, events);
        //debugTimer.save('render events');
        ctx.restore();
        //debugTimer.report();
    };
    OU.activity.Timegraph.prototype.clearLayoutEvents = function ( events ) {
        var i = 0, event;
        for (i = events.length; i--;) {
            event = events[i];
            event.hasLayout = false;
            if (event.events!=undefined) {
                event.events = this.clearLayoutEvents(event.events); // recurse through nested events
            }
        }
        return events;
    };
    OU.activity.Timegraph.prototype.detectEventCollision = function ( dims, events ) {
        var i = 0, event, collide;
        for (i = events.length; i--;) {
            event = events[i];
            if (event.hasLayout==false)
                return false; // reached an event that is not placed yet, so no collision
            var ErX = event.x + event.w,
                EbY = event.y + event.h,
                DrX = dims.x + dims.w,
                DbY = dims.y + dims.h;
            collide = true;
            if (event.x > DrX)
                collide = false;
            if (ErX < dims.x)
                collide = false;
            if (event.y > DbY)
                collide = false;
            if (EbY < dims.y)
                collide = false;
            if (collide==true)
                return true;
            if (event.events!=undefined) {
                if (this.detectEventCollision(dims, event.events)) // recurse through nested events
                    return true;
            }
        }
        return false;
    };
    OU.activity.Timegraph.prototype.layoutEvents = function ( ctx, events, yOffset ) {
        var i = 0, event, subDims = null,
            bH = this.data.barH,
            dims = {
                x:-1,
                y:-1,
                w:0,
                h:0
            },
            pad = bH / 2,
            durW, textW, startS, endS, y,
            graphS = this.data.startDate.valueOf() / 1000;
        for (i = events.length; i--;) {
            event = events[i];
            ctx.font = ((bH / 2) | 0) + 'px ' + OU.theme.font;
            textW = ctx.measureText(event.label).width + pad;
            if (event.type=='event')
                textW += bH * 1.2;
            startS = event.startDate.valueOf() / 1000;
            endS = event.endDate.valueOf() / 1000;
            durW = this.scale * (endS - startS);
            event.w = textW > durW ? textW : durW;
            event.h = bH;
            if (event.type=='duration')
                event.h *= 1.2;
            event.x = this.scale * (startS - graphS);
            if (event.image!==undefined) {
                event.h = event.h + event.image.height + pad / 2;
                if (event.w < event.image.width + pad)
                    event.w = event.image.width + pad;
            }
            // set my position Y
            event.hasLayout = false;
            y = yOffset;
            do {
                if (this.detectEventCollision({
                    x:event.x,
                    y:y,
                    w:event.w,
                    h:event.h
                }, events)) {
                    y = y + pad;
                }
                else {
                    event.y = y + pad / 4;
                    event.hasLayout = true;
                }
            } while (!event.hasLayout);
            if (event.events!=undefined) {
                subDims = this.layoutEvents(ctx, event.events, event.y + event.h); // recurse through nested events
                event.events = subDims.events;
                if (subDims.x + subDims.w > event.x + event.w)
                    event.w = subDims.x + subDims.w - event.x;
                event.h = event.h + subDims.h + pad;
                event = this.reCheckCollision(event, events); // adding child events may have incurred a collision
            }
            // extend OU.er dims if required
            if (dims.x==-1 || event.x < dims.x) {
                dims.w = dims.w + (dims.x - event.x);
                dims.x = event.x;
            }
            if (dims.y==-1 || event.y < dims.y) {
                dims.h = dims.h + (dims.y - event.y);
                dims.y = event.y;
            }
            if (event.y + event.h > dims.y + dims.h)
                dims.h = event.y + event.h - dims.y;
            if (event.x + event.w > dims.x + dims.w)
                dims.w = event.x + event.w - dims.x;
        }
        dims.h = dims.h + pad;
        dims.w = dims.w + pad;
        return {
            x:dims.x,
            y:dims.y,
            w:dims.w,
            h:dims.h,
            events:events
        };
    };
    OU.activity.Timegraph.prototype.moveEvent = function ( event, dY ) {
        if (event.hasLayout) {
            event.y = event.y + dY;
            if (event.events!==undefined) {
                for (var i = event.events.length; i--;) {
                    event.events[i] = this.moveEvent(event.events[i], dY);
                }
            }
        }
        return event;
    };
    OU.activity.Timegraph.prototype.reCheckCollision = function ( event, events ) {
        var y = event.y, dY,
            pad = this.data.barH / 2;
        event.hasLayout = false;
        do {
            if (this.detectEventCollision({
                x:event.x,
                y:y,
                w:event.w,
                h:event.h
            }, events)) {
                y = y + pad;
            }
            else {
                event.hasLayout = true;
                dY = y - event.y;
                if (dY > 0)
                    event = this.moveEvent(event, dY + pad / 4);
            }
        } while (!event.hasLayout);
        return event;
    };
    OU.activity.Timegraph.prototype.renderEvents = function ( ctx, origEvents ) {
        var i = 0, event, lH = this.data.barH, uW = this.unitPixels, eW, circle = Math.PI * 2,
            dY = this.y + this.graphY,
            dX = this.x + this.graphX,
            events = [],
            gW = this.w,
            gH = this.graphH;
        for (i = origEvents.length; i--;)
            events.push(origEvents[i]);
        ctx.font = ((lH / 2) | 0) + 'px ' + OU.theme.font;
        while (events.length > 0) {
            event = events[0];
            if (dX + event.x < gW && dX + event.x + event.w > 0 && dY + event.y < gH && dY + event.y + event.h > 0) {
                ctx.save();
                switch(event.type) {
                    case 'event':
                        ctx.background({
                            RGB:event.bgRGB,
                            alpha:event.bgAlpha,
                            radius:lH / 2
                        }, {
                            x:event.x + dX,
                            y:event.y + dY,
                            w:event.w,
                            h:event.h
                        });
                        ctx.strokeStyle = '#22c';
                        ctx.fillStyle = '#fff';
                        ctx.lineWidth = lH / 8;
                        ctx.beginPath();
                        ctx.moveTo(dX + event.x + lH * 1.1, dY + event.y + event.h - lH / 2);
                        ctx.lineTo(dX + event.x + (lH / 2) + (lH / 4), dY + event.y + event.h - lH / 2);
                        ctx.arc(dX + event.x + lH / 2, dY + event.y + event.h - lH / 2, lH / 4, 0, circle, false);
                        ctx.closePath();
                        ctx.fill();
                        ctx.stroke();
                        ctx.textAlign = 'left';
                        ctx.fillStyle = event.colour || '#22c';
                        ctx.fillText(event.label, dX + event.x + lH * 1.2, dY + event.y + event.h - lH / 2);
                        if (event.image!==undefined)
                            ctx.drawImage(event.image, dX + event.x + lH / 4, dY + event.y + lH / 4);
                        break;
                    case 'group':
                        ctx.background({
                            RGB:event.bgRGB,
                            alpha:event.bgAlpha,
                            borderCol:'#999',
                            radius:lH / 2
                        }, {
                            x:event.x + dX,
                            y:event.y + dY,
                            w:event.w,
                            h:event.h
                        });
                        ctx.textAlign = 'center';
                        ctx.fillStyle = event.colour || '#000';
                        ctx.font = 'bold ' + ctx.font;
                        if (event.image!==undefined) {
                            ctx.drawImage(event.image, dX + event.x + (event.w - event.image.width) / 2, dY + event.y + lH / 4);
                            ctx.fillText(event.label, dX + event.x + event.w / 2, dY + event.y + lH + event.image.height);
                        }
                        else {
                            ctx.fillText(event.label, dX + event.x + event.w / 2, dY + event.y + lH / 2);
                        }
                        break;
                    case 'duration':
                        ctx.background({
                            RGB:event.bgRGB,
                            alpha:event.bgAlpha,
                            radius:lH / 4
                        }, {
                            x:event.x + dX,
                            y:event.y + dY,
                            w:event.w,
                            h:event.h
                        });
                        eW = this.scale * (event.endDate.valueOf() - event.startDate.valueOf()) / 1000;
                        ctx.fillStyle = '#22c';
                        ctx.strokeStyle = '#22c';
                        ctx.lineWidth = lH / 16;
                        ctx.beginPath();
                        ctx.moveTo(dX + event.x, dY + event.y + event.h - lH * .3);
                        ctx.lineTo(dX + event.x + uW / 5, dY + event.y + event.h - lH * .4);
                        ctx.lineTo(dX + event.x + uW / 5, dY + event.y + event.h - lH * .2);
                        ctx.lineTo(dX + event.x, dY + event.y + event.h - lH * .3);
                        ctx.moveTo(dX + event.x + eW, dY + event.y + event.h - lH * .3);
                        ctx.lineTo(dX + event.x + eW - uW / 5, dY + event.y + event.h - lH * .4);
                        ctx.lineTo(dX + event.x + eW - uW / 5, dY + event.y + event.h - lH * .2);
                        ctx.lineTo(dX + event.x + eW, dY + event.y + event.h - lH * .3);
                        ctx.moveTo(dX + event.x, dY + event.y + event.h - lH * .3);
                        ctx.lineTo(dX + event.x + eW, dY + event.y + event.h - lH * .3);
                        ctx.closePath();
                        ctx.fill();
                        ctx.stroke();
                        ctx.textAlign = 'left';
                        ctx.fillStyle = event.colour || '#22c';
                        ctx.fillText(event.label, dX + event.x + lH / 4, dY + event.y + (event.h) - (lH * .7));
                        if (event.image!==undefined)
                            ctx.drawImage(event.image, dX + event.x + lH / 4, dY + event.y + lH / 4);
                        break;
                }
                ctx.restore();
                this.eventLayer.events.clickable.push(new this.EventClick(event, this)); // push event click area
            }
            if (event.events!==undefined) {
                for (i = event.events.length; i--;)
                    events.push(event.events[i]);
            }
            events.shift();
        }
    };
    OU.activity.Timegraph.prototype.initEvents = function ( events ) {
        var i = 0, event;
        for (i = events.length; i--;) {
            event = events[i];
            event.type = 'event';
            event.bgRGB = event.bgRGB || '255,255,255';
            event.bgAlpha = event.bgAlpha || 0.5;
            event.startDate = new Date(event.start);
            if (isNaN(event.startDate.valueOf())) {
                alert('Incorrect date format in event start date: ' + event.label);
                exit(); // throws an error
            }
            if (event.end!==undefined && event.end!='') {
                event.endDate = new Date(event.end);
                if (isNaN(event.endDate.valueOf())) {
                    alert('Incorrect date format in event end date: ' + event.label);
                    exit(); // throws an error
                }
                event.type = 'duration';
            }
            else {
                event.endDate = event.startDate;
            }
            this.updateStartEnd(event);
            if (event.events!==undefined) {
                event.type = 'group';
                event.events = this.initEvents(event.events); // recurse through nested events
            }
        }
        return events;
    };
    OU.activity.Timegraph.prototype.EventClick = function ( event, graph ) {
        this.event = event;
        this.graph = graph;
        OU.activity.Timegraph.prototype.EventClick.prototype.isHit = function ( x, y, pressed ) {
            var ev = this.event, h,
                dY = this.graph.graphY, dX = this.graph.graphX;
            if (ev.info!==undefined && ev.info.length > 0) {
                h = ev.h;
                if (ev.type=='group')
                    h = this.graph.data.barH;
                if (pressed && x > ev.x + dX && x < ev.x + ev.w + dX && y > ev.y + dY && y < ev.y + h + dY) {
                    this.graph.slider.halted = true;
                    new OU.util.PopUpInfo({
                        container:this,
                        txt:ev.info
                    });
                }
            }
        }
    };
    OU.activity.Timegraph.prototype.slideLeft = function () {
        var perc = (-this.graphX / this.range) - 0.2;
        this.slider.setTarget(perc < 0 ? 0 : perc);
    };
    OU.activity.Timegraph.prototype.slideRight = function () {
        var perc = (-this.graphX / this.range) + 0.2;
        this.slider.setTarget(perc > 1 ? 1 : perc);
    };
    OU.activity.Timegraph.prototype.initData = function () {
        var interval = this.data.interval || 'year', self = this;
        if (this.doneInitEvents===undefined)
            this.initEvents(this.data.events);
        this.doneInitEvents = true;
        this.data.startDate = new Date(this.data.startEvent.startDate);
        this.data.endDate = new Date(this.data.endEvent.endDate);
        interval = interval.toLowerCase();
        this.data.interval = interval;
        this.step = this.data.step || 1;
        switch(interval) {
            default:
            case 'year':
                this.data.startDate.decYear();
                this.data.endDate.incYear();
                this.data.numBars = 1;
                this.data.barH = this.h * .04;
                this.data.graphY = this.h * .14;
                this.data.lineY = this.h * .1;
                this.data.startDate.setMonth(0, 1);
                this.data.endDate.setMonth(0, 1);
                this.data.endDate.incYear();
                this.unitSecs = 31536000;
                this.unitPixels = this.w / 20; // render approx 40 units on screen in a single view
                //TODO use unitPixels to effect a zoom.
                break;
            case 'month':
                this.data.startDate.decMonth();
                this.data.endDate.incMonth();
                this.data.numBars = 2;
                this.data.barH = this.h * .04;
                this.data.graphY = this.h * .18;
                this.data.lineY = this.h * .14;
                this.data.startDate.setDate(1);
                this.data.endDate.setDate(1);
                this.data.endDate.incMonth();
                this.unitSecs = 2562000;
                this.unitPixels = this.w / 20; // render approx 40 units on screen in a single view
                //TODO use unitPixels to effect a zoom.
                break;
            case 'day':
                this.data.startDate.decDay();
                this.data.endDate.incDay();
                this.data.numBars = 4;
                this.data.barH = this.h * .04;
                this.data.graphY = this.h * .26;
                this.data.lineY = this.h * .18;
                this.unitSecs = 86400;
                this.unitPixels = this.w / 40; // render approx 40 units on screen in a single view
                //TODO use unitPixels to effect a zoom.
                break;
        }
        if (this.data.start!==undefined) {
            var givenStart = new Date(this.data.start);
            if (givenStart.valueOf() < this.data.startDate.valueOf())
                this.data.startDate = givenStart;
        }
        if (this.data.end!==undefined) {
            var givenEnd = new Date(this.data.end);
            if (givenEnd.valueOf() > this.data.endDate.valueOf())
                this.data.endDate = givenEnd;
        }
        this.durationSecs = (this.data.endDate.valueOf() - this.data.startDate.valueOf()) / 1000;
        this.scale = (this.unitPixels / this.unitSecs) / this.step;
        this.range = this.durationSecs * this.scale;
        if (this.range < 0)
            this.range = 0;
        this.graphX = 0;
        this.graphY = 0;
        this.imageLoader = new OU.util.ImageLoader({
            container:this,
            data:this.data,
            onLoad:function () {
                self.start();
            }
        });
    };
    OU.activity.Timegraph.prototype.initConstants = function () {
        this.constants = {};
        this.constants.monthText = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];
        this.constants.dayText = ['S', 'M', 'T', 'W', 'T', 'F', 'S'];
    };
    OU.activity.Timegraph.prototype.updateStartEnd = function ( ev ) {
        if (this.data.startEvent===undefined) {
            this.data.startEvent = ev;
            this.data.endEvent = ev;
            return;
        }
        if (this.data.startEvent.startDate > ev.startDate)
            this.data.startEvent = ev;
        if (this.data.endEvent.endDate < ev.endDate)
            this.data.endEvent = ev;
    };
    OU.activity.Timegraph.prototype.accessibleEvents = function ( events ) {
        var i, e, start, end,
            h = '<div class="indent">';
        for (i = 0; i < events.length; i++) {
            e = events[i];
            h += "<div>";
            if (( e.start ) && e.start.length > 0) {
                start = new Date(e.start);
                h += "<p>Event start: " + start.getFullYear();
                if (( e.end ) && e.end.length > 0) {
                    end = new Date(e.end);
                    h += " - Event end: " + end.getFullYear();
                }
                h += '</p>';
                if (( e.info ) && e.info.length > 0) {
                    h += "<p>" + e.info + "</p>";
                }
            }
            h += "</div>";
            if (e.events!==undefined && e.events.length > 0) {
                h += this.accessibleEvents(e.events);
            }
        }
        return h + "</div>";
    };
    OU.activity.Timegraph.prototype.renderAccessible = function () {
        var h = "<div id='timelineAccessible'>";
        if (this.data.title && this.data.title.length > 0) {
            h += "<h1>" + this.data.title + "</h1><p>This is a timeline marking important dates relating to " + this.data.title + "</p>";
        }
        h += "<h2>Timeline Events:</h2>" + this.accessibleEvents(this.data.events) + "</div>";
        document.body.innerHTML = '';
        var accessible = document.createElement('div');
        accessible.innerHTML = h;
        document.body.appendChild(accessible);
    };
    OU.base(this, data, instance, controller);
};
OU.inherits(OU.activity.Timegraph, OU.util.Activity);
